/*
  ZynAddSubFX - a software synthesizer

  EQ.C - EQ effect
  Copyright (C) 2002-2005 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  Modified for rakarrack by Josep Andreu

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

*/

#include "EQ_.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

EQ::EQ()
{
  for (i32 i = 0; i < MAX_EQ_BANDS; i++)
  {
    filter[i].Ptype = 0;
    filter[i].Pfreq = 64;
    filter[i].Pgain = 64;
    filter[i].Pq = 64;
    filter[i].Pstages = 0;
    filter[i].l = new AnalogFilter(6, 1000.0f, 1.0f, 0);
    filter[i].r = new AnalogFilter(6, 1000.0f, 1.0f, 0);
  }
  //default values
  Pvolume = 50;

  setpreset(0);
  cleanup();
}

void EQ::cleanup()
{
  for (i32 i = 0; i < MAX_EQ_BANDS; i++)
  {
    filter[i].l->cleanup();
    filter[i].r->cleanup();
  }
}

void EQ::processBlock(f32** inOutBlock, const i32 blockSize)
{
  for (i32 i = 0; i < MAX_EQ_BANDS; i++)
  {
    //if (filter[i].Ptype == 0)
    //  continue;
    filter[i].l->filterout(inOutBlock[0], blockSize);
    filter[i].r->filterout(inOutBlock[1], blockSize);
  }

  for (i32 i = 0; i < blockSize; i++)
  {
    inOutBlock[0][i] = inOutBlock[0][i] * outvolume;
    inOutBlock[1][i] = inOutBlock[1][i] * outvolume;
  }
}

void EQ::setvolume(const i32 Pvolume_)
{
  Pvolume = Pvolume_;

  outvolume = powf(0.005f, (1.0f - (f32)Pvolume_ / 127.0f)) * 10.0f;
}

void EQ::setpreset_Cabinet(const i32 npreset)
{
  const i32 PRESET_SIZE = 81;
  i32 presets[][PRESET_SIZE] = {

    //Marshall-4-12
    {2, 1135, 0, 0, 0, 8, 116, 87, 71, 0, 7, 128, 77, 53, 0, 7, 825, 84, 64, 0,
    7, 1021, 49, 19, 0, 7, 2657, 75, 95, 0, 7, 3116, 72, 86, 0, 7, 10580, 31, 64, 0,
    7, 17068, 28, 68, 0, 3, 17068, 0, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    67},

    //Celestion G12M
    {2, 704, 64, 64, 0, 7, 64, 90, 111, 0, 7, 151, 69, 59, 0, 7, 1021, 56, 25, 0,
    7, 1562, 56, 62, 0, 7, 2389, 71, 77, 0, 7, 5896, 53, 69, 0, 7, 6916, 59, 83, 0,
    7, 7691, 52, 102, 0, 7, 9021, 43, 64, 0, 7, 15347, 32, 59, 0, 3, 17068, 1, 58, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Jensen Alnico P12-N
    {2, 270, 0, 0, 0, 7, 414, 46, 56, 0, 7, 1647, 46, 84, 0, 7, 7293, 53, 69, 0,
    7, 10032, 43, 109, 0, 7, 12408, 41, 86, 0, 7, 14553, 41, 90, 0, 7, 17068, 35, 27, 0,
    3, 17068, 64, 61, 0, 7, 4065, 71, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Jensen Alnico P15-N
    {0, 600, 64, 64, 0, 2, 256, 64, 64, 0, 7, 414, 49, 43, 0, 7, 1832, 68, 58, 0,
    7, 4065, 46, 92, 0, 7, 8111, 52, 75, 0, 7, 11766, 43, 83, 0, 3, 15347, 64, 41, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Eminence Delta Demon
    {2, 68, 64, 64, 0, 7, 600, 55, 43, 0, 7, 1021, 72, 72, 0, 7, 1562, 68, 92, 0,
    7, 2519, 66, 80, 0, 7, 6218, 52, 66, 0, 7, 9513, 43, 81, 0, 3, 10580, 64, 66, 0,
    7, 12408, 58, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Celestion EVH12
    {2, 151, 64, 64, 0, 7, 1481, 56, 64, 0, 7, 2519, 66, 87, 0, 7, 3116, 66, 92, 0,
    7, 7293, 53, 87, 0, 7, 8554, 52, 87, 0, 7, 11157, 46, 59, 0, 3, 17068, 61, 30, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Eminence Copperhead

    {2, 301, 64, 64, 0, 7, 301, 58, 77, 0, 7, 1077, 68, 74, 0, 7, 2519, 71, 77, 0,
    7, 6558, 55, 87, 0, 7, 7293, 58, 37, 0, 7, 15347, 13, 75, 0, 3, 17068, 50, 43, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Mesa Boogie
    {2, 600, 64, 64, 0, 7, 128, 93, 52, 0, 7, 633, 58, 80, 0, 7, 1077, 52, 89, 0,
    7, 1647, 66, 89, 0, 7, 2037, 75, 86, 0, 7, 3466, 75, 90, 0, 7, 6218, 62, 52, 0,
    7, 11157, 58, 71, 0, 3, 1404, 64, 47, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Jazz-Chorus
    {4, 72, 0, 40, 0, 7, 72, 50, 43, 0, 7, 667, 38, 89, 0, 7, 3466, 77, 112, 0,
    7, 7293, 46, 93, 0, 7, 8111, 43, 105, 0, 7, 17068, 15, 40, 0, 3, 17068, 58, 55, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Vox-Bright  
    {2, 1021, 64, 64, 0, 7, 243, 68, 72, 0, 7, 2657, 75, 41, 0, 7, 3466, 69, 96, 0,
    7, 4767, 74, 74, 0, 7, 6218, 44, 81, 0, 7, 8554, 52, 100, 0, 7, 13086, 46, 72, 0,
    7, 15347, 52, 62, 0, 3, 13800, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67},

    //Marshall-I
    {2, 4287, 53, 64, 0, 7, 122, 80, 25, 0, 7, 633, 69, 86, 0, 7, 1021, 78, 59, 0,
    7, 1647, 75, 64, 0, 7, 2389, 86, 78, 0, 7, 3286, 95, 61, 0, 7, 6916, 61, 59, 0,
    7, 8554, 56, 84, 0, 7, 12408, 22, 18, 0, 3, 10032, 64, 61, 0, 0, 64, 64, 64, 0,
    0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 0, 64, 64, 64, 0, 67}
  };

  for (i32 n = 0; n < 16; n++)
  {
    changepar(n * 5 + 10, presets[npreset][n * 5]);
    changepar(n * 5 + 11, presets[npreset][n * 5 + 1]);
    changepar(n * 5 + 12, presets[npreset][n * 5 + 2]);
    changepar(n * 5 + 13, presets[npreset][n * 5 + 3]);
    changepar(n * 5 + 14, presets[npreset][n * 5 + 4]);
  }

  cabinet_preset = npreset;
}

void EQ::setpreset_EQ(i32 npreset)
{
  const i32 PRESET_SIZE = 12;
  i32 presets[][PRESET_SIZE] = {
    //Plain
    {64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
    //Pop
    {78, 71, 64, 64, 64, 64, 64, 64, 71, 80, 64, 42},
    //Jazz
    {71, 68, 64, 64, 64, 64, 64, 64, 66, 69, 64, 40}
  };

  for (i32 n = 0; n < 10; n++)
    changepar(n * 5 + 12, presets[npreset][n]);
  changepar(0, presets[npreset][10]);
  for (i32 n = 0; n < 10; n++)
    changepar(n * 5 + 13, presets[npreset][11]);
}

void EQ::setpreset_ParametricEQ(const i32 npreset)
{
  const i32 PRESET_SIZE = 10;
  i32 presets[][PRESET_SIZE] = {
    //Plain
    {72, 64, 64, 1077, 64, 64, 8111, 64, 64, 64},
    //Pop
    {72, 73, 45, 1077, 64, 64, 8111, 69, 38, 64},
    //Jazz
    {72, 71, 38, 1077, 64, 64, 10580, 69, 38, 64}
  };

  for (i32 n = 0; n < 3; n++)
  {
    changepar(n * 5 + 11, presets[npreset][n * 3]);
    changepar(n * 5 + 12, presets[npreset][n * 3 + 1]);
    changepar(n * 5 + 13, presets[npreset][n * 3 + 2]);
  }
  changepar(0, presets[npreset][9]);
}

void EQ::setpreset(const i32 npreset)
{
  const i32 PRESET_SIZE = 1;
  i32 presets[][PRESET_SIZE] = {
    //EQ 1
    {67},
    //EQ 2
    {67}
  };

  for (i32 n = 0; n < PRESET_SIZE; n++)
    changepar(n, presets[npreset][n]);
}

void EQ::changepar(i32 npar, i32 value)
{
  switch (npar)
  {
  case 0:
    setvolume(value);
    break;
  }
  if (npar < 10)
    return;

  i32 nb = (npar - 10) / 5;	//number of the band (filter)
  if (nb >= MAX_EQ_BANDS)
    return;
  i32 bp = npar % 5;		//band paramenter

  f32 tmp;
  switch (bp)
  {
  case 0:
    if (value > 9)
      value = 0;		//has to be changed if more filters will be added
    filter[nb].Ptype = value;
    if (value != 0)
    {
      filter[nb].l->settype(value - 1);
      filter[nb].r->settype(value - 1);
    };
    break;
  case 1:
    filter[nb].Pfreq = value;
    tmp = (f32)value;
    filter[nb].l->setfreq(tmp);
    filter[nb].r->setfreq(tmp);
    break;
  case 2:
    filter[nb].Pgain = value;
    tmp = 30.0f * ((f32)value - 64.0f) / 64.0f;
    filter[nb].l->setgain(tmp);
    filter[nb].r->setgain(tmp);
    break;
  case 3:
    filter[nb].Pq = value;
    tmp = powf(30.0f, ((f32)value - 64.0f) / 64.0f);
    filter[nb].l->setq(tmp);
    filter[nb].r->setq(tmp);
    break;
  case 4:
    if (value >= MAX_FILTER_STAGES)
      value = MAX_FILTER_STAGES - 1;
    filter[nb].Pstages = value;
    filter[nb].l->setstages(value);
    filter[nb].r->setstages(value);
    break;
  default:
    unreachable();
  }
}

i32 EQ::getpar(i32 npar)
{
  switch (npar)
  {
  case 0:
    return Pvolume;
  }

  if (npar < 10)
    return 0;

  i32 nb = (npar - 10) / 5;	//number of the band (filter)
  if (nb >= MAX_EQ_BANDS)
    return 0;
  i32 bp = npar % 5;		//band paramenter

  switch (bp)
  {
  case 0:
    return filter[nb].Ptype;
  case 1:
    return filter[nb].Pfreq;
  case 2:
    return filter[nb].Pgain;
  case 3:
    return filter[nb].Pq;
  case 4:
    return filter[nb].Pstages;
  }
  ASSERT(false);
  return 0;
}

f32 EQ::getfreqresponse(f32 freq)
{
  f32 resp = 1.0f;

  for (i32 i = 0; i < MAX_EQ_BANDS; i++)
  {
    if (filter[i].Ptype == 0)
      continue;
    resp *= filter[i].l->H(freq);
  }
  return rap2dB(resp * outvolume);
}

#endif // SHR3D_SFX_CORE_RAKARRACK
