// SPDX-License-Identifier: Unlicense

#include "file.h"

#include "global.h"
#include "skybox.h"
#include "stage.h"
#include "string_.h"

#include <stdio.h>
#include <string.h>
#include <string>
#include <fstream>
#include <sstream>
#include <filesystem>

#ifdef PLATFORM_WINDOWS
#include <Windows.h>
#ifndef SHR3D_MINIMAL_WIN32
#include <shlobj.h>
#endif // SHR3D_MINIMAL_WIN32
#endif // PLATFORM_WINDOWS

#ifdef __ANDROID__
#ifdef PLATFORM_OPENXR_ANDROID
#include "dirent.h"
#endif // PLATFORM_OPENXR_ANDROID
#include "collection.h"
#include <jni.h>
#endif // __ANDROID__

#include <sys/stat.h>


#ifdef PLATFORM_WINDOWS
static std::wstring windowsExtendedPath(std::wstring relPath)
{
  if (relPath.empty())
    return relPath;

  std::replace(relPath.begin(), relPath.end(), '/', '\\');

  if (relPath[0] == L'\\' && relPath[1] == L'\\')
    return relPath;

  std::wstring extendedPath;
  if (relPath.length() >= 2 && relPath[1] == L':')
  {
    extendedPath = relPath;
  }
  else
  {
    extendedPath.resize(4096);
    DWORD length2 = GetCurrentDirectoryW(u32(extendedPath.size()), extendedPath.data());
    ASSERT(extendedPath[length2] == L'\0');
    extendedPath.resize(length2);
    if (extendedPath.back() != L'\\')
      extendedPath += L'\\';
    extendedPath += relPath;
  }

  { // normalize path
    std::vector<std::wstring> components;
    size_t length = extendedPath.length();
    size_t start = extendedPath[0] == L'\\' ? 4 : 0;

    while (start < length)
    {
      size_t end = extendedPath.find(L'\\', start);
      if (end == std::wstring::npos)
        end = length;

      std::wstring token = extendedPath.substr(start, end - start);
      start = end + 1;

      if (token == L"." || token.empty())
        continue;

      if (token != L"..")
        components.push_back(token);
      else if (!components.empty())
        components.pop_back();
    }

    extendedPath = LR"(\\?)";
    for (const std::wstring& comp : components)
      extendedPath += L"\\" + comp;
  }

  if (extendedPath.back() == L':')
    extendedPath += L"\\";

  if (extendedPath.back() != L'\\' && relPath.back() == L'\\')
    extendedPath += L"\\";

  return extendedPath;
}
#endif // PLATFORM_WINDOWS

bool File::exists(const char* filePath)
{
#if defined(PLATFORM_WINDOWS)
  const std::wstring wFilepath = String::s2ws(filePath, i32(strlen(filePath)));
  const std::wstring wFilepathExpanded = String::ws_ExpandEnvironmentStrings(wFilepath);
  struct _stat buffer;
  return _wstat(wFilepathExpanded.c_str(), &buffer) == 0;
#elif defined(__ANDROID__)
  const std::string rootFilepath = filePath[0] == '/' ? filePath : (Global::rootPath + filePath);
  struct stat buffer;
  return stat(rootFilepath.c_str(), &buffer) == 0;
#else
  struct stat buffer;
  return stat(filePath, &buffer) == 0;
#endif
}

bool File::directoryExists(const char* filePath)
{
#if defined(PLATFORM_WINDOWS)
  const std::wstring wFilepath = String::s2ws(filePath, i32(strlen(filePath)));
  const std::wstring wFilepathExpanded = String::ws_ExpandEnvironmentStrings(wFilepath);
  const std::wstring wExtendedPath = windowsExtendedPath(wFilepathExpanded);
  const DWORD fileAttributes = GetFileAttributesW(wExtendedPath.c_str());
  if (fileAttributes == INVALID_FILE_ATTRIBUTES)
    return false;
  return (fileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
#elif defined(PLATFORM_OPENXR_ANDROID)
  const std::string rootFilepath = filePath[0] == '/' ? filePath : (Global::rootPath + filePath);
  struct stat buffer;
  if (stat(rootFilepath.c_str(), &buffer) != 0)
    return false;
  return buffer.st_mode & S_IFDIR;
#elif defined(PLATFORM_ANDROID_SDL)
  const std::filesystem::path path(filePath[0] == '/' ? filePath : (Global::rootPath + filePath).c_str());
  return std::filesystem::is_directory(path);
#else
  return std::filesystem::is_directory(filePath);
#endif
}

const char* File::filename(const char* filePath)
{
  const char* sep = strrchr(filePath, '/');
  if (!sep || sep == filePath)
    return filePath;
  return sep + 1;
}

const char* File::extension(const char* filePath)
{
  const char* sep = strrchr(filePath, '.');
  if (!sep || sep == filePath)
    return "";
  return sep;
}

std::string File::replaceExtension(const char* filePath, const char* newExtension)
{
  const char* dot = strrchr(filePath, '.');
  if (dot == nullptr)
    return std::string(filePath) + newExtension;

  return std::string(filePath, dot) + newExtension;
}

u64 File::modTime(const char* filePath)
{
#ifdef PLATFORM_WINDOWS
  WIN32_FILE_ATTRIBUTE_DATA data;
  if (!GetFileAttributesExW(String::s2ws(filePath, i32(strlen(filePath))).c_str(), GetFileExInfoStandard, &data))
    return 0;
  ULARGE_INTEGER ull;
  ull.LowPart = data.ftLastWriteTime.dwLowDateTime;
  ull.HighPart = data.ftLastWriteTime.dwHighDateTime;
  constexpr ULONGLONG WINDOWS_TO_UNIX_EPOCH = 116444736000000000LL;
  return (ull.QuadPart - WINDOWS_TO_UNIX_EPOCH) / 10000000LL;
#else // PLATFORM_WINDOWS
  struct stat attr{};
  stat(filePath, &attr);
  return attr.st_mtime;
#endif // PLATFORM_WINDOWS
}

void File::createDirectories(const char* filePath)
{
#ifndef PLATFORM_PICO_4
  if (Global::installMode != InstallMode::installed)
    return;

#ifdef PLATFORM_WINDOWS
  {
    const std::wstring extendedPath = windowsExtendedPath(String::s2ws(filePath, i32(strlen(filePath))));
    std::wstring partialExtendedPath;
    partialExtendedPath.reserve(extendedPath.size());

    for (size_t i = 0; i < extendedPath.size(); ++i)
    {
      partialExtendedPath += extendedPath[i];

      if (extendedPath[i] != L'\\')
        continue;

      if (partialExtendedPath.size() <= 7 || partialExtendedPath.back() == L':')
        continue;

      const bool success = CreateDirectoryW(partialExtendedPath.c_str(), nullptr);
      ASSERT(success || GetLastError() == ERROR_ALREADY_EXISTS);
    }
    return;
  }
#else // PLATFORM_WINDOWS

  if (!directoryExists(filePath))
  {
#if defined PLATFORM_ANDROID_SDL || defined PLATFORM_OPENXR_ANDROID
    const std::filesystem::path path(filePath[0] == '/' ? filePath : (Global::rootPath + filePath).c_str());
#else // PLATFORM_ANDROID_SDL
    std::filesystem::path path(filePath);
#endif // PLATFORM_ANDROID_SDL

    ASSERT(path.u8string().back() == '/');

    {
      const bool success = std::filesystem::create_directories(path.parent_path()); // remove trailing '/'. It works but create_directories will return false.
      ASSERT(success);
    }
  }

#endif // PLATFORM_WINDOWS
#else // PLATFORM_PICO_4
  const std::string rootFilepath = filePath[0] == '/' ? filePath : (Global::rootPath + filePath);
  struct stat buffer;

  if (directoryExists(rootFilepath.c_str()))
    return;

  // Find the parent directory
  const size_t pos = rootFilepath.find_last_of('/');
  if (pos != std::string::npos) {
    std::string parent = rootFilepath.substr(0, pos);
    if (!directoryExists(parent.c_str()))
      createDirectories(parent.c_str());
  }

  if (mkdir(rootFilepath.c_str(), 0755) == 0 || errno == EEXIST) {
    return;
  }
#endif // PLATFORM_PICO_4
}

void File::copy(const char* srcFilepath, const char* destFilepath)
{
#ifndef PLATFORM_OPENXR_ANDROID
  if (Global::installMode != InstallMode::installed)
    return;

#if defined PLATFORM_ANDROID_SDL || defined PLATFORM_OPENXR_ANDROID
  std::filesystem::copy(srcFilepath[0] == '/' ? std::filesystem::path(srcFilepath) : std::filesystem::path(Global::rootPath.c_str()) / std::filesystem::path(srcFilepath),
    destFilepath[0] == '/' ? std::filesystem::path(destFilepath) : std::filesystem::path(Global::rootPath.c_str()) / std::filesystem::path(destFilepath),
    std::filesystem::copy_options::overwrite_existing);
#else // PLATFORM_ANDROID_SDL
  std::filesystem::copy(std::filesystem::path(srcFilepath), std::filesystem::path(destFilepath), std::filesystem::copy_options::overwrite_existing);
#endif // PLATFORM_ANDROID_SDL
#endif // PLATFORM_OPENXR_ANDROID
}

std::vector<std::string> File::filesInDirectory(const char* filePath)
{
  std::vector<std::string> files;

#ifndef PLATFORM_OPENXR_ANDROID

  if (directoryExists(filePath))
  {
#if defined PLATFORM_ANDROID_SDL || defined PLATFORM_OPENXR_ANDROID
    const std::filesystem::path path(filePath[0] == '/' ? filePath : (Global::rootPath + filePath).c_str());
#else // PLATFORM_ANDROID_SDL

#ifdef PLATFORM_WINDOWS
    const std::wstring wFilepath = String::s2ws(filePath, i32(strlen(filePath)));
    const std::wstring wFilepathExpanded = String::ws_ExpandEnvironmentStrings(wFilepath);
    const std::wstring extendedPath = windowsExtendedPath(wFilepathExpanded);
    const std::filesystem::path path(extendedPath);
#else // PLATFORM_WINDOWS
    const std::filesystem::path path(filePath);
#endif // PLATFORM_WINDOWS

#endif // PLATFORM_ANDROID_SDL

    for (const std::filesystem::path& file : std::filesystem::recursive_directory_iterator(path))
    {
#if defined PLATFORM_ANDROID_SDL || defined PLATFORM_OPENXR_ANDROID
      files.emplace_back(reinterpret_cast<const char*>(file.generic_string().c_str()));
#else // PLATFORM_ANDROID_SDL && !PLATFORM_OPENXR_ANDROID
      const std::u8string u8str = file.generic_u8string();
      files.emplace_back(reinterpret_cast<const char*>(u8str.data()), u8str.size());
#endif // PLATFORM_ANDROID_SDL && !PLATFORM_OPENXR_ANDROID
    }
  }
#else // PLATFORM_OPENXR_ANDROID
  const std::string rootFilepath =
#ifdef __ANDROID__
    filePath[0] == '/' ? filePath : (Global::rootPath + filePath);
#else // __ANDROID__
    filePath;
#endif // __ANDROID__

  DIR* dir = opendir(rootFilepath.c_str());
  if (dir == nullptr) {
    //errorf("opendir %s error %d", path.c_str(), errno);
    return files;
  }
  struct dirent* file;
  while ((file = readdir(dir)) != nullptr) {
    if (strcmp(file->d_name, ".") == 0 || strcmp(file->d_name, "..") == 0) {
      continue;
    }
    if (file->d_type == DT_DIR)
    {
      std::string path_next = filePath;
      //path_next += "/";
      path_next += file->d_name;

      const std::vector<std::string> moreFiles = filesInDirectory(path_next.c_str());
      files.insert(files.end(), moreFiles.begin(), moreFiles.end());
    }
    else
    {
      std::string fileFullName = filePath;
      fileFullName += file->d_name;
      files.push_back(fileFullName);
    }
  }
#endif // PLATFORM_OPENXR_ANDROID

  return files;
}

std::string File::fixInvalidCharsInFilepath(const std::string& filePath)
{
  const u64 size = filePath.size();

  if (size == 0)
    return {};

  std::string s;
  s.reserve(size);

  u64 i = 0;

  for (; i < size; ++i) // trim left
    if (filePath[i] != ' ')
      break;
  for (; i < size; ++i)
  {
    switch (filePath[i])
    {
    default:
      s += filePath[i];
      break;
    case '\\':
      s += "{BACKSLASH}";
      break;
    case '/':
      s += "{SLASH}";
      break;
    case ':':
      s += "{COLON}";
      break;
    case '*':
      s += "{ASTERISK}";
      break;
    case '?':
      s += "{QUESTIONMARK}";
      break;
    case '\"':
      s += "{QUOTE}";
      break;
    case '<':
      s += "{LESSTHAN}";
      break;
    case '>':
      s += "{GREATERTHAN}";
      break;
    case '|':
      s += "{PIPE}";
      break;
    }
  }
  while (s.back() == ' ') // trim right
    s.pop_back();
  if (s.back() == '.') // last character '.' is not allowed on windows
  {
    s.pop_back();
    s += "{DOT}";
  }

  return s;
}

std::string File::uniqueFilepathInDirectory(const std::string& directory, const std::string& extension)
{
  std::string filePath = directory + "YYYY-MM-DD_HH-MM-SS" + extension;

  const time_t t = time(nullptr);
  const tm* tm = localtime(&t);

  sprintf(&filePath[directory.size()], "%4d-%02d-%02d_%02d-%02d-%02d%s", tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec, extension.c_str());

  if (!File::exists(filePath.c_str()))
    return filePath;

  filePath.resize(filePath.size() + 3);
  for (i32 i = 0; i < 100; ++i)
  {
    sprintf(&filePath[directory.size() + 16], "_%02d%s", i, extension.c_str());
    if (!File::exists(filePath.c_str()))
      return filePath;
  }

  ASSERT(false);
  return {};
}

//#ifndef PLATFORM_EMSCRIPTEN
FILE* File::fopen_u8(const char* filePath, const char* mode)
{
#if defined(PLATFORM_WINDOWS)
  return _wfopen(String::s2ws(filePath, i32(strlen(filePath))).c_str(), String::s2ws(mode, i32(strlen(mode))).c_str());
#elif defined(__ANDROID__)
  return fopen(filePath[0] == '/' ? filePath : (Global::rootPath + filePath).c_str(), mode);
#else
  return fopen(filePath, mode);
#endif
}

std::vector<u8> File::read(const char* filePath)
{
  DEBUG_PRINT("%s\n", filePath);

  FILE* file = fopen_u8(filePath, "rb");
  ASSERT(file != nullptr); // if this fails and strerror(errno); returns Permission denied on android then the user needs to be asked for runtime permissions

  fseek(file, 0, SEEK_END);
  const i32 lSize = ftell(file);
  rewind(file);

  std::vector<u8> fileData(lSize);

  fread(fileData.data(), lSize, 1, file);

  fclose(file);

  return fileData;
}

void File::write(const char* filePath, const void* content, size_t len)
{
  FILE* file = fopen_u8(filePath, "wb");
  ASSERT(file != nullptr);

  fwrite(content, len, 1, file);

  fclose(file);
}
//#endif // PLATFORM_EMSCRIPTEN

std::vector<std::string> File::openFileDialog(const wchar_t* filter, const bool allowMultiSelect)
{
#ifndef PLATFORM_PICO_4 // sadly the openFileDialog will crash on Pico4

#ifndef __ANDROID__
#ifdef PLATFORM_WINDOWS
#ifndef SHR3D_MINIMAL_WIN32
  wchar_t szFile[4096];
  OPENFILENAME ofn{};
  ofn.lStructSize = sizeof(ofn);
  ofn.hwndOwner = nullptr;
  ofn.lpstrFile = szFile;
  ofn.lpstrFile[0] = '\0';
  ofn.nMaxFile = sizeof(szFile);
  ofn.lpstrFilter = filter;
  ofn.nFilterIndex = 1;
  ofn.lpstrFileTitle = nullptr;
  ofn.nMaxFileTitle = 0;
  ofn.lpstrInitialDir = nullptr;
  ofn.Flags = OFN_HIDEREADONLY | OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_NOCHANGEDIR;
  if (allowMultiSelect)
    ofn.Flags |= OFN_ALLOWMULTISELECT;

  std::vector<std::string> filePaths;

  if (GetOpenFileNameW(&ofn))
  {
    const wchar_t* p = ofn.lpstrFile;

    std::string dirPathOrSingleFilepath = String::ws2s(p, i32(wcslen(p)));
    std::replace(dirPathOrSingleFilepath.begin(), dirPathOrSingleFilepath.end(), '\\', '/');

    p += dirPathOrSingleFilepath.size() + 1;

    if (*p == L'\0')
    {
      // single file
      ASSERT(File::exists(dirPathOrSingleFilepath.c_str()));
      filePaths.push_back(dirPathOrSingleFilepath);
    }
    else
    {
      // multiple files
      dirPathOrSingleFilepath += '/';
      while (*p != L'\0')
      {
        const i32 length = i32(wcslen(p));
        const std::string filepath = dirPathOrSingleFilepath + String::ws2s(p, length);
        ASSERT(File::exists(filepath.c_str()));
        filePaths.push_back(filepath);
        p += length + 1;
      }
    }
  }

  return filePaths;

#endif // SHR3D_MINIMAL_WIN32
#endif // PLATFORM_WINDOWS
#else // __ANDROID__
  JNIEnv* env = nullptr;
  Global::g_JVM->GetEnv((void**)&env, JNI_VERSION_1_4);

  jclass fileClass = env->FindClass("android/content/Intent");
  jmethodID intendClassConstructorId = env->GetMethodID(fileClass, "<init>", "(Ljava/lang/String;)V");
  jstring intentParam = env->NewStringUTF("android.intent.action.OPEN_DOCUMENT");
  jobject intent = env->NewObject(fileClass, intendClassConstructorId, intentParam);
  env->DeleteLocalRef(intentParam);

  jmethodID intendAddCategoryId = env->GetMethodID(fileClass, "addCategory", "(Ljava/lang/String;)Landroid/content/Intent;");
  jstring intentAddCategoryParam = env->NewStringUTF("android.intent.category.OPENABLE");
  env->CallObjectMethod(intent, intendAddCategoryId, intentAddCategoryParam);
  env->DeleteLocalRef(intentAddCategoryParam);

  jmethodID intendSetTypeId = env->GetMethodID(fileClass, "setType", "(Ljava/lang/String;)Landroid/content/Intent;");
  jstring intentSetTypeParam = env->NewStringUTF("*/*");
  env->CallObjectMethod(intent, intendSetTypeId, intentSetTypeParam);
  env->DeleteLocalRef(intentSetTypeParam);

  jmethodID intentPutExtraBoolId = env->GetMethodID(fileClass, "putExtra", "(Ljava/lang/String;Z)Landroid/content/Intent;");
  jstring intentExtraAllowMultiple = env->NewStringUTF("android.intent.extra.ALLOW_MULTIPLE");
  env->CallObjectMethod(intent, intentPutExtraBoolId, intentExtraAllowMultiple, JNI_TRUE);
  env->DeleteLocalRef(intentExtraAllowMultiple);

  //  jmethodID intendPutExtraId = env->GetMethodID(fileClass, "putExtra", "(Ljava/lang/String;Ljava/lang/String;)Landroid/content/Intent;");
  //  jstring intentPutExtraParam0 = env->NewStringUTF("android.intent.extra.TITLE");
  //  jstring intentPutExtraParam1 = env->NewStringUTF("_p.psarc");
  //  env->CallObjectMethod(intent, intendPutExtraId, intentPutExtraParam0, intentPutExtraParam1);
  //  env->DeleteLocalRef(intentPutExtraParam0);
  //  env->DeleteLocalRef(intentPutExtraParam1);

  jobject activity = (jobject)Global::androidActivity;
  jclass activityClass(env->GetObjectClass(activity));
  jmethodID startActivityForResultMethodId = env->GetMethodID(activityClass, "startActivityForResult", "(Landroid/content/Intent;I)V");
  env->CallVoidMethod(activity, startActivityForResultMethodId, intent, 1);

  env->DeleteLocalRef(intent);
  env->DeleteLocalRef(activityClass);
  env->DeleteLocalRef(fileClass);

  return {};

#endif // __ANDROID__
#endif // PLATFORM_PICO_4

  unreachable();
}

std::string File::openDirectoryDialog()
{
#if defined(PLATFORM_WINDOWS) && !defined(SHR3D_MINIMAL_WIN32)
  BROWSEINFO bi{};
  LPITEMIDLIST itemIdList = SHBrowseForFolder(&bi);
  if (itemIdList != 0)
  {
    wchar_t folderPath[512];
    const bool res = SHGetPathFromIDList(itemIdList, folderPath);
    ASSERT(res);

    CoTaskMemFree(itemIdList);
    std::string u8String = String::ws2s(folderPath, i32(wcslen(folderPath)));
    std::replace(u8String.begin(), u8String.end(), '\\', '/');
    return u8String;
  }
  return {};
#else // PLATFORM_WINDOWS && !SHR3D_MINIMAL_WIN32
  unreachable(); // TODO: not implemented
#endif // PLATFORM_WINDOWS && !SHR3D_MINIMAL_WIN32
}

void File::shellExecute(const char* filePathOrUrl)
{
#if defined(PLATFORM_WINDOWS) && !defined(SHR3D_MINIMAL_WIN32)
  ShellExecuteW(NULL, L"open", String::s2ws(filePathOrUrl, i32(strlen(filePathOrUrl))).c_str(), NULL, NULL, SW_SHOWDEFAULT);
#elif defined(__ANDROID__)
  JNIEnv* env = nullptr;
  Global::g_JVM->GetEnv((void**)&env, JNI_VERSION_1_4);

  jclass intentClass = env->FindClass("android/content/Intent");
  jmethodID intentConstructor = env->GetMethodID(intentClass, "<init>", "(Ljava/lang/String;)V");
  jstring actionView = env->NewStringUTF("android.intent.action.VIEW");
  jobject intent = env->NewObject(intentClass, intentConstructor, actionView);
  env->DeleteLocalRef(actionView);

  jmethodID setDataMethod = env->GetMethodID(intentClass, "setData", "(Landroid/net/Uri;)Landroid/content/Intent;");
  jclass uriClass = env->FindClass("android/net/Uri");
  jmethodID uriParseMethod = env->GetStaticMethodID(uriClass, "parse", "(Ljava/lang/String;)Landroid/net/Uri;");
  jstring urlString = env->NewStringUTF(filePathOrUrl);
  jobject uri = env->CallStaticObjectMethod(uriClass, uriParseMethod, urlString);
  env->DeleteLocalRef(urlString);
  env->DeleteLocalRef(uriClass);
  env->DeleteLocalRef(intentClass);

  env->CallObjectMethod(intent, setDataMethod, uri);
  env->DeleteLocalRef(uri);

  jclass contextClass = env->FindClass("android/content/Context");
  jmethodID startActivityMethod = env->GetMethodID(contextClass, "startActivity", "(Landroid/content/Intent;)V");
  jobject activity = (jobject)Global::androidActivity;
  env->CallVoidMethod(activity, startActivityMethod, intent);

  env->DeleteLocalRef(contextClass);
  env->DeleteLocalRef(intent);
#else // PLATFORM_WINDOWS && !SHR3D_MINIMAL_WIN32
  unreachable(); // TODO: not implemented
#endif // PLATFORM_WINDOWS && !SHR3D_MINIMAL_WIN32
}

void File::createPathDirectories()
{
#ifdef SHR3D_SHRED_OR_PSARC
  File::createDirectories(Settings::pathCache.c_str());
#endif // SHR3D_SHRED_OR_PSARC

#ifdef SHR3D_PSARC
  File::createDirectories(Settings::pathPsarc.c_str());
#endif // SHR3D_PSARC

#ifdef SHR3D_SHRED
  File::createDirectories(Settings::pathShred.c_str());
#endif // SHR3D_SHRED

#ifdef SHR3D_ENVIRONMENT_MILK
  File::createDirectories(Settings::pathMilk.c_str());
#endif // SHR3D_ENVIRONMENT_MILK

#ifdef SHR3D_SFX_CORE_NEURALAMPMODELER
  File::createDirectories(Settings::pathNam.c_str());
#endif // SHR3D_SFX_CORE_NEURALAMPMODELER

#ifdef SHR3D_SFX_PLUGIN_CLAP
  File::createDirectories(Settings::pathClap.c_str());
#endif // SHR3D_SFX_PLUGIN_CLAP

#ifdef SHR3D_SFX_PLUGIN_VST
  File::createDirectories(Settings::pathVst.c_str());
#endif // SHR3D_SFX_PLUGIN_VST

#ifdef SHR3D_SFX_PLUGIN_VST3
  File::createDirectories(Settings::pathVst3.c_str());
#endif // SHR3D_SFX_PLUGIN_VST3

#ifdef SHR3D_ENVIRONMENT_SKYBOX
  File::createDirectories(Settings::pathSkybox.c_str());
#endif // SHR3D_ENVIRONMENT_SKYBOX

#ifdef SHR3D_ENVIRONMENT_STAGE
  File::createDirectories(Settings::pathStage.c_str());
#endif // SHR3D_ENVIRONMENT_STAGE

#ifdef SHR3D_RECORDER
  File::createDirectories(Settings::pathRecorder.c_str());
#endif // SHR3D_RECORDER
}

#ifdef PLATFORM_EMSCRIPTEN
File::Type File::type_cacheIpfsCid(const char* filePath, const u8* fileData4CC)
{ // on ipfs we might not have a filename

  { // detect based on filename
    const char* extension = File::extension(filePath);
    if (strcmp(extension, ".shred") == 0)
      return File::Type::shred;
    if (strcmp(extension, ".psarc") == 0)
      return File::Type::psarc;
    if (strcmp(extension, ".tsv") == 0)
      return File::Type::tsv;
  }

  {
    File::Type fileType;

    if (memcmp(fileData4CC, "PK", 2) == 0)
    {
      DEBUG_PRINT("File::type4CC: .shred file detected.\n");
      fileType = File::Type::shred;
    }
    else if (memcmp(fileData4CC, "PSAR", 4) == 0)
    {
      DEBUG_PRINT("File::type4CC: .psarc file detected.\n");
      fileType = File::Type::psarc;
    }
    else if (memcmp(&fileData4CC[1], "PNG", 3) == 0)
    {
      DEBUG_PRINT("File::type4CC: .png file detected.\n");
      fileType = File::Type::png;
    }
    else if (memcmp(fileData4CC, "DDS", 3) == 0)
    {
      DEBUG_PRINT("File::type4CC: .dds file detected.\n");
      fileType = File::Type::dds;
    }
    else
    {
      DEBUG_PRINT("File::type4CC: unknown/urls.txt file detected.\n");
      fileType = File::Type::unknown;
    }

    Global::ipfsFileTypeCache[filePath] = fileType;

    return fileType;
  }
}
#endif // PLATFORM_EMSCRIPTEN

File::Type File::type(const char* filePath)
{
  DEBUG_PRINT("File::type() of %s\n", filePath);

  { // detect based on filename
    const char* extension = File::extension(filePath);
    if (strcmp(extension, ".shred") == 0)
      return File::Type::shred;
    if (strcmp(extension, ".psarc") == 0)
      return File::Type::psarc;
    if (strcmp(extension, ".png") == 0)
      return File::Type::png;
    if (strcmp(extension, ".dds") == 0)
      return File::Type::dds;
    if (strcmp(extension, ".astc") == 0)
      return File::Type::astc;
#ifdef SHR3D_ENVIRONMENT_SKYBOX
    if (strcmp(extension, ".astc6") == 0)
      return File::Type::astc6;
#endif // SHR3D_ENVIRONMENT_SKYBOX
#ifdef SHR3D_ENVIRONMENT_STAGE
    if (strcmp(extension, ".stage") == 0)
      return File::Type::stage;
    if (strcmp(extension, ".stage_a") == 0)
      return File::Type::stage_a;
#endif // SHR3D_ENVIRONMENT_STAGE
  }

#ifdef PLATFORM_EMSCRIPTEN
  ASSERT(Global::ipfsFileTypeCache.contains(filePath));
  return Global::ipfsFileTypeCache.at(filePath);
#else // PLATFORM_EMSCRIPTEN
  DEBUG_PRINT("Unknown file detected %s.\n", filePath);
  return File::Type::unknown;
#endif // PLATFORM_EMSCRIPTEN
}

#ifdef __ANDROID__
extern "C" JNIEXPORT void JNICALL
#ifdef SHR3D_WINDOW_SDL
Java_org_libsdl_app_SDLActivity_openFileDialogResult
#else // SHR3D_WINDOW_SDL
Java_app_shr3d_MainActivity_openFileDialogResult
#endif // SHR3D_WINDOW_SDL
(JNIEnv* env, jclass clazz, jstring path) {
  const char* pathRaw = env->GetStringUTFChars(path, nullptr);
  const std::string filePath = pathRaw;
  env->ReleaseStringUTFChars(path, pathRaw);

  const File::Type fileType = File::type(filePath.c_str());
  switch (fileType)
  {
  case File::Type::psarc:
  case File::Type::shred:
  {
    const std::vector<u8> fileContent = File::read(filePath.c_str());
    Collection::addSongFile(filePath, fileContent.data(), fileContent.size());
  }
  break;
#ifdef SHR3D_ENVIRONMENT_SKYBOX
  case File::Type::astc:
  case File::Type::astc6:
    Global::environmentSkyboxNames.clear();
    Skybox::init();
    break;
#endif // SHR3D_ENVIRONMENT_SKYBOX
#ifdef SHR3D_ENVIRONMENT_STAGE
  case File::Type::stage_a:
    Global::environmentStageNames.clear();
    Stage::init();
    break;
#endif // SHR3D_ENVIRONMENT_STAGE
  default:
    break;
  }
}
#endif // __ANDROID__
