/*=============================================================================
   Copyright (c) 2014-2024 Joel de Guzman. All rights reserved.
   Copyright (c) 2024 AshRa al-Baschir, Shr3D Industries. All rights reserved.

   Distributed under the MIT License [ https://opensource.org/licenses/MIT ]
=============================================================================*/

#include "hexfin.h"

#include <assert.h>
#include <limits.h>
#include <math.h>
#include <string.h>

#ifdef _MSC_VER
#include <intrin.h>
#endif // _MSC_VER

#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define MIN(a,b) ((a) < (b) ? (a) : (b))

static const double pi = 3.1415926535897932384626433832795;
static const double hysteresis = -40.0;
static const float periodicity_diff_factor = 0.008f;
static const float onset_threshold_db = -28.0f;
static const float release_threshold_db = -60.0f;
static const float makeup_gain = 4.0f;
static const float max_deviation = 0.90f;
static const float min_periodicity = 0.8f;
static const float pulse_threshold = 0.6f;
static const float harmonic_periodicity_factor = 16.0f;
static const float compressor_threshold = -18.0f;
static const float compressor_ratio = 1.0f / 4.0f;
static const float pulse_height_diff = 0.8f;
static const float pulse_width_diff = 0.85f;

static const float inv_db_table[] =
{
   1.000000000000000f, 1.011579454259899f, 1.023292992280754f, 1.035142166679344f, 1.047128548050900f, 1.059253725177289f, 1.071519305237606f, 1.083926914021204f,
   1.096478196143185f, 1.109174815262401f, 1.122018454301963f, 1.135010815672315f, 1.148153621496883f, 1.161448613840343f, 1.174897554939530f, 1.188502227437018f,
   1.202264434617413f, 1.216186000646368f, 1.230268770812382f, 1.244514611771385f, 1.258925411794167f, 1.273503081016662f, 1.288249551693134f, 1.303166778452299f,
   1.318256738556407f, 1.333521432163324f, 1.348962882591654f, 1.364583136588924f, 1.380384264602885f, 1.396368361055938f, 1.412537544622754f, 1.428893958511103f,
   1.445439770745927f, 1.462177174456718f, 1.479108388168207f, 1.496235656094433f, 1.513561248436208f, 1.531087461682030f, 1.548816618912481f, 1.566751070108149f,
   1.584893192461114f, 1.603245390690041f, 1.621810097358930f, 1.640589773199539f, 1.659586907437561f, 1.678804018122560f, 1.698243652461744f, 1.717908387157588f,
   1.737800828749375f, 1.757923613958693f, 1.778279410038923f, 1.798870915128788f, 1.819700858609983f, 1.840772001468956f, 1.862087136662868f, 1.883649089489801f,
   1.905460717963247f, 1.927524913190936f, 1.949844599758045f, 1.972422736114854f, 1.995262314968880f, 2.018366363681561f, 2.041737944669529f, 2.065380155810530f,
   2.089296130854040f, 2.113489039836647f, 2.137962089502232f, 2.162718523727020f, 2.187761623949553f, 2.213094709605638f, 2.238721138568339f, 2.264644307593060f,
   2.290867652767773f, 2.317394649968478f, 2.344228815319922f, 2.371373705661655f, 2.398832919019490f, 2.426610095082415f, 2.454708915685031f, 2.483133105295571f,
   2.511886431509580f, 2.540972705549305f, 2.570395782768863f, 2.600159563165272f, 2.630267991895382f, 2.660725059798810f, 2.691534803926916f, 2.722701308077912f,
   2.754228703338166f, 2.786121168629770f, 2.818382931264454f, 2.851018267503910f, 2.884031503126606f, 2.917427014001167f, 2.951209226666386f, 2.985382618917960f,
   3.019951720402016f, 3.054921113215513f, 3.090295432513591f, 3.126079367123955f, 3.162277660168380f, 3.198895109691398f, 3.235936569296283f, 3.273406948788382f,
   3.311311214825911f, 3.349654391578277f, 3.388441561392026f, 3.427677865464504f, 3.467368504525317f, 3.507518739525680f, 3.548133892335755f, 3.589219346450053f,
   3.630780547701014f, 3.672823004980846f, 3.715352290971725f, 3.758374042884441f, 3.801893963205611f, 3.845917820453535f, 3.890451449942806f, 3.935500754557774f,
   3.981071705534972f, 4.027170343254591f, 4.073802778041127f, 4.120975190973302f, 4.168693834703354f, 4.216965034285822f, 4.265795188015927f, 4.315190768277652f,
   4.365158322401660f, 4.415704473533125f, 4.466835921509632f, 4.518559443749224f, 4.570881896148751f, 4.623810213992603f, 4.677351412871983f, 4.731512589614805f,
   4.786300923226384f, 4.841723675840994f, 4.897788193684462f, 4.954501908047901f, 5.011872336272722f, 5.069907082747043f, 5.128613839913648f, 5.188000389289611f,
   5.248074602497725f, 5.308844442309884f, 5.370317963702527f, 5.432503314924332f, 5.495408738576246f, 5.559042572704036f, 5.623413251903491f, 5.688529308438414f,
   5.754399373371569f, 5.821032177708714f, 5.888436553555890f, 5.956621435290105f, 6.025595860743578f, 6.095368972401692f, 6.165950018614822f, 6.237348354824193f,
   6.309573444801933f, 6.382634861905488f, 6.456542290346556f, 6.531305526474723f, 6.606934480075959f, 6.683439175686146f, 6.760829753919817f, 6.839116472814292f,
   6.918309709189364f, 6.998419960022734f, 7.079457843841379f, 7.161434102129020f, 7.244359600749901f, 7.328245331389041f, 7.413102413009175f, 7.498942093324558f,
   7.585775750291837f, 7.673614893618190f, 7.762471166286917f, 7.852356346100718f, 7.943282347242816f, 8.035261221856173f, 8.128305161640993f, 8.222426499470712f,
   8.317637711026711f, 8.413951416451951f, 8.511380382023766f, 8.609937521846007f, 8.709635899560805f, 8.810488730080140f, 8.912509381337454f, 9.015711376059569f,
   9.120108393559097f, 9.225714271547631f, 9.332543007969910f, 9.440608762859233f, 9.549925860214358f, 9.660508789898133f, 9.772372209558107f, 9.885530946569387f,
   10.000000000000000f, 10.115794542598982f, 10.232929922807541f, 10.351421666793437f, 10.471285480508996f, 10.592537251772887f, 10.715193052376065f, 10.839269140212034f,
   10.964781961431852f, 11.091748152624008f, 11.220184543019636f, 11.350108156723149f, 11.481536214968829f, 11.614486138403427f, 11.748975549395297f, 11.885022274370183f,
   12.022644346174131f, 12.161860006463678f, 12.302687708123818f, 12.445146117713850f, 12.589254117941675f, 12.735030810166617f, 12.882495516931343f, 13.031667784522993f,
   13.182567385564074f, 13.335214321633240f, 13.489628825916533f, 13.645831365889245f, 13.803842646028846f, 13.963683610559377f, 14.125375446227540f, 14.288939585111029f,
   14.454397707459272f, 14.621771744567184f, 14.791083881682072f, 14.962356560944336f, 15.135612484362079f, 15.310874616820303f, 15.488166189124811f, 15.667510701081493f,
   15.848931924611133f, 16.032453906900418f, 16.218100973589298f, 16.405897731995395f, 16.595869074375607f, 16.788040181225607f, 16.982436524617444f, 17.179083871575884f,
   17.378008287493753f, 17.579236139586932f, 17.782794100389228f, 17.988709151287875f, 18.197008586099834f, 18.407720014689556f, 18.620871366628677f, 18.836490894898002f,
   19.054607179632473f, 19.275249131909355f, 19.498445997580454f, 19.724227361148536f, 19.952623149688797f, 20.183663636815606f, 20.417379446695296f, 20.653801558105290f,
   20.892961308540396f, 21.134890398366466f, 21.379620895022324f, 21.627185237270201f, 21.877616239495531f, 22.130947096056374f, 22.387211385683401f, 22.646443075930595f,
   22.908676527677734f, 23.173946499684785f, 23.442288153199225f, 23.713737056616552f, 23.988329190194900f, 24.266100950824157f, 24.547089156850298f, 24.831331052955704f,
   25.118864315095795f, 25.409727055493050f, 25.703957827688633f, 26.001595631652723f, 26.302679918953814f, 26.607250597988099f, 26.915348039269155f, 27.227013080779127f,
   27.542287033381662f, 27.861211686297708f, 28.183829312644534f, 28.510182675039097f, 28.840315031266059f, 29.174270140011675f, 29.512092266663856f, 29.853826189179603f,
   30.199517204020161f, 30.549211132155140f, 30.902954325135905f, 31.260793671239558f, 31.622776601683793f, 31.988951096913972f, 32.359365692962825f, 32.734069487883815f,
   33.113112148259113f, 33.496543915782759f, 33.884415613920254f, 34.276778654645028f, 34.673685045253166f, 35.075187395256798f, 35.481338923357548f, 35.892193464500515f,
   36.307805477010142f, 36.728230049808460f, 37.153522909717260f, 37.583740428844415f, 38.018939632056124f, 38.459178204535355f, 38.904514499428068f, 39.355007545577742f,
   39.810717055349734f, 40.271703432545905f, 40.738027780411279f, 41.209751909733022f, 41.686938347033553f, 42.169650342858226f, 42.657951880159253f, 43.151907682776525f,
   43.651583224016584f, 44.157044735331255f, 44.668359215096302f, 45.185594437492242f, 45.708818961487495f, 46.238102139926035f, 46.773514128719810f, 47.315125896148054f,
   47.863009232263828f, 48.417236758409942f, 48.977881936844611f, 49.545019080479030f, 50.118723362727224f, 50.699070827470443f, 51.286138399136483f, 51.880003892896120f,
   52.480746024977257f, 53.088444423098849f, 53.703179637025272f, 54.325033149243332f, 54.954087385762456f, 55.590425727040369f, 56.234132519034908f, 56.885293084384131f,
   57.543993733715695f, 58.210321777087131f, 58.884365535558899f, 59.566214352901035f, 60.255958607435780f, 60.953689724016904f, 61.659500186148222f, 62.373483548241914f,
   63.095734448019329f, 63.826348619054862f, 64.565422903465560f, 65.313055264747234f, 66.069344800759609f, 66.834391756861450f, 67.608297539198190f, 68.391164728142925f,
   69.183097091893657f, 69.984199600227342f, 70.794578438413808f, 71.614341021290201f, 72.443596007499025f, 73.282453313890400f, 74.131024130091774f, 74.989420933245583f,
   75.857757502918361f, 76.736148936181891f, 77.624711662869160f, 78.523563461007186f, 79.432823472428140f, 80.352612218561731f, 81.283051616409907f, 82.224264994707127f,
   83.176377110267083f, 84.139514164519511f, 85.113803820237635f, 86.099375218460082f, 87.096358995608057f, 88.104887300801423f, 89.125093813374548f, 90.157113760595706f,
   91.201083935590972f, 92.257142715476334f, 93.325430079699103f, 94.406087628592360f, 95.499258602143584f, 96.605087898981353f, 97.723722095581067f, 98.855309465693907f,
   100.000000000000000f, 101.157945425989823f, 102.329299228075357f, 103.514216667934406f, 104.712854805089961f, 105.925372517728860f, 107.151930523760598f, 108.392691402120391f,
   109.647819614318507f, 110.917481526240095f, 112.201845430196300f, 113.501081567231552f, 114.815362149688283f, 116.144861384034257f, 117.489755493952913f, 118.850222743701892f,
   120.226443461741312f, 121.618600064636794f, 123.026877081238112f, 124.451461177138569f, 125.892541179416753f, 127.350308101666158f, 128.824955169313370f, 130.316677845230004f,
   131.825673855640730f, 133.352143216332394f, 134.896288259165317f, 136.458313658892394f, 138.038426460288520f, 139.636836105593773f, 141.253754462275396f, 142.889395851110208f,
   144.543977074592789f, 146.217717445671838f, 147.910838816820728f, 149.623565609443290f, 151.356124843620876f, 153.108746168203027f, 154.881661891248115f, 156.675107010814855f,
   158.489319246111421f, 160.324539069004175f, 162.181009735892985f, 164.058977319953868f, 165.958690743756136f, 167.880401812256054f, 169.824365246174438f, 171.790838715758753f,
   173.780082874937619f, 175.792361395869307f, 177.827941003892278f, 179.887091512878726f, 181.970085860998267f, 184.077200146895620f, 186.208713666286741f, 188.364908948980002f,
   190.546071796324640f, 192.752491319093650f, 194.984459975804555f, 197.242273611485331f, 199.526231496887874f, 201.836636368156178f, 204.173794466952955f, 206.538015581052917f,
   208.929613085403872f, 211.348903983664769f, 213.796208950223246f, 216.271852372702000f, 218.776162394955179f, 221.309470960563885f, 223.872113856834005f, 226.464430759305969f,
   229.086765276777243f, 231.739464996847971f, 234.422881531992260f, 237.137370566165515f, 239.883291901948979f, 242.661009508241420f, 245.470891568503106f, 248.313310529557043f,
   251.188643150957972f, 254.097270554930390f, 257.039578276886459f, 260.015956316527195f, 263.026799189538167f, 266.072505979880873f, 269.153480392691677f, 272.270130807791247f,
   275.422870333816604f, 278.612116862976961f, 281.838293126445478f, 285.101826750390956f, 288.403150312660557f, 291.742701400116573f, 295.120922666638705f, 298.538261891796026f,
   301.995172040201624f, 305.492111321551249f, 309.029543251359200f, 312.607936712395599f, 316.227766016837961f, 319.889510969139735f, 323.593656929628082f, 327.340694878838292f,
   331.131121482591084f, 334.965439157827575f, 338.844156139202369f, 342.767786546450452f, 346.736850452531655f, 350.751873952567962f, 354.813389233575322f, 358.921934645005365f,
   363.078054770101403f, 367.282300498084624f, 371.535229097172419f, 375.837404288444304f, 380.189396320561286f, 384.591782045353511f, 389.045144994280463f, 393.550075455777630f,
   398.107170553497326f, 402.717034325459053f, 407.380277804112609f, 412.097519097330405f, 416.869383470335492f, 421.696503428582275f, 426.579518801592542f, 431.519076827764991f,
   436.515832240166105f, 441.570447353312545f, 446.683592150963023f, 451.855944374922160f, 457.088189614875205f, 462.381021399260362f, 467.735141287198132f, 473.151258961480266f,
   478.630092322638518f, 484.172367584099391f, 489.778819368446136f, 495.450190804790054f, 501.187233627272462f, 506.990708274704446f, 512.861383991364846f, 518.800038928960930f,
   524.807460249772816f, 530.884444230988493f, 537.031796370252664f, 543.250331492433020f, 549.540873857624774f, 555.904257270403718f, 562.341325190349039f, 568.852930843841364f,
   575.439937337156607f, 582.103217770871538f, 588.843655355589021f, 595.662143529010336f, 602.559586074357526f, 609.536897240169310f, 616.595001861482160f, 623.734835482419157f,
   630.957344480193001f, 638.263486190548974f, 645.654229034655600f, 653.130552647472314f, 660.693448007595748f, 668.343917568614870f, 676.082975391981904f, 683.911647281429282f,
   691.830970918936259f, 699.841996002273845f, 707.945784384138051f, 716.143410212902040f, 724.435960074989907f, 732.824533138904485f, 741.310241300917710f, 749.894209332455830f,
   758.577575029183549f, 767.361489361818599f, 776.247116628691970f, 785.235634610071770f, 794.328234724281288f, 803.526122185616941f, 812.830516164099549f, 822.242649947071186f,
   831.763771102670830f, 841.395141645194713f, 851.138038202376833f, 860.993752184600794f, 870.963589956080568f, 881.048873008013743f, 891.250938133745876f, 901.571137605957006f,
   912.010839355909638f, 922.571427154762887f, 933.254300796991515f, 944.060876285923541f, 954.992586021435955f, 966.050878989813100f, 977.237220955811154f, 988.553094656939038f,
   1000.000000000000000f, 1011.579454259898284f, 1023.292992280753651f, 1035.142166679344200f, 1047.128548050899553f, 1059.253725177288743f, 1071.519305237605977f, 1083.926914021203856f,
   1096.478196143185187f, 1109.174815262400898f, 1122.018454301962947f, 1135.010815672315402f, 1148.153621496882806f, 1161.448613840342659f, 1174.897554939529073f, 1188.502227437018973f,
   1202.264434617413144f, 1216.186000646367802f, 1230.268770812381035f, 1244.514611771385717f, 1258.925411794167530f, 1273.503081016661554f, 1288.249551693133526f, 1303.166778452299923f,
   1318.256738556407527f, 1333.521432163324107f, 1348.962882591653170f, 1364.583136588923935f, 1380.384264602885196f, 1396.368361055937612f, 1412.537544622754012f, 1428.893958511102255f,
   1445.439770745927945f, 1462.177174456718376f, 1479.108388168207284f, 1496.235656094432898f, 1513.561248436208643f, 1531.087461682030380f, 1548.816618912481090f, 1566.751070108148497f,
   1584.893192461114040f, 1603.245390690041859f, 1621.810097358929852f, 1640.589773199538740f, 1659.586907437561422f, 1678.804018122560592f, 1698.243652461744205f, 1717.908387157587640f,
   1737.800828749376251f, 1757.923613958693068f, 1778.279410038922833f, 1798.870915128787374f, 1819.700858609982561f, 1840.772001468956432f, 1862.087136662867579f, 1883.649089489800190f,
   1905.460717963246225f, 1927.524913190936559f, 1949.844599758045433f, 1972.422736114853478f, 1995.262314968878854f, 2018.366363681561779f, 2041.737944669529497f, 2065.380155810529232f,
   2089.296130854038893f, 2113.489039836647862f, 2137.962089502232629f, 2162.718523727020056f, 2187.761623949551904f, 2213.094709605638855f, 2238.721138568339939f, 2264.644307593059693f,
   2290.867652767772597f, 2317.394649968479825f, 2344.228815319922887f, 2371.373705661655094f, 2398.832919019489964f, 2426.610095082414318f, 2454.708915685031116f, 2483.133105295570203f,
   2511.886431509579779f, 2540.972705549304010f, 2570.395782768864592f, 2600.159563165272175f, 2630.267991895381329f, 2660.725059798808616f, 2691.534803926916538f, 2722.701308077912927f,
   2754.228703338166270f, 2786.121168629769272f, 2818.382931264454783f, 2851.018267503909556f, 2884.031503126605912f, 2917.427014001165844f, 2951.209226666387167f, 2985.382618917960372f,
   3019.951720402016235f, 3054.921113215512378f, 3090.295432513592004f, 3126.079367123955763f, 3162.277660168379498f, 3198.895109691397010f, 3235.936569296281050f, 3273.406948788383033f,
   3311.311214825911065f, 3349.654391578275863f, 3388.441561392024141f, 3427.677865464504521f, 3467.368504525316894f, 3507.518739525679393f, 3548.133892335752989f, 3589.219346450053308f,
   3630.780547701013802f, 3672.823004980846235f, 3715.352290971723960f, 3758.374042884443043f, 3801.893963205612636f, 3845.917820453535569f, 3890.451449942804629f, 3935.500754557776418f,
   3981.071705534973262f, 4027.170343254590989f, 4073.802778041126203f, 4120.975190973304052f, 4168.693834703354696f, 4216.965034285822185f, 4265.795188015925305f, 4315.190768277650022f,
   4365.158322401661280f, 4415.704473533125565f, 4466.835921509630680f, 4518.559443749221828f, 4570.881896148751366f, 4623.810213992603167f, 4677.351412871980756f, 4731.512589614802891f,
   4786.300923226384839f, 4841.723675840994474f, 4897.788193684461476f, 4954.501908047900542f, 5011.872336272725079f, 5069.907082747044115f, 5128.613839913648007f, 5188.000389289609302f,
   5248.074602497727938f, 5308.844442309884471f, 5370.317963702526868f, 5432.503314924330880f, 5495.408738576248652f, 5559.042572704037411f, 5623.413251903491073f, 5688.529308438412954f,
   5754.399373371566071f, 5821.032177708715608f, 5888.436553555889986f, 5956.621435290103364f, 6025.595860743575031f, 6095.368972401694009f, 6165.950018614822511f, 6237.348354824191119f,
   6309.573444801930236f, 6382.634861905488833f, 6456.542290346556001f, 6531.305526474722683f, 6606.934480075957254f, 6683.439175686148701f, 6760.829753919818359f, 6839.116472814293047f,
   6918.309709189362366f, 6998.419960022738451f, 7079.457843841380964f, 7161.434102129020175f, 7244.359600749898163f, 7328.245331389044622f, 7413.102413009177326f, 7498.942093324558300f,
   7585.775750291835720f, 7673.614893618185306f, 7762.471166286919470f, 7852.356346100717928f, 7943.282347242813557f, 8035.261221856168959f, 8128.305161640994811f, 8222.426499470711860f,
   8317.637711026709439f, 8413.951416451947807f, 8511.380382023768107f, 8609.937521846008167f, 8709.635899560806138f, 8810.488730080138339f, 8912.509381337458763f, 9015.711376059571194f,
   9120.108393559095930f, 9225.714271547629323f, 9332.543007969914470f, 9440.608762859235867f, 9549.925860214360000f, 9660.508789898130999f, 9772.372209558112445f, 9885.530946569389926f,
   10000.000000000000000f, 10115.794542598981934f, 10232.929922807536059f, 10351.421666793430632f, 10471.285480508984620f, 10592.537251772897434f, 10715.193052376071137f, 10839.269140212038110f,
   10964.781961431850505f, 11091.748152624008071f, 11220.184543019629928f, 11350.108156723143111f, 11481.536214968817148f, 11614.486138403437508f, 11748.975549395303460f, 11885.022274370188825f,
   12022.644346174130987f, 12161.860006463679383f, 12302.687708123810808f, 12445.146117713844433f, 12589.254117941662116f, 12735.030810166628726f, 12882.495516931348902f, 13031.667784523000591f,
   13182.567385564074357f, 13335.214321633240615f, 13489.628825916532151f, 13645.831365889238441f, 13803.842646028837407f, 13963.683610559362933f, 14125.375446227553766f, 14288.939585111036649f,
   14454.397707459280355f, 14621.771744567182395f, 14791.083881682072388f, 14962.356560944328521f, 15135.612484362070973f, 15310.874616820286974f, 15488.166189124827724f, 15667.510701081500883f,
   15848.931924611140857f, 16032.453906900416769f, 16218.100973589298519f, 16405.897731995388312f, 16595.869074375597847f, 16788.040181225587730f, 16982.436524617460236f, 17179.083871575894591f,
   17378.008287493761600f, 17579.236139586930221f, 17782.794100389226514f, 17988.709151287872373f, 18197.008586099826061f, 18407.720014689544769f, 18620.871366628656688f, 18836.490894898019178f,
   19054.607179632483167f, 19275.249131909367861f, 19498.445997580456606f, 19724.227361148532509f, 19952.623149688788544f, 20183.663636815595964f, 20417.379446695274964f, 20653.801558105311415f,
   20892.961308540408936f, 21134.890398366474983f, 21379.620895022322657f, 21627.185237270201469f, 21877.616239495517220f, 22130.947096056363080f, 22387.211385683378467f, 22646.443075930619671f,
   22908.676527677747799f, 23173.946499684796436f, 23442.288153199227963f, 23713.737056616551854f, 23988.329190194897819f, 24266.100950824144093f, 24547.089156850284780f, 24831.331052955676569f,
   25118.864315095823258f, 25409.727055493065563f, 25703.957827688645921f, 26001.595631652722659f, 26302.679918953814195f, 26607.250597988084337f, 26915.348039269138098f, 27227.013080779099255f,
   27542.287033381689980f, 27861.211686297723645f, 28183.829312644549645f, 28510.182675039097376f, 28840.315031266058213f, 29174.270140011660260f, 29512.092266663839837f, 29853.826189179570065f,
   30199.517204020190547f, 30549.211132155152882f, 30902.954325135920953f, 31260.793671239556716f, 31622.776601683792251f, 31988.951096913973743f, 32359.365692962808680f, 32734.069487883793045f,
   33113.112148259075184f, 33496.543915782793192f, 33884.415613920275064f, 34276.778654645044298f, 34673.685045253165299f, 35075.187395256791206f, 35481.338923357528984f, 35892.193464500494883f,
   36307.805477010100731f, 36728.230049808502372f, 37153.522909717277798f, 37583.740428844430426f, 38018.939632056128175f, 38459.178204535353871f, 38904.514499428049021f, 39355.007545577725978f,
   39810.717055349690781f, 40271.703432545946271f, 40738.027780411306594f, 41209.751909733044158f, 41686.938347033552418f, 42169.650342858221848f, 42657.951880159256689f, 43151.907682776502043f,
   43651.583224016561871f, 44157.044735331204720f, 44668.359215096344997f, 45185.594437492261932f, 45708.818961487515480f, 46238.102139926035306f, 46773.514128719813016f, 47315.125896148027095f,
   47863.009232263801096f, 48417.236758409890172f, 48977.881936844663869f, 49545.019080479054537f, 50118.723362727250787f, 50699.070827470444783f, 51286.138399136478256f, 51880.003892896093021f,
   52480.746024977226625f, 53088.444423098793777f, 53703.179637025328702f, 54325.033149243361549f, 54954.087385762482882f, 55590.425727040368656f, 56234.132519034908910f, 56885.293084384131362f,
   57543.993733715666167f, 58210.321777087097871f, 58884.365535558841657f, 59566.214352901093662f, 60255.958607435808517f, 60953.689724016934633f, 61659.500186148223293f, 62373.483548241914832f,
   63095.734448019298725f, 63826.348619054828305f, 64565.422903465492709f, 65313.055264747294132f, 66069.344800759645295f, 66834.391756861485192f, 67608.297539198189043f, 68391.164728142932290f,
   69183.097091893621837f, 69984.199600227308110f, 70794.578438413736876f, 71614.341021290267236f, 72443.596007499058032f, 73282.453313890437130f, 74131.024130091769621f, 74989.420933245579363f,
   75857.757502918364480f, 76736.148936181853060f, 77624.711662869114662f, 78523.563461007099249f, 79432.823472428208333f, 80352.612218561771442f, 81283.051616409953567f, 82224.264994707118603f,
   83176.377110267087119f, 84139.514164519481710f, 85113.803820237590116f, 86099.375218459987082f, 87096.358995608141413f, 88104.887300801463425f, 89125.093813374594902f, 90157.113760595704662f,
   91201.083935590970214f, 92257.142715476278681f, 93325.430079699057387f, 94406.087628592256806f, 95499.258602143687312f, 96605.087898981408216f, 97723.722095581120811f, 98855.309465693906532f,
   100000.000000000000000f, 101157.945425989833893f, 102329.299228075367864f, 103514.216667934306315f, 104712.854805089853471f, 105925.372517728974344f, 107151.930523760704091f, 108392.691402120384737f,
   109647.819614318505046f, 110917.481526240095263f, 112201.845430196292000f, 113501.081567231434747f, 114815.362149688167847f, 116144.861384034375078f, 117489.755493953038240f, 118850.222743701888248f,
   120226.443461741306237f, 121618.600064636790194f, 123026.877081238111714f, 124451.461177138437051f, 125892.541179416613886f, 127350.308101666290895f, 128824.955169313485385f, 130316.677845230005914f,
   131825.673855640750844f, 133352.143216332391603f, 134896.288259165332420f, 136458.313658892380772f, 138038.426460288377712f, 139636.836105593625689f, 141253.754462275537662f, 142889.395851110370131f,
   144543.977074592810823f, 146217.717445671820315f, 147910.838816820731154f, 149623.565609443292487f, 151356.124843620724278f, 153108.746168202866102f, 154881.661891248280881f, 156675.107010815001559f,
   158489.319246111408575f, 160324.539069004182238f, 162181.009735892992467f, 164058.977319953875849f, 165958.690743755956646f, 167880.401812255906407f, 169824.365246174595086f, 171790.838715758931357f,
   173780.082874937623274f, 175792.361395869316766f, 177827.941003892279696f, 179887.091512878745561f, 181970.085860998253338f, 184077.200146895454964f, 186208.713666286552325f, 188364.908948980213609f,
   190546.071796324831666f, 192752.491319093649508f, 194984.459975804551505f, 197242.273611485346919f, 199526.231496887892717f, 201836.636368155974196f, 204173.794466952735092f, 206538.015581053128699f,
   208929.613085404096637f, 211348.903983664757106f, 213796.208950223255670f, 216271.852372702007415f, 218776.162394955172203f, 221309.470960563630797f, 223872.113856833777390f, 226464.430759306182154f,
   229086.765276777470717f, 231739.464996847964358f, 234422.881531992257806f, 237137.370566165540367f, 239883.291901948978193f, 242661.009508241433650f, 245470.891568502847804f, 248313.310529556794791f,
   251188.643150958203478f, 254097.270554930641083f, 257039.578276886459207f, 260015.956316527212039f, 263026.799189538171049f, 266072.505979880865198f, 269153.480392691388261f, 272270.130807791021653f,
   275422.870333816914354f, 278612.116862977214623f, 281838.293126445496455f, 285101.826750390988309f, 288403.150312660553027f, 291742.701400116609875f, 295120.922666638391092f, 298538.261891795729753f,
   301995.172040201898199f, 305492.111321551550645f, 309029.543251359194983f, 312607.936712395574432f, 316227.766016837907955f, 319889.510969139693771f, 323593.656929628108628f, 327340.694878837966826f,
   331131.121482590795495f, 334965.439157827931922f, 338844.156139202765189f, 342767.786546450457536f, 346736.850452531652991f, 350751.873952567926608f, 354813.389233575318940f, 358921.934645004977938f,
   363078.054770101036411f, 367282.300498084980063f, 371535.229097172792535f, 375837.404288444318809f, 380189.396320561238099f, 384591.782045353553258f, 389045.144994280475657f, 393550.075455777230673f,
   398107.170553496922366f, 402717.034325459506363f, 407380.277804113051388f, 412097.519097330397926f, 416869.383470335509628f, 421696.503428582276683f, 426579.518801592581440f, 431519.076827765034977f,
   436515.832240165618714f, 441570.447353312047198f, 446683.592150963493623f, 451855.944374922604766f, 457088.189614875183906f, 462381.021399260323960f, 467735.141287198115606f, 473151.258961480285507f,
   478630.092322638025507f, 484172.367584098887164f, 489778.819368446653243f, 495450.190804790530819f, 501187.233627272478770f, 506990.708274704462383f, 512861.383991364797112f, 518800.038928960973863f,
   524807.460249772295356f, 530884.444230987923220f, 537031.796370253316127f, 543250.331492433557287f, 549540.873857624828815f, 555904.257270403672010f, 562341.325190349132754f, 568852.930843841284513f,
   575439.937337156618014f, 582103.217770871007815f, 588843.655355588416569f, 595662.143529010936618f, 602559.586074358085170f, 609536.897240169346333f, 616595.001861482160166f, 623734.835482419119217f,
   630957.344480192987248f, 638263.486190548283048f, 645654.229034654912539f, 653130.552647472941317f, 660693.448007596423849f, 668343.917568614939228f, 676082.975391981890425f, 683911.647281429264694f,
   691830.970918936305679f, 699841.996002273052000f, 707945.784384137368761f, 716143.410212902701460f, 724435.960074990638532f, 732824.533138904487714f, 741310.241300917696208f, 749894.209332455880940f,
   758577.575029183528386f, 767361.489361818530597f, 776247.116628691204824f, 785235.634610070963390f, 794328.234724282170646f, 803526.122185617685318f, 812830.516164099564776f, 822242.649947071215138f,
   831763.771102670812979f, 841395.141645194729790f, 851138.038202375872061f, 860993.752184599870816f, 870963.589956081472337f, 881048.873008014634252f, 891250.938133745919913f, 901571.137605956988409f,
   912010.839355909614824f, 922571.427154762903228f, 933254.300796990515664f, 944060.876285922597162f, 954992.586021436844021f, 966050.878989814082161f, 977237.220955811208114f, 988553.094656939036213f,
   1000000.000000000000000f
};

struct sub_collector_info
{
  int i1;
  int i2;
  int period;
  float periodicity;
  size_t harmonic;
};

struct sub_collector
{
  struct hexfin_context* ctx;
  float harmonic_threshold;
  float first_period;
  struct sub_collector_info fundamental;
  float period_diff_threshold;
  int range;
};

static inline float fast_log2(float x)
{
  union { float f; uint32_t i; } vx = { x };
  union { uint32_t i; float f; } mx = { (vx.i & 0x007FFFFF) | 0x3f000000 };
  float y = vx.i;
  y *= 1.1920928955078125e-7f;

  return y - 124.22551499f
    - 1.498030302f * mx.f
    - 1.72587999f / (0.3520887068f + mx.f);
}

static inline float fast_log10(float x)
{
  return 0.301029995663981f * fast_log2(x);
}

static inline float lin_to_db(float val)
{
  return 20.0f * fast_log10(val);
}

static float linear_interpolate(const float y1, const float y2, const float mu)
{
  return y1 + mu * (y2 - y1);
}

static float db2a(const float db)
{
  if (db < 0.0f)
    return 1.0f / db2a(-db);

  if (db < 120.0f)
  {
    float db_10 = db * 10.0f;
    unsigned int index = (unsigned int)db_10;
    float v1 = inv_db_table[index];
    float v2 = inv_db_table[index + 1];
    return linear_interpolate(v1, v2, db_10 - index);
  }
  return 1000000.0f;
}

static size_t smallest_pow2(size_t n, size_t one)
{
  return (one < n) ? smallest_pow2(n, one << 1) : one;
}

static float fast_exp3(float x)
{
  return (6.0f + x * (6.0f + x * (3.0f + x))) * 0.16666666f;
}

static inline bool rel_within(float a, float b, float eps)
{
  return fabsf(a - b) <= eps * MAX(fabsf(a), fabsf(b));
}

static uint64_t count_bits(uint64_t i)
{
#if defined(_MSC_VER)
  return _mm_popcnt_u64(i);
#elif defined(__GNUC__)
  return __builtin_popcountll(i);
#else
#error Unsupported compiler
#endif
}

static void ring_buffer_init(struct hx_ring_buffer* self, size_t size)
{
  self->capacity = smallest_pow2(size, 1);
  self->mask = self->capacity - 1;
  self->pos = 0;
  self->data = (struct hx_zero_crossing_info*)malloc(self->capacity * sizeof(struct hx_zero_crossing_info));

  {
    size_t i;
    for (i = 0; i < size; ++i)
    {
      self->data[i].crossing0 = 0.0f;
      self->data[i].crossing1 = 0.0f;
      self->data[i].peak = 0.0f;
      self->data[i].leading_edge = INT_MIN;
      self->data[i].trailing_edge = INT_MIN;
      self->data[i].width = 0.0f;
    }
  }
}
static void ring_buffer_push(struct hx_ring_buffer* self, struct hx_zero_crossing_info* zci)
{
  --self->pos;
  self->pos &= self->mask;
  memcpy(&self->data[self->pos], zci, sizeof(struct hx_zero_crossing_info));
}
static struct hx_zero_crossing_info* ring_buffer_at(struct hx_ring_buffer* self, int i)
{
  return &self->data[(self->pos + i) & self->mask];
}
static struct hx_zero_crossing_info* ring_buffer_at2(struct hx_ring_buffer* self, struct hexfin_context* ctx, int i)
{
  return ring_buffer_at(self, (ctx->num_edges - 1) - i);
}

static void bit_set_init(struct hx_bit_set* self, size_t num_bits)
{
  self->num_bits = num_bits;
  self->size = (num_bits + 63) / 64;
  self->data = (uint64_t*)malloc(self->size * sizeof(uint64_t));

  {
    size_t i;
    for (i = 0; i < self->size; ++i)
      self->data[i] = 0;
  }
}
static void bit_set_clear(struct hx_bit_set* self)
{
  memset(self->data, 0, self->size * sizeof(uint64_t));
}
static void bit_set_set(struct hx_bit_set* self, size_t i, size_t n, bool val)
{
  //assert(i < self->num_bits);
  if (i > self->num_bits)
    return;

  if ((i + n) > self->num_bits)
    n = self->num_bits - i;

  uint64_t* p = self->data;
  p += i / 64;

  size_t mod = i & 63;
  if (mod)
  {
    mod = 64 - mod;

    uint64_t mask = ~(UINT64_MAX >> mod);

    if (n < mod)
      mask &= UINT64_MAX >> (mod - n);

    if (val)
      *p |= mask;
    else
      *p &= ~mask;

    if (n < mod)
      return;

    n -= mod;
    ++p;
  }

  if (n >= 64)
  {
    uint64_t val_ = val ? UINT64_MAX : 0;

    do
    {
      *p++ = val_;
      n -= 64;
    } while (n >= 64);
  }

  if (n)
  {
    mod = n & 63;

    uint64_t mask = (1ull << mod) - 1;

    if (val)
      *p |= mask;
    else
      *p &= mask;
  }
}

static void bitstream_acf_init(struct hx_bitstream_acf* self, struct hx_bit_set* bitset)
{
  self->mid_array = MAX(((bitset->size) / 2) - 1, 1);
}
static size_t bitstream_acf_pass(struct hx_bitstream_acf* self, struct hx_bit_set* bitset, size_t period)
{
  const size_t index = period / 64;
  const size_t shift = period % 64;

  const uint64_t* p1 = bitset->data;
  const uint64_t* p2 = bitset->data + index;
  size_t count = 0;

  if (shift == 0)
  {
    size_t i;
    for (i = 0; i != self->mid_array; ++i)
      count += count_bits(*p1++ ^ *p2++);
  }
  else
  {
    const size_t shift2 = 64 - shift;
    {
      size_t i;
      for (i = 0; i != self->mid_array; ++i)
      {
        size_t v = *p2++ >> shift;
        v |= *p2 << shift2;
        count += count_bits(*p1++ ^ v);
      }
    }
  }
  return count;
}

static void peak_envelope_follower_init(struct hx_peak_envelope_follower* self, double release, float sample_rate)
{
  self->release = fast_exp3(-2.0f / (sample_rate * release));
  self->y = 0.0f;
}
static float peak_envelope_follower_pass(struct hx_peak_envelope_follower* self, float s)
{
  if (s > self->y)
    self->y = s;
  else
    self->y = s + self->release * (self->y - s);
  return self->y;
}

static void one_pole_lowpass_init(struct hx_one_pole_lowpass* self, float frequency, float sample_rate)
{
  self->a = 1.0f - fast_exp3(-2.0 * pi * frequency / sample_rate);
  self->y = 0.0f;
}
static float one_pole_lowpass_pass(struct hx_one_pole_lowpass* self, float s)
{
  self->y += self->a * (s - self->y);
  return self->y;
}

static void compressor_init(struct hx_compressor* self, double threshold, float ratio)
{
  self->threshold = threshold;
  self->slope = 1.0f - ratio;
}
static double compressor_pass(struct hx_compressor* self, double env)
{
  if (env <= self->threshold)
    return 0; //db
  return self->slope * (self->threshold - env);
}

static float clip_pass(float clip_max, float s)
{
  return (s > clip_max) ? clip_max : (s < -clip_max) ? -clip_max : s;
}

static float median3f(const float a, const float b, const float c)
{
  return MAX(MIN(a, b), MIN(MAX(a, b), c));
}
static void median3_init(struct hx_median3* self)
{
  self->b = 0.0f;
  self->c = 0.0f;
  self->median = 0.0f;
}
static float median3_pass(struct hx_median3* self, float a)
{
  self->median = median3f(a, self->b, self->c);
  self->c = self->b;
  self->b = a;
  return self->median;
}

static void sub_collector_init(struct sub_collector* self, struct hexfin_context* ctx, float period_diff_threshold, int range)
{
  self->ctx = ctx;
  self->harmonic_threshold = harmonic_periodicity_factor * 2 / ctx->window_size;
  self->period_diff_threshold = period_diff_threshold;
  self->range = range;

  self->fundamental.i1 = -1;
  self->fundamental.i2 = -1;
  self->fundamental.period = -1;
  self->fundamental.periodicity = 0.0f;
  //self->fundamental.harmonic = 0;
}
static bool similar(const struct hx_zero_crossing_info* first, const struct hx_zero_crossing_info* next)
{
  return rel_within(first->peak, next->peak, 1.0f - pulse_height_diff) && rel_within(first->width, next->width, 1.0f - pulse_width_diff);
}
static float fractional_period(const struct hx_zero_crossing_info* first, const struct hx_zero_crossing_info* next)
{
  assert(first->leading_edge <= next->leading_edge);

  const float prev1 = first->crossing0;
  const float curr1 = first->crossing1;
  const float dy1 = curr1 - prev1;
  const float dx1 = -prev1 / dy1;

  const float prev2 = next->crossing0;
  const float curr2 = next->crossing1;
  const float dy2 = curr2 - prev2;
  const float dx2 = -prev2 / dy2;

  const int result = next->leading_edge - first->leading_edge;
  return result + (dx2 - dx1);
}
static float period_of(struct hexfin_context* ctx, struct sub_collector_info* x)
{
  const struct hx_zero_crossing_info* first = ring_buffer_at2(&ctx->info, ctx, x->i1);
  const struct hx_zero_crossing_info* next = ring_buffer_at2(&ctx->info, ctx, x->i2);
  return fractional_period(first, next);
}
static void save(struct hexfin_context* ctx, struct sub_collector* self, struct sub_collector_info* incoming)
{
  self->fundamental = *incoming;
  self->fundamental.harmonic = 1;
  self->first_period = period_of(ctx, &self->fundamental);
}
static bool try_sub_harmonic(struct hexfin_context* ctx, struct sub_collector* self, size_t harmonic, struct sub_collector_info* incoming)
{
  int incoming_period = incoming->period / harmonic;
  int current_period = self->fundamental.period;
  if (abs(incoming_period - current_period) < self->period_diff_threshold)
  {
    if (incoming->periodicity > self->fundamental.periodicity && harmonic != self->fundamental.harmonic)
    {
      const float period_diff = fabsf(incoming->periodicity - self->fundamental.periodicity);

      if (period_diff <= self->harmonic_threshold)
      {
        self->fundamental.i1 = incoming->i1;
        self->fundamental.i2 = incoming->i2;
        self->fundamental.periodicity = incoming->periodicity;
        self->fundamental.harmonic = harmonic;
      }
      else
        save(ctx, self, incoming);
    }
    return true;
  }
  return false;
}
static bool process_harmonics(struct hexfin_context* ctx, struct sub_collector* self, struct sub_collector_info* incoming)
{
  if (incoming->period < self->first_period)
    return false;

  int multiple = MAX(1.0f, roundf(period_of(ctx, incoming) / self->first_period));
  return try_sub_harmonic(ctx, self, MIN(self->range, multiple), incoming);
}
static void sub_collector_pass(struct sub_collector* self, struct sub_collector_info* incoming)
{
  if (self->fundamental.period == -1.0f)
    save(self->ctx, self, incoming);
  else if (process_harmonics(self->ctx, self, incoming))
    return;
  else if (incoming->periodicity > self->fundamental.periodicity)
    save(self->ctx, self, incoming);
}

struct hexfin_context hexfin_create_context(const float sample_rate, const float lowest_possible_frequency, const float highest_possible_frequency)
{
  struct hexfin_context ctx;

  ctx.lowest_frequency = lowest_possible_frequency;
  ctx.highest_frequency = highest_possible_frequency;

  ctx.hysteresis = -db2a(hysteresis);

  const float lowest_frequency_period = 1.0f / lowest_possible_frequency;
  unsigned int window = lowest_frequency_period * 2.0f * sample_rate;
  // adjust window size
  ctx.window_size = (window + 63) / 64 * 64;
  assert(ctx.window_size >= 2);

  ring_buffer_init(&ctx.info, ctx.window_size / 2);

  const float highest_frequency_period = 1.0f / highest_possible_frequency;

  ctx.min_period = highest_frequency_period * sample_rate;
  ctx.range = highest_possible_frequency / lowest_possible_frequency;

  bit_set_init(&ctx.bits, ctx.window_size);

  ctx.weight = 2.0 / ctx.window_size;

  ctx.mid_point = ctx.window_size / 2;
  ctx.period_diff_threshold = ctx.mid_point * periodicity_diff_factor;

  ctx.frequency_current = 0.0f;
  ctx.frequency = 0.0f;
  ctx.sample_rate = sample_rate;

  peak_envelope_follower_init(&ctx.env, 0.03, sample_rate); // 30ms
  one_pole_lowpass_init(&ctx.lp, highest_possible_frequency, sample_rate);
  one_pole_lowpass_init(&ctx.lp2, lowest_possible_frequency, sample_rate);
  compressor_init(&ctx.comp, compressor_threshold, compressor_ratio);

  median3_init(&ctx.median);
  median3_init(&ctx.predicted_median);

  ctx.clip_max = 1.0f;
  ctx.onset_threshold = db2a(onset_threshold_db); // db
  ctx.release_threshold = db2a(release_threshold_db); // db
  ctx.threshold = ctx.onset_threshold;
  ctx.loudEnough = false;
  ctx.prev = 0.0f;
  ctx.state = false;
  ctx.num_edges = 0;
  ctx.frame = 0;
  ctx.ready = false;
  ctx.peak_update = 0.0f;
  ctx.peak = 0.0f;
  ctx.edge_mark = 0;
  ctx.predict_edge_mark = 0;
  ctx.predicted_period = -1.0f;
  ctx.fundamental_period = -1.0f;
  ctx.fundamental_periodicity = 0.0f;
  ctx.frames_after_shift = 0;

  return ctx;
}

void hexfin_destroy_context(struct hexfin_context* ctx)
{
  assert(ctx->bits.data != NULL);
  free(ctx->bits.data);
  assert(ctx->info.data != NULL);
  free(ctx->info.data);
}

static void reset(struct hexfin_context* ctx)
{
  ctx->num_edges = 0;
  ctx->state = false;
  ctx->frame = 0;
}

static void shift(struct hexfin_context* ctx, size_t n)
{
  ring_buffer_at(&ctx->info, 0)->leading_edge -= n;
  if (!ctx->state)
    ring_buffer_at(&ctx->info, 0)->trailing_edge -= n;
  size_t i = 1;
  for (; i != ctx->num_edges; ++i)
  {
    ring_buffer_at(&ctx->info, i)->leading_edge -= n;
    int edge = (ring_buffer_at(&ctx->info, i)->trailing_edge -= n);
    if (edge < 0)
      break;
  }
  ctx->num_edges = i;
}

static void update_state(struct hexfin_context* ctx, float s)
{
  if (ctx->ready)
  {
    shift(ctx, ctx->window_size / 2);
    ctx->ready = false;
    ctx->peak = ctx->peak_update;
    ctx->peak_update = 0.0f;
  }

  if (ctx->num_edges >= ctx->info.capacity)
    reset(ctx);

  if (s > 0.0f)
  {
    if (!ctx->state)
    {
      assert(ctx->num_edges < ctx->info.capacity);
      struct hx_zero_crossing_info zci = {
        .crossing0 = ctx->prev,
        .crossing1 = s,
        .peak = s,
        .leading_edge = (int)ctx->frame
      };
      ring_buffer_push(&ctx->info, &zci);
      ++ctx->num_edges;
      ctx->state = 1;
    }
    else
    {
      struct hx_zero_crossing_info* info = ring_buffer_at(&ctx->info, 0);
      info->peak = MAX(s, info->peak);
      if (info->width == 0.0f && s < (info->peak * 0.3))
        info->width = ctx->frame - info->leading_edge;
    }
    if (s > ctx->peak_update)
      ctx->peak_update = s;
  }
  else if (ctx->state && s < ctx->hysteresis)
  {
    ctx->state = 0;
    struct hx_zero_crossing_info* info = ring_buffer_at(&ctx->info, 0);
    info->trailing_edge = ctx->frame;
    if (ctx->peak == 0.0f)
      ctx->peak = ctx->peak_update;
  }

  ctx->prev = s;
}

static bool zero_crossing(struct hexfin_context* ctx, float s)
{
  s += ctx->hysteresis / 2.0;

  if (ctx->num_edges >= ctx->info.capacity)
    reset(ctx);

  if ((ctx->frame == ctx->window_size / 2) && ctx->num_edges == 0)
    reset(ctx);

  update_state(ctx, s);

  if (++ctx->frame >= ctx->window_size && !ctx->state)
  {
    ctx->frame -= ctx->window_size / 2;

    if (ctx->num_edges > 1)
      ctx->ready = true;
    else
      reset(ctx);
  }

  return ctx->state;
}

static void set_bitstream(struct hexfin_context* ctx)
{
  const float threshold = MAX(ctx->peak, ctx->peak_update) * pulse_threshold;

  bit_set_clear(&ctx->bits);
  {
    size_t i;
    for (i = 0; i != ctx->num_edges; ++i)
    {
      struct hx_zero_crossing_info* info = ring_buffer_at2(&ctx->info, ctx, i);
      if (info->peak >= threshold)
      {
        const int pos = MAX(info->leading_edge, 0);
        const int n = info->trailing_edge - pos;
        bit_set_set(&ctx->bits, pos, n, 1);
      }
    }
  }
}

static int autocorrelate2(struct hexfin_context* ctx, struct hx_bitstream_acf* ac, size_t* period, bool first)
{
  size_t count = bitstream_acf_pass(ac, &ctx->bits, *period);
  size_t mid = ac->mid_array * 64;
  size_t start = *period;

  if (first && count == 0)
  {
    if (bitstream_acf_pass(ac, &ctx->bits, *period / 2) == 0)
      return -1;
  }
  else if (*period < 32)
  {
    {
      size_t p;
      for (p = start + 1; p < mid; ++p)
      {
        const size_t c = bitstream_acf_pass(ac, &ctx->bits, p);
        if (c > count)
          break;
        count = c;
        *period = p;
      }
    }
    {
      size_t p;
      for (p = start - 1; p > ctx->min_period; --p)
      {
        size_t c = bitstream_acf_pass(ac, &ctx->bits, p);
        if (c > count)
          break;
        count = c;
        *period = p;
      }
    }
  }

  return count;
}

static void collect_get(struct hexfin_context* ctx, struct sub_collector_info* info, float* fundamental_period, float* fundamental_periodicity)
{
  if (info->period != -1.0f)
  {
    *fundamental_period = period_of(ctx, info) / info->harmonic;
    *fundamental_periodicity = info->periodicity;
  }
  else
  {
    *fundamental_period = -1.0f;
    *fundamental_periodicity = 0.0f;
  }
}

static void autocorrelate3(struct hexfin_context* ctx, struct hx_bitstream_acf* ac, struct sub_collector* collect, float threshold)
{
  {
    size_t i;
    for (i = 0; i != ctx->num_edges - 1; ++i)
    {
      const struct hx_zero_crossing_info* first = ring_buffer_at2(&ctx->info, ctx, i);
      if (first->peak >= threshold)
      {
        size_t j;
        for (j = i + 1; j != ctx->num_edges; ++j)
        {
          const struct hx_zero_crossing_info* next = ring_buffer_at2(&ctx->info, ctx, j);
          if (next->peak >= threshold)
          {
            assert(first->leading_edge <= next->leading_edge);
            size_t period = next->leading_edge - first->leading_edge;
            if (period > ctx->mid_point)
              break;
            if (period >= ctx->min_period)
            {
              int count = autocorrelate2(ctx, ac, &period, collect->fundamental.period == -1);
              if (count == -1)
                return;
              float periodicity = 1.0f - (count * ctx->weight);
              struct sub_collector_info info =
              {
                .i1 = (int)i,
                .i2 = (int)j,
                .period = (int)period,
                .periodicity = periodicity
              };
              sub_collector_pass(collect, &info);
              if (count == 0)
                return;
            }
          }
        }
      }
    }
  }
}

static void autocorrelate(struct hexfin_context* ctx)
{
  assert(ctx->num_edges > 1);

  struct hx_bitstream_acf ac;
  bitstream_acf_init(&ac, &ctx->bits);

  struct sub_collector collect;
  sub_collector_init(&collect, ctx, ctx->period_diff_threshold, ctx->range);

  const float threshold = MAX(ctx->peak, ctx->peak_update) * pulse_threshold;
  autocorrelate3(ctx, &ac, &collect, threshold);

  collect_get(ctx, &collect.fundamental, &ctx->fundamental_period, &ctx->fundamental_periodicity);
}

static bool pitch_detect_pass(struct hexfin_context* ctx, float s)
{
  bool prev = ctx->state;
  bool zc = zero_crossing(ctx, s);

  if (!zc && prev != zc)
  {
    ++ctx->edge_mark;
    ctx->predicted_period = -1.0f;
  }

  if (ctx->frame == 0)
  {
    ctx->fundamental_period = -1.0f;
    ctx->fundamental_periodicity = 0.0f;
  }

  if (ctx->ready)
  {
    set_bitstream(ctx);
    autocorrelate(ctx);
    return true;
  }

  return false;
}

static float calculate_frequency(struct hexfin_context* ctx)
{
  if (ctx->fundamental_period != -1.0f)
    return ctx->sample_rate / ctx->fundamental_period;
  return 0.0f;
}

static float bias2(struct hexfin_context* ctx, const float current, const float incoming, bool* shift)
{
  const float error = current / 32;
  const float diff = fabsf(current - incoming);

  if (diff < error)
    return incoming;

  if (ctx->frames_after_shift > 2)
  {
    if (current > incoming)
    {
      const int multiple = roundf(current / incoming);
      if (multiple > 1)
      {
        const float f = incoming * multiple;
        if (fabsf(current - f) < error)
          return f;
      }
    }
    else
    {
      const int multiple = roundf(incoming / current);
      if (multiple > 1)
      {
        const float f = incoming / multiple;
        if (fabsf(current - f) < error)
          return f;
      }
    }
  }

  if (ctx->fundamental_periodicity > min_periodicity)
  {
    *shift = true;
    return incoming;
  }
  return current;
}

static float predict_period(struct hexfin_context* ctx)
{
  if (ctx->predicted_period == -1.0f && ctx->edge_mark != ctx->predict_edge_mark)
  {
    ctx->predict_edge_mark = ctx->edge_mark;
    if (ctx->num_edges > 1)
    {
      const float threshold = MAX(ctx->peak, ctx->peak_update) * pulse_threshold;
      {
        int i;
        for (i = (int)ctx->num_edges - 1; i > 0; --i)
        {
          struct hx_zero_crossing_info* edge2 = ring_buffer_at2(&ctx->info, ctx, i);
          if (edge2->peak >= threshold)
          {
            int j;
            for (j = i - 1; j >= 0; --j)
            {
              struct hx_zero_crossing_info* edge1 = ring_buffer_at2(&ctx->info, ctx, j);
              if (similar(edge1, edge2))
              {
                ctx->predicted_period = fractional_period(edge1, edge2);
                return ctx->predicted_period;
              }
            }
          }
        }
      }
    }
  }
  return ctx->predicted_period;
}


static float predict_frequency(struct hexfin_context* ctx)
{
  const float period = predict_period(ctx);
  if (period < ctx->min_period)
    return 0.0f;

  float f = ctx->sample_rate / period;
  if (ctx->frequency_current != f)
    f = median3_pass(&ctx->predicted_median, f);

  return f;
}

static void bias(struct hexfin_context* ctx, float incoming)
{
  const float current = ctx->frequency_current;
  ++ctx->frames_after_shift;
  bool shift = false;
  const float f = bias2(ctx, current, incoming, &shift);
  if (shift)
  {
    if (ctx->fundamental_periodicity < max_deviation)
    {
      bool shift2 = false;
      const float predicted = predict_frequency(ctx);
      if (predicted > 0.0f)
      {
        const float f2 = bias2(ctx, current, predicted, &shift2);

        if (!shift2)
        {
          ctx->frequency_current = median3_pass(&ctx->median, f2);
        }
        else
        {
          const bool predicted2 = fabsf(current - f) >= fabsf(current - f2);
          ctx->frequency_current = median3_pass(&ctx->median, !predicted2 ? f : f2);
        }
      }
      else
      {
        ctx->frequency_current = median3_pass(&ctx->median, f);
      }
    }
    else
    {
      const float f = median3_pass(&ctx->median, incoming);
      if (f == incoming)
        ctx->frames_after_shift = 0;
      ctx->frequency_current = f;
    }
  }
  else
  {
    ctx->frequency_current = median3_pass(&ctx->median, f);
  }
}

static bool pitch_detect(struct hexfin_context* ctx, float s)
{
  pitch_detect_pass(ctx, s);

  if (ctx->ready)
  {
    if (ctx->frequency_current == 0.0f)
    {
      if (ctx->fundamental_periodicity >= max_deviation)
      {
        const float f = calculate_frequency(ctx);
        if (f > 0.0f)
        {
          median3_pass(&ctx->median, f);
          ctx->frequency_current = f;
          ctx->frames_after_shift = 0;
        }
      }
    }
    else
    {
      if (ctx->fundamental_periodicity < min_periodicity)
        ctx->frames_after_shift = 0;
      const float f = calculate_frequency(ctx);
      if (f > 0.0f)
        bias(ctx, f);
    }
  }

  return ctx->ready;
}

static void reinitalize_context(struct hexfin_context* ctx, float sample_rate, float lowest_frequency, float highest_frequency)
{
  ctx->sample_rate = sample_rate;
  ctx->lowest_frequency = lowest_frequency;
  ctx->highest_frequency = highest_frequency;

  ctx->prev = 0.0f;
  ctx->state = false;
  ctx->num_edges = 0;
  ctx->frame = 0;
  ctx->ready = false;
  ctx->peak_update = 0.0f;
  ctx->peak = 0.0f;

  const float lowest_frequency_period = 1.0f / lowest_frequency;
  unsigned int window = lowest_frequency_period * 2.0f * sample_rate;
  ctx->window_size = (window + 63) / 64 * 64;

  ctx->info.capacity = smallest_pow2(ctx->window_size / 2, 1);
  ctx->info.mask = ctx->info.capacity - 1;
  ctx->info.pos = 0;
  {
    unsigned int i;
    for (i = 0; i < ctx->window_size / 2; ++i)
    {
      ctx->info.data[i].crossing0 = 0.0f;
      ctx->info.data[i].crossing1 = 0.0f;
      ctx->info.data[i].peak = 0.0f;
      ctx->info.data[i].leading_edge = INT_MIN;
      ctx->info.data[i].trailing_edge = INT_MIN;
      ctx->info.data[i].width = 0.0f;
    }
  }

  const float highest_frequency_period = 1.0f / highest_frequency;

  ctx->min_period = highest_frequency_period * sample_rate;
  ctx->range = highest_frequency / lowest_frequency;

  bit_set_init(&ctx->bits, ctx->window_size);

  ctx->bits.num_bits = ctx->window_size;
  ctx->bits.size = (ctx->bits.num_bits + 63) / 64;

  {
    size_t i;
    for (i = 0; i < ctx->bits.size; ++i)
      ctx->bits.data[i] = 0;
  }

  ctx->weight = 2.0 / ctx->window_size;

  ctx->mid_point = ctx->window_size / 2;
  ctx->period_diff_threshold = ctx->mid_point * periodicity_diff_factor;

  ctx->edge_mark = 0;
  ctx->predict_edge_mark = 0;
  ctx->predicted_period = -1.0f;

  ctx->frequency_current = 0.0f;

  ctx->frames_after_shift = 0;

  median3_init(&ctx->median);
  median3_init(&ctx->predicted_median);
}

float hexfin_processBlock(struct hexfin_context* ctx, const float* inBlock, int blockSize, float sample_rate, float lowest_frequency, float highest_frequency)
{
  if (ctx->sample_rate != sample_rate || ctx->lowest_frequency != lowest_frequency || ctx->highest_frequency != highest_frequency)
    reinitalize_context(ctx, sample_rate, lowest_frequency, highest_frequency);

  struct hx_bitstream_acf bacf;
  bitstream_acf_init(&bacf, &ctx->bits);

  const float min_period = (1.0f / highest_frequency) * sample_rate;
  float volume = 0.0f;

  {
    int i;
    for (i = 0; i < blockSize; ++i)
    {
      float s = inBlock[i];
      float s_abs = fabsf(s);
      volume = MAX(volume, s_abs);

      // Bandpass filter
      s = one_pole_lowpass_pass(&ctx->lp, s);
      s -= one_pole_lowpass_pass(&ctx->lp2, s);

      // Envelope
      const float e = peak_envelope_follower_pass(&ctx->env, s_abs);
      const double e_db = lin_to_db(e);

      if (e > ctx->threshold)
      {
        // Compressor + makeup-gain + hard clip
        const float gain = db2a(compressor_pass(&ctx->comp, e_db)) * makeup_gain;
        s = clip_pass(ctx->clip_max, s * gain);
        ctx->threshold = ctx->release_threshold;
        ctx->loudEnough = true;
      }
      else
      {
        s = 0.0f;
        ctx->threshold = ctx->onset_threshold;
        ctx->loudEnough = false;
      }

      const bool ready = pitch_detect(ctx, s);

      if (ready)
      {
        ctx->frequency = ctx->frequency_current;
      }
    }
  }

  ctx->volume = volume;

  if (ctx->loudEnough)
    return ctx->frequency;

  return 0.0f;
}
