#include "milk.h"

/**
* projectM -- Milkdrop-esque visualisation SDK
* Copyright (C)2003-2009 projectM Team
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
* See 'LICENSE.txt' included within this release
*
*/

#ifdef SHR3D_ENVIRONMENT_MILK

#include "data.h"
#include "file.h"
#include "global.h"
#include "helper.h"
#include "opengl.h"
#include "shader.h"
#include "string_.h"

#ifdef SHR3D_SDL
#include "SDL.h"
#endif // SHR3D_SDL

#include <assert.h>
#include <cmath>
#include <cstdlib>
#include <ctype.h>
#include <errno.h>
#include <fstream>
#include <functional>
#include <iostream>
#include <locale>
#include <malloc.h>
#include <map>
#include <math.h>
#include <new>
#include <regex>
#include <set>
#include <sstream>
#include <stack>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>
#include <sys/stat.h>
#include <sys/types.h>
#include <vector>


static const int _maxLineLength = 2048;

namespace M4
{
  static char* mprintf_valist(int size, const char* fmt, va_list args) {
    ASSERT(size > 0);
    char* res = nullptr;
    va_list tmp;

    while (1) {
      res = new char[size];
      if (!res) return nullptr;

      va_copy(tmp, args);
      int len = vsnprintf(res, size, fmt, tmp);
      va_end(tmp);

      if ((len >= 0) && (size >= len + 1)) {
        break;
      }

      delete[] res;

      if (len > -1) {
        size = len + 1;
      }
      else {
        size *= 2;
      }
    }

    return res;
  }

  bool String_Equal(const char* a, const char* b) {
    if (a == b) return true;
    if (a == nullptr || b == nullptr) return false;
    return strcmp(a, b) == 0;
  }

  class Allocator {
  public:
    template <typename T> T* New() {
      return (T*)malloc(sizeof(T));
    }
    template <typename T> T* New(size_t count) {
      return (T*)malloc(sizeof(T) * count);
    }
    template <typename T> void Delete(T* ptr) {
      free((void*)ptr);
    }
    template <typename T> T* Realloc(T* ptr, size_t count) {
      return (T*)realloc(ptr, sizeof(T) * count);
    }
  };

  int String_PrintfArgList(char* buffer, int size, const char* format, va_list args)
  {

    va_list tmp;
    va_copy(tmp, args);

    int n = vsnprintf(buffer, size, format, tmp);

    va_end(tmp);

    if (n < 0 || n > size) return -1;
    return n;
  }

  int String_Printf(char* buffer, int size, const char* format, ...)
  {

    va_list args;
    va_start(args, format);

    int n = String_PrintfArgList(buffer, size, format, args);

    va_end(args);

    return n;
  }

  // Engine/Log.h

  void Log_ErrorArgList(const char* format, va_list args)
  {
#if 1 // @@ Don't we need to do this?
    va_list tmp;
    va_copy(tmp, args);
    vprintf(format, args);
    va_end(tmp);
#else
    vprintf(format, args);
#endif
  }

  void Log_Error(const char* format, ...)
  {
    va_list args;
    va_start(args, format);
    Log_ErrorArgList(format, args);
    va_end(args);
  }

  // Engine/Array.h

  template <typename T>
  void ConstructRange(T* buffer, int new_size, int old_size) {
    for (int i = old_size; i < new_size; i++) {
      new(buffer + i) T; // placement new
    }
  }

  template <typename T>
  void ConstructRange(T* buffer, int new_size, int old_size, const T& val) {
    for (int i = old_size; i < new_size; i++) {
      new(buffer + i) T(val); // placement new
    }
  }

  template <typename T>
  void DestroyRange(T* buffer, int new_size, int old_size) {
    for (int i = new_size; i < old_size; i++) {
      (buffer + i)->~T(); // Explicit call to the destructor
    }
  }


  template <typename T>
  class Array {
  public:
    Array(Allocator* _allocator) : allocator(_allocator), buffer(nullptr), size(0), capacity(0) {}

    void PushBack(const T& val) {
      ASSERT(&val < buffer || &val >= buffer + size);

      int old_size = size;
      int new_size = size + 1;

      SetSize(new_size);

      ConstructRange(buffer, new_size, old_size, val);
    }
    T& PushBackNew() {
      int old_size = size;
      int new_size = size + 1;

      SetSize(new_size);

      ConstructRange(buffer, new_size, old_size);

      return buffer[old_size];
    }
    void Resize(int new_size) {
      int old_size = size;

      DestroyRange(buffer, new_size, old_size);

      SetSize(new_size);

      ConstructRange(buffer, new_size, old_size);
    }

    int GetSize() const { return size; }
    const T& operator[](int i) const { ASSERT(i < size); return buffer[i]; }
    T& operator[](int i) { ASSERT(i < size); return buffer[i]; }

  private:

    // Change array size.
    void SetSize(int new_size) {
      size = new_size;

      if (new_size > capacity) {
        int new_buffer_size;
        if (capacity == 0) {
          // first allocation is exact
          new_buffer_size = new_size;
        }
        else {
          // following allocations grow array by 25%
          new_buffer_size = new_size + (new_size >> 2);
        }

        SetCapacity(new_buffer_size);
      }
    }

    // Change array capacity.
    void SetCapacity(int new_capacity) {
      ASSERT(new_capacity >= size);

      if (new_capacity == 0) {
        // free the buffer.
        if (buffer != nullptr) {
          allocator->Delete<T>(buffer);
          buffer = nullptr;
        }
      }
      else {
        // realloc the buffer
        buffer = allocator->Realloc<T>(buffer, new_capacity);
      }

      capacity = new_capacity;
    }


  private:
    Allocator* allocator; // @@ Do we really have to keep a pointer to this?
    T* buffer;
    int size;
    int capacity;
  };



  // @@ Implement this with a hash table!
  struct StringPool {
    StringPool(Allocator* allocator) : stringArray(allocator) {
    }
    ~StringPool()
    {
      for (int i = 0; i < stringArray.GetSize(); i++) {
        free((void*)stringArray[i]);
        stringArray[i] = nullptr;
      }
    }

    const char* AddString(const char* string)
    {
      for (int i = 0; i < stringArray.GetSize(); i++) {
        if (String_Equal(stringArray[i], string)) return stringArray[i];
      }
#if _MSC_VER
      const char* dup = _strdup(string);
#else
      const char* dup = strdup(string);
#endif
      stringArray.PushBack(dup);
      return dup;
    }

    const char* AddStringFormatList(const char* fmt, va_list args)
    {
      va_list tmp;
      va_copy(tmp, args);
      const char* string = mprintf_valist(256, fmt, tmp);
      va_end(tmp);

      for (int i = 0; i < stringArray.GetSize(); i++) {
        if (String_Equal(stringArray[i], string)) {
          delete[] string;
          return stringArray[i];
        }
      }

      stringArray.PushBack(string);
      return string;
    }

    const char* AddStringFormat(const char* fmt, ...)
    {
      va_list args;
      va_start(args, fmt);
      const char* string = AddStringFormatList(fmt, args);
      va_end(args);

      return string;
    }

    bool GetContainsString(const char* string) const
    {
      for (int i = 0; i < stringArray.GetSize(); i++) {
        if (String_Equal(stringArray[i], string)) return true;
      }
      return false;
    }

    Array<const char*> stringArray;
  };



  int String_FormatFloat(char* buffer, int size, float value) {
    std::ostringstream oss;
    oss.imbue(std::locale("C"));
    oss << value;

    return String_Printf(buffer, size, "%s", oss.str().c_str());
  }

  bool String_EqualNoCase(const char* a, const char* b) {
    if (a == b) return true;
    if (a == nullptr || b == nullptr) return false;
#if _MSC_VER
    return _stricmp(a, b) == 0;
#else
    return strcasecmp(a, b) == 0;
#endif
  }

  static inline double iss_strtod(const char* in, char** end) {
    char* in_var = const_cast<char*>(in);
    double df;
    std::istringstream iss(in);
    iss.imbue(std::locale("C"));
    iss >> df;
    if (iss.fail()) {
      *end = in_var;
      return 0.0;
    }
    if (iss.eof()) {
      *end = in_var + strlen(in);
      return df;
    }

    std::streampos pos = iss.tellg();
    if (iss.fail()) {
      *end = in_var;
      return 0.0;
    }
    *end = in_var + pos;
    return df;
  }

  double String_ToDouble(const char* str, char** endptr) {
    return iss_strtod(str, endptr);
  }

  int String_ToInteger(const char* str, char** endptr) {
    return strtol(str, endptr, 10);
  }

  //int String_ToIntegerHex(const char * str, char ** endptr) {
  //	return strtol(str, endptr, 16);
  //}




} // M4 namespace




//=============================================================================
//
// Render/CodeWriter.h
//
// Created by Max McGuire (max@unknownworlds.com)
// Copyright (c) 2013, Unknown Worlds Entertainment, Inc.
//
//=============================================================================


#if defined(__GNUC__)
#define M4_PRINTF_ATTR(string_index, first_to_check) __attribute__((format(printf, string_index, first_to_check)))
#else
#define M4_PRINTF_ATTR(string_index, first_to_check)
#endif

namespace M4
{

  class Allocator;

  /**
   * This class is used for outputting code. It handles indentation and inserting #line markers
   * to match the desired output line numbers.
   */
  class CodeWriter
  {

  public:
    CodeWriter(bool writeFileNames = true)
    {
      m_currentLine = 1;
      m_currentFileName = nullptr;
      m_spacesPerIndent = 4;
      m_writeLines = false;
      m_writeFileNames = writeFileNames;
    }

    void BeginLine(int indent, const char* fileName = nullptr, int lineNumber = -1)
    {
      if (m_writeLines)
      {
        bool outputLine = false;
        bool outputFile = false;

        // Output a line number pragma if necessary.
        if (fileName != nullptr && m_currentFileName != fileName)
        {
          m_currentFileName = fileName;
          fileName = m_currentFileName;
          outputFile = true;
        }
        if (lineNumber != -1 && m_currentLine != lineNumber)
        {
          m_currentLine = lineNumber;
          outputLine = true;
        }
        if (outputLine || outputFile)
        {
          char buffer[256];
          String_Printf(buffer, sizeof(buffer), "#line %d", lineNumber);
          m_buffer += buffer;
          if (outputFile && m_writeFileNames)
          {
            m_buffer += " \"";
            m_buffer += fileName;
            m_buffer += "\"\n\n";
          }
          else
          {
            m_buffer += "\n\n";
          }
        }
      }

      // Handle the indentation.
      for (int i = 0; i < indent * m_spacesPerIndent; ++i)
      {
        m_buffer += " ";
      }
    }


    M4_PRINTF_ATTR(2, 3) void Write(const char* format, ...)
    {
      va_list args;
      va_start(args, format);

      char buffer[_maxLineLength];
      String_PrintfArgList(buffer, sizeof(buffer), format, args);

      m_buffer += buffer;

      va_end(args);
    }

    void EndLine(const char* text = nullptr)
    {
      if (text != nullptr)
      {
        m_buffer += text;
      }
      m_buffer += "\n";
      ++m_currentLine;
    }

    M4_PRINTF_ATTR(3, 4) void WriteLine(int indent, const char* format, ...)
    {
      va_list args;
      va_start(args, format);

      char buffer[_maxLineLength];

      int result = String_PrintfArgList(buffer, sizeof(buffer), format, args);
      ASSERT(result != -1);
      (void)result;

      for (int i = 0; i < indent * m_spacesPerIndent; ++i)
      {
        m_buffer += " ";
      }
      m_buffer += buffer;

      EndLine();

      va_end(args);
    }

    M4_PRINTF_ATTR(5, 6) void WriteLineTagged(int indent, const char* fileName, int lineNumber, const char* format, ...)
    {
      va_list args;
      va_start(args, format);

      BeginLine(indent, fileName, lineNumber);

      char buffer[_maxLineLength];
      int result = String_PrintfArgList(buffer, sizeof(buffer), format, args);
      ASSERT(result != -1);
      (void)result;

      m_buffer += buffer;

      EndLine();

      va_end(args);
    }

    const char* GetResult() const
    {
      return m_buffer.c_str();
    }

    void Reset()
    {
      m_buffer.clear();
    }

  private:

    std::string     m_buffer;
    int             m_currentLine;
    const char* m_currentFileName;
    int             m_spacesPerIndent;
    bool            m_writeLines;
    bool            m_writeFileNames;

  };

}


namespace M4
{

  enum HLSLNodeType
  {
    HLSLNodeType_Root,
    HLSLNodeType_Declaration,
    HLSLNodeType_Struct,
    HLSLNodeType_StructField,
    HLSLNodeType_Buffer,
    HLSLNodeType_BufferField,
    HLSLNodeType_Function,
    HLSLNodeType_Argument,
    HLSLNodeType_Macro,
    HLSLNodeType_ExpressionStatement,
    HLSLNodeType_Expression,
    HLSLNodeType_ReturnStatement,
    HLSLNodeType_DiscardStatement,
    HLSLNodeType_BreakStatement,
    HLSLNodeType_ContinueStatement,
    HLSLNodeType_IfStatement,
    HLSLNodeType_ForStatement,
    HLSLNodeType_WhileStatement,
    HLSLNodeType_BlockStatement,
    HLSLNodeType_UnaryExpression,
    HLSLNodeType_BinaryExpression,
    HLSLNodeType_ConditionalExpression,
    HLSLNodeType_CastingExpression,
    HLSLNodeType_LiteralExpression,
    HLSLNodeType_IdentifierExpression,
    HLSLNodeType_ConstructorExpression,
    HLSLNodeType_MemberAccess,
    HLSLNodeType_ArrayAccess,
    HLSLNodeType_FunctionCall,
    HLSLNodeType_StateAssignment,
    HLSLNodeType_SamplerState,
    HLSLNodeType_Pass,
    HLSLNodeType_Technique,
    HLSLNodeType_Attribute,
    HLSLNodeType_Pipeline,
    HLSLNodeType_Stage,
  };


  enum HLSLBaseType
  {
    HLSLBaseType_Unknown,
    HLSLBaseType_Void,
    HLSLBaseType_Float,
    HLSLBaseType_FirstNumeric = HLSLBaseType_Float,
    HLSLBaseType_Float2,
    HLSLBaseType_Float3,
    HLSLBaseType_Float4,

    HLSLBaseType_Float2x4,
    HLSLBaseType_Float2x3,
    HLSLBaseType_Float2x2,

    HLSLBaseType_Float3x4,
    HLSLBaseType_Float3x3,
    HLSLBaseType_Float3x2,

    HLSLBaseType_Float4x4,
    HLSLBaseType_Float4x3,
    HLSLBaseType_Float4x2,

    HLSLBaseType_Bool,
    HLSLBaseType_FirstInteger = HLSLBaseType_Bool,
    HLSLBaseType_Bool2,
    HLSLBaseType_Bool3,
    HLSLBaseType_Bool4,
    HLSLBaseType_Int,
    HLSLBaseType_Int2,
    HLSLBaseType_Int3,
    HLSLBaseType_Int4,
    HLSLBaseType_Uint,
    HLSLBaseType_Uint2,
    HLSLBaseType_Uint3,
    HLSLBaseType_Uint4,
    /*HLSLBaseType_Short,   // @@ Separate dimension from Base type, this is getting out of control.
    HLSLBaseType_Short2,
    HLSLBaseType_Short3,
    HLSLBaseType_Short4,
    HLSLBaseType_Ushort,
    HLSLBaseType_Ushort2,
    HLSLBaseType_Ushort3,
    HLSLBaseType_Ushort4,*/
    HLSLBaseType_LastInteger = HLSLBaseType_Uint4,
    HLSLBaseType_LastNumeric = HLSLBaseType_Uint4,
    HLSLBaseType_Texture,
    HLSLBaseType_Sampler,           // @@ use type inference to determine sampler type.
    HLSLBaseType_Sampler2D,
    HLSLBaseType_Sampler3D,
    HLSLBaseType_SamplerCube,
    HLSLBaseType_Sampler2DShadow,
    HLSLBaseType_Sampler2DMS,
    HLSLBaseType_Sampler2DArray,
    HLSLBaseType_UserDefined,       // struct
    HLSLBaseType_Expression,        // type argument for defined() sizeof() and typeof().
    HLSLBaseType_Auto,

    HLSLBaseType_Count,
    HLSLBaseType_NumericCount = HLSLBaseType_LastNumeric - HLSLBaseType_FirstNumeric + 1
  };



  enum NumericType
  {
    NumericType_Float,
    NumericType_Bool,
    NumericType_Int,
    NumericType_Uint,
    NumericType_Count,
    NumericType_NaN,
  };


  struct BaseTypeDescription
  {
    const char* typeName;
    NumericType     numericType;
    int             numComponents;
    int             numDimensions;
    int             height;
    int             binaryOpRank;
  };

  const BaseTypeDescription baseTypeDescriptions[HLSLBaseType_Count] =
  {
      { "unknown type",       NumericType_NaN,        0, 0, 0, -1 },      // HLSLBaseType_Unknown
      { "void",               NumericType_NaN,        0, 0, 0, -1 },      // HLSLBaseType_Void
      { "float",              NumericType_Float,      1, 0, 1,  0 },      // HLSLBaseType_Float
      { "float2",             NumericType_Float,      2, 1, 1,  0 },      // HLSLBaseType_Float2
      { "float3",             NumericType_Float,      3, 1, 1,  0 },      // HLSLBaseType_Float3
      { "float4",             NumericType_Float,      4, 1, 1,  0 },      // HLSLBaseType_Float4

      { "float2x4",			NumericType_Float,		2, 2, 4,  0 },		// HLSLBaseType_Float2x4
      { "float2x3",			NumericType_Float,		2, 2, 3,  0 },		// HLSLBaseType_Float2x3
      { "float2x2",			NumericType_Float,		2, 2, 2,  0 },		// HLSLBaseType_Float2x2

      { "float3x4",           NumericType_Float,      3, 2, 4,  0 },      // HLSLBaseType_Float3x4
      { "float3x3",           NumericType_Float,      3, 2, 3,  0 },      // HLSLBaseType_Float3x3
      { "float3x2",           NumericType_Float,      3, 2, 2,  0 },      // HLSLBaseType_Float3x2

      { "float4x4",           NumericType_Float,      4, 2, 4,  0 },      // HLSLBaseType_Float4x4
      { "float4x3",           NumericType_Float,      4, 2, 3,  0 },      // HLSLBaseType_Float4x3
      { "float4x2",           NumericType_Float,      4, 2, 2,  0 },      // HLSLBaseType_Float4x2

      { "bool",               NumericType_Bool,       1, 0, 1,  4 },      // HLSLBaseType_Bool
      { "bool2",				NumericType_Bool,		2, 1, 1,  4 },      // HLSLBaseType_Bool2
      { "bool3",				NumericType_Bool,		3, 1, 1,  4 },      // HLSLBaseType_Bool3
      { "bool4",				NumericType_Bool,		4, 1, 1,  4 },      // HLSLBaseType_Bool4

      { "int",                NumericType_Int,        1, 0, 1,  3 },      // HLSLBaseType_Int
      { "int2",               NumericType_Int,        2, 1, 1,  3 },      // HLSLBaseType_Int2
      { "int3",               NumericType_Int,        3, 1, 1,  3 },      // HLSLBaseType_Int3
      { "int4",               NumericType_Int,        4, 1, 1,  3 },      // HLSLBaseType_Int4

      { "uint",               NumericType_Uint,       1, 0, 1,  2 },      // HLSLBaseType_Uint
      { "uint2",              NumericType_Uint,       2, 1, 1,  2 },      // HLSLBaseType_Uint2
      { "uint3",              NumericType_Uint,       3, 1, 1,  2 },      // HLSLBaseType_Uint3
      { "uint4",              NumericType_Uint,       4, 1, 1,  2 },      // HLSLBaseType_Uint4

      { "texture",            NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Texture
      { "sampler",            NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler
      { "sampler2D",          NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler2D
      { "sampler3D",          NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler3D
      { "samplerCUBE",        NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_SamplerCube
      { "sampler2DShadow",    NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler2DShadow
      { "sampler2DMS",        NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler2DMS
      { "sampler2DArray",     NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_Sampler2DArray
      { "user defined",       NumericType_NaN,        1, 0, 0, -1 },      // HLSLBaseType_UserDefined
      { "expression",         NumericType_NaN,        1, 0, 0, -1 }       // HLSLBaseType_Expression
  };

  extern const HLSLBaseType ScalarBaseType[HLSLBaseType_Count];

  inline bool IsSamplerType(HLSLBaseType baseType)
  {
    return baseType == HLSLBaseType_Sampler ||
      baseType == HLSLBaseType_Sampler2D ||
      baseType == HLSLBaseType_Sampler3D ||
      baseType == HLSLBaseType_SamplerCube ||
      baseType == HLSLBaseType_Sampler2DShadow ||
      baseType == HLSLBaseType_Sampler2DMS ||
      baseType == HLSLBaseType_Sampler2DArray;
  }

  inline bool IsMatrixType(HLSLBaseType baseType)
  {
    return baseType == HLSLBaseType_Float2x4 || baseType == HLSLBaseType_Float2x3 || baseType == HLSLBaseType_Float2x2 ||
      baseType == HLSLBaseType_Float3x4 || baseType == HLSLBaseType_Float3x3 || baseType == HLSLBaseType_Float3x2 ||
      baseType == HLSLBaseType_Float4x4 || baseType == HLSLBaseType_Float4x3 || baseType == HLSLBaseType_Float4x2;
  }

  inline bool IsScalarType(HLSLBaseType baseType)
  {
    return  baseType == HLSLBaseType_Float ||
      baseType == HLSLBaseType_Bool ||
      baseType == HLSLBaseType_Int ||
      baseType == HLSLBaseType_Uint;
  }

  inline bool IsVectorType(HLSLBaseType baseType)
  {
    return  baseType == HLSLBaseType_Float2 ||
      baseType == HLSLBaseType_Float3 ||
      baseType == HLSLBaseType_Float4 ||
      baseType == HLSLBaseType_Bool2 ||
      baseType == HLSLBaseType_Bool3 ||
      baseType == HLSLBaseType_Bool4 ||
      baseType == HLSLBaseType_Int2 ||
      baseType == HLSLBaseType_Int3 ||
      baseType == HLSLBaseType_Int4 ||
      baseType == HLSLBaseType_Uint2 ||
      baseType == HLSLBaseType_Uint3 ||
      baseType == HLSLBaseType_Uint4;
  }


  enum HLSLBinaryOp
  {
    HLSLBinaryOp_And,
    HLSLBinaryOp_Or,
    HLSLBinaryOp_Add,
    HLSLBinaryOp_Sub,
    HLSLBinaryOp_Mul,
    HLSLBinaryOp_Div,
    HLSLBinaryOp_Mod,
    HLSLBinaryOp_Less,
    HLSLBinaryOp_Greater,
    HLSLBinaryOp_LessEqual,
    HLSLBinaryOp_GreaterEqual,
    HLSLBinaryOp_Equal,
    HLSLBinaryOp_NotEqual,
    HLSLBinaryOp_BitAnd,
    HLSLBinaryOp_BitOr,
    HLSLBinaryOp_BitXor,
    HLSLBinaryOp_Assign,
    HLSLBinaryOp_AddAssign,
    HLSLBinaryOp_SubAssign,
    HLSLBinaryOp_MulAssign,
    HLSLBinaryOp_DivAssign,
  };

  inline bool IsCompareOp(HLSLBinaryOp op)
  {
    return op == HLSLBinaryOp_Less ||
      op == HLSLBinaryOp_Greater ||
      op == HLSLBinaryOp_LessEqual ||
      op == HLSLBinaryOp_GreaterEqual ||
      op == HLSLBinaryOp_Equal ||
      op == HLSLBinaryOp_NotEqual;
  }

  inline bool IsArithmeticOp(HLSLBinaryOp op)
  {
    return op == HLSLBinaryOp_Add ||
      op == HLSLBinaryOp_Sub ||
      op == HLSLBinaryOp_Mul ||
      op == HLSLBinaryOp_Div ||
      op == HLSLBinaryOp_Mod;
  }

  inline bool IsLogicOp(HLSLBinaryOp op)
  {
    return op == HLSLBinaryOp_And ||
      op == HLSLBinaryOp_Or;
  }

  inline bool IsAssignOp(HLSLBinaryOp op)
  {
    return op == HLSLBinaryOp_Assign ||
      op == HLSLBinaryOp_AddAssign ||
      op == HLSLBinaryOp_SubAssign ||
      op == HLSLBinaryOp_MulAssign ||
      op == HLSLBinaryOp_DivAssign;
  }


  enum HLSLUnaryOp
  {
    HLSLUnaryOp_Negative,       // -x
    HLSLUnaryOp_Positive,       // +x
    HLSLUnaryOp_Not,            // !x
    HLSLUnaryOp_PreIncrement,   // ++x
    HLSLUnaryOp_PreDecrement,   // --x
    HLSLUnaryOp_PostIncrement,  // x++
    HLSLUnaryOp_PostDecrement,  // x++
    HLSLUnaryOp_BitNot,         // ~x
  };

  enum HLSLArgumentModifier
  {
    HLSLArgumentModifier_None,
    HLSLArgumentModifier_In,
    HLSLArgumentModifier_Out,
    HLSLArgumentModifier_Inout,
    HLSLArgumentModifier_Uniform,
    HLSLArgumentModifier_Const,
  };

  enum HLSLTypeFlags
  {
    HLSLTypeFlag_None = 0,
    HLSLTypeFlag_Const = 0x01,
    HLSLTypeFlag_Static = 0x02,
    HLSLTypeFlag_Uniform = 0x04,
    //HLSLTypeFlag_Extern = 0x10,
    //HLSLTypeFlag_Volatile = 0x20,
    //HLSLTypeFlag_Shared = 0x40,
    //HLSLTypeFlag_Precise = 0x80,

    HLSLTypeFlag_Input = 0x100,
    HLSLTypeFlag_Output = 0x200,

    // Interpolation modifiers.
    HLSLTypeFlag_Linear = 0x10000,
    HLSLTypeFlag_Centroid = 0x20000,
    HLSLTypeFlag_NoInterpolation = 0x40000,
    HLSLTypeFlag_NoPerspective = 0x80000,
    HLSLTypeFlag_Sample = 0x100000,

    // Misc.
    HLSLTypeFlag_NoPromote = 0x200000,
  };

  enum HLSLAttributeType
  {
    HLSLAttributeType_Unknown,
    HLSLAttributeType_Unroll,
    HLSLAttributeType_Branch,
    HLSLAttributeType_Flatten,
    HLSLAttributeType_NoFastMath,
  };

  enum HLSLAddressSpace
  {
    HLSLAddressSpace_Undefined,
    HLSLAddressSpace_Constant,
    HLSLAddressSpace_Device,
    HLSLAddressSpace_Thread,
    HLSLAddressSpace_Shared,
  };


  struct HLSLNode;
  struct HLSLRoot;
  struct HLSLStatement;
  struct HLSLAttribute;
  struct HLSLDeclaration;
  struct HLSLStruct;
  struct HLSLStructField;
  struct HLSLBuffer;
  struct HLSLFunction;
  struct HLSLArgument;
  struct HLSLExpressionStatement;
  struct HLSLExpression;
  struct HLSLBinaryExpression;
  struct HLSLLiteralExpression;
  struct HLSLIdentifierExpression;
  struct HLSLConstructorExpression;
  struct HLSLFunctionCall;
  struct HLSLArrayAccess;
  struct HLSLAttribute;

  struct HLSLType
  {
    explicit HLSLType(HLSLBaseType _baseType = HLSLBaseType_Unknown)
    {
      baseType = _baseType;
      samplerType = HLSLBaseType_Float;
      typeName = nullptr;
      array = false;
      arraySize = nullptr;
      flags = 0;
      addressSpace = HLSLAddressSpace_Undefined;
    }
    HLSLBaseType        baseType;
    HLSLBaseType        samplerType;    // Half or Float
    const char* typeName;       // For user defined types.
    bool                array;
    HLSLExpression* arraySize;
    int                 flags;
    HLSLAddressSpace    addressSpace;
  };

  inline bool IsSamplerType(const HLSLType& type)
  {
    return IsSamplerType(type.baseType);
  }

  inline bool IsScalarType(const HLSLType& type)
  {
    return IsScalarType(type.baseType);
  }

  inline bool IsVectorType(const HLSLType& type)
  {
    return IsVectorType(type.baseType);
  }


  /** Base class for all nodes in the HLSL AST */
  struct HLSLNode
  {
    HLSLNodeType        nodeType;
    const char* fileName;
    int                 line;
  };

  struct HLSLRoot : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_Root;
    HLSLRoot() { statement = nullptr; }
    HLSLStatement* statement;          // First statement.
  };

  struct HLSLStatement : public HLSLNode
  {
    HLSLStatement()
    {
      nextStatement = nullptr;
      attributes = nullptr;
      hidden = false;
    }
    HLSLStatement* nextStatement;      // Next statement in the block.
    HLSLAttribute* attributes;
    mutable bool        hidden;
  };

  struct HLSLAttribute : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_Attribute;
    HLSLAttribute()
    {
      attributeType = HLSLAttributeType_Unknown;
      argument = nullptr;
      nextAttribute = nullptr;
    }
    HLSLAttributeType   attributeType;
    HLSLExpression* argument;
    HLSLAttribute* nextAttribute;
  };

  struct HLSLDeclaration : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Declaration;
    HLSLDeclaration()
    {
      name = nullptr;
      registerName = nullptr;
      semantic = nullptr;
      nextDeclaration = nullptr;
      assignment = nullptr;
      buffer = nullptr;
    }
    const char* name;
    HLSLType            type;
    const char* registerName;       // @@ Store register index?
    const char* semantic;
    HLSLDeclaration* nextDeclaration;    // If multiple variables declared on a line.
    HLSLExpression* assignment;
    HLSLBuffer* buffer;
  };

  struct HLSLStruct : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Struct;
    HLSLStruct()
    {
      name = nullptr;
      field = nullptr;
    }
    const char* name;
    HLSLStructField* field;              // First field in the structure.
  };

  struct HLSLStructField : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_StructField;
    HLSLStructField()
    {
      name = nullptr;
      semantic = nullptr;
      sv_semantic = nullptr;
      nextField = nullptr;
      hidden = false;
    }
    const char* name;
    HLSLType            type;
    const char* semantic;
    const char* sv_semantic;
    HLSLStructField* nextField;      // Next field in the structure.
    bool                hidden;
  };

  /** A cbuffer or tbuffer declaration. */
  struct HLSLBuffer : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Buffer;
    HLSLBuffer()
    {
      name = nullptr;
      registerName = nullptr;
      field = nullptr;
    }
    const char* name;
    const char* registerName;
    HLSLDeclaration* field;
  };


  /** Function declaration */
  struct HLSLFunction : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Function;
    HLSLFunction()
    {
      name = nullptr;
      semantic = nullptr;
      sv_semantic = nullptr;
      statement = nullptr;
      argument = nullptr;
      numArguments = 0;
      numOutputArguments = 0;
      forward = nullptr;
    }
    const char* name;
    HLSLType            returnType;
    const char* semantic;
    const char* sv_semantic;
    int                 numArguments;
    int                 numOutputArguments;     // Includes out and inout arguments.
    HLSLArgument* argument;
    HLSLStatement* statement;
    HLSLFunction* forward; // Which HLSLFunction this one forward-declares
  };

  /** Declaration of an argument to a function. */
  struct HLSLArgument : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_Argument;
    HLSLArgument()
    {
      name = nullptr;
      modifier = HLSLArgumentModifier_None;
      semantic = nullptr;
      sv_semantic = nullptr;
      defaultValue = nullptr;
      nextArgument = nullptr;
      hidden = false;
    }
    const char* name;
    HLSLArgumentModifier    modifier;
    HLSLType                type;
    const char* semantic;
    const char* sv_semantic;
    HLSLExpression* defaultValue;
    HLSLArgument* nextArgument;
    bool                    hidden;
  };

  /** Macro declaration */
  struct HLSLMacro : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Macro;
    HLSLMacro()
    {
      name = nullptr;
      argument = nullptr;
      numArguments = 0;
      macroAliased = nullptr;
    }
    const char* name;
    HLSLArgument* argument;
    unsigned int        numArguments;
    std::string         value;
    HLSLMacro* macroAliased;
  };

  /** A expression which forms a complete statement. */
  struct HLSLExpressionStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_ExpressionStatement;
    HLSLExpressionStatement()
    {
      expression = nullptr;
    }
    HLSLExpression* expression;
  };

  struct HLSLReturnStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_ReturnStatement;
    HLSLReturnStatement()
    {
      expression = nullptr;
    }
    HLSLExpression* expression;
  };

  struct HLSLDiscardStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_DiscardStatement;
  };

  struct HLSLBreakStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_BreakStatement;
  };

  struct HLSLContinueStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_ContinueStatement;
  };

  struct HLSLIfStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_IfStatement;
    HLSLIfStatement()
    {
      condition = nullptr;
      statement = nullptr;
      elseStatement = nullptr;
      isStatic = false;
    }
    HLSLExpression* condition;
    HLSLStatement* statement;
    HLSLStatement* elseStatement;
    bool                isStatic;
  };

  struct HLSLForStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_ForStatement;
    HLSLForStatement()
    {
      initialization = nullptr;
      condition = nullptr;
      increment = nullptr;
      statement = nullptr;
    }
    HLSLDeclaration* initialization;
    HLSLExpression* initializationWithoutType;
    HLSLExpression* condition;
    HLSLExpression* increment;
    HLSLStatement* statement;
  };

  struct HLSLWhileStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_WhileStatement;
    HLSLWhileStatement()
    {
      condition = nullptr;
      statement = nullptr;
    }
    HLSLExpression* condition;
    HLSLStatement* statement;
  };

  struct HLSLBlockStatement : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_BlockStatement;
    HLSLBlockStatement()
    {
      statement = nullptr;
    }
    HLSLStatement* statement;
  };


  /** Base type for all types of expressions. */
  struct HLSLExpression : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_Expression;
    HLSLExpression()
    {
      nextExpression = nullptr;
    }
    HLSLType            expressionType;
    HLSLExpression* nextExpression; // Used when the expression is part of a list, like in a function call.
  };

  struct HLSLUnaryExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_UnaryExpression;
    HLSLUnaryExpression()
    {
      expression = nullptr;
    }
    HLSLUnaryOp         unaryOp;
    HLSLExpression* expression;
  };

  struct HLSLBinaryExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_BinaryExpression;
    HLSLBinaryExpression()
    {
      expression1 = nullptr;
      expression2 = nullptr;
    }
    HLSLBinaryOp        binaryOp;
    HLSLExpression* expression1;
    HLSLExpression* expression2;
  };

  /** ? : construct */
  struct HLSLConditionalExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_ConditionalExpression;
    HLSLConditionalExpression()
    {
      condition = nullptr;
      trueExpression = nullptr;
      falseExpression = nullptr;
    }
    HLSLExpression* condition;
    HLSLExpression* trueExpression;
    HLSLExpression* falseExpression;
  };

  struct HLSLCastingExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_CastingExpression;
    HLSLCastingExpression()
    {
      expression = nullptr;
    }
    HLSLType            type;
    HLSLExpression* expression;
  };

  /** Float, integer, boolean, etc. literal constant. */
  struct HLSLLiteralExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_LiteralExpression;
    HLSLBaseType        type;   // Note, not all types can be literals.
    union
    {
      bool            bValue;
      float           fValue;
      int             iValue;
    };
  };

  /** An identifier, typically a variable name or structure field name. */
  struct HLSLIdentifierExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_IdentifierExpression;
    HLSLIdentifierExpression()
    {
      name = nullptr;
      global = false;
    }
    const char* name;
    bool                global; // This is a global variable.
  };

  /** float2(1, 2) */
  struct HLSLConstructorExpression : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_ConstructorExpression;
    HLSLConstructorExpression()
    {
      argument = nullptr;
    }
    HLSLType            type;
    HLSLExpression* argument;
  };

  /** object.member **/
  struct HLSLMemberAccess : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_MemberAccess;
    HLSLMemberAccess()
    {
      object = nullptr;
      field = nullptr;
      swizzle = false;
    }
    HLSLExpression* object;
    const char* field;
    bool                swizzle;
  };

  /** array[index] **/
  struct HLSLArrayAccess : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_ArrayAccess;
    HLSLArrayAccess()
    {
      array = nullptr;
      index = nullptr;
    }
    HLSLExpression* array;
    HLSLExpression* index;
  };

  struct HLSLFunctionCall : public HLSLExpression
  {
    static const HLSLNodeType s_type = HLSLNodeType_FunctionCall;
    HLSLFunctionCall()
    {
      function = nullptr;
      argument = nullptr;
      numArguments = 0;
    }
    const HLSLFunction* function;
    HLSLExpression* argument;
    int                 numArguments;
  };

  struct HLSLStateAssignment : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_StateAssignment;
    HLSLStateAssignment()
    {
      stateName = nullptr;
      sValue = nullptr;
      nextStateAssignment = nullptr;
    }

    const char* stateName;
    int                     d3dRenderState;
    union {
      int                 iValue;
      float               fValue;
      const char* sValue;
    };
    HLSLStateAssignment* nextStateAssignment;
  };

  struct HLSLSamplerState : public HLSLExpression // @@ Does this need to be an expression? Does it have a type? I guess type is useful.
  {
    static const HLSLNodeType s_type = HLSLNodeType_SamplerState;
    HLSLSamplerState()
    {
      numStateAssignments = 0;
      stateAssignments = nullptr;
    }

    int                     numStateAssignments;
    HLSLStateAssignment* stateAssignments;
  };

  struct HLSLPass : public HLSLNode
  {
    static const HLSLNodeType s_type = HLSLNodeType_Pass;
    HLSLPass()
    {
      name = nullptr;
      numStateAssignments = 0;
      stateAssignments = nullptr;
      nextPass = nullptr;
    }

    const char* name;
    int                     numStateAssignments;
    HLSLStateAssignment* stateAssignments;
    HLSLPass* nextPass;
  };

  struct HLSLTechnique : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Technique;
    HLSLTechnique()
    {
      name = nullptr;
      numPasses = 0;
      passes = nullptr;
    }

    const char* name;
    int                 numPasses;
    HLSLPass* passes;
  };

  struct HLSLPipeline : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Pipeline;
    HLSLPipeline()
    {
      name = nullptr;
      numStateAssignments = 0;
      stateAssignments = nullptr;
    }

    const char* name;
    int                     numStateAssignments;
    HLSLStateAssignment* stateAssignments;
  };

  struct HLSLStage : public HLSLStatement
  {
    static const HLSLNodeType s_type = HLSLNodeType_Stage;
    HLSLStage()
    {
      name = nullptr;
      statement = nullptr;
      inputs = nullptr;
      outputs = nullptr;
    }

    const char* name;
    HLSLStatement* statement;
    HLSLDeclaration* inputs;
    HLSLDeclaration* outputs;
  };

  struct matrixCtor {
    HLSLBaseType matrixType;
    std::vector<HLSLBaseType> argumentTypes;

    bool operator==(const matrixCtor& other) const
    {
      return  matrixType == other.matrixType &&
        argumentTypes == other.argumentTypes;
    }

    bool operator<(const matrixCtor& other) const
    {
      if (matrixType < other.matrixType)
      {
        return true;
      }
      else if (matrixType > other.matrixType)
      {
        return false;
      }

      return argumentTypes < other.argumentTypes;
    }
  };


  /**
   * Abstract syntax tree for parsed HLSL code.
   */
  class HLSLTree
  {

  public:

    explicit HLSLTree(Allocator* allocator);
    ~HLSLTree();

    /** Adds a string to the string pool used by the tree. */
    const char* AddString(const char* string);
    const char* AddStringFormat(const char* string, ...);

    /** Returns true if the string is contained within the tree. */
    bool GetContainsString(const char* string) const;

    /** Returns the root block in the tree */
    HLSLRoot* GetRoot() const;

    /** Adds a new node to the tree with the specified type. */
    template <class T>
    T* AddNode(const char* fileName, int line)
    {
      HLSLNode* node = new (AllocateMemory(sizeof(T))) T();
      node->nodeType = T::s_type;
      node->fileName = fileName;
      node->line = line;
      return static_cast<T*>(node);
    }

    HLSLFunction* FindFunction(const char* name);
    HLSLDeclaration* FindGlobalDeclaration(const char* name, HLSLBuffer** buffer_out = nullptr);
    HLSLStruct* FindGlobalStruct(const char* name);
    HLSLTechnique* FindTechnique(const char* name);
    HLSLPipeline* FindFirstPipeline();
    HLSLPipeline* FindNextPipeline(HLSLPipeline* current);
    HLSLPipeline* FindPipeline(const char* name);
    HLSLBuffer* FindBuffer(const char* name);

    bool GetExpressionValue(HLSLExpression* expression, int& value);
    int GetExpressionValue(HLSLExpression* expression, float values[4]);

    bool NeedsFunction(const char* name);
    bool ReplaceUniformsAssignments();
    void EnumerateMatrixCtorsNeeded(std::vector<matrixCtor>& matrixCtors);

  private:

    void* AllocateMemory(size_t size);
    void  AllocatePage();

  private:

    static const size_t s_nodePageSize = 1024 * 4;

    struct NodePage
    {
      NodePage* next;
      char        buffer[s_nodePageSize];
    };

    Allocator* m_allocator;
    StringPool      m_stringPool;
    HLSLRoot* m_root;

    NodePage* m_firstPage;
    NodePage* m_currentPage;
    size_t          m_currentPageOffset;

  };



  class HLSLTreeVisitor
  {
  public:
    virtual void VisitType(HLSLType& type);

    virtual void VisitRoot(HLSLRoot* node);
    virtual void VisitTopLevelStatement(HLSLStatement* node);
    virtual void VisitStatements(HLSLStatement* statement);
    virtual void VisitStatement(HLSLStatement* node);
    virtual void VisitDeclaration(HLSLDeclaration* node);
    virtual void VisitStruct(HLSLStruct* node);
    virtual void VisitStructField(HLSLStructField* node);
    virtual void VisitBuffer(HLSLBuffer* node);
    //virtual void VisitBufferField(HLSLBufferField * node);
    virtual void VisitFunction(HLSLFunction* node);
    virtual void VisitArgument(HLSLArgument* node);
    virtual void VisitExpressionStatement(HLSLExpressionStatement* node);
    virtual void VisitExpression(HLSLExpression* node);
    virtual void VisitReturnStatement(HLSLReturnStatement* node);
    virtual void VisitDiscardStatement(HLSLDiscardStatement* node);
    virtual void VisitBreakStatement(HLSLBreakStatement* node);
    virtual void VisitContinueStatement(HLSLContinueStatement* node);
    virtual void VisitIfStatement(HLSLIfStatement* node);
    virtual void VisitForStatement(HLSLForStatement* node);
    virtual void VisitWhileStatement(HLSLWhileStatement* node);
    virtual void VisitBlockStatement(HLSLBlockStatement* node);
    virtual void VisitUnaryExpression(HLSLUnaryExpression* node);
    virtual void VisitBinaryExpression(HLSLBinaryExpression* node);
    virtual void VisitConditionalExpression(HLSLConditionalExpression* node);
    virtual void VisitCastingExpression(HLSLCastingExpression* node);
    virtual void VisitLiteralExpression(HLSLLiteralExpression* node);
    virtual void VisitIdentifierExpression(HLSLIdentifierExpression* node);
    virtual void VisitConstructorExpression(HLSLConstructorExpression* node);
    virtual void VisitMemberAccess(HLSLMemberAccess* node);
    virtual void VisitArrayAccess(HLSLArrayAccess* node);
    virtual void VisitFunctionCall(HLSLFunctionCall* node);
    virtual void VisitStateAssignment(HLSLStateAssignment* node);
    virtual void VisitSamplerState(HLSLSamplerState* node);
    virtual void VisitPass(HLSLPass* node);
    virtual void VisitTechnique(HLSLTechnique* node);
    virtual void VisitPipeline(HLSLPipeline* node);


    virtual void VisitFunctions(HLSLRoot* root);
    virtual void VisitParameters(HLSLRoot* root);

    HLSLFunction* FindFunction(HLSLRoot* root, const char* name);
    HLSLDeclaration* FindGlobalDeclaration(HLSLRoot* root, const char* name);
    HLSLStruct* FindGlobalStruct(HLSLRoot* root, const char* name);
  };


  // Tree transformations:
  extern void PruneTree(HLSLTree* tree, const char* entryName0, const char* entryName1 = nullptr);
  extern void SortTree(HLSLTree* tree);
  extern void GroupParameters(HLSLTree* tree);
  extern void HideUnusedArguments(HLSLFunction* function);
  extern bool EmulateAlphaTest(HLSLTree* tree, const char* entryName, float alphaRef = 0.5f);
  extern void FlattenExpressions(HLSLTree* tree);

  extern matrixCtor matrixCtorBuilder(HLSLType type, HLSLExpression* arguments);


} // M4

namespace M4
{

  HLSLTree::HLSLTree(Allocator* allocator) :
    m_allocator(allocator), m_stringPool(allocator)
  {
    m_firstPage = m_allocator->New<NodePage>();
    m_firstPage->next = nullptr;

    m_currentPage = m_firstPage;
    m_currentPageOffset = 0;

    m_root = AddNode<HLSLRoot>(nullptr, 1);
  }

  HLSLTree::~HLSLTree()
  {
    NodePage* page = m_firstPage;
    while (page != nullptr)
    {
      NodePage* next = page->next;
      m_allocator->Delete(page);
      page = next;
    }
  }

  void HLSLTree::AllocatePage()
  {
    NodePage* newPage = m_allocator->New<NodePage>();
    newPage->next = nullptr;
    m_currentPage->next = newPage;
    m_currentPageOffset = 0;
    m_currentPage = newPage;
  }

  const char* HLSLTree::AddString(const char* string)
  {
    return m_stringPool.AddString(string);
  }

  const char* HLSLTree::AddStringFormat(const char* format, ...)
  {
    va_list args;
    va_start(args, format);
    const char* string = m_stringPool.AddStringFormatList(format, args);
    va_end(args);
    return string;
  }

  bool HLSLTree::GetContainsString(const char* string) const
  {
    return m_stringPool.GetContainsString(string);
  }

  HLSLRoot* HLSLTree::GetRoot() const
  {
    return m_root;
  }

  void* HLSLTree::AllocateMemory(size_t size)
  {
    if (m_currentPageOffset + size > s_nodePageSize)
    {
      AllocatePage();
    }
    void* buffer = m_currentPage->buffer + m_currentPageOffset;
    m_currentPageOffset += size;
    return buffer;
  }

  // @@ This doesn't do any parameter matching. Simply returns the first function with that name.
  HLSLFunction* HLSLTree::FindFunction(const char* name)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Function)
      {
        HLSLFunction* function = (HLSLFunction*)statement;
        if (String_Equal(name, function->name))
        {
          return function;
        }
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }

  HLSLDeclaration* HLSLTree::FindGlobalDeclaration(const char* name, HLSLBuffer** buffer_out/*=nullptr*/)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Declaration)
      {
        HLSLDeclaration* declaration = (HLSLDeclaration*)statement;
        if (String_Equal(name, declaration->name))
        {
          if (buffer_out) *buffer_out = nullptr;
          return declaration;
        }
      }
      else if (statement->nodeType == HLSLNodeType_Buffer)
      {
        HLSLBuffer* buffer = (HLSLBuffer*)statement;

        HLSLDeclaration* field = buffer->field;
        while (field != nullptr)
        {
          ASSERT(field->nodeType == HLSLNodeType_Declaration);
          if (String_Equal(name, field->name))
          {
            if (buffer_out) *buffer_out = buffer;
            return field;
          }
          field = (HLSLDeclaration*)field->nextStatement;
        }
      }

      statement = statement->nextStatement;
    }

    if (buffer_out) *buffer_out = nullptr;
    return nullptr;
  }

  HLSLStruct* HLSLTree::FindGlobalStruct(const char* name)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Struct)
      {
        HLSLStruct* declaration = (HLSLStruct*)statement;
        if (String_Equal(name, declaration->name))
        {
          return declaration;
        }
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }

  HLSLTechnique* HLSLTree::FindTechnique(const char* name)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Technique)
      {
        HLSLTechnique* technique = (HLSLTechnique*)statement;
        if (String_Equal(name, technique->name))
        {
          return technique;
        }
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }

  HLSLPipeline* HLSLTree::FindFirstPipeline()
  {
    return FindNextPipeline(nullptr);
  }

  HLSLPipeline* HLSLTree::FindNextPipeline(HLSLPipeline* current)
  {
    HLSLStatement* statement = current ? current : m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Pipeline)
      {
        return (HLSLPipeline*)statement;
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }

  HLSLPipeline* HLSLTree::FindPipeline(const char* name)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Pipeline)
      {
        HLSLPipeline* pipeline = (HLSLPipeline*)statement;
        if (String_Equal(name, pipeline->name))
        {
          return pipeline;
        }
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }

  HLSLBuffer* HLSLTree::FindBuffer(const char* name)
  {
    HLSLStatement* statement = m_root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Buffer)
      {
        HLSLBuffer* buffer = (HLSLBuffer*)statement;
        if (String_Equal(name, buffer->name))
        {
          return buffer;
        }
      }

      statement = statement->nextStatement;
    }

    return nullptr;
  }



  bool HLSLTree::GetExpressionValue(HLSLExpression* expression, int& value)
  {
    ASSERT(expression != nullptr);

    // Expression must be constant.
    if ((expression->expressionType.flags & HLSLTypeFlag_Const) == 0)
    {
      return false;
    }

    // We are expecting an integer scalar. @@ Add support for type conversion from other scalar types.
    if (expression->expressionType.baseType != HLSLBaseType_Int &&
      expression->expressionType.baseType != HLSLBaseType_Bool)
    {
      return false;
    }

    if (expression->expressionType.array)
    {
      return false;
    }

    if (expression->nodeType == HLSLNodeType_BinaryExpression)
    {
      HLSLBinaryExpression* binaryExpression = (HLSLBinaryExpression*)expression;

      int value1, value2;
      if (!GetExpressionValue(binaryExpression->expression1, value1) ||
        !GetExpressionValue(binaryExpression->expression2, value2))
      {
        return false;
      }

      switch (binaryExpression->binaryOp)
      {
      case HLSLBinaryOp_And:
        value = value1 && value2;
        return true;
      case HLSLBinaryOp_Or:
        value = value1 || value2;
        return true;
      case HLSLBinaryOp_Add:
        value = value1 + value2;
        return true;
      case HLSLBinaryOp_Sub:
        value = value1 - value2;
        return true;
      case HLSLBinaryOp_Mul:
        value = value1 * value2;
        return true;
      case HLSLBinaryOp_Div:
        value = value1 / value2;
        return true;
      case HLSLBinaryOp_Mod:
        value = value1 % value2;
        return true;
      case HLSLBinaryOp_Less:
        value = value1 < value2;
        return true;
      case HLSLBinaryOp_Greater:
        value = value1 > value2;
        return true;
      case HLSLBinaryOp_LessEqual:
        value = value1 <= value2;
        return true;
      case HLSLBinaryOp_GreaterEqual:
        value = value1 >= value2;
        return true;
      case HLSLBinaryOp_Equal:
        value = value1 == value2;
        return true;
      case HLSLBinaryOp_NotEqual:
        value = value1 != value2;
        return true;
      case HLSLBinaryOp_BitAnd:
        value = value1 & value2;
        return true;
      case HLSLBinaryOp_BitOr:
        value = value1 | value2;
        return true;
      case HLSLBinaryOp_BitXor:
        value = value1 ^ value2;
        return true;
      case HLSLBinaryOp_Assign:
      case HLSLBinaryOp_AddAssign:
      case HLSLBinaryOp_SubAssign:
      case HLSLBinaryOp_MulAssign:
      case HLSLBinaryOp_DivAssign:
        // IC: These are not valid on non-constant expressions and should fail earlier when querying expression value.
        return false;
      }
    }
    else if (expression->nodeType == HLSLNodeType_UnaryExpression)
    {
      HLSLUnaryExpression* unaryExpression = (HLSLUnaryExpression*)expression;

      if (!GetExpressionValue(unaryExpression->expression, value))
      {
        return false;
      }

      switch (unaryExpression->unaryOp)
      {
      case HLSLUnaryOp_Negative:
        value = -value;
        return true;
      case HLSLUnaryOp_Positive:
        // nop.
        return true;
      case HLSLUnaryOp_Not:
        value = !value;
        return true;
      case HLSLUnaryOp_BitNot:
        value = ~value;
        return true;
      case HLSLUnaryOp_PostDecrement:
      case HLSLUnaryOp_PostIncrement:
      case HLSLUnaryOp_PreDecrement:
      case HLSLUnaryOp_PreIncrement:
        // IC: These are not valid on non-constant expressions and should fail earlier when querying expression value.
        return false;
      }
    }
    else if (expression->nodeType == HLSLNodeType_IdentifierExpression)
    {
      HLSLIdentifierExpression* identifier = (HLSLIdentifierExpression*)expression;

      HLSLDeclaration* declaration = FindGlobalDeclaration(identifier->name);
      if (declaration == nullptr)
      {
        return false;
      }
      if ((declaration->type.flags & HLSLTypeFlag_Const) == 0)
      {
        return false;
      }

      return GetExpressionValue(declaration->assignment, value);
    }
    else if (expression->nodeType == HLSLNodeType_LiteralExpression)
    {
      HLSLLiteralExpression* literal = (HLSLLiteralExpression*)expression;

      if (literal->expressionType.baseType == HLSLBaseType_Int) value = literal->iValue;
      else if (literal->expressionType.baseType == HLSLBaseType_Bool) value = (int)literal->bValue;
      else return false;

      return true;
    }

    return false;
  }

  bool HLSLTree::NeedsFunction(const char* name)
  {
    // Early out
    if (!GetContainsString(name))
      return false;

    struct NeedsFunctionVisitor : HLSLTreeVisitor
    {
      const char* name;
      bool result;

      virtual void VisitTopLevelStatement(HLSLStatement* node)
      {
        if (!node->hidden)
          HLSLTreeVisitor::VisitTopLevelStatement(node);
      }

      virtual void VisitFunctionCall(HLSLFunctionCall* node)
      {
        result = result || String_Equal(name, node->function->name);

        HLSLTreeVisitor::VisitFunctionCall(node);
      }
    };

    NeedsFunctionVisitor visitor;
    visitor.name = name;
    visitor.result = false;

    visitor.VisitRoot(m_root);

    return visitor.result;
  }

  int GetVectorDimension(HLSLType& type)
  {
    if (type.baseType >= HLSLBaseType_FirstNumeric &&
      type.baseType <= HLSLBaseType_LastNumeric)
    {
      if (type.baseType == HLSLBaseType_Float) return 1;
      if (type.baseType == HLSLBaseType_Float2) return 2;
      if (type.baseType == HLSLBaseType_Float3) return 3;
      if (type.baseType == HLSLBaseType_Float4) return 4;

    }
    return 0;
  }

  // Returns dimension, 0 if invalid.
  int HLSLTree::GetExpressionValue(HLSLExpression* expression, float values[4])
  {
    ASSERT(expression != nullptr);

    // Expression must be constant.
    if ((expression->expressionType.flags & HLSLTypeFlag_Const) == 0)
    {
      return 0;
    }

    if (expression->expressionType.baseType == HLSLBaseType_Int ||
      expression->expressionType.baseType == HLSLBaseType_Bool)
    {
      int int_value;
      if (GetExpressionValue(expression, int_value)) {
        for (int i = 0; i < 4; i++) values[i] = (float)int_value;   // @@ Warn if conversion is not exact.
        return 1;
      }

      return 0;
    }
    if (expression->expressionType.baseType >= HLSLBaseType_FirstInteger && expression->expressionType.baseType <= HLSLBaseType_LastInteger)
    {
      // @@ Add support for uints?
      // @@ Add support for int vectors?
      return 0;
    }
    if (expression->expressionType.baseType > HLSLBaseType_LastNumeric)
    {
      return 0;
    }

    // @@ Not supported yet, but we may need it?
    if (expression->expressionType.array)
    {
      return false;
    }

    if (expression->nodeType == HLSLNodeType_BinaryExpression)
    {
      HLSLBinaryExpression* binaryExpression = (HLSLBinaryExpression*)expression;
      int dim = GetVectorDimension(binaryExpression->expressionType);

      float values1[4], values2[4];
      int dim1 = GetExpressionValue(binaryExpression->expression1, values1);
      int dim2 = GetExpressionValue(binaryExpression->expression2, values2);

      if (dim1 == 0 || dim2 == 0)
      {
        return 0;
      }

      if (dim1 != dim2)
      {
        // Brodacast scalar to vector size.
        if (dim1 == 1)
        {
          for (int i = 1; i < dim2; i++) values1[i] = values1[0];
          dim1 = dim2;
        }
        else if (dim2 == 1)
        {
          for (int i = 1; i < dim1; i++) values2[i] = values2[0];
          dim2 = dim1;
        }
        else
        {
          return 0;
        }
      }
      ASSERT(dim == dim1);

      switch (binaryExpression->binaryOp)
      {
      case HLSLBinaryOp_Add:
        for (int i = 0; i < dim; i++) values[i] = values1[i] + values2[i];
        return dim;
      case HLSLBinaryOp_Sub:
        for (int i = 0; i < dim; i++) values[i] = values1[i] - values2[i];
        return dim;
      case HLSLBinaryOp_Mul:
        for (int i = 0; i < dim; i++) values[i] = values1[i] * values2[i];
        return dim;
      case HLSLBinaryOp_Div:
        for (int i = 0; i < dim; i++) values[i] = values1[i] / values2[i];
        return dim;
      case HLSLBinaryOp_Mod:
        for (int i = 0; i < dim; i++) values[i] = int(values1[i]) % int(values2[i]);
        return dim;
      default:
        return 0;
      }
    }
    else if (expression->nodeType == HLSLNodeType_UnaryExpression)
    {
      HLSLUnaryExpression* unaryExpression = (HLSLUnaryExpression*)expression;
      int dim = GetVectorDimension(unaryExpression->expressionType);

      int dim1 = GetExpressionValue(unaryExpression->expression, values);
      if (dim1 == 0)
      {
        return 0;
      }
      ASSERT(dim == dim1);

      switch (unaryExpression->unaryOp)
      {
      case HLSLUnaryOp_Negative:
        for (int i = 0; i < dim; i++) values[i] = -values[i];
        return dim;
      case HLSLUnaryOp_Positive:
        // nop.
        return dim;
      default:
        return 0;
      }
    }
    else if (expression->nodeType == HLSLNodeType_ConstructorExpression)
    {
      HLSLConstructorExpression* constructor = (HLSLConstructorExpression*)expression;

      int dim = GetVectorDimension(constructor->expressionType);

      int idx = 0;
      HLSLExpression* arg = constructor->argument;
      while (arg != nullptr)
      {
        float tmp[4];
        int n = GetExpressionValue(arg, tmp);
        for (int i = 0; i < n; i++) values[idx + i] = tmp[i];
        idx += n;

        arg = arg->nextExpression;
      }
      ASSERT(dim == idx);

      return dim;
    }
    else if (expression->nodeType == HLSLNodeType_IdentifierExpression)
    {
      HLSLIdentifierExpression* identifier = (HLSLIdentifierExpression*)expression;

      HLSLDeclaration* declaration = FindGlobalDeclaration(identifier->name);
      if (declaration == nullptr)
      {
        return 0;
      }
      if ((declaration->type.flags & HLSLTypeFlag_Const) == 0)
      {
        return 0;
      }

      return GetExpressionValue(declaration->assignment, values);
    }
    else if (expression->nodeType == HLSLNodeType_LiteralExpression)
    {
      HLSLLiteralExpression* literal = (HLSLLiteralExpression*)expression;

      if (literal->expressionType.baseType == HLSLBaseType_Float) values[0] = literal->fValue;
      else if (literal->expressionType.baseType == HLSLBaseType_Bool) values[0] = literal->bValue;
      else if (literal->expressionType.baseType == HLSLBaseType_Int) values[0] = (float)literal->iValue;  // @@ Warn if conversion is not exact.
      else return 0;

      return 1;
    }

    return 0;
  }

  bool HLSLTree::ReplaceUniformsAssignments()
  {
    struct ReplaceUniformsAssignmentsVisitor : HLSLTreeVisitor
    {
      HLSLTree* tree;
      std::map<std::string, HLSLDeclaration*> uniforms;
      std::map<std::string, std::string> uniformsReplaced;
      bool withinAssignment;

      virtual void VisitDeclaration(HLSLDeclaration* node)
      {
        HLSLTreeVisitor::VisitDeclaration(node);

        // Enumerate uniforms
        if (node->type.flags & HLSLTypeFlag_Uniform)
        {
          uniforms[node->name] = node;
        }
      }

      virtual void VisitFunction(HLSLFunction* node)
      {
        uniformsReplaced.clear();

        // Detect uniforms assignments
        HLSLTreeVisitor::VisitFunction(node);

        // Declare uniforms replacements
        std::map<std::string, std::string>::const_iterator iter = uniformsReplaced.cbegin();
        for (; iter != uniformsReplaced.cend(); ++iter)
        {
          HLSLDeclaration* uniformDeclaration = uniforms[iter->first];
          HLSLDeclaration* declaration = tree->AddNode<HLSLDeclaration>(node->fileName, node->line);

          declaration->name = tree->AddString(iter->second.c_str());
          declaration->type = uniformDeclaration->type;

          // Add declaration within function statements
          declaration->nextStatement = node->statement;
          node->statement = declaration;
        }
      }

      virtual void VisitBinaryExpression(HLSLBinaryExpression* node)
      {
        // Visit expression 2 first to not replace possible uniform reading
        VisitExpression(node->expression2);

        if (IsAssignOp(node->binaryOp))
        {
          withinAssignment = true;
        }

        VisitExpression(node->expression1);

        withinAssignment = false;
      }

      virtual void VisitIdentifierExpression(HLSLIdentifierExpression* node)
      {
        if (withinAssignment)
        {
          // Check if variable is a uniform
          if (uniforms.find(node->name) != uniforms.end())
          {
            // Check if variable is not already replaced
            if (uniformsReplaced.find(node->name) == uniformsReplaced.end())
            {
              std::string newName(node->name);
              do
              {
                newName.insert(0, "new");
              } while (tree->GetContainsString(newName.c_str()));

              uniformsReplaced[node->name] = newName;
            }
          }
        }

        // Check if variable need to be replaced
        if (uniformsReplaced.find(node->name) != uniformsReplaced.end())
        {
          // Replace
          node->name = tree->AddString(uniformsReplaced[node->name].c_str());
        }
      }
    };

    ReplaceUniformsAssignmentsVisitor visitor;
    visitor.tree = this;
    visitor.withinAssignment = false;
    visitor.VisitRoot(m_root);

    return true;
  }


  matrixCtor matrixCtorBuilder(HLSLType type, HLSLExpression* arguments) {
    matrixCtor ctor;

    ctor.matrixType = type.baseType;

    // Fetch all arguments
    HLSLExpression* argument = arguments;
    while (argument != nullptr)
    {
      ctor.argumentTypes.push_back(argument->expressionType.baseType);
      argument = argument->nextExpression;
    }

    return ctor;
  }

  void HLSLTree::EnumerateMatrixCtorsNeeded(std::vector<matrixCtor>& matrixCtors) {

    struct EnumerateMatrixCtorsVisitor : HLSLTreeVisitor
    {
      std::vector<matrixCtor> matrixCtorsNeeded;

      virtual void VisitConstructorExpression(HLSLConstructorExpression* node)
      {
        if (IsMatrixType(node->expressionType.baseType))
        {
          matrixCtor ctor = matrixCtorBuilder(node->expressionType, node->argument);

          if (std::find(matrixCtorsNeeded.cbegin(), matrixCtorsNeeded.cend(), ctor) == matrixCtorsNeeded.cend())
          {
            matrixCtorsNeeded.push_back(ctor);
          }
        }

        HLSLTreeVisitor::VisitConstructorExpression(node);
      }

      virtual void VisitDeclaration(HLSLDeclaration* node)
      {
        if (IsMatrixType(node->type.baseType) &&
          (node->type.flags & HLSLArgumentModifier_Uniform) == 0)
        {
          matrixCtor ctor = matrixCtorBuilder(node->type, node->assignment);

          // No special constructor needed if it already a matrix
          bool matrixArgument = false;
          for (HLSLBaseType& type : ctor.argumentTypes)
          {
            if (IsMatrixType(type))
            {
              matrixArgument = true;
              break;
            }
          }

          if (!matrixArgument &&
            std::find(matrixCtorsNeeded.cbegin(), matrixCtorsNeeded.cend(), ctor) == matrixCtorsNeeded.cend())
          {
            matrixCtorsNeeded.push_back(ctor);
          }
        }

        HLSLTreeVisitor::VisitDeclaration(node);
      }
    };

    EnumerateMatrixCtorsVisitor visitor;
    visitor.VisitRoot(m_root);

    matrixCtors = visitor.matrixCtorsNeeded;
  }


  void HLSLTreeVisitor::VisitType(HLSLType& type)
  {
  }

  void HLSLTreeVisitor::VisitRoot(HLSLRoot* root)
  {
    HLSLStatement* statement = root->statement;
    while (statement != nullptr) {
      VisitTopLevelStatement(statement);
      statement = statement->nextStatement;
    }
  }

  void HLSLTreeVisitor::VisitTopLevelStatement(HLSLStatement* node)
  {
    if (node->nodeType == HLSLNodeType_Declaration) {
      VisitDeclaration((HLSLDeclaration*)node);
    }
    else if (node->nodeType == HLSLNodeType_Struct) {
      VisitStruct((HLSLStruct*)node);
    }
    else if (node->nodeType == HLSLNodeType_Buffer) {
      VisitBuffer((HLSLBuffer*)node);
    }
    else if (node->nodeType == HLSLNodeType_Function) {
      VisitFunction((HLSLFunction*)node);
    }
    else if (node->nodeType == HLSLNodeType_Technique) {
      VisitTechnique((HLSLTechnique*)node);
    }
    else if (node->nodeType == HLSLNodeType_Pipeline) {
      VisitPipeline((HLSLPipeline*)node);
    }
    else {
      ASSERT(0);
    }
  }

  void HLSLTreeVisitor::VisitStatements(HLSLStatement* statement)
  {
    while (statement != nullptr) {
      VisitStatement(statement);
      statement = statement->nextStatement;
    }
  }

  void HLSLTreeVisitor::VisitStatement(HLSLStatement* node)
  {
    // Function statements
    if (node->nodeType == HLSLNodeType_Declaration) {
      VisitDeclaration((HLSLDeclaration*)node);
    }
    else if (node->nodeType == HLSLNodeType_ExpressionStatement) {
      VisitExpressionStatement((HLSLExpressionStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_ReturnStatement) {
      VisitReturnStatement((HLSLReturnStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_DiscardStatement) {
      VisitDiscardStatement((HLSLDiscardStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_BreakStatement) {
      VisitBreakStatement((HLSLBreakStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_ContinueStatement) {
      VisitContinueStatement((HLSLContinueStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_IfStatement) {
      VisitIfStatement((HLSLIfStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_ForStatement) {
      VisitForStatement((HLSLForStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_WhileStatement) {
      VisitWhileStatement((HLSLWhileStatement*)node);
    }
    else if (node->nodeType == HLSLNodeType_BlockStatement) {
      VisitBlockStatement((HLSLBlockStatement*)node);
    }
    else {
      ASSERT(0);
    }
  }

  void HLSLTreeVisitor::VisitDeclaration(HLSLDeclaration* node)
  {
    VisitType(node->type);
    /*do {
        VisitExpression(node->assignment);
        node = node->nextDeclaration;
    } while (node);*/
    if (node->assignment != nullptr) {
      VisitExpression(node->assignment);
    }
    if (node->nextDeclaration != nullptr) {
      VisitDeclaration(node->nextDeclaration);
    }
  }

  void HLSLTreeVisitor::VisitStruct(HLSLStruct* node)
  {
    HLSLStructField* field = node->field;
    while (field != nullptr) {
      VisitStructField(field);
      field = field->nextField;
    }
  }

  void HLSLTreeVisitor::VisitStructField(HLSLStructField* node)
  {
    VisitType(node->type);
  }

  void HLSLTreeVisitor::VisitBuffer(HLSLBuffer* node)
  {
    HLSLDeclaration* field = node->field;
    while (field != nullptr) {
      ASSERT(field->nodeType == HLSLNodeType_Declaration);
      VisitDeclaration(field);
      ASSERT(field->nextDeclaration == nullptr);
      field = (HLSLDeclaration*)field->nextStatement;
    }
  }

  /*void HLSLTreeVisitor::VisitBufferField(HLSLBufferField * node)
  {
      VisitType(node->type);
  }*/

  void HLSLTreeVisitor::VisitFunction(HLSLFunction* node)
  {
    VisitType(node->returnType);

    HLSLArgument* argument = node->argument;
    while (argument != nullptr) {
      VisitArgument(argument);
      argument = argument->nextArgument;
    }

    VisitStatements(node->statement);
  }

  void HLSLTreeVisitor::VisitArgument(HLSLArgument* node)
  {
    VisitType(node->type);
    if (node->defaultValue != nullptr) {
      VisitExpression(node->defaultValue);
    }
  }

  void HLSLTreeVisitor::VisitExpressionStatement(HLSLExpressionStatement* node)
  {
    VisitExpression(node->expression);
  }

  void HLSLTreeVisitor::VisitExpression(HLSLExpression* node)
  {
    VisitType(node->expressionType);

    if (node->nodeType == HLSLNodeType_UnaryExpression) {
      VisitUnaryExpression((HLSLUnaryExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_BinaryExpression) {
      VisitBinaryExpression((HLSLBinaryExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_ConditionalExpression) {
      VisitConditionalExpression((HLSLConditionalExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_CastingExpression) {
      VisitCastingExpression((HLSLCastingExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_LiteralExpression) {
      VisitLiteralExpression((HLSLLiteralExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_IdentifierExpression) {
      VisitIdentifierExpression((HLSLIdentifierExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_ConstructorExpression) {
      VisitConstructorExpression((HLSLConstructorExpression*)node);
    }
    else if (node->nodeType == HLSLNodeType_MemberAccess) {
      VisitMemberAccess((HLSLMemberAccess*)node);
    }
    else if (node->nodeType == HLSLNodeType_ArrayAccess) {
      VisitArrayAccess((HLSLArrayAccess*)node);
    }
    else if (node->nodeType == HLSLNodeType_FunctionCall) {
      VisitFunctionCall((HLSLFunctionCall*)node);
    }
    // Acoget-TODO: This was missing. Did adding it break anything?
    else if (node->nodeType == HLSLNodeType_SamplerState) {
      VisitSamplerState((HLSLSamplerState*)node);
    }
    else {
      ASSERT(0);
    }
  }

  void HLSLTreeVisitor::VisitReturnStatement(HLSLReturnStatement* node)
  {
    VisitExpression(node->expression);
  }

  void HLSLTreeVisitor::VisitDiscardStatement(HLSLDiscardStatement* node) {}
  void HLSLTreeVisitor::VisitBreakStatement(HLSLBreakStatement* node) {}
  void HLSLTreeVisitor::VisitContinueStatement(HLSLContinueStatement* node) {}

  void HLSLTreeVisitor::VisitIfStatement(HLSLIfStatement* node)
  {
    VisitExpression(node->condition);
    VisitStatements(node->statement);
    if (node->elseStatement) {
      VisitStatements(node->elseStatement);
    }
  }

  void HLSLTreeVisitor::VisitForStatement(HLSLForStatement* node)
  {
    if (node->initialization) {
      VisitDeclaration(node->initialization);
    }
    if (node->condition) {
      VisitExpression(node->condition);
    }
    if (node->increment) {
      VisitExpression(node->increment);
    }
    VisitStatements(node->statement);
  }

  void HLSLTreeVisitor::VisitWhileStatement(HLSLWhileStatement* node)
  {
    if (node->condition) {
      VisitExpression(node->condition);
    }
    VisitStatements(node->statement);
  }

  void HLSLTreeVisitor::VisitBlockStatement(HLSLBlockStatement* node)
  {
    VisitStatements(node->statement);
  }

  void HLSLTreeVisitor::VisitUnaryExpression(HLSLUnaryExpression* node)
  {
    VisitExpression(node->expression);
  }

  void HLSLTreeVisitor::VisitBinaryExpression(HLSLBinaryExpression* node)
  {
    VisitExpression(node->expression1);
    VisitExpression(node->expression2);
  }

  void HLSLTreeVisitor::VisitConditionalExpression(HLSLConditionalExpression* node)
  {
    VisitExpression(node->condition);
    VisitExpression(node->falseExpression);
    VisitExpression(node->trueExpression);
  }

  void HLSLTreeVisitor::VisitCastingExpression(HLSLCastingExpression* node)
  {
    VisitType(node->type);
    VisitExpression(node->expression);
  }

  void HLSLTreeVisitor::VisitLiteralExpression(HLSLLiteralExpression* node) {}
  void HLSLTreeVisitor::VisitIdentifierExpression(HLSLIdentifierExpression* node) {}

  void HLSLTreeVisitor::VisitConstructorExpression(HLSLConstructorExpression* node)
  {
    HLSLExpression* argument = node->argument;
    while (argument != nullptr) {
      VisitExpression(argument);
      argument = argument->nextExpression;
    }
  }

  void HLSLTreeVisitor::VisitMemberAccess(HLSLMemberAccess* node)
  {
    VisitExpression(node->object);
  }

  void HLSLTreeVisitor::VisitArrayAccess(HLSLArrayAccess* node)
  {
    VisitExpression(node->array);
    VisitExpression(node->index);
  }

  void HLSLTreeVisitor::VisitFunctionCall(HLSLFunctionCall* node)
  {
    HLSLExpression* argument = node->argument;
    while (argument != nullptr) {
      VisitExpression(argument);
      argument = argument->nextExpression;
    }
  }

  void HLSLTreeVisitor::VisitStateAssignment(HLSLStateAssignment* node) {}

  void HLSLTreeVisitor::VisitSamplerState(HLSLSamplerState* node)
  {
    HLSLStateAssignment* stateAssignment = node->stateAssignments;
    while (stateAssignment != nullptr) {
      VisitStateAssignment(stateAssignment);
      stateAssignment = stateAssignment->nextStateAssignment;
    }
  }

  void HLSLTreeVisitor::VisitPass(HLSLPass* node)
  {
    HLSLStateAssignment* stateAssignment = node->stateAssignments;
    while (stateAssignment != nullptr) {
      VisitStateAssignment(stateAssignment);
      stateAssignment = stateAssignment->nextStateAssignment;
    }
  }

  void HLSLTreeVisitor::VisitTechnique(HLSLTechnique* node)
  {
    HLSLPass* pass = node->passes;
    while (pass != nullptr) {
      VisitPass(pass);
      pass = pass->nextPass;
    }
  }

  void HLSLTreeVisitor::VisitPipeline(HLSLPipeline* node)
  {
    // @@ ?
  }

  void HLSLTreeVisitor::VisitFunctions(HLSLRoot* root)
  {
    HLSLStatement* statement = root->statement;
    while (statement != nullptr) {
      if (statement->nodeType == HLSLNodeType_Function) {
        VisitFunction((HLSLFunction*)statement);
      }

      statement = statement->nextStatement;
    }
  }

  void HLSLTreeVisitor::VisitParameters(HLSLRoot* root)
  {
    HLSLStatement* statement = root->statement;
    while (statement != nullptr) {
      if (statement->nodeType == HLSLNodeType_Declaration) {
        VisitDeclaration((HLSLDeclaration*)statement);
      }

      statement = statement->nextStatement;
    }
  }


  class ResetHiddenFlagVisitor : public HLSLTreeVisitor
  {
  public:
    virtual void VisitTopLevelStatement(HLSLStatement* statement)
    {
      statement->hidden = true;

      if (statement->nodeType == HLSLNodeType_Buffer)
      {
        VisitBuffer((HLSLBuffer*)statement);
      }
    }

    // Hide buffer fields.
    virtual void VisitDeclaration(HLSLDeclaration* node)
    {
      node->hidden = true;
    }

    virtual void VisitArgument(HLSLArgument* node)
    {
      node->hidden = false;   // Arguments are visible by default.
    }
  };

  class MarkVisibleStatementsVisitor : public HLSLTreeVisitor
  {
  public:
    HLSLTree* tree;
    MarkVisibleStatementsVisitor(HLSLTree* _tree) : tree(_tree) {}

    virtual void VisitFunction(HLSLFunction* node)
    {
      node->hidden = false;
      HLSLTreeVisitor::VisitFunction(node);

      if (node->forward)
        VisitFunction(node->forward);
    }

    virtual void VisitFunctionCall(HLSLFunctionCall* node)
    {
      HLSLTreeVisitor::VisitFunctionCall(node);

      if (node->function->hidden)
      {
        VisitFunction(const_cast<HLSLFunction*>(node->function));
      }
    }

    virtual void VisitIdentifierExpression(HLSLIdentifierExpression* node)
    {
      HLSLTreeVisitor::VisitIdentifierExpression(node);

      if (node->global)
      {
        HLSLDeclaration* declaration = tree->FindGlobalDeclaration(node->name);
        if (declaration != nullptr && declaration->hidden)
        {
          declaration->hidden = false;
          VisitDeclaration(declaration);
        }
      }
    }

    virtual void VisitType(HLSLType& type)
    {
      if (type.baseType == HLSLBaseType_UserDefined)
      {
        HLSLStruct* globalStruct = tree->FindGlobalStruct(type.typeName);
        if (globalStruct != nullptr)
        {
          globalStruct->hidden = false;
          VisitStruct(globalStruct);
        }
      }
    }

  };


  void PruneTree(HLSLTree* tree, const char* entryName0, const char* entryName1/*=nullptr*/)
  {
    HLSLRoot* root = tree->GetRoot();

    // Reset all flags.
    ResetHiddenFlagVisitor reset;
    reset.VisitRoot(root);

    // Mark all the statements necessary for these entrypoints.
    HLSLFunction* entry = tree->FindFunction(entryName0);
    if (entry != nullptr)
    {
      MarkVisibleStatementsVisitor mark(tree);
      mark.VisitFunction(entry);
    }

    if (entryName1 != nullptr)
    {
      entry = tree->FindFunction(entryName1);
      if (entry != nullptr)
      {
        MarkVisibleStatementsVisitor mark(tree);
        mark.VisitFunction(entry);
      }
    }

    // Mark buffers visible, if any of their fields is visible.
    HLSLStatement* statement = root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Buffer)
      {
        HLSLBuffer* buffer = (HLSLBuffer*)statement;

        HLSLDeclaration* field = buffer->field;
        while (field != nullptr)
        {
          ASSERT(field->nodeType == HLSLNodeType_Declaration);
          if (!field->hidden)
          {
            buffer->hidden = false;
            break;
          }
          field = (HLSLDeclaration*)field->nextStatement;
        }
      }

      statement = statement->nextStatement;
    }
  }


  void SortTree(HLSLTree* tree)
  {
    // Stable sort so that statements are in this order:
    // structs, declarations, functions, techniques.
  // but their relative order is preserved.

    HLSLRoot* root = tree->GetRoot();

    HLSLStatement* structs = nullptr;
    HLSLStatement* lastStruct = nullptr;
    HLSLStatement* constDeclarations = nullptr;
    HLSLStatement* lastConstDeclaration = nullptr;
    HLSLStatement* declarations = nullptr;
    HLSLStatement* lastDeclaration = nullptr;
    HLSLStatement* functions = nullptr;
    HLSLStatement* lastFunction = nullptr;
    HLSLStatement* other = nullptr;
    HLSLStatement* lastOther = nullptr;

    HLSLStatement* statement = root->statement;
    while (statement != nullptr) {
      HLSLStatement* nextStatement = statement->nextStatement;
      statement->nextStatement = nullptr;

      if (statement->nodeType == HLSLNodeType_Struct) {
        if (structs == nullptr) structs = statement;
        if (lastStruct != nullptr) lastStruct->nextStatement = statement;
        lastStruct = statement;
      }
      else if (statement->nodeType == HLSLNodeType_Declaration || statement->nodeType == HLSLNodeType_Buffer) {
        if (statement->nodeType == HLSLNodeType_Declaration && (((HLSLDeclaration*)statement)->type.flags & HLSLTypeFlag_Const)) {
          if (constDeclarations == nullptr) constDeclarations = statement;
          if (lastConstDeclaration != nullptr) lastConstDeclaration->nextStatement = statement;
          lastConstDeclaration = statement;
        }
        else {
          if (declarations == nullptr) declarations = statement;
          if (lastDeclaration != nullptr) lastDeclaration->nextStatement = statement;
          lastDeclaration = statement;
        }
      }
      else if (statement->nodeType == HLSLNodeType_Function) {
        if (functions == nullptr) functions = statement;
        if (lastFunction != nullptr) lastFunction->nextStatement = statement;
        lastFunction = statement;
      }
      else {
        if (other == nullptr) other = statement;
        if (lastOther != nullptr) lastOther->nextStatement = statement;
        lastOther = statement;
      }

      statement = nextStatement;
    }

    // Chain all the statements in the order that we want.
    HLSLStatement* firstStatement = structs;
    HLSLStatement* lastStatement = lastStruct;

    if (constDeclarations != nullptr) {
      if (firstStatement == nullptr) firstStatement = constDeclarations;
      else lastStatement->nextStatement = constDeclarations;
      lastStatement = lastConstDeclaration;
    }

    if (declarations != nullptr) {
      if (firstStatement == nullptr) firstStatement = declarations;
      else lastStatement->nextStatement = declarations;
      lastStatement = lastDeclaration;
    }

    if (functions != nullptr) {
      if (firstStatement == nullptr) firstStatement = functions;
      else lastStatement->nextStatement = functions;
      lastStatement = lastFunction;
    }

    if (other != nullptr) {
      if (firstStatement == nullptr) firstStatement = other;
      else lastStatement->nextStatement = other;
      lastStatement = lastOther;
    }

    root->statement = firstStatement;
  }





  // First and last can be the same.
  void AddStatements(HLSLRoot* root, HLSLStatement* before, HLSLStatement* first, HLSLStatement* last)
  {
    if (before == nullptr) {
      last->nextStatement = root->statement;
      root->statement = first;
    }
    else {
      last->nextStatement = before->nextStatement;
      before->nextStatement = first;
    }
  }

  void AddSingleStatement(HLSLRoot* root, HLSLStatement* before, HLSLStatement* statement)
  {
    AddStatements(root, before, statement, statement);
  }



  // @@ This is very game-specific. Should be moved to pipeline_parser or somewhere else.
  void GroupParameters(HLSLTree* tree)
  {
    // Sort parameters based on semantic and group them in cbuffers.

    HLSLRoot* root = tree->GetRoot();

    HLSLDeclaration* firstPerItemDeclaration = nullptr;
    HLSLDeclaration* lastPerItemDeclaration = nullptr;

    HLSLDeclaration* instanceDataDeclaration = nullptr;

    HLSLDeclaration* firstPerPassDeclaration = nullptr;
    HLSLDeclaration* lastPerPassDeclaration = nullptr;

    HLSLDeclaration* firstPerItemSampler = nullptr;
    HLSLDeclaration* lastPerItemSampler = nullptr;

    HLSLDeclaration* firstPerPassSampler = nullptr;
    HLSLDeclaration* lastPerPassSampler = nullptr;

    HLSLStatement* statementBeforeBuffers = nullptr;

    HLSLStatement* previousStatement = nullptr;
    HLSLStatement* statement = root->statement;
    while (statement != nullptr)
    {
      HLSLStatement* nextStatement = statement->nextStatement;

      if (statement->nodeType == HLSLNodeType_Struct) // Do not remove this, or it will mess the else clause below.
      {
        statementBeforeBuffers = statement;
      }
      else if (statement->nodeType == HLSLNodeType_Declaration)
      {
        HLSLDeclaration* declaration = (HLSLDeclaration*)statement;

        // We insert buffers after the last const declaration.
        if ((declaration->type.flags & HLSLTypeFlag_Const) != 0)
        {
          statementBeforeBuffers = statement;
        }

        // Do not move samplers or static/const parameters.
        if ((declaration->type.flags & (HLSLTypeFlag_Static | HLSLTypeFlag_Const)) == 0)
        {
          // Unlink statement.
          statement->nextStatement = nullptr;
          if (previousStatement != nullptr) previousStatement->nextStatement = nextStatement;
          else root->statement = nextStatement;

          while (declaration != nullptr)
          {
            HLSLDeclaration* nextDeclaration = declaration->nextDeclaration;

            if (declaration->semantic != nullptr && String_EqualNoCase(declaration->semantic, "PER_INSTANCED_ITEM"))
            {
              ASSERT(instanceDataDeclaration == nullptr);
              instanceDataDeclaration = declaration;
            }
            else
            {
              // Select group based on type and semantic.
              HLSLDeclaration** first, ** last;
              if (declaration->semantic == nullptr || String_EqualNoCase(declaration->semantic, "PER_ITEM") || String_EqualNoCase(declaration->semantic, "PER_MATERIAL"))
              {
                if (IsSamplerType(declaration->type))
                {
                  first = &firstPerItemSampler;
                  last = &lastPerItemSampler;
                }
                else
                {
                  first = &firstPerItemDeclaration;
                  last = &lastPerItemDeclaration;
                }
              }
              else
              {
                if (IsSamplerType(declaration->type))
                {
                  first = &firstPerPassSampler;
                  last = &lastPerPassSampler;
                }
                else
                {
                  first = &firstPerPassDeclaration;
                  last = &lastPerPassDeclaration;
                }
              }

              // Add declaration to new list.
              if (*first == nullptr) *first = declaration;
              else (*last)->nextStatement = declaration;
              *last = declaration;
            }

            // Unlink from declaration list.
            declaration->nextDeclaration = nullptr;

            // Reset attributes.
            declaration->registerName = nullptr;
            //declaration->semantic = nullptr;         // @@ Don't do this!

            declaration = nextDeclaration;
          }
        }
      }
      /*else
      {
          if (statementBeforeBuffers == nullptr) {
              // This is the location where we will insert our buffers.
              statementBeforeBuffers = previousStatement;
          }
      }*/

      if (statement->nextStatement == nextStatement) {
        previousStatement = statement;
      }
      statement = nextStatement;
    }


    // Add instance data declaration at the end of the per_item buffer.
    if (instanceDataDeclaration != nullptr)
    {
      if (firstPerItemDeclaration == nullptr) firstPerItemDeclaration = instanceDataDeclaration;
      else lastPerItemDeclaration->nextStatement = instanceDataDeclaration;
    }


    // Add samplers.
    if (firstPerItemSampler != nullptr) {
      AddStatements(root, statementBeforeBuffers, firstPerItemSampler, lastPerItemSampler);
      statementBeforeBuffers = lastPerItemSampler;
    }
    if (firstPerPassSampler != nullptr) {
      AddStatements(root, statementBeforeBuffers, firstPerPassSampler, lastPerPassSampler);
      statementBeforeBuffers = lastPerPassSampler;
    }


    // @@ We are assuming per_item and per_pass buffers don't already exist. @@ We should assert on that.

    if (firstPerItemDeclaration != nullptr)
    {
      // Create buffer statement.
      HLSLBuffer* perItemBuffer = tree->AddNode<HLSLBuffer>(firstPerItemDeclaration->fileName, firstPerItemDeclaration->line - 1);
      perItemBuffer->name = tree->AddString("per_item");
      perItemBuffer->registerName = tree->AddString("b0");
      perItemBuffer->field = firstPerItemDeclaration;

      // Set declaration buffer pointers.
      HLSLDeclaration* field = perItemBuffer->field;
      while (field != nullptr)
      {
        field->buffer = perItemBuffer;
        field = (HLSLDeclaration*)field->nextStatement;
      }

      // Add buffer to statements.
      AddSingleStatement(root, statementBeforeBuffers, perItemBuffer);
      statementBeforeBuffers = perItemBuffer;
    }

    if (firstPerPassDeclaration != nullptr)
    {
      // Create buffer statement.
      HLSLBuffer* perPassBuffer = tree->AddNode<HLSLBuffer>(firstPerPassDeclaration->fileName, firstPerPassDeclaration->line - 1);
      perPassBuffer->name = tree->AddString("per_pass");
      perPassBuffer->registerName = tree->AddString("b1");
      perPassBuffer->field = firstPerPassDeclaration;

      // Set declaration buffer pointers.
      HLSLDeclaration* field = perPassBuffer->field;
      while (field != nullptr)
      {
        field->buffer = perPassBuffer;
        field = (HLSLDeclaration*)field->nextStatement;
      }

      // Add buffer to statements.
      AddSingleStatement(root, statementBeforeBuffers, perPassBuffer);
    }
  }


  class FindArgumentVisitor : public HLSLTreeVisitor
  {
  public:
    bool found;
    const char* name;

    FindArgumentVisitor()
    {
      found = false;
      name = nullptr;
    }

    bool FindArgument(const char* _name, HLSLFunction* function)
    {
      this->found = false;
      this->name = _name;
      VisitStatements(function->statement);
      return found;
    }

    virtual void VisitStatements(HLSLStatement* statement) override
    {
      while (statement != nullptr && !found)
      {
        VisitStatement(statement);
        statement = statement->nextStatement;
      }
    }

    virtual void VisitIdentifierExpression(HLSLIdentifierExpression* node) override
    {
      if (node->name == name)
      {
        found = true;
      }
    }
  };


  void HideUnusedArguments(HLSLFunction* function)
  {
    FindArgumentVisitor visitor;

    // For each argument.
    HLSLArgument* arg = function->argument;
    while (arg != nullptr)
    {
      if (!visitor.FindArgument(arg->name, function))
      {
        arg->hidden = true;
      }

      arg = arg->nextArgument;
    }
  }

  bool EmulateAlphaTest(HLSLTree* tree, const char* entryName, float alphaRef/*=0.5*/)
  {
    // Find all return statements of this entry point.
    HLSLFunction* entry = tree->FindFunction(entryName);
    if (entry != nullptr)
    {
      HLSLStatement** ptr = &entry->statement;
      HLSLStatement* statement = entry->statement;
      while (statement != nullptr)
      {
        if (statement->nodeType == HLSLNodeType_ReturnStatement)
        {
          HLSLReturnStatement* returnStatement = (HLSLReturnStatement*)statement;
          HLSLBaseType returnType = returnStatement->expression->expressionType.baseType;

          // Build statement: "if (%s.a < 0.5) discard;"

          HLSLDiscardStatement* discard = tree->AddNode<HLSLDiscardStatement>(statement->fileName, statement->line);

          HLSLExpression* alpha = nullptr;
          if (returnType == HLSLBaseType_Float4)
          {
            // @@ If return expression is a constructor, grab 4th argument.
            // That's not as easy, since we support 'float4(float3, float)' or 'float4(float, float3)', extracting
            // the latter is not that easy.
            /*if (returnStatement->expression->nodeType == HLSLNodeType_ConstructorExpression) {
                HLSLConstructorExpression * constructor = (HLSLConstructorExpression *)returnStatement->expression;
                //constructor->
            }
            */

            if (alpha == nullptr) {
              HLSLMemberAccess* access = tree->AddNode<HLSLMemberAccess>(statement->fileName, statement->line);
              access->expressionType = HLSLType(HLSLBaseType_Float);
              access->object = returnStatement->expression;     // @@ Is reference OK? Or should we clone expression?
              access->field = tree->AddString("a");
              access->swizzle = true;

              alpha = access;
            }
          }
          else if (returnType == HLSLBaseType_Float)
          {
            alpha = returnStatement->expression;     // @@ Is reference OK? Or should we clone expression?
          }
          else
          {
            return false;
          }

          HLSLLiteralExpression* threshold = tree->AddNode<HLSLLiteralExpression>(statement->fileName, statement->line);
          threshold->expressionType = HLSLType(HLSLBaseType_Float);
          threshold->fValue = alphaRef;
          threshold->type = HLSLBaseType_Float;

          HLSLBinaryExpression* condition = tree->AddNode<HLSLBinaryExpression>(statement->fileName, statement->line);
          condition->expressionType = HLSLType(HLSLBaseType_Bool);
          condition->binaryOp = HLSLBinaryOp_Less;
          condition->expression1 = alpha;
          condition->expression2 = threshold;

          // Insert statement.
          HLSLIfStatement* st = tree->AddNode<HLSLIfStatement>(statement->fileName, statement->line);
          st->condition = condition;
          st->statement = discard;
          st->nextStatement = statement;
          *ptr = st;
        }

        ptr = &statement->nextStatement;
        statement = statement->nextStatement;
      }
    }

    return true;
  }

  bool NeedsFlattening(HLSLExpression* expr, int level = 0) {
    if (expr == nullptr) {
      return false;
    }
    if (expr->nodeType == HLSLNodeType_UnaryExpression) {
      HLSLUnaryExpression* unaryExpr = (HLSLUnaryExpression*)expr;
      return NeedsFlattening(unaryExpr->expression, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_BinaryExpression) {
      HLSLBinaryExpression* binaryExpr = (HLSLBinaryExpression*)expr;
      if (IsAssignOp(binaryExpr->binaryOp)) {
        return NeedsFlattening(binaryExpr->expression2, level + 1) || NeedsFlattening(expr->nextExpression, level);
      }
      else {
        return NeedsFlattening(binaryExpr->expression1, level + 1) || NeedsFlattening(binaryExpr->expression2, level + 1) || NeedsFlattening(expr->nextExpression, level);
      }
    }
    else if (expr->nodeType == HLSLNodeType_ConditionalExpression) {
      HLSLConditionalExpression* conditionalExpr = (HLSLConditionalExpression*)expr;
      return NeedsFlattening(conditionalExpr->condition, level + 1) || NeedsFlattening(conditionalExpr->trueExpression, level + 1) || NeedsFlattening(conditionalExpr->falseExpression, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_CastingExpression) {
      HLSLCastingExpression* castingExpr = (HLSLCastingExpression*)expr;
      return NeedsFlattening(castingExpr->expression, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_LiteralExpression) {
      return NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_IdentifierExpression) {
      return NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_ConstructorExpression) {
      HLSLConstructorExpression* constructorExpr = (HLSLConstructorExpression*)expr;
      return NeedsFlattening(constructorExpr->argument, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_MemberAccess) {
      return NeedsFlattening(expr->nextExpression, level + 1);
    }
    else if (expr->nodeType == HLSLNodeType_ArrayAccess) {
      HLSLArrayAccess* arrayAccess = (HLSLArrayAccess*)expr;
      return NeedsFlattening(arrayAccess->array, level + 1) || NeedsFlattening(arrayAccess->index, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else if (expr->nodeType == HLSLNodeType_FunctionCall) {
      HLSLFunctionCall* functionCall = (HLSLFunctionCall*)expr;
      if (functionCall->function->numOutputArguments > 0) {
        if (level > 0) {
          return true;
        }
      }
      return NeedsFlattening(functionCall->argument, level + 1) || NeedsFlattening(expr->nextExpression, level);
    }
    else {
      //assert(false);
      return false;
    }
  }


  struct StatementList {
    HLSLStatement* head = nullptr;
    HLSLStatement* tail = nullptr;
    void append(HLSLStatement* st) {
      if (head == nullptr) {
        tail = head = st;
      }
      tail->nextStatement = st;
      tail = st;
    }
  };


  class ExpressionFlattener : public HLSLTreeVisitor
  {
  public:
    HLSLTree* m_tree;
    int tmp_index;
    HLSLStatement** statement_pointer;
    HLSLFunction* current_function;

    ExpressionFlattener()
    {
      m_tree = nullptr;
      tmp_index = 0;
      statement_pointer = nullptr;
      current_function = nullptr;
    }

    void FlattenExpressions(HLSLTree* tree)
    {
      m_tree = tree;
      VisitRoot(tree->GetRoot());
    }

    // Visit all statements updating the statement_pointer so that we can insert and replace statements. @@ Add this to the default visitor?
    virtual void VisitFunction(HLSLFunction* node) override
    {
      current_function = node;
      statement_pointer = &node->statement;
      VisitStatements(node->statement);
      statement_pointer = nullptr;
      current_function = nullptr;
    }

    virtual void VisitIfStatement(HLSLIfStatement* node) override
    {
      if (NeedsFlattening(node->condition, 1)) {
        assert(false);  // @@ Add statements before if statement.
      }

      statement_pointer = &node->statement;
      VisitStatements(node->statement);
      if (node->elseStatement) {
        statement_pointer = &node->elseStatement;
        VisitStatements(node->elseStatement);
      }
    }

    virtual void VisitForStatement(HLSLForStatement* node) override
    {
      if (NeedsFlattening(node->initialization->assignment, 1)) {
        assert(false);  // @@ Add statements before for statement.
      }
      if (NeedsFlattening(node->condition, 1) || NeedsFlattening(node->increment, 1)) {
        assert(false);  // @@ These are tricky to implement. Need to handle all loop exits.
      }

      statement_pointer = &node->statement;
      VisitStatements(node->statement);
    }

    virtual void VisitBlockStatement(HLSLBlockStatement* node) override
    {
      statement_pointer = &node->statement;
      VisitStatements(node->statement);
    }

    virtual void VisitStatements(HLSLStatement* statement) override
    {
      while (statement != nullptr) {
        VisitStatement(statement);
        statement_pointer = &statement->nextStatement;
        statement = statement->nextStatement;
      }
    }

    // This is usually a function call or assignment.
    virtual void VisitExpressionStatement(HLSLExpressionStatement* node) override
    {
      if (NeedsFlattening(node->expression, 0))
      {
        StatementList statements;
        Flatten(node->expression, statements, false);

        // Link beginning of statement list.
        *statement_pointer = statements.head;

        // Link end of statement list.
        HLSLStatement* tail = statements.tail;
        tail->nextStatement = node->nextStatement;

        // Update statement pointer.
        statement_pointer = &tail->nextStatement;

        // @@ Delete node?
      }
    }

    virtual void VisitDeclaration(HLSLDeclaration* node) override
    {
      // Skip global declarations.
      if (statement_pointer == nullptr) return;

      if (NeedsFlattening(node->assignment, 1))
      {
        StatementList statements;
        HLSLIdentifierExpression* ident = Flatten(node->assignment, statements, true);

        // @@ Delete node->assignment?

        node->assignment = ident;
        statements.append(node);

        // Link beginning of statement list.
        *statement_pointer = statements.head;

        // Link end of statement list.
        HLSLStatement* tail = statements.tail;
        tail->nextStatement = node->nextStatement;

        // Update statement pointer.
        statement_pointer = &tail->nextStatement;
      }
    }

    virtual void VisitReturnStatement(HLSLReturnStatement* node) override
    {
      if (NeedsFlattening(node->expression, 1))
      {
        StatementList statements;
        HLSLIdentifierExpression* ident = Flatten(node->expression, statements, true);

        // @@ Delete node->expression?

        node->expression = ident;
        statements.append(node);

        // Link beginning of statement list.
        *statement_pointer = statements.head;

        // Link end of statement list.
        HLSLStatement* tail = statements.tail;
        tail->nextStatement = node->nextStatement;

        // Update statement pointer.
        statement_pointer = &tail->nextStatement;
      }
    }


    HLSLDeclaration* BuildTemporaryDeclaration(HLSLExpression* expr)
    {
      assert(expr->expressionType.baseType != HLSLBaseType_Void);

      HLSLDeclaration* declaration = m_tree->AddNode<HLSLDeclaration>(expr->fileName, expr->line);
      declaration->name = m_tree->AddStringFormat("tmp%d", tmp_index++);
      declaration->type = expr->expressionType;
      declaration->assignment = expr;

      return declaration;
    }

    HLSLExpressionStatement* BuildExpressionStatement(HLSLExpression* expr)
    {
      HLSLExpressionStatement* statement = m_tree->AddNode<HLSLExpressionStatement>(expr->fileName, expr->line);
      statement->expression = expr;
      return statement;
    }

    HLSLIdentifierExpression* AddExpressionStatement(HLSLExpression* expr, StatementList& statements, bool wantIdent)
    {
      if (wantIdent) {
        HLSLDeclaration* declaration = BuildTemporaryDeclaration(expr);
        statements.append(declaration);

        HLSLIdentifierExpression* ident = m_tree->AddNode<HLSLIdentifierExpression>(expr->fileName, expr->line);
        ident->name = declaration->name;
        ident->expressionType = declaration->type;
        return ident;
      }
      else {
        HLSLExpressionStatement* statement = BuildExpressionStatement(expr);
        statements.append(statement);
        return nullptr;
      }
    }

    HLSLIdentifierExpression* Flatten(HLSLExpression* expr, StatementList& statements, bool wantIdent = true)
    {
      if (!NeedsFlattening(expr, wantIdent)) {
        return AddExpressionStatement(expr, statements, wantIdent);
      }

      if (expr->nodeType == HLSLNodeType_UnaryExpression) {
        assert(expr->nextExpression == nullptr);

        HLSLUnaryExpression* unaryExpr = (HLSLUnaryExpression*)expr;

        HLSLIdentifierExpression* tmp = Flatten(unaryExpr->expression, statements, true);

        HLSLUnaryExpression* newUnaryExpr = m_tree->AddNode<HLSLUnaryExpression>(unaryExpr->fileName, unaryExpr->line);
        newUnaryExpr->unaryOp = unaryExpr->unaryOp;
        newUnaryExpr->expression = tmp;
        newUnaryExpr->expressionType = unaryExpr->expressionType;

        return AddExpressionStatement(newUnaryExpr, statements, wantIdent);
      }
      else if (expr->nodeType == HLSLNodeType_BinaryExpression) {
        assert(expr->nextExpression == nullptr);

        HLSLBinaryExpression* binaryExpr = (HLSLBinaryExpression*)expr;

        if (IsAssignOp(binaryExpr->binaryOp)) {
          // Flatten right hand side only.
          HLSLIdentifierExpression* tmp2 = Flatten(binaryExpr->expression2, statements, true);

          HLSLBinaryExpression* newBinaryExpr = m_tree->AddNode<HLSLBinaryExpression>(binaryExpr->fileName, binaryExpr->line);
          newBinaryExpr->binaryOp = binaryExpr->binaryOp;
          newBinaryExpr->expression1 = binaryExpr->expression1;
          newBinaryExpr->expression2 = tmp2;
          newBinaryExpr->expressionType = binaryExpr->expressionType;

          return AddExpressionStatement(newBinaryExpr, statements, wantIdent);
        }
        else {
          HLSLIdentifierExpression* tmp1 = Flatten(binaryExpr->expression1, statements, true);
          HLSLIdentifierExpression* tmp2 = Flatten(binaryExpr->expression2, statements, true);

          HLSLBinaryExpression* newBinaryExpr = m_tree->AddNode<HLSLBinaryExpression>(binaryExpr->fileName, binaryExpr->line);
          newBinaryExpr->binaryOp = binaryExpr->binaryOp;
          newBinaryExpr->expression1 = tmp1;
          newBinaryExpr->expression2 = tmp2;
          newBinaryExpr->expressionType = binaryExpr->expressionType;

          return AddExpressionStatement(newBinaryExpr, statements, wantIdent);
        }
      }
      else if (expr->nodeType == HLSLNodeType_ConditionalExpression) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_CastingExpression) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_LiteralExpression) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_IdentifierExpression) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_ConstructorExpression) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_MemberAccess) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_ArrayAccess) {
        assert(false);
      }
      else if (expr->nodeType == HLSLNodeType_FunctionCall) {
        HLSLFunctionCall* functionCall = (HLSLFunctionCall*)expr;

        // @@ Output function as is?
        // @@ We have to flatten function arguments! This is tricky, need to handle input/output arguments.
        assert(!NeedsFlattening(functionCall->argument));

        return AddExpressionStatement(expr, statements, wantIdent);
      }
      else {
        assert(false);
      }
      return nullptr;
    }
  };


  void FlattenExpressions(HLSLTree* tree) {
    ExpressionFlattener flattener;
    flattener.FlattenExpressions(tree);
  }

} // M4

namespace M4
{

  /** In addition to the values in this enum, all of the ASCII characters are
  valid tokens. */
  enum HLSLToken
  {
    // Built-in types.
    HLSLToken_Float = 256,
    HLSLToken_Float1,
    HLSLToken_Float1x1,
    HLSLToken_Float2,
    HLSLToken_Float2x1,
    HLSLToken_Float3,
    HLSLToken_Float3x1,
    HLSLToken_Float4,
    HLSLToken_Float4x1,
    HLSLToken_Float2x4,
    HLSLToken_Float2x3,
    HLSLToken_Float2x2,
    HLSLToken_Float3x4,
    HLSLToken_Float3x3,
    HLSLToken_Float3x2,
    HLSLToken_Float4x4,
    HLSLToken_Float4x3,
    HLSLToken_Float4x2,

    HLSLToken_Half,
    HLSLToken_Half1,
    HLSLToken_Half1x1,
    HLSLToken_Half2,
    HLSLToken_Half2x1,
    HLSLToken_Half3,
    HLSLToken_Half3x1,
    HLSLToken_Half4,
    HLSLToken_Half4x1,
    HLSLToken_Half2x4,
    HLSLToken_Half2x3,
    HLSLToken_Half2x2,
    HLSLToken_Half3x4,
    HLSLToken_Half3x3,
    HLSLToken_Half3x2,
    HLSLToken_Half4x4,
    HLSLToken_Half4x3,
    HLSLToken_Half4x2,

    HLSLToken_Double,
    HLSLToken_Double1,
    HLSLToken_Double1x1,
    HLSLToken_Double2,
    HLSLToken_Double2x1,
    HLSLToken_Double3,
    HLSLToken_Double3x1,
    HLSLToken_Double4,
    HLSLToken_Double4x1,
    HLSLToken_Double2x4,
    HLSLToken_Double2x3,
    HLSLToken_Double2x2,
    HLSLToken_Double3x4,
    HLSLToken_Double3x3,
    HLSLToken_Double3x2,
    HLSLToken_Double4x4,
    HLSLToken_Double4x3,
    HLSLToken_Double4x2,

    HLSLToken_Bool,
    HLSLToken_Bool2,
    HLSLToken_Bool3,
    HLSLToken_Bool4,
    HLSLToken_Int,
    HLSLToken_Int2,
    HLSLToken_Int3,
    HLSLToken_Int4,
    HLSLToken_Uint,
    HLSLToken_Uint2,
    HLSLToken_Uint3,
    HLSLToken_Uint4,
    HLSLToken_Texture,
    HLSLToken_Sampler,
    HLSLToken_Sampler2D,
    HLSLToken_Sampler3D,
    HLSLToken_SamplerCube,
    HLSLToken_Sampler2DShadow,
    HLSLToken_Sampler2DMS,
    HLSLToken_Sampler2DArray,

    // Reserved words.
    HLSLToken_If,
    HLSLToken_Else,
    HLSLToken_For,
    HLSLToken_While,
    HLSLToken_Break,
    HLSLToken_True,
    HLSLToken_False,
    HLSLToken_Void,
    HLSLToken_Struct,
    HLSLToken_CBuffer,
    HLSLToken_TBuffer,
    HLSLToken_Register,
    HLSLToken_Return,
    HLSLToken_Continue,
    HLSLToken_Discard,
    HLSLToken_Const,
    HLSLToken_Static,
    HLSLToken_Inline,
    HLSLToken_PreprocessorDefine,
    HLSLToken_PreprocessorIf,
    HLSLToken_PreprocessorElse,
    HLSLToken_PreprocessorEndif,

    // Input modifiers.
    HLSLToken_Uniform,
    HLSLToken_In,
    HLSLToken_Out,
    HLSLToken_InOut,

    // Effect keywords.
    HLSLToken_SamplerState,
    HLSLToken_Technique,
    HLSLToken_Pass,

    // Multi-character symbols.
    HLSLToken_LessEqual,
    HLSLToken_GreaterEqual,
    HLSLToken_EqualEqual,
    HLSLToken_NotEqual,
    HLSLToken_PlusPlus,
    HLSLToken_MinusMinus,
    HLSLToken_PlusEqual,
    HLSLToken_MinusEqual,
    HLSLToken_TimesEqual,
    HLSLToken_DivideEqual,
    HLSLToken_AndAnd,       // &&
    HLSLToken_BarBar,       // ||

    // Other token types.
    HLSLToken_FloatLiteral,
    HLSLToken_IntLiteral,
    HLSLToken_Identifier,

    HLSLToken_EndOfLine,

    HLSLToken_EndOfStream,
  };


  /// Maximum string length of an identifier.
#define s_maxIdentifier 256

  class HLSLTokenizer
  {

  public:
    HLSLTokenizer() { }

    /** The file name is only used for error reporting. */
    HLSLTokenizer(const char* fileName, const char* buffer, size_t length);

    /** Advances to the next token in the stream. */
    void Next(const bool EOLSkipping = true);

    /** Returns the current token in the stream. */
    int GetToken() const;

    /** Returns the number of the current token. */
    float GetFloat() const;
    int   GetInt() const;

    /** Returns the identifier for the current token. */
    const char* GetIdentifier() const;

    /** Returns the line number where the current token began. */
    int GetLineNumber() const;

    /** Returns the file name where the current token began. */
    const char* GetFileName() const;

    /** Gets a human readable text description of the current token. */
    void GetTokenName(char buffer[s_maxIdentifier]) const;

    /** Reports an error using printf style formatting. The current line number
    is included. Only the first error reported will be output. */
    void Error(const char* format, ...);

    /** Gets a human readable text description of the specified token. */
    static void GetTokenName(int token, char buffer[s_maxIdentifier]);

    /** Returns true if the next caracterer is a whitespace. */
    bool NextIsWhitespace();

    const char* getLastPos(const bool trimmed);
    const char* getCurrentPos() { return m_buffer; }

    void ReturnToPos(const char* pos);

  private:

    bool SkipWhitespace(const bool EOLSkipping);
    bool SkipComment(const char** buffer, const bool EOLSkipping);
    bool SkipPragmaDirective();
    bool ScanNumber();
    bool ScanLineDirective();

  private:

    const char* m_fileName;
    const char* m_buffer;
    const char* m_bufferPrevious;
    const char* m_bufferEnd;
    int                 m_lineNumber;
    bool                m_error;

    int                 m_token;
    float               m_fValue;
    int                 m_iValue;
    char                m_identifier[s_maxIdentifier];
    char                m_lineDirectiveFileName[s_maxIdentifier];
    int                 m_tokenLineNumber;

  };

}


namespace M4
{
  // The order here must match the order in the Token enum.
  static const char* _reservedWords[] =
  {
      "float",
      "float1",
      "float1x1",
      "float2",
      "float2x1",
      "float3",
      "float3x1",
      "float4",
      "float4x1",
      "float2x4",
      "float2x3",
  "float2x2",
      "float3x4",
      "float3x3",
      "float3x2",
      "float4x4",
      "float4x3",
      "float4x2",

      "half",
      "half1",
      "half1x1",
      "half2",
      "half2x1",
      "half3",
      "half3x1",
      "half4",
      "half4x1",
      "half2x4",
      "half2x3",
  "half2x2",
      "half3x4",
      "half3x3",
      "half3x2",
      "half4x4",
      "half4x3",
      "half4x2",

      "double",
      "double1",
      "double1x1",
      "double2",
      "double2x1",
      "double3",
      "double3x1",
      "double4",
      "double4x1",
      "double2x4",
      "double2x3",
      "double2x2",
      "double3x4",
      "double3x3",
      "double3x2",
      "double4x4",
      "double4x3",
      "double4x2",

      "bool",
  "bool2",
  "bool3",
  "bool4",
      "int",
      "int2",
      "int3",
      "int4",
      "uint",
      "uint2",
      "uint3",
      "uint4",
      "texture",
      "sampler",
      "sampler2D",
      "sampler3D",
      "samplerCUBE",
      "sampler2DShadow",
      "sampler2DMS",
      "sampler2DArray",
      "if",
      "else",
      "for",
      "while",
      "break",
      "true",
      "false",
      "void",
      "struct",
      "cbuffer",
      "tbuffer",
      "register",
      "return",
      "continue",
      "discard",
      "const",
      "static",
      "inline",
      "#define",
      "#if",
      "#else",
      "#endif",
      "uniform",
      "in",
      "out",
      "inout",
      "sampler_state",
      "technique",
      "pass",
  };

  static bool GetIsSymbol(char c)
  {
    switch (c)
    {
    case ';':
    case ':':
    case '(': case ')':
    case '[': case ']':
    case '{': case '}':
    case '-': case '+':
    case '*': case '/': case '%':
    case '?':
    case '!':
    case ',':
    case '=':
    case '.':
    case '<': case '>':
    case '|': case '&': case '^': case '~':
    case '@':
      return true;
    }
    return false;
  }

  /** Returns true if the character is a valid token separator at the end of a number type token */
  static bool GetIsNumberSeparator(char c)
  {
    return c == 0 || isspace(c) || GetIsSymbol(c);
  }

  HLSLTokenizer::HLSLTokenizer(const char* fileName, const char* buffer, size_t length)
  {
    m_buffer = buffer;
    m_bufferPrevious = m_buffer;
    m_bufferEnd = buffer + length;
    m_fileName = fileName;
    m_lineNumber = 1;
    m_tokenLineNumber = 1;
    m_error = false;
    Next();
  }

  void HLSLTokenizer::Next(const bool EOLSkipping)
  {
    m_bufferPrevious = m_buffer;

    while (SkipWhitespace(EOLSkipping) || SkipComment(&m_buffer, EOLSkipping) || ScanLineDirective() || SkipPragmaDirective())
    {
    }

    if (m_error)
    {
      m_token = HLSLToken_EndOfStream;
      return;
    }

    if (!EOLSkipping && m_buffer[0] == '\n')
    {
      m_token = HLSLToken_EndOfLine;
      return;
    }

    m_tokenLineNumber = m_lineNumber;

    if (m_buffer >= m_bufferEnd || *m_buffer == '\0')
    {
      m_token = HLSLToken_EndOfStream;
      return;
    }

    const char* start = m_buffer;

    // +=, -=, *=, /=, ==, <=, >=
    if (m_buffer[0] == '+' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_PlusEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '-' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_MinusEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '*' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_TimesEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '/' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_DivideEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '=' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_EqualEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '!' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_NotEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '<' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_LessEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '>' && m_buffer[1] == '=')
    {
      m_token = HLSLToken_GreaterEqual;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '&' && m_buffer[1] == '&')
    {
      m_token = HLSLToken_AndAnd;
      m_buffer += 2;
      return;
    }
    else if (m_buffer[0] == '|' && m_buffer[1] == '|')
    {
      m_token = HLSLToken_BarBar;
      m_buffer += 2;
      return;
    }

    // ++, --
    if ((m_buffer[0] == '-' || m_buffer[0] == '+') && (m_buffer[1] == m_buffer[0]))
    {
      m_token = (m_buffer[0] == '+') ? HLSLToken_PlusPlus : HLSLToken_MinusMinus;
      m_buffer += 2;
      return;
    }

    // Check for the start of a number.
    if (ScanNumber())
    {
      return;
    }

    if (GetIsSymbol(m_buffer[0]))
    {
      m_token = static_cast<unsigned char>(m_buffer[0]);
      ++m_buffer;
      return;
    }

    // Must be an identifier or a reserved word.
    while (m_buffer < m_bufferEnd && m_buffer[0] != 0 && !GetIsSymbol(m_buffer[0]) && !isspace(m_buffer[0]))
    {
      ++m_buffer;
    }

    size_t length = m_buffer - start;
    memcpy(m_identifier, start, length);
    m_identifier[length] = 0;

    const int numReservedWords = sizeof(_reservedWords) / sizeof(const char*);
    for (int i = 0; i < numReservedWords; ++i)
    {
      if (strcmp(_reservedWords[i], m_identifier) == 0)
      {
        m_token = 256 + i;
        return;
      }
    }

    m_token = HLSLToken_Identifier;

  }

  bool HLSLTokenizer::NextIsWhitespace()
  {
    return isspace(m_buffer[0]);
  }

  const char* HLSLTokenizer::getLastPos(const bool trimmed)
  {
    const char* start = m_bufferPrevious;

    if (trimmed)
    {
      // Skip white space
      while (isspace(start[0])) {
        start++;
      }
    }

    return start;
  }


  bool HLSLTokenizer::SkipWhitespace(const bool EOLSkipping)
  {
    bool result = false;
    while (m_buffer < m_bufferEnd && isspace(m_buffer[0]) &&
      (EOLSkipping || (!EOLSkipping && m_buffer[0] != '\n')))
    {
      result = true;
      if (m_buffer[0] == '\n')
      {
        ++m_lineNumber;
      }
      ++m_buffer;
    }
    return result;
  }

  bool HLSLTokenizer::SkipComment(const char** buffer, const bool EOLSkipping)
  {
    bool result = false;
    if ((*buffer)[0] == '/')
    {
      if ((*buffer)[1] == '/')
      {
        // Single line comment.
        result = true;
        *buffer += 2;
        while (*buffer < m_bufferEnd)
        {
          if ((*buffer)[0] == '\n')
          {
            ++m_lineNumber;
            if (EOLSkipping)
            {
              ++*buffer;
            }
            break;
          }
          ++ * buffer;
        }
      }
      else if ((*buffer)[1] == '*')
      {
        // Multi-line comment.
        result = true;
        *buffer += 2;
        while (*buffer < m_bufferEnd)
        {
          if ((*buffer)[0] == '\n')
          {
            ++m_lineNumber;
          }
          if ((*buffer)[0] == '*' && (*buffer)[1] == '/')
          {
            break;
          }
          ++ * buffer;
        }
        if (*buffer < m_bufferEnd)
        {
          *buffer += 2;
        }
      }
    }
    return result;
  }

  bool HLSLTokenizer::SkipPragmaDirective()
  {
    bool result = false;
    if (m_bufferEnd - m_buffer > 7 && *m_buffer == '#')
    {
      const char* ptr = m_buffer + 1;
      while (isspace(*ptr))
        ptr++;

      if (strncmp(ptr, "pragma", 6) == 0 && isspace(ptr[6]))
      {
        m_buffer = ptr + 6;
        result = true;
        while (m_buffer < m_bufferEnd)
        {
          if (*(m_buffer++) == '\n')
          {
            ++m_lineNumber;
            break;
          }
        }
      }
    }
    return result;
  }

  bool HLSLTokenizer::ScanNumber()
  {

    // Don't treat the + or - as part of the number.
    if (m_buffer[0] == '+' || m_buffer[0] == '-')
    {
      return false;
    }

    // Parse hex literals.
    if (m_bufferEnd - m_buffer > 2 && m_buffer[0] == '0' && m_buffer[1] == 'x')
    {
      char* hEnd = nullptr;
      int     iValue = strtol(m_buffer + 2, &hEnd, 16);
      if (GetIsNumberSeparator(hEnd[0]))
      {
        m_buffer = hEnd;
        m_token = HLSLToken_IntLiteral;
        m_iValue = iValue;
        return true;
      }
    }

    char* fEnd = nullptr;
    double fValue = String_ToDouble(m_buffer, &fEnd);

    if (fEnd == m_buffer)
    {
      return false;
    }

    char* iEnd = nullptr;
    int    iValue = String_ToInteger(m_buffer, &iEnd);

    // If the character after the number is an f then the f is treated as part
    // of the number (to handle 1.0f syntax).
    if ((fEnd[0] == 'f' || fEnd[0] == 'h') && fEnd < m_bufferEnd)
    {
      ++fEnd;
    }

    if (fEnd > iEnd && GetIsNumberSeparator(fEnd[0]))
    {
      m_buffer = fEnd;
      m_token = HLSLToken_FloatLiteral;
      m_fValue = static_cast<float>(fValue);
      return true;
    }
    else if (iEnd > m_buffer && GetIsNumberSeparator(iEnd[0]))
    {
      m_buffer = iEnd;
      m_token = HLSLToken_IntLiteral;
      m_iValue = iValue;
      return true;
    }

    return false;
  }

  bool HLSLTokenizer::ScanLineDirective()
  {

    if (m_bufferEnd - m_buffer > 5 && strncmp(m_buffer, "#line", 5) == 0 && isspace(m_buffer[5]))
    {

      m_buffer += 5;

      while (m_buffer < m_bufferEnd && isspace(m_buffer[0]))
      {
        if (m_buffer[0] == '\n')
        {
          Error("Syntax error: expected line number after #line");
          return false;
        }
        ++m_buffer;
      }

      char* iEnd = nullptr;
      int lineNumber = String_ToInteger(m_buffer, &iEnd);

      if (!isspace(*iEnd))
      {
        Error("Syntax error: expected line number after #line");
        return false;
      }

      m_buffer = iEnd;
      while (m_buffer < m_bufferEnd && isspace(m_buffer[0]))
      {
        char c = m_buffer[0];
        ++m_buffer;
        if (c == '\n')
        {
          m_lineNumber = lineNumber;
          return true;
        }
      }

      if (m_buffer >= m_bufferEnd)
      {
        m_lineNumber = lineNumber;
        return true;
      }

      if (m_buffer[0] != '"')
      {
        Error("Syntax error: expected '\"' after line number near #line");
        return false;
      }

      ++m_buffer;

      int i = 0;
      while (i + 1 < s_maxIdentifier && m_buffer < m_bufferEnd && m_buffer[0] != '"')
      {
        if (m_buffer[0] == '\n')
        {
          Error("Syntax error: expected '\"' before end of line near #line");
          return false;
        }

        m_lineDirectiveFileName[i] = *m_buffer;
        ++m_buffer;
        ++i;
      }

      m_lineDirectiveFileName[i] = 0;

      if (m_buffer >= m_bufferEnd)
      {
        Error("Syntax error: expected '\"' before end of file near #line");
        return false;
      }

      if (i + 1 >= s_maxIdentifier)
      {
        Error("Syntax error: file name too long near #line");
        return false;
      }

      // Skip the closing quote
      ++m_buffer;

      while (m_buffer < m_bufferEnd && m_buffer[0] != '\n')
      {
        if (!isspace(m_buffer[0]))
        {
          Error("Syntax error: unexpected input after file name near #line");
          return false;
        }
        ++m_buffer;
      }

      // Skip new line
      ++m_buffer;

      m_lineNumber = lineNumber;
      m_fileName = m_lineDirectiveFileName;

      return true;

    }

    return false;

  }

  int HLSLTokenizer::GetToken() const
  {
    return m_token;
  }

  float HLSLTokenizer::GetFloat() const
  {
    return m_fValue;
  }

  int HLSLTokenizer::GetInt() const
  {
    return m_iValue;
  }

  const char* HLSLTokenizer::GetIdentifier() const
  {
    return m_identifier;
  }

  int HLSLTokenizer::GetLineNumber() const
  {
    return m_tokenLineNumber;
  }

  const char* HLSLTokenizer::GetFileName() const
  {
    return m_fileName;
  }

  void HLSLTokenizer::Error(const char* format, ...)
  {
    // It's not always convenient to stop executing when an error occurs,
    // so just track once we've hit an error and stop reporting them until
    // we successfully bail out of execution.
    if (m_error)
    {
      return;
    }
    m_error = true;


    char buffer[1024];
    va_list args;
    va_start(args, format);
    vsnprintf(buffer, sizeof(buffer) - 1, format, args);
    va_end(args);

    Log_Error("%s(%d) : %s\n", m_fileName, m_lineNumber, buffer);
  }

  void HLSLTokenizer::GetTokenName(char buffer[s_maxIdentifier]) const
  {
    if (m_token == HLSLToken_FloatLiteral)
    {
      sprintf(buffer, "%f", m_fValue);
    }
    else if (m_token == HLSLToken_IntLiteral)
    {
      sprintf(buffer, "%d", m_iValue);
    }
    else if (m_token == HLSLToken_Identifier)
    {
      strcpy(buffer, m_identifier);
    }
    else
    {
      GetTokenName(m_token, buffer);
    }
  }

  void HLSLTokenizer::GetTokenName(int token, char buffer[s_maxIdentifier])
  {
    if (token < 256)
    {
      buffer[0] = (char)token;
      buffer[1] = 0;
    }
    else if (token < HLSLToken_LessEqual)
    {
      strcpy(buffer, _reservedWords[token - 256]);
    }
    else
    {
      switch (token)
      {
      case HLSLToken_PlusPlus:
        strcpy(buffer, "++");
        break;
      case HLSLToken_MinusMinus:
        strcpy(buffer, "--");
        break;
      case HLSLToken_PlusEqual:
        strcpy(buffer, "+=");
        break;
      case HLSLToken_MinusEqual:
        strcpy(buffer, "-=");
        break;
      case HLSLToken_TimesEqual:
        strcpy(buffer, "*=");
        break;
      case HLSLToken_DivideEqual:
        strcpy(buffer, "/=");
        break;
      case HLSLToken_FloatLiteral:
        strcpy(buffer, "float");
        break;
      case HLSLToken_IntLiteral:
        strcpy(buffer, "int");
        break;
      case HLSLToken_Identifier:
        strcpy(buffer, "identifier");
        break;
      case HLSLToken_EndOfStream:
        strcpy(buffer, "<eof>");
        break;
      default:
        strcpy(buffer, "unknown");
        break;
      }
    }

  }

  void HLSLTokenizer::ReturnToPos(const char* pos)
  {
    m_buffer = pos;
  }


}

namespace M4
{

  struct EffectState;

  class HLSLParser
  {

  public:

    HLSLParser(Allocator* allocator, HLSLTree* tree);

    bool Parse(const char* fileName, const char* buffer, size_t length);
    bool ApplyPreprocessor(const char* fileName, const char* buffer, size_t length, std::string& sourcePreprocessed);

  private:

    bool Accept(int token);
    bool Expect(int token);

    /**
     * Special form of Accept for accepting a word that is not actually a token
     * but will be treated like one. This is useful for HLSL keywords that are
     * only tokens in specific contexts (like in/inout in parameter lists).
     */
    bool Accept(const char* token);
    bool Expect(const char* token);

    bool AcceptIdentifier(const char*& identifier);
    bool ExpectIdentifier(const char*& identifier);
    bool AcceptFloat(float& value);
    bool AcceptInt(int& value);
    bool AcceptType(bool allowVoid, HLSLType& type);
    bool ExpectType(bool allowVoid, HLSLType& type);
    bool AcceptBinaryOperator(int priority, HLSLBinaryOp& binaryOp);
    bool AcceptUnaryOperator(bool pre, HLSLUnaryOp& unaryOp);
    bool AcceptAssign(HLSLBinaryOp& binaryOp);
    bool AcceptTypeModifier(int& typeFlags);
    bool AcceptInterpolationModifier(int& flags);

    /**
     * Handles a declaration like: "float2 name[5]". If allowUnsizedArray is true, it is
     * is acceptable for the declaration to not specify the bounds of the array (i.e. name[]).
     */
    bool AcceptDeclaration(bool allowUnsizedArray, HLSLType& type, const char*& name);
    bool ExpectDeclaration(bool allowUnsizedArray, HLSLType& type, const char*& name);

    bool ParseTopLevel(HLSLStatement*& statement);
    bool ParseBlock(HLSLStatement*& firstStatement, const HLSLType& returnType);
    bool ParseStatementOrBlock(HLSLStatement*& firstStatement, const HLSLType& returnType, bool scoped = true);
    bool ParseStatement(HLSLStatement*& statement, const HLSLType& returnType);
    bool ParseDeclaration(HLSLDeclaration*& declaration);
    bool ParseFieldDeclaration(HLSLStructField*& field);
    //bool ParseBufferFieldDeclaration(HLSLBufferField*& field);
    bool ParseExpression(HLSLExpression*& expression);
    bool ParseBinaryExpression(int priority, HLSLExpression*& expression);
    bool ParseTerminalExpression(HLSLExpression*& expression, char& needsExpressionEndChar);
    bool ParseExpressionList(int endToken, bool allowEmptyEnd, HLSLExpression*& firstExpression, int& numExpressions);
    bool ParseArgumentList(HLSLArgument*& firstArgument, int& numArguments, int& numOutputArguments);
    bool ParseDeclarationAssignment(HLSLDeclaration* declaration);
    bool ParsePartialConstructor(HLSLExpression*& expression, HLSLBaseType type, const char* typeName);

    bool ParseStateName(bool isSamplerState, bool isPipelineState, const char*& name, const EffectState*& state);
    bool ParseColorMask(int& mask);
    bool ParseStateValue(const EffectState* state, HLSLStateAssignment* stateAssignment);
    bool ParseStateAssignment(HLSLStateAssignment*& stateAssignment, bool isSamplerState, bool isPipelineState);
    bool ParseSamplerState(HLSLExpression*& expression);
    bool ParseTechnique(HLSLStatement*& statement);
    bool ParsePass(HLSLPass*& pass);
    bool ParsePipeline(HLSLStatement*& pipeline);
    bool ParseStage(HLSLStatement*& stage);
    bool ParsePreprocessorDefine();

    bool ParseAttributeList(HLSLAttribute*& attribute);
    bool ParseAttributeBlock(HLSLAttribute*& attribute);

    bool CheckForUnexpectedEndOfStream(int endToken);

    const HLSLStruct* FindUserDefinedType(const char* name) const;

    void BeginScope();
    void EndScope();

    void DeclareVariable(const char* name, const HLSLType& type);

    /** Returned pointer is only valid until Declare or Begin/EndScope is called. */
    const HLSLType* FindVariable(const char* name, bool& global) const;

    const HLSLFunction* FindFunction(const char* name) const;
    const HLSLFunction* FindFunction(const HLSLFunction* fun) const;

    bool GetIsFunction(const char* name) const;

    /** Finds the overloaded function that matches the specified call. */
    const HLSLFunction* MatchFunctionCall(const HLSLFunctionCall* functionCall, const char* name);

    /** Gets the type of the named field on the specified object type (fieldName can also specify a swizzle. ) */
    bool GetMemberType(const HLSLType& objectType, HLSLMemberAccess* memberAccess);

    bool CheckTypeCast(const HLSLType& srcType, const HLSLType& dstType);

    const char* GetFileName();
    int GetLineNumber() const;

    bool ProcessMacroArguments(HLSLMacro* macro, std::string& sourcePreprocessed);
    HLSLMacro* ProcessMacroFromIdentifier(std::string& sourcePreprocessed, bool& addOriginalSource);

  private:

    struct Variable
    {
      const char* name;
      HLSLType        type;
    };

    HLSLTokenizer           m_tokenizer;
    Array<HLSLStruct*>      m_userTypes;
    Array<Variable>         m_variables;
    Array<HLSLFunction*>    m_functions;
    Array<HLSLMacro*>       m_macros;
    int                     m_numGlobals;

    HLSLTree* m_tree;

    bool                    m_allowUndeclaredIdentifiers = false;
    // not used bool                    m_disableSemanticValidation = false;
  };
}

namespace M4
{

  enum CompareFunctionsResult
  {
    FunctionsEqual,
    Function1Better,
    Function2Better
  };


  /** This structure stores a HLSLFunction-like declaration for an intrinsic function */
  struct Intrinsic
  {
    explicit Intrinsic(const char* name, HLSLBaseType returnType)
    {
      function.name = name;
      function.returnType.baseType = returnType;
      function.numArguments = 0;
    }
    explicit Intrinsic(const char* name, HLSLBaseType returnType, HLSLBaseType arg1)
    {
      function.name = name;
      function.returnType.baseType = returnType;
      function.numArguments = 1;
      function.argument = argument + 0;
      argument[0].type.baseType = arg1;
      argument[0].type.flags = HLSLTypeFlag_Const;
    }
    explicit Intrinsic(const char* name, HLSLBaseType returnType, HLSLBaseType arg1, HLSLBaseType arg2)
    {
      function.name = name;
      function.returnType.baseType = returnType;
      function.argument = argument + 0;
      function.numArguments = 2;
      argument[0].type.baseType = arg1;
      argument[0].type.flags = HLSLTypeFlag_Const;
      argument[0].nextArgument = argument + 1;
      argument[1].type.baseType = arg2;
      argument[1].type.flags = HLSLTypeFlag_Const;
    }
    explicit Intrinsic(const char* name, HLSLBaseType returnType, HLSLBaseType arg1, HLSLBaseType arg2, HLSLBaseType arg3)
    {
      function.name = name;
      function.returnType.baseType = returnType;
      function.argument = argument + 0;
      function.numArguments = 3;
      argument[0].type.baseType = arg1;
      argument[0].type.flags = HLSLTypeFlag_Const;
      argument[0].nextArgument = argument + 1;
      argument[1].type.baseType = arg2;
      argument[1].type.flags = HLSLTypeFlag_Const;
      argument[1].nextArgument = argument + 2;
      argument[2].type.baseType = arg3;
      argument[2].type.flags = HLSLTypeFlag_Const;
    }
    explicit Intrinsic(const char* name, HLSLBaseType returnType, HLSLBaseType arg1, HLSLBaseType arg2, HLSLBaseType arg3, HLSLBaseType arg4)
    {
      function.name = name;
      function.returnType.baseType = returnType;
      function.argument = argument + 0;
      function.numArguments = 4;
      argument[0].type.baseType = arg1;
      argument[0].type.flags = HLSLTypeFlag_Const;
      argument[0].nextArgument = argument + 1;
      argument[1].type.baseType = arg2;
      argument[1].type.flags = HLSLTypeFlag_Const;
      argument[1].nextArgument = argument + 2;
      argument[2].type.baseType = arg3;
      argument[2].type.flags = HLSLTypeFlag_Const;
      argument[2].nextArgument = argument + 3;
      argument[3].type.baseType = arg4;
      argument[3].type.flags = HLSLTypeFlag_Const;
    }
    HLSLFunction    function;
    HLSLArgument    argument[4];
  };

  Intrinsic SamplerIntrinsic(const char* name, HLSLBaseType returnType, HLSLBaseType arg1, HLSLBaseType samplerType, HLSLBaseType arg2)
  {
    Intrinsic i(name, returnType, arg1, arg2);
    i.argument[0].type.samplerType = samplerType;
    return i;
  }



  static const int _numberTypeRank[NumericType_Count][NumericType_Count] =
  {
    //F  B  I  U
    { 0, 4, 4, 4 },  // NumericType_Float
    { 5, 0, 5, 5 },  // NumericType_Bool
    { 2, 4, 0, 1 },  // NumericType_Int
    { 2, 4, 1, 0 }   // NumericType_Uint
  };


  struct EffectStateValue
  {
    const char* name;
    int value;
  };

  static const EffectStateValue textureFilteringValues[] = {
      {"None", 0},
      {"Point", 1},
      {"Linear", 2},
      {"Anisotropic", 3},
      {nullptr, 0}
  };

  static const EffectStateValue textureAddressingValues[] = {
      {"Wrap", 1},
      {"Mirror", 2},
      {"Clamp", 3},
      {"Border", 4},
      {"MirrorOnce", 5},
      {nullptr, 0}
  };

  static const EffectStateValue booleanValues[] = {
      {"False", 0},
      {"True", 1},
      {nullptr, 0}
  };

  static const EffectStateValue cullValues[] = {
      {"None", 1},
      {"CW", 2},
      {"CCW", 3},
      {nullptr, 0}
  };

  static const EffectStateValue cmpValues[] = {
      {"Never", 1},
      {"Less", 2},
      {"Equal", 3},
      {"LessEqual", 4},
      {"Greater", 5},
      {"NotEqual", 6},
      {"GreaterEqual", 7},
      {"Always", 8},
      {nullptr, 0}
  };

  static const EffectStateValue blendValues[] = {
      {"Zero", 1},
      {"One", 2},
      {"SrcColor", 3},
      {"InvSrcColor", 4},
      {"SrcAlpha", 5},
      {"InvSrcAlpha", 6},
      {"DestAlpha", 7},
      {"InvDestAlpha", 8},
      {"DestColor", 9},
      {"InvDestColor", 10},
      {"SrcAlphaSat", 11},
      {"BothSrcAlpha", 12},
      {"BothInvSrcAlpha", 13},
      {"BlendFactor", 14},
      {"InvBlendFactor", 15},
      {"SrcColor2", 16},          // Dual source blending. D3D9Ex only.
      {"InvSrcColor2", 17},
      {nullptr, 0}
  };

  static const EffectStateValue blendOpValues[] = {
      {"Add", 1},
      {"Subtract", 2},
      {"RevSubtract", 3},
      {"Min", 4},
      {"Max", 5},
      {nullptr, 0}
  };

  static const EffectStateValue fillModeValues[] = {
      {"Point", 1},
      {"Wireframe", 2},
      {"Solid", 3},
      {nullptr, 0}
  };

  static const EffectStateValue stencilOpValues[] = {
      {"Keep", 1},
      {"Zero", 2},
      {"Replace", 3},
      {"IncrSat", 4},
      {"DecrSat", 5},
      {"Invert", 6},
      {"Incr", 7},
      {"Decr", 8},
      {nullptr, 0}
  };

  // These are flags.
  static const EffectStateValue colorMaskValues[] = {
      {"False", 0},
      {"Red",   1 << 0},
      {"Green", 1 << 1},
      {"Blue",  1 << 2},
      {"Alpha", 1 << 3},
      {"X", 1 << 0},
      {"Y", 1 << 1},
      {"Z", 1 << 2},
      {"W", 1 << 3},
      {nullptr, 0}
  };

  static const EffectStateValue integerValues[] = {
      {nullptr, 0}
  };

  static const EffectStateValue floatValues[] = {
      {nullptr, 0}
  };


  struct EffectState
  {
    const char* name;
    int d3drs;
    const EffectStateValue* values;
  };

  static const EffectState samplerStates[] = {
      {"AddressU", 1, textureAddressingValues},
      {"AddressV", 2, textureAddressingValues},
      {"AddressW", 3, textureAddressingValues},
      // "BorderColor", 4, D3DCOLOR
      {"MagFilter", 5, textureFilteringValues},
      {"MinFilter", 6, textureFilteringValues},
      {"MipFilter", 7, textureFilteringValues},
      {"MipMapLodBias", 8, floatValues},
      {"MaxMipLevel", 9, integerValues},
      {"MaxAnisotropy", 10, integerValues},
      {"sRGBTexture", 11, booleanValues},
  };

  static const EffectState effectStates[] = {
      {"VertexShader", 0, nullptr},
      {"PixelShader", 0, nullptr},
      {"AlphaBlendEnable", 27, booleanValues},
      {"SrcBlend", 19, blendValues},
      {"DestBlend", 20, blendValues},
      {"BlendOp", 171, blendOpValues},
      {"SeparateAlphaBlendEanble", 206, booleanValues},
      {"SrcBlendAlpha", 207, blendValues},
      {"DestBlendAlpha", 208, blendValues},
      {"BlendOpAlpha", 209, blendOpValues},
      {"AlphaTestEnable", 15, booleanValues},
      {"AlphaRef", 24, integerValues},
      {"AlphaFunc", 25, cmpValues},
      {"CullMode", 22, cullValues},
      {"ZEnable", 7, booleanValues},
      {"ZWriteEnable", 14, booleanValues},
      {"ZFunc", 23, cmpValues},
      {"StencilEnable", 52, booleanValues},
      {"StencilFail", 53, stencilOpValues},
      {"StencilZFail", 54, stencilOpValues},
      {"StencilPass", 55, stencilOpValues},
      {"StencilFunc", 56, cmpValues},
      {"StencilRef", 57, integerValues},
      {"StencilMask", 58, integerValues},
      {"StencilWriteMask", 59, integerValues},
      {"TwoSidedStencilMode", 185, booleanValues},
      {"CCW_StencilFail", 186, stencilOpValues},
      {"CCW_StencilZFail", 187, stencilOpValues},
      {"CCW_StencilPass", 188, stencilOpValues},
      {"CCW_StencilFunc", 189, cmpValues},
      {"ColorWriteEnable", 168, colorMaskValues},
      {"FillMode", 8, fillModeValues},
      {"MultisampleAlias", 161, booleanValues},
      {"MultisampleMask", 162, integerValues},
      {"ScissorTestEnable", 174, booleanValues},
      {"SlopeScaleDepthBias", 175, floatValues},
      {"DepthBias", 195, floatValues}
  };


  static const EffectStateValue witnessCullModeValues[] = {
      {"None", 0},
      {"Back", 1},
      {"Front", 2},
      {nullptr, 0}
  };

  static const EffectStateValue witnessFillModeValues[] = {
      {"Solid", 0},
      {"Wireframe", 1},
      {nullptr, 0}
  };

  static const EffectStateValue witnessBlendModeValues[] = {
      {"Disabled", 0},
      {"AlphaBlend", 1},          // src * a + dst * (1-a)
      {"Add", 2},                 // src + dst
      {"Mixed", 3},               // src + dst * (1-a)
      {"Multiply", 4},            // src * dst
      {"Multiply2", 5},           // 2 * src * dst
      {nullptr, 0}
  };

  static const EffectStateValue witnessDepthFuncValues[] = {
      {"LessEqual", 0},
      {"Less", 1},
      {"Equal", 2},
      {"Greater", 3},
      {"Always", 4},
      {nullptr, 0}
  };

  static const EffectStateValue witnessStencilModeValues[] = {
      {"Disabled", 0},
      {"Set", 1},
      {"Test", 2},
      {nullptr, 0}
  };

  /* not used
  static const EffectStateValue witnessFilterModeValues[] = {
      {"Point", 0},
      {"Linear", 1},
      {"Mipmap_Nearest", 2},
      {"Mipmap_Best", 3},     // Quality of mipmap filtering depends on render settings.
      {"Anisotropic", 4},     // Aniso without mipmaps for reflection maps.
      {nullptr, 0}
  };

  static const EffectStateValue witnessWrapModeValues[] = {
      {"Repeat", 0},
      {"Clamp", 1},
      {"ClampToBorder", 2},
      {nullptr, 0}
  };
  */

  static const EffectState pipelineStates[] = {
      {"VertexShader", 0, nullptr},
      {"PixelShader", 0, nullptr},

      // Depth_Stencil_State
      {"DepthWrite", 0, booleanValues},
      {"DepthEnable", 0, booleanValues},
      {"DepthFunc", 0, witnessDepthFuncValues},
      {"StencilMode", 0, witnessStencilModeValues},

      // Raster_State
      {"CullMode", 0, witnessCullModeValues},
      {"FillMode", 0, witnessFillModeValues},
      {"MultisampleEnable", 0, booleanValues},
      {"PolygonOffset", 0, booleanValues},

      // Blend_State
      {"BlendMode", 0, witnessBlendModeValues},
      {"ColorWrite", 0, booleanValues},
      {"AlphaWrite", 0, booleanValues},
      {"AlphaTest", 0, booleanValues},       // This is really alpha to coverage.
  };





#define INTRINSIC_FLOAT1_FUNCTION(name) \
        Intrinsic( name, HLSLBaseType_Float,   HLSLBaseType_Float  ),   \
        Intrinsic( name, HLSLBaseType_Float2,  HLSLBaseType_Float2 ),   \
        Intrinsic( name, HLSLBaseType_Float3,  HLSLBaseType_Float3 ),   \
        Intrinsic( name, HLSLBaseType_Float4,  HLSLBaseType_Float4 )

#define INTRINSIC_FLOAT2_FUNCTION(name) \
        Intrinsic( name, HLSLBaseType_Float,   HLSLBaseType_Float,   HLSLBaseType_Float  ),   \
        Intrinsic( name, HLSLBaseType_Float2,  HLSLBaseType_Float2,  HLSLBaseType_Float2 ),   \
        Intrinsic( name, HLSLBaseType_Float3,  HLSLBaseType_Float3,  HLSLBaseType_Float3 ),   \
        Intrinsic( name, HLSLBaseType_Float4,  HLSLBaseType_Float4,  HLSLBaseType_Float4 )

#define INTRINSIC_FLOAT3_FUNCTION(name) \
        Intrinsic( name, HLSLBaseType_Float,   HLSLBaseType_Float,   HLSLBaseType_Float,  HLSLBaseType_Float ),   \
        Intrinsic( name, HLSLBaseType_Float2,  HLSLBaseType_Float2,  HLSLBaseType_Float2,  HLSLBaseType_Float2 ),  \
        Intrinsic( name, HLSLBaseType_Float3,  HLSLBaseType_Float3,  HLSLBaseType_Float3,  HLSLBaseType_Float3 ),  \
        Intrinsic( name, HLSLBaseType_Float4,  HLSLBaseType_Float4,  HLSLBaseType_Float4,  HLSLBaseType_Float4 )

#if 0
  // @@ IC: For some reason this is not working with the Visual Studio compiler:
#define SAMPLER_INTRINSIC_FUNCTION(name, sampler, arg1) \
        SamplerIntrinsic( name, HLSLBaseType_Float4, sampler, HLSLBaseType_Float, arg1)
#else
#define SAMPLER_INTRINSIC_FUNCTION(name, sampler, arg1) \
        Intrinsic( name, HLSLBaseType_Float4, sampler, arg1)
#endif

  const Intrinsic _intrinsic[] =
  {
      INTRINSIC_FLOAT1_FUNCTION("abs"),
      INTRINSIC_FLOAT1_FUNCTION("acos"),

      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float3),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float4),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float2x4),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float2x3),
  Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float2x2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float3x4),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float3x3),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float3x2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float4x4),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float4x3),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Float4x2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Bool),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Int),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Int2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Int3),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Int4),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Uint),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Uint2),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Uint3),
      Intrinsic("any", HLSLBaseType_Bool, HLSLBaseType_Uint4),

      INTRINSIC_FLOAT1_FUNCTION("asin"),
      INTRINSIC_FLOAT1_FUNCTION("atan"),
      INTRINSIC_FLOAT2_FUNCTION("atan2"),
      INTRINSIC_FLOAT3_FUNCTION("clamp"),
      INTRINSIC_FLOAT1_FUNCTION("cos"),
      INTRINSIC_FLOAT1_FUNCTION("tan"),

      INTRINSIC_FLOAT3_FUNCTION("lerp"),
      INTRINSIC_FLOAT3_FUNCTION("smoothstep"),

      INTRINSIC_FLOAT1_FUNCTION("floor"),
      INTRINSIC_FLOAT1_FUNCTION("ceil"),
      INTRINSIC_FLOAT1_FUNCTION("frac"),

      INTRINSIC_FLOAT2_FUNCTION("fmod"),

      Intrinsic("clip", HLSLBaseType_Void,  HLSLBaseType_Float),
      Intrinsic("clip", HLSLBaseType_Void,  HLSLBaseType_Float2),
      Intrinsic("clip", HLSLBaseType_Void,  HLSLBaseType_Float3),
      Intrinsic("clip", HLSLBaseType_Void,  HLSLBaseType_Float4),

      Intrinsic("dot", HLSLBaseType_Float,  HLSLBaseType_Float,   HLSLBaseType_Float),
      Intrinsic("dot", HLSLBaseType_Float,  HLSLBaseType_Float2,  HLSLBaseType_Float2),
      Intrinsic("dot", HLSLBaseType_Float,  HLSLBaseType_Float3,  HLSLBaseType_Float3),
      Intrinsic("dot", HLSLBaseType_Float,  HLSLBaseType_Float4,  HLSLBaseType_Float4),

      Intrinsic("cross", HLSLBaseType_Float3,  HLSLBaseType_Float3,  HLSLBaseType_Float3),

      Intrinsic("length", HLSLBaseType_Float,  HLSLBaseType_Float),
      Intrinsic("length", HLSLBaseType_Float,  HLSLBaseType_Float2),
      Intrinsic("length", HLSLBaseType_Float,  HLSLBaseType_Float3),
      Intrinsic("length", HLSLBaseType_Float,  HLSLBaseType_Float4),

      INTRINSIC_FLOAT2_FUNCTION("max"),
      INTRINSIC_FLOAT2_FUNCTION("min"),

      // @@ Add all combinations.
      // https://docs.microsoft.com/en-us/windows/desktop/direct3dhlsl/dx-graphics-hlsl-mul

      // scalar = mul(scalar, scalar)
      Intrinsic("mul", HLSLBaseType_Float, HLSLBaseType_Float, HLSLBaseType_Float),

      // vector<N> = mul(scalar, vector<N>)
      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float, HLSLBaseType_Float4),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float, HLSLBaseType_Float3),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float, HLSLBaseType_Float2),

      // matrix<N,M> = mul(scalar, matrix<M,N>)
      Intrinsic("mul", HLSLBaseType_Float4x4, HLSLBaseType_Float, HLSLBaseType_Float4x4),
      Intrinsic("mul", HLSLBaseType_Float4x3, HLSLBaseType_Float, HLSLBaseType_Float4x3),
      Intrinsic("mul", HLSLBaseType_Float4x2, HLSLBaseType_Float, HLSLBaseType_Float4x2),

      Intrinsic("mul", HLSLBaseType_Float3x4, HLSLBaseType_Float, HLSLBaseType_Float3x4),
      Intrinsic("mul", HLSLBaseType_Float3x3, HLSLBaseType_Float, HLSLBaseType_Float3x3),
      Intrinsic("mul", HLSLBaseType_Float3x2, HLSLBaseType_Float, HLSLBaseType_Float3x2),

      Intrinsic("mul", HLSLBaseType_Float2x4, HLSLBaseType_Float, HLSLBaseType_Float2x4),
      Intrinsic("mul", HLSLBaseType_Float2x3, HLSLBaseType_Float, HLSLBaseType_Float2x3),
      Intrinsic("mul", HLSLBaseType_Float2x2, HLSLBaseType_Float, HLSLBaseType_Float2x2),

      // vector<N> = mul(vector<N>, scalar)
      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float4, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float),

      // scalar = mul(vector<N>, vector<N>) it's a dot product !
      Intrinsic("mul", HLSLBaseType_Float, HLSLBaseType_Float4, HLSLBaseType_Float4),
      Intrinsic("mul", HLSLBaseType_Float, HLSLBaseType_Float3, HLSLBaseType_Float3),
      Intrinsic("mul", HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float2),

      // vector<M> = mul(vector<N>, matrix<N,M>)
      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float4, HLSLBaseType_Float4x4),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float4, HLSLBaseType_Float4x3),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float4, HLSLBaseType_Float4x2),

      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float3, HLSLBaseType_Float3x4),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float3x3),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3x2),

      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float2x4),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float2x3),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2x2),

      // matrix<N,M> = mul(matrix<N,M>,scalar)
      Intrinsic("mul", HLSLBaseType_Float4x4, HLSLBaseType_Float4x4, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float3x4, HLSLBaseType_Float3x4, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float2x4, HLSLBaseType_Float2x4, HLSLBaseType_Float),

      Intrinsic("mul", HLSLBaseType_Float4x3, HLSLBaseType_Float4x3, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float3x3, HLSLBaseType_Float3x3, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float2x3, HLSLBaseType_Float2x3, HLSLBaseType_Float),

      Intrinsic("mul", HLSLBaseType_Float4x2, HLSLBaseType_Float4x2, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float3x2, HLSLBaseType_Float3x2, HLSLBaseType_Float),
      Intrinsic("mul", HLSLBaseType_Float2x2, HLSLBaseType_Float2x2, HLSLBaseType_Float),

      // vector<N> = mul(matrix<N,M>, vector<M>)
      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float4x4, HLSLBaseType_Float4),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float3x4, HLSLBaseType_Float4),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float2x4, HLSLBaseType_Float4),

      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float4x3, HLSLBaseType_Float3),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float3x3, HLSLBaseType_Float3),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float2x3, HLSLBaseType_Float3),

      Intrinsic("mul", HLSLBaseType_Float4, HLSLBaseType_Float4x2, HLSLBaseType_Float2),
      Intrinsic("mul", HLSLBaseType_Float3, HLSLBaseType_Float3x2, HLSLBaseType_Float2),
      Intrinsic("mul", HLSLBaseType_Float2, HLSLBaseType_Float2x2, HLSLBaseType_Float2),

      // matrix<N,M> = mul(matrix<N,4>, matrix<4,M)
      Intrinsic("mul", HLSLBaseType_Float4x4, HLSLBaseType_Float4x4, HLSLBaseType_Float4x4),
      Intrinsic("mul", HLSLBaseType_Float4x3, HLSLBaseType_Float4x4, HLSLBaseType_Float4x3),
      Intrinsic("mul", HLSLBaseType_Float4x2, HLSLBaseType_Float4x4, HLSLBaseType_Float4x2),

      Intrinsic("mul", HLSLBaseType_Float3x4, HLSLBaseType_Float3x4, HLSLBaseType_Float4x4),
      Intrinsic("mul", HLSLBaseType_Float3x3, HLSLBaseType_Float3x4, HLSLBaseType_Float4x3),
      Intrinsic("mul", HLSLBaseType_Float3x2, HLSLBaseType_Float3x4, HLSLBaseType_Float4x2),

      Intrinsic("mul", HLSLBaseType_Float2x4, HLSLBaseType_Float2x4, HLSLBaseType_Float4x4),
      Intrinsic("mul", HLSLBaseType_Float2x3, HLSLBaseType_Float2x4, HLSLBaseType_Float4x3),
      Intrinsic("mul", HLSLBaseType_Float2x2, HLSLBaseType_Float2x4, HLSLBaseType_Float4x2),

      // matrix<N,M> = mul(matrix<N,3>, matrix<3,M>)
      Intrinsic("mul", HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float3x4),
      Intrinsic("mul", HLSLBaseType_Float4x3, HLSLBaseType_Float4x3, HLSLBaseType_Float3x3),
      Intrinsic("mul", HLSLBaseType_Float4x2, HLSLBaseType_Float4x3, HLSLBaseType_Float3x2),

      Intrinsic("mul", HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x4),
      Intrinsic("mul", HLSLBaseType_Float3x3, HLSLBaseType_Float3x3, HLSLBaseType_Float3x3),
      Intrinsic("mul", HLSLBaseType_Float3x2, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2),

      Intrinsic("mul", HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float3x4),
      Intrinsic("mul", HLSLBaseType_Float2x3, HLSLBaseType_Float2x3, HLSLBaseType_Float3x3),
      Intrinsic("mul", HLSLBaseType_Float2x2, HLSLBaseType_Float2x3, HLSLBaseType_Float3x2),

      // matrix<N,M> = mul(matrix<N,2>, matrix<2,M>)
      Intrinsic("mul", HLSLBaseType_Float4x4, HLSLBaseType_Float4x2, HLSLBaseType_Float2x4),
      Intrinsic("mul", HLSLBaseType_Float4x3, HLSLBaseType_Float4x2, HLSLBaseType_Float2x3),
      Intrinsic("mul", HLSLBaseType_Float4x2, HLSLBaseType_Float4x2, HLSLBaseType_Float2x2),

      Intrinsic("mul", HLSLBaseType_Float3x4, HLSLBaseType_Float3x2, HLSLBaseType_Float2x4),
      Intrinsic("mul", HLSLBaseType_Float3x3, HLSLBaseType_Float3x2, HLSLBaseType_Float2x3),
      Intrinsic("mul", HLSLBaseType_Float3x2, HLSLBaseType_Float3x2, HLSLBaseType_Float2x2),

      Intrinsic("mul", HLSLBaseType_Float2x4, HLSLBaseType_Float2x2, HLSLBaseType_Float2x4),
      Intrinsic("mul", HLSLBaseType_Float2x3, HLSLBaseType_Float2x2, HLSLBaseType_Float2x3),
      Intrinsic("mul", HLSLBaseType_Float2x2, HLSLBaseType_Float2x2, HLSLBaseType_Float2x2),


  Intrinsic("transpose", HLSLBaseType_Float2x2, HLSLBaseType_Float2x2),
      Intrinsic("transpose", HLSLBaseType_Float3x3, HLSLBaseType_Float3x3),
      Intrinsic("transpose", HLSLBaseType_Float4x4, HLSLBaseType_Float4x4),

      INTRINSIC_FLOAT2_FUNCTION("modf"),

      INTRINSIC_FLOAT1_FUNCTION("normalize"),
      INTRINSIC_FLOAT2_FUNCTION("pow"),
      INTRINSIC_FLOAT1_FUNCTION("saturate"),
      INTRINSIC_FLOAT1_FUNCTION("sin"),
      INTRINSIC_FLOAT1_FUNCTION("sqrt"),
      INTRINSIC_FLOAT1_FUNCTION("rsqrt"),
      INTRINSIC_FLOAT1_FUNCTION("rcp"),
      INTRINSIC_FLOAT1_FUNCTION("exp"),
      INTRINSIC_FLOAT1_FUNCTION("exp2"),
      INTRINSIC_FLOAT1_FUNCTION("log"),
      INTRINSIC_FLOAT1_FUNCTION("log2"),

      INTRINSIC_FLOAT1_FUNCTION("ddx"),
      INTRINSIC_FLOAT1_FUNCTION("ddy"),

      INTRINSIC_FLOAT1_FUNCTION("sign"),
      INTRINSIC_FLOAT2_FUNCTION("step"),
      INTRINSIC_FLOAT2_FUNCTION("reflect"),

  INTRINSIC_FLOAT1_FUNCTION("isnan"),
  INTRINSIC_FLOAT1_FUNCTION("isinf"),

  Intrinsic("asuint",    HLSLBaseType_Uint, HLSLBaseType_Float),

      SAMPLER_INTRINSIC_FUNCTION("tex2D", HLSLBaseType_Sampler2D, HLSLBaseType_Float2),

      Intrinsic("tex2Dproj", HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Float4),

      SAMPLER_INTRINSIC_FUNCTION("tex2Dlod", HLSLBaseType_Sampler2D, HLSLBaseType_Float4),

      Intrinsic("tex2Dlod",  HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Float4, HLSLBaseType_Int2),   // With offset.

      SAMPLER_INTRINSIC_FUNCTION("tex2Dbias", HLSLBaseType_Sampler2D, HLSLBaseType_Float4),

      Intrinsic("tex2Dgrad", HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2),
      Intrinsic("tex2Dgather", HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Float2, HLSLBaseType_Int),
      Intrinsic("tex2Dgather", HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Float2, HLSLBaseType_Int2, HLSLBaseType_Int),    // With offset.
      Intrinsic("tex2Dsize", HLSLBaseType_Int2, HLSLBaseType_Sampler2D),
      Intrinsic("tex2Dfetch", HLSLBaseType_Float4, HLSLBaseType_Sampler2D, HLSLBaseType_Int3),    // u,v,mipmap

      Intrinsic("tex2Dcmp", HLSLBaseType_Float4, HLSLBaseType_Sampler2DShadow, HLSLBaseType_Float4),                // @@ IC: This really takes a float3 (uvz) and returns a float.

      Intrinsic("tex2DMSfetch", HLSLBaseType_Float4, HLSLBaseType_Sampler2DMS, HLSLBaseType_Int2, HLSLBaseType_Int),
      Intrinsic("tex2DMSsize", HLSLBaseType_Int3, HLSLBaseType_Sampler2DMS),

      Intrinsic("tex2DArray", HLSLBaseType_Float4, HLSLBaseType_Sampler2DArray, HLSLBaseType_Float3),

      Intrinsic("tex3D",     HLSLBaseType_Float4, HLSLBaseType_Sampler3D, HLSLBaseType_Float3),
      Intrinsic("tex3Dlod",  HLSLBaseType_Float4, HLSLBaseType_Sampler3D, HLSLBaseType_Float4),
      Intrinsic("tex3Dbias", HLSLBaseType_Float4, HLSLBaseType_Sampler3D, HLSLBaseType_Float4),
      Intrinsic("tex3Dsize", HLSLBaseType_Int3, HLSLBaseType_Sampler3D),

      Intrinsic("texCUBE",       HLSLBaseType_Float4, HLSLBaseType_SamplerCube, HLSLBaseType_Float3),
      Intrinsic("texCUBElod", HLSLBaseType_Float4, HLSLBaseType_SamplerCube, HLSLBaseType_Float4),
      Intrinsic("texCUBEbias", HLSLBaseType_Float4, HLSLBaseType_SamplerCube, HLSLBaseType_Float4),
      Intrinsic("texCUBEsize", HLSLBaseType_Int, HLSLBaseType_SamplerCube),

      Intrinsic("sincos", HLSLBaseType_Void,  HLSLBaseType_Float,   HLSLBaseType_Float,  HLSLBaseType_Float),
      Intrinsic("sincos", HLSLBaseType_Void,  HLSLBaseType_Float2,  HLSLBaseType_Float,  HLSLBaseType_Float2),
      Intrinsic("sincos", HLSLBaseType_Void,  HLSLBaseType_Float3,  HLSLBaseType_Float,  HLSLBaseType_Float3),
      Intrinsic("sincos", HLSLBaseType_Void,  HLSLBaseType_Float4,  HLSLBaseType_Float,  HLSLBaseType_Float4),

      Intrinsic("mad", HLSLBaseType_Float, HLSLBaseType_Float, HLSLBaseType_Float, HLSLBaseType_Float),
      Intrinsic("mad", HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2),
      Intrinsic("mad", HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float3),
      Intrinsic("mad", HLSLBaseType_Float4, HLSLBaseType_Float4, HLSLBaseType_Float4, HLSLBaseType_Float4),
  };

  const int _numIntrinsics = sizeof(_intrinsic) / sizeof(Intrinsic);

  // The order in this array must match up with HLSLBinaryOp
  const int _binaryOpPriority[] =
  {
      2, 1, //  &&, ||
      8, 8, //  +,  -
      9, 9, //  *,  /
      7, 7, //  <,  >,
      7, 7, //  <=, >=,
      6, 6, //  ==, !=
      5, 3, 4, // &, |, ^
  };


  // IC: I'm not sure this table is right, but any errors should be caught by the backend compiler.
  // Also, this is operator dependent. The type resulting from (float4 * float4x4) is not the same as (float4 + float4x4).
  // We should probably distinguish between component-wise operator and only allow same dimensions
  HLSLBaseType _binaryOpTypeLookup[HLSLBaseType_NumericCount][HLSLBaseType_NumericCount] =
  {
      {   // float
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4
      },
      {   // float2
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2,
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2,
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2
      },
      {   // float3
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3,
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3,
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3
      },
      {   // float4
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4
      },
      {   // float2x4
          HLSLBaseType_Float2x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float2x3
          HLSLBaseType_Float2x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Float2x3, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float2x2
          HLSLBaseType_Float2x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Float2x2,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float2x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float3x4
          HLSLBaseType_Float3x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float3x3
          HLSLBaseType_Float3x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Float3x3, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float3x2
          HLSLBaseType_Float3x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Float3x2,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float3x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float4x4
          HLSLBaseType_Float4x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x4, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float4x3
          HLSLBaseType_Float4x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Float4x3, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x3, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
      {   // float4x2
          HLSLBaseType_Float4x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Float4x2,
          HLSLBaseType_Float4x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Float4x2, HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown
      },
       {   // bool
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // bool2
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // bool3
          HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Int3, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Int3, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint3, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // bool4
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // int
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int2,
          HLSLBaseType_Int, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // int2
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int2,
          HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int2, HLSLBaseType_Int2,
          HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2
      },
      {   // int3
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Int3, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int3,
          HLSLBaseType_Int3, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int3,
          HLSLBaseType_Uint3, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint3
      },
      {   // int4
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Int4, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Int4, HLSLBaseType_Int2, HLSLBaseType_Int3, HLSLBaseType_Int4,
          HLSLBaseType_Uint4, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // uint
          HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Float2x4, HLSLBaseType_Float2x3, HLSLBaseType_Float2x2,
          HLSLBaseType_Float3x4, HLSLBaseType_Float3x3, HLSLBaseType_Float3x2,
          HLSLBaseType_Float4x4, HLSLBaseType_Float4x3, HLSLBaseType_Float4x2,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4,
          HLSLBaseType_Uint, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
      {   // uint2
          HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2, HLSLBaseType_Float2,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2,
          HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2,
          HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2, HLSLBaseType_Uint2
      },
      {   // uint3
          HLSLBaseType_Float3, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float3,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Uint3, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint3,
          HLSLBaseType_Uint3, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint3,
          HLSLBaseType_Uint3, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint3
      },
      {   // uint4
          HLSLBaseType_Float4, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Unknown, HLSLBaseType_Unknown, HLSLBaseType_Unknown,
          HLSLBaseType_Uint4, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4,
          HLSLBaseType_Uint4, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4,
          HLSLBaseType_Uint4, HLSLBaseType_Uint2, HLSLBaseType_Uint3, HLSLBaseType_Uint4
      },
  };

  // Priority of the ? : operator.
  const int _conditionalOpPriority = 1;

  static const char* GetTypeName1(const HLSLType& type)
  {
    if (type.baseType == HLSLBaseType_UserDefined)
    {
      return type.typeName;
    }
    else
    {
      return baseTypeDescriptions[type.baseType].typeName;
    }
  }

  static const char* GetBinaryOpName(HLSLBinaryOp binaryOp)
  {
    switch (binaryOp)
    {
    case HLSLBinaryOp_And:          return "&&";
    case HLSLBinaryOp_Or:           return "||";
    case HLSLBinaryOp_Add:          return "+";
    case HLSLBinaryOp_Sub:          return "-";
    case HLSLBinaryOp_Mul:          return "*";
    case HLSLBinaryOp_Div:          return "/";
    case HLSLBinaryOp_Mod:          return "%";
    case HLSLBinaryOp_Less:         return "<";
    case HLSLBinaryOp_Greater:      return ">";
    case HLSLBinaryOp_LessEqual:    return "<=";
    case HLSLBinaryOp_GreaterEqual: return ">=";
    case HLSLBinaryOp_Equal:        return "==";
    case HLSLBinaryOp_NotEqual:     return "!=";
    case HLSLBinaryOp_BitAnd:       return "&";
    case HLSLBinaryOp_BitOr:        return "|";
    case HLSLBinaryOp_BitXor:       return "^";
    case HLSLBinaryOp_Assign:       return "=";
    case HLSLBinaryOp_AddAssign:    return "+=";
    case HLSLBinaryOp_SubAssign:    return "-=";
    case HLSLBinaryOp_MulAssign:    return "*=";
    case HLSLBinaryOp_DivAssign:    return "/=";
    default:
      ASSERT(0);
      return "???";
    }
  }


  /*
   * 1.) Match
   * 2.) Scalar dimension promotion (scalar -> vector/matrix)
   * 3.) Conversion
   * 4.) Conversion + scalar dimension promotion
   * 5.) Truncation (vector -> scalar or lower component vector, matrix -> scalar or lower component matrix)
   * 6.) Conversion + truncation
   */
  static int GetTypeCastRank(HLSLTree* tree, const HLSLType& srcType, const HLSLType& dstType)
  {
    /*if (srcType.array != dstType.array || srcType.arraySize != dstType.arraySize)
    {
        return -1;
    }*/

    if (srcType.array != dstType.array)
    {
      return -1;
    }

    if (srcType.array == true)
    {
      ASSERT(dstType.array == true);
      int srcArraySize = -1;
      int dstArraySize = -1;

      tree->GetExpressionValue(srcType.arraySize, srcArraySize);
      tree->GetExpressionValue(dstType.arraySize, dstArraySize);

      if (srcArraySize != dstArraySize) {
        return -1;
      }
    }

    if (srcType.baseType == HLSLBaseType_UserDefined && dstType.baseType == HLSLBaseType_UserDefined)
    {
      return strcmp(srcType.typeName, dstType.typeName) == 0 ? 0 : -1;
    }

    if (srcType.baseType == dstType.baseType)
    {
      if (IsSamplerType(srcType.baseType))
      {
        return srcType.samplerType == dstType.samplerType ? 0 : -1;
      }

      return 0;
    }

    const BaseTypeDescription& srcDesc = baseTypeDescriptions[srcType.baseType];
    const BaseTypeDescription& dstDesc = baseTypeDescriptions[dstType.baseType];
    if (srcDesc.numericType == NumericType_NaN || dstDesc.numericType == NumericType_NaN)
    {
      return -1;
    }

    // Result bits: T R R R P H C (T = truncation, R = conversion rank, P = dimension promotion, H = height promotion, C = component promotion)
    int result = _numberTypeRank[srcDesc.numericType][dstDesc.numericType] << 3;

    if (srcDesc.numDimensions == 0 && dstDesc.numDimensions > 0)
    {
      // Scalar dimension promotion
      result |= (1 << 2);
    }
    else if ((srcDesc.numDimensions == dstDesc.numDimensions && (srcDesc.numComponents > dstDesc.numComponents || srcDesc.height > dstDesc.height)) ||
      (srcDesc.numDimensions > 0 && dstDesc.numDimensions == 0))
    {
      // Truncation
      result |= (1 << 6);
    }
    else if (srcDesc.numDimensions != dstDesc.numDimensions)
    {
      // Can't convert
      return -1;
    }
    else if (srcDesc.height != dstDesc.height)
    {
      // Scalar height promotion
      result |= (1 << 1);
    }
    else if (srcDesc.numComponents != dstDesc.numComponents)
    {
      // Scalar components promotion
      result |= (1 << 0);
    }

    return result;

  }

  static bool GetFunctionCallCastRanks(HLSLTree* tree, const HLSLFunctionCall* call, const HLSLFunction* function, int* rankBuffer)
  {

    if (function == nullptr || function->numArguments < call->numArguments)
    {
      // Function not viable
      return false;
    }

    const HLSLExpression* expression = call->argument;
    const HLSLArgument* argument = function->argument;

    for (int i = 0; i < call->numArguments; ++i)
    {
      int rank = GetTypeCastRank(tree, expression->expressionType, argument->type);
      if (rank == -1)
      {
        return false;
      }

      rankBuffer[i] = rank;

      argument = argument->nextArgument;
      expression = expression->nextExpression;
    }

    for (int i = call->numArguments; i < function->numArguments; ++i)
    {
      if (argument->defaultValue == nullptr)
      {
        // Function not viable.
        return false;
      }
    }

    return true;

  }

  struct CompareRanks
  {
    bool operator() (const int& rank1, const int& rank2) { return rank1 > rank2; }
  };

  static CompareFunctionsResult CompareFunctions(HLSLTree* tree, const HLSLFunctionCall* call, const HLSLFunction* function1, const HLSLFunction* function2)
  {

#if defined WIN32 && !defined alloca
    int* function1Ranks = static_cast<int*>(_alloca(sizeof(int) * call->numArguments));
    int* function2Ranks = static_cast<int*>(_alloca(sizeof(int) * call->numArguments));
#else
    int* function1Ranks = static_cast<int*>(alloca(sizeof(int) * call->numArguments));
    int* function2Ranks = static_cast<int*>(alloca(sizeof(int) * call->numArguments));
#endif /** WIN32 */

    const bool function1Viable = GetFunctionCallCastRanks(tree, call, function1, function1Ranks);
    const bool function2Viable = GetFunctionCallCastRanks(tree, call, function2, function2Ranks);

    // Both functions have to be viable to be able to compare them
    if (!(function1Viable && function2Viable))
    {
      if (function1Viable)
      {
        return Function1Better;
      }
      else if (function2Viable)
      {
        return Function2Better;
      }
      else
      {
        return FunctionsEqual;
      }
    }

    std::sort(function1Ranks, function1Ranks + call->numArguments, CompareRanks());
    std::sort(function2Ranks, function2Ranks + call->numArguments, CompareRanks());

    for (int i = 0; i < call->numArguments; ++i)
    {
      if (function1Ranks[i] < function2Ranks[i])
      {
        return Function1Better;
      }
      else if (function2Ranks[i] < function1Ranks[i])
      {
        return Function2Better;
      }
    }

    //    // Dump matched function argument types
    //    std::string arg1;
    //    std::string arg2;
    //    const HLSLArgument* argument1 = function1->argument;
    //    const HLSLArgument* argument2 = function2->argument;
    //    for (int i = 0; i < call->numArguments; ++i)
    //    {
    //        arg1 += GetTypeName1(argument1->type);
    //        arg1 += " ";
    //        arg2 += GetTypeName1(argument2->type);
    //        arg2 += " ";

    //        argument1 = argument1->nextArgument;
    //        argument2 = argument2->nextArgument;
    //    }

    //    std::cout << "(" << arg1 << ") vs (" << arg2 << ")" << std::endl;


    return FunctionsEqual;

  }

  static bool GetBinaryOpResultType(HLSLBinaryOp binaryOp, const HLSLType& type1, const HLSLType& type2, HLSLType& result)
  {

    if (type1.baseType < HLSLBaseType_FirstNumeric || type1.baseType > HLSLBaseType_LastNumeric || type1.array ||
      type2.baseType < HLSLBaseType_FirstNumeric || type2.baseType > HLSLBaseType_LastNumeric || type2.array)
    {
      return false;
    }

    if (binaryOp == HLSLBinaryOp_BitAnd || binaryOp == HLSLBinaryOp_BitOr || binaryOp == HLSLBinaryOp_BitXor)
    {
      if (type1.baseType < HLSLBaseType_FirstInteger || type1.baseType > HLSLBaseType_LastInteger)
      {
        return false;
      }
    }

    switch (binaryOp)
    {
    case HLSLBinaryOp_And:
    case HLSLBinaryOp_Or:
    case HLSLBinaryOp_Less:
    case HLSLBinaryOp_Greater:
    case HLSLBinaryOp_LessEqual:
    case HLSLBinaryOp_GreaterEqual:
    case HLSLBinaryOp_Equal:
    case HLSLBinaryOp_NotEqual:
    {
      int numComponents = std::max(baseTypeDescriptions[type1.baseType].numComponents, baseTypeDescriptions[type2.baseType].numComponents);
      result.baseType = HLSLBaseType(HLSLBaseType_Bool + numComponents - 1);
      break;
    }
    case HLSLBinaryOp_Mod:
      result.baseType = HLSLBaseType_Int;
      break;

    default:
      result.baseType = _binaryOpTypeLookup[type1.baseType - HLSLBaseType_FirstNumeric][type2.baseType - HLSLBaseType_FirstNumeric];
      break;
    }

    result.typeName = nullptr;
    result.array = false;
    result.arraySize = nullptr;
    result.flags = (type1.flags & type2.flags) & HLSLTypeFlag_Const; // Propagate constness.

    return result.baseType != HLSLBaseType_Unknown;

  }

  HLSLParser::HLSLParser(Allocator* allocator, HLSLTree* tree) :
    m_userTypes(allocator),
    m_variables(allocator),
    m_functions(allocator),
    m_macros(allocator)
  {
    m_numGlobals = 0;
    m_tree = tree;
  }

  bool HLSLParser::Accept(int token)
  {
    if (m_tokenizer.GetToken() == token)
    {
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::Accept(const char* token)
  {
    if (m_tokenizer.GetToken() == HLSLToken_Identifier && String_Equal(token, m_tokenizer.GetIdentifier()))
    {
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::Expect(int token)
  {
    if (!Accept(token))
    {
      char want[s_maxIdentifier];
      m_tokenizer.GetTokenName(token, want);
      char near_[s_maxIdentifier];
      m_tokenizer.GetTokenName(near_);
      m_tokenizer.Error("Syntax error: expected '%s' near '%s'", want, near_);
      return false;
    }
    return true;
  }

  bool HLSLParser::Expect(const char* token)
  {
    if (!Accept(token))
    {
      const char* want = token;
      char near_[s_maxIdentifier];
      m_tokenizer.GetTokenName(near_);
      m_tokenizer.Error("Syntax error: expected '%s' near '%s'", want, near_);
      return false;
    }
    return true;
  }


  bool HLSLParser::AcceptIdentifier(const char*& identifier)
  {
    if (m_tokenizer.GetToken() == HLSLToken_Identifier)
    {
      identifier = m_tree->AddString(m_tokenizer.GetIdentifier());
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::ExpectIdentifier(const char*& identifier)
  {
    if (!AcceptIdentifier(identifier))
    {
      char near_[s_maxIdentifier];
      m_tokenizer.GetTokenName(near_);
      m_tokenizer.Error("Syntax error: expected identifier near '%s'", near_);
      identifier = "";
      return false;
    }
    return true;
  }

  bool HLSLParser::AcceptFloat(float& value)
  {
    if (m_tokenizer.GetToken() == HLSLToken_FloatLiteral)
    {
      value = m_tokenizer.GetFloat();
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::AcceptInt(int& value)
  {
    if (m_tokenizer.GetToken() == HLSLToken_IntLiteral)
    {
      value = m_tokenizer.GetInt();
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::ParseTopLevel(HLSLStatement*& statement)
  {
    HLSLAttribute* attributes = nullptr;
    ParseAttributeBlock(attributes);

    int line = GetLineNumber();
    const char* fileName = GetFileName();

    HLSLType type;
    //HLSLBaseType type;
    //const char*  typeName = nullptr;
    //int          typeFlags = false;

    bool doesNotExpectSemicolon = false;

    if (Accept(HLSLToken_Struct))
    {
      // Struct declaration.

      const char* structName = nullptr;
      if (!ExpectIdentifier(structName))
      {
        return false;
      }
      if (FindUserDefinedType(structName) != nullptr)
      {
        m_tokenizer.Error("struct %s already defined", structName);
        return false;
      }

      if (!Expect('{'))
      {
        return false;
      }

      HLSLStruct* structure = m_tree->AddNode<HLSLStruct>(fileName, line);
      structure->name = structName;

      m_userTypes.PushBack(structure);

      HLSLStructField* lastField = nullptr;

      // Add the struct to our list of user defined types.
      while (!Accept('}'))
      {
        if (CheckForUnexpectedEndOfStream('}'))
        {
          return false;
        }
        HLSLStructField* field = nullptr;
        if (!ParseFieldDeclaration(field))
        {
          return false;
        }
        ASSERT(field != nullptr);
        if (lastField == nullptr)
        {
          structure->field = field;
        }
        else
        {
          lastField->nextField = field;
        }
        lastField = field;
      }

      statement = structure;
    }
    else if (Accept(HLSLToken_CBuffer) || Accept(HLSLToken_TBuffer))
    {
      // cbuffer/tbuffer declaration.

      HLSLBuffer* buffer = m_tree->AddNode<HLSLBuffer>(fileName, line);
      AcceptIdentifier(buffer->name);

      // Optional register assignment.
      if (Accept(':'))
      {
        if (!Expect(HLSLToken_Register) || !Expect('(') || !ExpectIdentifier(buffer->registerName) || !Expect(')'))
        {
          return false;
        }
        // TODO: Check that we aren't re-using a register.
      }

      // Fields.
      if (!Expect('{'))
      {
        return false;
      }
      HLSLDeclaration* lastField = nullptr;
      while (!Accept('}'))
      {
        if (CheckForUnexpectedEndOfStream('}'))
        {
          return false;
        }
        HLSLDeclaration* field = nullptr;
        if (!ParseDeclaration(field))
        {
          m_tokenizer.Error("Expected variable declaration");
          return false;
        }
        DeclareVariable(field->name, field->type);
        field->buffer = buffer;
        if (buffer->field == nullptr)
        {
          buffer->field = field;
        }
        else
        {
          lastField->nextStatement = field;
        }
        lastField = field;

        if (!Expect(';')) {
          return false;
        }
      }

      statement = buffer;
    }
    else if (AcceptType(true, type))
    {
      // Global declaration (uniform or function).
      const char* globalName = nullptr;
      if (!ExpectIdentifier(globalName))
      {
        return false;
      }

      if (Accept('('))
      {
        // Function declaration.

        HLSLFunction* function = m_tree->AddNode<HLSLFunction>(fileName, line);
        function->name = globalName;
        function->returnType.baseType = type.baseType;
        function->returnType.typeName = type.typeName;
        function->attributes = attributes;

        BeginScope();

        if (!ParseArgumentList(function->argument, function->numArguments, function->numOutputArguments))
        {
          return false;
        }

        const HLSLFunction* declaration = FindFunction(function);

        // Forward declaration
        if (Accept(';'))
        {
          // Add a function entry so that calls can refer to it
          if (!declaration)
          {
            m_functions.PushBack(function);
            statement = function;
          }
          EndScope();
          return true;
        }

        // Optional semantic.
        if (Accept(':') && !ExpectIdentifier(function->semantic))
        {
          return false;
        }

        if (declaration)
        {
          if (declaration->forward || declaration->statement)
          {
            m_tokenizer.Error("Duplicate function definition");
            return false;
          }

          const_cast<HLSLFunction*>(declaration)->forward = function;
        }
        else
        {
          m_functions.PushBack(function);
        }

        if (!Expect('{') || !ParseBlock(function->statement, function->returnType))
        {
          return false;
        }

        EndScope();

        // Note, no semi-colon at the end of a function declaration.
        statement = function;

        return true;
      }
      else
      {
        HLSLDeclaration* firstDeclaration = nullptr;
        HLSLDeclaration* lastDeclaration = nullptr;

        do {
          if (firstDeclaration != nullptr) {
            if (!ExpectIdentifier(globalName))
            {
              return false;
            }
          }

          // Uniform declaration.
          HLSLDeclaration* declaration = m_tree->AddNode<HLSLDeclaration>(fileName, line);
          declaration->name = globalName;
          declaration->type = type;

          // Handle array syntax.
          if (Accept('['))
          {
            if (!Accept(']'))
            {
              if (!ParseExpression(declaration->type.arraySize) || !Expect(']'))
              {
                return false;
              }
            }
            declaration->type.array = true;
          }

          // Handle optional register.
          if (Accept(':'))
          {
            // @@ Currently we support either a semantic or a register, but not both.
            if (AcceptIdentifier(declaration->semantic)) {
              // nothing
            }
            else if (!Expect(HLSLToken_Register) || !Expect('(') || !ExpectIdentifier(declaration->registerName) || !Expect(')'))
            {
              return false;
            }
          }

          DeclareVariable(globalName, declaration->type);

          if (!ParseDeclarationAssignment(declaration))
          {
            return false;
          }
          if (firstDeclaration == nullptr) firstDeclaration = declaration;
          if (lastDeclaration != nullptr) lastDeclaration->nextDeclaration = declaration;
          lastDeclaration = declaration;

        } while (Accept(','));

        statement = firstDeclaration;
      }
    }
    else if (ParseTechnique(statement)) {
      doesNotExpectSemicolon = true;
    }
    else if (ParsePipeline(statement)) {
      doesNotExpectSemicolon = true;
    }
    else if (ParseStage(statement)) {
      doesNotExpectSemicolon = true;
    }

    if (statement != nullptr) {
      statement->attributes = attributes;
    }

    return doesNotExpectSemicolon || Expect(';');
  }

  bool HLSLParser::ParseStatementOrBlock(HLSLStatement*& firstStatement, const HLSLType& returnType, bool scoped/*=true*/)
  {
    if (scoped)
    {
      BeginScope();
    }
    if (Accept('{'))
    {
      if (!ParseBlock(firstStatement, returnType))
      {
        return false;
      }
    }
    else
    {
      if (!ParseStatement(firstStatement, returnType))
      {
        return false;
      }
    }
    if (scoped)
    {
      EndScope();
    }
    return true;
  }

  bool HLSLParser::ParseBlock(HLSLStatement*& firstStatement, const HLSLType& returnType)
  {
    HLSLStatement* lastStatement = nullptr;
    while (!Accept('}'))
    {
      if (CheckForUnexpectedEndOfStream('}'))
      {
        return false;
      }
      HLSLStatement* statement = nullptr;
      if (!ParseStatement(statement, returnType))
      {
        return false;
      }
      if (statement != nullptr)
      {
        if (firstStatement == nullptr)
        {
          firstStatement = statement;
        }
        else
        {
          lastStatement->nextStatement = statement;
        }
        lastStatement = statement;
        while (lastStatement->nextStatement) lastStatement = lastStatement->nextStatement;
      }
    }
    return true;
  }

  bool HLSLParser::ParseStatement(HLSLStatement*& statement, const HLSLType& returnType)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    // Empty statements.
    if (Accept(';'))
    {
      return true;
    }

    HLSLAttribute* attributes = nullptr;
    ParseAttributeBlock(attributes);    // @@ Leak if not assigned to node? 

#if 0 // @@ Work in progress.
    // Static statements: @if only for now.
    if (Accept('@'))
    {
      if (Accept(HLSLToken_If))
      {
        //HLSLIfStatement* ifStatement = m_tree->AddNode<HLSLIfStatement>(fileName, line);
        //ifStatement->isStatic = true;
        //ifStatement->attributes = attributes;

        HLSLExpression* condition = nullptr;

        m_allowUndeclaredIdentifiers = true;    // Not really correct... better to push to stack?
        if (!Expect('(') || !ParseExpression(condition) || !Expect(')'))
        {
          m_allowUndeclaredIdentifiers = false;
          return false;
        }
        m_allowUndeclaredIdentifiers = false;

        if ((condition->expressionType.flags & HLSLTypeFlag_Const) == 0)
        {
          m_tokenizer.Error("Syntax error: @if condition is not constant");
          return false;
        }

        int conditionValue;
        if (!m_tree->GetExpressionValue(condition, conditionValue))
        {
          m_tokenizer.Error("Syntax error: Cannot evaluate @if condition");
          return false;
        }

        if (!conditionValue) m_disableSemanticValidation = true;

        HLSLStatement* ifStatements = nullptr;
        HLSLStatement* elseStatements = nullptr;

        if (!ParseStatementOrBlock(ifStatements, returnType, /*scoped=*/false))
        {
          m_disableSemanticValidation = false;
          return false;
        }
        if (Accept(HLSLToken_Else))
        {
          if (conditionValue) m_disableSemanticValidation = true;

          if (!ParseStatementOrBlock(elseStatements, returnType, /*scoped=*/false))
          {
            m_disableSemanticValidation = false;
            return false;
          }
        }
        m_disableSemanticValidation = false;

        if (conditionValue) statement = ifStatements;
        else statement = elseStatements;

        // @@ Free the pruned statements?

        return true;
      }
      else {
        m_tokenizer.Error("Syntax error: unexpected token '@'");
      }
    }
#endif

    // If statement.
    if (Accept(HLSLToken_If))
    {
      HLSLIfStatement* ifStatement = m_tree->AddNode<HLSLIfStatement>(fileName, line);
      ifStatement->attributes = attributes;
      if (!Expect('(') || !ParseExpression(ifStatement->condition) || !Expect(')'))
      {
        return false;
      }
      statement = ifStatement;
      if (!ParseStatementOrBlock(ifStatement->statement, returnType))
      {
        return false;
      }
      if (Accept(HLSLToken_Else))
      {
        return ParseStatementOrBlock(ifStatement->elseStatement, returnType);
      }
      return true;
    }

    // For statement.
    if (Accept(HLSLToken_For))
    {
      HLSLForStatement* forStatement = m_tree->AddNode<HLSLForStatement>(fileName, line);
      forStatement->attributes = attributes;
      if (!Expect('('))
      {
        return false;
      }
      BeginScope();
      if (!ParseDeclaration(forStatement->initialization) && !ParseExpression(forStatement->initializationWithoutType))
      {
        return false;
      }
      if (!Expect(';'))
      {
        return false;
      }
      ParseExpression(forStatement->condition);
      if (!Expect(';'))
      {
        return false;
      }
      ParseExpression(forStatement->increment);
      if (!Expect(')'))
      {
        return false;
      }
      statement = forStatement;
      if (!ParseStatementOrBlock(forStatement->statement, returnType))
      {
        return false;
      }
      EndScope();
      return true;
    }

    // While statement.
    if (Accept(HLSLToken_While))
    {
      HLSLWhileStatement* whileStatement = m_tree->AddNode<HLSLWhileStatement>(fileName, line);
      whileStatement->attributes = attributes;
      if (!Expect('(') || !ParseExpression(whileStatement->condition) || !Expect(')'))
      {
        return false;
      }
      statement = whileStatement;
      if (!ParseStatementOrBlock(whileStatement->statement, returnType))
      {
        return false;
      }
      return true;
    }


    if (attributes != nullptr)
    {
      // @@ Error. Unexpected attribute. We only support attributes associated to if and for statements.
    }

    // Block statement.
    if (Accept('{'))
    {
      HLSLBlockStatement* blockStatement = m_tree->AddNode<HLSLBlockStatement>(fileName, line);
      statement = blockStatement;
      BeginScope();
      bool success = ParseBlock(blockStatement->statement, returnType);
      EndScope();
      return success;
    }

    // Discard statement.
    if (Accept(HLSLToken_Discard))
    {
      HLSLDiscardStatement* discardStatement = m_tree->AddNode<HLSLDiscardStatement>(fileName, line);
      statement = discardStatement;
      return Expect(';');
    }

    // Break statement.
    if (Accept(HLSLToken_Break))
    {
      HLSLBreakStatement* breakStatement = m_tree->AddNode<HLSLBreakStatement>(fileName, line);
      statement = breakStatement;
      return Expect(';');
    }

    // Continue statement.
    if (Accept(HLSLToken_Continue))
    {
      HLSLContinueStatement* continueStatement = m_tree->AddNode<HLSLContinueStatement>(fileName, line);
      statement = continueStatement;
      return Expect(';');
    }

    // Return statement
    if (Accept(HLSLToken_Return))
    {
      HLSLReturnStatement* returnStatement = m_tree->AddNode<HLSLReturnStatement>(fileName, line);
      if (!Accept(';') && !ParseExpression(returnStatement->expression))
      {
        return false;
      }
      // Check that the return expression can be cast to the return type of the function.
      HLSLType voidType(HLSLBaseType_Void);
      if (!CheckTypeCast(returnStatement->expression ? returnStatement->expression->expressionType : voidType, returnType))
      {
        return false;
      }

      statement = returnStatement;
      return Expect(';');
    }

    HLSLDeclaration* declaration = nullptr;
    HLSLExpression* expression = nullptr;

    if (ParseDeclaration(declaration))
    {
      statement = declaration;
    }
    else if (ParseExpression(expression))
    {
      HLSLExpressionStatement* expressionStatement;
      expressionStatement = m_tree->AddNode<HLSLExpressionStatement>(fileName, line);
      expressionStatement->expression = expression;
      statement = expressionStatement;
    }

    // Some statements are separated by a colon instead of a semicolon
    if (Accept(',')) {
      return true;
    }

    return Expect(';');
  }


  // IC: This is only used in block statements, or within control flow statements. So, it doesn't support semantics or layout modifiers.
  // @@ We should add suport for semantics for inline input/output declarations.
  bool HLSLParser::ParseDeclaration(HLSLDeclaration*& declaration)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    HLSLType type;
    if (!AcceptType(/*allowVoid=*/false, type))
    {
      return false;
    }

    bool allowUnsizedArray = true;  // @@ Really?

    HLSLDeclaration* firstDeclaration = nullptr;
    HLSLDeclaration* lastDeclaration = nullptr;

    do {
      const char* name;
      if (!ExpectIdentifier(name))
      {
        // TODO: false means we didn't accept a declaration and we had an error!
        return false;
      }
      // Handle array syntax.
      if (Accept('['))
      {
        type.array = true;
        // Optionally allow no size to the specified for the array.
        if (Accept(']') && allowUnsizedArray)
        {
          return true;
        }
        if (!ParseExpression(type.arraySize) || !Expect(']'))
        {
          return false;
        }
      }

      HLSLDeclaration* declaration2 = m_tree->AddNode<HLSLDeclaration>(fileName, line);
      declaration2->type = type;
      declaration2->name = name;

      DeclareVariable(declaration2->name, declaration2->type);

      // Handle option assignment of the declared variables(s).
      if (!ParseDeclarationAssignment(declaration2)) {
        return false;
      }

      if (firstDeclaration == nullptr) firstDeclaration = declaration2;
      if (lastDeclaration != nullptr) lastDeclaration->nextDeclaration = declaration2;
      lastDeclaration = declaration2;

    } while (Accept(','));

    declaration = firstDeclaration;

    return true;
  }

  bool HLSLParser::ParseDeclarationAssignment(HLSLDeclaration* declaration)
  {
    if (Accept('='))
    {
      // Handle array initialization syntax.
      if (declaration->type.array)
      {
        int numValues = 0;
        if (!Expect('{') || !ParseExpressionList('}', true, declaration->assignment, numValues))
        {
          return false;
        }
      }
      else if (IsSamplerType(declaration->type.baseType))
      {
        if (!ParseSamplerState(declaration->assignment))
        {
          return false;
        }
      }
      else if (!ParseExpression(declaration->assignment))
      {
        return false;
      }
    }
    return true;
  }

  bool HLSLParser::ParseFieldDeclaration(HLSLStructField*& field)
  {
    field = m_tree->AddNode<HLSLStructField>(GetFileName(), GetLineNumber());
    if (!ExpectDeclaration(false, field->type, field->name))
    {
      return false;
    }
    // Handle optional semantics.
    if (Accept(':'))
    {
      if (!ExpectIdentifier(field->semantic))
      {
        return false;
      }
    }
    return Expect(';');
  }

  // @@ Add support for packoffset to general declarations.
  /*bool HLSLParser::ParseBufferFieldDeclaration(HLSLBufferField*& field)
  {
      field = m_tree->AddNode<HLSLBufferField>( GetFileName(), GetLineNumber() );
      if (AcceptDeclaration(false, field->type, field->name))
      {
          // Handle optional packoffset.
          if (Accept(':'))
          {
              if (!Expect("packoffset"))
              {
                  return false;
              }
              const char* constantName = nullptr;
              const char* swizzleMask  = nullptr;
              if (!Expect('(') || !ExpectIdentifier(constantName) || !Expect('.') || !ExpectIdentifier(swizzleMask) || !Expect(')'))
              {
                  return false;
              }
          }
          return Expect(';');
      }
      return false;
  }*/

  bool HLSLParser::CheckTypeCast(const HLSLType& srcType, const HLSLType& dstType)
  {
    if (GetTypeCastRank(m_tree, srcType, dstType) == -1)
    {
      const char* srcTypeName = GetTypeName1(srcType);
      const char* dstTypeName = GetTypeName1(dstType);
      m_tokenizer.Error("Cannot implicitly convert from '%s' to '%s'", srcTypeName, dstTypeName);
      return false;
    }
    return true;
  }

  bool HLSLParser::ParseExpression(HLSLExpression*& expression)
  {
    if (!ParseBinaryExpression(0, expression))
    {
      return false;
    }

    HLSLBinaryOp assignOp;
    if (AcceptAssign(assignOp))
    {
      HLSLExpression* expression2 = nullptr;
      if (!ParseExpression(expression2))
      {
        return false;
      }
      HLSLBinaryExpression* binaryExpression = m_tree->AddNode<HLSLBinaryExpression>(expression->fileName, expression->line);
      binaryExpression->binaryOp = assignOp;
      binaryExpression->expression1 = expression;
      binaryExpression->expression2 = expression2;
      // This type is not strictly correct, since the type should be a reference.
      // However, for our usage of the types it should be sufficient.
      binaryExpression->expressionType = expression->expressionType;

      if (!CheckTypeCast(expression2->expressionType, expression->expressionType))
      {
        const char* srcTypeName = GetTypeName1(expression2->expressionType);
        const char* dstTypeName = GetTypeName1(expression->expressionType);
        m_tokenizer.Error("Cannot implicitly convert from '%s' to '%s'", srcTypeName, dstTypeName);
        return false;
      }

      expression = binaryExpression;
    }

    return true;
  }

  bool HLSLParser::AcceptBinaryOperator(int priority, HLSLBinaryOp& binaryOp)
  {
    int token = m_tokenizer.GetToken();
    switch (token)
    {
    case HLSLToken_AndAnd:          binaryOp = HLSLBinaryOp_And;          break;
    case HLSLToken_BarBar:          binaryOp = HLSLBinaryOp_Or;           break;
    case '+':                       binaryOp = HLSLBinaryOp_Add;          break;
    case '-':                       binaryOp = HLSLBinaryOp_Sub;          break;
    case '*':                       binaryOp = HLSLBinaryOp_Mul;          break;
    case '/':                       binaryOp = HLSLBinaryOp_Div;          break;
    case '%':                       binaryOp = HLSLBinaryOp_Mod;          break;
    case '<':                       binaryOp = HLSLBinaryOp_Less;         break;
    case '>':                       binaryOp = HLSLBinaryOp_Greater;      break;
    case HLSLToken_LessEqual:       binaryOp = HLSLBinaryOp_LessEqual;    break;
    case HLSLToken_GreaterEqual:    binaryOp = HLSLBinaryOp_GreaterEqual; break;
    case HLSLToken_EqualEqual:      binaryOp = HLSLBinaryOp_Equal;        break;
    case HLSLToken_NotEqual:        binaryOp = HLSLBinaryOp_NotEqual;     break;
    case '&':                       binaryOp = HLSLBinaryOp_BitAnd;       break;
    case '|':                       binaryOp = HLSLBinaryOp_BitOr;        break;
    case '^':                       binaryOp = HLSLBinaryOp_BitXor;       break;
    default:
      return false;
    }
    if (_binaryOpPriority[binaryOp] > priority)
    {
      m_tokenizer.Next();
      return true;
    }
    return false;
  }

  bool HLSLParser::AcceptUnaryOperator(bool pre, HLSLUnaryOp& unaryOp)
  {
    int token = m_tokenizer.GetToken();
    if (token == HLSLToken_PlusPlus)
    {
      unaryOp = pre ? HLSLUnaryOp_PreIncrement : HLSLUnaryOp_PostIncrement;
    }
    else if (token == HLSLToken_MinusMinus)
    {
      unaryOp = pre ? HLSLUnaryOp_PreDecrement : HLSLUnaryOp_PostDecrement;
    }
    else if (pre && token == '-')
    {
      unaryOp = HLSLUnaryOp_Negative;
    }
    else if (pre && token == '+')
    {
      unaryOp = HLSLUnaryOp_Positive;
    }
    else if (pre && token == '!')
    {
      unaryOp = HLSLUnaryOp_Not;
    }
    else if (pre && token == '~')
    {
      unaryOp = HLSLUnaryOp_Not;
    }
    else
    {
      return false;
    }
    m_tokenizer.Next();
    return true;
  }

  bool HLSLParser::AcceptAssign(HLSLBinaryOp& binaryOp)
  {
    if (Accept('='))
    {
      binaryOp = HLSLBinaryOp_Assign;
    }
    else if (Accept(HLSLToken_PlusEqual))
    {
      binaryOp = HLSLBinaryOp_AddAssign;
    }
    else if (Accept(HLSLToken_MinusEqual))
    {
      binaryOp = HLSLBinaryOp_SubAssign;
    }
    else if (Accept(HLSLToken_TimesEqual))
    {
      binaryOp = HLSLBinaryOp_MulAssign;
    }
    else if (Accept(HLSLToken_DivideEqual))
    {
      binaryOp = HLSLBinaryOp_DivAssign;
    }
    else
    {
      return false;
    }
    return true;
  }

  bool HLSLParser::ParseBinaryExpression(int priority, HLSLExpression*& expression)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    char needsExpressionEndChar;

    if (!ParseTerminalExpression(expression, needsExpressionEndChar))
    {
      return false;
    }

    // reset priority cause openned parenthesis
    if (needsExpressionEndChar != 0)
      priority = 0;

    bool acceptBinaryOp = false;
    HLSLBinaryOp binaryOp;
    while (1)
    {
      if (acceptBinaryOp || AcceptBinaryOperator(priority, binaryOp))
      {
        acceptBinaryOp = false;
        HLSLExpression* expression2 = nullptr;
        ASSERT(binaryOp < sizeof(_binaryOpPriority) / sizeof(int));
        if (!ParseBinaryExpression(_binaryOpPriority[binaryOp], expression2))
        {
          return false;
        }
        HLSLBinaryExpression* binaryExpression = m_tree->AddNode<HLSLBinaryExpression>(fileName, line);
        binaryExpression->binaryOp = binaryOp;
        binaryExpression->expression1 = expression;
        binaryExpression->expression2 = expression2;
        if (!GetBinaryOpResultType(binaryOp, expression->expressionType, expression2->expressionType, binaryExpression->expressionType))
        {
          const char* typeName1 = GetTypeName1(binaryExpression->expression1->expressionType);
          const char* typeName2 = GetTypeName1(binaryExpression->expression2->expressionType);
          m_tokenizer.Error("binary '%s' : no global operator found which takes types '%s' and '%s' (or there is no acceptable conversion)",
            GetBinaryOpName(binaryOp), typeName1, typeName2);

          return false;
        }

        // Propagate constness.
        binaryExpression->expressionType.flags = (expression->expressionType.flags | expression2->expressionType.flags) & HLSLTypeFlag_Const;

        expression = binaryExpression;
      }
      else if (_conditionalOpPriority > priority && Accept('?'))
      {

        HLSLConditionalExpression* conditionalExpression = m_tree->AddNode<HLSLConditionalExpression>(fileName, line);
        conditionalExpression->condition = expression;

        HLSLExpression* expression1 = nullptr;
        HLSLExpression* expression2 = nullptr;
        if (!ParseBinaryExpression(_conditionalOpPriority, expression1) || !Expect(':') || !ParseBinaryExpression(_conditionalOpPriority, expression2))
        {
          return false;
        }

        // Make sure both cases have compatible types.
        if (GetTypeCastRank(m_tree, expression1->expressionType, expression2->expressionType) == -1)
        {
          const char* srcTypeName = GetTypeName1(expression2->expressionType);
          const char* dstTypeName = GetTypeName1(expression1->expressionType);
          m_tokenizer.Error("':' no possible conversion from '%s' to '%s'", srcTypeName, dstTypeName);
          return false;
        }

        conditionalExpression->trueExpression = expression1;
        conditionalExpression->falseExpression = expression2;
        conditionalExpression->expressionType = expression1->expressionType;

        expression = conditionalExpression;
      }
      else
      {
        break;
      }

      //  First: try if next is a binary op before exiting the loop
      if (AcceptBinaryOperator(priority, binaryOp))
      {
        acceptBinaryOp = true;
        continue;
      }

      if (needsExpressionEndChar != 0)
      {
        if (!Expect(needsExpressionEndChar))
          return false;
        needsExpressionEndChar = 0;
      }
    }

    return needsExpressionEndChar == 0 || Expect(needsExpressionEndChar);
  }

  bool HLSLParser::ParsePartialConstructor(HLSLExpression*& expression, HLSLBaseType type, const char* typeName)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    HLSLConstructorExpression* constructorExpression = m_tree->AddNode<HLSLConstructorExpression>(fileName, line);
    constructorExpression->type.baseType = type;
    constructorExpression->type.typeName = typeName;
    int numArguments = 0;
    if (!ParseExpressionList(')', false, constructorExpression->argument, numArguments))
    {
      return false;
    }
    constructorExpression->expressionType = constructorExpression->type;
    constructorExpression->expressionType.flags = HLSLTypeFlag_Const;
    expression = constructorExpression;
    return true;
  }

  bool HLSLParser::ParseTerminalExpression(HLSLExpression*& expression, char& needsExpressionEndChar)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    needsExpressionEndChar = 0;

    HLSLUnaryOp unaryOp;
    if (AcceptUnaryOperator(true, unaryOp))
    {
      HLSLUnaryExpression* unaryExpression = m_tree->AddNode<HLSLUnaryExpression>(fileName, line);
      unaryExpression->unaryOp = unaryOp;
      if (!ParseTerminalExpression(unaryExpression->expression, needsExpressionEndChar))
      {
        return false;
      }
      if (unaryOp == HLSLUnaryOp_BitNot)
      {
        if (unaryExpression->expression->expressionType.baseType < HLSLBaseType_FirstInteger ||
          unaryExpression->expression->expressionType.baseType > HLSLBaseType_LastInteger)
        {
          const char* typeName = GetTypeName1(unaryExpression->expression->expressionType);
          m_tokenizer.Error("unary '~' : no global operator found which takes type '%s' (or there is no acceptable conversion)", typeName);
          return false;
        }
      }
      if (unaryOp == HLSLUnaryOp_Not)
      {
        if (unaryExpression->expression->expressionType.baseType != HLSLBaseType_Bool)
        {
          // Insert a cast to bool
          HLSLCastingExpression* castingExpression = m_tree->AddNode<HLSLCastingExpression>(fileName, line);
          castingExpression->type.baseType = HLSLBaseType_Bool;
          castingExpression->expression = unaryExpression->expression;
          castingExpression->expressionType.baseType = HLSLBaseType_Bool;

          unaryExpression->expression = castingExpression;
          expression = unaryExpression;
        }

        unaryExpression->expressionType = HLSLType(HLSLBaseType_Bool);

        // Propagate constness.
        unaryExpression->expressionType.flags = unaryExpression->expression->expressionType.flags & HLSLTypeFlag_Const;
      }

      if (unaryOp == HLSLUnaryOp_Negative || unaryOp == HLSLUnaryOp_Positive)
      {
        if (unaryExpression->expression->expressionType.baseType <= HLSLBaseType_Bool4 &&
          unaryExpression->expression->expressionType.baseType >= HLSLBaseType_Bool)
        {
          // Insert a cast bool to int
          int numComponents = baseTypeDescriptions[unaryExpression->expression->expressionType.baseType].numComponents;
          HLSLBaseType baseType = HLSLBaseType(HLSLBaseType_Int + numComponents - 1);

          unaryExpression->expressionType = unaryExpression->expression->expressionType;

          HLSLCastingExpression* castingExpression = m_tree->AddNode<HLSLCastingExpression>(fileName, line);
          castingExpression->type.baseType = baseType;
          castingExpression->expression = unaryExpression->expression;
          castingExpression->expressionType.baseType = baseType;

          unaryExpression->expression = castingExpression;
          expression = unaryExpression;
        }
        else
        {
          unaryExpression->expressionType = unaryExpression->expression->expressionType;
        }
      }
      else
      {
        unaryExpression->expressionType = unaryExpression->expression->expressionType;
      }
      expression = unaryExpression;
      return true;
    }

    // Expressions inside parenthesis or casts.
    char expressionEndChar = 0;
    if (Accept('('))
    {
      expressionEndChar = ')';
    }
    else if (Accept('{'))
    {
      expressionEndChar = '}';
    }


    if (expressionEndChar != 0)
    {
      // Check for a casting operator.
      HLSLType type;
      if (AcceptType(false, type))
      {
        // This is actually a type constructor like (float2(...
        if (Accept('('))
        {
          needsExpressionEndChar = expressionEndChar;
          return ParsePartialConstructor(expression, type.baseType, type.typeName);
        }
        HLSLCastingExpression* castingExpression = m_tree->AddNode<HLSLCastingExpression>(fileName, line);
        castingExpression->type = type;
        expression = castingExpression;
        castingExpression->expressionType = type;
        return Expect(')') && ParseExpression(castingExpression->expression);
      }

      int numArguments = 0;
      if (!ParseExpressionList(expressionEndChar, false, expression, numArguments))
      {
        return false;
      }

      if (expressionEndChar == ')')
      {
        // Allows comma separated expression lists, HLSL compiler keeps the last one
        HLSLExpression* lastExpression = expression;
        while (lastExpression->nextExpression != nullptr) {
          lastExpression = lastExpression->nextExpression;
        }
      }
    }
    else
    {
      // Terminal values.
      float fValue = 0.0f;
      int   iValue = 0;

      if (AcceptFloat(fValue))
      {
        HLSLLiteralExpression* literalExpression = m_tree->AddNode<HLSLLiteralExpression>(fileName, line);
        literalExpression->type = HLSLBaseType_Float;
        literalExpression->fValue = fValue;
        literalExpression->expressionType.baseType = literalExpression->type;
        literalExpression->expressionType.flags = HLSLTypeFlag_Const;
        expression = literalExpression;
        return true;
      }
      else if (AcceptInt(iValue))
      {
        HLSLLiteralExpression* literalExpression = m_tree->AddNode<HLSLLiteralExpression>(fileName, line);
        literalExpression->type = HLSLBaseType_Int;
        literalExpression->iValue = iValue;
        literalExpression->expressionType.baseType = literalExpression->type;
        literalExpression->expressionType.flags = HLSLTypeFlag_Const;
        expression = literalExpression;
        return true;
      }
      else if (Accept(HLSLToken_True))
      {
        HLSLLiteralExpression* literalExpression = m_tree->AddNode<HLSLLiteralExpression>(fileName, line);
        literalExpression->type = HLSLBaseType_Bool;
        literalExpression->bValue = true;
        literalExpression->expressionType.baseType = literalExpression->type;
        literalExpression->expressionType.flags = HLSLTypeFlag_Const;
        expression = literalExpression;
        return true;
      }
      else if (Accept(HLSLToken_False))
      {
        HLSLLiteralExpression* literalExpression = m_tree->AddNode<HLSLLiteralExpression>(fileName, line);
        literalExpression->type = HLSLBaseType_Bool;
        literalExpression->bValue = false;
        literalExpression->expressionType.baseType = literalExpression->type;
        literalExpression->expressionType.flags = HLSLTypeFlag_Const;
        expression = literalExpression;
        return true;
      }

      // Type constructor.
      HLSLType type;
      if (AcceptType(/*allowVoid=*/false, type))
      {
        Expect('(');
        if (!ParsePartialConstructor(expression, type.baseType, type.typeName))
        {
          return false;
        }
      }
      else
      {
        HLSLIdentifierExpression* identifierExpression = m_tree->AddNode<HLSLIdentifierExpression>(fileName, line);
        if (!ExpectIdentifier(identifierExpression->name))
        {
          return false;
        }

        bool undeclaredIdentifier = false;

        const HLSLType* identifierType = FindVariable(identifierExpression->name, identifierExpression->global);
        if (identifierType != nullptr)
        {
          identifierExpression->expressionType = *identifierType;
        }
        else
        {
          if (GetIsFunction(identifierExpression->name))
          {
            // Functions are always global scope.
            identifierExpression->global = true;
          }
          else
          {
            undeclaredIdentifier = true;
          }
        }

        if (undeclaredIdentifier)
        {
          if (m_allowUndeclaredIdentifiers)
          {
            HLSLLiteralExpression* literalExpression = m_tree->AddNode<HLSLLiteralExpression>(fileName, line);
            literalExpression->bValue = false;
            literalExpression->type = HLSLBaseType_Bool;
            literalExpression->expressionType.baseType = literalExpression->type;
            literalExpression->expressionType.flags = HLSLTypeFlag_Const;
            expression = literalExpression;
          }
          else
          {
            m_tokenizer.Error("Undeclared identifier '%s'", identifierExpression->name);
            return false;
          }
        }
        else {
          expression = identifierExpression;
        }
      }
    }

    bool done = false;
    while (!done)
    {
      done = true;

      // Post fix unary operator
      HLSLUnaryOp unaryOp2;
      while (AcceptUnaryOperator(false, unaryOp2))
      {
        HLSLUnaryExpression* unaryExpression = m_tree->AddNode<HLSLUnaryExpression>(fileName, line);
        unaryExpression->unaryOp = unaryOp2;
        unaryExpression->expression = expression;
        unaryExpression->expressionType = unaryExpression->expression->expressionType;
        expression = unaryExpression;
        done = false;
      }

      // Member access operator.
      while (Accept('.'))
      {
        HLSLMemberAccess* memberAccess = m_tree->AddNode<HLSLMemberAccess>(fileName, line);
        memberAccess->object = expression;
        if (!ExpectIdentifier(memberAccess->field))
        {
          return false;
        }
        if (!GetMemberType(expression->expressionType, memberAccess))
        {
          m_tokenizer.Error("Couldn't access '%s'", memberAccess->field);
          return false;
        }
        expression = memberAccess;
        done = false;
      }

      // Handle array access.
      while (Accept('['))
      {
        HLSLArrayAccess* arrayAccess = m_tree->AddNode<HLSLArrayAccess>(fileName, line);
        arrayAccess->array = expression;
        if (!ParseExpression(arrayAccess->index) || !Expect(']'))
        {
          return false;
        }

        if (expression->expressionType.array)
        {
          arrayAccess->expressionType = expression->expressionType;
          arrayAccess->expressionType.array = false;
          arrayAccess->expressionType.arraySize = nullptr;
        }
        else
        {
          switch (expression->expressionType.baseType)
          {
          case HLSLBaseType_Float2:
          case HLSLBaseType_Float3:
          case HLSLBaseType_Float4:
            arrayAccess->expressionType.baseType = HLSLBaseType_Float;
            break;
          case HLSLBaseType_Float4x4:
          case HLSLBaseType_Float3x4:
          case HLSLBaseType_Float2x4:
            arrayAccess->expressionType.baseType = HLSLBaseType_Float4;
            break;
          case HLSLBaseType_Float4x3:
          case HLSLBaseType_Float3x3:
          case HLSLBaseType_Float2x3:
            arrayAccess->expressionType.baseType = HLSLBaseType_Float3;
            break;
          case HLSLBaseType_Float4x2:
          case HLSLBaseType_Float3x2:
          case HLSLBaseType_Float2x2:
            arrayAccess->expressionType.baseType = HLSLBaseType_Float2;
            break;
          case HLSLBaseType_Bool2:
          case HLSLBaseType_Bool3:
          case HLSLBaseType_Bool4:
            arrayAccess->expressionType.baseType = HLSLBaseType_Bool;
            break;
          case HLSLBaseType_Int2:
          case HLSLBaseType_Int3:
          case HLSLBaseType_Int4:
            arrayAccess->expressionType.baseType = HLSLBaseType_Int;
            break;
          case HLSLBaseType_Uint2:
          case HLSLBaseType_Uint3:
          case HLSLBaseType_Uint4:
            arrayAccess->expressionType.baseType = HLSLBaseType_Uint;
            break;
          default:
            m_tokenizer.Error("array, matrix, vector, or indexable object type expected in index expression");
            return false;
          }
        }

        expression = arrayAccess;
        done = false;
      }

      // Handle function calls. Note, HLSL functions aren't like C function
      // pointers -- we can only directly call on an identifier, not on an
      // expression.
      if (Accept('('))
      {
        HLSLFunctionCall* functionCall = m_tree->AddNode<HLSLFunctionCall>(fileName, line);
        done = false;
        if (!ParseExpressionList(')', false, functionCall->argument, functionCall->numArguments))
        {
          return false;
        }

        if (expression->nodeType != HLSLNodeType_IdentifierExpression)
        {
          m_tokenizer.Error("Expected function identifier");
          return false;
        }

        const HLSLIdentifierExpression* identifierExpression = static_cast<const HLSLIdentifierExpression*>(expression);
        const HLSLFunction* function = MatchFunctionCall(functionCall, identifierExpression->name);
        if (function == nullptr)
        {
          return false;
        }

        functionCall->function = function;
        functionCall->expressionType = function->returnType;
        expression = functionCall;
      }

    }
    return true;

  }

  bool HLSLParser::ParseExpressionList(int endToken, bool allowEmptyEnd, HLSLExpression*& firstExpression, int& numExpressions)
  {
    numExpressions = 0;
    HLSLExpression* lastExpression = nullptr;
    while (!Accept(endToken))
    {
      if (CheckForUnexpectedEndOfStream(endToken))
      {
        return false;
      }
      if (numExpressions > 0 && !Expect(','))
      {
        return false;
      }
      // It is acceptable for the final element in the initialization list to
      // have a trailing comma in some cases, like array initialization such as {1, 2, 3,}
      if (allowEmptyEnd && Accept(endToken))
      {
        break;
      }
      HLSLExpression* expression = nullptr;
      if (!ParseExpression(expression))
      {
        return false;
      }
      if (firstExpression == nullptr)
      {
        firstExpression = expression;
      }
      else
      {
        lastExpression->nextExpression = expression;
      }
      lastExpression = expression;
      ++numExpressions;
    }
    return true;
  }

  bool HLSLParser::ParseArgumentList(HLSLArgument*& firstArgument, int& numArguments, int& numOutputArguments)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    HLSLArgument* lastArgument = nullptr;
    numArguments = 0;

    while (!Accept(')'))
    {
      if (CheckForUnexpectedEndOfStream(')'))
      {
        return false;
      }
      if (numArguments > 0 && !Expect(','))
      {
        return false;
      }

      HLSLArgument* argument = m_tree->AddNode<HLSLArgument>(fileName, line);

      if (Accept(HLSLToken_Uniform)) { argument->modifier = HLSLArgumentModifier_Uniform; }
      else if (Accept(HLSLToken_In)) { argument->modifier = HLSLArgumentModifier_In; }
      else if (Accept(HLSLToken_Out)) { argument->modifier = HLSLArgumentModifier_Out; }
      else if (Accept(HLSLToken_InOut)) { argument->modifier = HLSLArgumentModifier_Inout; }
      else if (Accept(HLSLToken_Const)) { argument->modifier = HLSLArgumentModifier_Const; }

      if (!ExpectDeclaration(/*allowUnsizedArray=*/true, argument->type, argument->name))
      {
        return false;
      }

      DeclareVariable(argument->name, argument->type);

      // Optional semantic.
      if (Accept(':') && !ExpectIdentifier(argument->semantic))
      {
        return false;
      }

      if (Accept('=') && !ParseExpression(argument->defaultValue))
      {
        // @@ Print error!
        return false;
      }

      if (lastArgument != nullptr)
      {
        lastArgument->nextArgument = argument;
      }
      else
      {
        firstArgument = argument;
      }
      lastArgument = argument;

      ++numArguments;
      if (argument->modifier == HLSLArgumentModifier_Out || argument->modifier == HLSLArgumentModifier_Inout)
      {
        ++numOutputArguments;
      }
    }
    return true;
  }


  bool HLSLParser::ParseSamplerState(HLSLExpression*& expression)
  {
    if (!Expect(HLSLToken_SamplerState))
    {
      return false;
    }

    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    HLSLSamplerState* samplerState = m_tree->AddNode<HLSLSamplerState>(fileName, line);

    if (!Expect('{'))
    {
      return false;
    }

    HLSLStateAssignment* lastStateAssignment = nullptr;

    // Parse state assignments.
    while (!Accept('}'))
    {
      if (CheckForUnexpectedEndOfStream('}'))
      {
        return false;
      }

      HLSLStateAssignment* stateAssignment = nullptr;
      if (!ParseStateAssignment(stateAssignment, /*isSamplerState=*/true, /*isPipeline=*/false))
      {
        return false;
      }
      ASSERT(stateAssignment != nullptr);
      if (lastStateAssignment == nullptr)
      {
        samplerState->stateAssignments = stateAssignment;
      }
      else
      {
        lastStateAssignment->nextStateAssignment = stateAssignment;
      }
      lastStateAssignment = stateAssignment;
      samplerState->numStateAssignments++;
    }

    expression = samplerState;
    return true;
  }

  bool HLSLParser::ParseTechnique(HLSLStatement*& statement)
  {
    if (!Accept(HLSLToken_Technique)) {
      return false;
    }

    const char* techniqueName = nullptr;
    if (!ExpectIdentifier(techniqueName))
    {
      return false;
    }

    if (!Expect('{'))
    {
      return false;
    }

    HLSLTechnique* technique = m_tree->AddNode<HLSLTechnique>(GetFileName(), GetLineNumber());
    technique->name = techniqueName;

    //m_techniques.PushBack(technique);

    HLSLPass* lastPass = nullptr;

    // Parse state assignments.
    while (!Accept('}'))
    {
      if (CheckForUnexpectedEndOfStream('}'))
      {
        return false;
      }

      HLSLPass* pass = nullptr;
      if (!ParsePass(pass))
      {
        return false;
      }
      ASSERT(pass != nullptr);
      if (lastPass == nullptr)
      {
        technique->passes = pass;
      }
      else
      {
        lastPass->nextPass = pass;
      }
      lastPass = pass;
      technique->numPasses++;
    }

    statement = technique;
    return true;
  }

  bool HLSLParser::ParsePass(HLSLPass*& pass)
  {
    if (!Accept(HLSLToken_Pass)) {
      return false;
    }

    // Optional pass name.
    const char* passName = nullptr;
    AcceptIdentifier(passName);

    if (!Expect('{'))
    {
      return false;
    }

    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    pass = m_tree->AddNode<HLSLPass>(fileName, line);
    pass->name = passName;

    HLSLStateAssignment* lastStateAssignment = nullptr;

    // Parse state assignments.
    while (!Accept('}'))
    {
      if (CheckForUnexpectedEndOfStream('}'))
      {
        return false;
      }

      HLSLStateAssignment* stateAssignment = nullptr;
      if (!ParseStateAssignment(stateAssignment, /*isSamplerState=*/false, /*isPipelineState=*/false))
      {
        return false;
      }
      ASSERT(stateAssignment != nullptr);
      if (lastStateAssignment == nullptr)
      {
        pass->stateAssignments = stateAssignment;
      }
      else
      {
        lastStateAssignment->nextStateAssignment = stateAssignment;
      }
      lastStateAssignment = stateAssignment;
      pass->numStateAssignments++;
    }
    return true;
  }


  bool HLSLParser::ParsePipeline(HLSLStatement*& statement)
  {
    if (!Accept("pipeline")) {
      return false;
    }

    // Optional pipeline name.
    const char* pipelineName = nullptr;
    AcceptIdentifier(pipelineName);

    if (!Expect('{'))
    {
      return false;
    }

    HLSLPipeline* pipeline = m_tree->AddNode<HLSLPipeline>(GetFileName(), GetLineNumber());
    pipeline->name = pipelineName;

    HLSLStateAssignment* lastStateAssignment = nullptr;

    // Parse state assignments.
    while (!Accept('}'))
    {
      if (CheckForUnexpectedEndOfStream('}'))
      {
        return false;
      }

      HLSLStateAssignment* stateAssignment = nullptr;
      if (!ParseStateAssignment(stateAssignment, /*isSamplerState=*/false, /*isPipeline=*/true))
      {
        return false;
      }
      ASSERT(stateAssignment != nullptr);
      if (lastStateAssignment == nullptr)
      {
        pipeline->stateAssignments = stateAssignment;
      }
      else
      {
        lastStateAssignment->nextStateAssignment = stateAssignment;
      }
      lastStateAssignment = stateAssignment;
      pipeline->numStateAssignments++;
    }

    statement = pipeline;
    return true;
  }


  const EffectState* GetEffectState(const char* name, bool isSamplerState, bool isPipeline)
  {
    const EffectState* validStates = effectStates;
    int count = sizeof(effectStates) / sizeof(effectStates[0]);

    if (isPipeline)
    {
      validStates = pipelineStates;
      count = sizeof(pipelineStates) / sizeof(pipelineStates[0]);
    }

    if (isSamplerState)
    {
      validStates = samplerStates;
      count = sizeof(samplerStates) / sizeof(samplerStates[0]);
    }

    // Case insensitive comparison.
    for (int i = 0; i < count; i++)
    {
      if (String_EqualNoCase(name, validStates[i].name))
      {
        return &validStates[i];
      }
    }

    return nullptr;
  }

  static const EffectStateValue* GetStateValue(const char* name, const EffectState* state)
  {
    // Case insensitive comparison.
    for (int i = 0; ; i++)
    {
      const EffectStateValue& value = state->values[i];
      if (value.name == nullptr) break;

      if (String_EqualNoCase(name, value.name))
      {
        return &value;
      }
    }

    return nullptr;
  }


  bool HLSLParser::ParseStateName(bool isSamplerState, bool isPipelineState, const char*& name, const EffectState*& state)
  {
    if (m_tokenizer.GetToken() != HLSLToken_Identifier)
    {
      char near_[s_maxIdentifier];
      m_tokenizer.GetTokenName(near_);
      m_tokenizer.Error("Syntax error: expected identifier near '%s'", near_);
      return false;
    }

    state = GetEffectState(m_tokenizer.GetIdentifier(), isSamplerState, isPipelineState);
    if (state == nullptr)
    {
      m_tokenizer.Error("Syntax error: unexpected identifier '%s'", m_tokenizer.GetIdentifier());
      return false;
    }

    m_tokenizer.Next();
    return true;
  }

  bool HLSLParser::ParseColorMask(int& mask)
  {
    mask = 0;

    do {
      if (m_tokenizer.GetToken() == HLSLToken_IntLiteral) {
        mask |= m_tokenizer.GetInt();
      }
      else if (m_tokenizer.GetToken() == HLSLToken_Identifier) {
        const char* ident = m_tokenizer.GetIdentifier();
        const EffectStateValue* stateValue = colorMaskValues;
        while (stateValue->name != nullptr) {
          if (String_EqualNoCase(stateValue->name, ident)) {
            mask |= stateValue->value;
            break;
          }
          ++stateValue;
        }
      }
      else {
        return false;
      }
      m_tokenizer.Next();
    } while (Accept('|'));

    return true;
  }

  bool HLSLParser::ParseStateValue(const EffectState* state, HLSLStateAssignment* stateAssignment)
  {
    const bool expectsExpression = state->values == colorMaskValues;
    const bool expectsInteger = state->values == integerValues;
    const bool expectsFloat = state->values == floatValues;
    const bool expectsBoolean = state->values == booleanValues;

    if (!expectsExpression && !expectsInteger && !expectsFloat && !expectsBoolean)
    {
      if (m_tokenizer.GetToken() != HLSLToken_Identifier)
      {
        char near_[s_maxIdentifier];
        m_tokenizer.GetTokenName(near_);
        m_tokenizer.Error("Syntax error: expected identifier near '%s'", near_);
        stateAssignment->iValue = 0;
        return false;
      }
    }

    if (state->values == nullptr)
    {
      if (strcmp(m_tokenizer.GetIdentifier(), "compile") != 0)
      {
        m_tokenizer.Error("Syntax error: unexpected identifier '%s' expected compile statement", m_tokenizer.GetIdentifier());
        stateAssignment->iValue = 0;
        return false;
      }

      // @@ Parse profile name, function name, argument expressions.

      // Skip the rest of the compile statement.
      while (m_tokenizer.GetToken() != ';')
      {
        m_tokenizer.Next();
      }
    }
    else {
      if (expectsInteger)
      {
        if (!AcceptInt(stateAssignment->iValue))
        {
          m_tokenizer.Error("Syntax error: expected integer near '%s'", m_tokenizer.GetIdentifier());
          stateAssignment->iValue = 0;
          return false;
        }
      }
      else if (expectsFloat)
      {
        if (!AcceptFloat(stateAssignment->fValue))
        {
          m_tokenizer.Error("Syntax error: expected float near '%s'", m_tokenizer.GetIdentifier());
          stateAssignment->iValue = 0;
          return false;
        }
      }
      else if (expectsBoolean)
      {
        const EffectStateValue* stateValue = GetStateValue(m_tokenizer.GetIdentifier(), state);

        if (stateValue != nullptr)
        {
          stateAssignment->iValue = stateValue->value;

          m_tokenizer.Next();
        }
        else if (AcceptInt(stateAssignment->iValue))
        {
          stateAssignment->iValue = (stateAssignment->iValue != 0);
        }
        else {
          m_tokenizer.Error("Syntax error: expected bool near '%s'", m_tokenizer.GetIdentifier());
          stateAssignment->iValue = 0;
          return false;
        }
      }
      else if (expectsExpression)
      {
        if (!ParseColorMask(stateAssignment->iValue))
        {
          m_tokenizer.Error("Syntax error: expected color mask near '%s'", m_tokenizer.GetIdentifier());
          stateAssignment->iValue = 0;
          return false;
        }
      }
      else
      {
        // Expect one of the allowed values.
        const EffectStateValue* stateValue = GetStateValue(m_tokenizer.GetIdentifier(), state);

        if (stateValue == nullptr)
        {
          m_tokenizer.Error("Syntax error: unexpected value '%s' for state '%s'", m_tokenizer.GetIdentifier(), state->name);
          stateAssignment->iValue = 0;
          return false;
        }

        stateAssignment->iValue = stateValue->value;

        m_tokenizer.Next();
      }
    }

    return true;
  }

  bool HLSLParser::ParseStateAssignment(HLSLStateAssignment*& stateAssignment, bool isSamplerState, bool isPipelineState)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    stateAssignment = m_tree->AddNode<HLSLStateAssignment>(fileName, line);

    const EffectState* state;
    if (!ParseStateName(isSamplerState, isPipelineState, stateAssignment->stateName, state)) {
      return false;
    }

    //stateAssignment->name = m_tree->AddString(m_tokenizer.GetIdentifier());
    stateAssignment->stateName = state->name;
    stateAssignment->d3dRenderState = state->d3drs;

    if (!Expect('=')) {
      return false;
    }

    if (!ParseStateValue(state, stateAssignment)) {
      return false;
    }

    if (!Expect(';')) {
      return false;
    }

    return true;
  }


  bool HLSLParser::ParseAttributeList(HLSLAttribute*& firstAttribute)
  {
    const char* fileName = GetFileName();
    int         line = GetLineNumber();

    HLSLAttribute* lastAttribute = firstAttribute;
    do {
      const char* identifier = nullptr;
      if (!ExpectIdentifier(identifier)) {
        return false;
      }

      HLSLAttribute* attribute = m_tree->AddNode<HLSLAttribute>(fileName, line);

      if (strcmp(identifier, "unroll") == 0) attribute->attributeType = HLSLAttributeType_Unroll;
      else if (strcmp(identifier, "flatten") == 0) attribute->attributeType = HLSLAttributeType_Flatten;
      else if (strcmp(identifier, "branch") == 0) attribute->attributeType = HLSLAttributeType_Branch;
      else if (strcmp(identifier, "nofastmath") == 0) attribute->attributeType = HLSLAttributeType_NoFastMath;

      // @@ parse arguments, () not required if attribute constructor has no arguments.

      if (firstAttribute == nullptr)
      {
        firstAttribute = attribute;
      }
      else
      {
        lastAttribute->nextAttribute = attribute;
      }
      lastAttribute = attribute;

    } while (Accept(','));

    return true;
  }

  // Attributes can have all these forms:
  //   [A] statement;
  //   [A,B] statement;
  //   [A][B] statement;
  // These are not supported yet:
  //   [A] statement [B];
  //   [A()] statement;
  //   [A(a)] statement;
  bool HLSLParser::ParseAttributeBlock(HLSLAttribute*& attribute)
  {
    HLSLAttribute** lastAttribute = &attribute;
    while (*lastAttribute != nullptr) { lastAttribute = &(*lastAttribute)->nextAttribute; }

    if (!Accept('['))
    {
      return false;
    }

    // Parse list of attribute constructors.
    ParseAttributeList(*lastAttribute);

    if (!Expect(']'))
    {
      return false;
    }

    // Parse additional [] blocks.
    ParseAttributeBlock(*lastAttribute);

    return true;
  }

  bool HLSLParser::ParseStage(HLSLStatement*& statement)
  {
    if (!Accept("stage"))
    {
      return false;
    }

    // Required stage name.
    const char* stageName = nullptr;
    if (!ExpectIdentifier(stageName))
    {
      return false;
    }

    if (!Expect('{'))
    {
      return false;
    }

    HLSLStage* stage = m_tree->AddNode<HLSLStage>(GetFileName(), GetLineNumber());
    stage->name = stageName;

    BeginScope();

    HLSLType voidType(HLSLBaseType_Void);
    if (!Expect('{') || !ParseBlock(stage->statement, voidType))
    {
      return false;
    }

    EndScope();

    // @@ To finish the stage definition we should traverse the statements recursively (including function calls) and find all the input/output declarations.

    statement = stage;
    return true;
  }




  bool HLSLParser::Parse(const char* fileName, const char* buffer, size_t length)
  {
    HLSLRoot* root = m_tree->GetRoot();
    HLSLStatement* lastStatement = nullptr;

    m_tokenizer = HLSLTokenizer(fileName, buffer, length);
    while (!Accept(HLSLToken_EndOfStream))
    {
      HLSLStatement* statement = nullptr;
      if (!ParseTopLevel(statement))
      {
        return false;
      }
      if (statement != nullptr)
      {
        if (lastStatement == nullptr)
        {
          root->statement = statement;
        }
        else
        {
          lastStatement->nextStatement = statement;
        }
        lastStatement = statement;
        while (lastStatement->nextStatement) lastStatement = lastStatement->nextStatement;
      }
    }
    return true;
  }


  bool HLSLParser::ParsePreprocessorDefine()
  {
    int line = GetLineNumber();
    const char* fileName = GetFileName();

    // Define identifier parsing:
    // Don't use AcceptIdentifier to be able to check for whitespace after define name
    // This is required to separate define name and value as both could contains parenthesis
    const char* macroName = nullptr;
    m_tokenizer.Next();
    if (m_tokenizer.GetToken() == HLSLToken_Identifier)
    {
      macroName = m_tree->AddString(m_tokenizer.GetIdentifier());
    }
    else
    {
      char near_[s_maxIdentifier];
      m_tokenizer.GetTokenName(near_);
      m_tokenizer.Error("Syntax error: expected identifier near '%s'", near_);
      return false;
    }

    bool macroWithArguments = !m_tokenizer.NextIsWhitespace();

    // Macro declaration
    HLSLMacro* macro = m_tree->AddNode<HLSLMacro>(fileName, line);
    macro->name = macroName;
    m_macros.PushBack(macro);

    // Prepare next token
    m_tokenizer.Next();

    std::string value;
    if (macroWithArguments)
    {
      // Macro with arguments
      if (Accept('('))
      {
        unsigned int numArguments = 0;
        HLSLArgument* lastArgument = nullptr;

        while (!Accept(')'))
        {
          if (CheckForUnexpectedEndOfStream(')'))
          {
            return false;
          }
          if (numArguments > 0 && !Expect(','))
          {
            return false;
          }

          HLSLArgument* argument = m_tree->AddNode<HLSLArgument>(fileName, line);
          if (!ExpectIdentifier(argument->name))
          {
            return false;
          }

          if (lastArgument != nullptr)
          {
            lastArgument->nextArgument = argument;
          }
          else
          {
            macro->argument = argument;
          }
          lastArgument = argument;

          ++numArguments;
        }

        macro->numArguments = numArguments;
      }

      // Macro value
      while (m_tokenizer.GetToken() != HLSLToken_EndOfLine)
      {
        bool addOriginalSource = true;

        if (m_tokenizer.GetToken() == HLSLToken_Identifier)
        {
          // Search for macro attributes
          HLSLArgument* argument = macro->argument;
          unsigned argNum = 0;
          while (argument != nullptr)
          {
            if (String_Equal(argument->name, m_tokenizer.GetIdentifier()))
            {
              value.append("#" + std::to_string(argNum) + "#");
              addOriginalSource = false;
              break;
            }

            argument = argument->nextArgument;
            argNum++;
          }
        }

        if (addOriginalSource)
        {
          value.append(m_tokenizer.getLastPos(true),
            m_tokenizer.getCurrentPos() - m_tokenizer.getLastPos(true));
        }

        m_tokenizer.Next(false);
      }


    }
    else
    {
      // Macro value
      while (m_tokenizer.GetToken() != HLSLToken_EndOfLine)
      {
        const char* start = m_tokenizer.getLastPos(true);

        std::string valueAdd(start, m_tokenizer.getCurrentPos() - start);
        value.append(valueAdd);

        m_tokenizer.Next(false);
      }
    }

    // Remove extra parenthesis
    if (value[0] == '(')
    {
      value.erase(value.length() - 1, 1);
      value.erase(0, 1);
    }

    macro->value = value;

    return true;
  }


  bool HLSLParser::ApplyPreprocessor(const char* fileName, const char* buffer, size_t length,
    std::string& sourcePreprocessed) {

    m_tokenizer = HLSLTokenizer(fileName, buffer, length);

    // Fisrt pass, grab #define
    while (m_tokenizer.GetToken() != HLSLToken_EndOfStream)
    {
      if (m_tokenizer.GetToken() == HLSLToken_PreprocessorDefine)
      {
        ParsePreprocessorDefine();
      }

      m_tokenizer.Next();
    }


    // Second pass, propagate macros definitions across macros values
    int index = 0;
    while (index < m_macros.GetSize())
    {
      HLSLMacro* macro = m_macros[index];
      m_tokenizer = HLSLTokenizer(fileName, macro->value.c_str(), macro->value.length());
      std::string valueProcessed;

      while (m_tokenizer.GetToken() != HLSLToken_EndOfStream)
      {
        bool addOriginalSource = true;
        if (m_tokenizer.GetToken() == HLSLToken_Identifier)
        {
          HLSLMacro* matchedMacro = ProcessMacroFromIdentifier(valueProcessed, addOriginalSource);
          if (matchedMacro != nullptr)
          {
            // Check if macro is just an alias
            // macros value must be equal to matched macro name
            if (macro->value == matchedMacro->name)
            {
              macro->macroAliased = matchedMacro;
            }
          }
        }

        if (addOriginalSource)
        {
          valueProcessed.append(m_tokenizer.getLastPos(true),
            m_tokenizer.getCurrentPos() - m_tokenizer.getLastPos(true));
        }

        m_tokenizer.Next();
      }

      if (valueProcessed == "main")
      {
        valueProcessed = "sampler_fw_main";
      }

      if (valueProcessed != macro->value)
      {
        // Define value referenced another define, it was replaced
        // try again until there is no replacement
        macro->value = valueProcessed;

      }
      else
      {
        index++;
      }
    }

    // Third pass, macro aliases get original value and arguments
    index = 0;
    while (index < m_macros.GetSize())
    {
      HLSLMacro* macro = m_macros[index];

      if (macro->macroAliased != nullptr)
      {
        macro->argument = macro->macroAliased->argument;
        macro->numArguments = macro->macroAliased->numArguments;
        macro->value = macro->macroAliased->value;
      }

      index++;
    }

    // Fouth pass, search and replace preprocessor directives
    std::stack<bool> isCodeActive;
    isCodeActive.push(true);
    m_tokenizer = HLSLTokenizer(fileName, buffer, length);
    sourcePreprocessed.clear();
    while (m_tokenizer.GetToken() != HLSLToken_EndOfStream)
    {
      bool addOriginalSource = true;

      if (m_tokenizer.GetToken() == HLSLToken_PreprocessorIf)
      {
        while (m_tokenizer.GetToken() != HLSLToken_IntLiteral && m_tokenizer.GetToken() != HLSLToken_EndOfLine)
        {
          m_tokenizer.Next(false);
        }

        if (m_tokenizer.GetToken() == HLSLToken_IntLiteral)
        {
          isCodeActive.push(m_tokenizer.GetInt() != 0);
        }
        else
        {
          m_tokenizer.Error("#if evaluation failed: not an integer");
          return false;
        }
        addOriginalSource = false;
      }
      else if (m_tokenizer.GetToken() == HLSLToken_PreprocessorElse)
      {
        // Invert stack state
        bool state = isCodeActive.top();
        isCodeActive.pop();
        isCodeActive.push(!state);
        addOriginalSource = false;
      }
      else if (m_tokenizer.GetToken() == HLSLToken_PreprocessorEndif)
      {
        isCodeActive.pop();
        addOriginalSource = false;
      }
      else if (m_tokenizer.GetToken() == HLSLToken_PreprocessorDefine)
      {
        // Skip macros definition
        while (m_tokenizer.GetToken() != HLSLToken_EndOfLine)
        {
          m_tokenizer.Next(false);
        }

        addOriginalSource = false;
      }
      else if (m_tokenizer.GetToken() == HLSLToken_Identifier && isCodeActive.top())
      {
        ProcessMacroFromIdentifier(sourcePreprocessed, addOriginalSource);
      }
      else if (!isCodeActive.top())
      {
        addOriginalSource = false;
      }

      if (addOriginalSource)
      {
        sourcePreprocessed.append(m_tokenizer.getLastPos(false),
          m_tokenizer.getCurrentPos() - m_tokenizer.getLastPos(false));
      }

      m_tokenizer.Next();
    }


    return isCodeActive.size() == 1;
  }


  HLSLMacro* HLSLParser::ProcessMacroFromIdentifier(std::string& sourcePreprocessed, bool& addOriginalSource)
  {
    // Search a define matching
    for (int i = m_macros.GetSize() - 1; i >= 0; --i)
    {
      if (String_Equal(m_macros[i]->name, m_tokenizer.GetIdentifier()))
      {
        if (m_macros[i]->argument == nullptr)
        {
          // Macro without arguments
          sourcePreprocessed.append("(");
          sourcePreprocessed.append(m_macros[i]->value);
          sourcePreprocessed.append(")");
          addOriginalSource = false;
        }
        else
        {
          // Macro with arguments
          const char* savePos = m_tokenizer.getLastPos(false);

          m_tokenizer.Next();

          if (ProcessMacroArguments(m_macros[i], sourcePreprocessed))
          {
            // Macro replaced
            addOriginalSource = false;
          }
          else
          {
            // Macro signatures does not match: writing back readed parts
            sourcePreprocessed.append(savePos,
              m_tokenizer.getLastPos(false) - savePos);

            m_tokenizer.ReturnToPos(m_tokenizer.getLastPos(false));

            addOriginalSource = true;
          }
        }

        return m_macros[i];
      }
    }

    return nullptr;
  }



  bool HLSLParser::ProcessMacroArguments(HLSLMacro* macro, std::string& sourcePreprocessed)
  {
    unsigned scopeLevel = 0;
    std::vector<std::string> argumentsValues;
    std::string argValue;
    bool firstToken = true;

    // Parse arguments values
    while (m_tokenizer.GetToken() != HLSLToken_EndOfStream) {
      bool addToValue = true;

      if (firstToken && m_tokenizer.GetToken() != '(')
      {
        break;
      }
      else if (m_tokenizer.GetToken() == '(')
      {
        scopeLevel++;
        if (scopeLevel == 1)
        {
          addToValue = false;
        }
      }
      else if (m_tokenizer.GetToken() == ')')
      {
        scopeLevel--;
        if (scopeLevel == 0)
        {
          argumentsValues.push_back(argValue);
          break;
        }
      }
      else if (m_tokenizer.GetToken() == ',')
      {
        if (scopeLevel == 1)
        {
          argumentsValues.push_back(argValue);
          argValue.clear();
          addToValue = false;
        }
      }
      else if (m_tokenizer.GetToken() == HLSLToken_Identifier)
      {
        ProcessMacroFromIdentifier(argValue, addToValue);
      }

      if (addToValue)
      {
        argValue.append(m_tokenizer.getLastPos(false),
          m_tokenizer.getCurrentPos() - m_tokenizer.getLastPos(false));
      }

      m_tokenizer.Next();
      firstToken = false;
    }

    // Arguments count does not match
    if (argumentsValues.size() != macro->numArguments) {
      return false;
    }

    // Surround macro definition with parenthesis
    sourcePreprocessed.append("(");

    // Write arguments value
    unsigned index = 0;
    std::string arg;
    bool argFound = false;
    while (index < macro->value.length())
    {
      if (macro->value[index] == '#' && !argFound)
      {
        argFound = true;
        arg.clear();
      }
      else if (macro->value[index] == '#' && argFound)
      {
        unsigned indexArg = std::stoi(arg);
        if (indexArg < argumentsValues.size())
        {
          sourcePreprocessed.append(argumentsValues[indexArg]);
        }
        argFound = false;
      }
      else if (argFound)
      {
        arg += macro->value[index];
      }
      else
      {
        sourcePreprocessed += macro->value[index];
      }

      index++;
    }

    // Surround macro definition with parenthesis
    sourcePreprocessed.append(")");

    return true;
  }


  bool HLSLParser::AcceptTypeModifier(int& flags)
  {
    if (Accept(HLSLToken_Const))
    {
      flags |= HLSLTypeFlag_Const;
      return true;
    }
    else if (Accept(HLSLToken_Static))
    {
      flags |= HLSLTypeFlag_Static;
      return true;
    }
    else if (Accept(HLSLToken_Uniform))
    {
      flags |= HLSLTypeFlag_Uniform;
      return true;
    }
    else if (Accept(HLSLToken_Inline))
    {
      //flags |= HLSLTypeFlag_Uniform;      // @@ Ignored. In HLSL all functions are inline.
      return true;
    }
    /*else if (Accept("in"))
    {
        flags |= HLSLTypeFlag_Input;
        return true;
    }
    else if (Accept("out"))
    {
        flags |= HLSLTypeFlag_Output;
        return true;
    }*/

    // Not an usage keyword.
    return false;
  }

  bool HLSLParser::AcceptInterpolationModifier(int& flags)
  {
    if (Accept("linear"))
    {
      flags |= HLSLTypeFlag_Linear;
      return true;
    }
    else if (Accept("centroid"))
    {
      flags |= HLSLTypeFlag_Centroid;
      return true;
    }
    else if (Accept("nointerpolation"))
    {
      flags |= HLSLTypeFlag_NoInterpolation;
      return true;
    }
    else if (Accept("noperspective"))
    {
      flags |= HLSLTypeFlag_NoPerspective;
      return true;
    }
    else if (Accept("sample"))
    {
      flags |= HLSLTypeFlag_Sample;
      return true;
    }

    return false;
  }


  bool HLSLParser::AcceptType(bool allowVoid, HLSLType& type/*, bool acceptFlags*/)
  {
    //if (type.flags != nullptr)
    {
      type.flags = 0;
      while (AcceptTypeModifier(type.flags) || AcceptInterpolationModifier(type.flags)) {}
    }

    int token = m_tokenizer.GetToken();

    // Check built-in types.
    type.baseType = HLSLBaseType_Void;
    switch (token)
    {
    case HLSLToken_Float:
    case HLSLToken_Float1:
    case HLSLToken_Float1x1:
    case HLSLToken_Half:
    case HLSLToken_Half1:
    case HLSLToken_Half1x1:
    case HLSLToken_Double:
    case HLSLToken_Double1:
    case HLSLToken_Double1x1:
      type.baseType = HLSLBaseType_Float;
      break;
    case HLSLToken_Float2:
    case HLSLToken_Float2x1:
    case HLSLToken_Half2:
    case HLSLToken_Half2x1:
    case HLSLToken_Double2:
    case HLSLToken_Double2x1:
      type.baseType = HLSLBaseType_Float2;
      break;
    case HLSLToken_Float3:
    case HLSLToken_Float3x1:
    case HLSLToken_Half3:
    case HLSLToken_Half3x1:
    case HLSLToken_Double3:
    case HLSLToken_Double3x1:
      type.baseType = HLSLBaseType_Float3;
      break;
    case HLSLToken_Float4:
    case HLSLToken_Float4x1:
    case HLSLToken_Half4:
    case HLSLToken_Half4x1:
    case HLSLToken_Double4:
    case HLSLToken_Double4x1:
      type.baseType = HLSLBaseType_Float4;
      break;
    case HLSLToken_Float2x4:
    case HLSLToken_Half2x4:
    case HLSLToken_Double2x4:
      type.baseType = HLSLBaseType_Float2x4;
      break;
    case HLSLToken_Float2x3:
    case HLSLToken_Half2x3:
    case HLSLToken_Double2x3:
      type.baseType = HLSLBaseType_Float2x3;
      break;
    case HLSLToken_Float2x2:
    case HLSLToken_Half2x2:
    case HLSLToken_Double2x2:
      type.baseType = HLSLBaseType_Float2x2;
      break;
    case HLSLToken_Float3x4:
    case HLSLToken_Half3x4:
    case HLSLToken_Double3x4:
      type.baseType = HLSLBaseType_Float3x4;
      break;
    case HLSLToken_Float3x3:
    case HLSLToken_Half3x3:
    case HLSLToken_Double3x3:
      type.baseType = HLSLBaseType_Float3x3;
      break;
    case HLSLToken_Float3x2:
    case HLSLToken_Half3x2:
    case HLSLToken_Double3x2:
      type.baseType = HLSLBaseType_Float3x2;
      break;
    case HLSLToken_Float4x4:
    case HLSLToken_Half4x4:
    case HLSLToken_Double4x4:
      type.baseType = HLSLBaseType_Float4x4;
      break;
    case HLSLToken_Float4x3:
    case HLSLToken_Half4x3:
    case HLSLToken_Double4x3:
      type.baseType = HLSLBaseType_Float4x3;
      break;
    case HLSLToken_Float4x2:
    case HLSLToken_Half4x2:
    case HLSLToken_Double4x2:
      type.baseType = HLSLBaseType_Float4x2;
      break;
    case HLSLToken_Bool:
      type.baseType = HLSLBaseType_Bool;
      break;
    case HLSLToken_Bool2:
      type.baseType = HLSLBaseType_Bool2;
      break;
    case HLSLToken_Bool3:
      type.baseType = HLSLBaseType_Bool3;
      break;
    case HLSLToken_Bool4:
      type.baseType = HLSLBaseType_Bool4;
      break;
    case HLSLToken_Int:
      type.baseType = HLSLBaseType_Int;
      break;
    case HLSLToken_Int2:
      type.baseType = HLSLBaseType_Int2;
      break;
    case HLSLToken_Int3:
      type.baseType = HLSLBaseType_Int3;
      break;
    case HLSLToken_Int4:
      type.baseType = HLSLBaseType_Int4;
      break;
    case HLSLToken_Uint:
      type.baseType = HLSLBaseType_Uint;
      break;
    case HLSLToken_Uint2:
      type.baseType = HLSLBaseType_Uint2;
      break;
    case HLSLToken_Uint3:
      type.baseType = HLSLBaseType_Uint3;
      break;
    case HLSLToken_Uint4:
      type.baseType = HLSLBaseType_Uint4;
      break;
    case HLSLToken_Texture:
      type.baseType = HLSLBaseType_Texture;
      break;
    case HLSLToken_Sampler:
      type.baseType = HLSLBaseType_Sampler2D;  // @@ IC: For now we assume that generic samplers are always sampler2D
      break;
    case HLSLToken_Sampler2D:
      type.baseType = HLSLBaseType_Sampler2D;
      break;
    case HLSLToken_Sampler3D:
      type.baseType = HLSLBaseType_Sampler3D;
      break;
    case HLSLToken_SamplerCube:
      type.baseType = HLSLBaseType_SamplerCube;
      break;
    case HLSLToken_Sampler2DShadow:
      type.baseType = HLSLBaseType_Sampler2DShadow;
      break;
    case HLSLToken_Sampler2DMS:
      type.baseType = HLSLBaseType_Sampler2DMS;
      break;
    case HLSLToken_Sampler2DArray:
      type.baseType = HLSLBaseType_Sampler2DArray;
      break;
    }
    if (type.baseType != HLSLBaseType_Void)
    {
      m_tokenizer.Next();

      if (IsSamplerType(type.baseType))
      {
        // Parse optional sampler type.
        if (Accept('<'))
        {
          int token2 = m_tokenizer.GetToken();
          if (token2 == HLSLToken_Float || token2 == HLSLToken_Float1)
          {
            type.samplerType = HLSLBaseType_Float;
          }
          else
          {
            m_tokenizer.Error("Expected half or float.");
            return false;
          }
          m_tokenizer.Next();

          if (!Expect('>'))
          {
            m_tokenizer.Error("Syntax error: '>' expected for sampler type");
            return false;
          }
        }
      }
      return true;
    }

    if (allowVoid && Accept(HLSLToken_Void))
    {
      type.baseType = HLSLBaseType_Void;
      return true;
    }
    if (token == HLSLToken_Identifier)
    {
      const char* identifier = m_tree->AddString(m_tokenizer.GetIdentifier());
      if (FindUserDefinedType(identifier) != nullptr)
      {
        m_tokenizer.Next();
        type.baseType = HLSLBaseType_UserDefined;
        type.typeName = identifier;
        return true;
      }
    }
    return false;
  }

  bool HLSLParser::ExpectType(bool allowVoid, HLSLType& type)
  {
    if (!AcceptType(allowVoid, type))
    {
      m_tokenizer.Error("Expected type");
      return false;
    }
    return true;
  }

  bool HLSLParser::AcceptDeclaration(bool allowUnsizedArray, HLSLType& type, const char*& name)
  {
    if (!AcceptType(/*allowVoid=*/false, type))
    {
      return false;
    }

    if (!ExpectIdentifier(name))
    {
      // TODO: false means we didn't accept a declaration and we had an error!
      return false;
    }
    // Handle array syntax.
    if (Accept('['))
    {
      type.array = true;
      // Optionally allow no size to the specified for the array.
      if (Accept(']') && allowUnsizedArray)
      {
        return true;
      }
      if (!ParseExpression(type.arraySize) || !Expect(']'))
      {
        return false;
      }
    }
    return true;
  }

  bool HLSLParser::ExpectDeclaration(bool allowUnsizedArray, HLSLType& type, const char*& name)
  {
    if (!AcceptDeclaration(allowUnsizedArray, type, name))
    {
      m_tokenizer.Error("Expected declaration");
      return false;
    }
    return true;
  }

  const HLSLStruct* HLSLParser::FindUserDefinedType(const char* name) const
  {
    // Pointer comparison is sufficient for strings since they exist in the
    // string pool.
    for (int i = 0; i < m_userTypes.GetSize(); ++i)
    {
      if (m_userTypes[i]->name == name)
      {
        return m_userTypes[i];
      }
    }
    return nullptr;
  }

  bool HLSLParser::CheckForUnexpectedEndOfStream(int endToken)
  {
    if (Accept(HLSLToken_EndOfStream))
    {
      char what[s_maxIdentifier];
      m_tokenizer.GetTokenName(endToken, what);
      m_tokenizer.Error("Unexpected end of file while looking for '%s'", what);
      return true;
    }
    return false;
  }

  int HLSLParser::GetLineNumber() const
  {
    return m_tokenizer.GetLineNumber();
  }

  const char* HLSLParser::GetFileName()
  {
    return m_tree->AddString(m_tokenizer.GetFileName());
  }

  void HLSLParser::BeginScope()
  {
    // Use nullptr as a sentinel that indices a new scope level.
    Variable& variable = m_variables.PushBackNew();
    variable.name = nullptr;
  }

  void HLSLParser::EndScope()
  {
    int numVariables = m_variables.GetSize() - 1;
    while (m_variables[numVariables].name != nullptr)
    {
      --numVariables;
      ASSERT(numVariables >= 0);
    }
    m_variables.Resize(numVariables);
  }

  const HLSLType* HLSLParser::FindVariable(const char* name, bool& global) const
  {
    for (int i = m_variables.GetSize() - 1; i >= 0; --i)
    {
      if (m_variables[i].name == name)
      {
        global = (i < m_numGlobals);
        return &m_variables[i].type;
      }
    }
    return nullptr;
  }

  const HLSLFunction* HLSLParser::FindFunction(const char* name) const
  {
    for (int i = 0; i < m_functions.GetSize(); ++i)
    {
      if (m_functions[i]->name == name)
      {
        return m_functions[i];
      }
    }
    return nullptr;
  }

  static bool AreTypesEqual(HLSLTree* tree, const HLSLType& lhs, const HLSLType& rhs)
  {
    return GetTypeCastRank(tree, lhs, rhs) == 0;
  }

  static bool AreArgumentListsEqual(HLSLTree* tree, HLSLArgument* lhs, HLSLArgument* rhs)
  {
    while (lhs && rhs)
    {
      if (!AreTypesEqual(tree, lhs->type, rhs->type))
        return false;

      if (lhs->modifier != rhs->modifier)
        return false;

      if (lhs->semantic != rhs->semantic || lhs->sv_semantic != rhs->sv_semantic)
        return false;

      lhs = lhs->nextArgument;
      rhs = rhs->nextArgument;
    }

    return lhs == nullptr && rhs == nullptr;
  }

  const HLSLFunction* HLSLParser::FindFunction(const HLSLFunction* fun) const
  {
    for (int i = 0; i < m_functions.GetSize(); ++i)
    {
      if (m_functions[i]->name == fun->name &&
        AreTypesEqual(m_tree, m_functions[i]->returnType, fun->returnType) &&
        AreArgumentListsEqual(m_tree, m_functions[i]->argument, fun->argument))
      {
        return m_functions[i];
      }
    }
    return nullptr;
  }

  void HLSLParser::DeclareVariable(const char* name, const HLSLType& type)
  {
    if (m_variables.GetSize() == m_numGlobals)
    {
      ++m_numGlobals;
    }
    Variable& variable = m_variables.PushBackNew();
    variable.name = name;
    variable.type = type;
  }

  bool HLSLParser::GetIsFunction(const char* name) const
  {
    for (int i = 0; i < m_functions.GetSize(); ++i)
    {
      // == is ok here because we're passed the strings through the string pool.
      if (m_functions[i]->name == name)
      {
        return true;
      }
    }
    for (int i = 0; i < _numIntrinsics; ++i)
    {
      // Intrinsic names are not in the string pool (since they are compile time
      // constants, so we need full string compare).
      if (String_Equal(name, _intrinsic[i].function.name))
      {
        return true;
      }
    }

    return false;
  }

  const HLSLFunction* HLSLParser::MatchFunctionCall(const HLSLFunctionCall* functionCall, const char* name)
  {
    const HLSLFunction* matchedFunction = nullptr;

    int  numMatchedOverloads = 0;
    bool nameMatches = false;

    // Get the user defined functions with the specified name.
    for (int i = 0; i < m_functions.GetSize(); ++i)
    {
      const HLSLFunction* function = m_functions[i];
      if (function->name == name)
      {
        nameMatches = true;

        CompareFunctionsResult result = CompareFunctions(m_tree, functionCall, function, matchedFunction);
        if (result == Function1Better)
        {
          matchedFunction = function;
          numMatchedOverloads = 1;
        }
        else if (result == FunctionsEqual)
        {
          ++numMatchedOverloads;
        }
      }
    }

    // Get the intrinsic functions with the specified name.
    for (int i = 0; i < _numIntrinsics; ++i)
    {
      const HLSLFunction* function = &_intrinsic[i].function;
      if (String_Equal(function->name, name))
      {
        nameMatches = true;

        CompareFunctionsResult result = CompareFunctions(m_tree, functionCall, function, matchedFunction);
        if (result == Function1Better)
        {
          matchedFunction = function;
          numMatchedOverloads = 1;
        }
        else if (result == FunctionsEqual)
        {
          ++numMatchedOverloads;
        }
      }
    }

    if (matchedFunction != nullptr && numMatchedOverloads > 1)
    {
      // Multiple overloads match.
      m_tokenizer.Error("'%s' %d overloads have similar conversions", name, numMatchedOverloads);
      return nullptr;
    }
    else if (matchedFunction == nullptr)
    {
      if (nameMatches)
      {
        m_tokenizer.Error("'%s' no overloaded function matched all of the arguments", name);
      }
      else
      {
        m_tokenizer.Error("Undeclared identifier '%s'", name);
      }
    }

    return matchedFunction;
  }

  bool HLSLParser::GetMemberType(const HLSLType& objectType, HLSLMemberAccess* memberAccess)
  {
    const char* fieldName = memberAccess->field;

    if (objectType.baseType == HLSLBaseType_UserDefined)
    {
      const HLSLStruct* structure = FindUserDefinedType(objectType.typeName);
      ASSERT(structure != nullptr);

      const HLSLStructField* field = structure->field;
      while (field != nullptr)
      {
        if (field->name == fieldName)
        {
          memberAccess->expressionType = field->type;
          return true;
        }
        field = field->nextField;
      }

      return false;
    }

    if (baseTypeDescriptions[objectType.baseType].numericType == NumericType_NaN)
    {
      // Currently we don't have an non-numeric types that allow member access.
      return false;
    }

    int swizzleLength = 0;

    if (baseTypeDescriptions[objectType.baseType].numDimensions <= 1)
    {
      // Check for a swizzle on the scalar/vector types.
      for (int i = 0; fieldName[i] != 0; ++i)
      {
        if (fieldName[i] != 'x' && fieldName[i] != 'y' && fieldName[i] != 'z' && fieldName[i] != 'w' &&
          fieldName[i] != 'r' && fieldName[i] != 'g' && fieldName[i] != 'b' && fieldName[i] != 'a')
        {
          m_tokenizer.Error("Invalid swizzle '%s'", fieldName);
          return false;
        }
        ++swizzleLength;
      }
      ASSERT(swizzleLength > 0);
    }
    else
    {

      // Check for a matrix element access (e.g. _m00 or _11)

      const char* n = fieldName;
      while (n[0] == '_')
      {
        ++n;
        int base = 1;
        if (n[0] == 'm')
        {
          base = 0;
          ++n;
        }
        if (!isdigit(n[0]) || !isdigit(n[1]))
        {
          return false;
        }

        int r = (n[0] - '0') - base;
        int c = (n[1] - '0') - base;
        if (r >= baseTypeDescriptions[objectType.baseType].height ||
          c >= baseTypeDescriptions[objectType.baseType].numComponents)
        {
          return false;
        }
        ++swizzleLength;
        n += 2;

      }

      if (n[0] != 0)
      {
        return false;
      }

    }

    if (swizzleLength > 4)
    {
      m_tokenizer.Error("Invalid swizzle '%s'", fieldName);
      return false;
    }

    static const HLSLBaseType floatType[] = { HLSLBaseType_Float, HLSLBaseType_Float2, HLSLBaseType_Float3, HLSLBaseType_Float4 };
    static const HLSLBaseType intType[] = { HLSLBaseType_Int,   HLSLBaseType_Int2,   HLSLBaseType_Int3,   HLSLBaseType_Int4 };
    static const HLSLBaseType uintType[] = { HLSLBaseType_Uint,  HLSLBaseType_Uint2,  HLSLBaseType_Uint3,  HLSLBaseType_Uint4 };
    static const HLSLBaseType boolType[] = { HLSLBaseType_Bool,  HLSLBaseType_Bool2,  HLSLBaseType_Bool3,  HLSLBaseType_Bool4 };

    switch (baseTypeDescriptions[objectType.baseType].numericType)
    {
    case NumericType_Float:
      memberAccess->expressionType.baseType = floatType[swizzleLength - 1];
      break;
    case NumericType_Int:
      memberAccess->expressionType.baseType = intType[swizzleLength - 1];
      break;
    case NumericType_Uint:
      memberAccess->expressionType.baseType = uintType[swizzleLength - 1];
      break;
    case NumericType_Bool:
      memberAccess->expressionType.baseType = boolType[swizzleLength - 1];
      break;
    default:
      ASSERT(0);
    }

    memberAccess->swizzle = true;

    return true;
  }

}


namespace M4
{

  class GLSLGenerator
  {

  public:
    enum Target
    {
      Target_VertexShader,
      Target_FragmentShader,
    };

    enum Version
    {
      Version_110, // OpenGL 2.0
      Version_120, // OpenGL 2.1
      Version_140, // OpenGL 3.1
      Version_150, // OpenGL 3.2
      Version_330, // OpenGL 3.3
      Version_100_ES, // OpenGL ES 2.0
      Version_300_ES, // OpenGL ES 3.0
    };

    enum Flags
    {
      Flag_FlipPositionOutput = 1 << 0,
      Flag_EmulateConstantBuffer = 1 << 1,
      Flag_PackMatrixRowMajor = 1 << 2,
      Flag_LowerMatrixMultiplication = 1 << 3,
    };

    struct Options
    {
      unsigned int flags;
      const char* constantBufferPrefix;

      Options()
      {
        flags = 0;
        constantBufferPrefix = "";
      }
    };

    GLSLGenerator();

    bool Generate(HLSLTree* tree, Target target, Version versiom, const char* entryName, const Options& options = Options());
    const char* GetResult() const;

  private:

    enum AttributeModifier
    {
      AttributeModifier_In,
      AttributeModifier_Out,
    };

    void OutputExpressionList(HLSLExpression* expression, HLSLArgument* argument = nullptr);
    void OutputExpression(HLSLExpression* expression, const HLSLType* dstType = nullptr);
    void OutputIdentifier(const char* name);
    void OutputArguments(HLSLArgument* argument);

    /**
     * If the statements are part of a function, then returnType can be used to specify the type
     * that a return statement is expected to produce so that correct casts will be generated.
     */
    void OutputStatements(int indent, HLSLStatement* statement, const HLSLType* returnType = nullptr);

    void OutputAttribute(const HLSLType& type, const char* semantic, AttributeModifier modifier);
    void OutputAttributes(HLSLFunction* entryFunction);
    void OutputEntryCaller(HLSLFunction* entryFunction);
    void OutputDeclaration(HLSLDeclaration* declaration, const bool skipAssignment);
    void OutputDeclarationType(const HLSLType& type);
    void OutputDeclarationBody(const HLSLType& type, const char* name);
    void OutputDeclaration(const HLSLType& type, const char* name);
    void OutputDeclarationAssignment(HLSLDeclaration* declaration);
    void OutputCast(const HLSLType& type);

    void OutputSetOutAttribute(const char* semantic, const char* resultName);

    void LayoutBuffer(HLSLBuffer* buffer, unsigned int& offset);
    void LayoutBuffer(const HLSLType& type, unsigned int& offset);
    void LayoutBufferElement(const HLSLType& type, unsigned int& offset);
    void LayoutBufferAlign(const HLSLType& type, unsigned int& offset);

    HLSLBuffer* GetBufferAccessExpression(HLSLExpression* expression);
    void OutputBufferAccessExpression(HLSLBuffer* buffer, HLSLExpression* expression, const HLSLType& type, unsigned int postOffset);
    unsigned int OutputBufferAccessIndex(HLSLExpression* expression, unsigned int postOffset);

    void OutputBuffer(int indent, HLSLBuffer* buffer);

    HLSLFunction* FindFunction(HLSLRoot* root, const char* name);
    HLSLStruct* FindStruct(HLSLRoot* root, const char* name);

    void Error(const char* format, ...);

    /** GLSL contains some reserved words that don't exist in HLSL. This function will
     * sanitize those names. */
    const char* GetSafeIdentifierName(const char* name) const;

    /** Generates a name of the format "base+n" where n is an integer such that the name
     * isn't used in the syntax tree. */
    bool ChooseUniqueName(const char* base, char* dst, int dstLength) const;

    const char* GetBuiltInSemantic(const char* semantic, AttributeModifier modifier, int* outputIndex = 0);
    const char* GetAttribQualifier(AttributeModifier modifier);
    void CompleteConstructorArguments(HLSLExpression* expression, HLSLBaseType dstType);
    void OutputMatrixCtors();

  private:

    static const int    s_numReservedWords = 9;
    static const char* s_reservedWord[s_numReservedWords];

    CodeWriter          m_writer;

    HLSLTree* m_tree;
    const char* m_entryName;
    Target              m_target;
    Version             m_version;
    bool                m_versionLegacy;
    Options             m_options;

    bool                m_outputPosition;
    int                 m_outputTargets;

    const char* m_outAttribPrefix;
    const char* m_inAttribPrefix;

    char                m_matrixRowFunction[64];
    char                m_matrixCtorFunction[64];
    char                m_matrixMulFunction[64];
    char                m_clipFunction[64];
    char                m_tex2DlodFunction[64];
    char                m_tex2DbiasFunction[64];
    char                m_tex2DgradFunction[64];
    char                m_tex3DlodFunction[64];
    char                m_texCUBEbiasFunction[64];
    char                m_texCUBElodFunction[64];
    char                m_scalarSwizzle2Function[64];
    char                m_scalarSwizzle3Function[64];
    char                m_scalarSwizzle4Function[64];
    char                m_sinCosFunction[64];
    char                m_bvecTernary[64];
    char                m_modfFunction[64];

    bool                m_error;

    char                m_reservedWord[s_numReservedWords][64];

    std::vector<matrixCtor> matrixCtors;
    std::map<matrixCtor, std::string> matrixCtorsId;
    std::vector<HLSLDeclaration*> globalVarsAssignments;

  };

}

namespace M4
{
  static const HLSLType kFloatType(HLSLBaseType_Float);
  static const HLSLType kUintType(HLSLBaseType_Uint);
  static const HLSLType kIntType(HLSLBaseType_Int);
  static const HLSLType kBoolType(HLSLBaseType_Bool);

  // These are reserved words in GLSL that aren't reserved in HLSL.
  const char* GLSLGenerator::s_reservedWord[] =
  {
      "output",
      "input",
      "mod",
      "mix",
      "fract",
      "dFdx",
      "dFdy",
      "filter",
  "main",
  };

  static const char* GetTypeName(const HLSLType& type)
  {
    switch (type.baseType)
    {
    case HLSLBaseType_Void:         return "void";
    case HLSLBaseType_Float:        return "float";
    case HLSLBaseType_Float2:       return "vec2";
    case HLSLBaseType_Float3:       return "vec3";
    case HLSLBaseType_Float4:       return "vec4";
    case HLSLBaseType_Float2x4:     return "mat4x2";
    case HLSLBaseType_Float2x3:     return "mat3x2";
    case HLSLBaseType_Float2x2:     return "mat2";
    case HLSLBaseType_Float3x4:     return "mat4x3";
    case HLSLBaseType_Float3x3:     return "mat3";
    case HLSLBaseType_Float3x2:     return "mat2x3";
    case HLSLBaseType_Float4x4:     return "mat4";
    case HLSLBaseType_Float4x3:     return "mat3x4";
    case HLSLBaseType_Float4x2:     return "mat2x4";
    case HLSLBaseType_Bool:         return "bool";
    case HLSLBaseType_Bool2:        return "bvec2";
    case HLSLBaseType_Bool3:        return "bvec3";
    case HLSLBaseType_Bool4:        return "bvec4";
    case HLSLBaseType_Int:          return "int";
    case HLSLBaseType_Int2:         return "ivec2";
    case HLSLBaseType_Int3:         return "ivec3";
    case HLSLBaseType_Int4:         return "ivec4";
    case HLSLBaseType_Uint:         return "uint";
    case HLSLBaseType_Uint2:        return "uvec2";
    case HLSLBaseType_Uint3:        return "uvec3";
    case HLSLBaseType_Uint4:        return "uvec4";
    case HLSLBaseType_Texture:      return "texture";
    case HLSLBaseType_Sampler:      return "sampler";
    case HLSLBaseType_Sampler2D:    return "sampler2D";
    case HLSLBaseType_Sampler3D:    return "sampler3D";
    case HLSLBaseType_SamplerCube:  return "samplerCube";
    case HLSLBaseType_Sampler2DMS:  return "sampler2DMS";
    case HLSLBaseType_Sampler2DArray:  return "sampler2DArray";
    case HLSLBaseType_UserDefined:  return type.typeName;
    default:
      ASSERT(0);
      return "?";
    }
  }

  static bool GetCanImplicitCast(const HLSLType& srcType, const HLSLType& dstType)
  {
    return srcType.baseType == dstType.baseType;
  }

  static int GetFunctionArguments(HLSLFunctionCall* functionCall, HLSLExpression* expression[], int maxArguments)
  {
    HLSLExpression* argument = functionCall->argument;
    int numArguments = 0;
    while (argument != nullptr)
    {
      if (numArguments < maxArguments)
      {
        expression[numArguments] = argument;
      }
      argument = argument->nextExpression;
      ++numArguments;
    }
    return numArguments;
  }

  GLSLGenerator::GLSLGenerator() :
    m_writer(/* writeFileNames= */ false)
  {
    m_tree = nullptr;
    m_entryName = nullptr;
    m_target = Target_VertexShader;
    m_version = Version_330;
    m_versionLegacy = false;
    m_inAttribPrefix = nullptr;
    m_outAttribPrefix = nullptr;
    m_error = false;
    m_matrixRowFunction[0] = 0;
    m_matrixCtorFunction[0] = 0;
    m_matrixMulFunction[0] = 0;
    m_clipFunction[0] = 0;
    m_tex2DlodFunction[0] = 0;
    m_tex2DbiasFunction[0] = 0;
    m_tex3DlodFunction[0] = 0;
    m_texCUBEbiasFunction[0] = 0;
    m_texCUBElodFunction[0] = 0;
    m_scalarSwizzle2Function[0] = 0;
    m_scalarSwizzle3Function[0] = 0;
    m_scalarSwizzle4Function[0] = 0;
    m_sinCosFunction[0] = 0;
    m_bvecTernary[0] = 0;
    m_modfFunction[0] = 0;
    m_outputPosition = false;
    m_outputTargets = 0;
  }

  bool GLSLGenerator::Generate(HLSLTree* tree, Target target, Version version, const char* entryName, const Options& options)
  {

    m_tree = tree;
    m_entryName = entryName;
    m_target = target;
    m_version = version;
    m_versionLegacy = (version == Version_110 || version == Version_120 || version == Version_100_ES);
    m_options = options;

    globalVarsAssignments.clear();

    ChooseUniqueName("matrix_row", m_matrixRowFunction, sizeof(m_matrixRowFunction));
    ChooseUniqueName("matrix_ctor", m_matrixCtorFunction, sizeof(m_matrixCtorFunction));
    ChooseUniqueName("matrix_mul", m_matrixMulFunction, sizeof(m_matrixMulFunction));
    ChooseUniqueName("clip", m_clipFunction, sizeof(m_clipFunction));
    ChooseUniqueName("tex2Dlod", m_tex2DlodFunction, sizeof(m_tex2DlodFunction));
    ChooseUniqueName("tex2Dbias", m_tex2DbiasFunction, sizeof(m_tex2DbiasFunction));
    ChooseUniqueName("tex2Dgrad", m_tex2DgradFunction, sizeof(m_tex2DgradFunction));
    ChooseUniqueName("tex3Dlod", m_tex3DlodFunction, sizeof(m_tex3DlodFunction));
    ChooseUniqueName("texCUBEbias", m_texCUBEbiasFunction, sizeof(m_texCUBEbiasFunction));
    ChooseUniqueName("texCUBElod", m_texCUBElodFunction, sizeof(m_texCUBElodFunction));
    ChooseUniqueName("modf", m_modfFunction, sizeof(m_modfFunction));

    for (int i = 0; i < s_numReservedWords; ++i)
    {
      ChooseUniqueName(s_reservedWord[i], m_reservedWord[i], sizeof(m_reservedWord[i]));
    }

    ChooseUniqueName("m_scalar_swizzle2", m_scalarSwizzle2Function, sizeof(m_scalarSwizzle2Function));
    ChooseUniqueName("m_scalar_swizzle3", m_scalarSwizzle3Function, sizeof(m_scalarSwizzle3Function));
    ChooseUniqueName("m_scalar_swizzle4", m_scalarSwizzle4Function, sizeof(m_scalarSwizzle4Function));

    ChooseUniqueName("sincos", m_sinCosFunction, sizeof(m_sinCosFunction));

    ChooseUniqueName("bvecTernary", m_bvecTernary, sizeof(m_bvecTernary));

    if (target == Target_VertexShader)
    {
      m_inAttribPrefix = "";
      m_outAttribPrefix = "frag_";
    }
    else
    {
      m_inAttribPrefix = "frag_";
      m_outAttribPrefix = "rast_";
    }

    m_tree->ReplaceUniformsAssignments();

    HLSLRoot* root = m_tree->GetRoot();
    HLSLStatement* statement = root->statement;

    // Find the entry point function.
    HLSLFunction* entryFunction = FindFunction(root, m_entryName);
    if (entryFunction == nullptr)
    {
      Error("Entry point '%s' doesn't exist", m_entryName);
      return false;
    }

    if (m_version == Version_110)
    {
      m_writer.WriteLine(0, "#version 110");
    }
    if (m_version == Version_120)
    {
      m_writer.WriteLine(0, "#version 120");
    }
    else if (m_version == Version_140)
    {
      m_writer.WriteLine(0, "#version 140");

      // Pragmas for NVIDIA.
      m_writer.WriteLine(0, "#pragma optionNV(fastmath on)");
      //m_writer.WriteLine(0, "#pragma optionNV(fastprecision on)");
      m_writer.WriteLine(0, "#pragma optionNV(ifcvt none)");
      m_writer.WriteLine(0, "#pragma optionNV(inline all)");
      m_writer.WriteLine(0, "#pragma optionNV(strict on)");
      m_writer.WriteLine(0, "#pragma optionNV(unroll all)");
    }
    else if (m_version == Version_150)
    {
      m_writer.WriteLine(0, "#version 150");
    }
    else if (m_version == Version_330)
    {
      m_writer.WriteLine(0, "#version 330");
    }
    else if (m_version == Version_100_ES)
    {
      m_writer.WriteLine(0, "#version 100");
      m_writer.WriteLine(0, "precision highp float;");
    }
    else if (m_version == Version_300_ES)
    {
      m_writer.WriteLine(0, "#version 300 es");
      m_writer.WriteLine(0, "precision highp float;");
      m_writer.WriteLine(0, "precision highp sampler3D;");
    }
    else
    {
      Error("Unrecognized target version");
      return false;
    }

    // Output the special function used to access rows in a matrix.
    m_writer.WriteLine(0, "vec2 %s(mat2 m, int i) { return vec2( m[0][i], m[1][i] ); }", m_matrixRowFunction);
    m_writer.WriteLine(0, "vec3 %s(mat3 m, int i) { return vec3( m[0][i], m[1][i], m[2][i] ); }", m_matrixRowFunction);
    m_writer.WriteLine(0, "vec4 %s(mat4 m, int i) { return vec4( m[0][i], m[1][i], m[2][i], m[3][i] ); }", m_matrixRowFunction);

    // Output the special function used to do matrix cast for OpenGL 2.0
    if (m_versionLegacy)
    {
      m_writer.WriteLine(0, "mat3 %s(mat4 m) { return mat3(m[0][0], m[0][1], m[0][2], m[1][0], m[1][1], m[1][2], m[2][0], m[2][1], m[2][2]); }", m_matrixCtorFunction);
    }

    // Output the special functions used for matrix multiplication lowering
    // They make sure glsl-optimizer can fold expressions better
    if (m_tree->NeedsFunction("mul") && (m_options.flags & Flag_LowerMatrixMultiplication))
    {
      m_writer.WriteLine(0, "vec2 %s(mat2 m, vec2 v) { return m[0] * v.x + m[1] * v.y; }", m_matrixMulFunction);
      m_writer.WriteLine(0, "vec2 %s(vec2 v, mat2 m) { return vec2(dot(m[0], v), dot(m[1], v)); }", m_matrixMulFunction);
      m_writer.WriteLine(0, "vec3 %s(mat3 m, vec3 v) { return m[0] * v.x + m[1] * v.y + m[2] * v.z; }", m_matrixMulFunction);
      m_writer.WriteLine(0, "vec3 %s(vec3 v, mat3 m) { return vec3(dot(m[0], v), dot(m[1], v), dot(m[2], v)); }", m_matrixMulFunction);
      m_writer.WriteLine(0, "vec4 %s(mat4 m, vec4 v) { return m[0] * v.x + m[1] * v.y + m[2] * v.z + m[3] * v.w; }", m_matrixMulFunction);
      m_writer.WriteLine(0, "vec4 %s(vec4 v, mat4 m) { return vec4(dot(m[0], v), dot(m[1], v), dot(m[2], v), dot(m[3], v)); }", m_matrixMulFunction);
    }

    // Output the special function used to emulate HLSL clip.
    if (m_tree->NeedsFunction("clip"))
    {
      const char* discard = m_target == Target_FragmentShader ? "discard" : "";
      m_writer.WriteLine(0, "void %s(float x) { if (x < 0.0) %s;  }", m_clipFunction, discard);
      m_writer.WriteLine(0, "void %s(vec2  x) { if (any(lessThan(x, vec2(0.0, 0.0)))) %s;  }", m_clipFunction, discard);
      m_writer.WriteLine(0, "void %s(vec3  x) { if (any(lessThan(x, vec3(0.0, 0.0, 0.0)))) %s;  }", m_clipFunction, discard);
      m_writer.WriteLine(0, "void %s(vec4  x) { if (any(lessThan(x, vec4(0.0, 0.0, 0.0, 0.0)))) %s;  }", m_clipFunction, discard);
    }

    // Output the special function used to emulate tex2Dlod.
    if (m_tree->NeedsFunction("tex2Dlod"))
    {
      const char* function = "textureLod";

      if (m_versionLegacy)
      {
        m_writer.WriteLine(0, "#extension GL_ARB_shader_texture_lod : require");
        function = "texture2DLod";
      }
      else if (m_version == Version_100_ES)
      {
        m_writer.WriteLine(0, "#extension GL_EXT_shader_texture_lod : require");
        function = "texture2DLodEXT";
      }

      m_writer.WriteLine(0, "vec4 %s(sampler2D samp, vec4 texCoord) { return %s(samp, texCoord.xy, texCoord.w);  }", m_tex2DlodFunction, function);
    }

    // Output the special function used to emulate tex2Dgrad.
    if (m_tree->NeedsFunction("tex2Dgrad"))
    {
      const char* function = "textureGrad";

      if (m_versionLegacy)
      {
        m_writer.WriteLine(0, "#extension GL_ARB_shader_texture_lod : require");
        function = "texture2DGradARB";
      }
      else if (m_version == Version_100_ES)
      {
        m_writer.WriteLine(0, "#extension GL_EXT_shader_texture_lod : require");
        function = "texture2DGradEXT";
      }

      m_writer.WriteLine(0, "vec4 %s(sampler2D samp, vec2 texCoord, vec2 dx, vec2 dy) { return %s(samp, texCoord, dx, dy);  }", m_tex2DgradFunction, function);
    }

    // Output the special function used to emulate tex2Dbias.
    if (m_tree->NeedsFunction("tex2Dbias"))
    {
      if (target == Target_FragmentShader)
      {
        m_writer.WriteLine(0, "vec4 %s(sampler2D samp, vec4 texCoord) { return %s(samp, texCoord.xy, texCoord.w);  }", m_tex2DbiasFunction, m_versionLegacy ? "texture2D" : "texture");
      }
      else
      {
        // Bias value is not supported in vertex shader.
        m_writer.WriteLine(0, "vec4 %s(sampler2D samp, vec4 texCoord) { return texture(samp, texCoord.xy);  }", m_tex2DbiasFunction);
      }
    }

    // Output the special function used to emulate tex2DMSfetch.
    if (m_tree->NeedsFunction("tex2DMSfetch"))
    {
      m_writer.WriteLine(0, "vec4 tex2DMSfetch(sampler2DMS samp, ivec2 texCoord, int sample) {");
      m_writer.WriteLine(1, "return texelFetch(samp, texCoord, sample);");
      m_writer.WriteLine(0, "}");
    }

    // Output the special function used to emulate tex3Dlod.
    if (m_tree->NeedsFunction("tex3Dlod"))
    {
      m_writer.WriteLine(0, "vec4 %s(sampler3D samp, vec4 texCoord) { return %s(samp, texCoord.xyz, texCoord.w);  }", m_tex3DlodFunction, m_versionLegacy ? "texture3D" : "texture");
    }

    // Output the special function used to emulate texCUBEbias.
    if (m_tree->NeedsFunction("texCUBEbias"))
    {
      if (target == Target_FragmentShader)
      {
        m_writer.WriteLine(0, "vec4 %s(samplerCube samp, vec4 texCoord) { return %s(samp, texCoord.xyz, texCoord.w);  }", m_texCUBEbiasFunction, m_versionLegacy ? "textureCube" : "texture");
      }
      else
      {
        // Bias value is not supported in vertex shader.
        m_writer.WriteLine(0, "vec4 %s(samplerCube samp, vec4 texCoord) { return texture(samp, texCoord.xyz);  }", m_texCUBEbiasFunction);
      }
    }

    // Output the special function used to emulate texCUBElod
    if (m_tree->NeedsFunction("texCUBElod"))
    {
      const char* function = "textureLod";

      if (m_version == Version_110 || m_version == Version_120)
      {
        m_writer.WriteLine(0, "#extension GL_ARB_shader_texture_lod : require");
        function = "textureCubeLod";
      }
      else if (m_version == Version_100_ES)
      {
        m_writer.WriteLine(0, "#extension GL_EXT_shader_texture_lod : require");
        function = "textureCubeLodEXT";
      }

      m_writer.WriteLine(0, "vec4 %s(samplerCube samp, vec4 texCoord) { return %s(samp, texCoord.xyz, texCoord.w);  }", m_texCUBElodFunction, function);
    }

    if (m_tree->NeedsFunction("modf"))
    {
      if (m_version == Version_110 || m_version == Version_120 || m_version == Version_100_ES)
      {
        m_writer.WriteLine(0, "float %s(float x, out int ip) { ip = int(x); return x - ip; }", m_modfFunction);
      }
      else {
        m_writer.WriteLine(0, "float %s(float x, out int ip) { return modf(x, ip); }", m_modfFunction);
      }
    }

    m_writer.WriteLine(0, "vec2  %s(float x) { return  vec2(x, x); }", m_scalarSwizzle2Function);
    m_writer.WriteLine(0, "ivec2 %s(int   x) { return ivec2(x, x); }", m_scalarSwizzle2Function);

    m_writer.WriteLine(0, "vec3  %s(float x) { return  vec3(x, x, x); }", m_scalarSwizzle3Function);
    m_writer.WriteLine(0, "ivec3 %s(int   x) { return ivec3(x, x, x); }", m_scalarSwizzle3Function);

    m_writer.WriteLine(0, "vec4  %s(float x) { return  vec4(x, x, x, x); }", m_scalarSwizzle4Function);
    m_writer.WriteLine(0, "ivec4 %s(int   x) { return ivec4(x, x, x, x); }", m_scalarSwizzle4Function);

    if (!m_versionLegacy)
    {
      m_writer.WriteLine(0, "uvec2 %s(uint  x) { return uvec2(x, x); }", m_scalarSwizzle2Function);
      m_writer.WriteLine(0, "uvec3 %s(uint  x) { return uvec3(x, x, x); }", m_scalarSwizzle3Function);
      m_writer.WriteLine(0, "uvec4 %s(uint  x) { return uvec4(x, x, x, x); }", m_scalarSwizzle4Function);
    }

    if (m_tree->NeedsFunction("sincos"))
    {
      const char* floatTypes[] = { "float", "vec2", "vec3", "vec4" };
      for (int i = 0; i < 4; ++i)
      {
        m_writer.WriteLine(0, "void %s(%s x, out %s s, out %s c) { s = sin(x); c = cos(x); }", m_sinCosFunction,
          floatTypes[i], floatTypes[i], floatTypes[i]);
      }
    }

    // special function to emulate ?: with bool{2,3,4} condition type
    m_writer.WriteLine(0, "vec2 %s(bvec2 cond, vec2 trueExpr, vec2 falseExpr) { vec2 ret; ret.x = cond.x ? trueExpr.x : falseExpr.x; ret.y = cond.y ? trueExpr.y : falseExpr.y; return ret; }", m_bvecTernary);
    m_writer.WriteLine(0, "vec3 %s(bvec3 cond, vec3 trueExpr, vec3 falseExpr) { vec3 ret; ret.x = cond.x ? trueExpr.x : falseExpr.x; ret.y = cond.y ? trueExpr.y : falseExpr.y; ret.z = cond.z ? trueExpr.z : falseExpr.z; return ret; }", m_bvecTernary);
    m_writer.WriteLine(0, "vec4 %s(bvec4 cond, vec4 trueExpr, vec4 falseExpr) { vec4 ret; ret.x = cond.x ? trueExpr.x : falseExpr.x; ret.y = cond.y ? trueExpr.y : falseExpr.y; ret.z = cond.z ? trueExpr.z : falseExpr.z; ret.w = cond.w ? trueExpr.w : falseExpr.w; return ret; }", m_bvecTernary);

    m_tree->EnumerateMatrixCtorsNeeded(matrixCtors);
    for (matrixCtor& ctor : matrixCtors)
    {
      std::string id = GetTypeName(HLSLType(ctor.matrixType));
      id += "_from";
      for (HLSLBaseType argType : ctor.argumentTypes)
      {
        id += "_";
        id += GetTypeName(HLSLType(argType));
      }
      matrixCtorsId[ctor] = id;
    }

    OutputMatrixCtors();

    // Output the extension used for dFdx/dFdy in GLES2
    if (m_version == Version_100_ES && (m_tree->NeedsFunction("ddx") || m_tree->NeedsFunction("ddy")))
    {
      m_writer.WriteLine(0, "#extension GL_OES_standard_derivatives : require");
    }

    OutputAttributes(entryFunction);

    if (m_target == Target_FragmentShader)
    {
      if (!m_outputTargets)
        Error("Fragment shader must output a color");

      if (!m_versionLegacy)
        m_writer.WriteLine(0, "out vec4 rast_FragData[%d];", m_outputTargets);
    }

    OutputStatements(0, statement);
    OutputEntryCaller(entryFunction);

    m_tree = nullptr;

    // The GLSL compilers don't check for this, so generate our own error message.
    if (target == Target_VertexShader && !m_outputPosition)
    {
      Error("Vertex shader must output a position");
    }

    return !m_error;

  }

  const char* GLSLGenerator::GetResult() const
  {
    return m_writer.GetResult();
  }

  void GLSLGenerator::OutputExpressionList(HLSLExpression* expression, HLSLArgument* argument)
  {
    int numExpressions = 0;
    while (expression != nullptr)
    {
      if (numExpressions > 0)
      {
        m_writer.Write(", ");
      }

      HLSLType* expectedType = nullptr;
      if (argument != nullptr)
      {
        expectedType = &argument->type;
        argument = argument->nextArgument;
      }

      OutputExpression(expression, expectedType);
      expression = expression->nextExpression;
      ++numExpressions;
    }
  }

  const HLSLType* commonScalarType(const HLSLType& lhs, const HLSLType& rhs)
  {
    if (!IsScalarType(lhs) || !IsScalarType(rhs))
      return nullptr;

    if (lhs.baseType == HLSLBaseType_Float || rhs.baseType == HLSLBaseType_Float)
      return &kFloatType;

    if (lhs.baseType == HLSLBaseType_Uint || rhs.baseType == HLSLBaseType_Uint)
      return &kUintType;

    if (lhs.baseType == HLSLBaseType_Int || rhs.baseType == HLSLBaseType_Int)
      return &kIntType;

    if (lhs.baseType == HLSLBaseType_Bool || rhs.baseType == HLSLBaseType_Bool)
      return &kBoolType;

    return nullptr;
  }

  void GLSLGenerator::OutputExpression(HLSLExpression* expression, const HLSLType* dstType)
  {

    bool cast = dstType != nullptr && !GetCanImplicitCast(expression->expressionType, *dstType);
    if (expression->nodeType == HLSLNodeType_CastingExpression)
    {
      // No need to include a cast if the expression is already doing it.
      cast = false;
    }

    if (cast)
    {
      OutputCast(*dstType);
      m_writer.Write("(");
    }

    HLSLBuffer* bufferAccess = (m_options.flags & Flag_EmulateConstantBuffer) ? GetBufferAccessExpression(expression) : 0;

    if (bufferAccess)
    {
      OutputBufferAccessExpression(bufferAccess, expression, expression->expressionType, 0);
    }
    else if (expression->nodeType == HLSLNodeType_IdentifierExpression)
    {
      HLSLIdentifierExpression* identifierExpression = static_cast<HLSLIdentifierExpression*>(expression);
      OutputIdentifier(identifierExpression->name);
    }
    else if (expression->nodeType == HLSLNodeType_ConstructorExpression)
    {
      HLSLConstructorExpression* constructorExpression = static_cast<HLSLConstructorExpression*>(expression);

      bool matrixCtorNeeded = false;
      if (IsMatrixType(constructorExpression->type.baseType))
      {
        matrixCtor ctor = matrixCtorBuilder(constructorExpression->type, constructorExpression->argument);
        if (std::find(matrixCtors.cbegin(), matrixCtors.cend(), ctor) != matrixCtors.cend())
        {
          matrixCtorNeeded = true;
        }
      }

      if (matrixCtorNeeded)
      {
        // Matrix contructors needs to be adapted since GLSL access a matrix as m[c][r] while HLSL is m[r][c]
        matrixCtor ctor = matrixCtorBuilder(constructorExpression->type, constructorExpression->argument);
        m_writer.Write("%s(", matrixCtorsId[ctor].c_str());
        OutputExpressionList(constructorExpression->argument);
        m_writer.Write(")");
      }
      else
      {
        m_writer.Write("%s(", GetTypeName(constructorExpression->type));
        OutputExpressionList(constructorExpression->argument);
        m_writer.Write(")");
      }
    }
    else if (expression->nodeType == HLSLNodeType_CastingExpression)
    {
      HLSLCastingExpression* castingExpression = static_cast<HLSLCastingExpression*>(expression);
      OutputCast(castingExpression->type);
      m_writer.Write("(");
      OutputExpression(castingExpression->expression);
      m_writer.Write(")");
    }
    else if (expression->nodeType == HLSLNodeType_LiteralExpression)
    {
      HLSLLiteralExpression* literalExpression = static_cast<HLSLLiteralExpression*>(expression);
      switch (literalExpression->type)
      {
      case HLSLBaseType_Float:
      {
        // Don't use printf directly so that we don't use the system locale.
        char buffer[64];
        String_FormatFloat(buffer, sizeof(buffer), literalExpression->fValue);
        m_writer.Write("%s", buffer);
      }
      break;
      case HLSLBaseType_Int:
      case HLSLBaseType_Uint:
        m_writer.Write("%d", literalExpression->iValue);
        break;
      case HLSLBaseType_Bool:
        m_writer.Write("%s", literalExpression->bValue ? "true" : "false");
        break;
      default:
        ASSERT(0);
      }
    }
    else if (expression->nodeType == HLSLNodeType_UnaryExpression)
    {
      HLSLUnaryExpression* unaryExpression = static_cast<HLSLUnaryExpression*>(expression);
      const char* op = "?";
      bool pre = true;
      const HLSLType* dstType2 = nullptr;
      switch (unaryExpression->unaryOp)
      {
      case HLSLUnaryOp_Negative:      op = "-";  break;
      case HLSLUnaryOp_Positive:      op = "+";  break;
      case HLSLUnaryOp_Not:           op = "!";  dstType2 = &unaryExpression->expressionType; break;
      case HLSLUnaryOp_PreIncrement:  op = "++"; break;
      case HLSLUnaryOp_PreDecrement:  op = "--"; break;
      case HLSLUnaryOp_PostIncrement: op = "++"; pre = false; break;
      case HLSLUnaryOp_PostDecrement: op = "--"; pre = false; break;
      case HLSLUnaryOp_BitNot:        op = "~";  break;
      }
      m_writer.Write("(");
      if (pre)
      {
        m_writer.Write("%s", op);
        OutputExpression(unaryExpression->expression, dstType2);
      }
      else
      {
        OutputExpression(unaryExpression->expression, dstType2);
        m_writer.Write("%s", op);
      }
      m_writer.Write(")");
    }
    else if (expression->nodeType == HLSLNodeType_BinaryExpression)
    {
      HLSLBinaryExpression* binaryExpression = static_cast<HLSLBinaryExpression*>(expression);
      const char* op = "?";
      const HLSLType* dstType1 = nullptr;
      const HLSLType* dstType2 = nullptr;

      //
      bool vectorExpression = IsVectorType(binaryExpression->expression1->expressionType) || IsVectorType(binaryExpression->expression2->expressionType);
      if (vectorExpression && IsCompareOp(binaryExpression->binaryOp))
      {
        switch (binaryExpression->binaryOp)
        {
        case HLSLBinaryOp_Less:         m_writer.Write("lessThan(");			break;
        case HLSLBinaryOp_Greater:      m_writer.Write("greaterThan(");			break;
        case HLSLBinaryOp_LessEqual:    m_writer.Write("lessThanEqual(");		break;
        case HLSLBinaryOp_GreaterEqual: m_writer.Write("greaterThanEqual(");	break;
        case HLSLBinaryOp_Equal:        m_writer.Write("equal(");				break;
        case HLSLBinaryOp_NotEqual:     m_writer.Write("notEqual(");			break;
        default:
          ASSERT(0); // is so, check isCompareOp
        }

        if (IsVectorType(binaryExpression->expression1->expressionType) && IsScalarType(binaryExpression->expression2->expressionType))
          dstType2 = &binaryExpression->expression1->expressionType;
        else if (IsScalarType(binaryExpression->expression1->expressionType) && IsVectorType(binaryExpression->expression2->expressionType))
          dstType1 = &binaryExpression->expression2->expressionType;
        // TODO if both expressions are vector but with different dimension handle it here or in parser?

        OutputExpression(binaryExpression->expression1, dstType1);
        m_writer.Write(", ");
        OutputExpression(binaryExpression->expression2, dstType2);
        m_writer.Write(")");
      }
      else
      {
        switch (binaryExpression->binaryOp)
        {
        case HLSLBinaryOp_Add:          op = " + "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_Sub:          op = " - "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_Mul:          op = " * "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_Div:          op = " / "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_Mod:          op = " % "; dstType1 = dstType2 = &kIntType; break;
        case HLSLBinaryOp_Less:         op = " < "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_Greater:      op = " > "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_LessEqual:    op = " <= "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_GreaterEqual: op = " >= "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_Equal:        op = " == "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_NotEqual:     op = " != "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_Assign:       op = " = ";  dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_AddAssign:    op = " += "; dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_SubAssign:    op = " -= "; dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_MulAssign:    op = " *= "; dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_DivAssign:    op = " /= "; dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_And:          op = " && "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_Or:           op = " || "; dstType1 = dstType2 = &binaryExpression->expressionType; break;
        case HLSLBinaryOp_BitAnd:       op = " & "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_BitOr:        op = " | "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        case HLSLBinaryOp_BitXor:       op = " ^ "; dstType1 = dstType2 = commonScalarType(binaryExpression->expression1->expressionType, binaryExpression->expression2->expressionType); break;
        default:
          ASSERT(0);
        }
        if ((m_version == Version_110 || m_version == Version_120 || m_version == Version_100_ES) && binaryExpression->binaryOp == HLSLBinaryOp_Mod) {
          m_writer.Write("(int(mod(");
          OutputExpression(binaryExpression->expression1, dstType1);
          m_writer.Write(",");
          OutputExpression(binaryExpression->expression2, dstType2);
          m_writer.Write(")))");
        }
        else {
          m_writer.Write("(");
          OutputExpression(binaryExpression->expression1, dstType1);
          m_writer.Write("%s", op);
          OutputExpression(binaryExpression->expression2, dstType2);
          m_writer.Write(")");
        }
      }
    }
    else if (expression->nodeType == HLSLNodeType_ConditionalExpression)
    {
      HLSLConditionalExpression* conditionalExpression = static_cast<HLSLConditionalExpression*>(expression);
      if (IsVectorType(conditionalExpression->condition->expressionType))
      {
        m_writer.Write("%s", m_bvecTernary);
        m_writer.Write("( ");
        OutputExpression(conditionalExpression->condition);
        m_writer.Write(", ");
        OutputExpression(conditionalExpression->trueExpression, &conditionalExpression->expressionType);
        m_writer.Write(", ");
        OutputExpression(conditionalExpression->falseExpression, &conditionalExpression->expressionType);
        m_writer.Write(" )");
      }
      else
      {
        m_writer.Write("((");
        OutputExpression(conditionalExpression->condition, &kBoolType);
        m_writer.Write(")?(");
        OutputExpression(conditionalExpression->trueExpression, dstType);
        m_writer.Write("):(");
        OutputExpression(conditionalExpression->falseExpression, dstType);
        m_writer.Write("))");
      }
    }
    else if (expression->nodeType == HLSLNodeType_MemberAccess)
    {

      HLSLMemberAccess* memberAccess = static_cast<HLSLMemberAccess*>(expression);

      if (memberAccess->object->expressionType.baseType == HLSLBaseType_Float ||
        memberAccess->object->expressionType.baseType == HLSLBaseType_Int ||
        memberAccess->object->expressionType.baseType == HLSLBaseType_Uint)
      {
        // Handle swizzling on scalar values.
        size_t swizzleLength = strlen(memberAccess->field);
        if (swizzleLength == 2)
        {
          m_writer.Write("%s", m_scalarSwizzle2Function);
        }
        else if (swizzleLength == 3)
        {
          m_writer.Write("%s", m_scalarSwizzle3Function);
        }
        else if (swizzleLength == 4)
        {
          m_writer.Write("%s", m_scalarSwizzle4Function);
        }
        m_writer.Write("(");
        OutputExpression(memberAccess->object);
        m_writer.Write(")");
      }
      else
      {
        m_writer.Write("(");
        OutputExpression(memberAccess->object);
        m_writer.Write(")");

        if (IsMatrixType(memberAccess->object->expressionType.baseType))
        {
          // Handle HLSL matrix "swizzling".
          // TODO: Properly handle multiple element selection such as _m00_m12
          const char* n = memberAccess->field;
          while (n[0] != 0)
          {
            if (n[0] != '_')
            {
              ASSERT(0);
              break;
            }
            ++n;
            char base = '1';
            if (n[0] == 'm')
            {
              base = '0';
              ++n;
            }
            if (isdigit(n[0]) && isdigit(n[1]))
            {
              m_writer.Write("[%d][%d]", n[1] - base, n[0] - base);
              n += 2;
            }
            else
            {
              ASSERT(0);
              break;
            }
          }
        }
        else
        {
          m_writer.Write(".%s", memberAccess->field);
        }

      }

    }
    else if (expression->nodeType == HLSLNodeType_ArrayAccess)
    {
      HLSLArrayAccess* arrayAccess = static_cast<HLSLArrayAccess*>(expression);

      if (!arrayAccess->array->expressionType.array &&
        IsMatrixType(arrayAccess->array->expressionType.baseType))
      {
        // GLSL access a matrix as m[c][r] while HLSL is m[r][c], so use our
        // special row access function to convert.
        m_writer.Write("%s(", m_matrixRowFunction);
        OutputExpression(arrayAccess->array);
        m_writer.Write(",");
        OutputExpression(arrayAccess->index);
        m_writer.Write(")");
      }
      else
      {
        OutputExpression(arrayAccess->array);
        m_writer.Write("[");
        OutputExpression(arrayAccess->index);
        m_writer.Write("]");
      }

    }
    else if (expression->nodeType == HLSLNodeType_FunctionCall)
    {
      HLSLFunctionCall* functionCall = static_cast<HLSLFunctionCall*>(expression);

      // Handle intrinsic funtions that are different between HLSL and GLSL.
      bool handled = false;
      const char* functionName = functionCall->function->name;

      if (String_Equal(functionName, "mul"))
      {
        HLSLExpression* argument[2];
        if (GetFunctionArguments(functionCall, argument, 2) != 2)
        {
          Error("mul expects 2 arguments");
          return;
        }

        const HLSLType& type0 = functionCall->function->argument->type;
        const HLSLType& type1 = functionCall->function->argument->nextArgument->type;

        if (IsVectorType(type0.baseType) && IsVectorType(type1.baseType))
        {
          m_writer.Write("dot((");
          OutputExpression(argument[0], &type0);
          m_writer.Write("),(");
          OutputExpression(argument[1], &type1);
          m_writer.Write("))");
        }
        else
        {
          m_writer.Write("((");
          OutputExpression(argument[0], &type0);
          m_writer.Write(")*(");
          OutputExpression(argument[1], &type1);
          m_writer.Write("))");
        }

        handled = true;
      }
      else if (String_Equal(functionName, "saturate"))
      {
        HLSLExpression* argument[1];
        if (GetFunctionArguments(functionCall, argument, 1) != 1)
        {
          Error("saturate expects 1 argument");
          return;
        }
        m_writer.Write("clamp(");
        OutputExpression(argument[0]);
        HLSLBaseType baseType = argument[0]->expressionType.baseType;
        switch (baseType) {
        case HLSLBaseType_Float:
        case HLSLBaseType_Float2:
        case HLSLBaseType_Float3:
        case HLSLBaseType_Float4:
          m_writer.Write(", 0.0, 1.0)");
          break;

        case HLSLBaseType_Int:
        case HLSLBaseType_Int2:
        case HLSLBaseType_Int3:
        case HLSLBaseType_Int4:
        case HLSLBaseType_Uint:
        case HLSLBaseType_Uint2:
        case HLSLBaseType_Uint3:
        case HLSLBaseType_Uint4:
          m_writer.Write(", 0, 1)");
          break;

        default:
          Error("saturate unhandled type: %s", GetTypeName(argument[0]->expressionType));
          break;
        }
        handled = true;
      }
      else if (String_Equal(functionName, "rsqrt"))
      {
        HLSLExpression* argument[1];
        if (GetFunctionArguments(functionCall, argument, 1) != 1)
        {
          Error("rsqrt expects 1 argument");
          return;
        }
        m_writer.Write("inversesqrt(");
        OutputExpression(argument[0]);
        m_writer.Write(")");
        handled = true;
      }

      if (!handled)
      {
        OutputIdentifier(functionName);
        m_writer.Write("(");
        OutputExpressionList(functionCall->argument, functionCall->function->argument);
        m_writer.Write(")");
      }
    }
    else
    {
      m_writer.Write("<unknown expression>");
    }

    if (cast)
    {
      if (IsVectorType(dstType->baseType) || IsMatrixType(dstType->baseType))
      {
        CompleteConstructorArguments(expression, dstType->baseType);
      }

      m_writer.Write(")");
    }

  }

  void GLSLGenerator::OutputIdentifier(const char* name)
  {

    // Remap intrinstic functions.
    if (String_Equal(name, "tex2D"))
    {
      name = m_versionLegacy ? "texture2D" : "texture";
    }
    else if (String_Equal(name, "tex2Dproj"))
    {
      name = m_versionLegacy ? "texture2DProj" : "textureProj";
    }
    else if (String_Equal(name, "texCUBE"))
    {
      name = m_versionLegacy ? "textureCube" : "texture";
    }
    else if (String_Equal(name, "tex3D"))
    {
      name = m_versionLegacy ? "texture3D" : "texture";
    }
    else if (String_Equal(name, "clip"))
    {
      name = m_clipFunction;
    }
    else if (String_Equal(name, "tex2Dlod"))
    {
      name = m_tex2DlodFunction;
    }
    else if (String_Equal(name, "tex2Dbias"))
    {
      name = m_tex2DbiasFunction;
    }
    else if (String_Equal(name, "tex2Dgrad"))
    {
      name = m_tex2DgradFunction;
    }
    else if (String_Equal(name, "tex2DArray"))
    {
      name = "texture";
    }
    else if (String_Equal(name, "texCUBEbias"))
    {
      name = m_texCUBEbiasFunction;
    }
    else if (String_Equal(name, "texCUBElod"))
    {
      name = m_texCUBElodFunction;
    }
    else if (String_Equal(name, "atan2"))
    {
      name = "atan";
    }
    else if (String_Equal(name, "sincos"))
    {
      name = m_sinCosFunction;
    }
    else if (String_Equal(name, "fmod"))
    {
      // mod is not the same as fmod if the parameter is negative!
      // The equivalent of fmod(x, y) is x - y * floor(x/y)
      // We use the mod version for performance.
      name = "mod";
    }
    else if (String_Equal(name, "lerp"))
    {
      name = "mix";
    }
    else if (String_Equal(name, "frac"))
    {
      name = "fract";
    }
    else if (String_Equal(name, "ddx"))
    {
      name = "dFdx";
    }
    else if (String_Equal(name, "ddy"))
    {
      name = "dFdy";
    }
    else if (String_Equal(name, "modf"))
    {
      name = m_modfFunction;
    }
    else
    {
      // The identifier could be a GLSL reserved word (if it's not also a HLSL reserved word).
      name = GetSafeIdentifierName(name);
    }
    m_writer.Write("%s", name);

  }

  void GLSLGenerator::OutputArguments(HLSLArgument* argument)
  {
    int numArgs = 0;
    while (argument != nullptr)
    {
      if (numArgs > 0)
      {
        m_writer.Write(", ");
      }

      switch (argument->modifier)
      {
      case HLSLArgumentModifier_In:
        m_writer.Write("in ");
        break;
      case HLSLArgumentModifier_Out:
        m_writer.Write("out ");
        break;
      case HLSLArgumentModifier_Inout:
        m_writer.Write("inout ");
        break;
      default:
        break;
      }

      OutputDeclaration(argument->type, argument->name);
      argument = argument->nextArgument;
      ++numArgs;
    }
  }

  void GLSLGenerator::OutputStatements(int indent, HLSLStatement* statement, const HLSLType* returnType)
  {

    while (statement != nullptr)
    {
      if (statement->hidden)
      {
        statement = statement->nextStatement;
        continue;
      }

      if (statement->nodeType == HLSLNodeType_Declaration)
      {
        HLSLDeclaration* declaration = static_cast<HLSLDeclaration*>(statement);

        // GLSL doesn't seem have texture uniforms, so just ignore them.
        if (declaration->type.baseType != HLSLBaseType_Texture)
        {
          bool skipAssignment = true;
          if (indent != 0)
          {
            skipAssignment = false;
          }

          m_writer.BeginLine(indent, declaration->fileName, declaration->line);
          if (indent == 0 && (declaration->type.flags & HLSLTypeFlag_Uniform))
          {
            // At the top level, we need the "uniform" keyword.
            m_writer.Write("uniform ");
            skipAssignment = false;
          }
          OutputDeclaration(declaration, skipAssignment);
          m_writer.EndLine(";");
        }
      }
      else if (statement->nodeType == HLSLNodeType_Struct)
      {
        HLSLStruct* structure = static_cast<HLSLStruct*>(statement);
        m_writer.WriteLine(indent, "struct %s {", structure->name);
        HLSLStructField* field = structure->field;
        while (field != nullptr)
        {
          m_writer.BeginLine(indent + 1, field->fileName, field->line);
          OutputDeclaration(field->type, field->name);
          m_writer.Write(";");
          m_writer.EndLine();
          field = field->nextField;
        }
        m_writer.WriteLine(indent, "};");
      }
      else if (statement->nodeType == HLSLNodeType_Buffer)
      {
        HLSLBuffer* buffer = static_cast<HLSLBuffer*>(statement);
        OutputBuffer(indent, buffer);
      }
      else if (statement->nodeType == HLSLNodeType_Function)
      {
        HLSLFunction* function = static_cast<HLSLFunction*>(statement);

        // Use an alternate name for the function which is supposed to be entry point
        // so that we can supply our own function which will be the actual entry point.
        const char* functionName = GetSafeIdentifierName(function->name);
        const char* returnTypeName = GetTypeName(function->returnType);

        m_writer.BeginLine(indent, function->fileName, function->line);
        m_writer.Write("%s %s(", returnTypeName, functionName);

        OutputArguments(function->argument);

        if (function->forward)
        {
          m_writer.WriteLine(indent, ");");
        }
        else
        {
          m_writer.Write(") {");
          m_writer.EndLine();

          OutputStatements(indent + 1, function->statement, &function->returnType);
          m_writer.WriteLine(indent, "}");
        }
      }
      else if (statement->nodeType == HLSLNodeType_ExpressionStatement)
      {
        HLSLExpressionStatement* expressionStatement = static_cast<HLSLExpressionStatement*>(statement);
        m_writer.BeginLine(indent, statement->fileName, statement->line);
        OutputExpression(expressionStatement->expression);
        m_writer.EndLine(";");
      }
      else if (statement->nodeType == HLSLNodeType_ReturnStatement)
      {
        HLSLReturnStatement* returnStatement = static_cast<HLSLReturnStatement*>(statement);
        if (returnStatement->expression != nullptr)
        {
          m_writer.BeginLine(indent, returnStatement->fileName, returnStatement->line);
          m_writer.Write("return ");
          OutputExpression(returnStatement->expression, returnType);
          m_writer.EndLine(";");
        }
        else
        {
          m_writer.WriteLineTagged(indent, returnStatement->fileName, returnStatement->line, "return;");
        }
      }
      else if (statement->nodeType == HLSLNodeType_DiscardStatement)
      {
        HLSLDiscardStatement* discardStatement = static_cast<HLSLDiscardStatement*>(statement);
        if (m_target == Target_FragmentShader)
        {
          m_writer.WriteLineTagged(indent, discardStatement->fileName, discardStatement->line, "discard;");
        }
      }
      else if (statement->nodeType == HLSLNodeType_BreakStatement)
      {
        HLSLBreakStatement* breakStatement = static_cast<HLSLBreakStatement*>(statement);
        m_writer.WriteLineTagged(indent, breakStatement->fileName, breakStatement->line, "break;");
      }
      else if (statement->nodeType == HLSLNodeType_ContinueStatement)
      {
        HLSLContinueStatement* continueStatement = static_cast<HLSLContinueStatement*>(statement);
        m_writer.WriteLineTagged(indent, continueStatement->fileName, continueStatement->line, "continue;");
      }
      else if (statement->nodeType == HLSLNodeType_IfStatement)
      {
        HLSLIfStatement* ifStatement = static_cast<HLSLIfStatement*>(statement);
        m_writer.BeginLine(indent, ifStatement->fileName, ifStatement->line);
        m_writer.Write("if (");
        OutputExpression(ifStatement->condition, &kBoolType);
        m_writer.Write(") {");
        m_writer.EndLine();
        OutputStatements(indent + 1, ifStatement->statement, returnType);
        m_writer.WriteLine(indent, "}");
        if (ifStatement->elseStatement != nullptr)
        {
          m_writer.WriteLine(indent, "else {");
          OutputStatements(indent + 1, ifStatement->elseStatement, returnType);
          m_writer.WriteLine(indent, "}");
        }
      }
      else if (statement->nodeType == HLSLNodeType_ForStatement)
      {
        HLSLForStatement* forStatement = static_cast<HLSLForStatement*>(statement);
        m_writer.BeginLine(indent, forStatement->fileName, forStatement->line);
        m_writer.Write("for (");
        if (forStatement->initialization != nullptr)
        {
          OutputDeclaration(forStatement->initialization, false);
        }
        else
        {
          OutputExpression(forStatement->initializationWithoutType);
        }
        m_writer.Write("; ");
        OutputExpression(forStatement->condition, &kBoolType);
        m_writer.Write("; ");
        OutputExpression(forStatement->increment);
        m_writer.Write(") {");
        m_writer.EndLine();
        OutputStatements(indent + 1, forStatement->statement, returnType);
        m_writer.WriteLine(indent, "}");
      }
      else if (statement->nodeType == HLSLNodeType_WhileStatement)
      {
        HLSLWhileStatement* whileStatement = static_cast<HLSLWhileStatement*>(statement);
        m_writer.BeginLine(indent, whileStatement->fileName, whileStatement->line);
        m_writer.Write("while (");
        OutputExpression(whileStatement->condition, &kBoolType);
        m_writer.Write(") {");
        m_writer.EndLine();
        OutputStatements(indent + 1, whileStatement->statement, returnType);
        m_writer.WriteLine(indent, "}");
      }
      else if (statement->nodeType == HLSLNodeType_BlockStatement)
      {
        HLSLBlockStatement* blockStatement = static_cast<HLSLBlockStatement*>(statement);
        m_writer.WriteLine(indent, "{");
        OutputStatements(indent + 1, blockStatement->statement, returnType);
        m_writer.WriteLine(indent, "}");
      }
      else
      {
        // Unhanded statement type.
        ASSERT(0);
      }

      statement = statement->nextStatement;

    }

  }

  void GLSLGenerator::OutputBuffer(int indent, HLSLBuffer* buffer)
  {
    // Empty uniform blocks cause compilation errors on NVIDIA, so don't emit them.
    if (buffer->field == nullptr)
      return;

    if (m_options.flags & Flag_EmulateConstantBuffer)
    {
      unsigned int size = 0;
      LayoutBuffer(buffer, size);

      unsigned int uniformSize = (size + 3) / 4;

      m_writer.WriteLineTagged(indent, buffer->fileName, buffer->line, "uniform vec4 %s%s[%d];", m_options.constantBufferPrefix, buffer->name, uniformSize);
    }
    else
    {
      m_writer.WriteLineTagged(indent, buffer->fileName, buffer->line, "layout (std140) uniform %s%s {", m_options.constantBufferPrefix, buffer->name);
      HLSLDeclaration* field = buffer->field;
      while (field != nullptr)
      {
        m_writer.BeginLine(indent + 1, field->fileName, field->line);
        OutputDeclaration(field->type, field->name);
        m_writer.Write(";");
        m_writer.EndLine();
        field = (HLSLDeclaration*)field->nextStatement;
      }
      m_writer.WriteLine(indent, "};");
    }
  }

  inline void alignForWrite(unsigned int& offset, unsigned int size)
  {
    ASSERT(size <= 4);

    if (offset / 4 != (offset + size - 1) / 4)
      offset = (offset + 3) & ~3;
  }

  void GLSLGenerator::LayoutBuffer(HLSLBuffer* buffer, unsigned int& offset)
  {
    for (HLSLDeclaration* field = buffer->field; field; field = (HLSLDeclaration*)field->nextStatement)
    {
      LayoutBuffer(field->type, offset);
    }
  }

  void GLSLGenerator::LayoutBuffer(const HLSLType& type, unsigned int& offset)
  {
    LayoutBufferAlign(type, offset);

    if (type.array)
    {
      int arraySize = 0;
      m_tree->GetExpressionValue(type.arraySize, arraySize);

      unsigned int elementSize = 0;
      LayoutBufferElement(type, elementSize);

      unsigned int alignedElementSize = (elementSize + 3) & ~3;

      offset += alignedElementSize * arraySize;
    }
    else
    {
      LayoutBufferElement(type, offset);
    }
  }

  void GLSLGenerator::LayoutBufferElement(const HLSLType& type, unsigned int& offset)
  {
    if (type.baseType == HLSLBaseType_Float)
    {
      offset += 1;
    }
    else if (type.baseType == HLSLBaseType_Float2)
    {
      offset += 2;
    }
    else if (type.baseType == HLSLBaseType_Float3)
    {
      offset += 3;
    }
    else if (type.baseType == HLSLBaseType_Float4)
    {
      offset += 4;
    }
    else if (type.baseType == HLSLBaseType_Float4x4)
    {
      offset += 16;
    }
    else if (type.baseType == HLSLBaseType_UserDefined)
    {
      HLSLStruct* st = m_tree->FindGlobalStruct(type.typeName);

      if (st)
      {
        for (HLSLStructField* field = st->field; field; field = field->nextField)
        {
          LayoutBuffer(field->type, offset);
        }
      }
      else
      {
        Error("Unknown type %s", type.typeName);
      }
    }
    else
    {
      Error("Constant buffer layout is not supported for %s", GetTypeName(type));
    }
  }

  void GLSLGenerator::LayoutBufferAlign(const HLSLType& type, unsigned int& offset)
  {
    if (type.array)
    {
      alignForWrite(offset, 4);
    }
    else if (type.baseType == HLSLBaseType_Float)
    {
      alignForWrite(offset, 1);
    }
    else if (type.baseType == HLSLBaseType_Float2)
    {
      alignForWrite(offset, 2);
    }
    else if (type.baseType == HLSLBaseType_Float3)
    {
      alignForWrite(offset, 3);
    }
    else if (type.baseType == HLSLBaseType_Float4)
    {
      alignForWrite(offset, 4);
    }
    else if (type.baseType == HLSLBaseType_Float4x4)
    {
      alignForWrite(offset, 4);
    }
    else if (type.baseType == HLSLBaseType_UserDefined)
    {
      alignForWrite(offset, 4);
    }
    else
    {
      Error("Constant buffer layout is not supported for %s", GetTypeName(type));
    }
  }

  HLSLBuffer* GLSLGenerator::GetBufferAccessExpression(HLSLExpression* expression)
  {
    if (expression->nodeType == HLSLNodeType_IdentifierExpression)
    {
      HLSLIdentifierExpression* identifierExpression = static_cast<HLSLIdentifierExpression*>(expression);

      if (identifierExpression->global)
      {
        HLSLDeclaration* declaration = m_tree->FindGlobalDeclaration(identifierExpression->name);

        if (declaration && declaration->buffer)
          return declaration->buffer;
      }
    }
    else if (expression->nodeType == HLSLNodeType_MemberAccess)
    {
      HLSLMemberAccess* memberAccess = static_cast<HLSLMemberAccess*>(expression);

      if (memberAccess->object->expressionType.baseType == HLSLBaseType_UserDefined)
        return GetBufferAccessExpression(memberAccess->object);
    }
    else if (expression->nodeType == HLSLNodeType_ArrayAccess)
    {
      HLSLArrayAccess* arrayAccess = static_cast<HLSLArrayAccess*>(expression);

      if (arrayAccess->array->expressionType.array)
        return GetBufferAccessExpression(arrayAccess->array);
    }

    return 0;
  }

  void GLSLGenerator::OutputBufferAccessExpression(HLSLBuffer* buffer, HLSLExpression* expression, const HLSLType& type, unsigned int postOffset)
  {
    if (type.array)
    {
      Error("Constant buffer access is not supported for arrays (use indexing instead)");
    }
    else if (type.baseType == HLSLBaseType_Float)
    {
      m_writer.Write("%s%s[", m_options.constantBufferPrefix, buffer->name);
      unsigned int index = OutputBufferAccessIndex(expression, postOffset);
      m_writer.Write("%d].%c", index / 4, "xyzw"[index % 4]);
    }
    else if (type.baseType == HLSLBaseType_Float2)
    {
      m_writer.Write("%s%s[", m_options.constantBufferPrefix, buffer->name);
      unsigned int index = OutputBufferAccessIndex(expression, postOffset);
      m_writer.Write("%d].%s", index / 4, index % 4 == 0 ? "xy" : index % 4 == 1 ? "yz" : "zw");
    }
    else if (type.baseType == HLSLBaseType_Float3)
    {
      m_writer.Write("%s%s[", m_options.constantBufferPrefix, buffer->name);
      unsigned int index = OutputBufferAccessIndex(expression, postOffset);
      m_writer.Write("%d].%s", index / 4, index % 4 == 0 ? "xyz" : "yzw");
    }
    else if (type.baseType == HLSLBaseType_Float4)
    {
      m_writer.Write("%s%s[", m_options.constantBufferPrefix, buffer->name);
      unsigned int index = OutputBufferAccessIndex(expression, postOffset);
      ASSERT(index % 4 == 0);
      m_writer.Write("%d]", index / 4);
    }
    else if (type.baseType == HLSLBaseType_Float4x4)
    {
      m_writer.Write("mat4(");
      for (int i = 0; i < 4; ++i)
      {
        m_writer.Write("%s%s[", m_options.constantBufferPrefix, buffer->name);
        unsigned int index = OutputBufferAccessIndex(expression, postOffset + i * 4);
        ASSERT(index % 4 == 0);
        m_writer.Write("%d]%c", index / 4, i == 3 ? ')' : ',');
      }
    }
    else if (type.baseType == HLSLBaseType_UserDefined)
    {
      HLSLStruct* st = m_tree->FindGlobalStruct(type.typeName);

      if (st)
      {
        m_writer.Write("%s(", st->name);

        unsigned int offset = postOffset;

        for (HLSLStructField* field = st->field; field; field = field->nextField)
        {
          OutputBufferAccessExpression(buffer, expression, field->type, offset);

          if (field->nextField)
            m_writer.Write(",");

          LayoutBuffer(field->type, offset);
        }

        m_writer.Write(")");
      }
      else
      {
        Error("Unknown type %s", type.typeName);
      }
    }
    else
    {
      Error("Constant buffer layout is not supported for %s", GetTypeName(type));
    }
  }

  unsigned int GLSLGenerator::OutputBufferAccessIndex(HLSLExpression* expression, unsigned int postOffset)
  {
    if (expression->nodeType == HLSLNodeType_IdentifierExpression)
    {
      HLSLIdentifierExpression* identifierExpression = static_cast<HLSLIdentifierExpression*>(expression);
      ASSERT(identifierExpression->global);

      HLSLDeclaration* declaration = m_tree->FindGlobalDeclaration(identifierExpression->name);
      ASSERT(declaration);

      HLSLBuffer* buffer = declaration->buffer;
      ASSERT(buffer);

      unsigned int offset = 0;

      for (HLSLDeclaration* field = buffer->field; field; field = (HLSLDeclaration*)field->nextStatement)
      {
        if (field == declaration)
        {
          LayoutBufferAlign(field->type, offset);
          break;
        }

        LayoutBuffer(field->type, offset);
      }

      return offset + postOffset;
    }
    else if (expression->nodeType == HLSLNodeType_MemberAccess)
    {
      HLSLMemberAccess* memberAccess = static_cast<HLSLMemberAccess*>(expression);

      const HLSLType& type = memberAccess->object->expressionType;
      ASSERT(type.baseType == HLSLBaseType_UserDefined);

      HLSLStruct* st = m_tree->FindGlobalStruct(type.typeName);

      if (st)
      {
        unsigned int offset = 0;

        for (HLSLStructField* field = st->field; field; field = field->nextField)
        {
          if (field->name == memberAccess->field)
          {
            LayoutBufferAlign(field->type, offset);
            break;
          }

          LayoutBuffer(field->type, offset);
        }

        return offset + OutputBufferAccessIndex(memberAccess->object, postOffset);
      }
      else
      {
        Error("Unknown type %s", type.typeName);
      }
    }
    else if (expression->nodeType == HLSLNodeType_ArrayAccess)
    {
      HLSLArrayAccess* arrayAccess = static_cast<HLSLArrayAccess*>(expression);

      const HLSLType& type = arrayAccess->array->expressionType;
      ASSERT(type.array);

      unsigned int elementSize = 0;
      LayoutBufferElement(type, elementSize);

      unsigned int alignedElementSize = (elementSize + 3) & ~3;

      int arrayIndex = 0;
      if (m_tree->GetExpressionValue(arrayAccess->index, arrayIndex))
      {
        unsigned int offset = arrayIndex * alignedElementSize;

        return offset + OutputBufferAccessIndex(arrayAccess->array, postOffset);
      }
      else
      {
        m_writer.Write("%d*(", alignedElementSize / 4);
        OutputExpression(arrayAccess->index);
        m_writer.Write(")+");

        return OutputBufferAccessIndex(arrayAccess->array, postOffset);
      }
    }
    else
    {
      ASSERT(!"IsBufferAccessExpression should have returned false");
    }

    return 0;
  }

  HLSLFunction* GLSLGenerator::FindFunction(HLSLRoot* root, const char* name)
  {
    HLSLStatement* statement = root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Function)
      {
        HLSLFunction* function = static_cast<HLSLFunction*>(statement);
        if (String_Equal(function->name, name))
        {
          return function;
        }
      }
      statement = statement->nextStatement;
    }
    return nullptr;
  }

  HLSLStruct* GLSLGenerator::FindStruct(HLSLRoot* root, const char* name)
  {
    HLSLStatement* statement = root->statement;
    while (statement != nullptr)
    {
      if (statement->nodeType == HLSLNodeType_Struct)
      {
        HLSLStruct* structDeclaration = static_cast<HLSLStruct*>(statement);
        if (String_Equal(structDeclaration->name, name))
        {
          return structDeclaration;
        }
      }
      statement = statement->nextStatement;
    }
    return nullptr;
  }


  const char* GLSLGenerator::GetAttribQualifier(AttributeModifier modifier)
  {
    if (m_versionLegacy)
    {
      if (m_target == Target_VertexShader)
        return (modifier == AttributeModifier_In) ? "attribute" : "varying";
      else
        return (modifier == AttributeModifier_In) ? "varying" : "out";
    }
    else
    {
      return (modifier == AttributeModifier_In) ? "in" : "out";
    }
  }

  void GLSLGenerator::OutputAttribute(const HLSLType& type, const char* semantic, AttributeModifier modifier)
  {
    const char* qualifier = GetAttribQualifier(modifier);
    const char* prefix = (modifier == AttributeModifier_In) ? m_inAttribPrefix : m_outAttribPrefix;

    HLSLRoot* root = m_tree->GetRoot();
    if (type.baseType == HLSLBaseType_UserDefined)
    {
      // If the argument is a struct with semantics specified, we need to
      // grab them.
      HLSLStruct* structDeclaration = FindStruct(root, type.typeName);
      ASSERT(structDeclaration != nullptr);
      HLSLStructField* field = structDeclaration->field;
      while (field != nullptr)
      {
        if (field->semantic != nullptr && GetBuiltInSemantic(field->semantic, modifier) == nullptr)
        {
          m_writer.Write("%s ", qualifier);
          char attribName[64];
          String_Printf(attribName, 64, "%s%s", prefix, field->semantic);
          OutputDeclaration(field->type, attribName);
          m_writer.EndLine(";");
        }
        field = field->nextField;
      }
    }
    else if (semantic != nullptr && GetBuiltInSemantic(semantic, modifier) == nullptr)
    {
      m_writer.Write("%s ", qualifier);
      char attribName[64];
      String_Printf(attribName, 64, "%s%s", prefix, semantic);
      OutputDeclaration(type, attribName);
      m_writer.EndLine(";");
    }
  }

  void GLSLGenerator::OutputAttributes(HLSLFunction* entryFunction)
  {
    // Write out the input/output attributes to the shader.
    HLSLArgument* argument = entryFunction->argument;
    while (argument != nullptr)
    {
      if (argument->modifier == HLSLArgumentModifier_None || argument->modifier == HLSLArgumentModifier_In)
        OutputAttribute(argument->type, argument->semantic, AttributeModifier_In);
      if (argument->modifier == HLSLArgumentModifier_Out)
        OutputAttribute(argument->type, argument->semantic, AttributeModifier_Out);

      argument = argument->nextArgument;
    }

    // Write out the output attributes from the shader.
    OutputAttribute(entryFunction->returnType, entryFunction->semantic, AttributeModifier_Out);
  }

  void GLSLGenerator::OutputSetOutAttribute(const char* semantic, const char* resultName)
  {
    int outputIndex = -1;
    const char* builtInSemantic = GetBuiltInSemantic(semantic, AttributeModifier_Out, &outputIndex);
    if (builtInSemantic != nullptr)
    {
      if (String_Equal(builtInSemantic, "gl_Position"))
      {
        if (m_options.flags & Flag_FlipPositionOutput)
        {
          // Mirror the y-coordinate when we're outputing from
          // the vertex shader so that we match the D3D texture
          // coordinate origin convention in render-to-texture
          // operations.
          // We also need to convert the normalized device
          // coordinates from the D3D convention of 0 to 1 to the
          // OpenGL convention of -1 to 1.
          m_writer.WriteLine(1, "vec4 temp = %s;", resultName);
          m_writer.WriteLine(1, "%s = temp * vec4(1,-1,2,1) - vec4(0,0,temp.w,0);", builtInSemantic);
        }
        else
        {
          m_writer.WriteLine(1, "%s = %s;", builtInSemantic, resultName);
        }

        m_outputPosition = true;
      }
      else if (String_Equal(builtInSemantic, "gl_FragDepth"))
      {
        // If the value goes outside of the 0 to 1 range, the
        // fragment will be rejected unlike in D3D, so clamp it.
        m_writer.WriteLine(1, "%s = clamp(float(%s), 0.0, 1.0);", builtInSemantic, resultName);
      }
      else if (outputIndex >= 0)
      {
        m_writer.WriteLine(1, "%s[%d] = %s;", builtInSemantic, outputIndex, resultName);
      }
      else
      {
        m_writer.WriteLine(1, "%s = %s;", builtInSemantic, resultName);
      }
    }
    else if (m_target == Target_FragmentShader)
    {
      Error("Output attribute %s does not map to any built-ins", semantic);
    }
    else
    {
      m_writer.WriteLine(1, "%s%s = %s;", m_outAttribPrefix, semantic, resultName);
    }
  }

  void GLSLGenerator::OutputEntryCaller(HLSLFunction* entryFunction)
  {
    HLSLRoot* root = m_tree->GetRoot();

    m_writer.WriteLine(0, "void main() {");

    // Create local variables for each of the parameters we'll need to pass
    // into the entry point function.
    HLSLArgument* argument = entryFunction->argument;
    while (argument != nullptr)
    {
      m_writer.BeginLine(1);
      OutputDeclaration(argument->type, argument->name);
      m_writer.EndLine(";");

      if (argument->modifier != HLSLArgumentModifier_Out)
      {
        // Set the value for the local variable.
        if (argument->type.baseType == HLSLBaseType_UserDefined)
        {
          HLSLStruct* structDeclaration = FindStruct(root, argument->type.typeName);
          ASSERT(structDeclaration != nullptr);
          HLSLStructField* field = structDeclaration->field;
          while (field != nullptr)
          {
            if (field->semantic != nullptr)
            {
              const char* builtInSemantic = GetBuiltInSemantic(field->semantic, AttributeModifier_In);
              if (builtInSemantic)
              {
                m_writer.WriteLine(1, "%s.%s = %s;", GetSafeIdentifierName(argument->name), GetSafeIdentifierName(field->name), builtInSemantic);
              }
              else
              {
                m_writer.WriteLine(1, "%s.%s = %s%s;", GetSafeIdentifierName(argument->name), GetSafeIdentifierName(field->name), m_inAttribPrefix, field->semantic);
              }
            }
            field = field->nextField;
          }
        }
        else if (argument->semantic != nullptr)
        {
          const char* builtInSemantic = GetBuiltInSemantic(argument->semantic, AttributeModifier_In);
          if (builtInSemantic)
          {
            m_writer.WriteLine(1, "%s = %s;", GetSafeIdentifierName(argument->name), builtInSemantic);
          }
          else
          {
            m_writer.WriteLine(1, "%s = %s%s;", GetSafeIdentifierName(argument->name), m_inAttribPrefix, argument->semantic);
          }
        }
      }

      argument = argument->nextArgument;
    }


    // Initialize global variables
    for (HLSLDeclaration* declaration : globalVarsAssignments)
    {
      m_writer.BeginLine(1, declaration->fileName, declaration->line);
      OutputDeclarationBody(declaration->type, GetSafeIdentifierName(declaration->name));

      OutputDeclarationAssignment(declaration);
      m_writer.EndLine(";");
    }


    const char* resultName = "result";

    // Call the original entry function.
    m_writer.BeginLine(1);
    if (entryFunction->returnType.baseType != HLSLBaseType_Void)
      m_writer.Write("%s %s = ", GetTypeName(entryFunction->returnType), resultName);
    m_writer.Write("%s(", m_entryName);

    int numArgs = 0;
    argument = entryFunction->argument;
    while (argument != nullptr)
    {
      if (numArgs > 0)
      {
        m_writer.Write(", ");
      }

      m_writer.Write("%s", GetSafeIdentifierName(argument->name));

      argument = argument->nextArgument;
      ++numArgs;
    }
    m_writer.EndLine(");");

    // Copy values from the result into the out attributes as necessary.
    argument = entryFunction->argument;
    while (argument != nullptr)
    {
      if (argument->modifier == HLSLArgumentModifier_Out && argument->semantic)
        OutputSetOutAttribute(argument->semantic, GetSafeIdentifierName(argument->name));

      argument = argument->nextArgument;
    }

    if (entryFunction->returnType.baseType == HLSLBaseType_UserDefined)
    {
      HLSLStruct* structDeclaration = FindStruct(root, entryFunction->returnType.typeName);
      ASSERT(structDeclaration != nullptr);
      HLSLStructField* field = structDeclaration->field;
      while (field != nullptr)
      {
        char fieldResultName[1024];
        String_Printf(fieldResultName, sizeof(fieldResultName), "%s.%s", resultName, field->name);
        OutputSetOutAttribute(field->semantic, fieldResultName);
        field = field->nextField;
      }
    }
    else if (entryFunction->semantic != nullptr)
    {
      OutputSetOutAttribute(entryFunction->semantic, resultName);
    }

    m_writer.WriteLine(0, "}");
  }

  void GLSLGenerator::OutputDeclaration(HLSLDeclaration* declaration, const bool skipAssignment)
  {
    OutputDeclarationType(declaration->type);

    HLSLDeclaration* lastDecl = nullptr;
    while (declaration)
    {
      if (lastDecl)
        m_writer.Write(", ");

      OutputDeclarationBody(declaration->type, GetSafeIdentifierName(declaration->name));

      if (declaration->assignment != nullptr)
      {
        if (!skipAssignment)
        {
          OutputDeclarationAssignment(declaration);
        }
        else
        {
          globalVarsAssignments.push_back(declaration);
        }
      }

      lastDecl = declaration;
      declaration = declaration->nextDeclaration;
    }
  }

  void GLSLGenerator::OutputDeclarationAssignment(HLSLDeclaration* declaration)
  {
    m_writer.Write(" = ");
    if (declaration->type.array)
    {
      m_writer.Write("%s[]( ", GetTypeName(declaration->type));
      OutputExpressionList(declaration->assignment);
      m_writer.Write(" )");
    }
    else
    {
      bool matrixCtorNeeded = false;
      if (IsMatrixType(declaration->type.baseType))
      {
        matrixCtor ctor = matrixCtorBuilder(declaration->type, declaration->assignment);
        if (std::find(matrixCtors.cbegin(), matrixCtors.cend(), ctor) != matrixCtors.cend())
        {
          matrixCtorNeeded = true;
        }
      }

      if (matrixCtorNeeded)
      {
        // Matrix contructors needs to be adapted since GLSL access a matrix as m[c][r] while HLSL is m[r][c]
        matrixCtor ctor = matrixCtorBuilder(declaration->type, declaration->assignment);
        m_writer.Write("%s(", matrixCtorsId[ctor].c_str());
        OutputExpressionList(declaration->assignment);
        m_writer.Write(")");
      }
      else
      {
        m_writer.Write("%s( ", GetTypeName(declaration->type));
        OutputExpressionList(declaration->assignment);
        m_writer.Write(" )");
      }
    }
  }

  void GLSLGenerator::OutputDeclaration(const HLSLType& type, const char* name)
  {
    OutputDeclarationType(type);
    OutputDeclarationBody(type, name);
  }

  void GLSLGenerator::OutputDeclarationType(const HLSLType& type)
  {
    m_writer.Write("%s ", GetTypeName(type));
  }

  void GLSLGenerator::OutputDeclarationBody(const HLSLType& type, const char* name)
  {
    if (!type.array)
    {
      m_writer.Write("%s", GetSafeIdentifierName(name));
    }
    else
    {
      m_writer.Write("%s[", GetSafeIdentifierName(name));
      if (type.arraySize != nullptr)
      {
        OutputExpression(type.arraySize);
      }
      m_writer.Write("]");
    }
  }

  void GLSLGenerator::OutputCast(const HLSLType& type)
  {
    if ((m_version == Version_110 || m_version == Version_120) && type.baseType == HLSLBaseType_Float3x3)
      m_writer.Write("%s", m_matrixCtorFunction);
    else
      OutputDeclaration(type, "");
  }

  void GLSLGenerator::Error(const char* format, ...)
  {
    // It's not always convenient to stop executing when an error occurs,
    // so just track once we've hit an error and stop reporting them until
    // we successfully bail out of execution.
    if (m_error)
    {
      return;
    }
    m_error = true;

    va_list arg;
    va_start(arg, format);
    Log_ErrorArgList(format, arg);
    va_end(arg);
  }

  const char* GLSLGenerator::GetSafeIdentifierName(const char* name) const
  {
    for (int i = 0; i < s_numReservedWords; ++i)
    {
      if (String_Equal(s_reservedWord[i], name))
      {
        return m_reservedWord[i];
      }
    }
    return name;
  }

  bool GLSLGenerator::ChooseUniqueName(const char* base, char* dst, int dstLength) const
  {
    for (int i = 0; i < 1024; ++i)
    {
      String_Printf(dst, dstLength, "%s%d", base, i);
      if (!m_tree->GetContainsString(dst))
      {
        return true;
      }
    }
    return false;
  }

  const char* GLSLGenerator::GetBuiltInSemantic(const char* semantic, AttributeModifier modifier, int* outputIndex)
  {
    if (outputIndex)
      *outputIndex = -1;

    if (m_target == Target_VertexShader && modifier == AttributeModifier_Out && String_Equal(semantic, "POSITION"))
      return "gl_Position";

    if (m_target == Target_VertexShader && modifier == AttributeModifier_Out && String_Equal(semantic, "SV_Position"))
      return "gl_Position";

    if (m_target == Target_VertexShader && modifier == AttributeModifier_Out && String_Equal(semantic, "PSIZE"))
      return "gl_PointSize";

    if (m_target == Target_VertexShader && modifier == AttributeModifier_In && String_Equal(semantic, "SV_InstanceID"))
      return "gl_InstanceID";

    if (m_target == Target_FragmentShader && modifier == AttributeModifier_Out && String_Equal(semantic, "SV_Depth"))
      return "gl_FragDepth";

    if (m_target == Target_FragmentShader && modifier == AttributeModifier_In && String_Equal(semantic, "SV_Position"))
      return "gl_FragCoord";

    if (m_target == Target_FragmentShader && modifier == AttributeModifier_Out)
    {
      int index = -1;

      if (strncmp(semantic, "COLOR", 5) == 0)
        index = atoi(semantic + 5);
      else if (strncmp(semantic, "SV_Target", 9) == 0)
        index = atoi(semantic + 9);

      if (index >= 0)
      {
        if (m_outputTargets <= index)
          m_outputTargets = index + 1;

        if (outputIndex)
          *outputIndex = index;

        return m_versionLegacy ? "gl_FragData" : "rast_FragData";
      }
    }

    return nullptr;
  }

  void GLSLGenerator::CompleteConstructorArguments(HLSLExpression* expression, HLSLBaseType dstType)
  {
    int nbComponentsProvided = 0;
    int nbComponentsNeeded = 0;

    const BaseTypeDescription& dstTypeDesc = baseTypeDescriptions[dstType];
    nbComponentsNeeded = dstTypeDesc.numComponents * dstTypeDesc.height;

    const BaseTypeDescription& srcTypeDesc = baseTypeDescriptions[expression->expressionType.baseType];
    nbComponentsProvided = srcTypeDesc.numComponents * srcTypeDesc.height;
    if (IsMatrixType(expression->expressionType.baseType) ||
      IsVectorType(expression->expressionType.baseType))
    {
      for (int i = nbComponentsProvided; i < nbComponentsNeeded; i++)
      {
        m_writer.Write(", 0");
      }
    }
  }


  void GLSLGenerator::OutputMatrixCtors() {
    for (matrixCtor& ctor : matrixCtors)
    {
      m_writer.Write("%s %s(",
        GetTypeName(HLSLType(ctor.matrixType)),
        matrixCtorsId[ctor].c_str());
      int argNum = 0;
      for (HLSLBaseType argType : ctor.argumentTypes)
      {
        if (argNum == 0)
        {
          m_writer.Write("%s %c", GetTypeName(HLSLType(argType)), 'a' + argNum);
        }
        else
        {
          m_writer.Write(", %s %c", GetTypeName(HLSLType(argType)), 'a' + argNum);
        }
        argNum++;
      }
      m_writer.Write(") { return %s(", GetTypeName(HLSLType(ctor.matrixType)));

      const BaseTypeDescription& ctorTypeDesc = baseTypeDescriptions[ctor.matrixType];
      std::vector<std::string> args(ctorTypeDesc.numComponents * ctorTypeDesc.height, "0");
      int argNumIn = 0;
      int argNumOut = 0;
      for (HLSLBaseType argType : ctor.argumentTypes)
      {
        std::string arg;
        arg += 'a' + argNumIn;

        if (IsScalarType(argType))
        {
          int index = (argNumOut % ctorTypeDesc.height) * (ctorTypeDesc.numComponents) +
            (argNumOut / ctorTypeDesc.height);
          args[index] = arg;
          argNumOut++;
        }
        else if (IsVectorType(argType))
        {
          const BaseTypeDescription& argTypeDesc = baseTypeDescriptions[argType];
          for (int dim = 0; dim < argTypeDesc.numComponents; dim++)
          {
            std::string argVect = arg + ".";
            argVect += "xyzw"[dim];
            int index = (argNumOut % ctorTypeDesc.height) * (ctorTypeDesc.numComponents) +
              (argNumOut / ctorTypeDesc.height);
            args[index] = argVect;
            argNumOut++;
          }
        }

        argNumIn++;
      }

      bool first = true;
      for (std::string& arg : args)
      {
        if (!first)
        {
          m_writer.Write(",%s", arg.c_str());
        }
        else
        {
          m_writer.Write("%s", arg.c_str());
        }
        first = false;
      }

      m_writer.Write("); }");
      m_writer.EndLine();
    }
  }
}





#ifndef __etc1_h__
#define __etc1_h__

#define ETC1_ENCODED_BLOCK_SIZE 8
#define ETC1_DECODED_BLOCK_SIZE 48

#ifndef ETC1_RGB8_OES
#define ETC1_RGB8_OES 0x8D64
#endif

typedef unsigned char etc1_byte;
typedef int etc1_bool;
typedef unsigned int etc1_uint32;

#ifdef __cplusplus
extern "C" {
#endif

  void etc1_encode_block(const etc1_byte* pIn, etc1_uint32 validPixelMask, etc1_byte* pOut);

  void etc1_decode_block(const etc1_byte* pIn, etc1_byte* pOut);


  etc1_uint32 etc1_get_encoded_data_size(etc1_uint32 width, etc1_uint32 height);

  int etc1_encode_image(const etc1_byte* pIn, etc1_uint32 width, etc1_uint32 height,
    etc1_uint32 pixelSize, etc1_uint32 stride, etc1_byte* pOut);

  int etc1_decode_image(const etc1_byte* pIn, etc1_byte* pOut,
    etc1_uint32 width, etc1_uint32 height,
    etc1_uint32 pixelSize, etc1_uint32 stride);

#define ETC_PKM_HEADER_SIZE 16

  void etc1_pkm_format_header(etc1_byte* pHeader, etc1_uint32 width, etc1_uint32 height);

  etc1_bool etc1_pkm_is_valid(const etc1_byte* pHeader);

  etc1_uint32 etc1_pkm_get_width(const etc1_byte* pHeader);

  etc1_uint32 etc1_pkm_get_height(const etc1_byte* pHeader);

#ifdef __cplusplus
}
#endif

#endif


static const int kModifierTable[] = {
  /* 0 */2, 8, -2, -8,
  /* 1 */5, 17, -5, -17,
  /* 2 */9, 29, -9, -29,
  /* 3 */13, 42, -13, -42,
  /* 4 */18, 60, -18, -60,
  /* 5 */24, 80, -24, -80,
  /* 6 */33, 106, -33, -106,
  /* 7 */47, 183, -47, -183 };

static const int kLookup[8] = { 0, 1, 2, 3, -4, -3, -2, -1 };

static inline etc1_byte clamp(int x) {
  return (etc1_byte)(x >= 0 ? (x < 255 ? x : 255) : 0);
}

static
inline int convert4To8(int b) {
  int c = b & 0xf;
  return (c << 4) | c;
}

static
inline int convert5To8(int b) {
  int c = b & 0x1f;
  return (c << 3) | (c >> 2);
}

static
inline int convert6To8(int b) {
  int c = b & 0x3f;
  return (c << 2) | (c >> 4);
}

static
inline int divideBy255(int d) {
  return (d + 128 + (d >> 8)) >> 8;
}

static
inline int convert8To4(int b) {
  //int c = b & 0xff;
  return divideBy255(b * 15);
}

static
inline int convert8To5(int b) {
  //int c = b & 0xff;
  return divideBy255(b * 31);
}

static
inline int convertDiff(int base, int diff) {
  return convert5To8((0x1f & base) + kLookup[0x7 & diff]);
}

static
void decode_subblock(etc1_byte* pOut, int r, int g, int b, const int* table,
  etc1_uint32 low, etc1_bool second, etc1_bool flipped) {
  int baseX = 0;
  int baseY = 0;
  int i;

  if (second) {
    if (flipped) {
      baseY = 2;
    }
    else {
      baseX = 2;
    }
  }
  for (i = 0; i < 8; i++) {
    int x, y;
    if (flipped) {
      x = baseX + (i >> 1);
      y = baseY + (i & 1);
    }
    else {
      x = baseX + (i >> 2);
      y = baseY + (i & 3);
    }
    int k = y + (x * 4);
    int offset = ((low >> k) & 1) | ((low >> (k + 15)) & 2);
    int delta = table[offset];
    etc1_byte* q = pOut + 3 * (x + 4 * y);
    *q++ = clamp(r + delta);
    *q++ = clamp(g + delta);
    *q++ = clamp(b + delta);
  }
}

// Input is an ETC1 compressed version of the data.
// Output is a 4 x 4 square of 3-byte pixels in form R, G, B

void etc1_decode_block(const etc1_byte* pIn, etc1_byte* pOut) {
  etc1_uint32 high = (pIn[0] << 24) | (pIn[1] << 16) | (pIn[2] << 8) | pIn[3];
  etc1_uint32 low = (pIn[4] << 24) | (pIn[5] << 16) | (pIn[6] << 8) | pIn[7];
  int r1, r2, g1, g2, b1, b2;
  if (high & 2) {
    // differential
    int rBase = high >> 27;
    int gBase = high >> 19;
    int bBase = high >> 11;
    r1 = convert5To8(rBase);
    r2 = convertDiff(rBase, high >> 24);
    g1 = convert5To8(gBase);
    g2 = convertDiff(gBase, high >> 16);
    b1 = convert5To8(bBase);
    b2 = convertDiff(bBase, high >> 8);
  }
  else {
    // not differential
    r1 = convert4To8(high >> 28);
    r2 = convert4To8(high >> 24);
    g1 = convert4To8(high >> 20);
    g2 = convert4To8(high >> 16);
    b1 = convert4To8(high >> 12);
    b2 = convert4To8(high >> 8);
  }
  int tableIndexA = 7 & (high >> 5);
  int tableIndexB = 7 & (high >> 2);
  const int* tableA = kModifierTable + tableIndexA * 4;
  const int* tableB = kModifierTable + tableIndexB * 4;
  etc1_bool flipped = (high & 1) != 0;
  decode_subblock(pOut, r1, g1, b1, tableA, low, 0, flipped);
  decode_subblock(pOut, r2, g2, b2, tableB, low, 1, flipped);
}

typedef struct {
  etc1_uint32 high;
  etc1_uint32 low;
  etc1_uint32 score; // Lower is more accurate
} etc_compressed;

static
inline void take_best(etc_compressed* a, const etc_compressed* b) {
  if (a->score > b->score) {
    *a = *b;
  }
}

static
void etc_average_colors_subblock(const etc1_byte* pIn, etc1_uint32 inMask,
  etc1_byte* pColors, etc1_bool flipped, etc1_bool second) {
  int r = 0;
  int g = 0;
  int b = 0;
  int y, x;

  if (flipped) {
    int by = 0;
    if (second) {
      by = 2;
    }
    for (y = 0; y < 2; y++) {
      int yy = by + y;
      for (x = 0; x < 4; x++) {
        int i = x + 4 * yy;
        if (inMask & (1 << i)) {
          const etc1_byte* p = pIn + i * 3;
          r += *(p++);
          g += *(p++);
          b += *(p++);
        }
      }
    }
  }
  else {
    int bx = 0;
    if (second) {
      bx = 2;
    }
    for (y = 0; y < 4; y++) {
      for (x = 0; x < 2; x++) {
        int xx = bx + x;
        int i = xx + 4 * y;
        if (inMask & (1 << i)) {
          const etc1_byte* p = pIn + i * 3;
          r += *(p++);
          g += *(p++);
          b += *(p++);
        }
      }
    }
  }
  pColors[0] = (etc1_byte)((r + 4) >> 3);
  pColors[1] = (etc1_byte)((g + 4) >> 3);
  pColors[2] = (etc1_byte)((b + 4) >> 3);
}

static
inline int square(int x) {
  return x * x;
}

static etc1_uint32 chooseModifier(const etc1_byte* pBaseColors,
  const etc1_byte* pIn, etc1_uint32* pLow, int bitIndex,
  const int* pModifierTable) {
  etc1_uint32 bestScore = ~0;
  int bestIndex = 0;
  int pixelR = pIn[0];
  int pixelG = pIn[1];
  int pixelB = pIn[2];
  int r = pBaseColors[0];
  int g = pBaseColors[1];
  int b = pBaseColors[2];
  int i;
  for (i = 0; i < 4; i++) {
    int modifier = pModifierTable[i];
    int decodedG = clamp(g + modifier);
    etc1_uint32 score = (etc1_uint32)(6 * square(decodedG - pixelG));
    if (score >= bestScore) {
      continue;
    }
    int decodedR = clamp(r + modifier);
    score += (etc1_uint32)(3 * square(decodedR - pixelR));
    if (score >= bestScore) {
      continue;
    }
    int decodedB = clamp(b + modifier);
    score += (etc1_uint32)square(decodedB - pixelB);
    if (score < bestScore) {
      bestScore = score;
      bestIndex = i;
    }
  }
  etc1_uint32 lowMask = (((bestIndex >> 1) << 16) | (bestIndex & 1))
    << bitIndex;
  *pLow |= lowMask;
  return bestScore;
}

static
void etc_encode_subblock_helper(const etc1_byte* pIn, etc1_uint32 inMask,
  etc_compressed* pCompressed, etc1_bool flipped, etc1_bool second,
  const etc1_byte* pBaseColors, const int* pModifierTable) {
  int score = pCompressed->score;
  int y, x;
  if (flipped) {
    int by = 0;
    if (second) {
      by = 2;
    }
    for (y = 0; y < 2; y++) {
      int yy = by + y;
      for (x = 0; x < 4; x++) {
        int i = x + 4 * yy;
        if (inMask & (1 << i)) {
          score += chooseModifier(pBaseColors, pIn + i * 3,
            &pCompressed->low, yy + x * 4, pModifierTable);
        }
      }
    }
  }
  else {
    int bx = 0;
    if (second) {
      bx = 2;
    }
    for (y = 0; y < 4; y++) {
      for (x = 0; x < 2; x++) {
        int xx = bx + x;
        int i = xx + 4 * y;
        if (inMask & (1 << i)) {
          score += chooseModifier(pBaseColors, pIn + i * 3,
            &pCompressed->low, y + xx * 4, pModifierTable);
        }
      }
    }
  }
  pCompressed->score = score;
}

static etc1_bool inRange4bitSigned(int color) {
  return color >= -4 && color <= 3;
}

static void etc_encodeBaseColors(etc1_byte* pBaseColors,
  const etc1_byte* pColors, etc_compressed* pCompressed) {
  int r1, g1, b1, r2 = 0, g2 = 0, b2 = 0; // 8 bit base colors for sub-blocks
  etc1_bool differential;
  {
    int r51 = convert8To5(pColors[0]);
    int g51 = convert8To5(pColors[1]);
    int b51 = convert8To5(pColors[2]);
    int r52 = convert8To5(pColors[3]);
    int g52 = convert8To5(pColors[4]);
    int b52 = convert8To5(pColors[5]);

    r1 = convert5To8(r51);
    g1 = convert5To8(g51);
    b1 = convert5To8(b51);

    int dr = r52 - r51;
    int dg = g52 - g51;
    int db = b52 - b51;

    differential = inRange4bitSigned(dr) && inRange4bitSigned(dg)
      && inRange4bitSigned(db);
    if (differential) {
      r2 = convert5To8(r51 + dr);
      g2 = convert5To8(g51 + dg);
      b2 = convert5To8(b51 + db);
      pCompressed->high |= (r51 << 27) | ((7 & dr) << 24) | (g51 << 19)
        | ((7 & dg) << 16) | (b51 << 11) | ((7 & db) << 8) | 2;
    }
  }

  if (!differential) {
    int r41 = convert8To4(pColors[0]);
    int g41 = convert8To4(pColors[1]);
    int b41 = convert8To4(pColors[2]);
    int r42 = convert8To4(pColors[3]);
    int g42 = convert8To4(pColors[4]);
    int b42 = convert8To4(pColors[5]);
    r1 = convert4To8(r41);
    g1 = convert4To8(g41);
    b1 = convert4To8(b41);
    r2 = convert4To8(r42);
    g2 = convert4To8(g42);
    b2 = convert4To8(b42);
    pCompressed->high |= (r41 << 28) | (r42 << 24) | (g41 << 20) | (g42
      << 16) | (b41 << 12) | (b42 << 8);
  }
  pBaseColors[0] = r1;
  pBaseColors[1] = g1;
  pBaseColors[2] = b1;
  pBaseColors[3] = r2;
  pBaseColors[4] = g2;
  pBaseColors[5] = b2;
}

static
void etc_encode_block_helper(const etc1_byte* pIn, etc1_uint32 inMask,
  const etc1_byte* pColors, etc_compressed* pCompressed, etc1_bool flipped) {
  int i;

  pCompressed->score = ~0;
  pCompressed->high = (flipped ? 1 : 0);
  pCompressed->low = 0;

  etc1_byte pBaseColors[6];

  etc_encodeBaseColors(pBaseColors, pColors, pCompressed);

  int originalHigh = pCompressed->high;

  const int* pModifierTable = kModifierTable;
  for (i = 0; i < 8; i++, pModifierTable += 4) {
    etc_compressed temp;
    temp.score = 0;
    temp.high = originalHigh | (i << 5);
    temp.low = 0;
    etc_encode_subblock_helper(pIn, inMask, &temp, flipped, 0,
      pBaseColors, pModifierTable);
    take_best(pCompressed, &temp);
  }
  pModifierTable = kModifierTable;
  etc_compressed firstHalf = *pCompressed;
  for (i = 0; i < 8; i++, pModifierTable += 4) {
    etc_compressed temp;
    temp.score = firstHalf.score;
    temp.high = firstHalf.high | (i << 2);
    temp.low = firstHalf.low;
    etc_encode_subblock_helper(pIn, inMask, &temp, flipped, 1,
      pBaseColors + 3, pModifierTable);
    if (i == 0) {
      *pCompressed = temp;
    }
    else {
      take_best(pCompressed, &temp);
    }
  }
}

static void writeBigEndian(etc1_byte* pOut, etc1_uint32 d) {
  pOut[0] = (etc1_byte)(d >> 24);
  pOut[1] = (etc1_byte)(d >> 16);
  pOut[2] = (etc1_byte)(d >> 8);
  pOut[3] = (etc1_byte)d;
}

// Input is a 4 x 4 square of 3-byte pixels in form R, G, B
// inmask is a 16-bit mask where bit (1 << (x + y * 4)) tells whether the corresponding (x,y)
// pixel is valid or not. Invalid pixel color values are ignored when compressing.
// Output is an ETC1 compressed version of the data.

void etc1_encode_block(const etc1_byte* pIn, etc1_uint32 inMask,
  etc1_byte* pOut) {
  etc1_byte colors[6];
  etc1_byte flippedColors[6];
  etc_average_colors_subblock(pIn, inMask, colors, 0, 0);
  etc_average_colors_subblock(pIn, inMask, colors + 3, 0, 1);
  etc_average_colors_subblock(pIn, inMask, flippedColors, 1, 0);
  etc_average_colors_subblock(pIn, inMask, flippedColors + 3, 1, 1);

  etc_compressed a, b;
  etc_encode_block_helper(pIn, inMask, colors, &a, 0);
  etc_encode_block_helper(pIn, inMask, flippedColors, &b, 1);
  take_best(&a, &b);
  writeBigEndian(pOut, a.high);
  writeBigEndian(pOut + 4, a.low);
}

// Return the size of the encoded image data (does not include size of PKM header).

etc1_uint32 etc1_get_encoded_data_size(etc1_uint32 width, etc1_uint32 height) {
  return (((width + 3) & ~3) * ((height + 3) & ~3)) >> 1;
}

// Encode an entire image.
// pIn - pointer to the image data. Formatted such that the Red component of
//       pixel (x,y) is at pIn + pixelSize * x + stride * y + redOffset;
// pOut - pointer to encoded data. Must be large enough to store entire encoded image.

int etc1_encode_image(const etc1_byte* pIn, etc1_uint32 width, etc1_uint32 height,
  etc1_uint32 pixelSize, etc1_uint32 stride, etc1_byte* pOut) {
  if (pixelSize < 2 || pixelSize > 3) {
    return -1;
  }
  static const unsigned short kYMask[] = { 0x0, 0xf, 0xff, 0xfff, 0xffff };
  static const unsigned short kXMask[] = { 0x0, 0x1111, 0x3333, 0x7777,
          0xffff };
  etc1_byte block[ETC1_DECODED_BLOCK_SIZE];
  etc1_byte encoded[ETC1_ENCODED_BLOCK_SIZE];
  etc1_uint32 y, x, cy, cx;

  etc1_uint32 encodedWidth = (width + 3) & ~3;
  etc1_uint32 encodedHeight = (height + 3) & ~3;

  for (y = 0; y < encodedHeight; y += 4) {
    etc1_uint32 yEnd = height - y;
    if (yEnd > 4) {
      yEnd = 4;
    }
    int ymask = kYMask[yEnd];
    for (x = 0; x < encodedWidth; x += 4) {
      etc1_uint32 xEnd = width - x;
      if (xEnd > 4) {
        xEnd = 4;
      }
      int mask = ymask & kXMask[xEnd];
      for (cy = 0; cy < yEnd; cy++) {
        etc1_byte* q = block + (cy * 4) * 3;
        const etc1_byte* p = pIn + pixelSize * x + stride * (y + cy);
        if (pixelSize == 3) {
          memcpy(q, p, xEnd * 3);
        }
        else {
          for (cx = 0; cx < xEnd; cx++) {
            int pixel = (p[1] << 8) | p[0];
            *q++ = convert5To8(pixel >> 11);
            *q++ = convert6To8(pixel >> 5);
            *q++ = convert5To8(pixel);
            p += pixelSize;
          }
        }
      }
      etc1_encode_block(block, mask, encoded);
      memcpy(pOut, encoded, sizeof(encoded));
      pOut += sizeof(encoded);
    }
  }
  return 0;
}

// Decode an entire image.
// pIn - pointer to encoded data.
// pOut - pointer to the image data. Will be written such that the Red component of
//       pixel (x,y) is at pIn + pixelSize * x + stride * y + redOffset. Must be
//        large enough to store entire image.


int etc1_decode_image(const etc1_byte* pIn, etc1_byte* pOut,
  etc1_uint32 width, etc1_uint32 height,
  etc1_uint32 pixelSize, etc1_uint32 stride) {
  if (pixelSize < 2 || pixelSize > 3) {
    return -1;
  }
  etc1_byte block[ETC1_DECODED_BLOCK_SIZE];

  etc1_uint32 encodedWidth = (width + 3) & ~3;
  etc1_uint32 encodedHeight = (height + 3) & ~3;

  etc1_uint32 y, x, cy, cx;

  for (y = 0; y < encodedHeight; y += 4) {
    etc1_uint32 yEnd = height - y;
    if (yEnd > 4) {
      yEnd = 4;
    }
    for (x = 0; x < encodedWidth; x += 4) {
      etc1_uint32 xEnd = width - x;
      if (xEnd > 4) {
        xEnd = 4;
      }
      etc1_decode_block(pIn, block);
      pIn += ETC1_ENCODED_BLOCK_SIZE;
      for (cy = 0; cy < yEnd; cy++) {
        const etc1_byte* q = block + (cy * 4) * 3;
        etc1_byte* p = pOut + pixelSize * x + stride * (y + cy);
        if (pixelSize == 3) {
          memcpy(p, q, xEnd * 3);
        }
        else {
          for (cx = 0; cx < xEnd; cx++) {
            etc1_byte r = *q++;
            etc1_byte g = *q++;
            etc1_byte b = *q++;
            etc1_uint32 pixel = ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
            *p++ = (etc1_byte)pixel;
            *p++ = (etc1_byte)(pixel >> 8);
          }
        }
      }
    }
  }
  return 0;
}

static const char kMagic[] = { 'P', 'K', 'M', ' ', '1', '0' };

static const etc1_uint32 ETC1_PKM_FORMAT_OFFSET = 6;
static const etc1_uint32 ETC1_PKM_ENCODED_WIDTH_OFFSET = 8;
static const etc1_uint32 ETC1_PKM_ENCODED_HEIGHT_OFFSET = 10;
static const etc1_uint32 ETC1_PKM_WIDTH_OFFSET = 12;
static const etc1_uint32 ETC1_PKM_HEIGHT_OFFSET = 14;

static const etc1_uint32 ETC1_RGB_NO_MIPMAPS = 0;

static void writeBEUint16(etc1_byte* pOut, etc1_uint32 data) {
  pOut[0] = (etc1_byte)(data >> 8);
  pOut[1] = (etc1_byte)data;
}

static etc1_uint32 readBEUint16(const etc1_byte* pIn) {
  return (pIn[0] << 8) | pIn[1];
}

// Format a PKM header

void etc1_pkm_format_header(etc1_byte* pHeader, etc1_uint32 width, etc1_uint32 height) {
  memcpy(pHeader, kMagic, sizeof(kMagic));
  etc1_uint32 encodedWidth = (width + 3) & ~3;
  etc1_uint32 encodedHeight = (height + 3) & ~3;
  writeBEUint16(pHeader + ETC1_PKM_FORMAT_OFFSET, ETC1_RGB_NO_MIPMAPS);
  writeBEUint16(pHeader + ETC1_PKM_ENCODED_WIDTH_OFFSET, encodedWidth);
  writeBEUint16(pHeader + ETC1_PKM_ENCODED_HEIGHT_OFFSET, encodedHeight);
  writeBEUint16(pHeader + ETC1_PKM_WIDTH_OFFSET, width);
  writeBEUint16(pHeader + ETC1_PKM_HEIGHT_OFFSET, height);
}

// Check if a PKM header is correctly formatted.

etc1_bool etc1_pkm_is_valid(const etc1_byte* pHeader) {
  if (memcmp(pHeader, kMagic, sizeof(kMagic))) {
    return 0;
  }
  etc1_uint32 format = readBEUint16(pHeader + ETC1_PKM_FORMAT_OFFSET);
  etc1_uint32 encodedWidth = readBEUint16(pHeader + ETC1_PKM_ENCODED_WIDTH_OFFSET);
  etc1_uint32 encodedHeight = readBEUint16(pHeader + ETC1_PKM_ENCODED_HEIGHT_OFFSET);
  etc1_uint32 width = readBEUint16(pHeader + ETC1_PKM_WIDTH_OFFSET);
  etc1_uint32 height = readBEUint16(pHeader + ETC1_PKM_HEIGHT_OFFSET);
  return format == ETC1_RGB_NO_MIPMAPS &&
    encodedWidth >= width && encodedWidth - width < 4 &&
    encodedHeight >= height && encodedHeight - height < 4;
}

// Read the image width from a PKM header

etc1_uint32 etc1_pkm_get_width(const etc1_byte* pHeader) {
  return readBEUint16(pHeader + ETC1_PKM_WIDTH_OFFSET);
}

// Read the image height from a PKM header

etc1_uint32 etc1_pkm_get_height(const etc1_byte* pHeader) {
  return readBEUint16(pHeader + ETC1_PKM_HEIGHT_OFFSET);
}



/*
    Jonathan Dummer

    image helper functions

    MIT license
*/


/*	Upscaling the image uses simple bilinear interpolation	*/
int
up_scale_image
(
  const unsigned char* const orig,
  int width, int height, int channels,
  unsigned char* resampled,
  int resampled_width, int resampled_height
)
{
  float dx, dy;
  int x, y, c;

  /* error(s) check	*/
  if ((width < 1) || (height < 1) ||
    (resampled_width < 2) || (resampled_height < 2) ||
    (channels < 1) ||
    (nullptr == orig) || (nullptr == resampled))
  {
    /*	signify badness	*/
    return 0;
  }
  /*
  for each given pixel in the new map, find the exact location
  from the original map which would contribute to this guy
*/
  dx = (width - 1.0f) / (resampled_width - 1.0f);
  dy = (height - 1.0f) / (resampled_height - 1.0f);
  for (y = 0; y < resampled_height; ++y)
  {
    /* find the base y index and fractional offset from that	*/
    float sampley = y * dy;
    int inty = (int)sampley;
    /*	if( inty < 0 ) { inty = 0; } else	*/
    if (inty > height - 2) { inty = height - 2; }
    sampley -= inty;
    for (x = 0; x < resampled_width; ++x)
    {
      float samplex = x * dx;
      int intx = (int)samplex;
      int base_index;
      /* find the base x index and fractional offset from that	*/
      /*	if( intx < 0 ) { intx = 0; } else	*/
      if (intx > width - 2) { intx = width - 2; }
      samplex -= intx;
      /*	base index into the original image	*/
      base_index = (inty * width + intx) * channels;
      for (c = 0; c < channels; ++c)
      {
        /*	do the sampling	*/
        float value = 0.5f;
        value += orig[base_index]
          * (1.0f - samplex) * (1.0f - sampley);
        value += orig[base_index + channels]
          * (samplex) * (1.0f - sampley);
        value += orig[base_index + width * channels]
          * (1.0f - samplex) * (sampley);
        value += orig[base_index + width * channels + channels]
          * (samplex) * (sampley);
        /*	move to the next channel	*/
        ++base_index;
        /*	save the new value	*/
        resampled[y * resampled_width * channels + x * channels + c] =
          (unsigned char)(value);
      }
    }
  }
  /*	done	*/
  return 1;
}

int
mipmap_image
(
  const unsigned char* const orig,
  int width, int height, int channels,
  unsigned char* resampled,
  int block_size_x, int block_size_y
)
{
  int mip_width, mip_height;
  int i, j, c;

  /*	error check	*/
  if ((width < 1) || (height < 1) ||
    (channels < 1) || (orig == nullptr) ||
    (resampled == nullptr) ||
    (block_size_x < 1) || (block_size_y < 1))
  {
    /*	nothing to do	*/
    return 0;
  }
  mip_width = width / block_size_x;
  mip_height = height / block_size_y;
  if (mip_width < 1)
  {
    mip_width = 1;
  }
  if (mip_height < 1)
  {
    mip_height = 1;
  }
  for (j = 0; j < mip_height; ++j)
  {
    for (i = 0; i < mip_width; ++i)
    {
      for (c = 0; c < channels; ++c)
      {
        const int index = (j * block_size_y) * width * channels + (i * block_size_x) * channels + c;
        int sum_value;
        int u, v;
        int u_block = block_size_x;
        int v_block = block_size_y;
        int block_area;
        /*	do a bit of checking so we don't over-run the boundaries
          (necessary for non-square textures!)	*/
        if (block_size_x * (i + 1) > width)
        {
          u_block = width - i * block_size_y;
        }
        if (block_size_y * (j + 1) > height)
        {
          v_block = height - j * block_size_y;
        }
        block_area = u_block * v_block;
        /*	for this pixel, see what the average
          of all the values in the block are.
          note: start the sum at the rounding value, not at 0	*/
        sum_value = block_area >> 1;
        for (v = 0; v < v_block; ++v)
          for (u = 0; u < u_block; ++u)
          {
            sum_value += orig[index + v * width * channels + u * channels];
          }
        resampled[j * mip_width * channels + i * channels + c] = sum_value / block_area;
      }
    }
  }
  return 1;
}

int
scale_image_RGB_to_NTSC_safe
(
  unsigned char* orig,
  int width, int height, int channels
)
{
  const float scale_lo = 16.0f - 0.499f;
  const float scale_hi = 235.0f + 0.499f;
  int i, j;
  int nc = channels;
  unsigned char scale_LUT[256];
  /*	error check	*/
  if ((width < 1) || (height < 1) ||
    (channels < 1) || (orig == nullptr))
  {
    /*	nothing to do	*/
    return 0;
  }
  /*	set up the scaling Look Up Table	*/
  for (i = 0; i < 256; ++i)
  {
    scale_LUT[i] = (unsigned char)((scale_hi - scale_lo) * i / 255.0f + scale_lo);
  }
  /*	for channels = 2 or 4, ignore the alpha component	*/
  nc -= 1 - (channels & 1);
  /*	OK, go through the image and scale any non-alpha components	*/
  for (i = 0; i < width * height * channels; i += channels)
  {
    for (j = 0; j < nc; ++j)
    {
      orig[i + j] = scale_LUT[orig[i + j]];
    }
  }
  return 1;
}

unsigned char clamp_byte(int x) { return ((x) < 0 ? (0) : ((x) > 255 ? 255 : (x))); }

/*
  This function takes the RGB components of the image
  and converts them into YCoCg.  3 components will be
  re-ordered to CoYCg (for optimum DXT1 compression),
  while 4 components will be ordered CoCgAY (for DXT5
  compression).
*/
int
convert_RGB_to_YCoCg
(
  unsigned char* orig,
  int width, int height, int channels
)
{
  int i;
  /*	error check	*/
  if ((width < 1) || (height < 1) ||
    (channels < 3) || (channels > 4) ||
    (orig == nullptr))
  {
    /*	nothing to do	*/
    return -1;
  }
  /*	do the conversion	*/
  if (channels == 3)
  {
    for (i = 0; i < width * height * 3; i += 3)
    {
      int r = orig[i + 0];
      int g = (orig[i + 1] + 1) >> 1;
      int b = orig[i + 2];
      int tmp = (2 + r + b) >> 2;
      /*	Co	*/
      orig[i + 0] = clamp_byte(128 + ((r - b + 1) >> 1));
      /*	Y	*/
      orig[i + 1] = clamp_byte(g + tmp);
      /*	Cg	*/
      orig[i + 2] = clamp_byte(128 + g - tmp);
    }
  }
  else
  {
    for (i = 0; i < width * height * 4; i += 4)
    {
      int r = orig[i + 0];
      int g = (orig[i + 1] + 1) >> 1;
      int b = orig[i + 2];
      unsigned char a = orig[i + 3];
      int tmp = (2 + r + b) >> 2;
      /*	Co	*/
      orig[i + 0] = clamp_byte(128 + ((r - b + 1) >> 1));
      /*	Cg	*/
      orig[i + 1] = clamp_byte(128 + g - tmp);
      /*	Alpha	*/
      orig[i + 2] = a;
      /*	Y	*/
      orig[i + 3] = clamp_byte(g + tmp);
    }
  }
  /*	done	*/
  return 0;
}

/*
  This function takes the YCoCg components of the image
  and converts them into RGB.  See above.
*/
int
convert_YCoCg_to_RGB
(
  unsigned char* orig,
  int width, int height, int channels
)
{
  int i;
  /*	error check	*/
  if ((width < 1) || (height < 1) ||
    (channels < 3) || (channels > 4) ||
    (orig == nullptr))
  {
    /*	nothing to do	*/
    return -1;
  }
  /*	do the conversion	*/
  if (channels == 3)
  {
    for (i = 0; i < width * height * 3; i += 3)
    {
      int co = orig[i + 0] - 128;
      int y = orig[i + 1];
      int cg = orig[i + 2] - 128;
      /*	R	*/
      orig[i + 0] = clamp_byte(y + co - cg);
      /*	G	*/
      orig[i + 1] = clamp_byte(y + cg);
      /*	B	*/
      orig[i + 2] = clamp_byte(y - co - cg);
    }
  }
  else
  {
    for (i = 0; i < width * height * 4; i += 4)
    {
      int co = orig[i + 0] - 128;
      int cg = orig[i + 1] - 128;
      unsigned char a = orig[i + 2];
      int y = orig[i + 3];
      /*	R	*/
      orig[i + 0] = clamp_byte(y + co - cg);
      /*	G	*/
      orig[i + 1] = clamp_byte(y + cg);
      /*	B	*/
      orig[i + 2] = clamp_byte(y - co - cg);
      /*	A	*/
      orig[i + 3] = a;
    }
  }
  /*	done	*/
  return 0;
}

float
find_max_RGBE
(
  unsigned char* image,
  int width, int height
)
{
  float max_val = 0.0f;
  unsigned char* img = image;
  int i, j;
  for (i = width * height; i > 0; --i)
  {
    /* float scale = powf( 2.0f, img[3] - 128.0f ) / 255.0f; */
    float scale = (float)ldexp(1.0f / 255.0f, (int)(img[3]) - 128);
    for (j = 0; j < 3; ++j)
    {
      if (img[j] * scale > max_val)
      {
        max_val = img[j] * scale;
      }
    }
    /* next pixel */
    img += 4;
  }
  return max_val;
}

int
RGBE_to_RGBdivA
(
  unsigned char* image,
  int width, int height,
  int rescale_to_max
)
{
  /* local variables */
  int i, iv;
  unsigned char* img = image;
  float scale = 1.0f;
  /* error check */
  if ((!image) || (width < 1) || (height < 1))
  {
    return 0;
  }
  /* convert (note: no negative numbers, but 0.0 is possible) */
  if (rescale_to_max)
  {
    scale = 255.0f / find_max_RGBE(image, width, height);
  }
  for (i = width * height; i > 0; --i)
  {
    /* decode this pixel, and find the max */
    float r, g, b, e, m;
    /* e = scale * powf( 2.0f, img[3] - 128.0f ) / 255.0f; */
    e = scale * (float)ldexp(1.0f / 255.0f, (int)(img[3]) - 128);
    r = e * img[0];
    g = e * img[1];
    b = e * img[2];
    m = (r > g) ? r : g;
    m = (b > m) ? b : m;
    /* and encode it into RGBdivA */
    iv = (m != 0.0f) ? (int)(255.0f / m) : 1;
    iv = (iv < 1) ? 1 : iv;
    img[3] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * r + 0.5f);
    img[0] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * g + 0.5f);
    img[1] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * b + 0.5f);
    img[2] = (iv > 255) ? 255 : iv;
    /* and on to the next pixel */
    img += 4;
  }
  return 1;
}

int
RGBE_to_RGBdivA2
(
  unsigned char* image,
  int width, int height,
  int rescale_to_max
)
{
  /* local variables */
  int i, iv;
  unsigned char* img = image;
  float scale = 1.0f;
  /* error check */
  if ((!image) || (width < 1) || (height < 1))
  {
    return 0;
  }
  /* convert (note: no negative numbers, but 0.0 is possible) */
  if (rescale_to_max)
  {
    scale = 255.0f * 255.0f / find_max_RGBE(image, width, height);
  }
  for (i = width * height; i > 0; --i)
  {
    /* decode this pixel, and find the max */
    float r, g, b, e, m;
    /* e = scale * powf( 2.0f, img[3] - 128.0f ) / 255.0f; */
    e = scale * (float)ldexp(1.0f / 255.0f, (int)(img[3]) - 128);
    r = e * img[0];
    g = e * img[1];
    b = e * img[2];
    m = (r > g) ? r : g;
    m = (b > m) ? b : m;
    /* and encode it into RGBdivA */
    iv = (m != 0.0f) ? (int)sqrtf(255.0f * 255.0f / m) : 1;
    iv = (iv < 1) ? 1 : iv;
    img[3] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * img[3] * r / 255.0f + 0.5f);
    img[0] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * img[3] * g / 255.0f + 0.5f);
    img[1] = (iv > 255) ? 255 : iv;
    iv = (int)(img[3] * img[3] * b / 255.0f + 0.5f);
    img[2] = (iv > 255) ? 255 : iv;
    /* and on to the next pixel */
    img += 4;
  }
  return 1;
}







/*
  Jonathan Dummer
  2007-07-31-10.32

  simple DXT compression / decompression code

  public domain
*/

#ifndef HEADER_IMAGE_DXT
#define HEADER_IMAGE_DXT

/**
  take an image and convert it to DXT1 (no alpha)
**/
unsigned char*
convert_image_to_DXT1
(
  const unsigned char* const uncompressed,
  int width, int height, int channels,
  int* out_size
);

/**
  take an image and convert it to DXT5 (with alpha)
**/
unsigned char*
convert_image_to_DXT5
(
  const unsigned char* const uncompressed,
  int width, int height, int channels,
  int* out_size
);

/**	A bunch of DirectDraw Surface structures and flags **/
typedef struct
{
  unsigned int    dwMagic;
  unsigned int    dwSize;
  unsigned int    dwFlags;
  unsigned int    dwHeight;
  unsigned int    dwWidth;
  unsigned int    dwPitchOrLinearSize;
  unsigned int    dwDepth;
  unsigned int    dwMipMapCount;
  unsigned int    dwReserved1[11];

  /*  DDPIXELFORMAT	*/
  struct
  {
    unsigned int    dwSize;
    unsigned int    dwFlags;
    unsigned int    dwFourCC;
    unsigned int    dwRGBBitCount;
    unsigned int    dwRBitMask;
    unsigned int    dwGBitMask;
    unsigned int    dwBBitMask;
    unsigned int    dwAlphaBitMask;
  }
  sPixelFormat;

  /*  DDCAPS2	*/
  struct
  {
    unsigned int    dwCaps1;
    unsigned int    dwCaps2;
    unsigned int    dwDDSX;
    unsigned int    dwReserved;
  }
  sCaps;
  unsigned int    dwReserved2;
}
DDS_header;

/*	the following constants were copied directly off the MSDN website	*/

/*	The dwFlags member of the original DDSURFACEDESC2 structure
  can be set to one or more of the following values.	*/
#define DDSD_CAPS	0x00000001
#define DDSD_HEIGHT	0x00000002
#define DDSD_WIDTH	0x00000004
#define DDSD_PITCH	0x00000008
#define DDSD_PIXELFORMAT	0x00001000
#define DDSD_MIPMAPCOUNT	0x00020000
#define DDSD_LINEARSIZE	0x00080000
#define DDSD_DEPTH	0x00800000

  /*	DirectDraw Pixel Format	*/
#define DDPF_ALPHAPIXELS	0x00000001
#define DDPF_FOURCC	0x00000004
#define DDPF_RGB	0x00000040

/*	The dwCaps1 member of the DDSCAPS2 structure can be
  set to one or more of the following values.	*/
#define DDSCAPS_COMPLEX	0x00000008
#define DDSCAPS_TEXTURE	0x00001000
#define DDSCAPS_MIPMAP	0x00400000

  /*	The dwCaps2 member of the DDSCAPS2 structure can be
    set to one or more of the following values.		*/
#define DDSCAPS2_CUBEMAP	0x00000200
#define DDSCAPS2_CUBEMAP_POSITIVEX	0x00000400
#define DDSCAPS2_CUBEMAP_NEGATIVEX	0x00000800
#define DDSCAPS2_CUBEMAP_POSITIVEY	0x00001000
#define DDSCAPS2_CUBEMAP_NEGATIVEY	0x00002000
#define DDSCAPS2_CUBEMAP_POSITIVEZ	0x00004000
#define DDSCAPS2_CUBEMAP_NEGATIVEZ	0x00008000
#define DDSCAPS2_VOLUME	0x00200000

#endif /* HEADER_IMAGE_DXT	*/


    /*	set this =1 if you want to use the covarince matrix method...
      which is better than my method of using standard deviations
      overall, except on the infintesimal chance that the power
      method fails for finding the largest eigenvector	*/
#define USE_COV_MAT	1

      /********* Function Prototypes *********/
      /*
        Takes a 4x4 block of pixels and compresses it into 8 bytes
        in DXT1 format (color only, no alpha).  Speed is valued
        over prettyness, at least for now.
      */
void compress_DDS_color_block(
  int channels,
  const unsigned char* const uncompressed,
  unsigned char compressed[8]);
/*
  Takes a 4x4 block of pixels and compresses the alpha
  component it into 8 bytes for use in DXT5 DDS files.
  Speed is valued over prettyness, at least for now.
*/
void compress_DDS_alpha_block(
  const unsigned char* const uncompressed,
  unsigned char compressed[8]);

unsigned char* convert_image_to_DXT1(
  const unsigned char* const uncompressed,
  int width, int height, int channels,
  int* out_size)
{
  unsigned char* compressed;
  int i, j, x, y;
  unsigned char ublock[16 * 3];
  unsigned char cblock[8];
  int index = 0, chan_step = 1;
  int block_count = 0;
  /*	error check	*/
  *out_size = 0;
  if ((width < 1) || (height < 1) ||
    (nullptr == uncompressed) ||
    (channels < 1) || (channels > 4))
  {
    return nullptr;
  }
  /*	for channels == 1 or 2, I do not step forward for R,G,B values	*/
  if (channels < 3)
  {
    chan_step = 0;
  }
  /*	get the RAM for the compressed image
    (8 bytes per 4x4 pixel block)	*/
  *out_size = ((width + 3) >> 2) * ((height + 3) >> 2) * 8;
  compressed = (unsigned char*)malloc(*out_size);
  /*	go through each block	*/
  for (j = 0; j < height; j += 4)
  {
    for (i = 0; i < width; i += 4)
    {
      /*	copy this block into a new one	*/
      int idx = 0;
      int mx = 4, my = 4;
      if (j + 4 >= height)
      {
        my = height - j;
      }
      if (i + 4 >= width)
      {
        mx = width - i;
      }
      for (y = 0; y < my; ++y)
      {
        for (x = 0; x < mx; ++x)
        {
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels];
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels + chan_step];
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels + chan_step + chan_step];
        }
        for (x = mx; x < 4; ++x)
        {
          ublock[idx++] = ublock[0];
          ublock[idx++] = ublock[1];
          ublock[idx++] = ublock[2];
        }
      }
      for (y = my; y < 4; ++y)
      {
        for (x = 0; x < 4; ++x)
        {
          ublock[idx++] = ublock[0];
          ublock[idx++] = ublock[1];
          ublock[idx++] = ublock[2];
        }
      }
      /*	compress the block	*/
      ++block_count;
      compress_DDS_color_block(3, ublock, cblock);
      /*	copy the data from the block into the main block	*/
      for (x = 0; x < 8; ++x)
      {
        compressed[index++] = cblock[x];
      }
    }
  }
  return compressed;
}

unsigned char* convert_image_to_DXT5(
  const unsigned char* const uncompressed,
  int width, int height, int channels,
  int* out_size)
{
  unsigned char* compressed;
  int i, j, x, y;
  unsigned char ublock[16 * 4];
  unsigned char cblock[8];
  int index = 0, chan_step = 1;
  int block_count = 0, has_alpha;
  /*	error check	*/
  *out_size = 0;
  if ((width < 1) || (height < 1) ||
    (nullptr == uncompressed) ||
    (channels < 1) || (channels > 4))
  {
    return nullptr;
  }
  /*	for channels == 1 or 2, I do not step forward for R,G,B vales	*/
  if (channels < 3)
  {
    chan_step = 0;
  }
  /*	# channels = 1 or 3 have no alpha, 2 & 4 do have alpha	*/
  has_alpha = 1 - (channels & 1);
  /*	get the RAM for the compressed image
    (16 bytes per 4x4 pixel block)	*/
  *out_size = ((width + 3) >> 2) * ((height + 3) >> 2) * 16;
  compressed = (unsigned char*)malloc(*out_size);
  /*	go through each block	*/
  for (j = 0; j < height; j += 4)
  {
    for (i = 0; i < width; i += 4)
    {
      /*	local variables, and my block counter	*/
      int idx = 0;
      int mx = 4, my = 4;
      if (j + 4 >= height)
      {
        my = height - j;
      }
      if (i + 4 >= width)
      {
        mx = width - i;
      }
      for (y = 0; y < my; ++y)
      {
        for (x = 0; x < mx; ++x)
        {
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels];
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels + chan_step];
          ublock[idx++] = uncompressed[(j + y) * width * channels + (i + x) * channels + chan_step + chan_step];
          ublock[idx++] =
            has_alpha * uncompressed[(j + y) * width * channels + (i + x) * channels + channels - 1]
            + (1 - has_alpha) * 255;
        }
        for (x = mx; x < 4; ++x)
        {
          ublock[idx++] = ublock[0];
          ublock[idx++] = ublock[1];
          ublock[idx++] = ublock[2];
          ublock[idx++] = ublock[3];
        }
      }
      for (y = my; y < 4; ++y)
      {
        for (x = 0; x < 4; ++x)
        {
          ublock[idx++] = ublock[0];
          ublock[idx++] = ublock[1];
          ublock[idx++] = ublock[2];
          ublock[idx++] = ublock[3];
        }
      }
      /*	now compress the alpha block	*/
      compress_DDS_alpha_block(ublock, cblock);
      /*	copy the data from the compressed alpha block into the main buffer	*/
      for (x = 0; x < 8; ++x)
      {
        compressed[index++] = cblock[x];
      }
      /*	then compress the color block	*/
      ++block_count;
      compress_DDS_color_block(4, ublock, cblock);
      /*	copy the data from the compressed color block into the main buffer	*/
      for (x = 0; x < 8; ++x)
      {
        compressed[index++] = cblock[x];
      }
    }
  }
  return compressed;
}

/********* Helper Functions *********/
int convert_bit_range(int c, int from_bits, int to_bits)
{
  int b = (1 << (from_bits - 1)) + c * ((1 << to_bits) - 1);
  return (b + (b >> from_bits)) >> from_bits;
}

int rgb_to_565(int r, int g, int b)
{
  return
    (convert_bit_range(r, 8, 5) << 11) |
    (convert_bit_range(g, 8, 6) << 05) |
    (convert_bit_range(b, 8, 5) << 00);
}

void rgb_888_from_565(unsigned int c, int* r, int* g, int* b)
{
  *r = convert_bit_range((c >> 11) & 31, 5, 8);
  *g = convert_bit_range((c >> 05) & 63, 6, 8);
  *b = convert_bit_range((c >> 00) & 31, 5, 8);
}

void compute_color_line_STDEV(
  const unsigned char* const uncompressed,
  int channels,
  float point[3], float direction[3])
{
  const float inv_16 = 1.0f / 16.0f;
  int i;
  float sum_r = 0.0f, sum_g = 0.0f, sum_b = 0.0f;
  float sum_rr = 0.0f, sum_gg = 0.0f, sum_bb = 0.0f;
  float sum_rg = 0.0f, sum_rb = 0.0f, sum_gb = 0.0f;
  /*	calculate all data needed for the covariance matrix
    ( to compare with _rygdxt code)	*/
  for (i = 0; i < 16 * channels; i += channels)
  {
    sum_r += uncompressed[i + 0];
    sum_rr += uncompressed[i + 0] * uncompressed[i + 0];
    sum_g += uncompressed[i + 1];
    sum_gg += uncompressed[i + 1] * uncompressed[i + 1];
    sum_b += uncompressed[i + 2];
    sum_bb += uncompressed[i + 2] * uncompressed[i + 2];
    sum_rg += uncompressed[i + 0] * uncompressed[i + 1];
    sum_rb += uncompressed[i + 0] * uncompressed[i + 2];
    sum_gb += uncompressed[i + 1] * uncompressed[i + 2];
  }
  /*	convert the sums to averages	*/
  sum_r *= inv_16;
  sum_g *= inv_16;
  sum_b *= inv_16;
  /*	and convert the squares to the squares of the value - avg_value	*/
  sum_rr -= 16.0f * sum_r * sum_r;
  sum_gg -= 16.0f * sum_g * sum_g;
  sum_bb -= 16.0f * sum_b * sum_b;
  sum_rg -= 16.0f * sum_r * sum_g;
  sum_rb -= 16.0f * sum_r * sum_b;
  sum_gb -= 16.0f * sum_g * sum_b;
  /*	the point on the color line is the average	*/
  point[0] = sum_r;
  point[1] = sum_g;
  point[2] = sum_b;
#if USE_COV_MAT
  /*
    The following idea was from ryg.
    (https://mollyrocket.com/forums/viewtopic.php?t=392)
    The method worked great (less RMSE than mine) most of
    the time, but had some issues handling some simple
    boundary cases, like full green next to full red,
    which would generate a covariance matrix like this:

    | 1  -1  0 |
    | -1  1  0 |
    | 0   0  0 |

    For a given starting vector, the power method can
    generate all zeros!  So no starting with {1,1,1}
    as I was doing!  This kind of error is still a
    slight posibillity, but will be very rare.
  */
  /*	use the covariance matrix directly
    (1st iteration, don't use all 1.0 values!)	*/
  sum_r = 1.0f;
  sum_g = 2.718281828f;
  sum_b = 3.141592654f;
  direction[0] = sum_r * sum_rr + sum_g * sum_rg + sum_b * sum_rb;
  direction[1] = sum_r * sum_rg + sum_g * sum_gg + sum_b * sum_gb;
  direction[2] = sum_r * sum_rb + sum_g * sum_gb + sum_b * sum_bb;
  /*	2nd iteration, use results from the 1st guy	*/
  sum_r = direction[0];
  sum_g = direction[1];
  sum_b = direction[2];
  direction[0] = sum_r * sum_rr + sum_g * sum_rg + sum_b * sum_rb;
  direction[1] = sum_r * sum_rg + sum_g * sum_gg + sum_b * sum_gb;
  direction[2] = sum_r * sum_rb + sum_g * sum_gb + sum_b * sum_bb;
  /*	3rd iteration, use results from the 2nd guy	*/
  sum_r = direction[0];
  sum_g = direction[1];
  sum_b = direction[2];
  direction[0] = sum_r * sum_rr + sum_g * sum_rg + sum_b * sum_rb;
  direction[1] = sum_r * sum_rg + sum_g * sum_gg + sum_b * sum_gb;
  direction[2] = sum_r * sum_rb + sum_g * sum_gb + sum_b * sum_bb;
#else
  /*	use my standard deviation method
    (very robust, a tiny bit slower and less accurate)	*/
  direction[0] = sqrt(sum_rr);
  direction[1] = sqrt(sum_gg);
  direction[2] = sqrt(sum_bb);
  /*	which has a greater component	*/
  if (sum_gg > sum_rr)
  {
    /*	green has greater component, so base the other signs off of green	*/
    if (sum_rg < 0.0f)
    {
      direction[0] = -direction[0];
    }
    if (sum_gb < 0.0f)
    {
      direction[2] = -direction[2];
    }
  }
  else
  {
    /*	red has a greater component	*/
    if (sum_rg < 0.0f)
    {
      direction[1] = -direction[1];
    }
    if (sum_rb < 0.0f)
    {
      direction[2] = -direction[2];
    }
  }
#endif
}

void LSE_master_colors_max_min(
  int* cmax, int* cmin,
  int channels,
  const unsigned char* const uncompressed)
{
  int i, j;
  /*	the master colors	*/
  int c0[3], c1[3];
  /*	used for fitting the line	*/
  float sum_x[] = { 0.0f, 0.0f, 0.0f };
  float sum_x2[] = { 0.0f, 0.0f, 0.0f };
  float dot_max = 1.0f, dot_min = -1.0f;
  float vec_len2 = 0.0f;
  float dot;
  /*	error check	*/
  if ((channels < 3) || (channels > 4))
  {
    return;
  }
  compute_color_line_STDEV(uncompressed, channels, sum_x, sum_x2);
  vec_len2 = 1.0f / (0.00001f +
    sum_x2[0] * sum_x2[0] + sum_x2[1] * sum_x2[1] + sum_x2[2] * sum_x2[2]);
  /*	finding the max and min vector values	*/
  dot_max =
    (
      sum_x2[0] * uncompressed[0] +
      sum_x2[1] * uncompressed[1] +
      sum_x2[2] * uncompressed[2]
      );
  dot_min = dot_max;
  for (i = 1; i < 16; ++i)
  {
    dot =
      (
        sum_x2[0] * uncompressed[i * channels + 0] +
        sum_x2[1] * uncompressed[i * channels + 1] +
        sum_x2[2] * uncompressed[i * channels + 2]
        );
    if (dot < dot_min)
    {
      dot_min = dot;
    }
    else if (dot > dot_max)
    {
      dot_max = dot;
    }
  }
  /*	and the offset (from the average location)	*/
  dot = sum_x2[0] * sum_x[0] + sum_x2[1] * sum_x[1] + sum_x2[2] * sum_x[2];
  dot_min -= dot;
  dot_max -= dot;
  /*	post multiply by the scaling factor	*/
  dot_min *= vec_len2;
  dot_max *= vec_len2;
  /*	OK, build the master colors	*/
  for (i = 0; i < 3; ++i)
  {
    /*	color 0	*/
    c0[i] = (int)(0.5f + sum_x[i] + dot_max * sum_x2[i]);
    if (c0[i] < 0)
    {
      c0[i] = 0;
    }
    else if (c0[i] > 255)
    {
      c0[i] = 255;
    }
    /*	color 1	*/
    c1[i] = (int)(0.5f + sum_x[i] + dot_min * sum_x2[i]);
    if (c1[i] < 0)
    {
      c1[i] = 0;
    }
    else if (c1[i] > 255)
    {
      c1[i] = 255;
    }
  }
  /*	down_sample (with rounding?)	*/
  i = rgb_to_565(c0[0], c0[1], c0[2]);
  j = rgb_to_565(c1[0], c1[1], c1[2]);
  if (i > j)
  {
    *cmax = i;
    *cmin = j;
  }
  else
  {
    *cmax = j;
    *cmin = i;
  }
}

void
compress_DDS_color_block
(
  int channels,
  const unsigned char* const uncompressed,
  unsigned char compressed[8]
)
{
  /*	variables	*/
  int i;
  int next_bit;
  int enc_c0, enc_c1;
  int c0[4], c1[4];
  float color_line[] = { 0.0f, 0.0f, 0.0f, 0.0f };
  float vec_len2 = 0.0f, dot_offset = 0.0f;
  /*	stupid order	*/
  int swizzle4[] = { 0, 2, 3, 1 };
  /*	get the master colors	*/
  LSE_master_colors_max_min(&enc_c0, &enc_c1, channels, uncompressed);
  /*	store the 565 color 0 and color 1	*/
  compressed[0] = (enc_c0 >> 0) & 255;
  compressed[1] = (enc_c0 >> 8) & 255;
  compressed[2] = (enc_c1 >> 0) & 255;
  compressed[3] = (enc_c1 >> 8) & 255;
  /*	zero out the compressed data	*/
  compressed[4] = 0;
  compressed[5] = 0;
  compressed[6] = 0;
  compressed[7] = 0;
  /*	reconstitute the master color vectors	*/
  rgb_888_from_565(enc_c0, &c0[0], &c0[1], &c0[2]);
  rgb_888_from_565(enc_c1, &c1[0], &c1[1], &c1[2]);
  /*	the new vector	*/
  vec_len2 = 0.0f;
  for (i = 0; i < 3; ++i)
  {
    color_line[i] = (float)(c1[i] - c0[i]);
    vec_len2 += color_line[i] * color_line[i];
  }
  if (vec_len2 > 0.0f)
  {
    vec_len2 = 1.0f / vec_len2;
  }
  /*	pre-proform the scaling	*/
  color_line[0] *= vec_len2;
  color_line[1] *= vec_len2;
  color_line[2] *= vec_len2;
  /*	compute the offset (constant) portion of the dot product	*/
  dot_offset = color_line[0] * c0[0] + color_line[1] * c0[1] + color_line[2] * c0[2];
  /*	store the rest of the bits	*/
  next_bit = 8 * 4;
  for (i = 0; i < 16; ++i)
  {
    /*	find the dot product of this color, to place it on the line
      (should be [-1,1])	*/
    int next_value = 0;
    float dot_product =
      color_line[0] * uncompressed[i * channels + 0] +
      color_line[1] * uncompressed[i * channels + 1] +
      color_line[2] * uncompressed[i * channels + 2] -
      dot_offset;
    /*	map to [0,3]	*/
    next_value = (int)(dot_product * 3.0f + 0.5f);
    if (next_value > 3)
    {
      next_value = 3;
    }
    else if (next_value < 0)
    {
      next_value = 0;
    }
    /*	OK, store this value	*/
    compressed[next_bit >> 3] |= swizzle4[next_value] << (next_bit & 7);
    next_bit += 2;
  }
  /*	done compressing to DXT1	*/
}

void
compress_DDS_alpha_block
(
  const unsigned char* const uncompressed,
  unsigned char compressed[8]
)
{
  /*	variables	*/
  int i;
  int next_bit;
  int a0, a1;
  float scale_me;
  /*	stupid order	*/
  int swizzle8[] = { 1, 7, 6, 5, 4, 3, 2, 0 };
  /*	get the alpha limits (a0 > a1)	*/
  a0 = a1 = uncompressed[3];
  for (i = 4 + 3; i < 16 * 4; i += 4)
  {
    if (uncompressed[i] > a0)
    {
      a0 = uncompressed[i];
    }
    else if (uncompressed[i] < a1)
    {
      a1 = uncompressed[i];
    }
  }
  /*	store those limits, and zero the rest of the compressed dataset	*/
  compressed[0] = a0;
  compressed[1] = a1;
  /*	zero out the compressed data	*/
  compressed[2] = 0;
  compressed[3] = 0;
  compressed[4] = 0;
  compressed[5] = 0;
  compressed[6] = 0;
  compressed[7] = 0;
  /*	store the all of the alpha values	*/
  next_bit = 8 * 2;
  scale_me = 7.9999f / (a0 - a1);
  for (i = 3; i < 16 * 4; i += 4)
  {
    /*	convert this alpha value to a 3 bit number	*/
    int svalue;
    int value = (int)((uncompressed[i] - a1) * scale_me);
    svalue = swizzle8[value & 7];
    /*	OK, store this value, start with the 1st byte	*/
    compressed[next_bit >> 3] |= svalue << (next_bit & 7);
    if ((next_bit & 7) > 5)
    {
      /*	spans 2 bytes, fill in the start of the 2nd byte	*/
      compressed[1 + (next_bit >> 3)] |= svalue >> (8 - (next_bit & 7));
    }
    next_bit += 3;
  }
  /*	done compressing to DXT1	*/
}














#define SOIL_CHECK_FOR_GL_ERRORS 0

#if defined( __APPLE_CC__ ) || defined ( __APPLE__ )
#include <TargetConditionals.h>

#if defined( __IPHONE__ ) || ( defined( TARGET_OS_IPHONE ) && TARGET_OS_IPHONE ) || ( defined( TARGET_IPHONE_SIMULATOR ) && TARGET_IPHONE_SIMULATOR )
#define SOIL_PLATFORM_IOS
#include <dlfcn.h>
#else
#define SOIL_PLATFORM_OSX
#endif
#elif defined( __ANDROID__ ) || defined( ANDROID )
#define SOIL_PLATFORM_ANDROID_SDL
#elif ( defined ( linux ) || defined( __linux__ ) || defined( __FreeBSD__ ) || defined(__OpenBSD__) || defined( __NetBSD__ ) || defined( __DragonFly__ ) || defined( __SVR4 ) )
#define SOIL_X11_PLATFORM
#endif

#if ( defined( SOIL_PLATFORM_IOS ) || defined( SOIL_PLATFORM_ANDROID_SDL ) ) && ( !defined( SOIL_GLES1 ) && !defined( SOIL_GLES2 ) )
#define SOIL_GLES2
#endif

#if ( defined( SOIL_GLES2 ) || defined( SOIL_GLES1 ) ) && !defined( SOIL_NO_EGL ) && !defined( SOIL_PLATFORM_IOS )
#include <EGL/egl.h>
#endif

#if defined( SOIL_GLES2 )
#ifdef SOIL_PLATFORM_IOS
#include <OpenGLES/ES2/gl.h>
#include <OpenGLES/ES2/glext.h>
#else
#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#endif

#define APIENTRY GL_APIENTRY
#elif defined( SOIL_GLES1 )
#ifndef GL_GLEXT_PROTOTYPES
#define GL_GLEXT_PROTOTYPES
#endif
#ifdef SOIL_PLATFORM_IOS
#include <OpenGLES/ES1/gl.h>
#include <OpenGLES/ES1/glext.h>
#else
#include <GLES/gl.h>
#include <GLES/glext.h>
#endif

#define APIENTRY GL_APIENTRY
#else

#if defined( __WIN32__ ) || defined( _WIN32 ) || defined( WIN32 )
#define SOIL_PLATFORM_WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <wingdi.h>
#include <GL/gl.h>

#ifndef GL_UNSIGNED_SHORT_4_4_4_4
#define GL_UNSIGNED_SHORT_4_4_4_4 0x8033
#endif
#ifndef GL_UNSIGNED_SHORT_5_5_5_1
#define GL_UNSIGNED_SHORT_5_5_5_1 0x8034
#endif
#ifndef GL_UNSIGNED_SHORT_5_6_5
#define GL_UNSIGNED_SHORT_5_6_5 0x8363
#endif
#elif defined(__APPLE__) || defined(__APPLE_CC__)
/*	I can't test this Apple stuff!	*/
#include <OpenGL/gl.h>
#include <Carbon/Carbon.h>
#define APIENTRY
#elif defined( SOIL_X11_PLATFORM )
#include <GL/gl.h>
#include <GL/glx.h>
#else
#include <GL/gl.h>
#endif

#endif

#ifndef GL_BGRA
#define GL_BGRA                                             0x80E1
#endif

#ifndef GL_RG
#define GL_RG                             0x8227
#endif

#ifndef HEADER_SIMPLE_OPENGL_IMAGE_LIBRARY
#define HEADER_SIMPLE_OPENGL_IMAGE_LIBRARY

#ifdef __cplusplus
extern "C" {
#endif

  enum
  {
    SOIL_LOAD_AUTO = 0,
    SOIL_LOAD_L = 1,
    SOIL_LOAD_LA = 2,
    SOIL_LOAD_RGB = 3,
    SOIL_LOAD_RGBA = 4
  };

  enum
  {
    SOIL_CREATE_NEW_ID = 0
  };

  enum
  {
    SOIL_FLAG_POWER_OF_TWO = 1,
    SOIL_FLAG_MIPMAPS = 2,
    SOIL_FLAG_TEXTURE_REPEATS = 4,
    SOIL_FLAG_MULTIPLY_ALPHA = 8,
    SOIL_FLAG_INVERT_Y = 16,
    SOIL_FLAG_COMPRESS_TO_DXT = 32,
    SOIL_FLAG_DDS_LOAD_DIRECT = 64,
    SOIL_FLAG_NTSC_SAFE_RGB = 128,
    SOIL_FLAG_CoCg_Y = 256,
    SOIL_FLAG_TEXTURE_RECTANGLE = 512,
    SOIL_FLAG_PVR_LOAD_DIRECT = 1024,
    SOIL_FLAG_ETC1_LOAD_DIRECT = 2048,
    SOIL_FLAG_GL_MIPMAPS = 4096
  };

  enum
  {
    SOIL_SAVE_TYPE_TGA = 0,
    SOIL_SAVE_TYPE_BMP = 1,
    SOIL_SAVE_TYPE_PNG = 2,
    SOIL_SAVE_TYPE_DDS = 3
  };

#define SOIL_DDS_CUBEMAP_FACE_ORDER "EWUDNS"

  enum
  {
    SOIL_HDR_RGBE = 0,
    SOIL_HDR_RGBdivA = 1,
    SOIL_HDR_RGBdivA2 = 2
  };

  unsigned int
    SOIL_load_OGL_texture
    (
      const char* filename,
      int force_channels,
      unsigned int reuse_texture_ID,
      unsigned int flags,
      int* width,
      int* height);

  unsigned int
    SOIL_load_OGL_texture_from_memory
    (const unsigned char* const buffer,
      unsigned int buffer_length,
      int force_channels,
      unsigned int reuse_texture_ID,
      unsigned int flags,
      int* width,
      int* height
    );

  unsigned char*
    SOIL_load_image
    (
      const char* filename,
      int* width, int* height, int* channels,
      int force_channels
    );

  unsigned char*
    SOIL_load_image_from_memory
    (
      const unsigned char* const buffer,
      int buffer_length,
      int* width, int* height, int* channels,
      int force_channels
    );

  void
    SOIL_free_image_data
    (
      unsigned char* img_data
    );

  unsigned int SOIL_direct_load_DDS(
    const char* filename,
    unsigned int reuse_texture_ID,
    int flags,
    int loading_as_cubemap);

  unsigned int SOIL_direct_load_DDS_from_memory(
    const unsigned char* const buffer,
    unsigned int buffer_length,
    unsigned int reuse_texture_ID,
    int flags,
    int loading_as_cubemap);

  unsigned int SOIL_direct_load_PVR(
    const char* filename,
    unsigned int reuse_texture_ID,
    int flags,
    int loading_as_cubemap);

  unsigned int SOIL_direct_load_PVR_from_memory(
    const unsigned char* const buffer,
    int buffer_length,
    unsigned int reuse_texture_ID,
    int flags,
    int loading_as_cubemap);

  unsigned int SOIL_direct_load_ETC1(const char* filename,
    unsigned int reuse_texture_ID,
    int flags);

  unsigned int SOIL_direct_load_ETC1_from_memory(const unsigned char* const buffer,
    int buffer_length,
    unsigned int reuse_texture_ID,
    int flags);

#ifdef __cplusplus
}
#endif

#endif /* HEADER_SIMPLE_OPENGL_IMAGE_LIBRARY	*/

#define STB_IMAGE_IMPLEMENTATION
/* stb_image - v2.05 - public domain image loader - http://nothings.org/stb_image.h
                                     no warranty implied; use at your own risk

   Do this:
      #define STB_IMAGE_IMPLEMENTATION
   before you include this file in *one* C or C++ file to create the implementation.

   // i.e. it should look like this:
   #include ...
   #include ...
   #include ...
   #define STB_IMAGE_IMPLEMENTATION
   #include "stb_image.h"

   You can #define STBI_ASSERT(x) before the #include to avoid using assert.h.
   And #define STBI_MALLOC, STBI_REALLOC, and STBI_FREE to avoid using malloc,realloc,free


   QUICK NOTES:
      Primarily of interest to game developers and other people who can
          avoid problematic images and only need the trivial interface

      JPEG baseline & progressive (12 bpc/arithmetic not supported, same as stock IJG lib)
      PNG 1/2/4/8-bit-per-channel (16 bpc not supported)

      TGA (not sure what subset, if a subset)
      BMP non-1bpp, non-RLE
      PSD (composited view only, no extra channels)

      GIF (*comp always reports as 4-channel)
      HDR (radiance rgbE format)
      PIC (Softimage PIC)
      PNM (PPM and PGM binary only)

      - decode from memory or through FILE (define STBI_NO_STDIO to remove code)
      - decode from arbitrary I/O callbacks
      - SIMD acceleration on x86/x64 (SSE2) and ARM (NEON)

   Full documentation under "DOCUMENTATION" below.


   Revision 2.00 release notes:

      - Progressive JPEG is now supported.

      - PPM and PGM binary formats are now supported, thanks to Ken Miller.

      - x86 platforms now make use of SSE2 SIMD instructions for
        JPEG decoding, and ARM platforms can use NEON SIMD if requested.
        This work was done by Fabian "ryg" Giesen. SSE2 is used by
        default, but NEON must be enabled explicitly; see docs.

        With other JPEG optimizations included in this version, we see
        2x speedup on a JPEG on an x86 machine, and a 1.5x speedup
        on a JPEG on an ARM machine, relative to previous versions of this
        library. The same results will not obtain for all JPGs and for all
        x86/ARM machines. (Note that progressive JPEGs are significantly
        slower to decode than regular JPEGs.) This doesn't mean that this
        is the fastest JPEG decoder in the land; rather, it brings it
        closer to parity with standard libraries. If you want the fastest
        decode, look elsewhere. (See "Philosophy" section of docs below.)

        See final bullet items below for more info on SIMD.

      - Added STBI_MALLOC, STBI_REALLOC, and STBI_FREE macros for replacing
        the memory allocator. Unlike other STBI libraries, these macros don't
        support a context parameter, so if you need to pass a context in to
        the allocator, you'll have to store it in a global or a thread-local
        variable.

      - Split existing STBI_NO_HDR flag into two flags, STBI_NO_HDR and
        STBI_NO_LINEAR.
            STBI_NO_HDR:     suppress implementation of .hdr reader format
            STBI_NO_LINEAR:  suppress high-dynamic-range light-linear float API

      - You can suppress implementation of any of the decoders to reduce
        your code footprint by #defining one or more of the following
        symbols before creating the implementation.

            STBI_NO_JPEG
            STBI_NO_PNG
            STBI_NO_BMP
            STBI_NO_PSD
            STBI_NO_TGA
            STBI_NO_GIF
            STBI_NO_HDR
            STBI_NO_PIC
            STBI_NO_PNM   (.ppm and .pgm)

      - You can request *only* certain decoders and suppress all other ones
        (this will be more forward-compatible, as addition of new decoders
        doesn't require you to disable them explicitly):

            STBI_ONLY_JPEG
            STBI_ONLY_PNG
            STBI_ONLY_BMP
            STBI_ONLY_PSD
            STBI_ONLY_TGA
            STBI_ONLY_GIF
            STBI_ONLY_HDR
            STBI_ONLY_PIC
            STBI_ONLY_PNM   (.ppm and .pgm)

         Note that you can define multiples of these, and you will get all
         of them ("only x" and "only y" is interpreted to mean "only x&y").

       - If you use STBI_NO_PNG (or _ONLY_ without PNG), and you still
         want the zlib decoder to be available, #define STBI_SUPPORT_ZLIB

      - Compilation of all SIMD code can be suppressed with
            #define STBI_NO_SIMD
        It should not be necessary to disable SIMD unless you have issues
        compiling (e.g. using an x86 compiler which doesn't support SSE
        intrinsics or that doesn't support the method used to detect
        SSE2 support at run-time), and even those can be reported as
        bugs so I can refine the built-in compile-time checking to be
        smarter.

      - The old STBI_SIMD system which allowed installing a user-defined
        IDCT etc. has been removed. If you need this, don't upgrade. My
        assumption is that almost nobody was doing this, and those who
        were will find the built-in SIMD more satisfactory anyway.

      - RGB values computed for JPEG images are slightly different from
        previous versions of stb_image. (This is due to using less
        integer precision in SIMD.) The C code has been adjusted so
        that the same RGB values will be computed regardless of whether
        SIMD support is available, so your app should always produce
        consistent results. But these results are slightly different from
        previous versions. (Specifically, about 3% of available YCbCr values
        will compute different RGB results from pre-1.49 versions by +-1;
        most of the deviating values are one smaller in the G channel.)

      - If you must produce consistent results with previous versions of
        stb_image, #define STBI_JPEG_OLD and you will get the same results
        you used to; however, you will not get the SIMD speedups for
        the YCbCr-to-RGB conversion step (although you should still see
        significant JPEG speedup from the other changes).

        Please note that STBI_JPEG_OLD is a temporary feature; it will be
        removed in future versions of the library. It is only intended for
        near-term back-compatibility use.


   Latest revision history:
      2.05  (2015-04-19) fix bug in progressive JPEG handling, fix warning
      2.04  (2015-04-15) try to re-enable SIMD on MinGW 64-bit
      2.03  (2015-04-12) additional corruption checking
                         stbi_set_flip_vertically_on_load
                         fix NEON support; fix mingw support
      2.02  (2015-01-19) fix incorrect assert, fix warning
      2.01  (2015-01-17) fix various warnings
      2.00b (2014-12-25) fix STBI_MALLOC in progressive JPEG
      2.00  (2014-12-25) optimize JPEG, including x86 SSE2 & ARM NEON SIMD
                         progressive JPEG
                         PGM/PPM support
                         STBI_MALLOC,STBI_REALLOC,STBI_FREE
                         STBI_NO_*, STBI_ONLY_*
                         GIF bugfix
      1.48  (2014-12-14) fix incorrectly-named assert()
      1.47  (2014-12-14) 1/2/4-bit PNG support (both grayscale and paletted)
                         optimize PNG
                         fix bug in interlaced PNG with user-specified channel count

   See end of file for full revision history.


 ============================    Contributors    =========================

 Image formats                                Bug fixes & warning fixes
    Sean Barrett (jpeg, png, bmp)                Marc LeBlanc
    Nicolas Schulz (hdr, psd)                    Christpher Lloyd
    Jonathan Dummer (tga)                        Dave Moore
    Jean-Marc Lienher (gif)                      Won Chun
    Tom Seddon (pic)                             the Horde3D community
    Thatcher Ulrich (psd)                        Janez Zemva
    Ken Miller (pgm, ppm)                        Jonathan Blow
                                                 Laurent Gomila
                                                 Aruelien Pocheville
 Extensions, features                            Ryamond Barbiero
    Jetro Lauha (stbi_info)                      David Woo
    Martin "SpartanJ" Golini (stbi_info)         Martin Golini
    James "moose2000" Brown (iPhone PNG)         Roy Eltham
    Ben "Disch" Wenger (io callbacks)            Luke Graham
    Omar Cornut (1/2/4-bit PNG)                  Thomas Ruf
    Nicolas Guillemot (vertical flip)            John Bartholomew
                                                 Ken Hamada
 Optimizations & bugfixes                        Cort Stratton
    Fabian "ryg" Giesen                          Blazej Dariusz Roszkowski
    Arseny Kapoulkine                            Thibault Reuille
                                                 Paul Du Bois
                                                 Guillaume George
  If your name should be here but                Jerry Jansson
  isn't, let Sean know.                          Hayaki Saito
                                                 Johan Duparc
                                                 Ronny Chevalier
                                                 Michal Cichon
                                                 Tero Hanninen
                                                 Sergio Gonzalez
                                                 Cass Everitt
                                                 Engin Manap
                                                 Martins Mozeiko
                                                 Joseph Thomson
                                                 Phil Jordan

License:
   This software is in the public domain. Where that dedication is not
   recognized, you are granted a perpetual, irrevocable license to copy
   and modify this file however you want.

*/

#ifndef STBI_INCLUDE_STB_IMAGE_H
#define STBI_INCLUDE_STB_IMAGE_H

// DOCUMENTATION
//
// Limitations:
//    - no 16-bit-per-channel PNG
//    - no 12-bit-per-channel JPEG
//    - no JPEGs with arithmetic coding
//    - no 1-bit BMP
//    - GIF always returns *comp=4
//
// Basic usage (see HDR discussion below for HDR usage):
//    int x,y,n;
//    unsigned char *data = stbi_load(filename, &x, &y, &n, 0);
//    // ... process data if not nullptr ...
//    // ... x = width, y = height, n = # 8-bit components per pixel ...
//    // ... replace '0' with '1'..'4' to force that many components per pixel
//    // ... but 'n' will always be the number that it would have been if you said 0
//    stbi_image_free(data)
//
// Standard parameters:
//    int *x       -- outputs image width in pixels
//    int *y       -- outputs image height in pixels
//    int *comp    -- outputs # of image components in image file
//    int req_comp -- if non-zero, # of image components requested in result
//
// The return value from an image loader is an 'unsigned char *' which points
// to the pixel data, or nullptr on an allocation failure or if the image is
// corrupt or invalid. The pixel data consists of *y scanlines of *x pixels,
// with each pixel consisting of N interleaved 8-bit components; the first
// pixel pointed to is top-left-most in the image. There is no padding between
// image scanlines or between pixels, regardless of format. The number of
// components N is 'req_comp' if req_comp is non-zero, or *comp otherwise.
// If req_comp is non-zero, *comp has the number of components that _would_
// have been output otherwise. E.g. if you set req_comp to 4, you will always
// get RGBA output, but you can check *comp to see if it's trivially opaque
// because e.g. there were only 3 channels in the source image.
//
// An output image with N components has the following components interleaved
// in this order in each pixel:
//
//     N=#comp     components
//       1           grey
//       2           grey, alpha
//       3           red, green, blue
//       4           red, green, blue, alpha
//
// If image loading fails for any reason, the return value will be nullptr,
// and *x, *y, *comp will be unchanged. The function stbi_failure_reason()
// can be queried for an extremely brief, end-user unfriendly explanation
// of why the load failed. Define STBI_NO_FAILURE_STRINGS to avoid
// compiling these strings at all, and STBI_FAILURE_USERMSG to get slightly
// more user-friendly ones.
//
// Paletted PNG, BMP, GIF, and PIC images are automatically depalettized.
//
// ===========================================================================
//
// Philosophy
//
// stb libraries are designed with the following priorities:
//
//    1. easy to use
//    2. easy to maintain
//    3. good performance
//
// Sometimes I let "good performance" creep up in priority over "easy to maintain",
// and for best performance I may provide less-easy-to-use APIs that give higher
// performance, in addition to the easy to use ones. Nevertheless, it's important
// to keep in mind that from the standpoint of you, a client of this library,
// all you care about is #1 and #3, and stb libraries do not emphasize #3 above all.
//
// Some secondary priorities arise directly from the first two, some of which
// make more explicit reasons why performance can't be emphasized.
//
//    - Portable ("ease of use")
//    - Small footprint ("easy to maintain")
//    - No dependencies ("ease of use")
//
// ===========================================================================
//
// I/O callbacks
//
// I/O callbacks allow you to read from arbitrary sources, like packaged
// files or some other source. Data read from callbacks are processed
// through a small internal buffer (currently 128 bytes) to try to reduce
// overhead.
//
// The three functions you must define are "read" (reads some bytes of data),
// "skip" (skips some bytes of data), "eof" (reports if the stream is at the end).
//
// ===========================================================================
//
// SIMD support
//
// The JPEG decoder will try to automatically use SIMD kernels on x86 when
// supported by the compiler. For ARM Neon support, you must explicitly
// request it.
//
// (The old do-it-yourself SIMD API is no longer supported in the current
// code.)
//
// On x86, SSE2 will automatically be used when available based on a run-time
// test; if not, the generic C versions are used as a fall-back. On ARM targets,
// the typical path is to have separate builds for NEON and non-NEON devices
// (at least this is true for iOS and Android). Therefore, the NEON support is
// toggled by a build flag: define STBI_NEON to get NEON loops.
//
// The output of the JPEG decoder is slightly different from versions where
// SIMD support was introduced (that is, for versions before 1.49). The
// difference is only +-1 in the 8-bit RGB channels, and only on a small
// fraction of pixels. You can force the pre-1.49 behavior by defining
// STBI_JPEG_OLD, but this will disable some of the SIMD decoding path
// and hence cost some performance.
//
// If for some reason you do not want to use any of SIMD code, or if
// you have issues compiling it, you can disable it entirely by
// defining STBI_NO_SIMD.
//
// ===========================================================================
//
// HDR image support   (disable by defining STBI_NO_HDR)
//
// stb_image now supports loading HDR images in general, and currently
// the Radiance .HDR file format, although the support is provided
// generically. You can still load any file through the existing interface;
// if you attempt to load an HDR file, it will be automatically remapped to
// LDR, assuming gamma 2.2 and an arbitrary scale factor defaulting to 1;
// both of these constants can be reconfigured through this interface:
//
//     stbi_hdr_to_ldr_gamma(2.2f);
//     stbi_hdr_to_ldr_scale(1.0f);
//
// (note, do not use _inverse_ constants; stbi_image will invert them
// appropriately).
//
// Additionally, there is a new, parallel interface for loading files as
// (linear) floats to preserve the full dynamic range:
//
//    float *data = stbi_loadf(filename, &x, &y, &n, 0);
//
// If you load LDR images through this interface, those images will
// be promoted to floating point values, run through the inverse of
// constants corresponding to the above:
//
//     stbi_ldr_to_hdr_scale(1.0f);
//     stbi_ldr_to_hdr_gamma(2.2f);
//
// Finally, given a filename (or an open file or memory block--see header
// file for details) containing image data, you can query for the "most
// appropriate" interface to use (that is, whether the image is HDR or
// not), using:
//
//     stbi_is_hdr(char *filename);
//
// ===========================================================================
//
// iPhone PNG support:
//
// By default we convert iphone-formatted PNGs back to RGB, even though
// they are internally encoded differently. You can disable this conversion
// by by calling stbi_convert_iphone_png_to_rgb(0), in which case
// you will always just get the native iphone "format" through (which
// is BGR stored in RGB).
//
// Call stbi_set_unpremultiply_on_load(1) as well to force a divide per
// pixel to remove any premultiplied alpha *only* if the image file explicitly
// says there's premultiplied data (currently only happens in iPhone images,
// and only if iPhone convert-to-rgb processing is on).
//


#ifndef STBI_NO_STDIO
#include <stdio.h>
#endif // STBI_NO_STDIO

#define STBI_VERSION 1

enum
{
  STBI_default = 0, // only used for req_comp

  STBI_grey = 1,
  STBI_grey_alpha = 2,
  STBI_rgb = 3,
  STBI_rgb_alpha = 4
};

typedef unsigned char stbi_uc;

#ifdef __cplusplus
extern "C" {
#endif

#ifdef STB_IMAGE_STATIC
#define STBIDEF static
#else
#define STBIDEF extern
#endif

  //////////////////////////////////////////////////////////////////////////////
  //
  // PRIMARY API - works on images of any type
  //

  //
  // load image by filename, open file, or memory buffer
  //

  typedef struct
  {
    int      (*read)  (void* user, char* data, int size);   // fill 'data' with 'size' bytes.  return number of bytes actually read
    void     (*skip)  (void* user, int n);                 // skip the next 'n' bytes, or 'unget' the last -n bytes if negative
    int      (*eof)   (void* user);                       // returns nonzero if we are at end of file/data
  } stbi_io_callbacks;

  STBIDEF stbi_uc* stbi_load(char              const* filename, int* x, int* y, int* comp, int req_comp);
  STBIDEF stbi_uc* stbi_load_from_memory(stbi_uc           const* buffer, int len, int* x, int* y, int* comp, int req_comp);
  STBIDEF stbi_uc* stbi_load_from_callbacks(stbi_io_callbacks const* clbk, void* user, int* x, int* y, int* comp, int req_comp);

#ifndef STBI_NO_STDIO
  STBIDEF stbi_uc* stbi_load_from_file(FILE* f, int* x, int* y, int* comp, int req_comp);
  // for stbi_load_from_file, file pointer is left pointing immediately after image
#endif

#ifndef STBI_NO_LINEAR
  STBIDEF float* stbi_loadf(char const* filename, int* x, int* y, int* comp, int req_comp);
  STBIDEF float* stbi_loadf_from_memory(stbi_uc const* buffer, int len, int* x, int* y, int* comp, int req_comp);
  STBIDEF float* stbi_loadf_from_callbacks(stbi_io_callbacks const* clbk, void* user, int* x, int* y, int* comp, int req_comp);

#ifndef STBI_NO_STDIO
  STBIDEF float* stbi_loadf_from_file(FILE* f, int* x, int* y, int* comp, int req_comp);
#endif
#endif

#ifndef STBI_NO_HDR
  STBIDEF void   stbi_hdr_to_ldr_gamma(float gamma);
  STBIDEF void   stbi_hdr_to_ldr_scale(float scale);
#endif

#ifndef STBI_NO_LINEAR
  STBIDEF void   stbi_ldr_to_hdr_gamma(float gamma);
  STBIDEF void   stbi_ldr_to_hdr_scale(float scale);
#endif // STBI_NO_HDR

  // stbi_is_hdr is always defined, but always returns false if STBI_NO_HDR
  STBIDEF int    stbi_is_hdr_from_callbacks(stbi_io_callbacks const* clbk, void* user);
  STBIDEF int    stbi_is_hdr_from_memory(stbi_uc const* buffer, int len);
#ifndef STBI_NO_STDIO
  STBIDEF int      stbi_is_hdr(char const* filename);
  STBIDEF int      stbi_is_hdr_from_file(FILE* f);
#endif // STBI_NO_STDIO


  // get a VERY brief reason for failure
  // NOT THREADSAFE
  STBIDEF const char* stbi_failure_reason(void);

  // free the loaded image -- this is just free()
  STBIDEF void     stbi_image_free(void* retval_from_stbi_load);

  // get image dimensions & components without fully decoding
  STBIDEF int      stbi_info_from_memory(stbi_uc const* buffer, int len, int* x, int* y, int* comp);
  STBIDEF int      stbi_info_from_callbacks(stbi_io_callbacks const* clbk, void* user, int* x, int* y, int* comp);

#ifndef STBI_NO_STDIO
  STBIDEF int      stbi_info(char const* filename, int* x, int* y, int* comp);
  STBIDEF int      stbi_info_from_file(FILE* f, int* x, int* y, int* comp);

#endif



  // for image formats that explicitly notate that they have premultiplied alpha,
  // we just return the colors as stored in the file. set this flag to force
  // unpremultiplication. results are undefined if the unpremultiply overflow.
  STBIDEF void stbi_set_unpremultiply_on_load(int flag_true_if_should_unpremultiply);

  // indicate whether we should process iphone images back to canonical format,
  // or just pass them through "as-is"
  STBIDEF void stbi_convert_iphone_png_to_rgb(int flag_true_if_should_convert);

  // flip the image vertically, so the first pixel in the output array is the bottom left
  STBIDEF void stbi_set_flip_vertically_on_load(int flag_true_if_should_flip);

  // ZLIB client - used by PNG, available for other purposes

  STBIDEF char* stbi_zlib_decode_malloc_guesssize(const char* buffer, int len, int initial_size, int* outlen);
  STBIDEF char* stbi_zlib_decode_malloc_guesssize_headerflag(const char* buffer, int len, int initial_size, int* outlen, int parse_header);
  STBIDEF char* stbi_zlib_decode_malloc(const char* buffer, int len, int* outlen);
  STBIDEF int   stbi_zlib_decode_buffer(char* obuffer, int olen, const char* ibuffer, int ilen);

  STBIDEF char* stbi_zlib_decode_noheader_malloc(const char* buffer, int len, int* outlen);
  STBIDEF int   stbi_zlib_decode_noheader_buffer(char* obuffer, int olen, const char* ibuffer, int ilen);


#define STBI_NO_PKM
#define STBI_NO_DDS
#define STBI_NO_PVR
#define STBI_NO_EXT

#ifdef __cplusplus
}
#endif

//
//
////   end header file   /////////////////////////////////////////////////////
#endif // STBI_INCLUDE_STB_IMAGE_H

#ifdef STB_IMAGE_IMPLEMENTATION

#if defined(STBI_ONLY_JPEG) || defined(STBI_ONLY_PNG) || defined(STBI_ONLY_BMP) \
  || defined(STBI_ONLY_TGA) || defined(STBI_ONLY_GIF) || defined(STBI_ONLY_PSD) \
  || defined(STBI_ONLY_HDR) || defined(STBI_ONLY_PIC) || defined(STBI_ONLY_PNM) \
  || defined(STBI_ONLY_ZLIB)
#ifndef STBI_ONLY_JPEG
#define STBI_NO_JPEG
#endif
#ifndef STBI_ONLY_PNG
#define STBI_NO_PNG
#endif
#ifndef STBI_ONLY_BMP
#define STBI_NO_BMP
#endif
#ifndef STBI_ONLY_PSD
#define STBI_NO_PSD
#endif
#ifndef STBI_ONLY_TGA
#define STBI_NO_TGA
#endif
#ifndef STBI_ONLY_GIF
#define STBI_NO_GIF
#endif
#ifndef STBI_ONLY_HDR
#define STBI_NO_HDR
#endif
#ifndef STBI_ONLY_PIC
#define STBI_NO_PIC
#endif
#ifndef STBI_ONLY_PNM
#define STBI_NO_PNM
#endif
#endif

#if defined(STBI_NO_PNG) && !defined(STBI_SUPPORT_ZLIB) && !defined(STBI_NO_ZLIB)
#define STBI_NO_ZLIB
#endif




#if !defined(STBI_NO_LINEAR) || !defined(STBI_NO_HDR)
#include <math.h>  // ldexp
#endif

#ifndef STBI_NO_STDIO
#include <stdio.h>
#endif

#ifndef STBI_ASSERT
#include <assert.h>
#define STBI_ASSERT(x) assert(x)
#endif


#ifndef _MSC_VER
#ifdef __cplusplus
#define stbi_inline inline
#else
#define stbi_inline
#endif
#else
#define stbi_inline __forceinline
#endif


#ifdef _MSC_VER
typedef unsigned short stbi__uint16;
typedef   signed short stbi__int16;
typedef unsigned int   stbi__uint32;
typedef   signed int   stbi__int32;
#else
#include <stdint.h>
typedef uint16_t stbi__uint16;
typedef int16_t  stbi__int16;
typedef uint32_t stbi__uint32;
typedef int32_t  stbi__int32;
#endif

// should produce compiler error if size is wrong
typedef unsigned char validate_uint32[sizeof(stbi__uint32) == 4 ? 1 : -1];

#ifdef _MSC_VER
#define STBI_NOTUSED(v)  (void)(v)
#else
#define STBI_NOTUSED(v)  (void)sizeof(v)
#endif

#ifdef _MSC_VER
#define STBI_HAS_LROTL
#endif

#ifdef STBI_HAS_LROTL
#define stbi_lrot(x,y)  _lrotl(x,y)
#else
#define stbi_lrot(x,y)  (((x) << (y)) | ((x) >> (32 - (y))))
#endif

#if defined(STBI_MALLOC) && defined(STBI_FREE) && defined(STBI_REALLOC)
// ok
#elif !defined(STBI_MALLOC) && !defined(STBI_FREE) && !defined(STBI_REALLOC)
// ok
#else
#error "Must define all or none of STBI_MALLOC, STBI_FREE, and STBI_REALLOC."
#endif

#ifndef STBI_MALLOC
#define STBI_MALLOC(sz)    malloc(sz)
#define STBI_REALLOC(p,sz) realloc(p,sz)
#define STBI_FREE(p)       free(p)
#endif

// x86/x64 detection
#if defined(__x86_64__) || defined(_M_X64)
#define STBI__X64_TARGET
#elif defined(__i386) || defined(_M_IX86)
#define STBI__X86_TARGET
#endif

#if defined(__GNUC__) && (defined(STBI__X86_TARGET) || defined(STBI__X64_TARGET)) && !defined(__SSE2__) && !defined(STBI_NO_SIMD)
// NOTE: not clear do we actually need this for the 64-bit path?
// gcc doesn't support sse2 intrinsics unless you compile with -msse2,
// (but compiling with -msse2 allows the compiler to use SSE2 everywhere;
// this is just broken and gcc are jerks for not fixing it properly
// http://www.virtualdub.org/blog/pivot/entry.php?id=363 )
#define STBI_NO_SIMD
#endif

#if defined(__MINGW32__) && !defined(__x86_64__) && !defined(STBI_NO_SIMD)
#define STBI_MINGW_ENABLE_SSE2
#define STBI_FORCE_STACK_ALIGN __attribute__((force_align_arg_pointer))
#else
#define STBI_FORCE_STACK_ALIGN
#endif

#if defined(__MINGW32__) && defined(STBI__X86_TARGET) && !defined(STBI_MINGW_ENABLE_SSE2) && !defined(STBI_NO_SIMD)
// Note that __MINGW32__ doesn't actually mean 32-bit, so we have to avoid STBI__X64_TARGET
//
// 32-bit MinGW wants ESP to be 16-byte aligned, but this is not in the
// Windows ABI and VC++ as well as Windows DLLs don't maintain that invariant.
// As a result, enabling SSE2 on 32-bit MinGW is dangerous when not
// simultaneously enabling "-mstackrealign".
//
// See https://github.com/nothings/stb/issues/81 for more information.
//
// So default to no SSE2 on 32-bit MinGW. If you've read this far and added
// -mstackrealign to your build settings, feel free to #define STBI_MINGW_ENABLE_SSE2.
#define STBI_NO_SIMD
#endif

#if !defined(STBI_NO_SIMD) && defined(STBI__X86_TARGET)
#define STBI_SSE2
#include <emmintrin.h>

#ifdef _MSC_VER

#if _MSC_VER >= 1400  // not VC6
#include <intrin.h> // __cpuid
static int stbi__cpuid3(void)
{
  int info[4];
  __cpuid(info, 1);
  return info[3];
}
#else
static int stbi__cpuid3(void)
{
  int res;
  __asm {
    mov  eax, 1
    cpuid
    mov  res, edx
  }
  return res;
}
#endif

#define STBI_SIMD_ALIGN(type, name) __declspec(align(16)) type name

static int stbi__sse2_available()
{
  int info3 = stbi__cpuid3();
  return ((info3 >> 26) & 1) != 0;
}
#else // assume GCC-style if not VC++
#define STBI_SIMD_ALIGN(type, name) type name __attribute__((aligned(16)))

static int stbi__sse2_available()
{
#if defined(__GNUC__) && (__GNUC__ * 100 + __GNUC_MINOR__) >= 408 // GCC 4.8 or later
  // GCC 4.8+ has a nice way to do this
  return __builtin_cpu_supports("sse2");
#else
  // portable way to do this, preferably without using GCC inline ASM?
  // just bail for now.
  return 0;
#endif
}
#endif
#endif

// ARM NEON
#if defined(STBI_NO_SIMD) && defined(STBI_NEON)
#undef STBI_NEON
#endif

#ifdef STBI_NEON
#include <arm_neon.h>
// assume GCC or Clang on ARM targets
#define STBI_SIMD_ALIGN(type, name) type name __attribute__((aligned(16)))
#endif

#ifndef STBI_SIMD_ALIGN
#define STBI_SIMD_ALIGN(type, name) type name
#endif

///////////////////////////////////////////////
//
//  stbi__context struct and start_xxx functions

// stbi__context structure is our basic context used by all images, so it
// contains all the IO context, plus some basic image information
typedef struct
{
  stbi__uint32 img_x, img_y;
  int img_n, img_out_n;

  stbi_io_callbacks io;
  void* io_user_data;

  int read_from_callbacks;
  int buflen;
  stbi_uc buffer_start[128];

  stbi_uc* img_buffer, * img_buffer_end;
  stbi_uc* img_buffer_original;
} stbi__context;


static void stbi__refill_buffer(stbi__context* s);

// initialize a memory-decode context
static void stbi__start_mem(stbi__context* s, stbi_uc const* buffer, int len)
{
  s->io.read = nullptr;
  s->read_from_callbacks = 0;
  s->img_buffer = s->img_buffer_original = (stbi_uc*)buffer;
  s->img_buffer_end = (stbi_uc*)buffer + len;
}

// initialize a callback-based context
static void stbi__start_callbacks(stbi__context* s, stbi_io_callbacks* c, void* user)
{
  s->io = *c;
  s->io_user_data = user;
  s->buflen = sizeof(s->buffer_start);
  s->read_from_callbacks = 1;
  s->img_buffer_original = s->buffer_start;
  stbi__refill_buffer(s);
}

#ifndef STBI_NO_STDIO

static int stbi__stdio_read(void* user, char* data, int size)
{
  return (int)fread(data, 1, size, (FILE*)user);
}

static void stbi__stdio_skip(void* user, int n)
{
  fseek((FILE*)user, n, SEEK_CUR);
}

static int stbi__stdio_eof(void* user)
{
  return feof((FILE*)user);
}

static stbi_io_callbacks stbi__stdio_callbacks =
{
   stbi__stdio_read,
   stbi__stdio_skip,
   stbi__stdio_eof,
};

static void stbi__start_file(stbi__context* s, FILE* f)
{
  stbi__start_callbacks(s, &stbi__stdio_callbacks, (void*)f);
}

//static void stop_file(stbi__context *s) { }

#endif // !STBI_NO_STDIO

static void stbi__rewind(stbi__context* s)
{
  // conceptually rewind SHOULD rewind to the beginning of the stream,
  // but we just rewind to the beginning of the initial buffer, because
  // we only use it after doing 'test', which only ever looks at at most 92 bytes
  s->img_buffer = s->img_buffer_original;
}

#ifndef STBI_NO_JPEG
static int      stbi__jpeg_test(stbi__context* s);
static stbi_uc* stbi__jpeg_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__jpeg_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_PNG
static int      stbi__png_test(stbi__context* s);
static stbi_uc* stbi__png_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__png_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_BMP
static int      stbi__bmp_test(stbi__context* s);
static stbi_uc* stbi__bmp_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__bmp_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_TGA
static int      stbi__tga_test(stbi__context* s);
static stbi_uc* stbi__tga_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__tga_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_PSD
static int      stbi__psd_test(stbi__context* s);
static stbi_uc* stbi__psd_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__psd_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_HDR
static int      stbi__hdr_test(stbi__context* s);
static float* stbi__hdr_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__hdr_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_PIC
static int      stbi__pic_test(stbi__context* s);
static stbi_uc* stbi__pic_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__pic_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_GIF
static int      stbi__gif_test(stbi__context* s);
static stbi_uc* stbi__gif_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__gif_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_PNM
static int      stbi__pnm_test(stbi__context* s);
static stbi_uc* stbi__pnm_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__pnm_info(stbi__context* s, int* x, int* y, int* comp);
#endif

#ifndef STBI_NO_PNM
static int      stbi__pnm_test(stbi__context* s);
static stbi_uc* stbi__pnm_load(stbi__context* s, int* x, int* y, int* comp, int req_comp);
static int      stbi__pnm_info(stbi__context* s, int* x, int* y, int* comp);
#endif

// this is not threadsafe
static const char* stbi__g_failure_reason;

STBIDEF const char* stbi_failure_reason(void)
{
  return stbi__g_failure_reason;
}

static int stbi__err(const char* str)
{
  stbi__g_failure_reason = str;
  return 0;
}

static void* stbi__malloc(size_t size)
{
  return STBI_MALLOC(size);
}

// stbi__err - error
// stbi__errpf - error returning pointer to float
// stbi__errpuc - error returning pointer to unsigned char

#ifdef STBI_NO_FAILURE_STRINGS
#define stbi__err(x,y)  0
#elif defined(STBI_FAILURE_USERMSG)
#define stbi__err(x,y)  stbi__err(y)
#else
#define stbi__err(x,y)  stbi__err(x)
#endif

#define stbi__errpf(x,y)   ((float *) (stbi__err(x,y)?nullptr:nullptr))
#define stbi__errpuc(x,y)  ((unsigned char *) (stbi__err(x,y)?nullptr:nullptr))

STBIDEF void stbi_image_free(void* retval_from_stbi_load)
{
  STBI_FREE(retval_from_stbi_load);
}

#ifndef STBI_NO_LINEAR
static float* stbi__ldr_to_hdr(stbi_uc* data, int x, int y, int comp);
#endif

#ifndef STBI_NO_HDR
static stbi_uc* stbi__hdr_to_ldr(float* data, int x, int y, int comp);
#endif

static int stbi__vertically_flip_on_load = 0;

STBIDEF void stbi_set_flip_vertically_on_load(int flag_true_if_should_flip)
{
  stbi__vertically_flip_on_load = flag_true_if_should_flip;
}

static unsigned char* stbi__load_main(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
#ifndef STBI_NO_JPEG
  if (stbi__jpeg_test(s)) return stbi__jpeg_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PNG
  if (stbi__png_test(s))  return stbi__png_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_BMP
  if (stbi__bmp_test(s))  return stbi__bmp_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_GIF
  if (stbi__gif_test(s))  return stbi__gif_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PSD
  if (stbi__psd_test(s))  return stbi__psd_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PIC
  if (stbi__pic_test(s))  return stbi__pic_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PNM
  if (stbi__pnm_test(s))  return stbi__pnm_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_DDS
  if (stbi__dds_test(s))  return stbi__dds_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PVR
  if (stbi__pvr_test(s))  return stbi__pvr_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_PKM
  if (stbi__pkm_test(s))  return stbi__pkm_load(s, x, y, comp, req_comp);
#endif
#ifndef STBI_NO_HDR
  if (stbi__hdr_test(s)) {
    float* hdr = stbi__hdr_load(s, x, y, comp, req_comp);
    return stbi__hdr_to_ldr(hdr, *x, *y, req_comp ? req_comp : *comp);
  }
#endif

#ifndef STBI_NO_TGA
  // test tga last because it's a crappy test!
  if (stbi__tga_test(s))
    return stbi__tga_load(s, x, y, comp, req_comp);
#endif

  return stbi__errpuc("unknown image type", "Image not of any known type, or corrupt");
}

static unsigned char* stbi__load_flip(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  unsigned char* result = stbi__load_main(s, x, y, comp, req_comp);

  if (stbi__vertically_flip_on_load && result != nullptr) {
    int w = *x, h = *y;
    int depth = req_comp ? req_comp : *comp;
    int row, col, z;
    stbi_uc temp;

    // @OPTIMIZE: use a bigger temp buffer and memcpy multiple pixels at once
    for (row = 0; row < (h >> 1); row++) {
      for (col = 0; col < w; col++) {
        for (z = 0; z < depth; z++) {
          temp = result[(row * w + col) * depth + z];
          result[(row * w + col) * depth + z] = result[((h - row - 1) * w + col) * depth + z];
          result[((h - row - 1) * w + col) * depth + z] = temp;
        }
      }
    }
  }

  return result;
}

static void stbi__float_postprocess(float* result, int* x, int* y, int* comp, int req_comp)
{
  if (stbi__vertically_flip_on_load && result != nullptr) {
    int w = *x, h = *y;
    int depth = req_comp ? req_comp : *comp;
    int row, col, z;
    float temp;

    // @OPTIMIZE: use a bigger temp buffer and memcpy multiple pixels at once
    for (row = 0; row < (h >> 1); row++) {
      for (col = 0; col < w; col++) {
        for (z = 0; z < depth; z++) {
          temp = result[(row * w + col) * depth + z];
          result[(row * w + col) * depth + z] = result[((h - row - 1) * w + col) * depth + z];
          result[((h - row - 1) * w + col) * depth + z] = temp;
        }
      }
    }
  }
}


#ifndef STBI_NO_STDIO

static FILE* stbi__fopen(char const* filename, char const* mode)
{
  FILE* f;
#if defined(_MSC_VER) && _MSC_VER >= 1400
  if (0 != fopen_s(&f, filename, mode))
    f = 0;
#else
  f = fopen(filename, mode);
#endif
  return f;
}


STBIDEF stbi_uc* stbi_load(char const* filename, int* x, int* y, int* comp, int req_comp)
{
  FILE* f = stbi__fopen(filename, "rb");
  unsigned char* result;
  if (!f) return stbi__errpuc("can't fopen", "Unable to open file");
  result = stbi_load_from_file(f, x, y, comp, req_comp);
  fclose(f);
  return result;
}

STBIDEF stbi_uc* stbi_load_from_file(FILE* f, int* x, int* y, int* comp, int req_comp)
{
  unsigned char* result;
  stbi__context s;
  stbi__start_file(&s, f);
  result = stbi__load_flip(&s, x, y, comp, req_comp);
  if (result) {
    // need to 'unget' all the characters in the IO buffer
    fseek(f, -(int)(s.img_buffer_end - s.img_buffer), SEEK_CUR);
  }
  return result;
}
#endif //!STBI_NO_STDIO

STBIDEF stbi_uc* stbi_load_from_memory(stbi_uc const* buffer, int len, int* x, int* y, int* comp, int req_comp)
{
  stbi__context s;
  stbi__start_mem(&s, buffer, len);
  return stbi__load_flip(&s, x, y, comp, req_comp);
}

STBIDEF stbi_uc* stbi_load_from_callbacks(stbi_io_callbacks const* clbk, void* user, int* x, int* y, int* comp, int req_comp)
{
  stbi__context s;
  stbi__start_callbacks(&s, (stbi_io_callbacks*)clbk, user);
  return stbi__load_flip(&s, x, y, comp, req_comp);
}

#ifndef STBI_NO_LINEAR
static float* stbi__loadf_main(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  unsigned char* data;
#ifndef STBI_NO_HDR
  if (stbi__hdr_test(s)) {
    float* hdr_data = stbi__hdr_load(s, x, y, comp, req_comp);
    if (hdr_data)
      stbi__float_postprocess(hdr_data, x, y, comp, req_comp);
    return hdr_data;
  }
#endif
  data = stbi__load_flip(s, x, y, comp, req_comp);
  if (data)
    return stbi__ldr_to_hdr(data, *x, *y, req_comp ? req_comp : *comp);
  return stbi__errpf("unknown image type", "Image not of any known type, or corrupt");
}

STBIDEF float* stbi_loadf_from_memory(stbi_uc const* buffer, int len, int* x, int* y, int* comp, int req_comp)
{
  stbi__context s;
  stbi__start_mem(&s, buffer, len);
  return stbi__loadf_main(&s, x, y, comp, req_comp);
}

STBIDEF float* stbi_loadf_from_callbacks(stbi_io_callbacks const* clbk, void* user, int* x, int* y, int* comp, int req_comp)
{
  stbi__context s;
  stbi__start_callbacks(&s, (stbi_io_callbacks*)clbk, user);
  return stbi__loadf_main(&s, x, y, comp, req_comp);
}

#ifndef STBI_NO_STDIO
STBIDEF float* stbi_loadf(char const* filename, int* x, int* y, int* comp, int req_comp)
{
  float* result;
  FILE* f = stbi__fopen(filename, "rb");
  if (!f) return stbi__errpf("can't fopen", "Unable to open file");
  result = stbi_loadf_from_file(f, x, y, comp, req_comp);
  fclose(f);
  return result;
}

STBIDEF float* stbi_loadf_from_file(FILE* f, int* x, int* y, int* comp, int req_comp)
{
  stbi__context s;
  stbi__start_file(&s, f);
  return stbi__loadf_main(&s, x, y, comp, req_comp);
}
#endif // !STBI_NO_STDIO

#endif // !STBI_NO_LINEAR

// these is-hdr-or-not is defined independent of whether STBI_NO_LINEAR is
// defined, for API simplicity; if STBI_NO_LINEAR is defined, it always
// reports false!

STBIDEF int stbi_is_hdr_from_memory(stbi_uc const* buffer, int len)
{
#ifndef STBI_NO_HDR
  stbi__context s;
  stbi__start_mem(&s, buffer, len);
  return stbi__hdr_test(&s);
#else
  STBI_NOTUSED(buffer);
  STBI_NOTUSED(len);
  return 0;
#endif
}

#ifndef STBI_NO_STDIO
STBIDEF int      stbi_is_hdr(char const* filename)
{
  FILE* f = stbi__fopen(filename, "rb");
  int result = 0;
  if (f) {
    result = stbi_is_hdr_from_file(f);
    fclose(f);
  }
  return result;
}

STBIDEF int      stbi_is_hdr_from_file(FILE* f)
{
#ifndef STBI_NO_HDR
  stbi__context s;
  stbi__start_file(&s, f);
  return stbi__hdr_test(&s);
#else
  return 0;
#endif
}
#endif // !STBI_NO_STDIO

STBIDEF int      stbi_is_hdr_from_callbacks(stbi_io_callbacks const* clbk, void* user)
{
#ifndef STBI_NO_HDR
  stbi__context s;
  stbi__start_callbacks(&s, (stbi_io_callbacks*)clbk, user);
  return stbi__hdr_test(&s);
#else
  return 0;
#endif
}

static float stbi__h2l_gamma_i = 1.0f / 2.2f, stbi__h2l_scale_i = 1.0f;
static float stbi__l2h_gamma = 2.2f, stbi__l2h_scale = 1.0f;

#ifndef STBI_NO_LINEAR
STBIDEF void   stbi_ldr_to_hdr_gamma(float gamma) { stbi__l2h_gamma = gamma; }
STBIDEF void   stbi_ldr_to_hdr_scale(float scale) { stbi__l2h_scale = scale; }
#endif

STBIDEF void   stbi_hdr_to_ldr_gamma(float gamma) { stbi__h2l_gamma_i = 1 / gamma; }
STBIDEF void   stbi_hdr_to_ldr_scale(float scale) { stbi__h2l_scale_i = 1 / scale; }


//////////////////////////////////////////////////////////////////////////////
//
// Common code used by all image loaders
//

enum
{
  STBI__SCAN_load = 0,
  STBI__SCAN_type,
  STBI__SCAN_header
};

static void stbi__refill_buffer(stbi__context* s)
{
  int n = (s->io.read)(s->io_user_data, (char*)s->buffer_start, s->buflen);
  if (n == 0) {
    // at end of file, treat same as if from memory, but need to handle case
    // where s->img_buffer isn't pointing to safe memory, e.g. 0-byte file
    s->read_from_callbacks = 0;
    s->img_buffer = s->buffer_start;
    s->img_buffer_end = s->buffer_start + 1;
    *s->img_buffer = 0;
  }
  else {
    s->img_buffer = s->buffer_start;
    s->img_buffer_end = s->buffer_start + n;
  }
}

stbi_inline static stbi_uc stbi__get8(stbi__context* s)
{
  if (s->img_buffer < s->img_buffer_end)
    return *s->img_buffer++;
  if (s->read_from_callbacks) {
    stbi__refill_buffer(s);
    return *s->img_buffer++;
  }
  return 0;
}

stbi_inline static int stbi__at_eof(stbi__context* s)
{
  if (s->io.read) {
    if (!(s->io.eof)(s->io_user_data)) return 0;
    // if feof() is true, check if buffer = end
    // special case: we've only got the special 0 character at the end
    if (s->read_from_callbacks == 0) return 1;
  }

  return s->img_buffer >= s->img_buffer_end;
}

static void stbi__skip(stbi__context* s, int n)
{
  if (n < 0) {
    s->img_buffer = s->img_buffer_end;
    return;
  }
  if (s->io.read) {
    int blen = (int)(s->img_buffer_end - s->img_buffer);
    if (blen < n) {
      s->img_buffer = s->img_buffer_end;
      (s->io.skip)(s->io_user_data, n - blen);
      return;
    }
  }
  s->img_buffer += n;
}

static int stbi__getn(stbi__context* s, stbi_uc* buffer, int n)
{
  if (s->io.read) {
    int blen = (int)(s->img_buffer_end - s->img_buffer);
    if (blen < n) {
      int res, count;

      memcpy(buffer, s->img_buffer, blen);

      count = (s->io.read)(s->io_user_data, (char*)buffer + blen, n - blen);
      res = (count == (n - blen));
      s->img_buffer = s->img_buffer_end;
      return res;
    }
  }

  if (s->img_buffer + n <= s->img_buffer_end) {
    memcpy(buffer, s->img_buffer, n);
    s->img_buffer += n;
    return 1;
  }
  else
    return 0;
}

static int stbi__get16be(stbi__context* s)
{
  int z = stbi__get8(s);
  return (z << 8) + stbi__get8(s);
}

static stbi__uint32 stbi__get32be(stbi__context* s)
{
  stbi__uint32 z = stbi__get16be(s);
  return (z << 16) + stbi__get16be(s);
}

static int stbi__get16le(stbi__context* s)
{
  int z = stbi__get8(s);
  return z + (stbi__get8(s) << 8);
}

static stbi__uint32 stbi__get32le(stbi__context* s)
{
  stbi__uint32 z = stbi__get16le(s);
  return z + (stbi__get16le(s) << 16);
}

#define STBI__BYTECAST(x)  ((stbi_uc) ((x) & 255))  // truncate int to byte without warnings


//////////////////////////////////////////////////////////////////////////////
//
//  generic converter from built-in img_n to req_comp
//    individual types do this automatically as much as possible (e.g. jpeg
//    does all cases internally since it needs to colorspace convert anyway,
//    and it never has alpha, so very few cases ). png can automatically
//    interleave an alpha=255 channel, but falls back to this for other cases
//
//  assume data buffer is malloced, so malloc a new one and free that one
//  only failure mode is malloc failing

static stbi_uc stbi__compute_y(int r, int g, int b)
{
  return (stbi_uc)(((r * 77) + (g * 150) + (29 * b)) >> 8);
}

static unsigned char* stbi__convert_format(unsigned char* data, int img_n, int req_comp, unsigned int x, unsigned int y)
{
  int i, j;
  unsigned char* good;

  if (req_comp == img_n) return data;
  STBI_ASSERT(req_comp >= 1 && req_comp <= 4);

  good = (unsigned char*)stbi__malloc(req_comp * x * y);
  if (good == nullptr) {
    STBI_FREE(data);
    return stbi__errpuc("outofmem", "Out of memory");
  }

  for (j = 0; j < (int)y; ++j) {
    unsigned char* src = data + j * x * img_n;
    unsigned char* dest = good + j * x * req_comp;

#define COMBO(a,b)  ((a)*8+(b))
#define CASE(a,b)   case COMBO(a,b): for(i=x-1; i >= 0; --i, src += a, dest += b)
    // convert source image with img_n components to one with req_comp components;
    // avoid switch per pixel, so use switch per scanline and massive macros
    switch (COMBO(img_n, req_comp)) {
      CASE(1, 2) dest[0] = src[0], dest[1] = 255;
      break;
      CASE(1, 3) dest[0] = dest[1] = dest[2] = src[0];
      break;
      CASE(1, 4) dest[0] = dest[1] = dest[2] = src[0], dest[3] = 255;
      break;
      CASE(2, 1) dest[0] = src[0];
      break;
      CASE(2, 3) dest[0] = dest[1] = dest[2] = src[0];
      break;
      CASE(2, 4) dest[0] = dest[1] = dest[2] = src[0], dest[3] = src[1];
      break;
      CASE(3, 4) dest[0] = src[0], dest[1] = src[1], dest[2] = src[2], dest[3] = 255;
      break;
      CASE(3, 1) dest[0] = stbi__compute_y(src[0], src[1], src[2]);
      break;
      CASE(3, 2) dest[0] = stbi__compute_y(src[0], src[1], src[2]), dest[1] = 255;
      break;
      CASE(4, 1) dest[0] = stbi__compute_y(src[0], src[1], src[2]);
      break;
      CASE(4, 2) dest[0] = stbi__compute_y(src[0], src[1], src[2]), dest[1] = src[3];
      break;
      CASE(4, 3) dest[0] = src[0], dest[1] = src[1], dest[2] = src[2];
      break;
    default: STBI_ASSERT(0);
    }
#undef CASE
  }

  STBI_FREE(data);
  return good;
}

#ifndef STBI_NO_LINEAR
static float* stbi__ldr_to_hdr(stbi_uc* data, int x, int y, int comp)
{
  int i, k, n;
  float* output = (float*)stbi__malloc(x * y * comp * sizeof(float));
  if (output == nullptr) { STBI_FREE(data); return stbi__errpf("outofmem", "Out of memory"); }
  // compute number of non-alpha components
  if (comp & 1) n = comp; else n = comp - 1;
  for (i = 0; i < x * y; ++i) {
    for (k = 0; k < n; ++k) {
      output[i * comp + k] = (float)(pow(data[i * comp + k] / 255.0f, stbi__l2h_gamma) * stbi__l2h_scale);
    }
    if (k < comp) output[i * comp + k] = data[i * comp + k] / 255.0f;
  }
  STBI_FREE(data);
  return output;
}
#endif

#ifndef STBI_NO_HDR
#define stbi__float2int(x)   ((int) (x))
static stbi_uc* stbi__hdr_to_ldr(float* data, int x, int y, int comp)
{
  int i, k, n;
  stbi_uc* output = (stbi_uc*)stbi__malloc(x * y * comp);
  if (output == nullptr) { STBI_FREE(data); return stbi__errpuc("outofmem", "Out of memory"); }
  // compute number of non-alpha components
  if (comp & 1) n = comp; else n = comp - 1;
  for (i = 0; i < x * y; ++i) {
    for (k = 0; k < n; ++k) {
      float z = (float)pow(data[i * comp + k] * stbi__h2l_scale_i, stbi__h2l_gamma_i) * 255 + 0.5f;
      if (z < 0) z = 0;
      if (z > 255) z = 255;
      output[i * comp + k] = (stbi_uc)stbi__float2int(z);
    }
    if (k < comp) {
      float z = data[i * comp + k] * 255 + 0.5f;
      if (z < 0) z = 0;
      if (z > 255) z = 255;
      output[i * comp + k] = (stbi_uc)stbi__float2int(z);
    }
  }
  STBI_FREE(data);
  return output;
}
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  "baseline" JPEG/JFIF decoder
//
//    simple implementation
//      - doesn't support delayed output of y-dimension
//      - simple interface (only one output format: 8-bit interleaved RGB)
//      - doesn't try to recover corrupt jpegs
//      - doesn't allow partial loading, loading multiple at once
//      - still fast on x86 (copying globals into locals doesn't help x86)
//      - allocates lots of intermediate memory (full size of all components)
//        - non-interleaved case requires this anyway
//        - allows good upsampling (see next)
//    high-quality
//      - upsampled channels are bilinearly interpolated, even across blocks
//      - quality integer IDCT derived from IJG's 'slow'
//    performance
//      - fast huffman; reasonable integer IDCT
//      - some SIMD kernels for common paths on targets with SSE2/NEON
//      - uses a lot of intermediate memory, could cache poorly

#ifndef STBI_NO_JPEG

// huffman decoding acceleration
#define FAST_BITS   9  // larger handles more cases; smaller stomps less cache

typedef struct
{
  stbi_uc  fast[1 << FAST_BITS];
  // weirdly, repacking this into AoS is a 10% speed loss, instead of a win
  stbi__uint16 code[256];
  stbi_uc  values[256];
  stbi_uc  size[257];
  unsigned int maxcode[18];
  int    delta[17];   // old 'firstsymbol' - old 'firstcode'
} stbi__huffman;

typedef struct
{
  stbi__context* s;
  stbi__huffman huff_dc[4];
  stbi__huffman huff_ac[4];
  stbi_uc dequant[4][64];
  stbi__int16 fast_ac[4][1 << FAST_BITS];

  // sizes for components, interleaved MCUs
  int img_h_max, img_v_max;
  int img_mcu_x, img_mcu_y;
  int img_mcu_w, img_mcu_h;

  // definition of jpeg image component
  struct
  {
    int id;
    int h, v;
    int tq;
    int hd, ha;
    int dc_pred;

    int x, y, w2, h2;
    stbi_uc* data;
    void* raw_data, * raw_coeff;
    stbi_uc* linebuf;
    short* coeff;   // progressive only
    int      coeff_w, coeff_h; // number of 8x8 coefficient blocks
  } img_comp[4];

  stbi__uint32   code_buffer; // jpeg entropy-coded buffer
  int            code_bits;   // number of valid bits
  unsigned char  marker;      // marker seen while filling entropy buffer
  int            nomore;      // flag if we saw a marker so must stop

  int            progressive;
  int            spec_start;
  int            spec_end;
  int            succ_high;
  int            succ_low;
  int            eob_run;

  int scan_n, order[4];
  int restart_interval, todo;

  // kernels
  void (*idct_block_kernel)(stbi_uc* out, int out_stride, short data[64]);
  void (*YCbCr_to_RGB_kernel)(stbi_uc* out, const stbi_uc* y, const stbi_uc* pcb, const stbi_uc* pcr, int count, int step);
  stbi_uc* (*resample_row_hv_2_kernel)(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs);
} stbi__jpeg;

static int stbi__build_huffman(stbi__huffman* h, int* count)
{
  int i, j, k = 0, code;
  // build size list for each symbol (from JPEG spec)
  for (i = 0; i < 16; ++i)
    for (j = 0; j < count[i]; ++j)
      h->size[k++] = (stbi_uc)(i + 1);
  h->size[k] = 0;

  // compute actual symbols (from jpeg spec)
  code = 0;
  k = 0;
  for (j = 1; j <= 16; ++j) {
    // compute delta to add to code to compute symbol id
    h->delta[j] = k - code;
    if (h->size[k] == j) {
      while (h->size[k] == j)
        h->code[k++] = (stbi__uint16)(code++);
      if (code - 1 >= (1 << j)) return stbi__err("bad code lengths", "Corrupt JPEG");
    }
    // compute largest code + 1 for this size, preshifted as needed later
    h->maxcode[j] = code << (16 - j);
    code <<= 1;
  }
  h->maxcode[j] = 0xffffffff;

  // build non-spec acceleration table; 255 is flag for not-accelerated
  memset(h->fast, 255, 1 << FAST_BITS);
  for (i = 0; i < k; ++i) {
    int s = h->size[i];
    if (s <= FAST_BITS) {
      int c = h->code[i] << (FAST_BITS - s);
      int m = 1 << (FAST_BITS - s);
      for (j = 0; j < m; ++j) {
        h->fast[c + j] = (stbi_uc)i;
      }
    }
  }
  return 1;
}

// build a table that decodes both magnitude and value of small ACs in
// one go.
static void stbi__build_fast_ac(stbi__int16* fast_ac, stbi__huffman* h)
{
  int i;
  for (i = 0; i < (1 << FAST_BITS); ++i) {
    stbi_uc fast = h->fast[i];
    fast_ac[i] = 0;
    if (fast < 255) {
      int rs = h->values[fast];
      int run = (rs >> 4) & 15;
      int magbits = rs & 15;
      int len = h->size[fast];

      if (magbits && len + magbits <= FAST_BITS) {
        // magnitude code followed by receive_extend code
        int k = ((i << len) & ((1 << FAST_BITS) - 1)) >> (FAST_BITS - magbits);
        int m = 1 << (magbits - 1);
        if (k < m) k += (-1 << magbits) + 1;
        // if the result is small enough, we can fit it in fast_ac table
        if (k >= -128 && k <= 127)
          fast_ac[i] = (stbi__int16)((k << 8) + (run << 4) + (len + magbits));
      }
    }
  }
}

static void stbi__grow_buffer_unsafe(stbi__jpeg* j)
{
  do {
    int b = j->nomore ? 0 : stbi__get8(j->s);
    if (b == 0xff) {
      int c = stbi__get8(j->s);
      if (c != 0) {
        j->marker = (unsigned char)c;
        j->nomore = 1;
        return;
      }
    }
    j->code_buffer |= b << (24 - j->code_bits);
    j->code_bits += 8;
  } while (j->code_bits <= 24);
}

// (1 << n) - 1
static stbi__uint32 stbi__bmask[17] = { 0,1,3,7,15,31,63,127,255,511,1023,2047,4095,8191,16383,32767,65535 };

// decode a jpeg huffman value from the bitstream
stbi_inline static int stbi__jpeg_huff_decode(stbi__jpeg* j, stbi__huffman* h)
{
  unsigned int temp;
  int c, k;

  if (j->code_bits < 16) stbi__grow_buffer_unsafe(j);

  // look at the top FAST_BITS and determine what symbol ID it is,
  // if the code is <= FAST_BITS
  c = (j->code_buffer >> (32 - FAST_BITS)) & ((1 << FAST_BITS) - 1);
  k = h->fast[c];
  if (k < 255) {
    int s = h->size[k];
    if (s > j->code_bits)
      return -1;
    j->code_buffer <<= s;
    j->code_bits -= s;
    return h->values[k];
  }

  // naive test is to shift the code_buffer down so k bits are
  // valid, then test against maxcode. To speed this up, we've
  // preshifted maxcode left so that it has (16-k) 0s at the
  // end; in other words, regardless of the number of bits, it
  // wants to be compared against something shifted to have 16;
  // that way we don't need to shift inside the loop.
  temp = j->code_buffer >> 16;
  for (k = FAST_BITS + 1; ; ++k)
    if (temp < h->maxcode[k])
      break;
  if (k == 17) {
    // error! code not found
    j->code_bits -= 16;
    return -1;
  }

  if (k > j->code_bits)
    return -1;

  // convert the huffman code to the symbol id
  c = ((j->code_buffer >> (32 - k)) & stbi__bmask[k]) + h->delta[k];
  STBI_ASSERT((((j->code_buffer) >> (32 - h->size[c])) & stbi__bmask[h->size[c]]) == h->code[c]);

  // convert the id to a symbol
  j->code_bits -= k;
  j->code_buffer <<= k;
  return h->values[c];
}

// bias[n] = (-1<<n) + 1
static int const stbi__jbias[16] = { 0,-1,-3,-7,-15,-31,-63,-127,-255,-511,-1023,-2047,-4095,-8191,-16383,-32767 };

// combined JPEG 'receive' and JPEG 'extend', since baseline
// always extends everything it receives.
stbi_inline static int stbi__extend_receive(stbi__jpeg* j, int n)
{
  unsigned int k;
  int sgn;
  if (j->code_bits < n) stbi__grow_buffer_unsafe(j);

  sgn = (stbi__int32)j->code_buffer >> 31; // sign bit is always in MSB
  k = stbi_lrot(j->code_buffer, n);
  STBI_ASSERT(n >= 0 && n < (int)(sizeof(stbi__bmask) / sizeof(*stbi__bmask)));
  j->code_buffer = k & ~stbi__bmask[n];
  k &= stbi__bmask[n];
  j->code_bits -= n;
  return k + (stbi__jbias[n] & ~sgn);
}

// get some unsigned bits
stbi_inline static int stbi__jpeg_get_bits(stbi__jpeg* j, int n)
{
  unsigned int k;
  if (j->code_bits < n) stbi__grow_buffer_unsafe(j);
  k = stbi_lrot(j->code_buffer, n);
  j->code_buffer = k & ~stbi__bmask[n];
  k &= stbi__bmask[n];
  j->code_bits -= n;
  return k;
}

stbi_inline static int stbi__jpeg_get_bit(stbi__jpeg* j)
{
  unsigned int k;
  if (j->code_bits < 1) stbi__grow_buffer_unsafe(j);
  k = j->code_buffer;
  j->code_buffer <<= 1;
  --j->code_bits;
  return k & 0x80000000;
}

// given a value that's at position X in the zigzag stream,
// where does it appear in the 8x8 matrix coded as row-major?
static stbi_uc stbi__jpeg_dezigzag[64 + 15] =
{
    0,  1,  8, 16,  9,  2,  3, 10,
   17, 24, 32, 25, 18, 11,  4,  5,
   12, 19, 26, 33, 40, 48, 41, 34,
   27, 20, 13,  6,  7, 14, 21, 28,
   35, 42, 49, 56, 57, 50, 43, 36,
   29, 22, 15, 23, 30, 37, 44, 51,
   58, 59, 52, 45, 38, 31, 39, 46,
   53, 60, 61, 54, 47, 55, 62, 63,
   // let corrupt input sample past end
   63, 63, 63, 63, 63, 63, 63, 63,
   63, 63, 63, 63, 63, 63, 63
};

// decode one 64-entry block--
static int stbi__jpeg_decode_block(stbi__jpeg* j, short data[64], stbi__huffman* hdc, stbi__huffman* hac, stbi__int16* fac, int b, stbi_uc* dequant)
{
  int diff, dc, k;
  int t;

  if (j->code_bits < 16) stbi__grow_buffer_unsafe(j);
  t = stbi__jpeg_huff_decode(j, hdc);
  if (t < 0) return stbi__err("bad huffman code", "Corrupt JPEG");

  // 0 all the ac values now so we can do it 32-bits at a time
  memset(data, 0, 64 * sizeof(data[0]));

  diff = t ? stbi__extend_receive(j, t) : 0;
  dc = j->img_comp[b].dc_pred + diff;
  j->img_comp[b].dc_pred = dc;
  data[0] = (short)(dc * dequant[0]);

  // decode AC components, see JPEG spec
  k = 1;
  do {
    unsigned int zig;
    int c, r, s;
    if (j->code_bits < 16) stbi__grow_buffer_unsafe(j);
    c = (j->code_buffer >> (32 - FAST_BITS)) & ((1 << FAST_BITS) - 1);
    r = fac[c];
    if (r) { // fast-AC path
      k += (r >> 4) & 15; // run
      s = r & 15; // combined length
      j->code_buffer <<= s;
      j->code_bits -= s;
      // decode into unzigzag'd location
      zig = stbi__jpeg_dezigzag[k++];
      data[zig] = (short)((r >> 8) * dequant[zig]);
    }
    else {
      int rs = stbi__jpeg_huff_decode(j, hac);
      if (rs < 0) return stbi__err("bad huffman code", "Corrupt JPEG");
      s = rs & 15;
      r = rs >> 4;
      if (s == 0) {
        if (rs != 0xf0) break; // end block
        k += 16;
      }
      else {
        k += r;
        // decode into unzigzag'd location
        zig = stbi__jpeg_dezigzag[k++];
        data[zig] = (short)(stbi__extend_receive(j, s) * dequant[zig]);
      }
    }
  } while (k < 64);
  return 1;
}

static int stbi__jpeg_decode_block_prog_dc(stbi__jpeg* j, short data[64], stbi__huffman* hdc, int b)
{
  int diff, dc;
  int t;
  if (j->spec_end != 0) return stbi__err("can't merge dc and ac", "Corrupt JPEG");

  if (j->code_bits < 16) stbi__grow_buffer_unsafe(j);

  if (j->succ_high == 0) {
    // first scan for DC coefficient, must be first
    memset(data, 0, 64 * sizeof(data[0])); // 0 all the ac values now
    t = stbi__jpeg_huff_decode(j, hdc);
    diff = t ? stbi__extend_receive(j, t) : 0;

    dc = j->img_comp[b].dc_pred + diff;
    j->img_comp[b].dc_pred = dc;
    data[0] = (short)(dc << j->succ_low);
  }
  else {
    // refinement scan for DC coefficient
    if (stbi__jpeg_get_bit(j))
      data[0] += (short)(1 << j->succ_low);
  }
  return 1;
}

// @OPTIMIZE: store non-zigzagged during the decode passes,
// and only de-zigzag when dequantizing
static int stbi__jpeg_decode_block_prog_ac(stbi__jpeg* j, short data[64], stbi__huffman* hac, stbi__int16* fac)
{
  int k;
  if (j->spec_start == 0) return stbi__err("can't merge dc and ac", "Corrupt JPEG");

  if (j->succ_high == 0) {
    int shift = j->succ_low;

    if (j->eob_run) {
      --j->eob_run;
      return 1;
    }

    k = j->spec_start;
    do {
      unsigned int zig;
      int c, r, s;
      if (j->code_bits < 16) stbi__grow_buffer_unsafe(j);
      c = (j->code_buffer >> (32 - FAST_BITS)) & ((1 << FAST_BITS) - 1);
      r = fac[c];
      if (r) { // fast-AC path
        k += (r >> 4) & 15; // run
        s = r & 15; // combined length
        j->code_buffer <<= s;
        j->code_bits -= s;
        zig = stbi__jpeg_dezigzag[k++];
        data[zig] = (short)((r >> 8) << shift);
      }
      else {
        int rs = stbi__jpeg_huff_decode(j, hac);
        if (rs < 0) return stbi__err("bad huffman code", "Corrupt JPEG");
        s = rs & 15;
        r = rs >> 4;
        if (s == 0) {
          if (r < 15) {
            j->eob_run = (1 << r);
            if (r)
              j->eob_run += stbi__jpeg_get_bits(j, r);
            --j->eob_run;
            break;
          }
          k += 16;
        }
        else {
          k += r;
          zig = stbi__jpeg_dezigzag[k++];
          data[zig] = (short)(stbi__extend_receive(j, s) << shift);
        }
      }
    } while (k <= j->spec_end);
  }
  else {
    // refinement scan for these AC coefficients

    short bit = (short)(1 << j->succ_low);

    if (j->eob_run) {
      --j->eob_run;
      for (k = j->spec_start; k <= j->spec_end; ++k) {
        short* p = &data[stbi__jpeg_dezigzag[k]];
        if (*p != 0)
          if (stbi__jpeg_get_bit(j))
            if ((*p & bit) == 0) {
              if (*p > 0)
                *p += bit;
              else
                *p -= bit;
            }
      }
    }
    else {
      k = j->spec_start;
      do {
        int r, s;
        int rs = stbi__jpeg_huff_decode(j, hac); // @OPTIMIZE see if we can use the fast path here, advance-by-r is so slow, eh
        if (rs < 0) return stbi__err("bad huffman code", "Corrupt JPEG");
        s = rs & 15;
        r = rs >> 4;
        if (s == 0) {
          if (r < 15) {
            j->eob_run = (1 << r) - 1;
            if (r)
              j->eob_run += stbi__jpeg_get_bits(j, r);
            r = 64; // force end of block
          }
          else {
            // r=15 s=0 should write 16 0s, so we just do
            // a run of 15 0s and then write s (which is 0),
            // so we don't have to do anything special here
          }
        }
        else {
          if (s != 1) return stbi__err("bad huffman code", "Corrupt JPEG");
          // sign bit
          if (stbi__jpeg_get_bit(j))
            s = bit;
          else
            s = -bit;
        }

        // advance by r
        while (k <= j->spec_end) {
          short* p = &data[stbi__jpeg_dezigzag[k++]];
          if (*p != 0) {
            if (stbi__jpeg_get_bit(j))
              if ((*p & bit) == 0) {
                if (*p > 0)
                  *p += bit;
                else
                  *p -= bit;
              }
          }
          else {
            if (r == 0) {
              *p = (short)s;
              break;
            }
            --r;
          }
        }
      } while (k <= j->spec_end);
    }
  }
  return 1;
}

// take a -128..127 value and stbi__clamp it and convert to 0..255
stbi_inline static stbi_uc stbi__clamp(int x)
{
  // trick to use a single test to catch both cases
  if ((unsigned int)x > 255) {
    if (x < 0) return 0;
    if (x > 255) return 255;
  }
  return (stbi_uc)x;
}

#define stbi__f2f(x)  ((int) (((x) * 4096 + 0.5)))
#define stbi__fsh(x)  ((x) << 12)

// derived from jidctint -- DCT_ISLOW
#define STBI__IDCT_1D(s0,s1,s2,s3,s4,s5,s6,s7) \
   int t0,t1,t2,t3,p1,p2,p3,p4,p5,x0,x1,x2,x3; \
   p2 = s2;                                    \
   p3 = s6;                                    \
   p1 = (p2+p3) * stbi__f2f(0.5411961f);       \
   t2 = p1 + p3*stbi__f2f(-1.847759065f);      \
   t3 = p1 + p2*stbi__f2f( 0.765366865f);      \
   p2 = s0;                                    \
   p3 = s4;                                    \
   t0 = stbi__fsh(p2+p3);                      \
   t1 = stbi__fsh(p2-p3);                      \
   x0 = t0+t3;                                 \
   x3 = t0-t3;                                 \
   x1 = t1+t2;                                 \
   x2 = t1-t2;                                 \
   t0 = s7;                                    \
   t1 = s5;                                    \
   t2 = s3;                                    \
   t3 = s1;                                    \
   p3 = t0+t2;                                 \
   p4 = t1+t3;                                 \
   p1 = t0+t3;                                 \
   p2 = t1+t2;                                 \
   p5 = (p3+p4)*stbi__f2f( 1.175875602f);      \
   t0 = t0*stbi__f2f( 0.298631336f);           \
   t1 = t1*stbi__f2f( 2.053119869f);           \
   t2 = t2*stbi__f2f( 3.072711026f);           \
   t3 = t3*stbi__f2f( 1.501321110f);           \
   p1 = p5 + p1*stbi__f2f(-0.899976223f);      \
   p2 = p5 + p2*stbi__f2f(-2.562915447f);      \
   p3 = p3*stbi__f2f(-1.961570560f);           \
   p4 = p4*stbi__f2f(-0.390180644f);           \
   t3 += p1+p4;                                \
   t2 += p2+p3;                                \
   t1 += p2+p4;                                \
   t0 += p1+p3;

static void stbi__idct_block(stbi_uc* out, int out_stride, short data[64])
{
  int i, val[64], * v = val;
  stbi_uc* o;
  short* d = data;

  // columns
  for (i = 0; i < 8; ++i, ++d, ++v) {
    // if all zeroes, shortcut -- this avoids dequantizing 0s and IDCTing
    if (d[8] == 0 && d[16] == 0 && d[24] == 0 && d[32] == 0
      && d[40] == 0 && d[48] == 0 && d[56] == 0) {
      //    no shortcut                 0     seconds
      //    (1|2|3|4|5|6|7)==0          0     seconds
      //    all separate               -0.047 seconds
      //    1 && 2|3 && 4|5 && 6|7:    -0.047 seconds
      int dcterm = d[0] << 2;
      v[0] = v[8] = v[16] = v[24] = v[32] = v[40] = v[48] = v[56] = dcterm;
    }
    else {
      STBI__IDCT_1D(d[0], d[8], d[16], d[24], d[32], d[40], d[48], d[56])
        // constants scaled things up by 1<<12; let's bring them back
        // down, but keep 2 extra bits of precision
        x0 += 512; x1 += 512; x2 += 512; x3 += 512;
      v[0] = (x0 + t3) >> 10;
      v[56] = (x0 - t3) >> 10;
      v[8] = (x1 + t2) >> 10;
      v[48] = (x1 - t2) >> 10;
      v[16] = (x2 + t1) >> 10;
      v[40] = (x2 - t1) >> 10;
      v[24] = (x3 + t0) >> 10;
      v[32] = (x3 - t0) >> 10;
    }
  }

  for (i = 0, v = val, o = out; i < 8; ++i, v += 8, o += out_stride) {
    // no fast case since the first 1D IDCT spread components out
    STBI__IDCT_1D(v[0], v[1], v[2], v[3], v[4], v[5], v[6], v[7])
      // constants scaled things up by 1<<12, plus we had 1<<2 from first
      // loop, plus horizontal and vertical each scale by sqrt(8) so together
      // we've got an extra 1<<3, so 1<<17 total we need to remove.
      // so we want to round that, which means adding 0.5 * 1<<17,
      // aka 65536. Also, we'll end up with -128 to 127 that we want
      // to encode as 0..255 by adding 128, so we'll add that before the shift
      x0 += 65536 + (128 << 17);
    x1 += 65536 + (128 << 17);
    x2 += 65536 + (128 << 17);
    x3 += 65536 + (128 << 17);
    // tried computing the shifts into temps, or'ing the temps to see
    // if any were out of range, but that was slower
    o[0] = stbi__clamp((x0 + t3) >> 17);
    o[7] = stbi__clamp((x0 - t3) >> 17);
    o[1] = stbi__clamp((x1 + t2) >> 17);
    o[6] = stbi__clamp((x1 - t2) >> 17);
    o[2] = stbi__clamp((x2 + t1) >> 17);
    o[5] = stbi__clamp((x2 - t1) >> 17);
    o[3] = stbi__clamp((x3 + t0) >> 17);
    o[4] = stbi__clamp((x3 - t0) >> 17);
  }
}

#ifdef STBI_SSE2
// sse2 integer IDCT. not the fastest possible implementation but it
// produces bit-identical results to the generic C version so it's
// fully "transparent".
static void stbi__idct_simd(stbi_uc* out, int out_stride, short data[64])
{
  // This is constructed to match our regular (generic) integer IDCT exactly.
  __m128i row0, row1, row2, row3, row4, row5, row6, row7;
  __m128i tmp;

  // dot product constant: even elems=x, odd elems=y
#define dct_const(x,y)  _mm_setr_epi16((x),(y),(x),(y),(x),(y),(x),(y))

// out(0) = c0[even]*x + c0[odd]*y   (c0, x, y 16-bit, out 32-bit)
// out(1) = c1[even]*x + c1[odd]*y
#define dct_rot(out0,out1, x,y,c0,c1) \
      __m128i c0##lo = _mm_unpacklo_epi16((x),(y)); \
      __m128i c0##hi = _mm_unpackhi_epi16((x),(y)); \
      __m128i out0##_l = _mm_madd_epi16(c0##lo, c0); \
      __m128i out0##_h = _mm_madd_epi16(c0##hi, c0); \
      __m128i out1##_l = _mm_madd_epi16(c0##lo, c1); \
      __m128i out1##_h = _mm_madd_epi16(c0##hi, c1)

   // out = in << 12  (in 16-bit, out 32-bit)
#define dct_widen(out, in) \
      __m128i out##_l = _mm_srai_epi32(_mm_unpacklo_epi16(_mm_setzero_si128(), (in)), 4); \
      __m128i out##_h = _mm_srai_epi32(_mm_unpackhi_epi16(_mm_setzero_si128(), (in)), 4)

   // wide add
#define dct_wadd(out, a, b) \
      __m128i out##_l = _mm_add_epi32(a##_l, b##_l); \
      __m128i out##_h = _mm_add_epi32(a##_h, b##_h)

   // wide sub
#define dct_wsub(out, a, b) \
      __m128i out##_l = _mm_sub_epi32(a##_l, b##_l); \
      __m128i out##_h = _mm_sub_epi32(a##_h, b##_h)

   // butterfly a/b, add bias, then shift by "s" and pack
#define dct_bfly32o(out0, out1, a,b,bias,s) \
      { \
         __m128i abiased_l = _mm_add_epi32(a##_l, bias); \
         __m128i abiased_h = _mm_add_epi32(a##_h, bias); \
         dct_wadd(sum, abiased, b); \
         dct_wsub(dif, abiased, b); \
         out0 = _mm_packs_epi32(_mm_srai_epi32(sum_l, s), _mm_srai_epi32(sum_h, s)); \
         out1 = _mm_packs_epi32(_mm_srai_epi32(dif_l, s), _mm_srai_epi32(dif_h, s)); \
      }

   // 8-bit interleave step (for transposes)
#define dct_interleave8(a, b) \
      tmp = a; \
      a = _mm_unpacklo_epi8(a, b); \
      b = _mm_unpackhi_epi8(tmp, b)

   // 16-bit interleave step (for transposes)
#define dct_interleave16(a, b) \
      tmp = a; \
      a = _mm_unpacklo_epi16(a, b); \
      b = _mm_unpackhi_epi16(tmp, b)

#define dct_pass(bias,shift) \
      { \
         /* even part */ \
         dct_rot(t2e,t3e, row2,row6, rot0_0,rot0_1); \
         __m128i sum04 = _mm_add_epi16(row0, row4); \
         __m128i dif04 = _mm_sub_epi16(row0, row4); \
         dct_widen(t0e, sum04); \
         dct_widen(t1e, dif04); \
         dct_wadd(x0, t0e, t3e); \
         dct_wsub(x3, t0e, t3e); \
         dct_wadd(x1, t1e, t2e); \
         dct_wsub(x2, t1e, t2e); \
         /* odd part */ \
         dct_rot(y0o,y2o, row7,row3, rot2_0,rot2_1); \
         dct_rot(y1o,y3o, row5,row1, rot3_0,rot3_1); \
         __m128i sum17 = _mm_add_epi16(row1, row7); \
         __m128i sum35 = _mm_add_epi16(row3, row5); \
         dct_rot(y4o,y5o, sum17,sum35, rot1_0,rot1_1); \
         dct_wadd(x4, y0o, y4o); \
         dct_wadd(x5, y1o, y5o); \
         dct_wadd(x6, y2o, y5o); \
         dct_wadd(x7, y3o, y4o); \
         dct_bfly32o(row0,row7, x0,x7,bias,shift); \
         dct_bfly32o(row1,row6, x1,x6,bias,shift); \
         dct_bfly32o(row2,row5, x2,x5,bias,shift); \
         dct_bfly32o(row3,row4, x3,x4,bias,shift); \
      }

  __m128i rot0_0 = dct_const(stbi__f2f(0.5411961f), stbi__f2f(0.5411961f) + stbi__f2f(-1.847759065f));
  __m128i rot0_1 = dct_const(stbi__f2f(0.5411961f) + stbi__f2f(0.765366865f), stbi__f2f(0.5411961f));
  __m128i rot1_0 = dct_const(stbi__f2f(1.175875602f) + stbi__f2f(-0.899976223f), stbi__f2f(1.175875602f));
  __m128i rot1_1 = dct_const(stbi__f2f(1.175875602f), stbi__f2f(1.175875602f) + stbi__f2f(-2.562915447f));
  __m128i rot2_0 = dct_const(stbi__f2f(-1.961570560f) + stbi__f2f(0.298631336f), stbi__f2f(-1.961570560f));
  __m128i rot2_1 = dct_const(stbi__f2f(-1.961570560f), stbi__f2f(-1.961570560f) + stbi__f2f(3.072711026f));
  __m128i rot3_0 = dct_const(stbi__f2f(-0.390180644f) + stbi__f2f(2.053119869f), stbi__f2f(-0.390180644f));
  __m128i rot3_1 = dct_const(stbi__f2f(-0.390180644f), stbi__f2f(-0.390180644f) + stbi__f2f(1.501321110f));

  // rounding biases in column/row passes, see stbi__idct_block for explanation.
  __m128i bias_0 = _mm_set1_epi32(512);
  __m128i bias_1 = _mm_set1_epi32(65536 + (128 << 17));

  // load
  row0 = _mm_load_si128((const __m128i*) (data + 0 * 8));
  row1 = _mm_load_si128((const __m128i*) (data + 1 * 8));
  row2 = _mm_load_si128((const __m128i*) (data + 2 * 8));
  row3 = _mm_load_si128((const __m128i*) (data + 3 * 8));
  row4 = _mm_load_si128((const __m128i*) (data + 4 * 8));
  row5 = _mm_load_si128((const __m128i*) (data + 5 * 8));
  row6 = _mm_load_si128((const __m128i*) (data + 6 * 8));
  row7 = _mm_load_si128((const __m128i*) (data + 7 * 8));

  // column pass
  dct_pass(bias_0, 10);

  {
    // 16bit 8x8 transpose pass 1
    dct_interleave16(row0, row4);
    dct_interleave16(row1, row5);
    dct_interleave16(row2, row6);
    dct_interleave16(row3, row7);

    // transpose pass 2
    dct_interleave16(row0, row2);
    dct_interleave16(row1, row3);
    dct_interleave16(row4, row6);
    dct_interleave16(row5, row7);

    // transpose pass 3
    dct_interleave16(row0, row1);
    dct_interleave16(row2, row3);
    dct_interleave16(row4, row5);
    dct_interleave16(row6, row7);
  }

  // row pass
  dct_pass(bias_1, 17);

  {
    // pack
    __m128i p0 = _mm_packus_epi16(row0, row1); // a0a1a2a3...a7b0b1b2b3...b7
    __m128i p1 = _mm_packus_epi16(row2, row3);
    __m128i p2 = _mm_packus_epi16(row4, row5);
    __m128i p3 = _mm_packus_epi16(row6, row7);

    // 8bit 8x8 transpose pass 1
    dct_interleave8(p0, p2); // a0e0a1e1...
    dct_interleave8(p1, p3); // c0g0c1g1...

    // transpose pass 2
    dct_interleave8(p0, p1); // a0c0e0g0...
    dct_interleave8(p2, p3); // b0d0f0h0...

    // transpose pass 3
    dct_interleave8(p0, p2); // a0b0c0d0...
    dct_interleave8(p1, p3); // a4b4c4d4...

    // store
    _mm_storel_epi64((__m128i*) out, p0); out += out_stride;
    _mm_storel_epi64((__m128i*) out, _mm_shuffle_epi32(p0, 0x4e)); out += out_stride;
    _mm_storel_epi64((__m128i*) out, p2); out += out_stride;
    _mm_storel_epi64((__m128i*) out, _mm_shuffle_epi32(p2, 0x4e)); out += out_stride;
    _mm_storel_epi64((__m128i*) out, p1); out += out_stride;
    _mm_storel_epi64((__m128i*) out, _mm_shuffle_epi32(p1, 0x4e)); out += out_stride;
    _mm_storel_epi64((__m128i*) out, p3); out += out_stride;
    _mm_storel_epi64((__m128i*) out, _mm_shuffle_epi32(p3, 0x4e));
  }

#undef dct_const
#undef dct_rot
#undef dct_widen
#undef dct_wadd
#undef dct_wsub
#undef dct_bfly32o
#undef dct_interleave8
#undef dct_interleave16
#undef dct_pass
}

#endif // STBI_SSE2

#ifdef STBI_NEON

// NEON integer IDCT. should produce bit-identical
// results to the generic C version.
static void stbi__idct_simd(stbi_uc* out, int out_stride, short data[64])
{
  int16x8_t row0, row1, row2, row3, row4, row5, row6, row7;

  int16x4_t rot0_0 = vdup_n_s16(stbi__f2f(0.5411961f));
  int16x4_t rot0_1 = vdup_n_s16(stbi__f2f(-1.847759065f));
  int16x4_t rot0_2 = vdup_n_s16(stbi__f2f(0.765366865f));
  int16x4_t rot1_0 = vdup_n_s16(stbi__f2f(1.175875602f));
  int16x4_t rot1_1 = vdup_n_s16(stbi__f2f(-0.899976223f));
  int16x4_t rot1_2 = vdup_n_s16(stbi__f2f(-2.562915447f));
  int16x4_t rot2_0 = vdup_n_s16(stbi__f2f(-1.961570560f));
  int16x4_t rot2_1 = vdup_n_s16(stbi__f2f(-0.390180644f));
  int16x4_t rot3_0 = vdup_n_s16(stbi__f2f(0.298631336f));
  int16x4_t rot3_1 = vdup_n_s16(stbi__f2f(2.053119869f));
  int16x4_t rot3_2 = vdup_n_s16(stbi__f2f(3.072711026f));
  int16x4_t rot3_3 = vdup_n_s16(stbi__f2f(1.501321110f));

#define dct_long_mul(out, inq, coeff) \
   int32x4_t out##_l = vmull_s16(vget_low_s16(inq), coeff); \
   int32x4_t out##_h = vmull_s16(vget_high_s16(inq), coeff)

#define dct_long_mac(out, acc, inq, coeff) \
   int32x4_t out##_l = vmlal_s16(acc##_l, vget_low_s16(inq), coeff); \
   int32x4_t out##_h = vmlal_s16(acc##_h, vget_high_s16(inq), coeff)

#define dct_widen(out, inq) \
   int32x4_t out##_l = vshll_n_s16(vget_low_s16(inq), 12); \
   int32x4_t out##_h = vshll_n_s16(vget_high_s16(inq), 12)

  // wide add
#define dct_wadd(out, a, b) \
   int32x4_t out##_l = vaddq_s32(a##_l, b##_l); \
   int32x4_t out##_h = vaddq_s32(a##_h, b##_h)

// wide sub
#define dct_wsub(out, a, b) \
   int32x4_t out##_l = vsubq_s32(a##_l, b##_l); \
   int32x4_t out##_h = vsubq_s32(a##_h, b##_h)

// butterfly a/b, then shift using "shiftop" by "s" and pack
#define dct_bfly32o(out0,out1, a,b,shiftop,s) \
   { \
      dct_wadd(sum, a, b); \
      dct_wsub(dif, a, b); \
      out0 = vcombine_s16(shiftop(sum_l, s), shiftop(sum_h, s)); \
      out1 = vcombine_s16(shiftop(dif_l, s), shiftop(dif_h, s)); \
   }

#define dct_pass(shiftop, shift) \
   { \
      /* even part */ \
      int16x8_t sum26 = vaddq_s16(row2, row6); \
      dct_long_mul(p1e, sum26, rot0_0); \
      dct_long_mac(t2e, p1e, row6, rot0_1); \
      dct_long_mac(t3e, p1e, row2, rot0_2); \
      int16x8_t sum04 = vaddq_s16(row0, row4); \
      int16x8_t dif04 = vsubq_s16(row0, row4); \
      dct_widen(t0e, sum04); \
      dct_widen(t1e, dif04); \
      dct_wadd(x0, t0e, t3e); \
      dct_wsub(x3, t0e, t3e); \
      dct_wadd(x1, t1e, t2e); \
      dct_wsub(x2, t1e, t2e); \
      /* odd part */ \
      int16x8_t sum15 = vaddq_s16(row1, row5); \
      int16x8_t sum17 = vaddq_s16(row1, row7); \
      int16x8_t sum35 = vaddq_s16(row3, row5); \
      int16x8_t sum37 = vaddq_s16(row3, row7); \
      int16x8_t sumodd = vaddq_s16(sum17, sum35); \
      dct_long_mul(p5o, sumodd, rot1_0); \
      dct_long_mac(p1o, p5o, sum17, rot1_1); \
      dct_long_mac(p2o, p5o, sum35, rot1_2); \
      dct_long_mul(p3o, sum37, rot2_0); \
      dct_long_mul(p4o, sum15, rot2_1); \
      dct_wadd(sump13o, p1o, p3o); \
      dct_wadd(sump24o, p2o, p4o); \
      dct_wadd(sump23o, p2o, p3o); \
      dct_wadd(sump14o, p1o, p4o); \
      dct_long_mac(x4, sump13o, row7, rot3_0); \
      dct_long_mac(x5, sump24o, row5, rot3_1); \
      dct_long_mac(x6, sump23o, row3, rot3_2); \
      dct_long_mac(x7, sump14o, row1, rot3_3); \
      dct_bfly32o(row0,row7, x0,x7,shiftop,shift); \
      dct_bfly32o(row1,row6, x1,x6,shiftop,shift); \
      dct_bfly32o(row2,row5, x2,x5,shiftop,shift); \
      dct_bfly32o(row3,row4, x3,x4,shiftop,shift); \
   }

   // load
  row0 = vld1q_s16(data + 0 * 8);
  row1 = vld1q_s16(data + 1 * 8);
  row2 = vld1q_s16(data + 2 * 8);
  row3 = vld1q_s16(data + 3 * 8);
  row4 = vld1q_s16(data + 4 * 8);
  row5 = vld1q_s16(data + 5 * 8);
  row6 = vld1q_s16(data + 6 * 8);
  row7 = vld1q_s16(data + 7 * 8);

  // add DC bias
  row0 = vaddq_s16(row0, vsetq_lane_s16(1024, vdupq_n_s16(0), 0));

  // column pass
  dct_pass(vrshrn_n_s32, 10);

  // 16bit 8x8 transpose
  {
    // these three map to a single VTRN.16, VTRN.32, and VSWP, respectively.
    // whether compilers actually get this is another story, sadly.
#define dct_trn16(x, y) { int16x8x2_t t = vtrnq_s16(x, y); x = t.val[0]; y = t.val[1]; }
#define dct_trn32(x, y) { int32x4x2_t t = vtrnq_s32(vreinterpretq_s32_s16(x), vreinterpretq_s32_s16(y)); x = vreinterpretq_s16_s32(t.val[0]); y = vreinterpretq_s16_s32(t.val[1]); }
#define dct_trn64(x, y) { int16x8_t x0 = x; int16x8_t y0 = y; x = vcombine_s16(vget_low_s16(x0), vget_low_s16(y0)); y = vcombine_s16(vget_high_s16(x0), vget_high_s16(y0)); }

      // pass 1
    dct_trn16(row0, row1); // a0b0a2b2a4b4a6b6
    dct_trn16(row2, row3);
    dct_trn16(row4, row5);
    dct_trn16(row6, row7);

    // pass 2
    dct_trn32(row0, row2); // a0b0c0d0a4b4c4d4
    dct_trn32(row1, row3);
    dct_trn32(row4, row6);
    dct_trn32(row5, row7);

    // pass 3
    dct_trn64(row0, row4); // a0b0c0d0e0f0g0h0
    dct_trn64(row1, row5);
    dct_trn64(row2, row6);
    dct_trn64(row3, row7);

#undef dct_trn16
#undef dct_trn32
#undef dct_trn64
  }

  // row pass
  // vrshrn_n_s32 only supports shifts up to 16, we need
  // 17. so do a non-rounding shift of 16 first then follow
  // up with a rounding shift by 1.
  dct_pass(vshrn_n_s32, 16);

  {
    // pack and round
    uint8x8_t p0 = vqrshrun_n_s16(row0, 1);
    uint8x8_t p1 = vqrshrun_n_s16(row1, 1);
    uint8x8_t p2 = vqrshrun_n_s16(row2, 1);
    uint8x8_t p3 = vqrshrun_n_s16(row3, 1);
    uint8x8_t p4 = vqrshrun_n_s16(row4, 1);
    uint8x8_t p5 = vqrshrun_n_s16(row5, 1);
    uint8x8_t p6 = vqrshrun_n_s16(row6, 1);
    uint8x8_t p7 = vqrshrun_n_s16(row7, 1);

    // again, these can translate into one instruction, but often don't.
#define dct_trn8_8(x, y) { uint8x8x2_t t = vtrn_u8(x, y); x = t.val[0]; y = t.val[1]; }
#define dct_trn8_16(x, y) { uint16x4x2_t t = vtrn_u16(vreinterpret_u16_u8(x), vreinterpret_u16_u8(y)); x = vreinterpret_u8_u16(t.val[0]); y = vreinterpret_u8_u16(t.val[1]); }
#define dct_trn8_32(x, y) { uint32x2x2_t t = vtrn_u32(vreinterpret_u32_u8(x), vreinterpret_u32_u8(y)); x = vreinterpret_u8_u32(t.val[0]); y = vreinterpret_u8_u32(t.val[1]); }

      // sadly can't use interleaved stores here since we only write
      // 8 bytes to each scan line!

      // 8x8 8-bit transpose pass 1
    dct_trn8_8(p0, p1);
    dct_trn8_8(p2, p3);
    dct_trn8_8(p4, p5);
    dct_trn8_8(p6, p7);

    // pass 2
    dct_trn8_16(p0, p2);
    dct_trn8_16(p1, p3);
    dct_trn8_16(p4, p6);
    dct_trn8_16(p5, p7);

    // pass 3
    dct_trn8_32(p0, p4);
    dct_trn8_32(p1, p5);
    dct_trn8_32(p2, p6);
    dct_trn8_32(p3, p7);

    // store
    vst1_u8(out, p0); out += out_stride;
    vst1_u8(out, p1); out += out_stride;
    vst1_u8(out, p2); out += out_stride;
    vst1_u8(out, p3); out += out_stride;
    vst1_u8(out, p4); out += out_stride;
    vst1_u8(out, p5); out += out_stride;
    vst1_u8(out, p6); out += out_stride;
    vst1_u8(out, p7);

#undef dct_trn8_8
#undef dct_trn8_16
#undef dct_trn8_32
  }

#undef dct_long_mul
#undef dct_long_mac
#undef dct_widen
#undef dct_wadd
#undef dct_wsub
#undef dct_bfly32o
#undef dct_pass
}

#endif // STBI_NEON

#define STBI__MARKER_none  0xff
// if there's a pending marker from the entropy stream, return that
// otherwise, fetch from the stream and get a marker. if there's no
// marker, return 0xff, which is never a valid marker value
static stbi_uc stbi__get_marker(stbi__jpeg* j)
{
  stbi_uc x;
  if (j->marker != STBI__MARKER_none) { x = j->marker; j->marker = STBI__MARKER_none; return x; }
  x = stbi__get8(j->s);
  if (x != 0xff) return STBI__MARKER_none;
  while (x == 0xff)
    x = stbi__get8(j->s);
  return x;
}

// in each scan, we'll have scan_n components, and the order
// of the components is specified by order[]
#define STBI__RESTART(x)     ((x) >= 0xd0 && (x) <= 0xd7)

// after a restart interval, stbi__jpeg_reset the entropy decoder and
// the dc prediction
static void stbi__jpeg_reset(stbi__jpeg* j)
{
  j->code_bits = 0;
  j->code_buffer = 0;
  j->nomore = 0;
  j->img_comp[0].dc_pred = j->img_comp[1].dc_pred = j->img_comp[2].dc_pred = 0;
  j->marker = STBI__MARKER_none;
  j->todo = j->restart_interval ? j->restart_interval : 0x7fffffff;
  j->eob_run = 0;
  // no more than 1<<31 MCUs if no restart_interal? that's plenty safe,
  // since we don't even allow 1<<30 pixels
}

static int stbi__parse_entropy_coded_data(stbi__jpeg* z)
{
  stbi__jpeg_reset(z);
  if (!z->progressive) {
    if (z->scan_n == 1) {
      int i, j;
      STBI_SIMD_ALIGN(short, data[64]);
      int n = z->order[0];
      // non-interleaved data, we just need to process one block at a time,
      // in trivial scanline order
      // number of blocks to do just depends on how many actual "pixels" this
      // component has, independent of interleaved MCU blocking and such
      int w = (z->img_comp[n].x + 7) >> 3;
      int h = (z->img_comp[n].y + 7) >> 3;
      for (j = 0; j < h; ++j) {
        for (i = 0; i < w; ++i) {
          int ha = z->img_comp[n].ha;
          if (!stbi__jpeg_decode_block(z, data, z->huff_dc + z->img_comp[n].hd, z->huff_ac + ha, z->fast_ac[ha], n, z->dequant[z->img_comp[n].tq])) return 0;
          z->idct_block_kernel(z->img_comp[n].data + z->img_comp[n].w2 * j * 8 + i * 8, z->img_comp[n].w2, data);
          // every data block is an MCU, so countdown the restart interval
          if (--z->todo <= 0) {
            if (z->code_bits < 24) stbi__grow_buffer_unsafe(z);
            // if it's NOT a restart, then just bail, so we get corrupt data
            // rather than no data
            if (!STBI__RESTART(z->marker)) return 1;
            stbi__jpeg_reset(z);
          }
        }
      }
      return 1;
    }
    else { // interleaved
      int i, j, k, x, y;
      STBI_SIMD_ALIGN(short, data[64]);
      for (j = 0; j < z->img_mcu_y; ++j) {
        for (i = 0; i < z->img_mcu_x; ++i) {
          // scan an interleaved mcu... process scan_n components in order
          for (k = 0; k < z->scan_n; ++k) {
            int n = z->order[k];
            // scan out an mcu's worth of this component; that's just determined
            // by the basic H and V specified for the component
            for (y = 0; y < z->img_comp[n].v; ++y) {
              for (x = 0; x < z->img_comp[n].h; ++x) {
                int x2 = (i * z->img_comp[n].h + x) * 8;
                int y2 = (j * z->img_comp[n].v + y) * 8;
                int ha = z->img_comp[n].ha;
                if (!stbi__jpeg_decode_block(z, data, z->huff_dc + z->img_comp[n].hd, z->huff_ac + ha, z->fast_ac[ha], n, z->dequant[z->img_comp[n].tq])) return 0;
                z->idct_block_kernel(z->img_comp[n].data + z->img_comp[n].w2 * y2 + x2, z->img_comp[n].w2, data);
              }
            }
          }
          // after all interleaved components, that's an interleaved MCU,
          // so now count down the restart interval
          if (--z->todo <= 0) {
            if (z->code_bits < 24) stbi__grow_buffer_unsafe(z);
            if (!STBI__RESTART(z->marker)) return 1;
            stbi__jpeg_reset(z);
          }
        }
      }
      return 1;
    }
  }
  else {
    if (z->scan_n == 1) {
      int i, j;
      int n = z->order[0];
      // non-interleaved data, we just need to process one block at a time,
      // in trivial scanline order
      // number of blocks to do just depends on how many actual "pixels" this
      // component has, independent of interleaved MCU blocking and such
      int w = (z->img_comp[n].x + 7) >> 3;
      int h = (z->img_comp[n].y + 7) >> 3;
      for (j = 0; j < h; ++j) {
        for (i = 0; i < w; ++i) {
          short* data = z->img_comp[n].coeff + 64 * (i + j * z->img_comp[n].coeff_w);
          if (z->spec_start == 0) {
            if (!stbi__jpeg_decode_block_prog_dc(z, data, &z->huff_dc[z->img_comp[n].hd], n))
              return 0;
          }
          else {
            int ha = z->img_comp[n].ha;
            if (!stbi__jpeg_decode_block_prog_ac(z, data, &z->huff_ac[ha], z->fast_ac[ha]))
              return 0;
          }
          // every data block is an MCU, so countdown the restart interval
          if (--z->todo <= 0) {
            if (z->code_bits < 24) stbi__grow_buffer_unsafe(z);
            if (!STBI__RESTART(z->marker)) return 1;
            stbi__jpeg_reset(z);
          }
        }
      }
      return 1;
    }
    else { // interleaved
      int i, j, k, x, y;
      for (j = 0; j < z->img_mcu_y; ++j) {
        for (i = 0; i < z->img_mcu_x; ++i) {
          // scan an interleaved mcu... process scan_n components in order
          for (k = 0; k < z->scan_n; ++k) {
            int n = z->order[k];
            // scan out an mcu's worth of this component; that's just determined
            // by the basic H and V specified for the component
            for (y = 0; y < z->img_comp[n].v; ++y) {
              for (x = 0; x < z->img_comp[n].h; ++x) {
                int x2 = (i * z->img_comp[n].h + x);
                int y2 = (j * z->img_comp[n].v + y);
                short* data = z->img_comp[n].coeff + 64 * (x2 + y2 * z->img_comp[n].coeff_w);
                if (!stbi__jpeg_decode_block_prog_dc(z, data, &z->huff_dc[z->img_comp[n].hd], n))
                  return 0;
              }
            }
          }
          // after all interleaved components, that's an interleaved MCU,
          // so now count down the restart interval
          if (--z->todo <= 0) {
            if (z->code_bits < 24) stbi__grow_buffer_unsafe(z);
            if (!STBI__RESTART(z->marker)) return 1;
            stbi__jpeg_reset(z);
          }
        }
      }
      return 1;
    }
  }
}

static void stbi__jpeg_dequantize(short* data, stbi_uc* dequant)
{
  int i;
  for (i = 0; i < 64; ++i)
    data[i] *= dequant[i];
}

static void stbi__jpeg_finish(stbi__jpeg* z)
{
  if (z->progressive) {
    // dequantize and idct the data
    int i, j, n;
    for (n = 0; n < z->s->img_n; ++n) {
      int w = (z->img_comp[n].x + 7) >> 3;
      int h = (z->img_comp[n].y + 7) >> 3;
      for (j = 0; j < h; ++j) {
        for (i = 0; i < w; ++i) {
          short* data = z->img_comp[n].coeff + 64 * (i + j * z->img_comp[n].coeff_w);
          stbi__jpeg_dequantize(data, z->dequant[z->img_comp[n].tq]);
          z->idct_block_kernel(z->img_comp[n].data + z->img_comp[n].w2 * j * 8 + i * 8, z->img_comp[n].w2, data);
        }
      }
    }
  }
}

static int stbi__process_marker(stbi__jpeg* z, int m)
{
  int L;
  switch (m) {
  case STBI__MARKER_none: // no marker found
    return stbi__err("expected marker", "Corrupt JPEG");

  case 0xDD: // DRI - specify restart interval
    if (stbi__get16be(z->s) != 4) return stbi__err("bad DRI len", "Corrupt JPEG");
    z->restart_interval = stbi__get16be(z->s);
    return 1;

  case 0xDB: // DQT - define quantization table
    L = stbi__get16be(z->s) - 2;
    while (L > 0) {
      int q = stbi__get8(z->s);
      int p = q >> 4;
      int t = q & 15, i;
      if (p != 0) return stbi__err("bad DQT type", "Corrupt JPEG");
      if (t > 3) return stbi__err("bad DQT table", "Corrupt JPEG");
      for (i = 0; i < 64; ++i)
        z->dequant[t][stbi__jpeg_dezigzag[i]] = stbi__get8(z->s);
      L -= 65;
    }
    return L == 0;

  case 0xC4: // DHT - define huffman table
    L = stbi__get16be(z->s) - 2;
    while (L > 0) {
      stbi_uc* v;
      int sizes[16], i, n = 0;
      int q = stbi__get8(z->s);
      int tc = q >> 4;
      int th = q & 15;
      if (tc > 1 || th > 3) return stbi__err("bad DHT header", "Corrupt JPEG");
      for (i = 0; i < 16; ++i) {
        sizes[i] = stbi__get8(z->s);
        n += sizes[i];
      }
      L -= 17;
      if (tc == 0) {
        if (!stbi__build_huffman(z->huff_dc + th, sizes)) return 0;
        v = z->huff_dc[th].values;
      }
      else {
        if (!stbi__build_huffman(z->huff_ac + th, sizes)) return 0;
        v = z->huff_ac[th].values;
      }
      for (i = 0; i < n; ++i)
        v[i] = stbi__get8(z->s);
      if (tc != 0)
        stbi__build_fast_ac(z->fast_ac[th], z->huff_ac + th);
      L -= n;
    }
    return L == 0;
  }
  // check for comment block or APP blocks
  if ((m >= 0xE0 && m <= 0xEF) || m == 0xFE) {
    stbi__skip(z->s, stbi__get16be(z->s) - 2);
    return 1;
  }
  return 0;
}

// after we see SOS
static int stbi__process_scan_header(stbi__jpeg* z)
{
  int i;
  int Ls = stbi__get16be(z->s);
  z->scan_n = stbi__get8(z->s);
  if (z->scan_n < 1 || z->scan_n > 4 || z->scan_n > (int)z->s->img_n) return stbi__err("bad SOS component count", "Corrupt JPEG");
  if (Ls != 6 + 2 * z->scan_n) return stbi__err("bad SOS len", "Corrupt JPEG");
  for (i = 0; i < z->scan_n; ++i) {
    int id = stbi__get8(z->s), which;
    int q = stbi__get8(z->s);
    for (which = 0; which < z->s->img_n; ++which)
      if (z->img_comp[which].id == id)
        break;
    if (which == z->s->img_n) return 0; // no match
    z->img_comp[which].hd = q >> 4;   if (z->img_comp[which].hd > 3) return stbi__err("bad DC huff", "Corrupt JPEG");
    z->img_comp[which].ha = q & 15;   if (z->img_comp[which].ha > 3) return stbi__err("bad AC huff", "Corrupt JPEG");
    z->order[i] = which;
  }

  {
    int aa;
    z->spec_start = stbi__get8(z->s);
    z->spec_end = stbi__get8(z->s); // should be 63, but might be 0
    aa = stbi__get8(z->s);
    z->succ_high = (aa >> 4);
    z->succ_low = (aa & 15);
    if (z->progressive) {
      if (z->spec_start > 63 || z->spec_end > 63 || z->spec_start > z->spec_end || z->succ_high > 13 || z->succ_low > 13)
        return stbi__err("bad SOS", "Corrupt JPEG");
    }
    else {
      if (z->spec_start != 0) return stbi__err("bad SOS", "Corrupt JPEG");
      if (z->succ_high != 0 || z->succ_low != 0) return stbi__err("bad SOS", "Corrupt JPEG");
      z->spec_end = 63;
    }
  }

  return 1;
}

static int stbi__process_frame_header(stbi__jpeg* z, int scan)
{
  stbi__context* s = z->s;
  int Lf, p, i, q, h_max = 1, v_max = 1, c;
  Lf = stbi__get16be(s);         if (Lf < 11) return stbi__err("bad SOF len", "Corrupt JPEG"); // JPEG
  p = stbi__get8(s);            if (p != 8) return stbi__err("only 8-bit", "JPEG format not supported: 8-bit only"); // JPEG baseline
  s->img_y = stbi__get16be(s);   if (s->img_y == 0) return stbi__err("no header height", "JPEG format not supported: delayed height"); // Legal, but we don't handle it--but neither does IJG
  s->img_x = stbi__get16be(s);   if (s->img_x == 0) return stbi__err("0 width", "Corrupt JPEG"); // JPEG requires
  c = stbi__get8(s);
  if (c != 3 && c != 1) return stbi__err("bad component count", "Corrupt JPEG");    // JFIF requires
  s->img_n = c;
  for (i = 0; i < c; ++i) {
    z->img_comp[i].data = nullptr;
    z->img_comp[i].linebuf = nullptr;
  }

  if (Lf != 8 + 3 * s->img_n) return stbi__err("bad SOF len", "Corrupt JPEG");

  for (i = 0; i < s->img_n; ++i) {
    z->img_comp[i].id = stbi__get8(s);
    if (z->img_comp[i].id != i + 1)   // JFIF requires
      if (z->img_comp[i].id != i)  // some version of jpegtran outputs non-JFIF-compliant files!
        return stbi__err("bad component ID", "Corrupt JPEG");
    q = stbi__get8(s);
    z->img_comp[i].h = (q >> 4);  if (!z->img_comp[i].h || z->img_comp[i].h > 4) return stbi__err("bad H", "Corrupt JPEG");
    z->img_comp[i].v = q & 15;    if (!z->img_comp[i].v || z->img_comp[i].v > 4) return stbi__err("bad V", "Corrupt JPEG");
    z->img_comp[i].tq = stbi__get8(s);  if (z->img_comp[i].tq > 3) return stbi__err("bad TQ", "Corrupt JPEG");
  }

  if (scan != STBI__SCAN_load) return 1;

  if ((1 << 30) / s->img_x / s->img_n < s->img_y) return stbi__err("too large", "Image too large to decode");

  for (i = 0; i < s->img_n; ++i) {
    if (z->img_comp[i].h > h_max) h_max = z->img_comp[i].h;
    if (z->img_comp[i].v > v_max) v_max = z->img_comp[i].v;
  }

  // compute interleaved mcu info
  z->img_h_max = h_max;
  z->img_v_max = v_max;
  z->img_mcu_w = h_max * 8;
  z->img_mcu_h = v_max * 8;
  z->img_mcu_x = (s->img_x + z->img_mcu_w - 1) / z->img_mcu_w;
  z->img_mcu_y = (s->img_y + z->img_mcu_h - 1) / z->img_mcu_h;

  for (i = 0; i < s->img_n; ++i) {
    // number of effective pixels (e.g. for non-interleaved MCU)
    z->img_comp[i].x = (s->img_x * z->img_comp[i].h + h_max - 1) / h_max;
    z->img_comp[i].y = (s->img_y * z->img_comp[i].v + v_max - 1) / v_max;
    // to simplify generation, we'll allocate enough memory to decode
    // the bogus oversized data from using interleaved MCUs and their
    // big blocks (e.g. a 16x16 iMCU on an image of width 33); we won't
    // discard the extra data until colorspace conversion
    z->img_comp[i].w2 = z->img_mcu_x * z->img_comp[i].h * 8;
    z->img_comp[i].h2 = z->img_mcu_y * z->img_comp[i].v * 8;
    z->img_comp[i].raw_data = stbi__malloc(z->img_comp[i].w2 * z->img_comp[i].h2 + 15);

    if (z->img_comp[i].raw_data == nullptr) {
      for (--i; i >= 0; --i) {
        STBI_FREE(z->img_comp[i].raw_data);
        z->img_comp[i].data = nullptr;
      }
      return stbi__err("outofmem", "Out of memory");
    }
    // align blocks for idct using mmx/sse
    z->img_comp[i].data = (stbi_uc*)(((size_t)z->img_comp[i].raw_data + 15) & ~15);
    z->img_comp[i].linebuf = nullptr;
    if (z->progressive) {
      z->img_comp[i].coeff_w = (z->img_comp[i].w2 + 7) >> 3;
      z->img_comp[i].coeff_h = (z->img_comp[i].h2 + 7) >> 3;
      z->img_comp[i].raw_coeff = STBI_MALLOC(z->img_comp[i].coeff_w * z->img_comp[i].coeff_h * 64 * sizeof(short) + 15);
      z->img_comp[i].coeff = (short*)(((size_t)z->img_comp[i].raw_coeff + 15) & ~15);
    }
    else {
      z->img_comp[i].coeff = 0;
      z->img_comp[i].raw_coeff = 0;
    }
  }

  return 1;
}

// use comparisons since in some cases we handle more than one case (e.g. SOF)
#define stbi__DNL(x)         ((x) == 0xdc)
#define stbi__SOI(x)         ((x) == 0xd8)
#define stbi__EOI(x)         ((x) == 0xd9)
#define stbi__SOF(x)         ((x) == 0xc0 || (x) == 0xc1 || (x) == 0xc2)
#define stbi__SOS(x)         ((x) == 0xda)

#define stbi__SOF_progressive(x)   ((x) == 0xc2)

static int stbi__decode_jpeg_header(stbi__jpeg* z, int scan)
{
  int m;
  z->marker = STBI__MARKER_none; // initialize cached marker to empty
  m = stbi__get_marker(z);
  if (!stbi__SOI(m)) return stbi__err("no SOI", "Corrupt JPEG");
  if (scan == STBI__SCAN_type) return 1;
  m = stbi__get_marker(z);
  while (!stbi__SOF(m)) {
    if (!stbi__process_marker(z, m)) return 0;
    m = stbi__get_marker(z);
    while (m == STBI__MARKER_none) {
      // some files have extra padding after their blocks, so ok, we'll scan
      if (stbi__at_eof(z->s)) return stbi__err("no SOF", "Corrupt JPEG");
      m = stbi__get_marker(z);
    }
  }
  z->progressive = stbi__SOF_progressive(m);
  if (!stbi__process_frame_header(z, scan)) return 0;
  return 1;
}

// decode image to YCbCr format
static int stbi__decode_jpeg_image(stbi__jpeg* j)
{
  int m;
  for (m = 0; m < 4; m++) {
    j->img_comp[m].raw_data = nullptr;
    j->img_comp[m].raw_coeff = nullptr;
  }
  j->restart_interval = 0;
  if (!stbi__decode_jpeg_header(j, STBI__SCAN_load)) return 0;
  m = stbi__get_marker(j);
  while (!stbi__EOI(m)) {
    if (stbi__SOS(m)) {
      if (!stbi__process_scan_header(j)) return 0;
      if (!stbi__parse_entropy_coded_data(j)) return 0;
      if (j->marker == STBI__MARKER_none) {
        // handle 0s at the end of image data from IP Kamera 9060
        while (!stbi__at_eof(j->s)) {
          int x = stbi__get8(j->s);
          if (x == 255) {
            j->marker = stbi__get8(j->s);
            break;
          }
          else if (x != 0) {
            return stbi__err("junk before marker", "Corrupt JPEG");
          }
        }
        // if we reach eof without hitting a marker, stbi__get_marker() below will fail and we'll eventually return 0
      }
    }
    else {
      if (!stbi__process_marker(j, m)) return 0;
    }
    m = stbi__get_marker(j);
  }
  if (j->progressive)
    stbi__jpeg_finish(j);
  return 1;
}

// static jfif-centered resampling (across block boundaries)

typedef stbi_uc* (*resample_row_func)(stbi_uc* out, stbi_uc* in0, stbi_uc* in1,
  int w, int hs);

#define stbi__div4(x) ((stbi_uc) ((x) >> 2))

static stbi_uc* resample_row_1(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  STBI_NOTUSED(out);
  STBI_NOTUSED(in_far);
  STBI_NOTUSED(w);
  STBI_NOTUSED(hs);
  return in_near;
}

static stbi_uc* stbi__resample_row_v_2(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  // need to generate two samples vertically for every one in input
  int i;
  STBI_NOTUSED(hs);
  for (i = 0; i < w; ++i)
    out[i] = stbi__div4(3 * in_near[i] + in_far[i] + 2);
  return out;
}

static stbi_uc* stbi__resample_row_h_2(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  // need to generate two samples horizontally for every one in input
  int i;
  stbi_uc* input = in_near;

  if (w == 1) {
    // if only one sample, can't do any interpolation
    out[0] = out[1] = input[0];
    return out;
  }

  out[0] = input[0];
  out[1] = stbi__div4(input[0] * 3 + input[1] + 2);
  for (i = 1; i < w - 1; ++i) {
    int n = 3 * input[i] + 2;
    out[i * 2 + 0] = stbi__div4(n + input[i - 1]);
    out[i * 2 + 1] = stbi__div4(n + input[i + 1]);
  }
  out[i * 2 + 0] = stbi__div4(input[w - 2] * 3 + input[w - 1] + 2);
  out[i * 2 + 1] = input[w - 1];

  STBI_NOTUSED(in_far);
  STBI_NOTUSED(hs);

  return out;
}

#define stbi__div16(x) ((stbi_uc) ((x) >> 4))

static stbi_uc* stbi__resample_row_hv_2(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  // need to generate 2x2 samples for every one in input
  int i, t0, t1;
  if (w == 1) {
    out[0] = out[1] = stbi__div4(3 * in_near[0] + in_far[0] + 2);
    return out;
  }

  t1 = 3 * in_near[0] + in_far[0];
  out[0] = stbi__div4(t1 + 2);
  for (i = 1; i < w; ++i) {
    t0 = t1;
    t1 = 3 * in_near[i] + in_far[i];
    out[i * 2 - 1] = stbi__div16(3 * t0 + t1 + 8);
    out[i * 2] = stbi__div16(3 * t1 + t0 + 8);
  }
  out[w * 2 - 1] = stbi__div4(t1 + 2);

  STBI_NOTUSED(hs);

  return out;
}

#if defined(STBI_SSE2) || defined(STBI_NEON)
static stbi_uc* stbi__resample_row_hv_2_simd(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  // need to generate 2x2 samples for every one in input
  int i = 0, t0, t1;

  if (w == 1) {
    out[0] = out[1] = stbi__div4(3 * in_near[0] + in_far[0] + 2);
    return out;
  }

  t1 = 3 * in_near[0] + in_far[0];
  // process groups of 8 pixels for as long as we can.
  // note we can't handle the last pixel in a row in this loop
  // because we need to handle the filter boundary conditions.
  for (; i < ((w - 1) & ~7); i += 8) {
#if defined(STBI_SSE2)
    // load and perform the vertical filtering pass
    // this uses 3*x + y = 4*x + (y - x)
    __m128i zero = _mm_setzero_si128();
    __m128i farb = _mm_loadl_epi64((__m128i*) (in_far + i));
    __m128i nearb = _mm_loadl_epi64((__m128i*) (in_near + i));
    __m128i farw = _mm_unpacklo_epi8(farb, zero);
    __m128i nearw = _mm_unpacklo_epi8(nearb, zero);
    __m128i diff = _mm_sub_epi16(farw, nearw);
    __m128i nears = _mm_slli_epi16(nearw, 2);
    __m128i curr = _mm_add_epi16(nears, diff); // current row

    // horizontal filter works the same based on shifted vers of current
    // row. "prev" is current row shifted right by 1 pixel; we need to
    // insert the previous pixel value (from t1).
    // "next" is current row shifted left by 1 pixel, with first pixel
    // of next block of 8 pixels added in.
    __m128i prv0 = _mm_slli_si128(curr, 2);
    __m128i nxt0 = _mm_srli_si128(curr, 2);
    __m128i prev = _mm_insert_epi16(prv0, t1, 0);
    __m128i next = _mm_insert_epi16(nxt0, 3 * in_near[i + 8] + in_far[i + 8], 7);

    // horizontal filter, polyphase implementation since it's convenient:
    // even pixels = 3*cur + prev = cur*4 + (prev - cur)
    // odd  pixels = 3*cur + next = cur*4 + (next - cur)
    // note the shared term.
    __m128i bias = _mm_set1_epi16(8);
    __m128i curs = _mm_slli_epi16(curr, 2);
    __m128i prvd = _mm_sub_epi16(prev, curr);
    __m128i nxtd = _mm_sub_epi16(next, curr);
    __m128i curb = _mm_add_epi16(curs, bias);
    __m128i even = _mm_add_epi16(prvd, curb);
    __m128i odd = _mm_add_epi16(nxtd, curb);

    // interleave even and odd pixels, then undo scaling.
    __m128i int0 = _mm_unpacklo_epi16(even, odd);
    __m128i int1 = _mm_unpackhi_epi16(even, odd);
    __m128i de0 = _mm_srli_epi16(int0, 4);
    __m128i de1 = _mm_srli_epi16(int1, 4);

    // pack and write output
    __m128i outv = _mm_packus_epi16(de0, de1);
    _mm_storeu_si128((__m128i*) (out + i * 2), outv);
#elif defined(STBI_NEON)
    // load and perform the vertical filtering pass
    // this uses 3*x + y = 4*x + (y - x)
    uint8x8_t farb = vld1_u8(in_far + i);
    uint8x8_t nearb = vld1_u8(in_near + i);
    int16x8_t diff = vreinterpretq_s16_u16(vsubl_u8(farb, nearb));
    int16x8_t nears = vreinterpretq_s16_u16(vshll_n_u8(nearb, 2));
    int16x8_t curr = vaddq_s16(nears, diff); // current row

    // horizontal filter works the same based on shifted vers of current
    // row. "prev" is current row shifted right by 1 pixel; we need to
    // insert the previous pixel value (from t1).
    // "next" is current row shifted left by 1 pixel, with first pixel
    // of next block of 8 pixels added in.
    int16x8_t prv0 = vextq_s16(curr, curr, 7);
    int16x8_t nxt0 = vextq_s16(curr, curr, 1);
    int16x8_t prev = vsetq_lane_s16(t1, prv0, 0);
    int16x8_t next = vsetq_lane_s16(3 * in_near[i + 8] + in_far[i + 8], nxt0, 7);

    // horizontal filter, polyphase implementation since it's convenient:
    // even pixels = 3*cur + prev = cur*4 + (prev - cur)
    // odd  pixels = 3*cur + next = cur*4 + (next - cur)
    // note the shared term.
    int16x8_t curs = vshlq_n_s16(curr, 2);
    int16x8_t prvd = vsubq_s16(prev, curr);
    int16x8_t nxtd = vsubq_s16(next, curr);
    int16x8_t even = vaddq_s16(curs, prvd);
    int16x8_t odd = vaddq_s16(curs, nxtd);

    // undo scaling and round, then store with even/odd phases interleaved
    uint8x8x2_t o;
    o.val[0] = vqrshrun_n_s16(even, 4);
    o.val[1] = vqrshrun_n_s16(odd, 4);
    vst2_u8(out + i * 2, o);
#endif

    // "previous" value for next iter
    t1 = 3 * in_near[i + 7] + in_far[i + 7];
  }

  t0 = t1;
  t1 = 3 * in_near[i] + in_far[i];
  out[i * 2] = stbi__div16(3 * t1 + t0 + 8);

  for (++i; i < w; ++i) {
    t0 = t1;
    t1 = 3 * in_near[i] + in_far[i];
    out[i * 2 - 1] = stbi__div16(3 * t0 + t1 + 8);
    out[i * 2] = stbi__div16(3 * t1 + t0 + 8);
  }
  out[w * 2 - 1] = stbi__div4(t1 + 2);

  STBI_NOTUSED(hs);

  return out;
}
#endif

static stbi_uc* stbi__resample_row_generic(stbi_uc* out, stbi_uc* in_near, stbi_uc* in_far, int w, int hs)
{
  // resample with nearest-neighbor
  int i, j;
  STBI_NOTUSED(in_far);
  for (i = 0; i < w; ++i)
    for (j = 0; j < hs; ++j)
      out[i * hs + j] = in_near[i];
  return out;
}

#ifdef STBI_JPEG_OLD
// this is the same YCbCr-to-RGB calculation that stb_image has used
// historically before the algorithm changes in 1.49
#define float2fixed(x)  ((int) ((x) * 65536 + 0.5))
static void stbi__YCbCr_to_RGB_row(stbi_uc* out, const stbi_uc* y, const stbi_uc* pcb, const stbi_uc* pcr, int count, int step)
{
  int i;
  for (i = 0; i < count; ++i) {
    int y_fixed = (y[i] << 16) + 32768; // rounding
    int r, g, b;
    int cr = pcr[i] - 128;
    int cb = pcb[i] - 128;
    r = y_fixed + cr * float2fixed(1.40200f);
    g = y_fixed - cr * float2fixed(0.71414f) - cb * float2fixed(0.34414f);
    b = y_fixed + cb * float2fixed(1.77200f);
    r >>= 16;
    g >>= 16;
    b >>= 16;
    if ((unsigned)r > 255) { if (r < 0) r = 0; else r = 255; }
    if ((unsigned)g > 255) { if (g < 0) g = 0; else g = 255; }
    if ((unsigned)b > 255) { if (b < 0) b = 0; else b = 255; }
    out[0] = (stbi_uc)r;
    out[1] = (stbi_uc)g;
    out[2] = (stbi_uc)b;
    out[3] = 255;
    out += step;
  }
}
#else
// this is a reduced-precision calculation of YCbCr-to-RGB introduced
// to make sure the code produces the same results in both SIMD and scalar
#define float2fixed(x)  (((int) ((x) * 4096.0f + 0.5f)) << 8)
static void stbi__YCbCr_to_RGB_row(stbi_uc* out, const stbi_uc* y, const stbi_uc* pcb, const stbi_uc* pcr, int count, int step)
{
  int i;
  for (i = 0; i < count; ++i) {
    int y_fixed = (y[i] << 20) + (1 << 19); // rounding
    int r, g, b;
    int cr = pcr[i] - 128;
    int cb = pcb[i] - 128;
    r = y_fixed + cr * float2fixed(1.40200f);
    g = y_fixed + (cr * -float2fixed(0.71414f)) + ((cb * -float2fixed(0.34414f)) & 0xffff0000);
    b = y_fixed + cb * float2fixed(1.77200f);
    r >>= 20;
    g >>= 20;
    b >>= 20;
    if ((unsigned)r > 255) { if (r < 0) r = 0; else r = 255; }
    if ((unsigned)g > 255) { if (g < 0) g = 0; else g = 255; }
    if ((unsigned)b > 255) { if (b < 0) b = 0; else b = 255; }
    out[0] = (stbi_uc)r;
    out[1] = (stbi_uc)g;
    out[2] = (stbi_uc)b;
    out[3] = 255;
    out += step;
  }
}
#endif

#if defined(STBI_SSE2) || defined(STBI_NEON)
static void stbi__YCbCr_to_RGB_simd(stbi_uc* out, stbi_uc const* y, stbi_uc const* pcb, stbi_uc const* pcr, int count, int step)
{
  int i = 0;

#ifdef STBI_SSE2
  // step == 3 is pretty ugly on the final interleave, and i'm not convinced
  // it's useful in practice (you wouldn't use it for textures, for example).
  // so just accelerate step == 4 case.
  if (step == 4) {
    // this is a fairly straightforward implementation and not super-optimized.
    __m128i signflip = _mm_set1_epi8(-0x80);
    __m128i cr_const0 = _mm_set1_epi16((short)(1.40200f * 4096.0f + 0.5f));
    __m128i cr_const1 = _mm_set1_epi16(-(short)(0.71414f * 4096.0f + 0.5f));
    __m128i cb_const0 = _mm_set1_epi16(-(short)(0.34414f * 4096.0f + 0.5f));
    __m128i cb_const1 = _mm_set1_epi16((short)(1.77200f * 4096.0f + 0.5f));
    __m128i y_bias = _mm_set1_epi8((char)(unsigned char)128);
    __m128i xw = _mm_set1_epi16(255); // alpha channel

    for (; i + 7 < count; i += 8) {
      // load
      __m128i y_bytes = _mm_loadl_epi64((__m128i*) (y + i));
      __m128i cr_bytes = _mm_loadl_epi64((__m128i*) (pcr + i));
      __m128i cb_bytes = _mm_loadl_epi64((__m128i*) (pcb + i));
      __m128i cr_biased = _mm_xor_si128(cr_bytes, signflip); // -128
      __m128i cb_biased = _mm_xor_si128(cb_bytes, signflip); // -128

      // unpack to short (and left-shift cr, cb by 8)
      __m128i yw = _mm_unpacklo_epi8(y_bias, y_bytes);
      __m128i crw = _mm_unpacklo_epi8(_mm_setzero_si128(), cr_biased);
      __m128i cbw = _mm_unpacklo_epi8(_mm_setzero_si128(), cb_biased);

      // color transform
      __m128i yws = _mm_srli_epi16(yw, 4);
      __m128i cr0 = _mm_mulhi_epi16(cr_const0, crw);
      __m128i cb0 = _mm_mulhi_epi16(cb_const0, cbw);
      __m128i cb1 = _mm_mulhi_epi16(cbw, cb_const1);
      __m128i cr1 = _mm_mulhi_epi16(crw, cr_const1);
      __m128i rws = _mm_add_epi16(cr0, yws);
      __m128i gwt = _mm_add_epi16(cb0, yws);
      __m128i bws = _mm_add_epi16(yws, cb1);
      __m128i gws = _mm_add_epi16(gwt, cr1);

      // descale
      __m128i rw = _mm_srai_epi16(rws, 4);
      __m128i bw = _mm_srai_epi16(bws, 4);
      __m128i gw = _mm_srai_epi16(gws, 4);

      // back to byte, set up for transpose
      __m128i brb = _mm_packus_epi16(rw, bw);
      __m128i gxb = _mm_packus_epi16(gw, xw);

      // transpose to interleave channels
      __m128i t0 = _mm_unpacklo_epi8(brb, gxb);
      __m128i t1 = _mm_unpackhi_epi8(brb, gxb);
      __m128i o0 = _mm_unpacklo_epi16(t0, t1);
      __m128i o1 = _mm_unpackhi_epi16(t0, t1);

      // store
      _mm_storeu_si128((__m128i*) (out + 0), o0);
      _mm_storeu_si128((__m128i*) (out + 16), o1);
      out += 32;
    }
  }
#endif

#ifdef STBI_NEON
  // in this version, step=3 support would be easy to add. but is there demand?
  if (step == 4) {
    // this is a fairly straightforward implementation and not super-optimized.
    uint8x8_t signflip = vdup_n_u8(0x80);
    int16x8_t cr_const0 = vdupq_n_s16((short)(1.40200f * 4096.0f + 0.5f));
    int16x8_t cr_const1 = vdupq_n_s16(-(short)(0.71414f * 4096.0f + 0.5f));
    int16x8_t cb_const0 = vdupq_n_s16(-(short)(0.34414f * 4096.0f + 0.5f));
    int16x8_t cb_const1 = vdupq_n_s16((short)(1.77200f * 4096.0f + 0.5f));

    for (; i + 7 < count; i += 8) {
      // load
      uint8x8_t y_bytes = vld1_u8(y + i);
      uint8x8_t cr_bytes = vld1_u8(pcr + i);
      uint8x8_t cb_bytes = vld1_u8(pcb + i);
      int8x8_t cr_biased = vreinterpret_s8_u8(vsub_u8(cr_bytes, signflip));
      int8x8_t cb_biased = vreinterpret_s8_u8(vsub_u8(cb_bytes, signflip));

      // expand to s16
      int16x8_t yws = vreinterpretq_s16_u16(vshll_n_u8(y_bytes, 4));
      int16x8_t crw = vshll_n_s8(cr_biased, 7);
      int16x8_t cbw = vshll_n_s8(cb_biased, 7);

      // color transform
      int16x8_t cr0 = vqdmulhq_s16(crw, cr_const0);
      int16x8_t cb0 = vqdmulhq_s16(cbw, cb_const0);
      int16x8_t cr1 = vqdmulhq_s16(crw, cr_const1);
      int16x8_t cb1 = vqdmulhq_s16(cbw, cb_const1);
      int16x8_t rws = vaddq_s16(yws, cr0);
      int16x8_t gws = vaddq_s16(vaddq_s16(yws, cb0), cr1);
      int16x8_t bws = vaddq_s16(yws, cb1);

      // undo scaling, round, convert to byte
      uint8x8x4_t o;
      o.val[0] = vqrshrun_n_s16(rws, 4);
      o.val[1] = vqrshrun_n_s16(gws, 4);
      o.val[2] = vqrshrun_n_s16(bws, 4);
      o.val[3] = vdup_n_u8(255);

      // store, interleaving r/g/b/a
      vst4_u8(out, o);
      out += 8 * 4;
    }
  }
#endif

  for (; i < count; ++i) {
    int y_fixed = (y[i] << 20) + (1 << 19); // rounding
    int r, g, b;
    int cr = pcr[i] - 128;
    int cb = pcb[i] - 128;
    r = y_fixed + cr * float2fixed(1.40200f);
    g = y_fixed + cr * -float2fixed(0.71414f) + ((cb * -float2fixed(0.34414f)) & 0xffff0000);
    b = y_fixed + cb * float2fixed(1.77200f);
    r >>= 20;
    g >>= 20;
    b >>= 20;
    if ((unsigned)r > 255) { if (r < 0) r = 0; else r = 255; }
    if ((unsigned)g > 255) { if (g < 0) g = 0; else g = 255; }
    if ((unsigned)b > 255) { if (b < 0) b = 0; else b = 255; }
    out[0] = (stbi_uc)r;
    out[1] = (stbi_uc)g;
    out[2] = (stbi_uc)b;
    out[3] = 255;
    out += step;
  }
}
#endif

// set up the kernels
static void stbi__setup_jpeg(stbi__jpeg* j)
{
  j->idct_block_kernel = stbi__idct_block;
  j->YCbCr_to_RGB_kernel = stbi__YCbCr_to_RGB_row;
  j->resample_row_hv_2_kernel = stbi__resample_row_hv_2;

#ifdef STBI_SSE2
  if (stbi__sse2_available()) {
    j->idct_block_kernel = stbi__idct_simd;
#ifndef STBI_JPEG_OLD
    j->YCbCr_to_RGB_kernel = stbi__YCbCr_to_RGB_simd;
#endif
    j->resample_row_hv_2_kernel = stbi__resample_row_hv_2_simd;
  }
#endif

#ifdef STBI_NEON
  j->idct_block_kernel = stbi__idct_simd;
#ifndef STBI_JPEG_OLD
  j->YCbCr_to_RGB_kernel = stbi__YCbCr_to_RGB_simd;
#endif
  j->resample_row_hv_2_kernel = stbi__resample_row_hv_2_simd;
#endif
}

// clean up the temporary component buffers
static void stbi__cleanup_jpeg(stbi__jpeg* j)
{
  int i;
  for (i = 0; i < j->s->img_n; ++i) {
    if (j->img_comp[i].raw_data) {
      STBI_FREE(j->img_comp[i].raw_data);
      j->img_comp[i].raw_data = nullptr;
      j->img_comp[i].data = nullptr;
    }
    if (j->img_comp[i].raw_coeff) {
      STBI_FREE(j->img_comp[i].raw_coeff);
      j->img_comp[i].raw_coeff = 0;
      j->img_comp[i].coeff = 0;
    }
    if (j->img_comp[i].linebuf) {
      STBI_FREE(j->img_comp[i].linebuf);
      j->img_comp[i].linebuf = nullptr;
    }
  }
}

typedef struct
{
  resample_row_func resample;
  stbi_uc* line0, * line1;
  int hs, vs;   // expansion factor in each axis
  int w_lores; // horizontal pixels pre-expansion
  int ystep;   // how far through vertical expansion we are
  int ypos;    // which pre-expansion row we're on
} stbi__resample;

static stbi_uc* load_jpeg_image(stbi__jpeg* z, int* out_x, int* out_y, int* comp, int req_comp)
{
  int n, decode_n;
  z->s->img_n = 0; // make stbi__cleanup_jpeg safe

  // validate req_comp
  if (req_comp < 0 || req_comp > 4) return stbi__errpuc("bad req_comp", "Internal error");

  // load a jpeg image from whichever source, but leave in YCbCr format
  if (!stbi__decode_jpeg_image(z)) { stbi__cleanup_jpeg(z); return nullptr; }

  // determine actual number of components to generate
  n = req_comp ? req_comp : z->s->img_n;

  if (z->s->img_n == 3 && n < 3)
    decode_n = 1;
  else
    decode_n = z->s->img_n;

  // resample and color-convert
  {
    int k;
    unsigned int i, j;
    stbi_uc* output;
    stbi_uc* coutput[4];

    stbi__resample res_comp[4];

    for (k = 0; k < decode_n; ++k) {
      stbi__resample* r = &res_comp[k];

      // allocate line buffer big enough for upsampling off the edges
      // with upsample factor of 4
      z->img_comp[k].linebuf = (stbi_uc*)stbi__malloc(z->s->img_x + 3);
      if (!z->img_comp[k].linebuf) { stbi__cleanup_jpeg(z); return stbi__errpuc("outofmem", "Out of memory"); }

      r->hs = z->img_h_max / z->img_comp[k].h;
      r->vs = z->img_v_max / z->img_comp[k].v;
      r->ystep = r->vs >> 1;
      r->w_lores = (z->s->img_x + r->hs - 1) / r->hs;
      r->ypos = 0;
      r->line0 = r->line1 = z->img_comp[k].data;

      if (r->hs == 1 && r->vs == 1) r->resample = resample_row_1;
      else if (r->hs == 1 && r->vs == 2) r->resample = stbi__resample_row_v_2;
      else if (r->hs == 2 && r->vs == 1) r->resample = stbi__resample_row_h_2;
      else if (r->hs == 2 && r->vs == 2) r->resample = z->resample_row_hv_2_kernel;
      else                               r->resample = stbi__resample_row_generic;
    }

    // can't error after this so, this is safe
    output = (stbi_uc*)stbi__malloc(n * z->s->img_x * z->s->img_y + 1);
    if (!output) { stbi__cleanup_jpeg(z); return stbi__errpuc("outofmem", "Out of memory"); }

    // now go ahead and resample
    for (j = 0; j < z->s->img_y; ++j) {
      stbi_uc* out = output + n * z->s->img_x * j;
      for (k = 0; k < decode_n; ++k) {
        stbi__resample* r = &res_comp[k];
        int y_bot = r->ystep >= (r->vs >> 1);
        coutput[k] = r->resample(z->img_comp[k].linebuf,
          y_bot ? r->line1 : r->line0,
          y_bot ? r->line0 : r->line1,
          r->w_lores, r->hs);
        if (++r->ystep >= r->vs) {
          r->ystep = 0;
          r->line0 = r->line1;
          if (++r->ypos < z->img_comp[k].y)
            r->line1 += z->img_comp[k].w2;
        }
      }
      if (n >= 3) {
        stbi_uc* y = coutput[0];
        if (z->s->img_n == 3) {
          z->YCbCr_to_RGB_kernel(out, y, coutput[1], coutput[2], z->s->img_x, n);
        }
        else
          for (i = 0; i < z->s->img_x; ++i) {
            out[0] = out[1] = out[2] = y[i];
            out[3] = 255; // not used if n==3
            out += n;
          }
      }
      else {
        stbi_uc* y = coutput[0];
        if (n == 1)
          for (i = 0; i < z->s->img_x; ++i) out[i] = y[i];
        else
          for (i = 0; i < z->s->img_x; ++i) *out++ = y[i], *out++ = 255;
      }
    }
    stbi__cleanup_jpeg(z);
    *out_x = z->s->img_x;
    *out_y = z->s->img_y;
    if (comp) *comp = z->s->img_n; // report original components, not output
    return output;
  }
}

static unsigned char* STBI_FORCE_STACK_ALIGN stbi__jpeg_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  stbi__jpeg j;
  j.s = s;
  stbi__setup_jpeg(&j);
  return load_jpeg_image(&j, x, y, comp, req_comp);
}

static int stbi__jpeg_test(stbi__context* s)
{
  int r;
  stbi__jpeg j;
  j.s = s;
  stbi__setup_jpeg(&j);
  r = stbi__decode_jpeg_header(&j, STBI__SCAN_type);
  stbi__rewind(s);
  return r;
}

static int stbi__jpeg_info_raw(stbi__jpeg* j, int* x, int* y, int* comp)
{
  if (!stbi__decode_jpeg_header(j, STBI__SCAN_header)) {
    stbi__rewind(j->s);
    return 0;
  }
  if (x) *x = j->s->img_x;
  if (y) *y = j->s->img_y;
  if (comp) *comp = j->s->img_n;
  return 1;
}

static int stbi__jpeg_info(stbi__context* s, int* x, int* y, int* comp)
{
  stbi__jpeg j;
  j.s = s;
  return stbi__jpeg_info_raw(&j, x, y, comp);
}
#endif

// public domain zlib decode    v0.2  Sean Barrett 2006-11-18
//    simple implementation
//      - all input must be provided in an upfront buffer
//      - all output is written to a single output buffer (can malloc/realloc)
//    performance
//      - fast huffman

#ifndef STBI_NO_ZLIB

// fast-way is faster to check than jpeg huffman, but slow way is slower
#define STBI__ZFAST_BITS  9 // accelerate all cases in default tables
#define STBI__ZFAST_MASK  ((1 << STBI__ZFAST_BITS) - 1)

// zlib-style huffman encoding
// (jpegs packs from left, zlib from right, so can't share code)
typedef struct
{
  stbi__uint16 fast[1 << STBI__ZFAST_BITS];
  stbi__uint16 firstcode[16];
  int maxcode[17];
  stbi__uint16 firstsymbol[16];
  stbi_uc  size[288];
  stbi__uint16 value[288];
} stbi__zhuffman;

stbi_inline static int stbi__bitreverse16(int n)
{
  n = ((n & 0xAAAA) >> 1) | ((n & 0x5555) << 1);
  n = ((n & 0xCCCC) >> 2) | ((n & 0x3333) << 2);
  n = ((n & 0xF0F0) >> 4) | ((n & 0x0F0F) << 4);
  n = ((n & 0xFF00) >> 8) | ((n & 0x00FF) << 8);
  return n;
}

stbi_inline static int stbi__bit_reverse(int v, int bits)
{
  STBI_ASSERT(bits <= 16);
  // to bit reverse n bits, reverse 16 and shift
  // e.g. 11 bits, bit reverse and shift away 5
  return stbi__bitreverse16(v) >> (16 - bits);
}

static int stbi__zbuild_huffman(stbi__zhuffman* z, stbi_uc* sizelist, int num)
{
  int i, k = 0;
  int code, next_code[16], sizes[17];

  // DEFLATE spec for generating codes
  memset(sizes, 0, sizeof(sizes));
  memset(z->fast, 0, sizeof(z->fast));
  for (i = 0; i < num; ++i)
    ++sizes[sizelist[i]];
  sizes[0] = 0;
  for (i = 1; i < 16; ++i)
    if (sizes[i] > (1 << i))
      return stbi__err("bad sizes", "Corrupt PNG");
  code = 0;
  for (i = 1; i < 16; ++i) {
    next_code[i] = code;
    z->firstcode[i] = (stbi__uint16)code;
    z->firstsymbol[i] = (stbi__uint16)k;
    code = (code + sizes[i]);
    if (sizes[i])
      if (code - 1 >= (1 << i)) return stbi__err("bad codelengths", "Corrupt PNG");
    z->maxcode[i] = code << (16 - i); // preshift for inner loop
    code <<= 1;
    k += sizes[i];
  }
  z->maxcode[16] = 0x10000; // sentinel
  for (i = 0; i < num; ++i) {
    int s = sizelist[i];
    if (s) {
      int c = next_code[s] - z->firstcode[s] + z->firstsymbol[s];
      stbi__uint16 fastv = (stbi__uint16)((s << 9) | i);
      z->size[c] = (stbi_uc)s;
      z->value[c] = (stbi__uint16)i;
      if (s <= STBI__ZFAST_BITS) {
        int k2 = stbi__bit_reverse(next_code[s], s);
        while (k2 < (1 << STBI__ZFAST_BITS)) {
          z->fast[k2] = fastv;
          k2 += (1 << s);
        }
      }
      ++next_code[s];
    }
  }
  return 1;
}

// zlib-from-memory implementation for PNG reading
//    because PNG allows splitting the zlib stream arbitrarily,
//    and it's annoying structurally to have PNG call ZLIB call PNG,
//    we require PNG read all the IDATs and combine them into a single
//    memory buffer

typedef struct
{
  stbi_uc* zbuffer, * zbuffer_end;
  int num_bits;
  stbi__uint32 code_buffer;

  char* zout;
  char* zout_start;
  char* zout_end;
  int   z_expandable;

  stbi__zhuffman z_length, z_distance;
} stbi__zbuf;

stbi_inline static stbi_uc stbi__zget8(stbi__zbuf* z)
{
  if (z->zbuffer >= z->zbuffer_end) return 0;
  return *z->zbuffer++;
}

static void stbi__fill_bits(stbi__zbuf* z)
{
  do {
    STBI_ASSERT(z->code_buffer < (1U << z->num_bits));
    z->code_buffer |= stbi__zget8(z) << z->num_bits;
    z->num_bits += 8;
  } while (z->num_bits <= 24);
}

stbi_inline static unsigned int stbi__zreceive(stbi__zbuf* z, int n)
{
  unsigned int k;
  if (z->num_bits < n) stbi__fill_bits(z);
  k = z->code_buffer & ((1 << n) - 1);
  z->code_buffer >>= n;
  z->num_bits -= n;
  return k;
}

static int stbi__zhuffman_decode_slowpath(stbi__zbuf* a, stbi__zhuffman* z)
{
  int b, s, k;
  // not resolved by fast table, so compute it the slow way
  // use jpeg approach, which requires MSbits at top
  k = stbi__bit_reverse(a->code_buffer, 16);
  for (s = STBI__ZFAST_BITS + 1; ; ++s)
    if (k < z->maxcode[s])
      break;
  if (s == 16) return -1; // invalid code!
  // code size is s, so:
  b = (k >> (16 - s)) - z->firstcode[s] + z->firstsymbol[s];
  STBI_ASSERT(z->size[b] == s);
  a->code_buffer >>= s;
  a->num_bits -= s;
  return z->value[b];
}

stbi_inline static int stbi__zhuffman_decode(stbi__zbuf* a, stbi__zhuffman* z)
{
  int b, s;
  if (a->num_bits < 16) stbi__fill_bits(a);
  b = z->fast[a->code_buffer & STBI__ZFAST_MASK];
  if (b) {
    s = b >> 9;
    a->code_buffer >>= s;
    a->num_bits -= s;
    return b & 511;
  }
  return stbi__zhuffman_decode_slowpath(a, z);
}

static int stbi__zexpand(stbi__zbuf* z, char* zout, int n)  // need to make room for n bytes
{
  char* q;
  int cur, limit;
  z->zout = zout;
  if (!z->z_expandable) return stbi__err("output buffer limit", "Corrupt PNG");
  cur = (int)(z->zout - z->zout_start);
  limit = (int)(z->zout_end - z->zout_start);
  while (cur + n > limit)
    limit *= 2;
  q = (char*)STBI_REALLOC(z->zout_start, limit);
  if (q == nullptr) return stbi__err("outofmem", "Out of memory");
  z->zout_start = q;
  z->zout = q + cur;
  z->zout_end = q + limit;
  return 1;
}

static int stbi__zlength_base[31] = {
   3,4,5,6,7,8,9,10,11,13,
   15,17,19,23,27,31,35,43,51,59,
   67,83,99,115,131,163,195,227,258,0,0 };

static int stbi__zlength_extra[31] =
{ 0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0,0,0 };

static int stbi__zdist_base[32] = { 1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,
257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0 };

static int stbi__zdist_extra[32] =
{ 0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13 };

static int stbi__parse_huffman_block(stbi__zbuf* a)
{
  char* zout = a->zout;
  for (;;) {
    int z = stbi__zhuffman_decode(a, &a->z_length);
    if (z < 256) {
      if (z < 0) return stbi__err("bad huffman code", "Corrupt PNG"); // error in huffman codes
      if (zout >= a->zout_end) {
        if (!stbi__zexpand(a, zout, 1)) return 0;
        zout = a->zout;
      }
      *zout++ = (char)z;
    }
    else {
      stbi_uc* p;
      int len, dist;
      if (z == 256) {
        a->zout = zout;
        return 1;
      }
      z -= 257;
      len = stbi__zlength_base[z];
      if (stbi__zlength_extra[z]) len += stbi__zreceive(a, stbi__zlength_extra[z]);
      z = stbi__zhuffman_decode(a, &a->z_distance);
      if (z < 0) return stbi__err("bad huffman code", "Corrupt PNG");
      dist = stbi__zdist_base[z];
      if (stbi__zdist_extra[z]) dist += stbi__zreceive(a, stbi__zdist_extra[z]);
      if (zout - a->zout_start < dist) return stbi__err("bad dist", "Corrupt PNG");
      if (zout + len > a->zout_end) {
        if (!stbi__zexpand(a, zout, len)) return 0;
        zout = a->zout;
      }
      p = (stbi_uc*)(zout - dist);
      if (dist == 1) { // run of one byte; common in images.
        stbi_uc v = *p;
        if (len) { do *zout++ = v; while (--len); }
      }
      else {
        if (len) { do *zout++ = *p++; while (--len); }
      }
    }
  }
}

static int stbi__compute_huffman_codes(stbi__zbuf* a)
{
  static stbi_uc length_dezigzag[19] = { 16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15 };
  stbi__zhuffman z_codelength;
  stbi_uc lencodes[286 + 32 + 137];//padding for maximum single op
  stbi_uc codelength_sizes[19];
  int i, n;

  int hlit = stbi__zreceive(a, 5) + 257;
  int hdist = stbi__zreceive(a, 5) + 1;
  int hclen = stbi__zreceive(a, 4) + 4;

  memset(codelength_sizes, 0, sizeof(codelength_sizes));
  for (i = 0; i < hclen; ++i) {
    int s = stbi__zreceive(a, 3);
    codelength_sizes[length_dezigzag[i]] = (stbi_uc)s;
  }
  if (!stbi__zbuild_huffman(&z_codelength, codelength_sizes, 19)) return 0;

  n = 0;
  while (n < hlit + hdist) {
    int c = stbi__zhuffman_decode(a, &z_codelength);
    if (c < 0 || c >= 19) return stbi__err("bad codelengths", "Corrupt PNG");
    if (c < 16)
      lencodes[n++] = (stbi_uc)c;
    else if (c == 16) {
      c = stbi__zreceive(a, 2) + 3;
      memset(lencodes + n, lencodes[n - 1], c);
      n += c;
    }
    else if (c == 17) {
      c = stbi__zreceive(a, 3) + 3;
      memset(lencodes + n, 0, c);
      n += c;
    }
    else {
      STBI_ASSERT(c == 18);
      c = stbi__zreceive(a, 7) + 11;
      memset(lencodes + n, 0, c);
      n += c;
    }
  }
  if (n != hlit + hdist) return stbi__err("bad codelengths", "Corrupt PNG");
  if (!stbi__zbuild_huffman(&a->z_length, lencodes, hlit)) return 0;
  if (!stbi__zbuild_huffman(&a->z_distance, lencodes + hlit, hdist)) return 0;
  return 1;
}

static int stbi__parse_uncomperssed_block(stbi__zbuf* a)
{
  stbi_uc header[4];
  int len, nlen, k;
  if (a->num_bits & 7)
    stbi__zreceive(a, a->num_bits & 7); // discard
  // drain the bit-packed data into header
  k = 0;
  while (a->num_bits > 0) {
    header[k++] = (stbi_uc)(a->code_buffer & 255); // suppress MSVC run-time check
    a->code_buffer >>= 8;
    a->num_bits -= 8;
  }
  STBI_ASSERT(a->num_bits == 0);
  // now fill header the normal way
  while (k < 4)
    header[k++] = stbi__zget8(a);
  len = header[1] * 256 + header[0];
  nlen = header[3] * 256 + header[2];
  if (nlen != (len ^ 0xffff)) return stbi__err("zlib corrupt", "Corrupt PNG");
  if (a->zbuffer + len > a->zbuffer_end) return stbi__err("read past buffer", "Corrupt PNG");
  if (a->zout + len > a->zout_end)
    if (!stbi__zexpand(a, a->zout, len)) return 0;
  memcpy(a->zout, a->zbuffer, len);
  a->zbuffer += len;
  a->zout += len;
  return 1;
}

static int stbi__parse_zlib_header(stbi__zbuf* a)
{
  int cmf = stbi__zget8(a);
  int cm = cmf & 15;
  /* int cinfo = cmf >> 4; */
  int flg = stbi__zget8(a);
  if ((cmf * 256 + flg) % 31 != 0) return stbi__err("bad zlib header", "Corrupt PNG"); // zlib spec
  if (flg & 32) return stbi__err("no preset dict", "Corrupt PNG"); // preset dictionary not allowed in png
  if (cm != 8) return stbi__err("bad compression", "Corrupt PNG"); // DEFLATE required for png
  // window = 1 << (8 + cinfo)... but who cares, we fully buffer output
  return 1;
}

// @TODO: should statically initialize these for optimal thread safety
static stbi_uc stbi__zdefault_length[288], stbi__zdefault_distance[32];
static void stbi__init_zdefaults(void)
{
  int i;   // use <= to match clearly with spec
  for (i = 0; i <= 143; ++i)     stbi__zdefault_length[i] = 8;
  for (; i <= 255; ++i)     stbi__zdefault_length[i] = 9;
  for (; i <= 279; ++i)     stbi__zdefault_length[i] = 7;
  for (; i <= 287; ++i)     stbi__zdefault_length[i] = 8;

  for (i = 0; i <= 31; ++i)     stbi__zdefault_distance[i] = 5;
}

static int stbi__parse_zlib(stbi__zbuf* a, int parse_header)
{
  int final, type;
  if (parse_header)
    if (!stbi__parse_zlib_header(a)) return 0;
  a->num_bits = 0;
  a->code_buffer = 0;
  do {
    final = stbi__zreceive(a, 1);
    type = stbi__zreceive(a, 2);
    if (type == 0) {
      if (!stbi__parse_uncomperssed_block(a)) return 0;
    }
    else if (type == 3) {
      return 0;
    }
    else {
      if (type == 1) {
        // use fixed code lengths
        if (!stbi__zdefault_distance[31]) stbi__init_zdefaults();
        if (!stbi__zbuild_huffman(&a->z_length, stbi__zdefault_length, 288)) return 0;
        if (!stbi__zbuild_huffman(&a->z_distance, stbi__zdefault_distance, 32)) return 0;
      }
      else {
        if (!stbi__compute_huffman_codes(a)) return 0;
      }
      if (!stbi__parse_huffman_block(a)) return 0;
    }
  } while (!final);
  return 1;
}

static int stbi__do_zlib(stbi__zbuf* a, char* obuf, int olen, int exp, int parse_header)
{
  a->zout_start = obuf;
  a->zout = obuf;
  a->zout_end = obuf + olen;
  a->z_expandable = exp;

  return stbi__parse_zlib(a, parse_header);
}

STBIDEF char* stbi_zlib_decode_malloc_guesssize(const char* buffer, int len, int initial_size, int* outlen)
{
  stbi__zbuf a;
  char* p = (char*)stbi__malloc(initial_size);
  if (p == nullptr) return nullptr;
  a.zbuffer = (stbi_uc*)buffer;
  a.zbuffer_end = (stbi_uc*)buffer + len;
  if (stbi__do_zlib(&a, p, initial_size, 1, 1)) {
    if (outlen) *outlen = (int)(a.zout - a.zout_start);
    return a.zout_start;
  }
  else {
    STBI_FREE(a.zout_start);
    return nullptr;
  }
}

STBIDEF char* stbi_zlib_decode_malloc(char const* buffer, int len, int* outlen)
{
  return stbi_zlib_decode_malloc_guesssize(buffer, len, 16384, outlen);
}

STBIDEF char* stbi_zlib_decode_malloc_guesssize_headerflag(const char* buffer, int len, int initial_size, int* outlen, int parse_header)
{
  stbi__zbuf a;
  char* p = (char*)stbi__malloc(initial_size);
  if (p == nullptr) return nullptr;
  a.zbuffer = (stbi_uc*)buffer;
  a.zbuffer_end = (stbi_uc*)buffer + len;
  if (stbi__do_zlib(&a, p, initial_size, 1, parse_header)) {
    if (outlen) *outlen = (int)(a.zout - a.zout_start);
    return a.zout_start;
  }
  else {
    STBI_FREE(a.zout_start);
    return nullptr;
  }
}

STBIDEF int stbi_zlib_decode_buffer(char* obuffer, int olen, char const* ibuffer, int ilen)
{
  stbi__zbuf a;
  a.zbuffer = (stbi_uc*)ibuffer;
  a.zbuffer_end = (stbi_uc*)ibuffer + ilen;
  if (stbi__do_zlib(&a, obuffer, olen, 0, 1))
    return (int)(a.zout - a.zout_start);
  else
    return -1;
}

STBIDEF char* stbi_zlib_decode_noheader_malloc(char const* buffer, int len, int* outlen)
{
  stbi__zbuf a;
  char* p = (char*)stbi__malloc(16384);
  if (p == nullptr) return nullptr;
  a.zbuffer = (stbi_uc*)buffer;
  a.zbuffer_end = (stbi_uc*)buffer + len;
  if (stbi__do_zlib(&a, p, 16384, 1, 0)) {
    if (outlen) *outlen = (int)(a.zout - a.zout_start);
    return a.zout_start;
  }
  else {
    STBI_FREE(a.zout_start);
    return nullptr;
  }
}

STBIDEF int stbi_zlib_decode_noheader_buffer(char* obuffer, int olen, const char* ibuffer, int ilen)
{
  stbi__zbuf a;
  a.zbuffer = (stbi_uc*)ibuffer;
  a.zbuffer_end = (stbi_uc*)ibuffer + ilen;
  if (stbi__do_zlib(&a, obuffer, olen, 0, 0))
    return (int)(a.zout - a.zout_start);
  else
    return -1;
}
#endif

// public domain "baseline" PNG decoder   v0.10  Sean Barrett 2006-11-18
//    simple implementation
//      - only 8-bit samples
//      - no CRC checking
//      - allocates lots of intermediate memory
//        - avoids problem of streaming data between subsystems
//        - avoids explicit window management
//    performance
//      - uses stb_zlib, a PD zlib implementation with fast huffman decoding

#ifndef STBI_NO_PNG
typedef struct
{
  stbi__uint32 length;
  stbi__uint32 type;
} stbi__pngchunk;

static stbi__pngchunk stbi__get_chunk_header(stbi__context* s)
{
  stbi__pngchunk c;
  c.length = stbi__get32be(s);
  c.type = stbi__get32be(s);
  return c;
}

static int stbi__check_png_header(stbi__context* s)
{
  static stbi_uc png_sig[8] = { 137,80,78,71,13,10,26,10 };
  int i;
  for (i = 0; i < 8; ++i)
    if (stbi__get8(s) != png_sig[i]) return stbi__err("bad png sig", "Not a PNG");
  return 1;
}

typedef struct
{
  stbi__context* s;
  stbi_uc* idata, * expanded, * out;
} stbi__png;


enum {
  STBI__F_none = 0,
  STBI__F_sub = 1,
  STBI__F_up = 2,
  STBI__F_avg = 3,
  STBI__F_paeth = 4,
  // synthetic filters used for first scanline to avoid needing a dummy row of 0s
  STBI__F_avg_first,
  STBI__F_paeth_first
};

static stbi_uc first_row_filter[5] =
{
   STBI__F_none,
   STBI__F_sub,
   STBI__F_none,
   STBI__F_avg_first,
   STBI__F_paeth_first
};

static int stbi__paeth(int a, int b, int c)
{
  int p = a + b - c;
  int pa = abs(p - a);
  int pb = abs(p - b);
  int pc = abs(p - c);
  if (pa <= pb && pa <= pc) return a;
  if (pb <= pc) return b;
  return c;
}

static stbi_uc stbi__depth_scale_table[9] = { 0, 0xff, 0x55, 0, 0x11, 0,0,0, 0x01 };

// create the png data from post-deflated data
static int stbi__create_png_image_raw(stbi__png* a, stbi_uc* raw, stbi__uint32 raw_len, int out_n, stbi__uint32 x, stbi__uint32 y, int depth, int color)
{
  stbi__context* s = a->s;
  stbi__uint32 i, j, stride = x * out_n;
  stbi__uint32 img_len, img_width_bytes;
  int k;
  int img_n = s->img_n; // copy it into a local for later

  STBI_ASSERT(out_n == s->img_n || out_n == s->img_n + 1);
  a->out = (stbi_uc*)stbi__malloc(x * y * out_n); // extra bytes to write off the end into
  if (!a->out) return stbi__err("outofmem", "Out of memory");

  img_width_bytes = (((img_n * x * depth) + 7) >> 3);
  img_len = (img_width_bytes + 1) * y;
  if (s->img_x == x && s->img_y == y) {
    if (raw_len != img_len) return stbi__err("not enough pixels", "Corrupt PNG");
  }
  else { // interlaced:
    if (raw_len < img_len) return stbi__err("not enough pixels", "Corrupt PNG");
  }

  for (j = 0; j < y; ++j) {
    stbi_uc* cur = a->out + stride * j;
    stbi_uc* prior = cur - stride;
    int filter = *raw++;
    int filter_bytes = img_n;
    int width = x;
    if (filter > 4)
      return stbi__err("invalid filter", "Corrupt PNG");

    if (depth < 8) {
      STBI_ASSERT(img_width_bytes <= x);
      cur += x * out_n - img_width_bytes; // store output to the rightmost img_len bytes, so we can decode in place
      filter_bytes = 1;
      width = img_width_bytes;
    }

    // if first row, use special filter that doesn't sample previous row
    if (j == 0) filter = first_row_filter[filter];

    // handle first byte explicitly
    for (k = 0; k < filter_bytes; ++k) {
      switch (filter) {
      case STBI__F_none: cur[k] = raw[k]; break;
      case STBI__F_sub: cur[k] = raw[k]; break;
      case STBI__F_up: cur[k] = STBI__BYTECAST(raw[k] + prior[k]); break;
      case STBI__F_avg: cur[k] = STBI__BYTECAST(raw[k] + (prior[k] >> 1)); break;
      case STBI__F_paeth: cur[k] = STBI__BYTECAST(raw[k] + stbi__paeth(0, prior[k], 0)); break;
      case STBI__F_avg_first: cur[k] = raw[k]; break;
      case STBI__F_paeth_first: cur[k] = raw[k]; break;
      }
    }

    if (depth == 8) {
      if (img_n != out_n)
        cur[img_n] = 255; // first pixel
      raw += img_n;
      cur += out_n;
      prior += out_n;
    }
    else {
      raw += 1;
      cur += 1;
      prior += 1;
    }

    // this is a little gross, so that we don't switch per-pixel or per-component
    if (depth < 8 || img_n == out_n) {
      int nk = (width - 1) * img_n;
#define CASE(f) \
             case f:     \
                for (k=0; k < nk; ++k)
      switch (filter) {
        // "none" filter turns into a memcpy here; make that explicit.
      case STBI__F_none:         memcpy(cur, raw, nk);
        break;
        CASE(STBI__F_sub)          cur[k] = STBI__BYTECAST(raw[k] + cur[k - filter_bytes]);
        break;
        CASE(STBI__F_up)           cur[k] = STBI__BYTECAST(raw[k] + prior[k]);
        break;
        CASE(STBI__F_avg)          cur[k] = STBI__BYTECAST(raw[k] + ((prior[k] + cur[k - filter_bytes]) >> 1));
        break;
        CASE(STBI__F_paeth)        cur[k] = STBI__BYTECAST(raw[k] + stbi__paeth(cur[k - filter_bytes], prior[k], prior[k - filter_bytes]));
        break;
        CASE(STBI__F_avg_first)    cur[k] = STBI__BYTECAST(raw[k] + (cur[k - filter_bytes] >> 1));
        break;
        CASE(STBI__F_paeth_first)  cur[k] = STBI__BYTECAST(raw[k] + stbi__paeth(cur[k - filter_bytes], 0, 0));
        break;
      }
#undef CASE
      raw += nk;
    }
    else {
      STBI_ASSERT(img_n + 1 == out_n);
#define CASE(f) \
             case f:     \
                for (i=x-1; i >= 1; --i, cur[img_n]=255,raw+=img_n,cur+=out_n,prior+=out_n) \
                   for (k=0; k < img_n; ++k)
      switch (filter) {
        CASE(STBI__F_none)         cur[k] = raw[k];
        break;
        CASE(STBI__F_sub)          cur[k] = STBI__BYTECAST(raw[k] + cur[k - out_n]);
        break;
        CASE(STBI__F_up)           cur[k] = STBI__BYTECAST(raw[k] + prior[k]);
        break;
        CASE(STBI__F_avg)          cur[k] = STBI__BYTECAST(raw[k] + ((prior[k] + cur[k - out_n]) >> 1));
        break;
        CASE(STBI__F_paeth)        cur[k] = STBI__BYTECAST(raw[k] + stbi__paeth(cur[k - out_n], prior[k], prior[k - out_n]));
        break;
        CASE(STBI__F_avg_first)    cur[k] = STBI__BYTECAST(raw[k] + (cur[k - out_n] >> 1));
        break;
        CASE(STBI__F_paeth_first)  cur[k] = STBI__BYTECAST(raw[k] + stbi__paeth(cur[k - out_n], 0, 0));
        break;
      }
#undef CASE
    }
  }

  // we make a separate pass to expand bits to pixels; for performance,
  // this could run two scanlines behind the above code, so it won't
  // intefere with filtering but will still be in the cache.
  if (depth < 8) {
    for (j = 0; j < y; ++j) {
      stbi_uc* cur = a->out + stride * j;
      stbi_uc* in = a->out + stride * j + x * out_n - img_width_bytes;
      // unpack 1/2/4-bit into a 8-bit buffer. allows us to keep the common 8-bit path optimal at minimal cost for 1/2/4-bit
      // png guarante byte alignment, if width is not multiple of 8/4/2 we'll decode dummy trailing data that will be skipped in the later loop
      stbi_uc scale = (color == 0) ? stbi__depth_scale_table[depth] : 1; // scale grayscale values to 0..255 range

      // note that the final byte might overshoot and write more data than desired.
      // we can allocate enough data that this never writes out of memory, but it
      // could also overwrite the next scanline. can it overwrite non-empty data
      // on the next scanline? yes, consider 1-pixel-wide scanlines with 1-bit-per-pixel.
      // so we need to explicitly clamp the final ones

      if (depth == 4) {
        for (k = x * img_n; k >= 2; k -= 2, ++in) {
          *cur++ = scale * ((*in >> 4));
          *cur++ = scale * ((*in) & 0x0f);
        }
        if (k > 0) *cur++ = scale * ((*in >> 4));
      }
      else if (depth == 2) {
        for (k = x * img_n; k >= 4; k -= 4, ++in) {
          *cur++ = scale * ((*in >> 6));
          *cur++ = scale * ((*in >> 4) & 0x03);
          *cur++ = scale * ((*in >> 2) & 0x03);
          *cur++ = scale * ((*in) & 0x03);
        }
        if (k > 0) *cur++ = scale * ((*in >> 6));
        if (k > 1) *cur++ = scale * ((*in >> 4) & 0x03);
        if (k > 2) *cur++ = scale * ((*in >> 2) & 0x03);
      }
      else if (depth == 1) {
        for (k = x * img_n; k >= 8; k -= 8, ++in) {
          *cur++ = scale * ((*in >> 7));
          *cur++ = scale * ((*in >> 6) & 0x01);
          *cur++ = scale * ((*in >> 5) & 0x01);
          *cur++ = scale * ((*in >> 4) & 0x01);
          *cur++ = scale * ((*in >> 3) & 0x01);
          *cur++ = scale * ((*in >> 2) & 0x01);
          *cur++ = scale * ((*in >> 1) & 0x01);
          *cur++ = scale * ((*in) & 0x01);
        }
        if (k > 0) *cur++ = scale * ((*in >> 7));
        if (k > 1) *cur++ = scale * ((*in >> 6) & 0x01);
        if (k > 2) *cur++ = scale * ((*in >> 5) & 0x01);
        if (k > 3) *cur++ = scale * ((*in >> 4) & 0x01);
        if (k > 4) *cur++ = scale * ((*in >> 3) & 0x01);
        if (k > 5) *cur++ = scale * ((*in >> 2) & 0x01);
        if (k > 6) *cur++ = scale * ((*in >> 1) & 0x01);
      }
      if (img_n != out_n) {
        // insert alpha = 255
        stbi_uc* cur2 = a->out + stride * j;
        int i2;
        if (img_n == 1) {
          for (i2 = x - 1; i2 >= 0; --i2) {
            cur2[i2 * 2 + 1] = 255;
            cur2[i2 * 2 + 0] = cur2[i2];
          }
        }
        else {
          STBI_ASSERT(img_n == 3);
          for (i2 = x - 1; i2 >= 0; --i2) {
            cur2[i2 * 4 + 3] = 255;
            cur2[i2 * 4 + 2] = cur2[i2 * 3 + 2];
            cur2[i2 * 4 + 1] = cur2[i2 * 3 + 1];
            cur2[i2 * 4 + 0] = cur2[i2 * 3 + 0];
          }
        }
      }
    }
  }

  return 1;
}

static int stbi__create_png_image(stbi__png* a, stbi_uc* image_data, stbi__uint32 image_data_len, int out_n, int depth, int color, int interlaced)
{
  stbi_uc* final;
  int p;
  if (!interlaced)
    return stbi__create_png_image_raw(a, image_data, image_data_len, out_n, a->s->img_x, a->s->img_y, depth, color);

  // de-interlacing
  final = (stbi_uc*)stbi__malloc(a->s->img_x * a->s->img_y * out_n);
  for (p = 0; p < 7; ++p) {
    int xorig[] = { 0,4,0,2,0,1,0 };
    int yorig[] = { 0,0,4,0,2,0,1 };
    int xspc[] = { 8,8,4,4,2,2,1 };
    int yspc[] = { 8,8,8,4,4,2,2 };
    int i, j, x, y;
    // pass1_x[4] = 0, pass1_x[5] = 1, pass1_x[12] = 1
    x = (a->s->img_x - xorig[p] + xspc[p] - 1) / xspc[p];
    y = (a->s->img_y - yorig[p] + yspc[p] - 1) / yspc[p];
    if (x && y) {
      stbi__uint32 img_len = ((((a->s->img_n * x * depth) + 7) >> 3) + 1) * y;
      if (!stbi__create_png_image_raw(a, image_data, image_data_len, out_n, x, y, depth, color)) {
        STBI_FREE(final);
        return 0;
      }
      for (j = 0; j < y; ++j) {
        for (i = 0; i < x; ++i) {
          int out_y = j * yspc[p] + yorig[p];
          int out_x = i * xspc[p] + xorig[p];
          memcpy(final + out_y * a->s->img_x * out_n + out_x * out_n,
            a->out + (j * x + i) * out_n, out_n);
        }
      }
      STBI_FREE(a->out);
      image_data += img_len;
      image_data_len -= img_len;
    }
  }
  a->out = final;

  return 1;
}

static int stbi__compute_transparency(stbi__png* z, stbi_uc tc[3], int out_n)
{
  stbi__context* s = z->s;
  stbi__uint32 i, pixel_count = s->img_x * s->img_y;
  stbi_uc* p = z->out;

  // compute color-based transparency, assuming we've
  // already got 255 as the alpha value in the output
  STBI_ASSERT(out_n == 2 || out_n == 4);

  if (out_n == 2) {
    for (i = 0; i < pixel_count; ++i) {
      p[1] = (p[0] == tc[0] ? 0 : 255);
      p += 2;
    }
  }
  else {
    for (i = 0; i < pixel_count; ++i) {
      if (p[0] == tc[0] && p[1] == tc[1] && p[2] == tc[2])
        p[3] = 0;
      p += 4;
    }
  }
  return 1;
}

static int stbi__expand_png_palette(stbi__png* a, stbi_uc* palette, int len, int pal_img_n)
{
  stbi__uint32 i, pixel_count = a->s->img_x * a->s->img_y;
  stbi_uc* p, * temp_out, * orig = a->out;

  p = (stbi_uc*)stbi__malloc(pixel_count * pal_img_n);
  if (p == nullptr) return stbi__err("outofmem", "Out of memory");

  // between here and free(out) below, exitting would leak
  temp_out = p;

  if (pal_img_n == 3) {
    for (i = 0; i < pixel_count; ++i) {
      int n = orig[i] * 4;
      p[0] = palette[n];
      p[1] = palette[n + 1];
      p[2] = palette[n + 2];
      p += 3;
    }
  }
  else {
    for (i = 0; i < pixel_count; ++i) {
      int n = orig[i] * 4;
      p[0] = palette[n];
      p[1] = palette[n + 1];
      p[2] = palette[n + 2];
      p[3] = palette[n + 3];
      p += 4;
    }
  }
  STBI_FREE(a->out);
  a->out = temp_out;

  STBI_NOTUSED(len);

  return 1;
}

static int stbi__unpremultiply_on_load = 0;
static int stbi__de_iphone_flag = 0;

STBIDEF void stbi_set_unpremultiply_on_load(int flag_true_if_should_unpremultiply)
{
  stbi__unpremultiply_on_load = flag_true_if_should_unpremultiply;
}

STBIDEF void stbi_convert_iphone_png_to_rgb(int flag_true_if_should_convert)
{
  stbi__de_iphone_flag = flag_true_if_should_convert;
}

static void stbi__de_iphone(stbi__png* z)
{
  stbi__context* s = z->s;
  stbi__uint32 i, pixel_count = s->img_x * s->img_y;
  stbi_uc* p = z->out;

  if (s->img_out_n == 3) {  // convert bgr to rgb
    for (i = 0; i < pixel_count; ++i) {
      stbi_uc t = p[0];
      p[0] = p[2];
      p[2] = t;
      p += 3;
    }
  }
  else {
    STBI_ASSERT(s->img_out_n == 4);
    if (stbi__unpremultiply_on_load) {
      // convert bgr to rgb and unpremultiply
      for (i = 0; i < pixel_count; ++i) {
        stbi_uc a = p[3];
        stbi_uc t = p[0];
        if (a) {
          p[0] = p[2] * 255 / a;
          p[1] = p[1] * 255 / a;
          p[2] = t * 255 / a;
        }
        else {
          p[0] = p[2];
          p[2] = t;
        }
        p += 4;
      }
    }
    else {
      // convert bgr to rgb
      for (i = 0; i < pixel_count; ++i) {
        stbi_uc t = p[0];
        p[0] = p[2];
        p[2] = t;
        p += 4;
      }
    }
  }
}

#define STBI__PNG_TYPE(a,b,c,d)  (((a) << 24) + ((b) << 16) + ((c) << 8) + (d))

static int stbi__parse_png_file(stbi__png* z, int scan, int req_comp)
{
  stbi_uc palette[1024], pal_img_n = 0;
  stbi_uc has_trans = 0, tc[3];
  stbi__uint32 ioff = 0, idata_limit = 0, i, pal_len = 0;
  int first = 1, k, interlace = 0, color = 0, depth = 0, is_iphone = 0;
  stbi__context* s = z->s;

  z->expanded = nullptr;
  z->idata = nullptr;
  z->out = nullptr;

  if (!stbi__check_png_header(s)) return 0;

  if (scan == STBI__SCAN_type) return 1;

  for (;;) {
    stbi__pngchunk c = stbi__get_chunk_header(s);
    switch (c.type) {
    case STBI__PNG_TYPE('C', 'g', 'B', 'I'):
      is_iphone = 1;
      stbi__skip(s, c.length);
      break;
    case STBI__PNG_TYPE('I', 'H', 'D', 'R'): {
      int comp, filter;
      if (!first) return stbi__err("multiple IHDR", "Corrupt PNG");
      first = 0;
      if (c.length != 13) return stbi__err("bad IHDR len", "Corrupt PNG");
      s->img_x = stbi__get32be(s); if (s->img_x > (1 << 24)) return stbi__err("too large", "Very large image (corrupt?)");
      s->img_y = stbi__get32be(s); if (s->img_y > (1 << 24)) return stbi__err("too large", "Very large image (corrupt?)");
      depth = stbi__get8(s);  if (depth != 1 && depth != 2 && depth != 4 && depth != 8)  return stbi__err("1/2/4/8-bit only", "PNG not supported: 1/2/4/8-bit only");
      color = stbi__get8(s);  if (color > 6)         return stbi__err("bad ctype", "Corrupt PNG");
      if (color == 3) pal_img_n = 3; else if (color & 1) return stbi__err("bad ctype", "Corrupt PNG");
      comp = stbi__get8(s);  if (comp) return stbi__err("bad comp method", "Corrupt PNG");
      filter = stbi__get8(s);  if (filter) return stbi__err("bad filter method", "Corrupt PNG");
      interlace = stbi__get8(s); if (interlace > 1) return stbi__err("bad interlace method", "Corrupt PNG");
      if (!s->img_x || !s->img_y) return stbi__err("0-pixel image", "Corrupt PNG");
      if (!pal_img_n) {
        s->img_n = (color & 2 ? 3 : 1) + (color & 4 ? 1 : 0);
        if ((1 << 30) / s->img_x / s->img_n < s->img_y) return stbi__err("too large", "Image too large to decode");
        if (scan == STBI__SCAN_header) return 1;
      }
      else {
        // if paletted, then pal_n is our final components, and
        // img_n is # components to decompress/filter.
        s->img_n = 1;
        if ((1 << 30) / s->img_x / 4 < s->img_y) return stbi__err("too large", "Corrupt PNG");
        // if SCAN_header, have to scan to see if we have a tRNS
      }
      break;
    }

    case STBI__PNG_TYPE('P', 'L', 'T', 'E'): {
      if (first) return stbi__err("first not IHDR", "Corrupt PNG");
      if (c.length > 256 * 3) return stbi__err("invalid PLTE", "Corrupt PNG");
      pal_len = c.length / 3;
      if (pal_len * 3 != c.length) return stbi__err("invalid PLTE", "Corrupt PNG");
      for (i = 0; i < pal_len; ++i) {
        palette[i * 4 + 0] = stbi__get8(s);
        palette[i * 4 + 1] = stbi__get8(s);
        palette[i * 4 + 2] = stbi__get8(s);
        palette[i * 4 + 3] = 255;
      }
      break;
    }

    case STBI__PNG_TYPE('t', 'R', 'N', 'S'): {
      if (first) return stbi__err("first not IHDR", "Corrupt PNG");
      if (z->idata) return stbi__err("tRNS after IDAT", "Corrupt PNG");
      if (pal_img_n) {
        if (scan == STBI__SCAN_header) { s->img_n = 4; return 1; }
        if (pal_len == 0) return stbi__err("tRNS before PLTE", "Corrupt PNG");
        if (c.length > pal_len) return stbi__err("bad tRNS len", "Corrupt PNG");
        pal_img_n = 4;
        for (i = 0; i < c.length; ++i)
          palette[i * 4 + 3] = stbi__get8(s);
      }
      else {
        if (!(s->img_n & 1)) return stbi__err("tRNS with alpha", "Corrupt PNG");
        if (c.length != (stbi__uint32)s->img_n * 2) return stbi__err("bad tRNS len", "Corrupt PNG");
        has_trans = 1;
        for (k = 0; k < s->img_n; ++k)
          tc[k] = (stbi_uc)(stbi__get16be(s) & 255) * stbi__depth_scale_table[depth]; // non 8-bit images will be larger
      }
      break;
    }

    case STBI__PNG_TYPE('I', 'D', 'A', 'T'): {
      if (first) return stbi__err("first not IHDR", "Corrupt PNG");
      if (pal_img_n && !pal_len) return stbi__err("no PLTE", "Corrupt PNG");
      if (scan == STBI__SCAN_header) { s->img_n = pal_img_n; return 1; }
      if ((int)(ioff + c.length) < (int)ioff) return 0;
      if (ioff + c.length > idata_limit) {
        stbi_uc* p;
        if (idata_limit == 0) idata_limit = c.length > 4096 ? c.length : 4096;
        while (ioff + c.length > idata_limit)
          idata_limit *= 2;
        p = (stbi_uc*)STBI_REALLOC(z->idata, idata_limit); if (p == nullptr) return stbi__err("outofmem", "Out of memory");
        z->idata = p;
      }
      if (!stbi__getn(s, z->idata + ioff, c.length)) return stbi__err("outofdata", "Corrupt PNG");
      ioff += c.length;
      break;
    }

    case STBI__PNG_TYPE('I', 'E', 'N', 'D'): {
      stbi__uint32 raw_len, bpl;
      if (first) return stbi__err("first not IHDR", "Corrupt PNG");
      if (scan != STBI__SCAN_load) return 1;
      if (z->idata == nullptr) return stbi__err("no IDAT", "Corrupt PNG");
      // initial guess for decoded data size to avoid unnecessary reallocs
      bpl = (s->img_x * depth + 7) / 8; // bytes per line, per component
      raw_len = bpl * s->img_y * s->img_n /* pixels */ + s->img_y /* filter mode per row */;
      z->expanded = (stbi_uc*)stbi_zlib_decode_malloc_guesssize_headerflag((char*)z->idata, ioff, raw_len, (int*)&raw_len, !is_iphone);
      if (z->expanded == nullptr) return 0; // zlib should set error
      STBI_FREE(z->idata); z->idata = nullptr;
      if ((req_comp == s->img_n + 1 && req_comp != 3 && !pal_img_n) || has_trans)
        s->img_out_n = s->img_n + 1;
      else
        s->img_out_n = s->img_n;
      if (!stbi__create_png_image(z, z->expanded, raw_len, s->img_out_n, depth, color, interlace)) return 0;
      if (has_trans)
        if (!stbi__compute_transparency(z, tc, s->img_out_n)) return 0;
      if (is_iphone && stbi__de_iphone_flag && s->img_out_n > 2)
        stbi__de_iphone(z);
      if (pal_img_n) {
        // pal_img_n == 3 or 4
        s->img_n = pal_img_n; // record the actual colors we had
        s->img_out_n = pal_img_n;
        if (req_comp >= 3) s->img_out_n = req_comp;
        if (!stbi__expand_png_palette(z, palette, pal_len, s->img_out_n))
          return 0;
      }
      STBI_FREE(z->expanded); z->expanded = nullptr;
      return 1;
    }

    default:
      // if critical, fail
      if (first) return stbi__err("first not IHDR", "Corrupt PNG");
      if ((c.type & (1 << 29)) == 0) {
#ifndef STBI_NO_FAILURE_STRINGS
        // not threadsafe
        static char invalid_chunk[] = "XXXX PNG chunk not known";
        invalid_chunk[0] = STBI__BYTECAST(c.type >> 24);
        invalid_chunk[1] = STBI__BYTECAST(c.type >> 16);
        invalid_chunk[2] = STBI__BYTECAST(c.type >> 8);
        invalid_chunk[3] = STBI__BYTECAST(c.type >> 0);
#endif
        return stbi__err(invalid_chunk, "PNG not supported: unknown PNG chunk type");
      }
      stbi__skip(s, c.length);
      break;
    }
    // end of PNG chunk, read and skip CRC
    stbi__get32be(s);
  }
}

static unsigned char* stbi__do_png(stbi__png* p, int* x, int* y, int* n, int req_comp)
{
  unsigned char* result = nullptr;
  if (req_comp < 0 || req_comp > 4) return stbi__errpuc("bad req_comp", "Internal error");
  if (stbi__parse_png_file(p, STBI__SCAN_load, req_comp)) {
    result = p->out;
    p->out = nullptr;
    if (req_comp && req_comp != p->s->img_out_n) {
      result = stbi__convert_format(result, p->s->img_out_n, req_comp, p->s->img_x, p->s->img_y);
      p->s->img_out_n = req_comp;
      if (result == nullptr) return result;
    }
    *x = p->s->img_x;
    *y = p->s->img_y;
    if (n) *n = p->s->img_out_n;
  }
  STBI_FREE(p->out);      p->out = nullptr;
  STBI_FREE(p->expanded); p->expanded = nullptr;
  STBI_FREE(p->idata);    p->idata = nullptr;

  return result;
}

static unsigned char* stbi__png_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  stbi__png p;
  p.s = s;
  return stbi__do_png(&p, x, y, comp, req_comp);
}

static int stbi__png_test(stbi__context* s)
{
  int r;
  r = stbi__check_png_header(s);
  stbi__rewind(s);
  return r;
}

static int stbi__png_info_raw(stbi__png* p, int* x, int* y, int* comp)
{
  if (!stbi__parse_png_file(p, STBI__SCAN_header, 0)) {
    stbi__rewind(p->s);
    return 0;
  }
  if (x) *x = p->s->img_x;
  if (y) *y = p->s->img_y;
  if (comp) *comp = p->s->img_n;
  return 1;
}

static int stbi__png_info(stbi__context* s, int* x, int* y, int* comp)
{
  stbi__png p;
  p.s = s;
  return stbi__png_info_raw(&p, x, y, comp);
}
#endif

// Microsoft/Windows BMP image

#ifndef STBI_NO_BMP
static int stbi__bmp_test_raw(stbi__context* s)
{
  int r;
  int sz;
  if (stbi__get8(s) != 'B') return 0;
  if (stbi__get8(s) != 'M') return 0;
  stbi__get32le(s); // discard filesize
  stbi__get16le(s); // discard reserved
  stbi__get16le(s); // discard reserved
  stbi__get32le(s); // discard data offset
  sz = stbi__get32le(s);
  r = (sz == 12 || sz == 40 || sz == 56 || sz == 108 || sz == 124);
  return r;
}

static int stbi__bmp_test(stbi__context* s)
{
  int r = stbi__bmp_test_raw(s);
  stbi__rewind(s);
  return r;
}


// returns 0..31 for the highest set bit
static int stbi__high_bit(unsigned int z)
{
  int n = 0;
  if (z == 0) return -1;
  if (z >= 0x10000) n += 16, z >>= 16;
  if (z >= 0x00100) n += 8, z >>= 8;
  if (z >= 0x00010) n += 4, z >>= 4;
  if (z >= 0x00004) n += 2, z >>= 2;
  if (z >= 0x00002) n += 1, z >>= 1;
  return n;
}

static int stbi__bitcount(unsigned int a)
{
  a = (a & 0x55555555) + ((a >> 1) & 0x55555555); // max 2
  a = (a & 0x33333333) + ((a >> 2) & 0x33333333); // max 4
  a = (a + (a >> 4)) & 0x0f0f0f0f; // max 8 per 4, now 8 bits
  a = (a + (a >> 8)); // max 16 per 8 bits
  a = (a + (a >> 16)); // max 32 per 8 bits
  return a & 0xff;
}

static int stbi__shiftsigned(int v, int shift, int bits)
{
  int result;
  int z = 0;

  if (shift < 0) v <<= -shift;
  else v >>= shift;
  result = v;

  z = bits;
  while (z < 8) {
    result += v >> z;
    z += bits;
  }
  return result;
}

static stbi_uc* stbi__bmp_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  stbi_uc* out;
  unsigned int mr = 0, mg = 0, mb = 0, ma = 0, fake_a = 0;
  stbi_uc pal[256][4];
  int psize = 0, i, j, compress = 0, width;
  int bpp, flip_vertically, pad, target, offset, hsz;
  if (stbi__get8(s) != 'B' || stbi__get8(s) != 'M') return stbi__errpuc("not BMP", "Corrupt BMP");
  stbi__get32le(s); // discard filesize
  stbi__get16le(s); // discard reserved
  stbi__get16le(s); // discard reserved
  offset = stbi__get32le(s);
  hsz = stbi__get32le(s);
  if (hsz != 12 && hsz != 40 && hsz != 56 && hsz != 108 && hsz != 124) return stbi__errpuc("unknown BMP", "BMP type not supported: unknown");
  if (hsz == 12) {
    s->img_x = stbi__get16le(s);
    s->img_y = stbi__get16le(s);
  }
  else {
    s->img_x = stbi__get32le(s);
    s->img_y = stbi__get32le(s);
  }
  if (stbi__get16le(s) != 1) return stbi__errpuc("bad BMP", "bad BMP");
  bpp = stbi__get16le(s);
  if (bpp == 1) return stbi__errpuc("monochrome", "BMP type not supported: 1-bit");
  flip_vertically = ((int)s->img_y) > 0;
  s->img_y = abs((int)s->img_y);
  if (hsz == 12) {
    if (bpp < 24)
      psize = (offset - 14 - 24) / 3;
  }
  else {
    compress = stbi__get32le(s);
    if (compress == 1 || compress == 2) return stbi__errpuc("BMP RLE", "BMP type not supported: RLE");
    stbi__get32le(s); // discard sizeof
    stbi__get32le(s); // discard hres
    stbi__get32le(s); // discard vres
    stbi__get32le(s); // discard colorsused
    stbi__get32le(s); // discard max important
    if (hsz == 40 || hsz == 56) {
      if (hsz == 56) {
        stbi__get32le(s);
        stbi__get32le(s);
        stbi__get32le(s);
        stbi__get32le(s);
      }
      if (bpp == 16 || bpp == 32) {
        mr = mg = mb = 0;
        if (compress == 0) {
          if (bpp == 32) {
            mr = 0xffu << 16;
            mg = 0xffu << 8;
            mb = 0xffu << 0;
            ma = 0xffu << 24;
            fake_a = 1; // @TODO: check for cases like alpha value is all 0 and switch it to 255
            STBI_NOTUSED(fake_a);
          }
          else {
            mr = 31u << 10;
            mg = 31u << 5;
            mb = 31u << 0;
          }
        }
        else if (compress == 3) {
          mr = stbi__get32le(s);
          mg = stbi__get32le(s);
          mb = stbi__get32le(s);
          // not documented, but generated by photoshop and handled by mspaint
          if (mr == mg && mg == mb) {
            // ?!?!?
            return stbi__errpuc("bad BMP", "bad BMP");
          }
        }
        else
          return stbi__errpuc("bad BMP", "bad BMP");
      }
    }
    else {
      STBI_ASSERT(hsz == 108 || hsz == 124);
      mr = stbi__get32le(s);
      mg = stbi__get32le(s);
      mb = stbi__get32le(s);
      ma = stbi__get32le(s);
      stbi__get32le(s); // discard color space
      for (i = 0; i < 12; ++i)
        stbi__get32le(s); // discard color space parameters
      if (hsz == 124) {
        stbi__get32le(s); // discard rendering intent
        stbi__get32le(s); // discard offset of profile data
        stbi__get32le(s); // discard size of profile data
        stbi__get32le(s); // discard reserved
      }
    }
    if (bpp < 16)
      psize = (offset - 14 - hsz) >> 2;
  }
  s->img_n = ma ? 4 : 3;
  if (req_comp && req_comp >= 3) // we can directly decode 3 or 4
    target = req_comp;
  else
    target = s->img_n; // if they want monochrome, we'll post-convert
  out = (stbi_uc*)stbi__malloc(target * s->img_x * s->img_y);
  if (!out) return stbi__errpuc("outofmem", "Out of memory");
  if (bpp < 16) {
    int z = 0;
    if (psize == 0 || psize > 256) { STBI_FREE(out); return stbi__errpuc("invalid", "Corrupt BMP"); }
    for (i = 0; i < psize; ++i) {
      pal[i][2] = stbi__get8(s);
      pal[i][1] = stbi__get8(s);
      pal[i][0] = stbi__get8(s);
      if (hsz != 12) stbi__get8(s);
      pal[i][3] = 255;
    }
    stbi__skip(s, offset - 14 - hsz - psize * (hsz == 12 ? 3 : 4));
    if (bpp == 4) width = (s->img_x + 1) >> 1;
    else if (bpp == 8) width = s->img_x;
    else { STBI_FREE(out); return stbi__errpuc("bad bpp", "Corrupt BMP"); }
    pad = (-width) & 3;
    for (j = 0; j < (int)s->img_y; ++j) {
      for (i = 0; i < (int)s->img_x; i += 2) {
        int v = stbi__get8(s), v2 = 0;
        if (bpp == 4) {
          v2 = v & 15;
          v >>= 4;
        }
        out[z++] = pal[v][0];
        out[z++] = pal[v][1];
        out[z++] = pal[v][2];
        if (target == 4) out[z++] = 255;
        if (i + 1 == (int)s->img_x) break;
        v = (bpp == 8) ? stbi__get8(s) : v2;
        out[z++] = pal[v][0];
        out[z++] = pal[v][1];
        out[z++] = pal[v][2];
        if (target == 4) out[z++] = 255;
      }
      stbi__skip(s, pad);
    }
  }
  else {
    int rshift = 0, gshift = 0, bshift = 0, ashift = 0, rcount = 0, gcount = 0, bcount = 0, acount = 0;
    int z = 0;
    int easy = 0;
    stbi__skip(s, offset - 14 - hsz);
    if (bpp == 24) width = 3 * s->img_x;
    else if (bpp == 16) width = 2 * s->img_x;
    else /* bpp = 32 and pad = 0 */ width = 0;
    pad = (-width) & 3;
    if (bpp == 24) {
      easy = 1;
    }
    else if (bpp == 32) {
      if (mb == 0xff && mg == 0xff00 && mr == 0x00ff0000 && ma == 0xff000000)
        easy = 2;
    }
    if (!easy) {
      if (!mr || !mg || !mb) { STBI_FREE(out); return stbi__errpuc("bad masks", "Corrupt BMP"); }
      // right shift amt to put high bit in position #7
      rshift = stbi__high_bit(mr) - 7; rcount = stbi__bitcount(mr);
      gshift = stbi__high_bit(mg) - 7; gcount = stbi__bitcount(mg);
      bshift = stbi__high_bit(mb) - 7; bcount = stbi__bitcount(mb);
      ashift = stbi__high_bit(ma) - 7; acount = stbi__bitcount(ma);
    }
    for (j = 0; j < (int)s->img_y; ++j) {
      if (easy) {
        for (i = 0; i < (int)s->img_x; ++i) {
          unsigned char a;
          out[z + 2] = stbi__get8(s);
          out[z + 1] = stbi__get8(s);
          out[z + 0] = stbi__get8(s);
          z += 3;
          a = (easy == 2 ? stbi__get8(s) : 255);
          if (target == 4) out[z++] = a;
        }
      }
      else {
        for (i = 0; i < (int)s->img_x; ++i) {
          stbi__uint32 v = (bpp == 16 ? (stbi__uint32)stbi__get16le(s) : stbi__get32le(s));
          int a;
          out[z++] = STBI__BYTECAST(stbi__shiftsigned(v & mr, rshift, rcount));
          out[z++] = STBI__BYTECAST(stbi__shiftsigned(v & mg, gshift, gcount));
          out[z++] = STBI__BYTECAST(stbi__shiftsigned(v & mb, bshift, bcount));
          a = (ma ? stbi__shiftsigned(v & ma, ashift, acount) : 255);
          if (target == 4) out[z++] = STBI__BYTECAST(a);
        }
      }
      stbi__skip(s, pad);
    }
  }
  if (flip_vertically) {
    stbi_uc t;
    for (j = 0; j < (int)s->img_y >> 1; ++j) {
      stbi_uc* p1 = out + j * s->img_x * target;
      stbi_uc* p2 = out + (s->img_y - 1 - j) * s->img_x * target;
      for (i = 0; i < (int)s->img_x * target; ++i) {
        t = p1[i], p1[i] = p2[i], p2[i] = t;
      }
    }
  }

  if (req_comp && req_comp != target) {
    out = stbi__convert_format(out, target, req_comp, s->img_x, s->img_y);
    if (out == nullptr) return out; // stbi__convert_format frees input on failure
  }

  *x = s->img_x;
  *y = s->img_y;
  if (comp) *comp = s->img_n;
  return out;
}
#endif

// Targa Truevision - TGA
// by Jonathan Dummer
#ifndef STBI_NO_TGA
static int stbi__tga_info(stbi__context* s, int* x, int* y, int* comp)
{
  int tga_w, tga_h, tga_comp;
  int sz;
  stbi__get8(s);                   // discard Offset
  sz = stbi__get8(s);              // color type
  if (sz > 1) {
    stbi__rewind(s);
    return 0;      // only RGB or indexed allowed
  }
  sz = stbi__get8(s);              // image type
  // only RGB or grey allowed, +/- RLE
  if ((sz != 1) && (sz != 2) && (sz != 3) && (sz != 9) && (sz != 10) && (sz != 11)) return 0;
  stbi__skip(s, 9);
  tga_w = stbi__get16le(s);
  if (tga_w < 1) {
    stbi__rewind(s);
    return 0;   // test width
  }
  tga_h = stbi__get16le(s);
  if (tga_h < 1) {
    stbi__rewind(s);
    return 0;   // test height
  }
  sz = stbi__get8(s);               // bits per pixel
  // only RGB or RGBA or grey allowed
  if ((sz != 8) && (sz != 16) && (sz != 24) && (sz != 32)) {
    stbi__rewind(s);
    return 0;
  }
  tga_comp = sz;
  if (x) *x = tga_w;
  if (y) *y = tga_h;
  if (comp) *comp = tga_comp / 8;
  return 1;                   // seems to have passed everything
}

static int stbi__tga_test(stbi__context* s)
{
  int res;
  int sz;
  stbi__get8(s);      //   discard Offset
  sz = stbi__get8(s);   //   color type
  if (sz > 1) return 0;   //   only RGB or indexed allowed
  sz = stbi__get8(s);   //   image type
  if ((sz != 1) && (sz != 2) && (sz != 3) && (sz != 9) && (sz != 10) && (sz != 11)) return 0;   //   only RGB or grey allowed, +/- RLE
  stbi__get16be(s);      //   discard palette start
  stbi__get16be(s);      //   discard palette length
  stbi__get8(s);         //   discard bits per palette color entry
  stbi__get16be(s);      //   discard x origin
  stbi__get16be(s);      //   discard y origin
  if (stbi__get16be(s) < 1) return 0;      //   test width
  if (stbi__get16be(s) < 1) return 0;      //   test height
  sz = stbi__get8(s);   //   bits per pixel
  if ((sz != 8) && (sz != 16) && (sz != 24) && (sz != 32))
    res = 0;
  else
    res = 1;
  stbi__rewind(s);
  return res;
}

static stbi_uc* stbi__tga_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  //   read in the TGA header stuff
  int tga_offset = stbi__get8(s);
  int tga_indexed = stbi__get8(s);
  int tga_image_type = stbi__get8(s);
  int tga_is_RLE = 0;
  int tga_palette_start = stbi__get16le(s);
  int tga_palette_len = stbi__get16le(s);
  int tga_palette_bits = stbi__get8(s);
  int tga_x_origin = stbi__get16le(s);
  int tga_y_origin = stbi__get16le(s);
  int tga_width = stbi__get16le(s);
  int tga_height = stbi__get16le(s);
  int tga_bits_per_pixel = stbi__get8(s);
  int tga_comp = tga_bits_per_pixel / 8;
  int tga_inverted = stbi__get8(s);
  //   image data
  unsigned char* tga_data;
  unsigned char* tga_palette = nullptr;
  int i, j;
  unsigned char raw_data[4];
  int RLE_count = 0;
  int RLE_repeating = 0;
  int read_next_pixel = 1;

  //   do a tiny bit of precessing
  if (tga_image_type >= 8)
  {
    tga_image_type -= 8;
    tga_is_RLE = 1;
  }
  /* int tga_alpha_bits = tga_inverted & 15; */
  tga_inverted = 1 - ((tga_inverted >> 5) & 1);

  //   error check
  if ( //(tga_indexed) ||
    (tga_width < 1) || (tga_height < 1) ||
    (tga_image_type < 1) || (tga_image_type > 3) ||
    ((tga_bits_per_pixel != 8) && (tga_bits_per_pixel != 16) &&
      (tga_bits_per_pixel != 24) && (tga_bits_per_pixel != 32))
    )
  {
    return nullptr; // we don't report this as a bad TGA because we don't even know if it's TGA
  }

  //   If I'm paletted, then I'll use the number of bits from the palette
  if (tga_indexed)
  {
    tga_comp = tga_palette_bits / 8;
  }

  //   tga info
  *x = tga_width;
  *y = tga_height;
  if (comp) *comp = tga_comp;

  tga_data = (unsigned char*)stbi__malloc((size_t)tga_width * tga_height * tga_comp);
  if (!tga_data) return stbi__errpuc("outofmem", "Out of memory");

  // skip to the data's starting position (offset usually = 0)
  stbi__skip(s, tga_offset);

  if (!tga_indexed && !tga_is_RLE) {
    for (i = 0; i < tga_height; ++i) {
      int y2 = tga_inverted ? tga_height - i - 1 : i;
      stbi_uc* tga_row = tga_data + y2 * tga_width * tga_comp;
      stbi__getn(s, tga_row, tga_width * tga_comp);
    }
  }
  else {
    //   do I need to load a palette?
    if (tga_indexed)
    {
      //   any data to skip? (offset usually = 0)
      stbi__skip(s, tga_palette_start);
      //   load the palette
      tga_palette = (unsigned char*)stbi__malloc(tga_palette_len * tga_palette_bits / 8);
      if (!tga_palette) {
        STBI_FREE(tga_data);
        return stbi__errpuc("outofmem", "Out of memory");
      }
      if (!stbi__getn(s, tga_palette, tga_palette_len * tga_palette_bits / 8)) {
        STBI_FREE(tga_data);
        STBI_FREE(tga_palette);
        return stbi__errpuc("bad palette", "Corrupt TGA");
      }
    }
    //   load the data
    for (i = 0; i < tga_width * tga_height; ++i)
    {
      //   if I'm in RLE mode, do I need to get a RLE stbi__pngchunk?
      if (tga_is_RLE)
      {
        if (RLE_count == 0)
        {
          //   yep, get the next byte as a RLE command
          int RLE_cmd = stbi__get8(s);
          RLE_count = 1 + (RLE_cmd & 127);
          RLE_repeating = RLE_cmd >> 7;
          read_next_pixel = 1;
        }
        else if (!RLE_repeating)
        {
          read_next_pixel = 1;
        }
      }
      else
      {
        read_next_pixel = 1;
      }
      //   OK, if I need to read a pixel, do it now
      if (read_next_pixel)
      {
        //   load however much data we did have
        if (tga_indexed)
        {
          //   read in 1 byte, then perform the lookup
          int pal_idx = stbi__get8(s);
          if (pal_idx >= tga_palette_len)
          {
            //   invalid index
            pal_idx = 0;
          }
          pal_idx *= tga_bits_per_pixel / 8;
          for (j = 0; j * 8 < tga_bits_per_pixel; ++j)
          {
            raw_data[j] = tga_palette[pal_idx + j];
          }
        }
        else
        {
          //   read in the data raw
          for (j = 0; j * 8 < tga_bits_per_pixel; ++j)
          {
            raw_data[j] = stbi__get8(s);
          }
        }
        //   clear the reading flag for the next pixel
        read_next_pixel = 0;
      } // end of reading a pixel

      // copy data
      for (j = 0; j < tga_comp; ++j)
        tga_data[i * tga_comp + j] = raw_data[j];

      //   in case we're in RLE mode, keep counting down
      --RLE_count;
    }
    //   do I need to invert the image?
    if (tga_inverted)
    {
      for (j = 0; j * 2 < tga_height; ++j)
      {
        int index1 = j * tga_width * tga_comp;
        int index2 = (tga_height - 1 - j) * tga_width * tga_comp;
        for (i = tga_width * tga_comp; i > 0; --i)
        {
          unsigned char temp = tga_data[index1];
          tga_data[index1] = tga_data[index2];
          tga_data[index2] = temp;
          ++index1;
          ++index2;
        }
      }
    }
    //   clear my palette, if I had one
    if (tga_palette != nullptr)
    {
      STBI_FREE(tga_palette);
    }
  }

  // swap RGB
  if (tga_comp >= 3)
  {
    unsigned char* tga_pixel = tga_data;
    for (i = 0; i < tga_width * tga_height; ++i)
    {
      unsigned char temp = tga_pixel[0];
      tga_pixel[0] = tga_pixel[2];
      tga_pixel[2] = temp;
      tga_pixel += tga_comp;
    }
  }

  // convert to target component count
  if (req_comp && req_comp != tga_comp)
    tga_data = stbi__convert_format(tga_data, tga_comp, req_comp, tga_width, tga_height);

  //   the things I do to get rid of an error message, and yet keep
  //   Microsoft's C compilers happy... [8^(
  tga_palette_start = tga_palette_len = tga_palette_bits =
    tga_x_origin = tga_y_origin = 0;
  //   OK, done
  return tga_data;
}
#endif

// *************************************************************************************************
// Photoshop PSD loader -- PD by Thatcher Ulrich, integration by Nicolas Schulz, tweaked by STB

#ifndef STBI_NO_PSD
static int stbi__psd_test(stbi__context* s)
{
  int r = (stbi__get32be(s) == 0x38425053);
  stbi__rewind(s);
  return r;
}

static stbi_uc* stbi__psd_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  int   pixelCount;
  int channelCount, compression;
  int channel, i, count, len;
  int w, h;
  stbi_uc* out;

  // Check identifier
  if (stbi__get32be(s) != 0x38425053)   // "8BPS"
    return stbi__errpuc("not PSD", "Corrupt PSD image");

  // Check file type version.
  if (stbi__get16be(s) != 1)
    return stbi__errpuc("wrong version", "Unsupported version of PSD image");

  // Skip 6 reserved bytes.
  stbi__skip(s, 6);

  // Read the number of channels (R, G, B, A, etc).
  channelCount = stbi__get16be(s);
  if (channelCount < 0 || channelCount > 16)
    return stbi__errpuc("wrong channel count", "Unsupported number of channels in PSD image");

  // Read the rows and columns of the image.
  h = stbi__get32be(s);
  w = stbi__get32be(s);

  // Make sure the depth is 8 bits.
  if (stbi__get16be(s) != 8)
    return stbi__errpuc("unsupported bit depth", "PSD bit depth is not 8 bit");

  // Make sure the color mode is RGB.
  // Valid options are:
  //   0: Bitmap
  //   1: Grayscale
  //   2: Indexed color
  //   3: RGB color
  //   4: CMYK color
  //   7: Multichannel
  //   8: Duotone
  //   9: Lab color
  if (stbi__get16be(s) != 3)
    return stbi__errpuc("wrong color format", "PSD is not in RGB color format");

  // Skip the Mode Data.  (It's the palette for indexed color; other info for other modes.)
  stbi__skip(s, stbi__get32be(s));

  // Skip the image resources.  (resolution, pen tool paths, etc)
  stbi__skip(s, stbi__get32be(s));

  // Skip the reserved data.
  stbi__skip(s, stbi__get32be(s));

  // Find out if the data is compressed.
  // Known values:
  //   0: no compression
  //   1: RLE compressed
  compression = stbi__get16be(s);
  if (compression > 1)
    return stbi__errpuc("bad compression", "PSD has an unknown compression format");

  // Create the destination image.
  out = (stbi_uc*)stbi__malloc(channelCount * w * h);
  if (!out) return stbi__errpuc("outofmem", "Out of memory");
  pixelCount = w * h;

  // Initialize the data to zero.
  //memset( out, 0, pixelCount * 4 );

  // Finally, the image data.
  if (compression) {
    // RLE as used by .PSD and .TIFF
    // Loop until you get the number of unpacked bytes you are expecting:
    //     Read the next source byte into n.
    //     If n is between 0 and 127 inclusive, copy the next n+1 bytes literally.
    //     Else if n is between -127 and -1 inclusive, copy the next byte -n+1 times.
    //     Else if n is 128, noop.
    // Endloop

    // The RLE-compressed data is preceeded by a 2-byte data count for each row in the data,
    // which we're going to just skip.
    stbi__skip(s, h * channelCount * 2);

    // Read the RLE data by channel.
    for (channel = 0; channel < channelCount; channel++) {
      stbi_uc* p;

      p = out + channel;
      if (channel >= channelCount) {
        // Fill this channel with default data.
        for (i = 0; i < pixelCount; i++) *p = (channel == 3 ? 255 : 0), p += channelCount;
      }
      else {
        // Read the RLE data.
        count = 0;
        while (count < pixelCount) {
          len = stbi__get8(s);
          if (len == 128) {
            // No-op.
          }
          else if (len < 128) {
            // Copy next len+1 bytes literally.
            len++;
            count += len;
            while (len) {
              *p = stbi__get8(s);
              p += channelCount;
              len--;
            }
          }
          else if (len > 128) {
            stbi_uc   val;
            // Next -len+1 bytes in the dest are replicated from next source byte.
            // (Interpret len as a negative 8-bit int.)
            len ^= 0x0FF;
            len += 2;
            val = stbi__get8(s);
            count += len;
            while (len) {
              *p = val;
              p += channelCount;
              len--;
            }
          }
        }
      }
    }

  }
  else {
    // We're at the raw image data.  It's each channel in order (Red, Green, Blue, Alpha, ...)
    // where each channel consists of an 8-bit value for each pixel in the image.

    // Read the data by channel.
    for (channel = 0; channel < channelCount; channel++) {
      stbi_uc* p;

      p = out + channel;
      if (channel > channelCount) {
        // Fill this channel with default data.
        for (i = 0; i < pixelCount; i++) *p = channel == 3 ? 255 : 0, p += channelCount;
      }
      else {
        // Read the data.
        for (i = 0; i < pixelCount; i++)
          *p = stbi__get8(s), p += channelCount;
      }
    }
  }

  if (req_comp && req_comp != channelCount) {
    out = stbi__convert_format(out, channelCount, req_comp, w, h);
    if (out == nullptr) return out; // stbi__convert_format frees input on failure
  }

  if (comp) *comp = channelCount;
  *y = h;
  *x = w;

  return out;
}
#endif

// *************************************************************************************************
// Softimage PIC loader
// by Tom Seddon
//
// See http://softimage.wiki.softimage.com/index.php/INFO:_PIC_file_format
// See http://ozviz.wasp.uwa.edu.au/~pbourke/dataformats/softimagepic/

#ifndef STBI_NO_PIC
static int stbi__pic_is4(stbi__context* s, const char* str)
{
  int i;
  for (i = 0; i < 4; ++i)
    if (stbi__get8(s) != (stbi_uc)str[i])
      return 0;

  return 1;
}

static int stbi__pic_test_core(stbi__context* s)
{
  int i;

  if (!stbi__pic_is4(s, "\x53\x80\xF6\x34"))
    return 0;

  for (i = 0; i < 84; ++i)
    stbi__get8(s);

  if (!stbi__pic_is4(s, "PICT"))
    return 0;

  return 1;
}

typedef struct
{
  stbi_uc size, type, channel;
} stbi__pic_packet;

static stbi_uc* stbi__readval(stbi__context* s, int channel, stbi_uc* dest)
{
  int mask = 0x80, i;

  for (i = 0; i < 4; ++i, mask >>= 1) {
    if (channel & mask) {
      if (stbi__at_eof(s)) return stbi__errpuc("bad file", "PIC file too short");
      dest[i] = stbi__get8(s);
    }
  }

  return dest;
}

static void stbi__copyval(int channel, stbi_uc* dest, const stbi_uc* src)
{
  int mask = 0x80, i;

  for (i = 0; i < 4; ++i, mask >>= 1)
    if (channel & mask)
      dest[i] = src[i];
}

static stbi_uc* stbi__pic_load_core(stbi__context* s, int width, int height, int* comp, stbi_uc* result)
{
  int act_comp = 0, num_packets = 0, y, chained;
  stbi__pic_packet packets[10];

  // this will (should...) cater for even some bizarre stuff like having data
   // for the same channel in multiple packets.
  do {
    stbi__pic_packet* packet;

    if (num_packets == sizeof(packets) / sizeof(packets[0]))
      return stbi__errpuc("bad format", "too many packets");

    packet = &packets[num_packets++];

    chained = stbi__get8(s);
    packet->size = stbi__get8(s);
    packet->type = stbi__get8(s);
    packet->channel = stbi__get8(s);

    act_comp |= packet->channel;

    if (stbi__at_eof(s))          return stbi__errpuc("bad file", "file too short (reading packets)");
    if (packet->size != 8)  return stbi__errpuc("bad format", "packet isn't 8bpp");
  } while (chained);

  *comp = (act_comp & 0x10 ? 4 : 3); // has alpha channel?

  for (y = 0; y < height; ++y) {
    int packet_idx;

    for (packet_idx = 0; packet_idx < num_packets; ++packet_idx) {
      stbi__pic_packet* packet = &packets[packet_idx];
      stbi_uc* dest = result + y * width * 4;

      switch (packet->type) {
      default:
        return stbi__errpuc("bad format", "packet has bad compression type");

      case 0: {//uncompressed
        int x;

        for (x = 0; x < width; ++x, dest += 4)
          if (!stbi__readval(s, packet->channel, dest))
            return 0;
        break;
      }

      case 1://Pure RLE
      {
        int left = width, i;

        while (left > 0) {
          stbi_uc count, value[4];

          count = stbi__get8(s);
          if (stbi__at_eof(s))   return stbi__errpuc("bad file", "file too short (pure read count)");

          if (count > left)
            count = (stbi_uc)left;

          if (!stbi__readval(s, packet->channel, value))  return 0;

          for (i = 0; i < count; ++i, dest += 4)
            stbi__copyval(packet->channel, dest, value);
          left -= count;
        }
      }
      break;

      case 2: {//Mixed RLE
        int left = width;
        while (left > 0) {
          int count = stbi__get8(s), i;
          if (stbi__at_eof(s))  return stbi__errpuc("bad file", "file too short (mixed read count)");

          if (count >= 128) { // Repeated
            stbi_uc value[4];

            if (count == 128)
              count = stbi__get16be(s);
            else
              count -= 127;
            if (count > left)
              return stbi__errpuc("bad file", "scanline overrun");

            if (!stbi__readval(s, packet->channel, value))
              return 0;

            for (i = 0; i < count; ++i, dest += 4)
              stbi__copyval(packet->channel, dest, value);
          }
          else { // Raw
            ++count;
            if (count > left) return stbi__errpuc("bad file", "scanline overrun");

            for (i = 0; i < count; ++i, dest += 4)
              if (!stbi__readval(s, packet->channel, dest))
                return 0;
          }
          left -= count;
        }
        break;
      }
      }
    }
  }

  return result;
}

static stbi_uc* stbi__pic_load(stbi__context* s, int* px, int* py, int* comp, int req_comp)
{
  stbi_uc* result;
  int i, x, y;

  for (i = 0; i < 92; ++i)
    stbi__get8(s);

  x = stbi__get16be(s);
  y = stbi__get16be(s);
  if (stbi__at_eof(s))  return stbi__errpuc("bad file", "file too short (pic header)");
  if ((1 << 28) / x < y) return stbi__errpuc("too large", "Image too large to decode");

  stbi__get32be(s); //skip `ratio'
  stbi__get16be(s); //skip `fields'
  stbi__get16be(s); //skip `pad'

  // intermediate buffer is RGBA
  result = (stbi_uc*)stbi__malloc(x * y * 4);
  memset(result, 0xff, x * y * 4);

  if (!stbi__pic_load_core(s, x, y, comp, result)) {
    STBI_FREE(result);
    result = 0;
  }
  *px = x;
  *py = y;
  if (req_comp == 0) req_comp = *comp;
  result = stbi__convert_format(result, 4, req_comp, x, y);

  return result;
}

static int stbi__pic_test(stbi__context* s)
{
  int r = stbi__pic_test_core(s);
  stbi__rewind(s);
  return r;
}
#endif

// *************************************************************************************************
// GIF loader -- public domain by Jean-Marc Lienher -- simplified/shrunk by stb

#ifndef STBI_NO_GIF
typedef struct
{
  stbi__int16 prefix;
  stbi_uc first;
  stbi_uc suffix;
} stbi__gif_lzw;

typedef struct
{
  int w, h;
  stbi_uc* out;                 // output buffer (always 4 components)
  int flags, bgindex, ratio, transparent, eflags;
  stbi_uc  pal[256][4];
  stbi_uc lpal[256][4];
  stbi__gif_lzw codes[4096];
  stbi_uc* color_table;
  int parse, step;
  int lflags;
  int start_x, start_y;
  int max_x, max_y;
  int cur_x, cur_y;
  int line_size;
} stbi__gif;

static int stbi__gif_test_raw(stbi__context* s)
{
  int sz;
  if (stbi__get8(s) != 'G' || stbi__get8(s) != 'I' || stbi__get8(s) != 'F' || stbi__get8(s) != '8') return 0;
  sz = stbi__get8(s);
  if (sz != '9' && sz != '7') return 0;
  if (stbi__get8(s) != 'a') return 0;
  return 1;
}

static int stbi__gif_test(stbi__context* s)
{
  int r = stbi__gif_test_raw(s);
  stbi__rewind(s);
  return r;
}

static void stbi__gif_parse_colortable(stbi__context* s, stbi_uc pal[256][4], int num_entries, int transp)
{
  int i;
  for (i = 0; i < num_entries; ++i) {
    pal[i][2] = stbi__get8(s);
    pal[i][1] = stbi__get8(s);
    pal[i][0] = stbi__get8(s);
    pal[i][3] = transp == i ? 0 : 255;
  }
}

static int stbi__gif_header(stbi__context* s, stbi__gif* g, int* comp, int is_info)
{
  stbi_uc version;
  if (stbi__get8(s) != 'G' || stbi__get8(s) != 'I' || stbi__get8(s) != 'F' || stbi__get8(s) != '8')
    return stbi__err("not GIF", "Corrupt GIF");

  version = stbi__get8(s);
  if (version != '7' && version != '9')    return stbi__err("not GIF", "Corrupt GIF");
  if (stbi__get8(s) != 'a')                return stbi__err("not GIF", "Corrupt GIF");

  stbi__g_failure_reason = "";
  g->w = stbi__get16le(s);
  g->h = stbi__get16le(s);
  g->flags = stbi__get8(s);
  g->bgindex = stbi__get8(s);
  g->ratio = stbi__get8(s);
  g->transparent = -1;

  if (comp != 0) *comp = 4;  // can't actually tell whether it's 3 or 4 until we parse the comments

  if (is_info) return 1;

  if (g->flags & 0x80)
    stbi__gif_parse_colortable(s, g->pal, 2 << (g->flags & 7), -1);

  return 1;
}

static int stbi__gif_info_raw(stbi__context* s, int* x, int* y, int* comp)
{
  stbi__gif g;
  if (!stbi__gif_header(s, &g, comp, 1)) {
    stbi__rewind(s);
    return 0;
  }
  if (x) *x = g.w;
  if (y) *y = g.h;
  return 1;
}

static void stbi__out_gif_code(stbi__gif* g, stbi__uint16 code)
{
  stbi_uc* p, * c;

  // recurse to decode the prefixes, since the linked-list is backwards,
  // and working backwards through an interleaved image would be nasty
  if (g->codes[code].prefix >= 0)
    stbi__out_gif_code(g, g->codes[code].prefix);

  if (g->cur_y >= g->max_y) return;

  p = &g->out[g->cur_x + g->cur_y];
  c = &g->color_table[g->codes[code].suffix * 4];

  if (c[3] >= 128) {
    p[0] = c[2];
    p[1] = c[1];
    p[2] = c[0];
    p[3] = c[3];
  }
  g->cur_x += 4;

  if (g->cur_x >= g->max_x) {
    g->cur_x = g->start_x;
    g->cur_y += g->step;

    while (g->cur_y >= g->max_y && g->parse > 0) {
      g->step = (1 << g->parse) * g->line_size;
      g->cur_y = g->start_y + (g->step >> 1);
      --g->parse;
    }
  }
}

static stbi_uc* stbi__process_gif_raster(stbi__context* s, stbi__gif* g)
{
  stbi_uc lzw_cs;
  stbi__int32 len, code;
  stbi__uint32 first;
  stbi__int32 codesize, codemask, avail, oldcode, bits, valid_bits, clear;
  stbi__gif_lzw* p;

  lzw_cs = stbi__get8(s);
  if (lzw_cs > 12) return nullptr;
  clear = 1 << lzw_cs;
  first = 1;
  codesize = lzw_cs + 1;
  codemask = (1 << codesize) - 1;
  bits = 0;
  valid_bits = 0;
  for (code = 0; code < clear; code++) {
    g->codes[code].prefix = -1;
    g->codes[code].first = (stbi_uc)code;
    g->codes[code].suffix = (stbi_uc)code;
  }

  // support no starting clear code
  avail = clear + 2;
  oldcode = -1;

  len = 0;
  for (;;) {
    if (valid_bits < codesize) {
      if (len == 0) {
        len = stbi__get8(s); // start new block
        if (len == 0)
          return g->out;
      }
      --len;
      bits |= (stbi__int32)stbi__get8(s) << valid_bits;
      valid_bits += 8;
    }
    else {
      stbi__int32 code2 = bits & codemask;
      bits >>= codesize;
      valid_bits -= codesize;
      // @OPTIMIZE: is there some way we can accelerate the non-clear path?
      if (code2 == clear) {  // clear code
        codesize = lzw_cs + 1;
        codemask = (1 << codesize) - 1;
        avail = clear + 2;
        oldcode = -1;
        first = 0;
      }
      else if (code2 == clear + 1) { // end of stream code
        stbi__skip(s, len);
        while ((len = stbi__get8(s)) > 0)
          stbi__skip(s, len);
        return g->out;
      }
      else if (code2 <= avail) {
        if (first) return stbi__errpuc("no clear code", "Corrupt GIF");

        if (oldcode >= 0) {
          p = &g->codes[avail++];
          if (avail > 4096)        return stbi__errpuc("too many codes", "Corrupt GIF");
          p->prefix = (stbi__int16)oldcode;
          p->first = g->codes[oldcode].first;
          p->suffix = (code2 == avail) ? p->first : g->codes[code2].first;
        }
        else if (code2 == avail)
          return stbi__errpuc("illegal code in raster", "Corrupt GIF");

        stbi__out_gif_code(g, (stbi__uint16)code2);

        if ((avail & codemask) == 0 && avail <= 0x0FFF) {
          codesize++;
          codemask = (1 << codesize) - 1;
        }

        oldcode = code2;
      }
      else {
        return stbi__errpuc("illegal code in raster", "Corrupt GIF");
      }
    }
  }
}

static void stbi__fill_gif_background(stbi__gif* g)
{
  int i;
  stbi_uc* c = g->pal[g->bgindex];
  // @OPTIMIZE: write a dword at a time
  for (i = 0; i < g->w * g->h * 4; i += 4) {
    stbi_uc* p = &g->out[i];
    p[0] = c[2];
    p[1] = c[1];
    p[2] = c[0];
    p[3] = c[3];
  }
}

// this function is designed to support animated gifs, although stb_image doesn't support it
static stbi_uc* stbi__gif_load_next(stbi__context* s, stbi__gif* g, int* comp, int req_comp)
{
  int i;
  stbi_uc* old_out = 0;

  if (g->out == 0) {
    if (!stbi__gif_header(s, g, comp, 0))     return 0; // stbi__g_failure_reason set by stbi__gif_header
    g->out = (stbi_uc*)stbi__malloc(4 * g->w * g->h);
    if (g->out == 0)                      return stbi__errpuc("outofmem", "Out of memory");
    stbi__fill_gif_background(g);
  }
  else {
    // animated-gif-only path
    if (((g->eflags & 0x1C) >> 2) == 3) {
      old_out = g->out;
      g->out = (stbi_uc*)stbi__malloc(4 * g->w * g->h);
      if (g->out == 0)                   return stbi__errpuc("outofmem", "Out of memory");
      memcpy(g->out, old_out, g->w * g->h * 4);
    }
  }

  for (;;) {
    switch (stbi__get8(s)) {
    case 0x2C: /* Image Descriptor */
    {
      stbi__int32 x, y, w, h;
      stbi_uc* o;

      x = stbi__get16le(s);
      y = stbi__get16le(s);
      w = stbi__get16le(s);
      h = stbi__get16le(s);
      if (((x + w) > (g->w)) || ((y + h) > (g->h)))
        return stbi__errpuc("bad Image Descriptor", "Corrupt GIF");

      g->line_size = g->w * 4;
      g->start_x = x * 4;
      g->start_y = y * g->line_size;
      g->max_x = g->start_x + w * 4;
      g->max_y = g->start_y + h * g->line_size;
      g->cur_x = g->start_x;
      g->cur_y = g->start_y;

      g->lflags = stbi__get8(s);

      if (g->lflags & 0x40) {
        g->step = 8 * g->line_size; // first interlaced spacing
        g->parse = 3;
      }
      else {
        g->step = g->line_size;
        g->parse = 0;
      }

      if (g->lflags & 0x80) {
        stbi__gif_parse_colortable(s, g->lpal, 2 << (g->lflags & 7), g->eflags & 0x01 ? g->transparent : -1);
        g->color_table = (stbi_uc*)g->lpal;
      }
      else if (g->flags & 0x80) {
        for (i = 0; i < 256; ++i)  // @OPTIMIZE: stbi__jpeg_reset only the previous transparent
          g->pal[i][3] = 255;
        if (g->transparent >= 0 && (g->eflags & 0x01))
          g->pal[g->transparent][3] = 0;
        g->color_table = (stbi_uc*)g->pal;
      }
      else
        return stbi__errpuc("missing color table", "Corrupt GIF");

      o = stbi__process_gif_raster(s, g);
      if (o == nullptr) return nullptr;

      if (req_comp && req_comp != 4)
        o = stbi__convert_format(o, 4, req_comp, g->w, g->h);
      return o;
    }

    case 0x21: // Comment Extension.
    {
      int len;
      if (stbi__get8(s) == 0xF9) { // Graphic Control Extension.
        len = stbi__get8(s);
        if (len == 4) {
          g->eflags = stbi__get8(s);
          stbi__get16le(s); // delay
          g->transparent = stbi__get8(s);
        }
        else {
          stbi__skip(s, len);
          break;
        }
      }
      while ((len = stbi__get8(s)) != 0)
        stbi__skip(s, len);
      break;
    }

    case 0x3B: // gif stream termination code
      return (stbi_uc*)s; // using '1' causes warning on some compilers

    default:
      return stbi__errpuc("unknown code", "Corrupt GIF");
    }
  }
}

static stbi_uc* stbi__gif_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  stbi_uc* u = 0;
  stbi__gif g;
  memset(&g, 0, sizeof(g));

  u = stbi__gif_load_next(s, &g, comp, req_comp);
  if (u == (stbi_uc*)s) u = 0;  // end of animated gif marker
  if (u) {
    *x = g.w;
    *y = g.h;
  }

  return u;
}

static int stbi__gif_info(stbi__context* s, int* x, int* y, int* comp)
{
  return stbi__gif_info_raw(s, x, y, comp);
}
#endif

// *************************************************************************************************
// Radiance RGBE HDR loader
// originally by Nicolas Schulz
#ifndef STBI_NO_HDR
static int stbi__hdr_test_core(stbi__context* s)
{
  const char* signature = "#?RADIANCE\n";
  int i;
  for (i = 0; signature[i]; ++i)
    if (stbi__get8(s) != signature[i])
      return 0;
  return 1;
}

static int stbi__hdr_test(stbi__context* s)
{
  int r = stbi__hdr_test_core(s);
  stbi__rewind(s);
  return r;
}

#define STBI__HDR_BUFLEN  1024
static char* stbi__hdr_gettoken(stbi__context* z, char* buffer)
{
  int len = 0;
  char c = '\0';

  c = (char)stbi__get8(z);

  while (!stbi__at_eof(z) && c != '\n') {
    buffer[len++] = c;
    if (len == STBI__HDR_BUFLEN - 1) {
      // flush to end of line
      while (!stbi__at_eof(z) && stbi__get8(z) != '\n')
        ;
      break;
    }
    c = (char)stbi__get8(z);
  }

  buffer[len] = 0;
  return buffer;
}

static void stbi__hdr_convert(float* output, stbi_uc* input, int req_comp)
{
  if (input[3] != 0) {
    float f1;
    // Exponent
    f1 = (float)ldexp(1.0f, input[3] - (int)(128 + 8));
    if (req_comp <= 2)
      output[0] = (input[0] + input[1] + input[2]) * f1 / 3;
    else {
      output[0] = input[0] * f1;
      output[1] = input[1] * f1;
      output[2] = input[2] * f1;
    }
    if (req_comp == 2) output[1] = 1;
    if (req_comp == 4) output[3] = 1;
  }
  else {
    switch (req_comp) {
    case 4: output[3] = 1; /* fallthrough */
    case 3: output[0] = output[1] = output[2] = 0;
      break;
    case 2: output[1] = 1; /* fallthrough */
    case 1: output[0] = 0;
      break;
    }
  }
}

static float* stbi__hdr_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  char buffer[STBI__HDR_BUFLEN];
  char* token;
  int valid = 0;
  int width, height;
  stbi_uc* scanline;
  float* hdr_data;
  int len;
  unsigned char count, value;
  int i, j, k, c1, c2, z;


  // Check identifier
  if (strcmp(stbi__hdr_gettoken(s, buffer), "#?RADIANCE") != 0)
    return stbi__errpf("not HDR", "Corrupt HDR image");

  // Parse header
  for (;;) {
    token = stbi__hdr_gettoken(s, buffer);
    if (token[0] == 0) break;
    if (strcmp(token, "FORMAT=32-bit_rle_rgbe") == 0) valid = 1;
  }

  if (!valid)    return stbi__errpf("unsupported format", "Unsupported HDR format");

  // Parse width and height
  // can't use sscanf() if we're not using stdio!
  token = stbi__hdr_gettoken(s, buffer);
  if (strncmp(token, "-Y ", 3))  return stbi__errpf("unsupported data layout", "Unsupported HDR format");
  token += 3;
  height = (int)strtol(token, &token, 10);
  while (*token == ' ') ++token;
  if (strncmp(token, "+X ", 3))  return stbi__errpf("unsupported data layout", "Unsupported HDR format");
  token += 3;
  width = (int)strtol(token, nullptr, 10);

  *x = width;
  *y = height;

  if (comp) *comp = 3;
  if (req_comp == 0) req_comp = 3;

  // Read data
  hdr_data = (float*)stbi__malloc(height * width * req_comp * sizeof(float));

  // Load image data
  // image data is stored as some number of sca
  if (width < 8 || width >= 32768) {
    // Read flat data
    for (j = 0; j < height; ++j) {
      for (i = 0; i < width; ++i) {
        stbi_uc rgbe[4];
      main_decode_loop:
        stbi__getn(s, rgbe, 4);
        stbi__hdr_convert(hdr_data + j * width * req_comp + i * req_comp, rgbe, req_comp);
      }
    }
  }
  else {
    // Read RLE-encoded data
    scanline = nullptr;

    for (j = 0; j < height; ++j) {
      c1 = stbi__get8(s);
      c2 = stbi__get8(s);
      len = stbi__get8(s);
      if (c1 != 2 || c2 != 2 || (len & 0x80)) {
        // not run-length encoded, so we have to actually use THIS data as a decoded
        // pixel (note this can't be a valid pixel--one of RGB must be >= 128)
        stbi_uc rgbe[4];
        rgbe[0] = (stbi_uc)c1;
        rgbe[1] = (stbi_uc)c2;
        rgbe[2] = (stbi_uc)len;
        rgbe[3] = (stbi_uc)stbi__get8(s);
        stbi__hdr_convert(hdr_data, rgbe, req_comp);
        i = 1;
        j = 0;
        STBI_FREE(scanline);
        goto main_decode_loop; // yes, this makes no sense
      }
      len <<= 8;
      len |= stbi__get8(s);
      if (len != width) { STBI_FREE(hdr_data); STBI_FREE(scanline); return stbi__errpf("invalid decoded scanline length", "corrupt HDR"); }
      if (scanline == nullptr) scanline = (stbi_uc*)stbi__malloc(width * 4);

      for (k = 0; k < 4; ++k) {
        i = 0;
        while (i < width) {
          count = stbi__get8(s);
          if (count > 128) {
            // Run
            value = stbi__get8(s);
            count -= 128;
            for (z = 0; z < count; ++z)
              scanline[i++ * 4 + k] = value;
          }
          else {
            // Dump
            for (z = 0; z < count; ++z)
              scanline[i++ * 4 + k] = stbi__get8(s);
          }
        }
      }
      for (i = 0; i < width; ++i)
        stbi__hdr_convert(hdr_data + (j * width + i) * req_comp, scanline + i * 4, req_comp);
    }
    STBI_FREE(scanline);
  }

  return hdr_data;
}

static int stbi__hdr_info(stbi__context* s, int* x, int* y, int* comp)
{
  char buffer[STBI__HDR_BUFLEN];
  char* token;
  int valid = 0;

  if (strcmp(stbi__hdr_gettoken(s, buffer), "#?RADIANCE") != 0) {
    stbi__rewind(s);
    return 0;
  }

  for (;;) {
    token = stbi__hdr_gettoken(s, buffer);
    if (token[0] == 0) break;
    if (strcmp(token, "FORMAT=32-bit_rle_rgbe") == 0) valid = 1;
  }

  if (!valid) {
    stbi__rewind(s);
    return 0;
  }
  token = stbi__hdr_gettoken(s, buffer);
  if (strncmp(token, "-Y ", 3)) {
    stbi__rewind(s);
    return 0;
  }
  token += 3;
  *y = (int)strtol(token, &token, 10);
  while (*token == ' ') ++token;
  if (strncmp(token, "+X ", 3)) {
    stbi__rewind(s);
    return 0;
  }
  token += 3;
  *x = (int)strtol(token, nullptr, 10);
  *comp = 3;
  return 1;
}
#endif // STBI_NO_HDR

#ifndef STBI_NO_BMP
static int stbi__bmp_info(stbi__context* s, int* x, int* y, int* comp)
{
  int hsz;
  if (stbi__get8(s) != 'B' || stbi__get8(s) != 'M') {
    stbi__rewind(s);
    return 0;
  }
  stbi__skip(s, 12);
  hsz = stbi__get32le(s);
  if (hsz != 12 && hsz != 40 && hsz != 56 && hsz != 108 && hsz != 124) {
    stbi__rewind(s);
    return 0;
  }
  if (hsz == 12) {
    *x = stbi__get16le(s);
    *y = stbi__get16le(s);
  }
  else {
    *x = stbi__get32le(s);
    *y = stbi__get32le(s);
  }
  if (stbi__get16le(s) != 1) {
    stbi__rewind(s);
    return 0;
  }
  *comp = stbi__get16le(s) / 8;
  return 1;
}
#endif

#ifndef STBI_NO_PSD
static int stbi__psd_info(stbi__context* s, int* x, int* y, int* comp)
{
  int channelCount;
  if (stbi__get32be(s) != 0x38425053) {
    stbi__rewind(s);
    return 0;
  }
  if (stbi__get16be(s) != 1) {
    stbi__rewind(s);
    return 0;
  }
  stbi__skip(s, 6);
  channelCount = stbi__get16be(s);
  if (channelCount < 0 || channelCount > 16) {
    stbi__rewind(s);
    return 0;
  }
  *y = stbi__get32be(s);
  *x = stbi__get32be(s);
  if (stbi__get16be(s) != 8) {
    stbi__rewind(s);
    return 0;
  }
  if (stbi__get16be(s) != 3) {
    stbi__rewind(s);
    return 0;
  }
  *comp = 4;
  return 1;
}
#endif

#ifndef STBI_NO_PIC
static int stbi__pic_info(stbi__context* s, int* x, int* y, int* comp)
{
  int act_comp = 0, num_packets = 0, chained;
  stbi__pic_packet packets[10];

  stbi__skip(s, 92);

  *x = stbi__get16be(s);
  *y = stbi__get16be(s);
  if (stbi__at_eof(s))  return 0;
  if ((*x) != 0 && (1 << 28) / (*x) < (*y)) {
    stbi__rewind(s);
    return 0;
  }

  stbi__skip(s, 8);

  do {
    stbi__pic_packet* packet;

    if (num_packets == sizeof(packets) / sizeof(packets[0]))
      return 0;

    packet = &packets[num_packets++];
    chained = stbi__get8(s);
    packet->size = stbi__get8(s);
    packet->type = stbi__get8(s);
    packet->channel = stbi__get8(s);
    act_comp |= packet->channel;

    if (stbi__at_eof(s)) {
      stbi__rewind(s);
      return 0;
    }
    if (packet->size != 8) {
      stbi__rewind(s);
      return 0;
    }
  } while (chained);

  *comp = (act_comp & 0x10 ? 4 : 3);

  return 1;
}
#endif

// *************************************************************************************************
// Portable Gray Map and Portable Pixel Map loader
// by Ken Miller
//
// PGM: http://netpbm.sourceforge.net/doc/pgm.html
// PPM: http://netpbm.sourceforge.net/doc/ppm.html
//
// Known limitations:
//    Does not support comments in the header section
//    Does not support ASCII image data (formats P2 and P3)
//    Does not support 16-bit-per-channel

#ifndef STBI_NO_PNM

static int      stbi__pnm_test(stbi__context* s)
{
  char p, t;
  p = (char)stbi__get8(s);
  t = (char)stbi__get8(s);
  if (p != 'P' || (t != '5' && t != '6')) {
    stbi__rewind(s);
    return 0;
  }
  return 1;
}

static stbi_uc* stbi__pnm_load(stbi__context* s, int* x, int* y, int* comp, int req_comp)
{
  stbi_uc* out;
  if (!stbi__pnm_info(s, (int*)&s->img_x, (int*)&s->img_y, (int*)&s->img_n))
    return 0;
  *x = s->img_x;
  *y = s->img_y;
  *comp = s->img_n;

  out = (stbi_uc*)stbi__malloc(s->img_n * s->img_x * s->img_y);
  if (!out) return stbi__errpuc("outofmem", "Out of memory");
  stbi__getn(s, out, s->img_n * s->img_x * s->img_y);

  if (req_comp && req_comp != s->img_n) {
    out = stbi__convert_format(out, s->img_n, req_comp, s->img_x, s->img_y);
    if (out == nullptr) return out; // stbi__convert_format frees input on failure
  }
  return out;
}

static int      stbi__pnm_isspace(char c)
{
  return c == ' ' || c == '\t' || c == '\n' || c == '\v' || c == '\f' || c == '\r';
}

static void     stbi__pnm_skip_whitespace(stbi__context* s, char* c)
{
  while (!stbi__at_eof(s) && stbi__pnm_isspace(*c))
    *c = (char)stbi__get8(s);
}

static int      stbi__pnm_isdigit(char c)
{
  return c >= '0' && c <= '9';
}

static int      stbi__pnm_getinteger(stbi__context* s, char* c)
{
  int value = 0;

  while (!stbi__at_eof(s) && stbi__pnm_isdigit(*c)) {
    value = value * 10 + (*c - '0');
    *c = (char)stbi__get8(s);
  }

  return value;
}

static int      stbi__pnm_info(stbi__context* s, int* x, int* y, int* comp)
{
  int maxv;
  char c, p, t;

  stbi__rewind(s);

  // Get identifier
  p = (char)stbi__get8(s);
  t = (char)stbi__get8(s);
  if (p != 'P' || (t != '5' && t != '6')) {
    stbi__rewind(s);
    return 0;
  }

  *comp = (t == '6') ? 3 : 1;  // '5' is 1-component .pgm; '6' is 3-component .ppm

  c = (char)stbi__get8(s);
  stbi__pnm_skip_whitespace(s, &c);

  *x = stbi__pnm_getinteger(s, &c); // read width
  stbi__pnm_skip_whitespace(s, &c);

  *y = stbi__pnm_getinteger(s, &c); // read height
  stbi__pnm_skip_whitespace(s, &c);

  maxv = stbi__pnm_getinteger(s, &c);  // read max value

  if (maxv > 255)
    return stbi__err("max value > 255", "PPM image not 8-bit");
  else
    return 1;
}
#endif

static int stbi__info_main(stbi__context* s, int* x, int* y, int* comp)
{
#ifndef STBI_NO_JPEG
  if (stbi__jpeg_info(s, x, y, comp)) return 1;
#endif

#ifndef STBI_NO_PNG
  if (stbi__png_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_GIF
  if (stbi__gif_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_BMP
  if (stbi__bmp_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_PSD
  if (stbi__psd_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_PIC
  if (stbi__pic_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_PNM
  if (stbi__pnm_info(s, x, y, comp))  return 1;
#endif

#ifndef STBI_NO_HDR
  if (stbi__hdr_info(s, x, y, comp))  return 1;
#endif

  // test tga last because it's a crappy test!
#ifndef STBI_NO_TGA
  if (stbi__tga_info(s, x, y, comp))
    return 1;
#endif
  return stbi__err("unknown image type", "Image not of any known type, or corrupt");
}

#ifndef STBI_NO_STDIO
STBIDEF int stbi_info(char const* filename, int* x, int* y, int* comp)
{
  FILE* f = stbi__fopen(filename, "rb");
  int result;
  if (!f) return stbi__err("can't fopen", "Unable to open file");
  result = stbi_info_from_file(f, x, y, comp);
  fclose(f);
  return result;
}

STBIDEF int stbi_info_from_file(FILE* f, int* x, int* y, int* comp)
{
  int r;
  stbi__context s;
  long pos = ftell(f);
  stbi__start_file(&s, f);
  r = stbi__info_main(&s, x, y, comp);
  fseek(f, pos, SEEK_SET);
  return r;
}
#endif // !STBI_NO_STDIO

STBIDEF int stbi_info_from_memory(stbi_uc const* buffer, int len, int* x, int* y, int* comp)
{
  stbi__context s;
  stbi__start_mem(&s, buffer, len);
  return stbi__info_main(&s, x, y, comp);
}

STBIDEF int stbi_info_from_callbacks(stbi_io_callbacks const* c, void* user, int* x, int* y, int* comp)
{
  stbi__context s;
  stbi__start_callbacks(&s, (stbi_io_callbacks*)c, user);
  return stbi__info_main(&s, x, y, comp);
}


#endif // STB_IMAGE_IMPLEMENTATION





#ifndef PVR_HELPER_H
#define PVR_HELPER_H

// Taken from PowerVR SDK

/*!***************************************************************************
 Describes the header of a PVR header-texture
 *****************************************************************************/
typedef struct
{
  unsigned int dwHeaderSize;			/*!< size of the structure */
  unsigned int dwHeight;				/*!< height of surface to be created */
  unsigned int dwWidth;				/*!< width of input surface */
  unsigned int dwMipMapCount;			/*!< number of mip-map levels requested */
  unsigned int dwpfFlags;				/*!< pixel format flags */
  unsigned int dwTextureDataSize;		/*!< Total size in bytes */
  unsigned int dwBitCount;			/*!< number of bits per pixel  */
  unsigned int dwRBitMask;			/*!< mask for red bit */
  unsigned int dwGBitMask;			/*!< mask for green bits */
  unsigned int dwBBitMask;			/*!< mask for blue bits */
  unsigned int dwAlphaBitMask;		/*!< mask for alpha channel */
  unsigned int dwPVR;					/*!< magic number identifying pvr file */
  unsigned int dwNumSurfs;			/*!< the number of surfaces present in the pvr */
} PVR_Texture_Header;

/*****************************************************************************
 * ENUMS
 *****************************************************************************/

enum PixelType
{
  MGLPT_ARGB_4444 = 0x00,
  MGLPT_ARGB_1555,
  MGLPT_RGB_565,
  MGLPT_RGB_555,
  MGLPT_RGB_888,
  MGLPT_ARGB_8888,
  MGLPT_ARGB_8332,
  MGLPT_I_8,
  MGLPT_AI_88,
  MGLPT_1_BPP,
  MGLPT_VY1UY0,
  MGLPT_Y1VY0U,
  MGLPT_PVRTC2,
  MGLPT_PVRTC4,
  MGLPT_PVRTC2_2,
  MGLPT_PVRTC2_4,

  OGL_RGBA_4444 = 0x10,
  OGL_RGBA_5551,
  OGL_RGBA_8888,
  OGL_RGB_565,
  OGL_RGB_555,
  OGL_RGB_888,
  OGL_I_8,
  OGL_AI_88,
  OGL_PVRTC2,
  OGL_PVRTC4,

  // OGL_BGRA_8888 extension
  OGL_BGRA_8888,

  D3D_DXT1 = 0x20,
  D3D_DXT2,
  D3D_DXT3,
  D3D_DXT4,
  D3D_DXT5,

  D3D_RGB_332,
  D3D_AI_44,
  D3D_LVU_655,
  D3D_XLVU_8888,
  D3D_QWVU_8888,

  //10 bits per channel
  D3D_ABGR_2101010,
  D3D_ARGB_2101010,
  D3D_AWVU_2101010,

  //16 bits per channel
  D3D_GR_1616,
  D3D_VU_1616,
  D3D_ABGR_16161616,

  //HDR formats
  D3D_R16F,
  D3D_GR_1616F,
  D3D_ABGR_16161616F,

  //32 bits per channel
  D3D_R32F,
  D3D_GR_3232F,
  D3D_ABGR_32323232F,

  // Ericsson
  ETC_RGB_4BPP,
  ETC_RGBA_EXPLICIT,
  ETC_RGBA_INTERPOLATED,

  // DX10


  ePT_DX10_R32G32B32A32_FLOAT = 0x50,
  ePT_DX10_R32G32B32A32_UINT,
  ePT_DX10_R32G32B32A32_SINT,

  ePT_DX10_R32G32B32_FLOAT,
  ePT_DX10_R32G32B32_UINT,
  ePT_DX10_R32G32B32_SINT,

  ePT_DX10_R16G16B16A16_FLOAT,
  ePT_DX10_R16G16B16A16_UNORM,
  ePT_DX10_R16G16B16A16_UINT,
  ePT_DX10_R16G16B16A16_SNORM,
  ePT_DX10_R16G16B16A16_SINT,

  ePT_DX10_R32G32_FLOAT,
  ePT_DX10_R32G32_UINT,
  ePT_DX10_R32G32_SINT,

  ePT_DX10_R10G10B10A2_UNORM,
  ePT_DX10_R10G10B10A2_UINT,

  ePT_DX10_R11G11B10_FLOAT,

  ePT_DX10_R8G8B8A8_UNORM,
  ePT_DX10_R8G8B8A8_UNORM_SRGB,
  ePT_DX10_R8G8B8A8_UINT,
  ePT_DX10_R8G8B8A8_SNORM,
  ePT_DX10_R8G8B8A8_SINT,

  ePT_DX10_R16G16_FLOAT,
  ePT_DX10_R16G16_UNORM,
  ePT_DX10_R16G16_UINT,
  ePT_DX10_R16G16_SNORM,
  ePT_DX10_R16G16_SINT,

  ePT_DX10_R32_FLOAT,
  ePT_DX10_R32_UINT,
  ePT_DX10_R32_SINT,

  ePT_DX10_R8G8_UNORM,
  ePT_DX10_R8G8_UINT,
  ePT_DX10_R8G8_SNORM,
  ePT_DX10_R8G8_SINT,

  ePT_DX10_R16_FLOAT,
  ePT_DX10_R16_UNORM,
  ePT_DX10_R16_UINT,
  ePT_DX10_R16_SNORM,
  ePT_DX10_R16_SINT,

  ePT_DX10_R8_UNORM,
  ePT_DX10_R8_UINT,
  ePT_DX10_R8_SNORM,
  ePT_DX10_R8_SINT,

  ePT_DX10_A8_UNORM,
  ePT_DX10_R1_UNORM,
  ePT_DX10_R9G9B9E5_SHAREDEXP,
  ePT_DX10_R8G8_B8G8_UNORM,
  ePT_DX10_G8R8_G8B8_UNORM,

  ePT_DX10_BC1_UNORM,
  ePT_DX10_BC1_UNORM_SRGB,

  ePT_DX10_BC2_UNORM,
  ePT_DX10_BC2_UNORM_SRGB,

  ePT_DX10_BC3_UNORM,
  ePT_DX10_BC3_UNORM_SRGB,

  ePT_DX10_BC4_UNORM,
  ePT_DX10_BC4_SNORM,

  ePT_DX10_BC5_UNORM,
  ePT_DX10_BC5_SNORM,

  //ePT_DX10_B5G6R5_UNORM,			// defined but obsolete - won't actually load in DX10
  //ePT_DX10_B5G5R5A1_UNORM,
  //ePT_DX10_B8G8R8A8_UNORM,
  //ePT_DX10_B8G8R8X8_UNORM,

  // OpenVG

  /* RGB{A,X} channel ordering */
  ePT_VG_sRGBX_8888 = 0x90,
  ePT_VG_sRGBA_8888,
  ePT_VG_sRGBA_8888_PRE,
  ePT_VG_sRGB_565,
  ePT_VG_sRGBA_5551,
  ePT_VG_sRGBA_4444,
  ePT_VG_sL_8,
  ePT_VG_lRGBX_8888,
  ePT_VG_lRGBA_8888,
  ePT_VG_lRGBA_8888_PRE,
  ePT_VG_lL_8,
  ePT_VG_A_8,
  ePT_VG_BW_1,

  /* {A,X}RGB channel ordering */
  ePT_VG_sXRGB_8888,
  ePT_VG_sARGB_8888,
  ePT_VG_sARGB_8888_PRE,
  ePT_VG_sARGB_1555,
  ePT_VG_sARGB_4444,
  ePT_VG_lXRGB_8888,
  ePT_VG_lARGB_8888,
  ePT_VG_lARGB_8888_PRE,

  /* BGR{A,X} channel ordering */
  ePT_VG_sBGRX_8888,
  ePT_VG_sBGRA_8888,
  ePT_VG_sBGRA_8888_PRE,
  ePT_VG_sBGR_565,
  ePT_VG_sBGRA_5551,
  ePT_VG_sBGRA_4444,
  ePT_VG_lBGRX_8888,
  ePT_VG_lBGRA_8888,
  ePT_VG_lBGRA_8888_PRE,

  /* {A,X}BGR channel ordering */
  ePT_VG_sXBGR_8888,
  ePT_VG_sABGR_8888,
  ePT_VG_sABGR_8888_PRE,
  ePT_VG_sABGR_1555,
  ePT_VG_sABGR_4444,
  ePT_VG_lXBGR_8888,
  ePT_VG_lABGR_8888,
  ePT_VG_lABGR_8888_PRE,

  // max cap for iterating
  END_OF_PIXEL_TYPES,

  MGLPT_NOTYPE = 0xff

};

#define PVRTEX_MIPMAP		(1<<8)		// has mip map levels
#define PVRTEX_TWIDDLE		(1<<9)		// is twiddled
#define PVRTEX_BUMPMAP		(1<<10)		// has normals encoded for a bump map
#define PVRTEX_TILING		(1<<11)		// is bordered for tiled pvr
#define PVRTEX_CUBEMAP		(1<<12)		// is a cubemap/skybox
#define PVRTEX_FALSEMIPCOL	(1<<13)		//
#define PVRTEX_VOLUME		(1<<14)
#define PVRTEX_PIXELTYPE	0xff			// pixel type is always in the last 16bits of the flags
#define PVRTEX_IDENTIFIER	0x21525650	// the pvr identifier is the characters 'P','V','R'

#define PVRTEX_V1_HEADER_SIZE 44			// old header size was 44 for identification purposes

#define PVRTC2_MIN_TEXWIDTH		16
#define PVRTC2_MIN_TEXHEIGHT	8
#define PVRTC4_MIN_TEXWIDTH		8
#define PVRTC4_MIN_TEXHEIGHT	8
#define ETC_MIN_TEXWIDTH		4
#define ETC_MIN_TEXHEIGHT		4
#define DXT_MIN_TEXWIDTH		4
#define DXT_MIN_TEXHEIGHT		4

#endif




typedef struct {
  char aName[6];
  unsigned short iBlank;
  unsigned char iPaddedWidthMSB;
  unsigned char iPaddedWidthLSB;
  unsigned char iPaddedHeightMSB;
  unsigned char iPaddedHeightLSB;
  unsigned char iWidthMSB;
  unsigned char iWidthLSB;
  unsigned char iHeightMSB;
  unsigned char iHeightLSB;
} PKMHeader;
#define PKM_HEADER_SIZE 16




/*	error reporting	*/
const char* result_string_pointer = "SOIL initialized";

/*	for loading cube maps	*/
enum {
  SOIL_CAPABILITY_UNKNOWN = -1,
  SOIL_CAPABILITY_NONE = 0,
  SOIL_CAPABILITY_PRESENT = 1
};
static int has_cubemap_capability = SOIL_CAPABILITY_UNKNOWN;
#define SOIL_TEXTURE_WRAP_R					0x8072
#define SOIL_CLAMP_TO_EDGE					0x812F
#define SOIL_NORMAL_MAP						0x8511
#define SOIL_REFLECTION_MAP					0x8512
#define SOIL_TEXTURE_CUBE_MAP				0x8513
#define SOIL_TEXTURE_BINDING_CUBE_MAP		0x8514
#define SOIL_TEXTURE_CUBE_MAP_POSITIVE_X	0x8515
#define SOIL_TEXTURE_CUBE_MAP_NEGATIVE_X	0x8516
#define SOIL_TEXTURE_CUBE_MAP_POSITIVE_Y	0x8517
#define SOIL_TEXTURE_CUBE_MAP_NEGATIVE_Y	0x8518
#define SOIL_TEXTURE_CUBE_MAP_POSITIVE_Z	0x8519
#define SOIL_TEXTURE_CUBE_MAP_NEGATIVE_Z	0x851A
#define SOIL_PROXY_TEXTURE_CUBE_MAP			0x851B
#define SOIL_MAX_CUBE_MAP_TEXTURE_SIZE		0x851C
/*	for non-power-of-two texture	*/
#define SOIL_IS_POW2( v ) ( ( v & ( v - 1 ) ) == 0 )
static int has_NPOT_capability = SOIL_CAPABILITY_PRESENT;
/*	for texture rectangles	*/
static int has_tex_rectangle_capability = SOIL_CAPABILITY_PRESENT;
#define SOIL_TEXTURE_RECTANGLE_ARB				0x84F5
#define SOIL_MAX_RECTANGLE_TEXTURE_SIZE_ARB		0x84F8
/*	for using DXT compression	*/
static int has_DXT_capability = SOIL_CAPABILITY_PRESENT;
#define SOIL_RGB_S3TC_DXT1		0x83F0
#define SOIL_RGBA_S3TC_DXT1		0x83F1
#define SOIL_RGBA_S3TC_DXT3		0x83F2
#define SOIL_RGBA_S3TC_DXT5		0x83F3
typedef void (APIENTRY* P_SOIL_GLCOMPRESSEDTEXIMAGE2DPROC) (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLint border, GLsizei imageSize, const GLvoid* data);
static P_SOIL_GLCOMPRESSEDTEXIMAGE2DPROC soilGlCompressedTexImage2D = nullptr;

typedef void (APIENTRY* P_SOIL_GLGENERATEMIPMAPPROC)(GLenum target);
static P_SOIL_GLGENERATEMIPMAPPROC soilGlGenerateMipmap = nullptr;

static int has_gen_mipmap_capability = SOIL_CAPABILITY_PRESENT;

static int has_PVR_capability = SOIL_CAPABILITY_PRESENT;
static int has_BGRA8888_capability = SOIL_CAPABILITY_UNKNOWN;
int query_BGRA8888_capability(void);
static int has_ETC1_capability = SOIL_CAPABILITY_UNKNOWN;
int query_ETC1_capability(void);

/* GL_IMG_texture_compression_pvrtc */
#define SOIL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG                      0x8C00
#define SOIL_COMPRESSED_RGB_PVRTC_2BPPV1_IMG                      0x8C01
#define SOIL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG                     0x8C02
#define SOIL_COMPRESSED_RGBA_PVRTC_2BPPV1_IMG                     0x8C03
#define SOIL_GL_ETC1_RGB8_OES                                     0x8D64

#if defined( SOIL_X11_PLATFORM ) || defined( SOIL_PLATFORM_WIN32 ) || defined( SOIL_PLATFORM_OSX )
typedef const GLubyte* (APIENTRY* P_SOIL_glGetStringiFunc) (GLenum, GLuint);
static P_SOIL_glGetStringiFunc soilGlGetStringiFunc = nullptr;

static int isAtLeastGL3()
{
  static int is_gl3 = SOIL_CAPABILITY_UNKNOWN;

  if (SOIL_CAPABILITY_UNKNOWN == is_gl3)
  {
    const char* verstr = (const char*)glGetString(GL_VERSION);
    is_gl3 = (verstr && (atoi(verstr) >= 3));
  }

  return is_gl3;
}
#endif

#ifdef SOIL_PLATFORM_WIN32
static int soilTestWinProcPointer(const PROC pTest)
{
  ptrdiff_t iTest;
  if (!pTest) return 0;
  iTest = (ptrdiff_t)pTest;
  if (iTest == 1 || iTest == 2 || iTest == 3 || iTest == -1) return 0;
  return 1;
}
#endif

/*	other functions	*/
unsigned int
SOIL_internal_create_OGL_texture
(
  const unsigned char* const data,
  int* width, int* height, int channels,
  unsigned int reuse_texture_ID,
  unsigned int flags,
  unsigned int opengl_texture_type,
  unsigned int opengl_texture_target,
  unsigned int texture_check_size_enum
);

/*	and the code magic begins here [8^)	*/
unsigned int
SOIL_load_OGL_texture
(
  const char* filename,
  int force_channels,
  unsigned int reuse_texture_ID,
  unsigned int flags,
  int* width,
  int* height
)
{
  /*	variables	*/
  unsigned char* img;
  int channels;
  unsigned int tex_id;
  /*	does the user want direct uploading of the image as a DDS file?	*/
  if (flags & SOIL_FLAG_DDS_LOAD_DIRECT)
  {
    /*	1st try direct loading of the image as a DDS file
      note: direct uploading will only load what is in the
      DDS file, no MIPmaps will be generated, the image will
      not be flipped, etc.	*/
    tex_id = SOIL_direct_load_DDS(filename, reuse_texture_ID, flags, 0);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  if (flags & SOIL_FLAG_PVR_LOAD_DIRECT)
  {
    tex_id = SOIL_direct_load_PVR(filename, reuse_texture_ID, flags, 0);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  if (flags & SOIL_FLAG_ETC1_LOAD_DIRECT)
  {
    tex_id = SOIL_direct_load_ETC1(filename, reuse_texture_ID, flags);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  /*	try to load the image	*/
  img = SOIL_load_image(filename, width, height, &channels, force_channels);
  /*	channels holds the original number of channels, which may have been forced	*/
  if ((force_channels >= 1) && (force_channels <= 4))
  {
    channels = force_channels;
  }
  if (nullptr == img)
  {
    /*	image loading failed	*/
    result_string_pointer = stbi_failure_reason();
    return 0;
  }
  /*	OK, make it a texture!	*/
  tex_id = SOIL_internal_create_OGL_texture(
    img, width, height, channels,
    reuse_texture_ID, flags,
    GL_TEXTURE_2D, GL_TEXTURE_2D,
    GL_MAX_TEXTURE_SIZE);
  /*	and nuke the image data	*/
  SOIL_free_image_data(img);
  /*	and return the handle, such as it is	*/
  return tex_id;
}


unsigned int
SOIL_load_OGL_texture_from_memory
(
  const unsigned char* const buffer,
  unsigned int buffer_length,
  int force_channels,
  unsigned int reuse_texture_ID,
  unsigned int flags,
  int* width,
  int* height
)
{
  /*	variables	*/
  unsigned char* img;
  int channels;
  unsigned int tex_id;
  /*	does the user want direct uploading of the image as a DDS file?	*/
  if (flags & SOIL_FLAG_DDS_LOAD_DIRECT)
  {
    /*	1st try direct loading of the image as a DDS file
      note: direct uploading will only load what is in the
      DDS file, no MIPmaps will be generated, the image will
      not be flipped, etc.	*/
    tex_id = SOIL_direct_load_DDS_from_memory(
      buffer, buffer_length,
      reuse_texture_ID, flags, 0);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  if (flags & SOIL_FLAG_PVR_LOAD_DIRECT)
  {
    tex_id = SOIL_direct_load_PVR_from_memory(
      buffer, buffer_length,
      reuse_texture_ID, flags, 0);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  if (flags & SOIL_FLAG_ETC1_LOAD_DIRECT)
  {
    tex_id = SOIL_direct_load_ETC1_from_memory(
      buffer, buffer_length,
      reuse_texture_ID, flags);
    if (tex_id)
    {
      /*	hey, it worked!!	*/
      return tex_id;
    }
  }

  /*	try to load the image	*/
  img = SOIL_load_image_from_memory(
    buffer, buffer_length,
    width, height, &channels,
    force_channels);
  /*	channels holds the original number of channels, which may have been forced	*/
  if ((force_channels >= 1) && (force_channels <= 4))
  {
    channels = force_channels;
  }
  if (nullptr == img)
  {
    /*	image loading failed	*/
    result_string_pointer = stbi_failure_reason();
    return 0;
  }
  /*	OK, make it a texture!	*/
  tex_id = SOIL_internal_create_OGL_texture(
    img, width, height, channels,
    reuse_texture_ID, flags,
    GL_TEXTURE_2D, GL_TEXTURE_2D,
    GL_MAX_TEXTURE_SIZE);
  /*	and nuke the image data	*/
  SOIL_free_image_data(img);
  /*	and return the handle, such as it is	*/
  return tex_id;
}

#if SOIL_CHECK_FOR_GL_ERRORS
void check_for_GL_errors(const char* calling_location)
{
  /*	check for errors	*/
  GLenum err_code = glGetError();
  while (GL_NO_ERROR != err_code)
  {
    printf("OpenGL Error @ %s: %i", calling_location, err_code);
    err_code = glGetError();
  }
}
#else
void check_for_GL_errors(const char* calling_location)
{
  /*	no check for errors	*/
}
#endif

static void createMipmaps(const unsigned char* const img,
  int width, int height, int channels,
  unsigned int flags,
  unsigned int opengl_texture_target,
  unsigned int internal_texture_format,
  unsigned int original_texture_format,
  int DXT_mode)
{
  if (flags & SOIL_FLAG_GL_MIPMAPS)
  {
    soilGlGenerateMipmap(opengl_texture_target);
  }
  else
  {
    int MIPlevel = 1;
    int MIPwidth = (width + 1) / 2;
    int MIPheight = (height + 1) / 2;
    unsigned char* resampled = (unsigned char*)malloc(channels * MIPwidth * MIPheight);

    while (((1 << MIPlevel) <= width) || ((1 << MIPlevel) <= height))
    {
      /*	do this MIPmap level	*/
      mipmap_image(
        img, width, height, channels,
        resampled,
        (1 << MIPlevel), (1 << MIPlevel));

      /*  upload the MIPmaps	*/
      if (DXT_mode == SOIL_CAPABILITY_PRESENT)
      {
        /*	user wants me to do the DXT conversion!	*/
        int DDS_size;
        unsigned char* DDS_data = nullptr;
        if ((channels & 1) == 1)
        {
          /*	RGB, use DXT1	*/
          DDS_data = convert_image_to_DXT1(
            resampled, MIPwidth, MIPheight, channels, &DDS_size);
        }
        else
        {
          /*	RGBA, use DXT5	*/
          DDS_data = convert_image_to_DXT5(
            resampled, MIPwidth, MIPheight, channels, &DDS_size);
        }
        if (DDS_data)
        {
          soilGlCompressedTexImage2D(
            opengl_texture_target, MIPlevel,
            internal_texture_format, MIPwidth, MIPheight, 0,
            DDS_size, DDS_data);
          check_for_GL_errors("glCompressedTexImage2D");
          SOIL_free_image_data(DDS_data);
        }
        else
        {
          /*	my compression failed, try the OpenGL driver's version	*/
          glTexImage2D(
            opengl_texture_target, MIPlevel,
            internal_texture_format, MIPwidth, MIPheight, 0,
            original_texture_format, GL_UNSIGNED_BYTE, resampled);
          check_for_GL_errors("glTexImage2D");
        }
      }
      else
      {
        /*	user want OpenGL to do all the work!	*/
        glTexImage2D(
          opengl_texture_target, MIPlevel,
          internal_texture_format, MIPwidth, MIPheight, 0,
          original_texture_format, GL_UNSIGNED_BYTE, resampled);
        check_for_GL_errors("glTexImage2D");
      }
      /*	prep for the next level	*/
      ++MIPlevel;
      MIPwidth = (MIPwidth + 1) / 2;
      MIPheight = (MIPheight + 1) / 2;
    }

    SOIL_free_image_data(resampled);
  }
}

unsigned int
SOIL_internal_create_OGL_texture
(
  const unsigned char* const data,
  int* width, int* height, int channels,
  unsigned int reuse_texture_ID,
  unsigned int flags,
  unsigned int opengl_texture_type,
  unsigned int opengl_texture_target,
  unsigned int texture_check_size_enum
)
{
  /*	variables	*/
  unsigned char* img = nullptr;
  unsigned int tex_id;
  unsigned int internal_texture_format = 0, original_texture_format = 0;
  int DXT_mode = SOIL_CAPABILITY_UNKNOWN;
  int max_supported_size;
  int iwidth = *width;
  int iheight = *height;
  int needCopy;
  GLint unpack_aligment;

  /*	how large of a texture can this OpenGL implementation handle?	*/
  /*	texture_check_size_enum will be GL_MAX_TEXTURE_SIZE or SOIL_MAX_CUBE_MAP_TEXTURE_SIZE	*/
  glGetIntegerv(texture_check_size_enum, &max_supported_size);


  needCopy = ((flags & SOIL_FLAG_INVERT_Y) ||
    (flags & SOIL_FLAG_NTSC_SAFE_RGB) ||
    (flags & SOIL_FLAG_MULTIPLY_ALPHA) ||
    (flags & SOIL_FLAG_CoCg_Y)
    );

  /*	create a copy the image data only if needed */
  if (needCopy) {
    img = (unsigned char*)malloc(iwidth * iheight * channels);
    memcpy(img, data, iwidth * iheight * channels);
  }

  /*	does the user want me to invert the image?	*/
  if (flags & SOIL_FLAG_INVERT_Y)
  {
    int i, j;
    for (j = 0; j * 2 < iheight; ++j)
    {
      int index1 = j * iwidth * channels;
      int index2 = (iheight - 1 - j) * iwidth * channels;
      for (i = iwidth * channels; i > 0; --i)
      {
        unsigned char temp = img[index1];
        img[index1] = img[index2];
        img[index2] = temp;
        ++index1;
        ++index2;
      }
    }
  }
  /*	does the user want me to scale the colors into the NTSC safe RGB range?	*/
  if (flags & SOIL_FLAG_NTSC_SAFE_RGB)
  {
    scale_image_RGB_to_NTSC_safe(img, iwidth, iheight, channels);
  }
  /*	does the user want me to convert from straight to pre-multiplied alpha?
    (and do we even _have_ alpha?)	*/
  if (flags & SOIL_FLAG_MULTIPLY_ALPHA)
  {
    int i;
    switch (channels)
    {
    case 2:
      for (i = 0; i < 2 * iwidth * iheight; i += 2)
      {
        img[i] = (img[i] * img[i + 1] + 128) >> 8;
      }
      break;
    case 4:
      for (i = 0; i < 4 * iwidth * iheight; i += 4)
      {
        img[i + 0] = (img[i + 0] * img[i + 3] + 128) >> 8;
        img[i + 1] = (img[i + 1] * img[i + 3] + 128) >> 8;
        img[i + 2] = (img[i + 2] * img[i + 3] + 128) >> 8;
      }
      break;
    default:
      /*	no other number of channels contains alpha data	*/
      break;
    }
  }

  /*	now, if it is too large...	*/
  if ((iwidth > max_supported_size) || (iheight > max_supported_size))
  {
    /*	I've already made it a power of two, so simply use the MIPmapping
      code to reduce its size to the allowable maximum.	*/
    unsigned char* resampled;
    int reduce_block_x = 1, reduce_block_y = 1;
    int new_width, new_height;
    if (iwidth > max_supported_size)
    {
      reduce_block_x = iwidth / max_supported_size;
    }
    if (iheight > max_supported_size)
    {
      reduce_block_y = iheight / max_supported_size;
    }
    new_width = iwidth / reduce_block_x;
    new_height = iheight / reduce_block_y;
    resampled = (unsigned char*)malloc(channels * new_width * new_height);
    /*	perform the actual reduction	*/
    mipmap_image(nullptr != img ? img : data, iwidth, iheight, channels,
      resampled, reduce_block_x, reduce_block_y);
    /*	nuke the old guy, then point it at the new guy	*/
    SOIL_free_image_data(img);
    img = resampled;
    *width = new_width;
    *height = new_height;
    iwidth = new_width;
    iheight = new_height;
  }
  /*	does the user want us to use YCoCg color space?	*/
  if (flags & SOIL_FLAG_CoCg_Y)
  {
    /*	this will only work with RGB and RGBA images */
    convert_RGB_to_YCoCg(img, iwidth, iheight, channels);
  }
  /*	create the OpenGL texture ID handle
    (note: allowing a forced texture ID lets me reload a texture)	*/
  tex_id = reuse_texture_ID;
  if (tex_id == 0)
  {
    glGenTextures(1, &tex_id);
  }
  check_for_GL_errors("glGenTextures");
  /* Note: sometimes glGenTextures fails (usually no OpenGL context)	*/
  if (tex_id)
  {
    /*	and what type am I using as the internal texture format?	*/
    switch (channels)
    {
    case 1:
      original_texture_format = GL_LUMINANCE;
      break;
    case 2:
      original_texture_format = GL_LUMINANCE_ALPHA;
      break;
    case 3:
      original_texture_format = GL_RGB;
      break;
    case 4:
      original_texture_format = GL_RGBA;
      break;
    }
    internal_texture_format = original_texture_format;
    /*	does the user want me to, and can I, save as DXT?	*/
    if (flags & SOIL_FLAG_COMPRESS_TO_DXT)
    {
      /*	I can use DXT, whether I compress it or OpenGL does	*/
      if ((channels & 1) == 1)
      {
        /*	1 or 3 channels = DXT1	*/
        internal_texture_format = SOIL_RGB_S3TC_DXT1;
      }
      else
      {
        /*	2 or 4 channels = DXT5	*/
        internal_texture_format = SOIL_RGBA_S3TC_DXT5;
      }
    }
    /*  bind an OpenGL texture ID	*/
    glBindTexture(opengl_texture_type, tex_id);
    check_for_GL_errors("glBindTexture");

    /* set the unpack aligment */
    glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack_aligment);
    if (1 != unpack_aligment)
    {
      glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    }

    /*  upload the main image	*/
    if (DXT_mode == SOIL_CAPABILITY_PRESENT)
    {
      /*	user wants me to do the DXT conversion!	*/
      int DDS_size;
      unsigned char* DDS_data = nullptr;
      if ((channels & 1) == 1)
      {
        /*	RGB, use DXT1	*/
        DDS_data = convert_image_to_DXT1(nullptr != img ? img : data, iwidth, iheight, channels, &DDS_size);
      }
      else
      {
        /*	RGBA, use DXT5	*/
        DDS_data = convert_image_to_DXT5(nullptr != img ? img : data, iwidth, iheight, channels, &DDS_size);
      }
      if (DDS_data)
      {
        soilGlCompressedTexImage2D(
          opengl_texture_target, 0,
          internal_texture_format, iwidth, iheight, 0,
          DDS_size, DDS_data);
        check_for_GL_errors("glCompressedTexImage2D");
        SOIL_free_image_data(DDS_data);
        /*	printf( "Internal DXT compressor\n" );	*/
      }
      else
      {
        /*	my compression failed, try the OpenGL driver's version	*/
        glTexImage2D(
          opengl_texture_target, 0,
          internal_texture_format, iwidth, iheight, 0,
          original_texture_format, GL_UNSIGNED_BYTE, nullptr != img ? img : data);
        check_for_GL_errors("glTexImage2D");
        /*	printf( "OpenGL DXT compressor\n" );	*/
      }
    }
    else
    {
      /*	user want OpenGL to do all the work!	*/
      glTexImage2D(
        opengl_texture_target, 0,
        internal_texture_format, iwidth, iheight, 0,
        original_texture_format, GL_UNSIGNED_BYTE, nullptr != img ? img : data);

      check_for_GL_errors("glTexImage2D");
      /*printf( "OpenGL DXT compressor\n" );	*/
    }

    /*	are any MIPmaps desired?	*/
    if (flags & SOIL_FLAG_MIPMAPS || flags & SOIL_FLAG_GL_MIPMAPS)
    {
      createMipmaps(nullptr != img ? img : data, iwidth, iheight, channels, flags, opengl_texture_target, internal_texture_format, original_texture_format, DXT_mode);

      /*	instruct OpenGL to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
      check_for_GL_errors("GL_TEXTURE_MIN/MAG_FILTER");
    }
    else
    {
      /*	instruct OpenGL _NOT_ to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      check_for_GL_errors("GL_TEXTURE_MIN/MAG_FILTER");
    }

    /* recover the unpack aligment */
    if (1 != unpack_aligment)
    {
      glPixelStorei(GL_UNPACK_ALIGNMENT, unpack_aligment);
    }

    /*	does the user want clamping, or wrapping?	*/
    if (flags & SOIL_FLAG_TEXTURE_REPEATS)
    {
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, GL_REPEAT);
      if (opengl_texture_type == SOIL_TEXTURE_CUBE_MAP)
      {
        /*	SOIL_TEXTURE_WRAP_R is invalid if cubemaps aren't supported	*/
        glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, GL_REPEAT);
      }
      check_for_GL_errors("GL_TEXTURE_WRAP_*");
    }
    else
    {
      unsigned int clamp_mode = SOIL_CLAMP_TO_EDGE;
      /* unsigned int clamp_mode = GL_CLAMP; */
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, clamp_mode);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, clamp_mode);
      if (opengl_texture_type == SOIL_TEXTURE_CUBE_MAP)
      {
        /*	SOIL_TEXTURE_WRAP_R is invalid if cubemaps aren't supported	*/
        glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, clamp_mode);
      }
      check_for_GL_errors("GL_TEXTURE_WRAP_*");
    }
    /*	done	*/
    result_string_pointer = "Image loaded as an OpenGL texture";
  }
  else
  {
    /*	failed	*/
    result_string_pointer = "Failed to generate an OpenGL texture name; missing OpenGL context?";
  }

  SOIL_free_image_data(img);

  return tex_id;
}

unsigned char*
SOIL_load_image
(
  const char* filename,
  int* width, int* height, int* channels,
  int force_channels
)
{
  unsigned char* result = stbi_load(filename,
    width, height, channels, force_channels);
  if (result == nullptr)
  {
    result_string_pointer = stbi_failure_reason();
  }
  else
  {
    result_string_pointer = "Image loaded";
  }
  return result;
}

unsigned char*
SOIL_load_image_from_memory
(
  const unsigned char* const buffer,
  int buffer_length,
  int* width, int* height, int* channels,
  int force_channels
)
{
  unsigned char* result = stbi_load_from_memory(
    buffer, buffer_length,
    width, height, channels,
    force_channels);
  if (result == nullptr)
  {
    result_string_pointer = stbi_failure_reason();
  }
  else
  {
    result_string_pointer = "Image loaded from memory";
  }
  return result;
}

void
SOIL_free_image_data
(
  unsigned char* img_data
)
{
  if (img_data)
    free((void*)img_data);
}

unsigned int SOIL_direct_load_DDS_from_memory(
  const unsigned char* const buffer,
  unsigned int buffer_length,
  unsigned int reuse_texture_ID,
  int flags,
  int loading_as_cubemap)
{
  /*	variables	*/
  DDS_header header;
  unsigned int buffer_index = 0;
  unsigned int tex_ID = 0;
  /*	file reading variables	*/
  unsigned int S3TC_type = 0;
  unsigned char* DDS_data;
  unsigned int DDS_main_size;
  unsigned int DDS_full_size;
  unsigned int width, height;
  int mipmaps, cubemap, uncompressed, block_size = 16;
  unsigned int flag;
  unsigned int cf_target, ogl_target_start, ogl_target_end;
  unsigned int opengl_texture_type;
  int i;
  /*	1st off, does the filename even exist?	*/
  if (nullptr == buffer)
  {
    /*	we can't do it!	*/
    result_string_pointer = "nullptr buffer";
    return 0;
  }
  if (buffer_length < sizeof(DDS_header))
  {
    /*	we can't do it!	*/
    result_string_pointer = "DDS file was too small to contain the DDS header";
    return 0;
  }
  /*	try reading in the header	*/
  memcpy((void*)(&header), (const void*)buffer, sizeof(DDS_header));
  buffer_index = sizeof(DDS_header);
  /*	guilty until proven innocent	*/
  result_string_pointer = "Failed to read a known DDS header";
  /*	validate the header (warning, "goto"'s ahead, shield your eyes!!)	*/
  flag = ('D' << 0) | ('D' << 8) | ('S' << 16) | (' ' << 24);
  if (header.dwMagic != flag) { goto quick_exit; }
  if (header.dwSize != 124) { goto quick_exit; }
  /*	I need all of these	*/
  flag = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
  if ((header.dwFlags & flag) != flag) { goto quick_exit; }
  /*	According to the MSDN spec, the dwFlags should contain
    DDSD_LINEARSIZE if it's compressed, or DDSD_PITCH if
    uncompressed.  Some DDS writers do not conform to the
    spec, so I need to make my reader more tolerant	*/
    /*	I need one of these	*/
  flag = DDPF_FOURCC | DDPF_RGB;
  if ((header.sPixelFormat.dwFlags & flag) == 0) { goto quick_exit; }
  if (header.sPixelFormat.dwSize != 32) { goto quick_exit; }
  if ((header.sCaps.dwCaps1 & DDSCAPS_TEXTURE) == 0) { goto quick_exit; }
  /*	make sure it is a type we can upload	*/
  if ((header.sPixelFormat.dwFlags & DDPF_FOURCC) &&
    !(
      (header.sPixelFormat.dwFourCC == (('D' << 0) | ('X' << 8) | ('T' << 16) | ('1' << 24))) ||
      (header.sPixelFormat.dwFourCC == (('D' << 0) | ('X' << 8) | ('T' << 16) | ('3' << 24))) ||
      (header.sPixelFormat.dwFourCC == (('D' << 0) | ('X' << 8) | ('T' << 16) | ('5' << 24)))
      ))
  {
    goto quick_exit;
  }
  /*	OK, validated the header, let's load the image data	*/
  result_string_pointer = "DDS header loaded and validated";
  width = header.dwWidth;
  height = header.dwHeight;
  uncompressed = 1 - (header.sPixelFormat.dwFlags & DDPF_FOURCC) / DDPF_FOURCC;
  cubemap = (header.sCaps.dwCaps2 & DDSCAPS2_CUBEMAP) / DDSCAPS2_CUBEMAP;
  if (uncompressed)
  {
    S3TC_type = GL_RGB;
    block_size = 3;
    if (header.sPixelFormat.dwFlags & DDPF_ALPHAPIXELS)
    {
      S3TC_type = GL_RGBA;
      block_size = 4;
    }
    DDS_main_size = width * height * block_size;
  }
  else
  {
    /*	well, we know it is DXT1/3/5, because we checked above	*/
    switch ((header.sPixelFormat.dwFourCC >> 24) - '0')
    {
    case 1:
      S3TC_type = SOIL_RGBA_S3TC_DXT1;
      block_size = 8;
      break;
    case 3:
      S3TC_type = SOIL_RGBA_S3TC_DXT3;
      block_size = 16;
      break;
    case 5:
      S3TC_type = SOIL_RGBA_S3TC_DXT5;
      block_size = 16;
      break;
    }
    DDS_main_size = ((width + 3) >> 2) * ((height + 3) >> 2) * block_size;
  }
  if (cubemap)
  {
    /* does the user want a cubemap?	*/
    if (!loading_as_cubemap)
    {
      /*	we can't do it!	*/
      result_string_pointer = "DDS image was a cubemap";
      return 0;
    }
    ogl_target_start = SOIL_TEXTURE_CUBE_MAP_POSITIVE_X;
    ogl_target_end = SOIL_TEXTURE_CUBE_MAP_NEGATIVE_Z;
    opengl_texture_type = SOIL_TEXTURE_CUBE_MAP;
  }
  else
  {
    /* does the user want a non-cubemap?	*/
    if (loading_as_cubemap)
    {
      /*	we can't do it!	*/
      result_string_pointer = "DDS image was not a cubemap";
      return 0;
    }
    ogl_target_start = GL_TEXTURE_2D;
    ogl_target_end = GL_TEXTURE_2D;
    opengl_texture_type = GL_TEXTURE_2D;
  }
  if ((header.sCaps.dwCaps1 & DDSCAPS_MIPMAP) && (header.dwMipMapCount > 1))
  {
    int shift_offset;
    mipmaps = header.dwMipMapCount - 1;
    DDS_full_size = DDS_main_size;
    if (uncompressed)
    {
      /*	uncompressed DDS, simple MIPmap size calculation	*/
      shift_offset = 0;
    }
    else
    {
      /*	compressed DDS, MIPmap size calculation is block based	*/
      shift_offset = 2;
    }
    for (i = 1; i <= mipmaps; ++i)
    {
      int w, h;
      w = width >> (shift_offset + i);
      h = height >> (shift_offset + i);
      if (w < 1)
      {
        w = 1;
      }
      if (h < 1)
      {
        h = 1;
      }
      DDS_full_size += w * h * block_size;
    }
  }
  else
  {
    mipmaps = 0;
    DDS_full_size = DDS_main_size;
  }
  DDS_data = (unsigned char*)malloc(DDS_full_size);
  /*	got the image data RAM, create or use an existing OpenGL texture handle	*/
  tex_ID = reuse_texture_ID;
  if (tex_ID == 0)
  {
    glGenTextures(1, &tex_ID);
  }
  /*  bind an OpenGL texture ID	*/
  glBindTexture(opengl_texture_type, tex_ID);
  /*	do this for each face of the cubemap!	*/
  for (cf_target = ogl_target_start; cf_target <= ogl_target_end; ++cf_target)
  {
    if (buffer_index + DDS_full_size <= buffer_length)
    {
      unsigned int byte_offset = DDS_main_size;
      memcpy((void*)DDS_data, (const void*)(&buffer[buffer_index]), DDS_full_size);
      buffer_index += DDS_full_size;
      /*	upload the main chunk	*/
      if (uncompressed)
      {
        /*	and remember, DXT uncompressed uses BGR(A),
          so swap to RGB(A) for ALL MIPmap levels	*/
        for (i = 0; i < (int)DDS_full_size; i += block_size)
        {
          unsigned char temp = DDS_data[i];
          DDS_data[i] = DDS_data[i + 2];
          DDS_data[i + 2] = temp;
        }
        glTexImage2D(
          cf_target, 0,
          S3TC_type, width, height, 0,
          S3TC_type, GL_UNSIGNED_BYTE, DDS_data);
      }
      else
      {
        soilGlCompressedTexImage2D(
          cf_target, 0,
          S3TC_type, width, height, 0,
          DDS_main_size, DDS_data);
      }
      /*	upload the mipmaps, if we have them	*/
      for (i = 1; i <= mipmaps; ++i)
      {
        int w, h, mip_size;
        w = width >> i;
        h = height >> i;
        if (w < 1)
        {
          w = 1;
        }
        if (h < 1)
        {
          h = 1;
        }
        /*	upload this mipmap	*/
        if (uncompressed)
        {
          mip_size = w * h * block_size;
          glTexImage2D(
            cf_target, i,
            S3TC_type, w, h, 0,
            S3TC_type, GL_UNSIGNED_BYTE, &DDS_data[byte_offset]);
        }
        else
        {
          mip_size = ((w + 3) / 4) * ((h + 3) / 4) * block_size;
          soilGlCompressedTexImage2D(
            cf_target, i,
            S3TC_type, w, h, 0,
            mip_size, &DDS_data[byte_offset]);
        }
        /*	and move to the next mipmap	*/
        byte_offset += mip_size;
      }
      /*	it worked!	*/
      result_string_pointer = "DDS file loaded";
    }
    else
    {
      glDeleteTextures(1, &tex_ID);
      tex_ID = 0;
      cf_target = ogl_target_end + 1;
      result_string_pointer = "DDS file was too small for expected image data";
    }
  }/* end reading each face */
  SOIL_free_image_data(DDS_data);
  if (tex_ID)
  {
    /*	did I have MIPmaps?	*/
    if (mipmaps > 0)
    {
      /*	instruct OpenGL to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    }
    else
    {
      /*	instruct OpenGL _NOT_ to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    }
    /*	does the user want clamping, or wrapping?	*/
    if (flags & SOIL_FLAG_TEXTURE_REPEATS)
    {
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, GL_REPEAT);
    }
    else
    {
      unsigned int clamp_mode = SOIL_CLAMP_TO_EDGE;
      /* unsigned int clamp_mode = GL_CLAMP; */
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, clamp_mode);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, clamp_mode);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, clamp_mode);
    }
  }

quick_exit:
  /*	report success or failure	*/
  return tex_ID;
}

unsigned int SOIL_direct_load_DDS(
  const char* filename,
  unsigned int reuse_texture_ID,
  int flags,
  int loading_as_cubemap)
{
  FILE* f;
  unsigned char* buffer;
  size_t buffer_length, bytes_read;
  unsigned int tex_ID = 0;
  /*	error checks	*/
  if (nullptr == filename)
  {
    result_string_pointer = "nullptr filename";
    return 0;
  }
  f = fopen(filename, "rb");
  if (nullptr == f)
  {
    /*	the file doesn't seem to exist (or be open-able)	*/
    result_string_pointer = "Can not find DDS file";
    return 0;
  }
  fseek(f, 0, SEEK_END);
  buffer_length = ftell(f);
  fseek(f, 0, SEEK_SET);
  buffer = (unsigned char*)malloc(buffer_length);
  if (nullptr == buffer)
  {
    result_string_pointer = "malloc failed";
    fclose(f);
    return 0;
  }
  bytes_read = fread((void*)buffer, 1, buffer_length, f);
  fclose(f);
  if (bytes_read < buffer_length)
  {
    /*	huh?	*/
    buffer_length = bytes_read;
  }
  /*	now try to do the loading	*/
  tex_ID = SOIL_direct_load_DDS_from_memory(
    (const unsigned char* const)buffer, (int)buffer_length,
    reuse_texture_ID, flags, loading_as_cubemap);
  SOIL_free_image_data(buffer);
  return tex_ID;
}

unsigned int SOIL_direct_load_PVR_from_memory(
  const unsigned char* const buffer,
  int buffer_length,
  unsigned int reuse_texture_ID,
  int flags,
  int loading_as_cubemap)
{
  PVR_Texture_Header* header = (PVR_Texture_Header*)buffer;
  int num_surfs = 1;
  GLuint tex_ID = 0;
  GLenum PVR_format = 0;
  GLenum PVR_type = GL_RGB;
  unsigned int opengl_texture_type = loading_as_cubemap ? SOIL_TEXTURE_CUBE_MAP : GL_TEXTURE_2D;
  int is_PVRTC_supported = 1;
  int is_BGRA8888_supported = 1;
  int is_compressed_format_supported = 0;
  int is_compressed_format = 0;
  int mipmaps = 0;
  int i;
  GLint unpack_aligment;

  // Check the header size
  if (header->dwHeaderSize != sizeof(PVR_Texture_Header)) {
    if (header->dwHeaderSize == PVRTEX_V1_HEADER_SIZE) {
      result_string_pointer = "this is an old pvr ( update the PVR file )";

      if (loading_as_cubemap) {
        if (header->dwpfFlags & PVRTEX_CUBEMAP) {
          num_surfs = 6;
        }
        else {
          result_string_pointer = "tried to load a non-cubemap PVR as cubemap";
          return 0;
        }
      }
    }
    else {
      result_string_pointer = "invalid PVR header";

      return 0;
    }
  }
  else {
    if (loading_as_cubemap) {
      // Header V2
      if (header->dwNumSurfs < 1) {
        if (header->dwpfFlags & PVRTEX_CUBEMAP) {
          num_surfs = 6;
        }
        else {
          result_string_pointer = "tried to load a non-cubemap PVR as cubemap";
          return 0;
        }
      }
      else {
        num_surfs = header->dwNumSurfs;
      }
    }
  }

  // Check the magic identifier
  if (header->dwPVR != PVRTEX_IDENTIFIER) {
    result_string_pointer = "invalid PVR header";
    return 0;
  }

  /* Only accept untwiddled data UNLESS texture format is PVRTC */
  if (((header->dwpfFlags & PVRTEX_TWIDDLE) == PVRTEX_TWIDDLE)
    && ((header->dwpfFlags & PVRTEX_PIXELTYPE) != OGL_PVRTC2)
    && ((header->dwpfFlags & PVRTEX_PIXELTYPE) != OGL_PVRTC4))
  {
    // We need to load untwiddled textures -- hw will twiddle for us.
    result_string_pointer = "pvr is not compressed ( untwiddled texture )";
    return 0;
  }

  switch (header->dwpfFlags & PVRTEX_PIXELTYPE)
  {
  case OGL_RGBA_4444:
    PVR_format = GL_UNSIGNED_SHORT_4_4_4_4;
    PVR_type = GL_RGBA;
    break;
  case OGL_RGBA_5551:
    PVR_format = GL_UNSIGNED_SHORT_5_5_5_1;
    PVR_type = GL_RGBA;
    break;
  case OGL_RGBA_8888:
    PVR_format = GL_UNSIGNED_BYTE;
    PVR_type = GL_RGBA;
    break;
  case OGL_RGB_565:
    PVR_format = GL_UNSIGNED_SHORT_5_6_5;
    PVR_type = GL_RGB;
    break;
  case OGL_RGB_555:
    result_string_pointer = "failed: pixel type OGL_RGB_555 not supported.";
    return 0;
  case OGL_RGB_888:
    PVR_format = GL_UNSIGNED_BYTE;
    PVR_type = GL_RGB;
    break;
  case OGL_I_8:
    PVR_format = GL_UNSIGNED_BYTE;
    PVR_type = GL_LUMINANCE;
    break;
  case OGL_AI_88:
    PVR_format = GL_UNSIGNED_BYTE;
    PVR_type = GL_LUMINANCE_ALPHA;
    break;
  case MGLPT_PVRTC2:
  case OGL_PVRTC2:
    if (is_PVRTC_supported) {
      is_compressed_format_supported = is_compressed_format = 1;
      PVR_format = header->dwAlphaBitMask == 0 ? SOIL_COMPRESSED_RGB_PVRTC_2BPPV1_IMG : SOIL_COMPRESSED_RGBA_PVRTC_2BPPV1_IMG;	// PVRTC2
    }
    else {
      result_string_pointer = "error: PVRTC2 not supported.Decompress the texture first.";
      return 0;
    }
    break;
  case MGLPT_PVRTC4:
  case OGL_PVRTC4:
    if (is_PVRTC_supported) {
      is_compressed_format_supported = is_compressed_format = 1;
      PVR_format = header->dwAlphaBitMask == 0 ? SOIL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG : SOIL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;	// PVRTC4
    }
    else {
      result_string_pointer = "error: PVRTC4 not supported. Decompress the texture first.";
      return 0;
    }
    break;
  case OGL_BGRA_8888:
    if (is_BGRA8888_supported) {
      PVR_format = GL_UNSIGNED_BYTE;
      PVR_type = GL_BGRA;
      break;
    }
    else {
      result_string_pointer = "Unable to load GL_BGRA texture as extension GL_IMG_texture_format_BGRA8888 is unsupported.";
      return 0;
    }
  default:											// NOT SUPPORTED
    result_string_pointer = "failed: pixel type not supported.";
    return 0;
  }

#ifdef SOIL_GLES1
  //  check that this data is cube map data or not.
  if (loading_as_cubemap) {
    result_string_pointer = "cube map textures are not available in GLES1.x.";
    return 0;
  }
#endif

  // load the texture up
  tex_ID = reuse_texture_ID;
  if (tex_ID == 0)
  {
    glGenTextures(1, &tex_ID);
  }

  glBindTexture(opengl_texture_type, tex_ID);

  if (glGetError()) {
    result_string_pointer = "failed: glBindTexture() failed.";
    return 0;
  }

  glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack_aligment);
  if (1 != unpack_aligment)
  {
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);				// Never have row-aligned in headers
  }

#define _MAX( a, b ) (( a <= b )? b : a)
  for (i = 0; i < num_surfs; i++) {
    char* texture_ptr = (char*)buffer + header->dwHeaderSize + header->dwTextureDataSize * i;
    char* cur_texture_ptr = 0;
    int	mipmap_level;
    unsigned int width = header->dwWidth, height = header->dwHeight;
    unsigned int compressed_image_size = 0;

    mipmaps = ((flags & SOIL_FLAG_MIPMAPS) && (header->dwpfFlags & PVRTEX_MIPMAP)) ? header->dwMipMapCount : 0;

    for (mipmap_level = 0; mipmap_level <= mipmaps; width = _MAX(width / 2, (unsigned int)1), height = _MAX(height / 2, (unsigned int)1), mipmap_level++) {
      // Do Alpha-swap if needed
      cur_texture_ptr = texture_ptr;

      // Load the Texture_
      /* If the texture is PVRTC then use GLCompressedTexImage2D */
      if (is_compressed_format) {
        /* Calculate how many bytes this MIP level occupies */
        if ((header->dwpfFlags & PVRTEX_PIXELTYPE) == OGL_PVRTC2) {
          compressed_image_size = (_MAX(width, PVRTC2_MIN_TEXWIDTH) * _MAX(height, PVRTC2_MIN_TEXHEIGHT) * header->dwBitCount + 7) / 8;
        }
        else {// PVRTC4 case
          compressed_image_size = (_MAX(width, PVRTC4_MIN_TEXWIDTH) * _MAX(height, PVRTC4_MIN_TEXHEIGHT) * header->dwBitCount + 7) / 8;
        }

        if (is_compressed_format_supported) {
          /* Load compressed texture data at selected MIP level */
          if (loading_as_cubemap) {
            soilGlCompressedTexImage2D(SOIL_TEXTURE_CUBE_MAP_POSITIVE_X + i, mipmap_level, PVR_format, width, height, 0, compressed_image_size, cur_texture_ptr);
          }
          else {
            soilGlCompressedTexImage2D(opengl_texture_type, mipmap_level, PVR_format, width, height, 0, compressed_image_size, cur_texture_ptr);
          }
        }
        else {
          result_string_pointer = "failed: GPU doesnt support compressed textures";
        }
      }
      else {
        /* Load uncompressed texture data at selected MIP level */
        if (loading_as_cubemap) {
          glTexImage2D(SOIL_TEXTURE_CUBE_MAP_POSITIVE_X + i, mipmap_level, PVR_type, width, height, 0, PVR_type, PVR_format, cur_texture_ptr);
        }
        else {
          glTexImage2D(opengl_texture_type, mipmap_level, PVR_type, width, height, 0, PVR_type, PVR_format, cur_texture_ptr);
        }
      }

      if (glGetError()) {
        result_string_pointer = "failed: glCompressedTexImage2D() failed.";
        if (1 != unpack_aligment)
        {
          glPixelStorei(GL_UNPACK_ALIGNMENT, unpack_aligment);
        }
        return 0;
      }

      // offset the texture pointer by one mip-map level
      /* PVRTC case */
      if (is_compressed_format) {
        texture_ptr += compressed_image_size;
      }
      else {
        /* New formula that takes into account bit counts inferior to 8 (e.g. 1 bpp) */
        texture_ptr += (width * height * header->dwBitCount + 7) / 8;
      }
    }
  }
#undef _MAX

  if (1 != unpack_aligment)
  {
    glPixelStorei(GL_UNPACK_ALIGNMENT, unpack_aligment);
  }

  if (tex_ID)
  {
    /*	did I have MIPmaps?	*/
    if (mipmaps)
    {
      /*	instruct OpenGL to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    }
    else
    {
      /*	instruct OpenGL _NOT_ to use the MIPmaps	*/
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    }

    /*	does the user want clamping, or wrapping?	*/
    if (flags & SOIL_FLAG_TEXTURE_REPEATS)
    {
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, GL_REPEAT);
    }
    else
    {
      unsigned int clamp_mode = SOIL_CLAMP_TO_EDGE;
      /* unsigned int clamp_mode = GL_CLAMP; */
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, clamp_mode);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, clamp_mode);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, clamp_mode);
    }
  }

  return tex_ID;
}

unsigned int SOIL_direct_load_PVR(
  const char* filename,
  unsigned int reuse_texture_ID,
  int flags,
  int loading_as_cubemap)
{
  FILE* f;
  unsigned char* buffer;
  size_t buffer_length, bytes_read;
  unsigned int tex_ID = 0;
  /*	error checks	*/
  if (nullptr == filename)
  {
    result_string_pointer = "nullptr filename";
    return 0;
  }
  f = fopen(filename, "rb");
  if (nullptr == f)
  {
    /*	the file doesn't seem to exist (or be open-able)	*/
    result_string_pointer = "Can not find PVR file";
    return 0;
  }
  fseek(f, 0, SEEK_END);
  buffer_length = ftell(f);
  fseek(f, 0, SEEK_SET);
  buffer = (unsigned char*)malloc(buffer_length);
  if (nullptr == buffer)
  {
    result_string_pointer = "malloc failed";
    fclose(f);
    return 0;
  }
  bytes_read = fread((void*)buffer, 1, buffer_length, f);
  fclose(f);
  if (bytes_read < buffer_length)
  {
    /*	huh?	*/
    buffer_length = bytes_read;
  }
  /*	now try to do the loading	*/
  tex_ID = SOIL_direct_load_PVR_from_memory(
    (const unsigned char* const)buffer, (int)buffer_length,
    reuse_texture_ID, flags, loading_as_cubemap);
  SOIL_free_image_data(buffer);
  return tex_ID;
}

unsigned int SOIL_direct_load_ETC1_from_memory(
  const unsigned char* const buffer,
  int buffer_length,
  unsigned int reuse_texture_ID,
  int flags)
{
  GLuint tex_ID = 0;
  PKMHeader* header = (PKMHeader*)buffer;
  unsigned int opengl_texture_type = GL_TEXTURE_2D;
  unsigned int width;
  unsigned int height;
  unsigned long compressed_image_size = buffer_length - PKM_HEADER_SIZE;
  char* texture_ptr = (char*)buffer + PKM_HEADER_SIZE;
  GLint unpack_aligment;

  if (0 != strcmp(header->aName, "PKM 10")) {
    result_string_pointer = "error: PKM 10 header not found.";
    return 0;
  }

  width = (header->iWidthMSB << 8) | header->iWidthLSB;
  height = (header->iHeightMSB << 8) | header->iHeightLSB;
  compressed_image_size = (((width + 3) & ~3) * ((height + 3) & ~3)) >> 1;

  // load the texture up
  tex_ID = reuse_texture_ID;
  if (tex_ID == 0)
  {
    glGenTextures(1, &tex_ID);
  }

  glBindTexture(opengl_texture_type, tex_ID);

  if (glGetError()) {
    result_string_pointer = "failed: glBindTexture() failed.";
    return 0;
  }

  glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack_aligment);
  if (1 != unpack_aligment)
  {
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);				// Never have row-aligned in headers
  }

  soilGlCompressedTexImage2D(opengl_texture_type, 0, SOIL_GL_ETC1_RGB8_OES, width, height, 0, compressed_image_size, texture_ptr);

  if (glGetError()) {
    result_string_pointer = "failed: glCompressedTexImage2D() failed.";

    if (1 != unpack_aligment)
    {
      glPixelStorei(GL_UNPACK_ALIGNMENT, unpack_aligment);
    }
    return 0;
  }

  if (1 != unpack_aligment)
  {
    glPixelStorei(GL_UNPACK_ALIGNMENT, unpack_aligment);
  }

  if (tex_ID)
  {
    /* No MIPmaps for ETC1 */
    glTexParameteri(opengl_texture_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(opengl_texture_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    /*	does the user want clamping, or wrapping?	*/
    if (flags & SOIL_FLAG_TEXTURE_REPEATS)
    {
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, GL_REPEAT);
    }
    else
    {
      unsigned int clamp_mode = SOIL_CLAMP_TO_EDGE;
      /* unsigned int clamp_mode = GL_CLAMP; */
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_S, clamp_mode);
      glTexParameteri(opengl_texture_type, GL_TEXTURE_WRAP_T, clamp_mode);
      glTexParameteri(opengl_texture_type, SOIL_TEXTURE_WRAP_R, clamp_mode);
    }
  }

  return tex_ID;
}

unsigned int SOIL_direct_load_ETC1(const char* filename,
  unsigned int reuse_texture_ID,
  int flags)
{
  FILE* f;
  unsigned char* buffer;
  size_t buffer_length, bytes_read;
  unsigned int tex_ID = 0;
  /*	error checks	*/
  if (nullptr == filename)
  {
    result_string_pointer = "nullptr filename";
    return 0;
  }
  f = fopen(filename, "rb");
  if (nullptr == f)
  {
    /*	the file doesn't seem to exist (or be open-able)	*/
    result_string_pointer = "Can not find PVR file";
    return 0;
  }
  fseek(f, 0, SEEK_END);
  buffer_length = ftell(f);
  fseek(f, 0, SEEK_SET);
  buffer = (unsigned char*)malloc(buffer_length);
  if (nullptr == buffer)
  {
    result_string_pointer = "malloc failed";
    fclose(f);
    return 0;
  }
  bytes_read = fread((void*)buffer, 1, buffer_length, f);
  fclose(f);
  if (bytes_read < buffer_length)
  {
    /*	huh?	*/
    buffer_length = bytes_read;
  }
  /*	now try to do the loading	*/
  tex_ID = SOIL_direct_load_ETC1_from_memory(
    (const unsigned char* const)buffer, (int)buffer_length,
    reuse_texture_ID, flags);
  SOIL_free_image_data(buffer);
  return tex_ID;
}

namespace glm
{
  using std::size_t;
  typedef int length_t;
}//namespace glm

namespace glm
{
  template<typename T, std::size_t N>
  constexpr std::size_t countof(T const (&)[N])
  {
    return N;
  }
}//namespace glm

namespace glm {
  namespace detail
  {
    template<typename T>
    struct is_int
    {
      enum test { value = 0 };
    };

    template<>
    struct is_int<unsigned int>
    {
      enum test { value = ~0 };
    };

    template<>
    struct is_int<signed int>
    {
      enum test { value = ~0 };
    };
  }//namespace detail

  typedef unsigned int	uint;
}//namespace glm

///////////////////////////////////////////////////////////////////////////////////
// 64-bit int

namespace glm {
  namespace detail
  {
    typedef std::uint64_t						uint64;
    typedef std::int64_t						int64;
  }//namespace detail
}//namespace glm

namespace glm {
  namespace detail
  {
    using std::make_unsigned;
  }//namespace detail
}//namespace glm

namespace glm
{
  /// Qualify GLM types in term of alignment (packed, aligned) and precision in term of ULPs (lowp, mediump, highp)
  enum qualifier
  {
    packed_highp, ///< Typed data is tightly packed in memory and operations are executed with high precision in term of ULPs
    packed_mediump, ///< Typed data is tightly packed in memory  and operations are executed with medium precision in term of ULPs for higher performance
    packed_lowp, ///< Typed data is tightly packed in memory  and operations are executed with low precision in term of ULPs to maximize performance
    highp = packed_highp, ///< By default highp qualifier is also packed
    mediump = packed_mediump, ///< By default mediump qualifier is also packed
    lowp = packed_lowp, ///< By default lowp qualifier is also packed
    packed = packed_highp, ///< By default packed qualifier is also high precision
    defaultp = highp
  };

  typedef qualifier precision;

  template<length_t L, typename T, qualifier Q = defaultp> struct vec;
  template<length_t C, length_t R, typename T, qualifier Q = defaultp> struct mat;
  template<typename T, qualifier Q = defaultp> struct qua;


  template <typename T, qualifier Q = defaultp> using tvec1 = vec<1, T, Q>;
  template <typename T, qualifier Q = defaultp> using tvec2 = vec<2, T, Q>;
  template <typename T, qualifier Q = defaultp> using tvec3 = vec<3, T, Q>;
  template <typename T, qualifier Q = defaultp> using tvec4 = vec<4, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat2x2 = mat<2, 2, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat2x3 = mat<2, 3, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat2x4 = mat<2, 4, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat3x2 = mat<3, 2, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat3x3 = mat<3, 3, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat3x4 = mat<3, 4, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat4x2 = mat<4, 2, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat4x3 = mat<4, 3, T, Q>;
  template <typename T, qualifier Q = defaultp> using tmat4x4 = mat<4, 4, T, Q>;
  template <typename T, qualifier Q = defaultp> using tquat = qua<T, Q>;


  namespace detail
  {
    template<glm::qualifier P>
    struct is_aligned
    {
      static const bool value = false;
    };

    template<length_t L, typename T, bool is_aligned>
    struct storage
    {
      typedef struct type {
        T data[L];
      } type;
    };


    template<length_t L, typename T>
    struct storage<L, T, true>
    {
      typedef struct alignas(L * sizeof(T)) type {
        T data[L];
      } type;
    };

    template<typename T>
    struct storage<3, T, true>
    {
      typedef struct alignas(4 * sizeof(T)) type {
        T data[4];
      } type;
    };

    enum genTypeEnum
    {
      GENTYPE_VEC,
      GENTYPE_MAT,
      GENTYPE_QUAT
    };

    template <typename genType>
    struct genTypeTrait
    {};

    template <length_t C, length_t R, typename T>
    struct genTypeTrait<mat<C, R, T> >
    {
      static const genTypeEnum GENTYPE = GENTYPE_MAT;
    };

    template<typename genType, genTypeEnum type>
    struct init_gentype
    {
    };

    template<typename genType>
    struct init_gentype<genType, GENTYPE_QUAT>
    {
      inline constexpr static genType identity()
      {
        return genType(1, 0, 0, 0);
      }
    };

    template<typename genType>
    struct init_gentype<genType, GENTYPE_MAT>
    {
      inline constexpr static genType identity()
      {
        return genType(1);
      }
    };
  }//namespace detail
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct vec<4, T, Q>
  {
    // -- Implementation detail --

    typedef T value_type;
    typedef vec<4, T, Q> type;
    typedef vec<4, bool, Q> bool_type;

    // -- Data --
    union { T x, r, s; };
    union { T y, g, t; };
    union { T z, b, p; };
    union { T w, a, q; };

    // -- Component accesses --

    typedef length_t length_type;

    /// Return the count of components of the vector
    static constexpr length_type length() { return 4; }

    constexpr T& operator[](length_type i);
    constexpr T const& operator[](length_type i) const;

    // -- Implicit basic constructors --

    constexpr vec() = default;
    constexpr vec(vec<4, T, Q> const& v) = default;
    template<qualifier P>
    constexpr vec(vec<4, T, P> const& v);

    // -- Explicit basic constructors --

    constexpr explicit vec(T scalar);
    constexpr vec(T x, T y, T z, T w);

    // -- Conversion scalar constructors --

    template<typename U, qualifier P>
    constexpr explicit vec(vec<1, U, P> const& v);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, Y _y, Z _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, Z _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, Z _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, Y _y, vec<1, Z, Q> const& _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, W _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, Z _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, Z _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, Y _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w);
    template<typename X, typename Y, typename Z, typename W>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _Y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w);

    // -- Conversion vector constructors --

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, B _z, C _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z, C _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, B _z, vec<1, C, P> const& _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z, vec<1, C, P> const& _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(A _x, vec<2, B, P> const& _yz, C _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz, C _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(A _x, vec<2, B, P> const& _yz, vec<1, C, P> const& _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz, vec<1, C, P> const& _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(A _x, B _y, vec<2, C, P> const& _zw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, B _y, vec<2, C, P> const& _zw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(A _x, vec<1, B, P> const& _y, vec<2, C, P> const& _zw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, typename C, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, vec<1, B, P> const& _y, vec<2, C, P> const& _zw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<3, A, P> const& _xyz, B _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<3, A, P> const& _xyz, vec<1, B, P> const& _w);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(A _x, vec<3, B, P> const& _yzw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, vec<3, B, P> const& _yzw);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, vec<2, B, P> const& _zw);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<4, U, P> const& v);

    // -- Swizzle constructors --

    // -- Unary arithmetic operators --

    constexpr vec<4, T, Q>& operator=(vec<4, T, Q> const& v) = default;

    template<typename U>
    constexpr vec<4, T, Q>& operator=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator+=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator+=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator+=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator-=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator-=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator-=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator*=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator*=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator*=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator/=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator/=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator/=(vec<4, U, Q> const& v);

    // -- Increment and decrement operators --

    constexpr vec<4, T, Q>& operator++();
    constexpr vec<4, T, Q>& operator--();
    constexpr vec<4, T, Q> operator++(int);
    constexpr vec<4, T, Q> operator--(int);

    // -- Unary bit operators --

    template<typename U>
    constexpr vec<4, T, Q>& operator%=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator%=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator%=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator&=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator&=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator&=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator|=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator|=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator|=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator^=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator^=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator^=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator<<=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator<<=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator<<=(vec<4, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator>>=(U scalar);
    template<typename U>
    constexpr vec<4, T, Q>& operator>>=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<4, T, Q>& operator>>=(vec<4, U, Q> const& v);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v);

  // -- Binary operators --

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v, T const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v, T const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v, T const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator*(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator*(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v, T const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator/(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator/(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator%(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator%(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator&(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator&(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator|(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator|(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator^(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator^(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator<<(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator<<(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator>>(T scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator>>(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<4, T, Q> operator~(vec<4, T, Q> const& v);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  constexpr bool operator==(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr bool operator!=(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2);

  template<qualifier Q>
  constexpr vec<4, bool, Q> operator&&(vec<4, bool, Q> const& v1, vec<4, bool, Q> const& v2);

  template<qualifier Q>
  constexpr vec<4, bool, Q> operator||(vec<4, bool, Q> const& v1, vec<4, bool, Q> const& v2);
}//namespace glm

namespace glm {
  namespace detail
  {
    template <typename T, bool isFloat>
    struct compute_equal
    {
      inline constexpr static bool call(T a, T b)
      {
        return a == b;
      }
    };
    /*
      template <typename T>
      struct compute_equal<T, true>
      {
        GLM_FUNC_QUALIFIER GLM_CONSTEXPR static bool call(T a, T b)
        {
          return detail::compute_abs<T, std::numeric_limits<T>::is_signed>::call(b - a) <= static_cast<T>(0);
          //return std::memcmp(&a, &b, sizeof(T)) == 0;
        }
      };
    */
  }//namespace detail
}//namespace glm


namespace glm {
  namespace detail
  {
    template<typename T, qualifier Q, bool Aligned>
    struct compute_vec4_add
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x + b.x, a.y + b.y, a.z + b.z, a.w + b.w);
      }
    };

    template<typename T, qualifier Q, bool Aligned>
    struct compute_vec4_sub
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x - b.x, a.y - b.y, a.z - b.z, a.w - b.w);
      }
    };

    template<typename T, qualifier Q, bool Aligned>
    struct compute_vec4_mul
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x * b.x, a.y * b.y, a.z * b.z, a.w * b.w);
      }
    };

    template<typename T, qualifier Q, bool Aligned>
    struct compute_vec4_div
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x / b.x, a.y / b.y, a.z / b.z, a.w / b.w);
      }
    };

    template<typename T, qualifier Q, bool Aligned>
    struct compute_vec4_mod
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x % b.x, a.y % b.y, a.z % b.z, a.w % b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_and
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x & b.x, a.y & b.y, a.z & b.z, a.w & b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_or
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x | b.x, a.y | b.y, a.z | b.z, a.w | b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_xor
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x ^ b.x, a.y ^ b.y, a.z ^ b.z, a.w ^ b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_shift_left
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x << b.x, a.y << b.y, a.z << b.z, a.w << b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_shift_right
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& a, vec<4, T, Q> const& b)
      {
        return vec<4, T, Q>(a.x >> b.x, a.y >> b.y, a.z >> b.z, a.w >> b.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_equal
    {
      inline constexpr static bool call(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
      {
        return
          detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.x, v2.x) &&
          detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.y, v2.y) &&
          detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.z, v2.z) &&
          detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.w, v2.w);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_nequal
    {
      inline constexpr static bool call(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
      {
        return !compute_vec4_equal<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(v1, v2);
      }
    };

    template<typename T, qualifier Q, int IsInt, std::size_t Size, bool Aligned>
    struct compute_vec4_bitwise_not
    {
      inline constexpr static vec<4, T, Q> call(vec<4, T, Q> const& v)
      {
        return vec<4, T, Q>(~v.x, ~v.y, ~v.z, ~v.w);
      }
    };
  }//namespace detail

    // -- Implicit basic constructors --















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<4, T, P> const& v)
    : x(v.x), y(v.y), z(v.z), w(v.w)
  {}

  // -- Explicit basic constructors --

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q>::vec(T scalar)
    : x(scalar), y(scalar), z(scalar), w(scalar)
  {}

  template <typename T, qualifier Q>
  inline constexpr vec<4, T, Q>::vec(T _x, T _y, T _z, T _w)
    : x(_x), y(_y), z(_z), w(_w)
  {}

  // -- Conversion scalar constructors --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.x))
    , z(static_cast<T>(v.x))
    , w(static_cast<T>(v.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, Y _y, Z _z, W _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, Z _z, W _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, Z _z, W _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z, W _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, Y _y, vec<1, Z, Q> const& _z, W _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z, W _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, W _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, W _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, Z _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, Z _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, Y _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z, typename W>
  inline constexpr vec<4, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z, vec<1, W, Q> const& _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w.x))
  {}

  // -- Conversion vector constructors --

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<2, A, P> const& _xy, B _z, C _w)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z, C _w)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<2, A, P> const& _xy, B _z, vec<1, C, P> const& _w)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z, vec<1, C, P> const& _w)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z.x))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(A _x, vec<2, B, P> const& _yz, C _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz, C _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(A _x, vec<2, B, P> const& _yz, vec<1, C, P> const& _w)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz, vec<1, C, P> const& _w)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(A _x, B _y, vec<2, C, P> const& _zw)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_zw.x))
    , w(static_cast<T>(_zw.y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, A, P> const& _x, B _y, vec<2, C, P> const& _zw)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_zw.x))
    , w(static_cast<T>(_zw.y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(A _x, vec<1, B, P> const& _y, vec<2, C, P> const& _zw)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_zw.x))
    , w(static_cast<T>(_zw.y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, typename C, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, A, P> const& _x, vec<1, B, P> const& _y, vec<2, C, P> const& _zw)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_zw.x))
    , w(static_cast<T>(_zw.y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<3, A, P> const& _xyz, B _w)
    : x(static_cast<T>(_xyz.x))
    , y(static_cast<T>(_xyz.y))
    , z(static_cast<T>(_xyz.z))
    , w(static_cast<T>(_w))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<3, A, P> const& _xyz, vec<1, B, P> const& _w)
    : x(static_cast<T>(_xyz.x))
    , y(static_cast<T>(_xyz.y))
    , z(static_cast<T>(_xyz.z))
    , w(static_cast<T>(_w.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<4, T, Q>::vec(A _x, vec<3, B, P> const& _yzw)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_yzw.x))
    , z(static_cast<T>(_yzw.y))
    , w(static_cast<T>(_yzw.z))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<1, A, P> const& _x, vec<3, B, P> const& _yzw)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_yzw.x))
    , z(static_cast<T>(_yzw.y))
    , w(static_cast<T>(_yzw.z))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<2, A, P> const& _xy, vec<2, B, P> const& _zw)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_zw.x))
    , w(static_cast<T>(_zw.y))
  {}

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<4, T, Q>::vec(vec<4, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
    , z(static_cast<T>(v.z))
    , w(static_cast<T>(v.w))
  {}

  // -- Component accesses --

  template<typename T, qualifier Q>
  inline constexpr T& vec<4, T, Q>::operator[](typename vec<4, T, Q>::length_type i)
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    case 3:
      return w;
    }
  }

  template<typename T, qualifier Q>
  inline constexpr T const& vec<4, T, Q>::operator[](typename vec<4, T, Q>::length_type i) const
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    case 3:
      return w;
    }
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator=(vec<4, U, Q> const& v)
  {
    this->x = static_cast<T>(v.x);
    this->y = static_cast<T>(v.y);
    this->z = static_cast<T>(v.z);
    this->w = static_cast<T>(v.w);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator+=(U scalar)
  {
    return (*this = detail::compute_vec4_add<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator+=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_add<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v.x)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator+=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_add<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator-=(U scalar)
  {
    return (*this = detail::compute_vec4_sub<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator-=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_sub<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v.x)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator-=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_sub<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator*=(U scalar)
  {
    return (*this = detail::compute_vec4_mul<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator*=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_mul<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v.x)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator*=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_mul<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator/=(U scalar)
  {
    return (*this = detail::compute_vec4_div<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator/=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_div<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v.x)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator/=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_div<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator++()
  {
    ++this->x;
    ++this->y;
    ++this->z;
    ++this->w;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator--()
  {
    --this->x;
    --this->y;
    --this->z;
    --this->w;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> vec<4, T, Q>::operator++(int)
  {
    vec<4, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> vec<4, T, Q>::operator--(int)
  {
    vec<4, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary bit operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator%=(U scalar)
  {
    return (*this = detail::compute_vec4_mod<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator%=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_mod<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator%=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_mod<T, Q, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator&=(U scalar)
  {
    return (*this = detail::compute_vec4_and<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator&=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_and<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator&=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_and<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator|=(U scalar)
  {
    return (*this = detail::compute_vec4_or<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator|=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_or<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator|=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_or<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator^=(U scalar)
  {
    return (*this = detail::compute_vec4_xor<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator^=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_xor<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator^=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_xor<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator<<=(U scalar)
  {
    return (*this = detail::compute_vec4_shift_left<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator<<=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_shift_left<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator<<=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_shift_left<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator>>=(U scalar)
  {
    return (*this = detail::compute_vec4_shift_right<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(scalar)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator>>=(vec<1, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_shift_right<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<4, T, Q>& vec<4, T, Q>::operator>>=(vec<4, U, Q> const& v)
  {
    return (*this = detail::compute_vec4_shift_right<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(*this, vec<4, T, Q>(v)));
  }

  // -- Unary constant operators --

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v)
  {
    return v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(0) -= v;
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v, T const& scalar)
  {
    return vec<4, T, Q>(v) += scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) += v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(v) += scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v2) += v1;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator+(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) += v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v, T const& scalar)
  {
    return vec<4, T, Q>(v) -= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) -= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) -= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) -= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator-(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) -= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v, T const& scalar)
  {
    return vec<4, T, Q>(v) *= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) *= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator*(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(v) *= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator*(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v2) *= v1;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator*(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) *= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v, T const& scalar)
  {
    return vec<4, T, Q>(v) /= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) /= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator/(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) /= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator/(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) /= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator/(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) /= v2;
  }

  // -- Binary bit operators --

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) %= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) %= v2.x;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator%(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) %= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator%(vec<1, T, Q> const& scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar.x) %= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator%(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) %= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) &= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<4, T, Q>(v) &= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator&(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) &= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator&(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) &= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator&(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) &= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) |= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) |= v2.x;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator|(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) |= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator|(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) |= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator|(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) |= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) ^= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) ^= v2.x;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator^(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) ^= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator^(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) ^= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator^(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) ^= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) <<= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) <<= v2.x;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator<<(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) <<= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator<<(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) <<= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator<<(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) <<= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v, T scalar)
  {
    return vec<4, T, Q>(v) >>= scalar;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) >>= v2.x;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator>>(T scalar, vec<4, T, Q> const& v)
  {
    return vec<4, T, Q>(scalar) >>= v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator>>(vec<1, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1.x) >>= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator>>(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return vec<4, T, Q>(v1) >>= v2;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<4, T, Q> operator~(vec<4, T, Q> const& v)
  {
    return detail::compute_vec4_bitwise_not<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(v);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline constexpr bool operator==(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return detail::compute_vec4_equal<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(v1, v2);
  }

  template<typename T, qualifier Q>
  inline constexpr bool operator!=(vec<4, T, Q> const& v1, vec<4, T, Q> const& v2)
  {
    return detail::compute_vec4_nequal<T, Q, detail::is_int<T>::value, sizeof(T) * 8, detail::is_aligned<Q>::value>::call(v1, v2);
  }

  template<qualifier Q>
  inline constexpr vec<4, bool, Q> operator&&(vec<4, bool, Q> const& v1, vec<4, bool, Q> const& v2)
  {
    return vec<4, bool, Q>(v1.x && v2.x, v1.y && v2.y, v1.z && v2.z, v1.w && v2.w);
  }

  template<qualifier Q>
  inline constexpr vec<4, bool, Q> operator||(vec<4, bool, Q> const& v1, vec<4, bool, Q> const& v2)
  {
    return vec<4, bool, Q>(v1.x || v2.x, v1.y || v2.y, v1.z || v2.z, v1.w || v2.w);
  }
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<4, 4, T, Q>
  {
    typedef vec<4, T, Q> col_type;
    typedef vec<4, T, Q> row_type;
    typedef mat<4, 4, T, Q> type;
    typedef mat<4, 4, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[4];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 4; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<4, 4, T, P> const& m);

    explicit constexpr mat(T const& x);
    constexpr mat(
      T const& x0, T const& y0, T const& z0, T const& w0,
      T const& x1, T const& y1, T const& z1, T const& w1,
      T const& x2, T const& y2, T const& z2, T const& w2,
      T const& x3, T const& y3, T const& z3, T const& w3);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2,
      col_type const& v3);

    // -- Conversions --

    template<
      typename X1, typename Y1, typename Z1, typename W1,
      typename X2, typename Y2, typename Z2, typename W2,
      typename X3, typename Y3, typename Z3, typename W3,
      typename X4, typename Y4, typename Z4, typename W4>
    constexpr mat(
      X1 const& x1, Y1 const& y1, Z1 const& z1, W1 const& w1,
      X2 const& x2, Y2 const& y2, Z2 const& z2, W2 const& w2,
      X3 const& x3, Y3 const& y3, Z3 const& z3, W3 const& w3,
      X4 const& x4, Y4 const& y4, Z4 const& z4, W4 const& w4);

    template<typename V1, typename V2, typename V3, typename V4>
    constexpr mat(
      vec<4, V1, Q> const& v1,
      vec<4, V2, Q> const& v2,
      vec<4, V3, Q> const& v3,
      vec<4, V4, Q> const& v4);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<4, 4, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<4, 4, T, Q>& operator=(mat<4, 4, U, Q> const& m);
    template<typename U>
    mat<4, 4, T, Q>& operator+=(U s);
    template<typename U>
    mat<4, 4, T, Q>& operator+=(mat<4, 4, U, Q> const& m);
    template<typename U>
    mat<4, 4, T, Q>& operator-=(U s);
    template<typename U>
    mat<4, 4, T, Q>& operator-=(mat<4, 4, U, Q> const& m);
    template<typename U>
    mat<4, 4, T, Q>& operator*=(U s);
    template<typename U>
    mat<4, 4, T, Q>& operator*=(mat<4, 4, U, Q> const& m);
    template<typename U>
    mat<4, 4, T, Q>& operator/=(U s);
    template<typename U>
    mat<4, 4, T, Q>& operator/=(mat<4, 4, U, Q> const& m);

    // -- Increment and decrement operators --

    mat<4, 4, T, Q>& operator++();
    mat<4, 4, T, Q>& operator--();
    mat<4, 4, T, Q> operator++(int);
    mat<4, 4, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator+(mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator-(mat<4, 4, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator+(mat<4, 4, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator+(T const& s, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator+(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator-(mat<4, 4, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator-(T const& s, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator-(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator*(mat<4, 4, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator*(T const& s, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<4, 4, T, Q>::col_type operator*(mat<4, 4, T, Q> const& m, typename mat<4, 4, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<4, 4, T, Q>::row_type operator*(typename mat<4, 4, T, Q>::col_type const& v, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator*(mat<4, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator*(mat<4, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator*(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator/(mat<4, 4, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator/(T const& s, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<4, 4, T, Q>::col_type operator/(mat<4, 4, T, Q> const& m, typename mat<4, 4, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<4, 4, T, Q>::row_type operator/(typename mat<4, 4, T, Q>::col_type const& v, mat<4, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator/(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<4, 4, T, Q> const& m1, mat<4, 4, T, Q> const& m2);
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct vec<2, T, Q>
  {
    // -- Implementation detail --

    typedef T value_type;
    typedef vec<2, T, Q> type;
    typedef vec<2, bool, Q> bool_type;

    // -- Data --
    union { T x, r, s; };
    union { T y, g, t; };


    // -- Component accesses --

    /// Return the count of components of the vector
    typedef length_t length_type;
    static constexpr length_type length() { return 2; }

    constexpr T& operator[](length_type i);
    constexpr T const& operator[](length_type i) const;

    // -- Implicit basic constructors --

    constexpr vec() = default;
    constexpr vec(vec const& v) = default;
    template<qualifier P>
    constexpr vec(vec<2, T, P> const& v);

    // -- Explicit basic constructors --

    constexpr explicit vec(T scalar);
    constexpr vec(T x, T y);

    // -- Conversion constructors --

    template<typename U, qualifier P>
    constexpr explicit vec(vec<1, U, P> const& v);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B>
    constexpr vec(A x, B y);
    template<typename A, typename B>
    constexpr vec(vec<1, A, Q> const& x, B y);
    template<typename A, typename B>
    constexpr vec(A x, vec<1, B, Q> const& y);
    template<typename A, typename B>
    constexpr vec(vec<1, A, Q> const& x, vec<1, B, Q> const& y);

    // -- Conversion vector constructors --

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<3, U, P> const& v);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<4, U, P> const& v);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<2, U, P> const& v);

    // -- Unary arithmetic operators --

    constexpr vec<2, T, Q>& operator=(vec const& v) = default;

    template<typename U>
    constexpr vec<2, T, Q>& operator=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator+=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator+=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator+=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator-=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator-=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator-=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator*=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator*=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator*=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator/=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator/=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator/=(vec<2, U, Q> const& v);

    // -- Increment and decrement operators --

    constexpr vec<2, T, Q>& operator++();
    constexpr vec<2, T, Q>& operator--();
    constexpr vec<2, T, Q> operator++(int);
    constexpr vec<2, T, Q> operator--(int);

    // -- Unary bit operators --

    template<typename U>
    constexpr vec<2, T, Q>& operator%=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator%=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator%=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator&=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator&=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator&=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator|=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator|=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator|=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator^=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator^=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator^=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator<<=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator<<=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator<<=(vec<2, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator>>=(U scalar);
    template<typename U>
    constexpr vec<2, T, Q>& operator>>=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<2, T, Q>& operator>>=(vec<2, U, Q> const& v);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v);

  // -- Binary operators --

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator*(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator*(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator/(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator/(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator%(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator%(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator&(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator&(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator|(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator|(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator^(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator^(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator<<(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator<<(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator>>(T scalar, vec<2, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator>>(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<2, T, Q> operator~(vec<2, T, Q> const& v);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  constexpr bool operator==(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr bool operator!=(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2);

  template<qualifier Q>
  constexpr vec<2, bool, Q> operator&&(vec<2, bool, Q> const& v1, vec<2, bool, Q> const& v2);

  template<qualifier Q>
  constexpr vec<2, bool, Q> operator||(vec<2, bool, Q> const& v1, vec<2, bool, Q> const& v2);
}//namespace glm

namespace glm
{
  // -- Implicit basic constructors --
  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr vec<2, T, Q>::vec(vec<2, T, P> const& v)
    : x(v.x), y(v.y)
  {}

  // -- Explicit basic constructors --

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q>::vec(T scalar)
    : x(scalar), y(scalar)
  {}

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q>::vec(T _x, T _y)
    : x(_x), y(_y)
  {}

  // -- Conversion scalar constructors --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<2, T, Q>::vec(vec<1, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B>
  inline constexpr vec<2, T, Q>::vec(A _x, B _y)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B>
  inline constexpr vec<2, T, Q>::vec(vec<1, A, Q> const& _x, B _y)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B>
  inline constexpr vec<2, T, Q>::vec(A _x, vec<1, B, Q> const& _y)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B>
  inline constexpr vec<2, T, Q>::vec(vec<1, A, Q> const& _x, vec<1, B, Q> const& _y)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
  {}

  // -- Conversion vector constructors --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<2, T, Q>::vec(vec<2, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
  {}

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<2, T, Q>::vec(vec<3, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
  {}

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<2, T, Q>::vec(vec<4, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
  {}

  // -- Component accesses --

  template<typename T, qualifier Q>
  inline constexpr T& vec<2, T, Q>::operator[](typename vec<2, T, Q>::length_type i)
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    }
  }

  template<typename T, qualifier Q>
  inline constexpr T const& vec<2, T, Q>::operator[](typename vec<2, T, Q>::length_type i) const
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    }
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator=(vec<2, U, Q> const& v)
  {
    this->x = static_cast<T>(v.x);
    this->y = static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator+=(U scalar)
  {
    this->x += static_cast<T>(scalar);
    this->y += static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator+=(vec<1, U, Q> const& v)
  {
    this->x += static_cast<T>(v.x);
    this->y += static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator+=(vec<2, U, Q> const& v)
  {
    this->x += static_cast<T>(v.x);
    this->y += static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator-=(U scalar)
  {
    this->x -= static_cast<T>(scalar);
    this->y -= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator-=(vec<1, U, Q> const& v)
  {
    this->x -= static_cast<T>(v.x);
    this->y -= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator-=(vec<2, U, Q> const& v)
  {
    this->x -= static_cast<T>(v.x);
    this->y -= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator*=(U scalar)
  {
    this->x *= static_cast<T>(scalar);
    this->y *= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator*=(vec<1, U, Q> const& v)
  {
    this->x *= static_cast<T>(v.x);
    this->y *= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator*=(vec<2, U, Q> const& v)
  {
    this->x *= static_cast<T>(v.x);
    this->y *= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator/=(U scalar)
  {
    this->x /= static_cast<T>(scalar);
    this->y /= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator/=(vec<1, U, Q> const& v)
  {
    this->x /= static_cast<T>(v.x);
    this->y /= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator/=(vec<2, U, Q> const& v)
  {
    this->x /= static_cast<T>(v.x);
    this->y /= static_cast<T>(v.y);
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator++()
  {
    ++this->x;
    ++this->y;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator--()
  {
    --this->x;
    --this->y;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> vec<2, T, Q>::operator++(int)
  {
    vec<2, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> vec<2, T, Q>::operator--(int)
  {
    vec<2, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary bit operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator%=(U scalar)
  {
    this->x %= static_cast<T>(scalar);
    this->y %= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator%=(vec<1, U, Q> const& v)
  {
    this->x %= static_cast<T>(v.x);
    this->y %= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator%=(vec<2, U, Q> const& v)
  {
    this->x %= static_cast<T>(v.x);
    this->y %= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator&=(U scalar)
  {
    this->x &= static_cast<T>(scalar);
    this->y &= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator&=(vec<1, U, Q> const& v)
  {
    this->x &= static_cast<T>(v.x);
    this->y &= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator&=(vec<2, U, Q> const& v)
  {
    this->x &= static_cast<T>(v.x);
    this->y &= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator|=(U scalar)
  {
    this->x |= static_cast<T>(scalar);
    this->y |= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator|=(vec<1, U, Q> const& v)
  {
    this->x |= static_cast<T>(v.x);
    this->y |= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator|=(vec<2, U, Q> const& v)
  {
    this->x |= static_cast<T>(v.x);
    this->y |= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator^=(U scalar)
  {
    this->x ^= static_cast<T>(scalar);
    this->y ^= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator^=(vec<1, U, Q> const& v)
  {
    this->x ^= static_cast<T>(v.x);
    this->y ^= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator^=(vec<2, U, Q> const& v)
  {
    this->x ^= static_cast<T>(v.x);
    this->y ^= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator<<=(U scalar)
  {
    this->x <<= static_cast<T>(scalar);
    this->y <<= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator<<=(vec<1, U, Q> const& v)
  {
    this->x <<= static_cast<T>(v.x);
    this->y <<= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator<<=(vec<2, U, Q> const& v)
  {
    this->x <<= static_cast<T>(v.x);
    this->y <<= static_cast<T>(v.y);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator>>=(U scalar)
  {
    this->x >>= static_cast<T>(scalar);
    this->y >>= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator>>=(vec<1, U, Q> const& v)
  {
    this->x >>= static_cast<T>(v.x);
    this->y >>= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<2, T, Q>& vec<2, T, Q>::operator>>=(vec<2, U, Q> const& v)
  {
    this->x >>= static_cast<T>(v.x);
    this->y >>= static_cast<T>(v.y);
    return *this;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v)
  {
    return v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      -v.x,
      -v.y);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x + scalar,
      v.y + scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x + v2.x,
      v1.y + v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar + v.x,
      scalar + v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x + v2.x,
      v1.x + v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator+(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x + v2.x,
      v1.y + v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x - scalar,
      v.y - scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x - v2.x,
      v1.y - v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar - v.x,
      scalar - v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x - v2.x,
      v1.x - v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator-(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x - v2.x,
      v1.y - v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x * scalar,
      v.y * scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x * v2.x,
      v1.y * v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator*(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar * v.x,
      scalar * v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator*(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x * v2.x,
      v1.x * v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator*(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x * v2.x,
      v1.y * v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x / scalar,
      v.y / scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x / v2.x,
      v1.y / v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator/(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar / v.x,
      scalar / v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator/(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x / v2.x,
      v1.x / v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator/(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x / v2.x,
      v1.y / v2.y);
  }

  // -- Binary bit operators --

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x % scalar,
      v.y % scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x % v2.x,
      v1.y % v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator%(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar % v.x,
      scalar % v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator%(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x % v2.x,
      v1.x % v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator%(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x % v2.x,
      v1.y % v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x & scalar,
      v.y & scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x & v2.x,
      v1.y & v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator&(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar & v.x,
      scalar & v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator&(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x & v2.x,
      v1.x & v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator&(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x & v2.x,
      v1.y & v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x | scalar,
      v.y | scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x | v2.x,
      v1.y | v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator|(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar | v.x,
      scalar | v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator|(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x | v2.x,
      v1.x | v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator|(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x | v2.x,
      v1.y | v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x ^ scalar,
      v.y ^ scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x ^ v2.x,
      v1.y ^ v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator^(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar ^ v.x,
      scalar ^ v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator^(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x ^ v2.x,
      v1.x ^ v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator^(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x ^ v2.x,
      v1.y ^ v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x << scalar,
      v.y << scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x << v2.x,
      v1.y << v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator<<(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar << v.x,
      scalar << v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator<<(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x << v2.x,
      v1.x << v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator<<(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x << v2.x,
      v1.y << v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v, T scalar)
  {
    return vec<2, T, Q>(
      v.x >> scalar,
      v.y >> scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v1, vec<1, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x >> v2.x,
      v1.y >> v2.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator>>(T scalar, vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      scalar >> v.x,
      scalar >> v.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator>>(vec<1, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x >> v2.x,
      v1.x >> v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator>>(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return vec<2, T, Q>(
      v1.x >> v2.x,
      v1.y >> v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<2, T, Q> operator~(vec<2, T, Q> const& v)
  {
    return vec<2, T, Q>(
      ~v.x,
      ~v.y);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline constexpr bool operator==(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return
      detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.x, v2.x) &&
      detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.y, v2.y);
  }

  template<typename T, qualifier Q>
  inline constexpr bool operator!=(vec<2, T, Q> const& v1, vec<2, T, Q> const& v2)
  {
    return !(v1 == v2);
  }

  template<qualifier Q>
  inline constexpr vec<2, bool, Q> operator&&(vec<2, bool, Q> const& v1, vec<2, bool, Q> const& v2)
  {
    return vec<2, bool, Q>(v1.x && v2.x, v1.y && v2.y);
  }

  template<qualifier Q>
  inline constexpr vec<2, bool, Q> operator||(vec<2, bool, Q> const& v1, vec<2, bool, Q> const& v2)
  {
    return vec<2, bool, Q>(v1.x || v2.x, v1.y || v2.y);
  }
}//namespace glm




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 2 components vector of boolean.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<2, bool, defaultp>		bvec2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_bool2_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 2 components vector of high qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, bool, highp>		highp_bvec2;

  /// 2 components vector of medium qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, bool, mediump>	mediump_bvec2;

  /// 2 components vector of low qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, bool, lowp>		lowp_bvec2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float2.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 2 components vector of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<2, float, defaultp>	vec2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float2_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 2 components vector of high single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, float, highp>		highp_vec2;

  /// 2 components vector of medium single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, float, mediump>		mediump_vec2;

  /// 2 components vector of low single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<2, float, lowp>			lowp_vec2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_double2.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 2 components vector of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<2, double, defaultp>		dvec2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_double2_precision.hpp



namespace glm {
  namespace detail
  {
    typedef std::int8_t			int8;
    typedef std::int16_t		int16;
    typedef std::int32_t		int32;

    template<>
    struct is_int<int8>
    {
      enum test { value = ~0 };
    };

    template<>
    struct is_int<int16>
    {
      enum test { value = ~0 };
    };

    template<>
    struct is_int<int64>
    {
      enum test { value = ~0 };
    };
  }//namespace detail


    /// @addtogroup ext_scalar_int_sized
    /// @{

    /// 8 bit signed integer type.
  typedef detail::int8		int8;

  /// 16 bit signed integer type.
  typedef detail::int16		int16;

  /// 32 bit signed integer type.
  typedef detail::int32		int32;

  /// 64 bit signed integer type.
  typedef detail::int64		int64;

  /// @}
}//namespace glm

namespace glm
{
  /// @addtogroup ext_vector_int2_sized
  /// @{

  /// 8 bit signed integer vector of 2 components type.
  ///
  /// @see ext_vector_int2_sized
  typedef vec<2, int8, defaultp>		i8vec2;

  /// 16 bit signed integer vector of 2 components type.
  ///
  /// @see ext_vector_int2_sized
  typedef vec<2, int16, defaultp>		i16vec2;

  /// 32 bit signed integer vector of 2 components type.
  ///
  /// @see ext_vector_int2_sized
  typedef vec<2, int32, defaultp>		i32vec2;

  /// 64 bit signed integer vector of 2 components type.
  ///
  /// @see ext_vector_int2_sized
  typedef vec<2, int64, defaultp>		i64vec2;

  /// @}
}//namespace glm

namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 2 components vector of unsigned integer numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<2, unsigned int, defaultp>		uvec2;

  /// @}
}//namespace glm

namespace glm {
  namespace detail
  {
    typedef std::uint8_t		uint8;
    typedef std::uint16_t		uint16;
    typedef std::uint32_t		uint32;

    template<>
    struct is_int<uint8>
    {
      enum test { value = ~0 };
    };

    template<>
    struct is_int<uint16>
    {
      enum test { value = ~0 };
    };

    template<>
    struct is_int<uint64>
    {
      enum test { value = ~0 };
    };
  }//namespace detail


    /// @addtogroup ext_scalar_uint_sized
    /// @{

    /// 8 bit unsigned integer type.
  typedef detail::uint8		uint8;

  /// 16 bit unsigned integer type.
  typedef detail::uint16		uint16;

  /// 32 bit unsigned integer type.
  typedef detail::uint32		uint32;

  /// 64 bit unsigned integer type.
  typedef detail::uint64		uint64;

  /// @}
}//namespace glm

namespace glm
{
  /// @addtogroup ext_vector_uint2_sized
  /// @{

  /// 8 bit unsigned integer vector of 2 components type.
  ///
  /// @see ext_vector_uint2_sized
  typedef vec<2, uint8, defaultp>		u8vec2;

  /// 16 bit unsigned integer vector of 2 components type.
  ///
  /// @see ext_vector_uint2_sized
  typedef vec<2, uint16, defaultp>	u16vec2;

  /// 32 bit unsigned integer vector of 2 components type.
  ///
  /// @see ext_vector_uint2_sized
  typedef vec<2, uint32, defaultp>	u32vec2;

  /// 64 bit unsigned integer vector of 2 components type.
  ///
  /// @see ext_vector_uint2_sized
  typedef vec<2, uint64, defaultp>	u64vec2;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct vec<3, T, Q>
  {
    // -- Implementation detail --

    typedef T value_type;
    typedef vec<3, T, Q> type;
    typedef vec<3, bool, Q> bool_type;

    // -- Data --
    union { T x, r, s; };
    union { T y, g, t; };
    union { T z, b, p; };

    // -- Component accesses --

    /// Return the count of components of the vector
    typedef length_t length_type;
    static constexpr length_type length() { return 3; }

    constexpr T& operator[](length_type i);
    constexpr T const& operator[](length_type i) const;

    // -- Implicit basic constructors --

    constexpr vec() = default;
    constexpr vec(vec const& v) = default;
    template<qualifier P>
    constexpr vec(vec<3, T, P> const& v);

    // -- Explicit basic constructors --

    constexpr explicit vec(T scalar);
    constexpr vec(T a, T b, T c);

    // -- Conversion scalar constructors --

    template<typename U, qualifier P>
    constexpr explicit vec(vec<1, U, P> const& v);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename X, typename Y, typename Z>
    constexpr vec(X x, Y y, Z z);
    template<typename X, typename Y, typename Z>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, Z _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, Z _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(X _x, Y _y, vec<1, Z, Q> const& _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z);
    template<typename X, typename Y, typename Z>
    constexpr vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z);

    // -- Conversion vector constructors --

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, B _z);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(A _x, vec<2, B, P> const& _yz);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename A, typename B, qualifier P>
    constexpr vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz);
    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<4, U, P> const& v);

    /// Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)
    template<typename U, qualifier P>
    constexpr  vec(vec<3, U, P> const& v);

    // -- Unary arithmetic operators --

    constexpr vec<3, T, Q>& operator=(vec<3, T, Q> const& v) = default;

    template<typename U>
    constexpr vec<3, T, Q>& operator=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator+=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator+=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator+=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator-=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator-=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator-=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator*=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator*=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator*=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator/=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator/=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator/=(vec<3, U, Q> const& v);

    // -- Increment and decrement operators --

    constexpr vec<3, T, Q>& operator++();
    constexpr vec<3, T, Q>& operator--();
    constexpr vec<3, T, Q> operator++(int);
    constexpr vec<3, T, Q> operator--(int);

    // -- Unary bit operators --

    template<typename U>
    constexpr vec<3, T, Q>& operator%=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator%=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator%=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator&=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator&=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator&=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator|=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator|=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator|=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator^=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator^=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator^=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator<<=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator<<=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator<<=(vec<3, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator>>=(U scalar);
    template<typename U>
    constexpr vec<3, T, Q>& operator>>=(vec<1, U, Q> const& v);
    template<typename U>
    constexpr vec<3, T, Q>& operator>>=(vec<3, U, Q> const& v);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v);

  // -- Binary operators --

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator*(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator*(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator/(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator/(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator%(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator%(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v1, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator&(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator&(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator|(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator|(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator^(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator^(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator<<(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator<<(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v, T scalar);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v1, vec<1, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator>>(T scalar, vec<3, T, Q> const& v);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator>>(vec<1, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr vec<3, T, Q> operator~(vec<3, T, Q> const& v);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  constexpr bool operator==(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<typename T, qualifier Q>
  constexpr bool operator!=(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2);

  template<qualifier Q>
  constexpr vec<3, bool, Q> operator&&(vec<3, bool, Q> const& v1, vec<3, bool, Q> const& v2);

  template<qualifier Q>
  constexpr vec<3, bool, Q> operator||(vec<3, bool, Q> const& v1, vec<3, bool, Q> const& v2);
}//namespace glm

/// @ref core

namespace glm
{
  // -- Implicit basic constructors --

  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<3, T, P> const& v)
    : x(v.x), y(v.y), z(v.z)
  {}

  // -- Explicit basic constructors --

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q>::vec(T scalar)
    : x(scalar), y(scalar), z(scalar)
  {}

  template <typename T, qualifier Q>
  inline constexpr vec<3, T, Q>::vec(T _x, T _y, T _z)
    : x(_x), y(_y), z(_z)
  {}

  // -- Conversion scalar constructors --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<1, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.x))
    , z(static_cast<T>(v.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(X _x, Y _y, Z _z)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, Z _z)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, Z _z)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, Z _z)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(X _x, Y _y, vec<1, Z, Q> const& _z)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(vec<1, X, Q> const& _x, Y _y, vec<1, Z, Q> const& _z)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y))
    , z(static_cast<T>(_z.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(X _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
  {}

  template<typename T, qualifier Q>
  template<typename X, typename Y, typename Z>
  inline constexpr vec<3, T, Q>::vec(vec<1, X, Q> const& _x, vec<1, Y, Q> const& _y, vec<1, Z, Q> const& _z)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_y.x))
    , z(static_cast<T>(_z.x))
  {}

  // -- Conversion vector constructors --

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<2, A, P> const& _xy, B _z)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<2, A, P> const& _xy, vec<1, B, P> const& _z)
    : x(static_cast<T>(_xy.x))
    , y(static_cast<T>(_xy.y))
    , z(static_cast<T>(_z.x))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<3, T, Q>::vec(A _x, vec<2, B, P> const& _yz)
    : x(static_cast<T>(_x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
  {}

  template<typename T, qualifier Q>
  template<typename A, typename B, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<1, A, P> const& _x, vec<2, B, P> const& _yz)
    : x(static_cast<T>(_x.x))
    , y(static_cast<T>(_yz.x))
    , z(static_cast<T>(_yz.y))
  {}

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<3, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
    , z(static_cast<T>(v.z))
  {}

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr vec<3, T, Q>::vec(vec<4, U, P> const& v)
    : x(static_cast<T>(v.x))
    , y(static_cast<T>(v.y))
    , z(static_cast<T>(v.z))
  {}

  // -- Component accesses --

  template<typename T, qualifier Q>
  inline constexpr T& vec<3, T, Q>::operator[](typename vec<3, T, Q>::length_type i)
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    }
  }

  template<typename T, qualifier Q>
  inline constexpr T const& vec<3, T, Q>::operator[](typename vec<3, T, Q>::length_type i) const
  {
    assert(i >= 0 && i < this->length());
    switch (i)
    {
    default:
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    }
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator=(vec<3, U, Q> const& v)
  {
    this->x = static_cast<T>(v.x);
    this->y = static_cast<T>(v.y);
    this->z = static_cast<T>(v.z);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator+=(U scalar)
  {
    this->x += static_cast<T>(scalar);
    this->y += static_cast<T>(scalar);
    this->z += static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator+=(vec<1, U, Q> const& v)
  {
    this->x += static_cast<T>(v.x);
    this->y += static_cast<T>(v.x);
    this->z += static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator+=(vec<3, U, Q> const& v)
  {
    this->x += static_cast<T>(v.x);
    this->y += static_cast<T>(v.y);
    this->z += static_cast<T>(v.z);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator-=(U scalar)
  {
    this->x -= static_cast<T>(scalar);
    this->y -= static_cast<T>(scalar);
    this->z -= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator-=(vec<1, U, Q> const& v)
  {
    this->x -= static_cast<T>(v.x);
    this->y -= static_cast<T>(v.x);
    this->z -= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator-=(vec<3, U, Q> const& v)
  {
    this->x -= static_cast<T>(v.x);
    this->y -= static_cast<T>(v.y);
    this->z -= static_cast<T>(v.z);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator*=(U scalar)
  {
    this->x *= static_cast<T>(scalar);
    this->y *= static_cast<T>(scalar);
    this->z *= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator*=(vec<1, U, Q> const& v)
  {
    this->x *= static_cast<T>(v.x);
    this->y *= static_cast<T>(v.x);
    this->z *= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator*=(vec<3, U, Q> const& v)
  {
    this->x *= static_cast<T>(v.x);
    this->y *= static_cast<T>(v.y);
    this->z *= static_cast<T>(v.z);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator/=(U v)
  {
    this->x /= static_cast<T>(v);
    this->y /= static_cast<T>(v);
    this->z /= static_cast<T>(v);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator/=(vec<1, U, Q> const& v)
  {
    this->x /= static_cast<T>(v.x);
    this->y /= static_cast<T>(v.x);
    this->z /= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator/=(vec<3, U, Q> const& v)
  {
    this->x /= static_cast<T>(v.x);
    this->y /= static_cast<T>(v.y);
    this->z /= static_cast<T>(v.z);
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator++()
  {
    ++this->x;
    ++this->y;
    ++this->z;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator--()
  {
    --this->x;
    --this->y;
    --this->z;
    return *this;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> vec<3, T, Q>::operator++(int)
  {
    vec<3, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> vec<3, T, Q>::operator--(int)
  {
    vec<3, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary bit operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator%=(U scalar)
  {
    this->x %= scalar;
    this->y %= scalar;
    this->z %= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator%=(vec<1, U, Q> const& v)
  {
    this->x %= v.x;
    this->y %= v.x;
    this->z %= v.x;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator%=(vec<3, U, Q> const& v)
  {
    this->x %= v.x;
    this->y %= v.y;
    this->z %= v.z;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator&=(U scalar)
  {
    this->x &= scalar;
    this->y &= scalar;
    this->z &= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator&=(vec<1, U, Q> const& v)
  {
    this->x &= v.x;
    this->y &= v.x;
    this->z &= v.x;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator&=(vec<3, U, Q> const& v)
  {
    this->x &= v.x;
    this->y &= v.y;
    this->z &= v.z;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator|=(U scalar)
  {
    this->x |= scalar;
    this->y |= scalar;
    this->z |= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator|=(vec<1, U, Q> const& v)
  {
    this->x |= v.x;
    this->y |= v.x;
    this->z |= v.x;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator|=(vec<3, U, Q> const& v)
  {
    this->x |= v.x;
    this->y |= v.y;
    this->z |= v.z;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator^=(U scalar)
  {
    this->x ^= scalar;
    this->y ^= scalar;
    this->z ^= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator^=(vec<1, U, Q> const& v)
  {
    this->x ^= v.x;
    this->y ^= v.x;
    this->z ^= v.x;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator^=(vec<3, U, Q> const& v)
  {
    this->x ^= v.x;
    this->y ^= v.y;
    this->z ^= v.z;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator<<=(U scalar)
  {
    this->x <<= scalar;
    this->y <<= scalar;
    this->z <<= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator<<=(vec<1, U, Q> const& v)
  {
    this->x <<= static_cast<T>(v.x);
    this->y <<= static_cast<T>(v.x);
    this->z <<= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator<<=(vec<3, U, Q> const& v)
  {
    this->x <<= static_cast<T>(v.x);
    this->y <<= static_cast<T>(v.y);
    this->z <<= static_cast<T>(v.z);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator>>=(U scalar)
  {
    this->x >>= static_cast<T>(scalar);
    this->y >>= static_cast<T>(scalar);
    this->z >>= static_cast<T>(scalar);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator>>=(vec<1, U, Q> const& v)
  {
    this->x >>= static_cast<T>(v.x);
    this->y >>= static_cast<T>(v.x);
    this->z >>= static_cast<T>(v.x);
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline constexpr vec<3, T, Q>& vec<3, T, Q>::operator>>=(vec<3, U, Q> const& v)
  {
    this->x >>= static_cast<T>(v.x);
    this->y >>= static_cast<T>(v.y);
    this->z >>= static_cast<T>(v.z);
    return *this;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v)
  {
    return v;
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      -v.x,
      -v.y,
      -v.z);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x + scalar,
      v.y + scalar,
      v.z + scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x + scalar.x,
      v.y + scalar.x,
      v.z + scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar + v.x,
      scalar + v.y,
      scalar + v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x + v.x,
      scalar.x + v.y,
      scalar.x + v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator+(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x + v2.x,
      v1.y + v2.y,
      v1.z + v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x - scalar,
      v.y - scalar,
      v.z - scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x - scalar.x,
      v.y - scalar.x,
      v.z - scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar - v.x,
      scalar - v.y,
      scalar - v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x - v.x,
      scalar.x - v.y,
      scalar.x - v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator-(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x - v2.x,
      v1.y - v2.y,
      v1.z - v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x * scalar,
      v.y * scalar,
      v.z * scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x * scalar.x,
      v.y * scalar.x,
      v.z * scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator*(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar * v.x,
      scalar * v.y,
      scalar * v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator*(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x * v.x,
      scalar.x * v.y,
      scalar.x * v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator*(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x * v2.x,
      v1.y * v2.y,
      v1.z * v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x / scalar,
      v.y / scalar,
      v.z / scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x / scalar.x,
      v.y / scalar.x,
      v.z / scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator/(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar / v.x,
      scalar / v.y,
      scalar / v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator/(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x / v.x,
      scalar.x / v.y,
      scalar.x / v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator/(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x / v2.x,
      v1.y / v2.y,
      v1.z / v2.z);
  }

  // -- Binary bit operators --

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x % scalar,
      v.y % scalar,
      v.z % scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x % scalar.x,
      v.y % scalar.x,
      v.z % scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator%(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar % v.x,
      scalar % v.y,
      scalar % v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator%(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x % v.x,
      scalar.x % v.y,
      scalar.x % v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator%(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x % v2.x,
      v1.y % v2.y,
      v1.z % v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x & scalar,
      v.y & scalar,
      v.z & scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x & scalar.x,
      v.y & scalar.x,
      v.z & scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator&(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar & v.x,
      scalar & v.y,
      scalar & v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator&(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x & v.x,
      scalar.x & v.y,
      scalar.x & v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator&(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x & v2.x,
      v1.y & v2.y,
      v1.z & v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x | scalar,
      v.y | scalar,
      v.z | scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x | scalar.x,
      v.y | scalar.x,
      v.z | scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator|(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar | v.x,
      scalar | v.y,
      scalar | v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator|(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x | v.x,
      scalar.x | v.y,
      scalar.x | v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator|(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x | v2.x,
      v1.y | v2.y,
      v1.z | v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x ^ scalar,
      v.y ^ scalar,
      v.z ^ scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x ^ scalar.x,
      v.y ^ scalar.x,
      v.z ^ scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator^(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar ^ v.x,
      scalar ^ v.y,
      scalar ^ v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator^(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x ^ v.x,
      scalar.x ^ v.y,
      scalar.x ^ v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator^(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x ^ v2.x,
      v1.y ^ v2.y,
      v1.z ^ v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x << scalar,
      v.y << scalar,
      v.z << scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x << scalar.x,
      v.y << scalar.x,
      v.z << scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator<<(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar << v.x,
      scalar << v.y,
      scalar << v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator<<(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x << v.x,
      scalar.x << v.y,
      scalar.x << v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator<<(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x << v2.x,
      v1.y << v2.y,
      v1.z << v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v, T scalar)
  {
    return vec<3, T, Q>(
      v.x >> scalar,
      v.y >> scalar,
      v.z >> scalar);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v, vec<1, T, Q> const& scalar)
  {
    return vec<3, T, Q>(
      v.x >> scalar.x,
      v.y >> scalar.x,
      v.z >> scalar.x);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator>>(T scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar >> v.x,
      scalar >> v.y,
      scalar >> v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator>>(vec<1, T, Q> const& scalar, vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      scalar.x >> v.x,
      scalar.x >> v.y,
      scalar.x >> v.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator>>(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return vec<3, T, Q>(
      v1.x >> v2.x,
      v1.y >> v2.y,
      v1.z >> v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr vec<3, T, Q> operator~(vec<3, T, Q> const& v)
  {
    return vec<3, T, Q>(
      ~v.x,
      ~v.y,
      ~v.z);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline constexpr bool operator==(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return
      detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.x, v2.x) &&
      detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.y, v2.y) &&
      detail::compute_equal<T, std::numeric_limits<T>::is_iec559>::call(v1.z, v2.z);
  }

  template<typename T, qualifier Q>
  inline constexpr bool operator!=(vec<3, T, Q> const& v1, vec<3, T, Q> const& v2)
  {
    return !(v1 == v2);
  }

  template<qualifier Q>
  inline constexpr vec<3, bool, Q> operator&&(vec<3, bool, Q> const& v1, vec<3, bool, Q> const& v2)
  {
    return vec<3, bool, Q>(v1.x && v2.x, v1.y && v2.y, v1.z && v2.z);
  }

  template<qualifier Q>
  inline constexpr vec<3, bool, Q> operator||(vec<3, bool, Q> const& v1, vec<3, bool, Q> const& v2)
  {
    return vec<3, bool, Q>(v1.x || v2.x, v1.y || v2.y, v1.z || v2.z);
  }
}//namespace glm




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 3 components vector of boolean.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<3, bool, defaultp>		bvec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_bool3_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 3 components vector of high qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, bool, highp>		highp_bvec3;

  /// 3 components vector of medium qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, bool, mediump>	mediump_bvec3;

  /// 3 components vector of low qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, bool, lowp>		lowp_bvec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float3.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 3 components vector of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<3, float, defaultp>		vec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float3_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 3 components vector of high single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, float, highp>		highp_vec3;

  /// 3 components vector of medium single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, float, mediump>		mediump_vec3;

  /// 3 components vector of low single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, float, lowp>			lowp_vec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_double3.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 3 components vector of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<3, double, defaultp>		dvec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_double3_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 3 components vector of high double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, double, highp>		highp_dvec3;

  /// 3 components vector of medium double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, double, mediump>		mediump_dvec3;

  /// 3 components vector of low double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<3, double, lowp>		lowp_dvec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_int3.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 3 components vector of signed integer numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<3, int, defaultp>		ivec3;

  /// @}
}//namespace glm


/// @ref ext_vector_int3_sized
/// @file glm/ext/vector_int3_sized.hpp
///
/// @defgroup ext_vector_int3_sized GLM_EXT_vector_int3_sized
/// @ingroup ext
///
/// Exposes sized signed integer vector of 3 components type.
///
/// Include <glm/ext/vector_int3_sized.hpp> to use the features of this extension.
///
/// @see ext_scalar_int_sized
/// @see ext_vector_uint3_sized










namespace glm
{
  /// @addtogroup ext_vector_int3_sized
  /// @{

  /// 8 bit signed integer vector of 3 components type.
  ///
  /// @see ext_vector_int3_sized
  typedef vec<3, int8, defaultp>		i8vec3;

  /// 16 bit signed integer vector of 3 components type.
  ///
  /// @see ext_vector_int3_sized
  typedef vec<3, int16, defaultp>		i16vec3;

  /// 32 bit signed integer vector of 3 components type.
  ///
  /// @see ext_vector_int3_sized
  typedef vec<3, int32, defaultp>		i32vec3;

  /// 64 bit signed integer vector of 3 components type.
  ///
  /// @see ext_vector_int3_sized
  typedef vec<3, int64, defaultp>		i64vec3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_uint3.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 3 components vector of unsigned integer numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<3, unsigned int, defaultp>		uvec3;

  /// @}
}//namespace glm


/// @ref ext_vector_uint3_sized
/// @file glm/ext/vector_uint3_sized.hpp
///
/// @defgroup ext_vector_uint3_sized GLM_EXT_vector_uint3_sized
/// @ingroup ext
///
/// Exposes sized unsigned integer vector of 3 components type.
///
/// Include <glm/ext/vector_uint3_sized.hpp> to use the features of this extension.
///
/// @see ext_scalar_uint_sized
/// @see ext_vector_int3_sized










namespace glm
{
  /// @addtogroup ext_vector_uint3_sized
  /// @{

  /// 8 bit unsigned integer vector of 3 components type.
  ///
  /// @see ext_vector_uint3_sized
  typedef vec<3, uint8, defaultp>		u8vec3;

  /// 16 bit unsigned integer vector of 3 components type.
  ///
  /// @see ext_vector_uint3_sized
  typedef vec<3, uint16, defaultp>	u16vec3;

  /// 32 bit unsigned integer vector of 3 components type.
  ///
  /// @see ext_vector_uint3_sized
  typedef vec<3, uint32, defaultp>	u32vec3;

  /// 64 bit unsigned integer vector of 3 components type.
  ///
  /// @see ext_vector_uint3_sized
  typedef vec<3, uint64, defaultp>	u64vec3;

  /// @}
}//namespace glm



/// @ref core
/// @file glm/vec4.hpp



/// @ref core
/// @file glm/ext/vector_bool4.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 4 components vector of boolean.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<4, bool, defaultp>		bvec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_bool4_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 4 components vector of high qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, bool, highp>		highp_bvec4;

  /// 4 components vector of medium qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, bool, mediump>	mediump_bvec4;

  /// 4 components vector of low qualifier bool numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, bool, lowp>		lowp_bvec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float4.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 4 components vector of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<4, float, defaultp>		vec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_float4_precision.hpp




namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 4 components vector of high single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, float, highp>		highp_vec4;

  /// 4 components vector of medium single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, float, mediump>		mediump_vec4;

  /// 4 components vector of low single-qualifier floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, float, lowp>			lowp_vec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_double4.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 4 components vector of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<4, double, defaultp>		dvec4;

  /// @}
}//namespace glm


namespace glm
{
  /// @addtogroup core_vector_precision
  /// @{

  /// 4 components vector of high double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, double, highp>		highp_dvec4;

  /// 4 components vector of medium double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, double, mediump>		mediump_dvec4;

  /// 4 components vector of low double-qualifier floating-point numbers.
  /// There is no guarantee on the actual qualifier.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef vec<4, double, lowp>		lowp_dvec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_int4.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 4 components vector of signed integer numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<4, int, defaultp>		ivec4;

  /// @}
}//namespace glm


/// @ref ext_vector_int4_sized
/// @file glm/ext/vector_int4_sized.hpp
///
/// @defgroup ext_vector_int4_sized GLM_EXT_vector_int4_sized
/// @ingroup ext
///
/// Exposes sized signed integer vector of 4 components type.
///
/// Include <glm/ext/vector_int4_sized.hpp> to use the features of this extension.
///
/// @see ext_scalar_int_sized
/// @see ext_vector_uint4_sized










namespace glm
{
  /// @addtogroup ext_vector_int4_sized
  /// @{

  /// 8 bit signed integer vector of 4 components type.
  ///
  /// @see ext_vector_int4_sized
  typedef vec<4, int8, defaultp>		i8vec4;

  /// 16 bit signed integer vector of 4 components type.
  ///
  /// @see ext_vector_int4_sized
  typedef vec<4, int16, defaultp>		i16vec4;

  /// 32 bit signed integer vector of 4 components type.
  ///
  /// @see ext_vector_int4_sized
  typedef vec<4, int32, defaultp>		i32vec4;

  /// 64 bit signed integer vector of 4 components type.
  ///
  /// @see ext_vector_int4_sized
  typedef vec<4, int64, defaultp>		i64vec4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/vector_uint4.hpp




namespace glm
{
  /// @addtogroup core_vector
  /// @{

  /// 4 components vector of unsigned integer numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.5 Vectors</a>
  typedef vec<4, unsigned int, defaultp>		uvec4;

  /// @}
}//namespace glm


/// @ref ext_vector_uint4_sized
/// @file glm/ext/vector_uint4_sized.hpp
///
/// @defgroup ext_vector_uint4_sized GLM_EXT_vector_uint4_sized
/// @ingroup ext
///
/// Exposes sized unsigned integer vector of 4 components type.
///
/// Include <glm/ext/vector_uint4_sized.hpp> to use the features of this extension.
///
/// @see ext_scalar_uint_sized
/// @see ext_vector_int4_sized










namespace glm
{
  /// @addtogroup ext_vector_uint4_sized
  /// @{

  /// 8 bit unsigned integer vector of 4 components type.
  ///
  /// @see ext_vector_uint4_sized
  typedef vec<4, uint8, defaultp>		u8vec4;

  /// 16 bit unsigned integer vector of 4 components type.
  ///
  /// @see ext_vector_uint4_sized
  typedef vec<4, uint16, defaultp>	u16vec4;

  /// 32 bit unsigned integer vector of 4 components type.
  ///
  /// @see ext_vector_uint4_sized
  typedef vec<4, uint32, defaultp>	u32vec4;

  /// 64 bit unsigned integer vector of 4 components type.
  ///
  /// @see ext_vector_uint4_sized
  typedef vec<4, uint64, defaultp>	u64vec4;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<2, 2, T, Q>
  {
    typedef vec<2, T, Q> col_type;
    typedef vec<2, T, Q> row_type;
    typedef mat<2, 2, T, Q> type;
    typedef mat<2, 2, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[2];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 2; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<2, 2, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T const& x1, T const& y1,
      T const& x2, T const& y2);
    constexpr mat(
      col_type const& v1,
      col_type const& v2);

    // -- Conversions --

    template<typename U, typename V, typename M, typename N>
    constexpr mat(
      U const& x1, V const& y1,
      M const& x2, N const& y2);

    template<typename U, typename V>
    constexpr mat(
      vec<2, U, Q> const& v1,
      vec<2, V, Q> const& v2);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<2, 2, U, P> const& m);

    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<2, 2, T, Q>& operator=(mat<2, 2, U, Q> const& m);
    template<typename U>
    mat<2, 2, T, Q>& operator+=(U s);
    template<typename U>
    mat<2, 2, T, Q>& operator+=(mat<2, 2, U, Q> const& m);
    template<typename U>
    mat<2, 2, T, Q>& operator-=(U s);
    template<typename U>
    mat<2, 2, T, Q>& operator-=(mat<2, 2, U, Q> const& m);
    template<typename U>
    mat<2, 2, T, Q>& operator*=(U s);
    template<typename U>
    mat<2, 2, T, Q>& operator*=(mat<2, 2, U, Q> const& m);
    template<typename U>
    mat<2, 2, T, Q>& operator/=(U s);
    template<typename U>
    mat<2, 2, T, Q>& operator/=(mat<2, 2, U, Q> const& m);

    // -- Increment and decrement operators --

    mat<2, 2, T, Q>& operator++ ();
    mat<2, 2, T, Q>& operator-- ();
    mat<2, 2, T, Q> operator++(int);
    mat<2, 2, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator+(T scalar, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator-(T scalar, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator*(mat<2, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator*(T scalar, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<2, 2, T, Q>::col_type operator*(mat<2, 2, T, Q> const& m, typename mat<2, 2, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<2, 2, T, Q>::row_type operator*(typename mat<2, 2, T, Q>::col_type const& v, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator/(mat<2, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator/(T scalar, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<2, 2, T, Q>::col_type operator/(mat<2, 2, T, Q> const& m, typename mat<2, 2, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<2, 2, T, Q>::row_type operator/(typename mat<2, 2, T, Q>::col_type const& v, mat<2, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator/(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2);
} //namespace glm





namespace glm
{
  // -- Constructors --















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<2, 2, T, Q>::mat(mat<2, 2, T, P> const& m)

    : value{ m[0], m[1] }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(T scalar)

    : value{ col_type(scalar, 0), col_type(0, scalar) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat
  (
    T const& x0, T const& y0,
    T const& x1, T const& y1
  )

    : value{ col_type(x0, y0), col_type(x1, y1) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(col_type const& v0, col_type const& v1)

    : value{ v0, v1 }

  {




  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<typename X1, typename Y1, typename X2, typename Y2>
  inline constexpr mat<2, 2, T, Q>::mat
  (
    X1 const& x1, Y1 const& y1,
    X2 const& x2, Y2 const& y2
  )

    : value{ col_type(static_cast<T>(x1), value_type(y1)), col_type(static_cast<T>(x2), value_type(y2)) }

  {




  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2>
  inline constexpr mat<2, 2, T, Q>::mat(vec<2, V1, Q> const& v1, vec<2, V2, Q> const& v2)

    : value{ col_type(v1), col_type(v2) }

  {




  }

  // -- mat2x2 matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<2, 2, T, Q>::mat(mat<2, 2, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 2, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<2, 2, T, Q>::col_type& mat<2, 2, T, Q>::operator[](typename mat<2, 2, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<2, 2, T, Q>::col_type const& mat<2, 2, T, Q>::operator[](typename mat<2, 2, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator=(mat<2, 2, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator+=(U scalar)
  {
    this->value[0] += scalar;
    this->value[1] += scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator+=(mat<2, 2, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator-=(U scalar)
  {
    this->value[0] -= scalar;
    this->value[1] -= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator-=(mat<2, 2, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator*=(U scalar)
  {
    this->value[0] *= scalar;
    this->value[1] *= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator*=(mat<2, 2, U, Q> const& m)
  {
    return (*this = *this * m);
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator/=(U scalar)
  {
    this->value[0] /= scalar;
    this->value[1] /= scalar;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator/=(mat<2, 2, U, Q> const& m)
  {
    return *this *= inverse(m);
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q>& mat<2, 2, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> mat<2, 2, T, Q>::operator++(int)
  {
    mat<2, 2, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> mat<2, 2, T, Q>::operator--(int)
  {
    mat<2, 2, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m)
  {
    return mat<2, 2, T, Q>(
      -m[0],
      -m[1]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m, T scalar)
  {
    return mat<2, 2, T, Q>(
      m[0] + scalar,
      m[1] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator+(T scalar, mat<2, 2, T, Q> const& m)
  {
    return mat<2, 2, T, Q>(
      m[0] + scalar,
      m[1] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator+(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return mat<2, 2, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m, T scalar)
  {
    return mat<2, 2, T, Q>(
      m[0] - scalar,
      m[1] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator-(T scalar, mat<2, 2, T, Q> const& m)
  {
    return mat<2, 2, T, Q>(
      scalar - m[0],
      scalar - m[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator-(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return mat<2, 2, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator*(mat<2, 2, T, Q> const& m, T scalar)
  {
    return mat<2, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator*(T scalar, mat<2, 2, T, Q> const& m)
  {
    return mat<2, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 2, T, Q>::col_type operator*
    (
      mat<2, 2, T, Q> const& m,
      typename mat<2, 2, T, Q>::row_type const& v
      )
  {
    return vec<2, T, Q>(
      m[0][0] * v.x + m[1][0] * v.y,
      m[0][1] * v.x + m[1][1] * v.y);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 2, T, Q>::row_type operator*
    (
      typename mat<2, 2, T, Q>::col_type const& v,
      mat<2, 2, T, Q> const& m
      )
  {
    return vec<2, T, Q>(
      v.x * m[0][0] + v.y * m[0][1],
      v.x * m[1][0] + v.y * m[1][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return mat<2, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return mat<3, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator*(mat<2, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return mat<4, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator/(mat<2, 2, T, Q> const& m, T scalar)
  {
    return mat<2, 2, T, Q>(
      m[0] / scalar,
      m[1] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator/(T scalar, mat<2, 2, T, Q> const& m)
  {
    return mat<2, 2, T, Q>(
      scalar / m[0],
      scalar / m[1]);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 2, T, Q>::col_type operator/(mat<2, 2, T, Q> const& m, typename mat<2, 2, T, Q>::row_type const& v)
  {
    return inverse(m) * v;
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 2, T, Q>::row_type operator/(typename mat<2, 2, T, Q>::col_type const& v, mat<2, 2, T, Q> const& m)
  {
    return v * inverse(m);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator/(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    mat<2, 2, T, Q> m1_copy(m1);
    return m1_copy /= m2;
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<2, 2, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 2 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 2, double, defaultp>		dmat2x2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 2, double, defaultp>		dmat2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double2x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, lowp>		lowp_dmat2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, mediump>	mediump_dmat2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, highp>	highp_dmat2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, lowp>		lowp_dmat2x2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, mediump>	mediump_dmat2x2;

  /// 2 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, double, highp>	highp_dmat2x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x2.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 2 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 2, float, defaultp>		mat2x2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 2, float, defaultp>		mat2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, lowp>		lowp_mat2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, mediump>	mediump_mat2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, highp>		highp_mat2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, lowp>		lowp_mat2x2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, mediump>	mediump_mat2x2;

  /// 2 columns of 2 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 2, float, highp>		highp_mat2x2;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<2, 3, T, Q>
  {
    typedef vec<3, T, Q> col_type;
    typedef vec<2, T, Q> row_type;
    typedef mat<2, 3, T, Q> type;
    typedef mat<3, 2, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[2];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 2; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<2, 3, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0, T z0,
      T x1, T y1, T z1);
    constexpr mat(
      col_type const& v0,
      col_type const& v1);

    // -- Conversions --

    template<typename X1, typename Y1, typename Z1, typename X2, typename Y2, typename Z2>
    constexpr mat(
      X1 x1, Y1 y1, Z1 z1,
      X2 x2, Y2 y2, Z2 z2);

    template<typename U, typename V>
    constexpr mat(
      vec<3, U, Q> const& v1,
      vec<3, V, Q> const& v2);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<2, 3, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<2, 3, T, Q>& operator=(mat<2, 3, U, Q> const& m);
    template<typename U>
    mat<2, 3, T, Q>& operator+=(U s);
    template<typename U>
    mat<2, 3, T, Q>& operator+=(mat<2, 3, U, Q> const& m);
    template<typename U>
    mat<2, 3, T, Q>& operator-=(U s);
    template<typename U>
    mat<2, 3, T, Q>& operator-=(mat<2, 3, U, Q> const& m);
    template<typename U>
    mat<2, 3, T, Q>& operator*=(U s);
    template<typename U>
    mat<2, 3, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<2, 3, T, Q>& operator++ ();
    mat<2, 3, T, Q>& operator-- ();
    mat<2, 3, T, Q> operator++(int);
    mat<2, 3, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator*(mat<2, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator*(T scalar, mat<2, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<2, 3, T, Q>::col_type operator*(mat<2, 3, T, Q> const& m, typename mat<2, 3, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<2, 3, T, Q>::row_type operator*(typename mat<2, 3, T, Q>::col_type const& v, mat<2, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator/(mat<2, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator/(T scalar, mat<2, 3, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2);
}//namespace glm



namespace glm
{
  // -- Constructors --















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<2, 3, T, Q>::mat(mat<2, 3, T, P> const& m)

    : value{ m.value[0], m.value[1] }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(T scalar)

    : value{ col_type(scalar, 0, 0), col_type(0, scalar, 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat
  (
    T x0, T y0, T z0,
    T x1, T y1, T z1
  )

    : value{ col_type(x0, y0, z0), col_type(x1, y1, z1) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(col_type const& v0, col_type const& v1)

    : value{ col_type(v0), col_type(v1) }

  {




  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X1, typename Y1, typename Z1,
    typename X2, typename Y2, typename Z2>
  inline constexpr mat<2, 3, T, Q>::mat
  (
    X1 x1, Y1 y1, Z1 z1,
    X2 x2, Y2 y2, Z2 z2
  )

    : value{ col_type(x1, y1, z1), col_type(x2, y2, z2) }

  {




  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2>
  inline constexpr mat<2, 3, T, Q>::mat(vec<3, V1, Q> const& v1, vec<3, V2, Q> const& v2)

    : value{ col_type(v1), col_type(v2) }

  {




  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<2, 3, T, Q>::mat(mat<2, 3, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr  mat<2, 3, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 3, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<2, 3, T, Q>::col_type& mat<2, 3, T, Q>::operator[](typename mat<2, 3, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<2, 3, T, Q>::col_type const& mat<2, 3, T, Q>::operator[](typename mat<2, 3, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator=(mat<2, 3, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator+=(mat<2, 3, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator-=(mat<2, 3, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q>& mat<2, 3, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> mat<2, 3, T, Q>::operator++(int)
  {
    mat<2, 3, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> mat<2, 3, T, Q>::operator--(int)
  {
    mat<2, 3, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m)
  {
    return mat<2, 3, T, Q>(
      -m[0],
      -m[1]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m, T scalar)
  {
    return mat<2, 3, T, Q>(
      m[0] + scalar,
      m[1] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator+(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return mat<2, 3, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m, T scalar)
  {
    return mat<2, 3, T, Q>(
      m[0] - scalar,
      m[1] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator-(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return mat<2, 3, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator*(mat<2, 3, T, Q> const& m, T scalar)
  {
    return mat<2, 3, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator*(T scalar, mat<2, 3, T, Q> const& m)
  {
    return mat<2, 3, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 3, T, Q>::col_type operator*
    (
      mat<2, 3, T, Q> const& m,
      typename mat<2, 3, T, Q>::row_type const& v)
  {
    return typename mat<2, 3, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y,
      m[0][1] * v.x + m[1][1] * v.y,
      m[0][2] * v.x + m[1][2] * v.y);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 3, T, Q>::row_type operator*
    (
      typename mat<2, 3, T, Q>::col_type const& v,
      mat<2, 3, T, Q> const& m)
  {
    return typename mat<2, 3, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1] + v.z * m[0][2],
      v.x * m[1][0] + v.y * m[1][1] + v.z * m[1][2]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return mat<2, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    T SrcA00 = m1[0][0];
    T SrcA01 = m1[0][1];
    T SrcA02 = m1[0][2];
    T SrcA10 = m1[1][0];
    T SrcA11 = m1[1][1];
    T SrcA12 = m1[1][2];

    T SrcB00 = m2[0][0];
    T SrcB01 = m2[0][1];
    T SrcB10 = m2[1][0];
    T SrcB11 = m2[1][1];
    T SrcB20 = m2[2][0];
    T SrcB21 = m2[2][1];

    mat<3, 3, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01;
    Result[0][2] = SrcA02 * SrcB00 + SrcA12 * SrcB01;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11;
    Result[1][2] = SrcA02 * SrcB10 + SrcA12 * SrcB11;
    Result[2][0] = SrcA00 * SrcB20 + SrcA10 * SrcB21;
    Result[2][1] = SrcA01 * SrcB20 + SrcA11 * SrcB21;
    Result[2][2] = SrcA02 * SrcB20 + SrcA12 * SrcB21;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator*(mat<2, 3, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return mat<4, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1],
      m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1],
      m1[0][2] * m2[3][0] + m1[1][2] * m2[3][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator/(mat<2, 3, T, Q> const& m, T scalar)
  {
    return mat<2, 3, T, Q>(
      m[0] / scalar,
      m[1] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator/(T scalar, mat<2, 3, T, Q> const& m)
  {
    return mat<2, 3, T, Q>(
      scalar / m[0],
      scalar / m[1]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<2, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 3 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 3, double, defaultp>		dmat2x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double2x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 3 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, double, lowp>		lowp_dmat2x3;

  /// 2 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, double, mediump>	mediump_dmat2x3;

  /// 2 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, double, highp>	highp_dmat2x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x3.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 3 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 3, float, defaultp>		mat2x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 3 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, float, lowp>		lowp_mat2x3;

  /// 2 columns of 3 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, float, mediump>	mediump_mat2x3;

  /// 2 columns of 3 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 3, float, highp>		highp_mat2x3;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<2, 4, T, Q>
  {
    typedef vec<4, T, Q> col_type;
    typedef vec<2, T, Q> row_type;
    typedef mat<2, 4, T, Q> type;
    typedef mat<4, 2, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[2];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 2; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<2, 4, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0, T z0, T w0,
      T x1, T y1, T z1, T w1);
    constexpr mat(
      col_type const& v0,
      col_type const& v1);

    // -- Conversions --

    template<
      typename X1, typename Y1, typename Z1, typename W1,
      typename X2, typename Y2, typename Z2, typename W2>
    constexpr mat(
      X1 x1, Y1 y1, Z1 z1, W1 w1,
      X2 x2, Y2 y2, Z2 z2, W2 w2);

    template<typename U, typename V>
    constexpr mat(
      vec<4, U, Q> const& v1,
      vec<4, V, Q> const& v2);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<2, 4, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<2, 4, T, Q>& operator=(mat<2, 4, U, Q> const& m);
    template<typename U>
    mat<2, 4, T, Q>& operator+=(U s);
    template<typename U>
    mat<2, 4, T, Q>& operator+=(mat<2, 4, U, Q> const& m);
    template<typename U>
    mat<2, 4, T, Q>& operator-=(U s);
    template<typename U>
    mat<2, 4, T, Q>& operator-=(mat<2, 4, U, Q> const& m);
    template<typename U>
    mat<2, 4, T, Q>& operator*=(U s);
    template<typename U>
    mat<2, 4, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<2, 4, T, Q>& operator++ ();
    mat<2, 4, T, Q>& operator-- ();
    mat<2, 4, T, Q> operator++(int);
    mat<2, 4, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator*(mat<2, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator*(T scalar, mat<2, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<2, 4, T, Q>::col_type operator*(mat<2, 4, T, Q> const& m, typename mat<2, 4, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<2, 4, T, Q>::row_type operator*(typename mat<2, 4, T, Q>::col_type const& v, mat<2, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<2, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator/(mat<2, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator/(T scalar, mat<2, 4, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2);
}//namespace glm



namespace glm
{
  // -- Constructors --















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<2, 4, T, Q>::mat(mat<2, 4, T, P> const& m)

    : value{ m[0], m[1] }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(T s)

    : value{ col_type(s, 0, 0, 0), col_type(0, s, 0, 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat
  (
    T x0, T y0, T z0, T w0,
    T x1, T y1, T z1, T w1
  )

    : value{ col_type(x0, y0, z0, w0), col_type(x1, y1, z1, w1) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(col_type const& v0, col_type const& v1)

    : value{ col_type(v0), col_type(v1) }

  {




  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X1, typename Y1, typename Z1, typename W1,
    typename X2, typename Y2, typename Z2, typename W2>
  inline constexpr mat<2, 4, T, Q>::mat
  (
    X1 x1, Y1 y1, Z1 z1, W1 w1,
    X2 x2, Y2 y2, Z2 z2, W2 w2
  )

    : value{
      col_type(x1, y1, z1, w1),
      col_type(x2, y2, z2, w2) }

  {




  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2>
  inline constexpr mat<2, 4, T, Q>::mat(vec<4, V1, Q> const& v1, vec<4, V2, Q> const& v2)

    : value{ col_type(v1), col_type(v2) }

  {




  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<2, 4, T, Q>::mat(mat<2, 4, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0) }

  {




  }

  template<typename T, qualifier Q>
  inline constexpr mat<2, 4, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0) }

  {




  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<2, 4, T, Q>::col_type& mat<2, 4, T, Q>::operator[](typename mat<2, 4, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<2, 4, T, Q>::col_type const& mat<2, 4, T, Q>::operator[](typename mat<2, 4, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator=(mat<2, 4, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator+=(mat<2, 4, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator-=(mat<2, 4, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q>& mat<2, 4, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> mat<2, 4, T, Q>::operator++(int)
  {
    mat<2, 4, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> mat<2, 4, T, Q>::operator--(int)
  {
    mat<2, 4, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m)
  {
    return mat<2, 4, T, Q>(
      -m[0],
      -m[1]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m, T scalar)
  {
    return mat<2, 4, T, Q>(
      m[0] + scalar,
      m[1] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator+(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    return mat<2, 4, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m, T scalar)
  {
    return mat<2, 4, T, Q>(
      m[0] - scalar,
      m[1] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator-(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    return mat<2, 4, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator*(mat<2, 4, T, Q> const& m, T scalar)
  {
    return mat<2, 4, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator*(T scalar, mat<2, 4, T, Q> const& m)
  {
    return mat<2, 4, T, Q>(
      m[0] * scalar,
      m[1] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 4, T, Q>::col_type operator*(mat<2, 4, T, Q> const& m, typename mat<2, 4, T, Q>::row_type const& v)
  {
    return typename mat<2, 4, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y,
      m[0][1] * v.x + m[1][1] * v.y,
      m[0][2] * v.x + m[1][2] * v.y,
      m[0][3] * v.x + m[1][3] * v.y);
  }

  template<typename T, qualifier Q>
  inline typename mat<2, 4, T, Q>::row_type operator*(typename mat<2, 4, T, Q>::col_type const& v, mat<2, 4, T, Q> const& m)
  {
    return typename mat<2, 4, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1] + v.z * m[0][2] + v.w * m[0][3],
      v.x * m[1][0] + v.y * m[1][1] + v.z * m[1][2] + v.w * m[1][3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    T SrcA00 = m1[0][0];
    T SrcA01 = m1[0][1];
    T SrcA02 = m1[0][2];
    T SrcA03 = m1[0][3];
    T SrcA10 = m1[1][0];
    T SrcA11 = m1[1][1];
    T SrcA12 = m1[1][2];
    T SrcA13 = m1[1][3];

    T SrcB00 = m2[0][0];
    T SrcB01 = m2[0][1];
    T SrcB10 = m2[1][0];
    T SrcB11 = m2[1][1];
    T SrcB20 = m2[2][0];
    T SrcB21 = m2[2][1];
    T SrcB30 = m2[3][0];
    T SrcB31 = m2[3][1];

    mat<4, 4, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01;
    Result[0][2] = SrcA02 * SrcB00 + SrcA12 * SrcB01;
    Result[0][3] = SrcA03 * SrcB00 + SrcA13 * SrcB01;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11;
    Result[1][2] = SrcA02 * SrcB10 + SrcA12 * SrcB11;
    Result[1][3] = SrcA03 * SrcB10 + SrcA13 * SrcB11;
    Result[2][0] = SrcA00 * SrcB20 + SrcA10 * SrcB21;
    Result[2][1] = SrcA01 * SrcB20 + SrcA11 * SrcB21;
    Result[2][2] = SrcA02 * SrcB20 + SrcA12 * SrcB21;
    Result[2][3] = SrcA03 * SrcB20 + SrcA13 * SrcB21;
    Result[3][0] = SrcA00 * SrcB30 + SrcA10 * SrcB31;
    Result[3][1] = SrcA01 * SrcB30 + SrcA11 * SrcB31;
    Result[3][2] = SrcA02 * SrcB30 + SrcA12 * SrcB31;
    Result[3][3] = SrcA03 * SrcB30 + SrcA13 * SrcB31;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<2, 2, T, Q> const& m2)
  {
    return mat<2, 4, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1],
      m1[0][3] * m2[0][0] + m1[1][3] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1],
      m1[0][3] * m2[1][0] + m1[1][3] * m2[1][1]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator*(mat<2, 4, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return mat<3, 4, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1],
      m1[0][3] * m2[0][0] + m1[1][3] * m2[0][1],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1],
      m1[0][3] * m2[1][0] + m1[1][3] * m2[1][1],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1],
      m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1],
      m1[0][3] * m2[2][0] + m1[1][3] * m2[2][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator/(mat<2, 4, T, Q> const& m, T scalar)
  {
    return mat<2, 4, T, Q>(
      m[0] / scalar,
      m[1] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator/(T scalar, mat<2, 4, T, Q> const& m)
  {
    return mat<2, 4, T, Q>(
      scalar / m[0],
      scalar / m[1]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<2, 4, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 4 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 4, double, defaultp>		dmat2x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double2x4_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 4 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, double, lowp>		lowp_dmat2x4;

  /// 2 columns of 4 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, double, mediump>	mediump_dmat2x4;

  /// 2 columns of 4 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, double, highp>	highp_dmat2x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x4.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 2 columns of 4 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<2, 4, float, defaultp>		mat2x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x4_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 2 columns of 4 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, float, lowp>		lowp_mat2x4;

  /// 2 columns of 4 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, float, mediump>	mediump_mat2x4;

  /// 2 columns of 4 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<2, 4, float, highp>		highp_mat2x4;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<3, 2, T, Q>
  {
    typedef vec<2, T, Q> col_type;
    typedef vec<3, T, Q> row_type;
    typedef mat<3, 2, T, Q> type;
    typedef mat<2, 3, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[3];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 3; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<3, 2, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0,
      T x1, T y1,
      T x2, T y2);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2);

    // -- Conversions --

    template<
      typename X1, typename Y1,
      typename X2, typename Y2,
      typename X3, typename Y3>
    constexpr mat(
      X1 x1, Y1 y1,
      X2 x2, Y2 y2,
      X3 x3, Y3 y3);

    template<typename V1, typename V2, typename V3>
    constexpr mat(
      vec<2, V1, Q> const& v1,
      vec<2, V2, Q> const& v2,
      vec<2, V3, Q> const& v3);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<3, 2, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<3, 2, T, Q>& operator=(mat<3, 2, U, Q> const& m);
    template<typename U>
    mat<3, 2, T, Q>& operator+=(U s);
    template<typename U>
    mat<3, 2, T, Q>& operator+=(mat<3, 2, U, Q> const& m);
    template<typename U>
    mat<3, 2, T, Q>& operator-=(U s);
    template<typename U>
    mat<3, 2, T, Q>& operator-=(mat<3, 2, U, Q> const& m);
    template<typename U>
    mat<3, 2, T, Q>& operator*=(U s);
    template<typename U>
    mat<3, 2, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<3, 2, T, Q>& operator++ ();
    mat<3, 2, T, Q>& operator-- ();
    mat<3, 2, T, Q> operator++(int);
    mat<3, 2, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator*(mat<3, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator*(T scalar, mat<3, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<3, 2, T, Q>::col_type operator*(mat<3, 2, T, Q> const& m, typename mat<3, 2, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<3, 2, T, Q>::row_type operator*(typename mat<3, 2, T, Q>::col_type const& v, mat<3, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator/(mat<3, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator/(T scalar, mat<3, 2, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2);

}//namespace glm



namespace glm
{
  // -- Constructors --
















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<3, 2, T, Q>::mat(mat<3, 2, T, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(T s)

    : value{ col_type(s, 0), col_type(0, s), col_type(0, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat
  (
    T x0, T y0,
    T x1, T y1,
    T x2, T y2
  )

    : value{ col_type(x0, y0), col_type(x1, y1), col_type(x2, y2) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(col_type const& v0, col_type const& v1, col_type const& v2)

    : value{ col_type(v0), col_type(v1), col_type(v2) }

  {





  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X0, typename Y0,
    typename X1, typename Y1,
    typename X2, typename Y2>
  inline constexpr mat<3, 2, T, Q>::mat
  (
    X0 x0, Y0 y0,
    X1 x1, Y1 y1,
    X2 x2, Y2 y2
  )

    : value{ col_type(x0, y0), col_type(x1, y1), col_type(x2, y2) }

  {





  }

  template<typename T, qualifier Q>
  template<typename V0, typename V1, typename V2>
  inline constexpr mat<3, 2, T, Q>::mat(vec<2, V0, Q> const& v0, vec<2, V1, Q> const& v1, vec<2, V2, Q> const& v2)

    : value{ col_type(v0), col_type(v1), col_type(v2) }

  {





  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<3, 2, T, Q>::mat(mat<3, 2, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 2, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<3, 2, T, Q>::col_type& mat<3, 2, T, Q>::operator[](typename mat<3, 2, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<3, 2, T, Q>::col_type const& mat<3, 2, T, Q>::operator[](typename mat<3, 2, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator=(mat<3, 2, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    this->value[2] = m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    this->value[2] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator+=(mat<3, 2, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    this->value[2] += m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    this->value[2] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator-=(mat<3, 2, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    this->value[2] -= m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    this->value[2] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    this->value[2] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    ++this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q>& mat<3, 2, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    --this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> mat<3, 2, T, Q>::operator++(int)
  {
    mat<3, 2, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> mat<3, 2, T, Q>::operator--(int)
  {
    mat<3, 2, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m)
  {
    return mat<3, 2, T, Q>(
      -m[0],
      -m[1],
      -m[2]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m, T scalar)
  {
    return mat<3, 2, T, Q>(
      m[0] + scalar,
      m[1] + scalar,
      m[2] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator+(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return mat<3, 2, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1],
      m1[2] + m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m, T scalar)
  {
    return mat<3, 2, T, Q>(
      m[0] - scalar,
      m[1] - scalar,
      m[2] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator-(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return mat<3, 2, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1],
      m1[2] - m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator*(mat<3, 2, T, Q> const& m, T scalar)
  {
    return mat<3, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator*(T scalar, mat<3, 2, T, Q> const& m)
  {
    return mat<3, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 2, T, Q>::col_type operator*(mat<3, 2, T, Q> const& m, typename mat<3, 2, T, Q>::row_type const& v)
  {
    return typename mat<3, 2, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z,
      m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 2, T, Q>::row_type operator*(typename mat<3, 2, T, Q>::col_type const& v, mat<3, 2, T, Q> const& m)
  {
    return typename mat<3, 2, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1],
      v.x * m[1][0] + v.y * m[1][1],
      v.x * m[2][0] + v.y * m[2][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    const T SrcA00 = m1[0][0];
    const T SrcA01 = m1[0][1];
    const T SrcA10 = m1[1][0];
    const T SrcA11 = m1[1][1];
    const T SrcA20 = m1[2][0];
    const T SrcA21 = m1[2][1];

    const T SrcB00 = m2[0][0];
    const T SrcB01 = m2[0][1];
    const T SrcB02 = m2[0][2];
    const T SrcB10 = m2[1][0];
    const T SrcB11 = m2[1][1];
    const T SrcB12 = m2[1][2];

    mat<2, 2, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01 + SrcA20 * SrcB02;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01 + SrcA21 * SrcB02;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11 + SrcA20 * SrcB12;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11 + SrcA21 * SrcB12;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return mat<3, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator*(mat<3, 2, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return mat<4, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1] + m1[2][0] * m2[3][2],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1] + m1[2][1] * m2[3][2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator/(mat<3, 2, T, Q> const& m, T scalar)
  {
    return mat<3, 2, T, Q>(
      m[0] / scalar,
      m[1] / scalar,
      m[2] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator/(T scalar, mat<3, 2, T, Q> const& m)
  {
    return mat<3, 2, T, Q>(
      scalar / m[0],
      scalar / m[1],
      scalar / m[2]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]) && (m1[2] == m2[2]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<3, 2, T, Q> const& m1, mat<3, 2, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]) || (m1[2] != m2[2]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 3 columns of 2 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 2, double, defaultp>		dmat3x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double3x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 2 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, double, lowp>		lowp_dmat3x2;

  /// 3 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, double, mediump>	mediump_dmat3x2;

  /// 3 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, double, highp>	highp_dmat3x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x2.hpp




namespace glm
{
  /// @addtogroup core
  /// @{

  /// 3 columns of 2 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 2, float, defaultp>			mat3x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 2 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, float, lowp>		lowp_mat3x2;

  /// 3 columns of 2 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, float, mediump>	mediump_mat3x2;

  /// 3 columns of 2 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 2, float, highp>		highp_mat3x2;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<3, 3, T, Q>
  {
    typedef vec<3, T, Q> col_type;
    typedef vec<3, T, Q> row_type;
    typedef mat<3, 3, T, Q> type;
    typedef mat<3, 3, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[3];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 3; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<3, 3, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0, T z0,
      T x1, T y1, T z1,
      T x2, T y2, T z2);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2);

    // -- Conversions --

    template<
      typename X1, typename Y1, typename Z1,
      typename X2, typename Y2, typename Z2,
      typename X3, typename Y3, typename Z3>
    constexpr mat(
      X1 x1, Y1 y1, Z1 z1,
      X2 x2, Y2 y2, Z2 z2,
      X3 x3, Y3 y3, Z3 z3);

    template<typename V1, typename V2, typename V3>
    constexpr mat(
      vec<3, V1, Q> const& v1,
      vec<3, V2, Q> const& v2,
      vec<3, V3, Q> const& v3);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<3, 3, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<3, 3, T, Q>& operator=(mat<3, 3, U, Q> const& m);
    template<typename U>
    mat<3, 3, T, Q>& operator+=(U s);
    template<typename U>
    mat<3, 3, T, Q>& operator+=(mat<3, 3, U, Q> const& m);
    template<typename U>
    mat<3, 3, T, Q>& operator-=(U s);
    template<typename U>
    mat<3, 3, T, Q>& operator-=(mat<3, 3, U, Q> const& m);
    template<typename U>
    mat<3, 3, T, Q>& operator*=(U s);
    template<typename U>
    mat<3, 3, T, Q>& operator*=(mat<3, 3, U, Q> const& m);
    template<typename U>
    mat<3, 3, T, Q>& operator/=(U s);
    template<typename U>
    mat<3, 3, T, Q>& operator/=(mat<3, 3, U, Q> const& m);

    // -- Increment and decrement operators --

    mat<3, 3, T, Q>& operator++();
    mat<3, 3, T, Q>& operator--();
    mat<3, 3, T, Q> operator++(int);
    mat<3, 3, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator+(T scalar, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator-(T scalar, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator*(mat<3, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator*(T scalar, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<3, 3, T, Q>::col_type operator*(mat<3, 3, T, Q> const& m, typename mat<3, 3, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<3, 3, T, Q>::row_type operator*(typename mat<3, 3, T, Q>::col_type const& v, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator/(mat<3, 3, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator/(T scalar, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<3, 3, T, Q>::col_type operator/(mat<3, 3, T, Q> const& m, typename mat<3, 3, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<3, 3, T, Q>::row_type operator/(typename mat<3, 3, T, Q>::col_type const& v, mat<3, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator/(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  constexpr bool operator==(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2);
}//namespace glm





namespace glm
{
  // -- Constructors --
















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<3, 3, T, Q>::mat(mat<3, 3, T, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(T s)

    : value{ col_type(s, 0, 0), col_type(0, s, 0), col_type(0, 0, s) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat
  (
    T x0, T y0, T z0,
    T x1, T y1, T z1,
    T x2, T y2, T z2
  )

    : value{ col_type(x0, y0, z0), col_type(x1, y1, z1), col_type(x2, y2, z2) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(col_type const& v0, col_type const& v1, col_type const& v2)

    : value{ col_type(v0), col_type(v1), col_type(v2) }

  {





  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X1, typename Y1, typename Z1,
    typename X2, typename Y2, typename Z2,
    typename X3, typename Y3, typename Z3>
  inline constexpr mat<3, 3, T, Q>::mat
  (
    X1 x1, Y1 y1, Z1 z1,
    X2 x2, Y2 y2, Z2 z2,
    X3 x3, Y3 y3, Z3 z3
  )

    : value{ col_type(x1, y1, z1), col_type(x2, y2, z2), col_type(x3, y3, z3) }

  {





  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2, typename V3>
  inline constexpr mat<3, 3, T, Q>::mat(vec<3, V1, Q> const& v1, vec<3, V2, Q> const& v2, vec<3, V3, Q> const& v3)

    : value{ col_type(v1), col_type(v2), col_type(v3) }

  {





  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<3, 3, T, Q>::mat(mat<3, 3, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(0, 0, 1) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0, 0, 1) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 1) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0, 0, 1) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 1) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 3, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<3, 3, T, Q>::col_type& mat<3, 3, T, Q>::operator[](typename mat<3, 3, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<3, 3, T, Q>::col_type const& mat<3, 3, T, Q>::operator[](typename mat<3, 3, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator=(mat<3, 3, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    this->value[2] = m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    this->value[2] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator+=(mat<3, 3, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    this->value[2] += m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    this->value[2] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator-=(mat<3, 3, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    this->value[2] -= m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    this->value[2] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator*=(mat<3, 3, U, Q> const& m)
  {
    return (*this = *this * m);
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    this->value[2] /= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator/=(mat<3, 3, U, Q> const& m)
  {
    return *this *= inverse(m);
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    ++this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q>& mat<3, 3, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    --this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> mat<3, 3, T, Q>::operator++(int)
  {
    mat<3, 3, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> mat<3, 3, T, Q>::operator--(int)
  {
    mat<3, 3, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m)
  {
    return mat<3, 3, T, Q>(
      -m[0],
      -m[1],
      -m[2]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m, T scalar)
  {
    return mat<3, 3, T, Q>(
      m[0] + scalar,
      m[1] + scalar,
      m[2] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator+(T scalar, mat<3, 3, T, Q> const& m)
  {
    return mat<3, 3, T, Q>(
      m[0] + scalar,
      m[1] + scalar,
      m[2] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator+(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return mat<3, 3, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1],
      m1[2] + m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m, T scalar)
  {
    return mat<3, 3, T, Q>(
      m[0] - scalar,
      m[1] - scalar,
      m[2] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator-(T scalar, mat<3, 3, T, Q> const& m)
  {
    return mat<3, 3, T, Q>(
      scalar - m[0],
      scalar - m[1],
      scalar - m[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator-(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return mat<3, 3, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1],
      m1[2] - m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator*(mat<3, 3, T, Q> const& m, T scalar)
  {
    return mat<3, 3, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator*(T scalar, mat<3, 3, T, Q> const& m)
  {
    return mat<3, 3, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 3, T, Q>::col_type operator*(mat<3, 3, T, Q> const& m, typename mat<3, 3, T, Q>::row_type const& v)
  {
    return typename mat<3, 3, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z,
      m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z,
      m[0][2] * v.x + m[1][2] * v.y + m[2][2] * v.z);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 3, T, Q>::row_type operator*(typename mat<3, 3, T, Q>::col_type const& v, mat<3, 3, T, Q> const& m)
  {
    return typename mat<3, 3, T, Q>::row_type(
      m[0][0] * v.x + m[0][1] * v.y + m[0][2] * v.z,
      m[1][0] * v.x + m[1][1] * v.y + m[1][2] * v.z,
      m[2][0] * v.x + m[2][1] * v.y + m[2][2] * v.z);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    T const SrcA00 = m1[0][0];
    T const SrcA01 = m1[0][1];
    T const SrcA02 = m1[0][2];
    T const SrcA10 = m1[1][0];
    T const SrcA11 = m1[1][1];
    T const SrcA12 = m1[1][2];
    T const SrcA20 = m1[2][0];
    T const SrcA21 = m1[2][1];
    T const SrcA22 = m1[2][2];

    T const SrcB00 = m2[0][0];
    T const SrcB01 = m2[0][1];
    T const SrcB02 = m2[0][2];
    T const SrcB10 = m2[1][0];
    T const SrcB11 = m2[1][1];
    T const SrcB12 = m2[1][2];
    T const SrcB20 = m2[2][0];
    T const SrcB21 = m2[2][1];
    T const SrcB22 = m2[2][2];

    mat<3, 3, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01 + SrcA20 * SrcB02;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01 + SrcA21 * SrcB02;
    Result[0][2] = SrcA02 * SrcB00 + SrcA12 * SrcB01 + SrcA22 * SrcB02;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11 + SrcA20 * SrcB12;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11 + SrcA21 * SrcB12;
    Result[1][2] = SrcA02 * SrcB10 + SrcA12 * SrcB11 + SrcA22 * SrcB12;
    Result[2][0] = SrcA00 * SrcB20 + SrcA10 * SrcB21 + SrcA20 * SrcB22;
    Result[2][1] = SrcA01 * SrcB20 + SrcA11 * SrcB21 + SrcA21 * SrcB22;
    Result[2][2] = SrcA02 * SrcB20 + SrcA12 * SrcB21 + SrcA22 * SrcB22;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return mat<2, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator*(mat<3, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return mat<4, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2],
      m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1] + m1[2][2] * m2[2][2],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1] + m1[2][0] * m2[3][2],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1] + m1[2][1] * m2[3][2],
      m1[0][2] * m2[3][0] + m1[1][2] * m2[3][1] + m1[2][2] * m2[3][2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator/(mat<3, 3, T, Q> const& m, T scalar)
  {
    return mat<3, 3, T, Q>(
      m[0] / scalar,
      m[1] / scalar,
      m[2] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator/(T scalar, mat<3, 3, T, Q> const& m)
  {
    return mat<3, 3, T, Q>(
      scalar / m[0],
      scalar / m[1],
      scalar / m[2]);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 3, T, Q>::col_type operator/(mat<3, 3, T, Q> const& m, typename mat<3, 3, T, Q>::row_type const& v)
  {
    return  inverse(m) * v;
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 3, T, Q>::row_type operator/(typename mat<3, 3, T, Q>::col_type const& v, mat<3, 3, T, Q> const& m)
  {
    return v * inverse(m);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator/(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    mat<3, 3, T, Q> m1_copy(m1);
    return m1_copy /= m2;
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline constexpr bool operator==(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]) && (m1[2] == m2[2]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<3, 3, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]) || (m1[2] != m2[2]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 3 columns of 3 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 3, double, defaultp>		dmat3x3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 3, double, defaultp>		dmat3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double3x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, lowp>		lowp_dmat3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, mediump>	mediump_dmat3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, highp>	highp_dmat3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, lowp>		lowp_dmat3x3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, mediump>	mediump_dmat3x3;

  /// 3 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, double, highp>	highp_dmat3x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x3.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 3 columns of 3 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 3, float, defaultp>			mat3x3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 3, float, defaultp>			mat3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, lowp>		lowp_mat3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, mediump>	mediump_mat3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, highp>		highp_mat3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, lowp>		lowp_mat3x3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, mediump>	mediump_mat3x3;

  /// 3 columns of 3 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 3, float, highp>		highp_mat3x3;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<3, 4, T, Q>
  {
    typedef vec<4, T, Q> col_type;
    typedef vec<3, T, Q> row_type;
    typedef mat<3, 4, T, Q> type;
    typedef mat<4, 3, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[3];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 3; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<3, 4, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0, T z0, T w0,
      T x1, T y1, T z1, T w1,
      T x2, T y2, T z2, T w2);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2);

    // -- Conversions --

    template<
      typename X1, typename Y1, typename Z1, typename W1,
      typename X2, typename Y2, typename Z2, typename W2,
      typename X3, typename Y3, typename Z3, typename W3>
    constexpr mat(
      X1 x1, Y1 y1, Z1 z1, W1 w1,
      X2 x2, Y2 y2, Z2 z2, W2 w2,
      X3 x3, Y3 y3, Z3 z3, W3 w3);

    template<typename V1, typename V2, typename V3>
    constexpr mat(
      vec<4, V1, Q> const& v1,
      vec<4, V2, Q> const& v2,
      vec<4, V3, Q> const& v3);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<3, 4, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<3, 4, T, Q>& operator=(mat<3, 4, U, Q> const& m);
    template<typename U>
    mat<3, 4, T, Q>& operator+=(U s);
    template<typename U>
    mat<3, 4, T, Q>& operator+=(mat<3, 4, U, Q> const& m);
    template<typename U>
    mat<3, 4, T, Q>& operator-=(U s);
    template<typename U>
    mat<3, 4, T, Q>& operator-=(mat<3, 4, U, Q> const& m);
    template<typename U>
    mat<3, 4, T, Q>& operator*=(U s);
    template<typename U>
    mat<3, 4, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<3, 4, T, Q>& operator++();
    mat<3, 4, T, Q>& operator--();
    mat<3, 4, T, Q> operator++(int);
    mat<3, 4, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator*(mat<3, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator*(T scalar, mat<3, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<3, 4, T, Q>::col_type operator*(mat<3, 4, T, Q> const& m, typename mat<3, 4, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<3, 4, T, Q>::row_type operator*(typename mat<3, 4, T, Q>::col_type const& v, mat<3, 4, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<2, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<2, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<3, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator/(mat<3, 4, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<3, 4, T, Q> operator/(T scalar, mat<3, 4, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2);
}//namespace glm



namespace glm
{
  // -- Constructors --
















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<3, 4, T, Q>::mat(mat<3, 4, T, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(T s)

    : value{ col_type(s, 0, 0, 0), col_type(0, s, 0, 0), col_type(0, 0, s, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat
  (
    T x0, T y0, T z0, T w0,
    T x1, T y1, T z1, T w1,
    T x2, T y2, T z2, T w2
  )

    : value{
      col_type(x0, y0, z0, w0),
      col_type(x1, y1, z1, w1),
      col_type(x2, y2, z2, w2) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(col_type const& v0, col_type const& v1, col_type const& v2)

    : value{ col_type(v0), col_type(v1), col_type(v2) }

  {





  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X0, typename Y0, typename Z0, typename W0,
    typename X1, typename Y1, typename Z1, typename W1,
    typename X2, typename Y2, typename Z2, typename W2>
  inline constexpr mat<3, 4, T, Q>::mat
  (
    X0 x0, Y0 y0, Z0 z0, W0 w0,
    X1 x1, Y1 y1, Z1 z1, W1 w1,
    X2 x2, Y2 y2, Z2 z2, W2 w2
  )

    : value{
      col_type(x0, y0, z0, w0),
      col_type(x1, y1, z1, w1),
      col_type(x2, y2, z2, w2) }

  {





  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2, typename V3>
  inline constexpr mat<3, 4, T, Q>::mat(vec<4, V1, Q> const& v0, vec<4, V2, Q> const& v1, vec<4, V3, Q> const& v2)

    : value{ col_type(v0), col_type(v1), col_type(v2) }

  {





  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<3, 4, T, Q>::mat(mat<3, 4, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0), col_type(0, 0, 1, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(0, 0, 1, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0), col_type(m[2], 1, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0, 0, 1, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0], 0, 0), col_type(m[1], 0, 0), col_type(m[2], 1, 0) }

  {





  }

  template<typename T, qualifier Q>
  inline constexpr mat<3, 4, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 0) }

  {





  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<3, 4, T, Q>::col_type& mat<3, 4, T, Q>::operator[](typename mat<3, 4, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<3, 4, T, Q>::col_type const& mat<3, 4, T, Q>::operator[](typename mat<3, 4, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator=(mat<3, 4, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    this->value[2] = m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    this->value[2] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator+=(mat<3, 4, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    this->value[2] += m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    this->value[2] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator-=(mat<3, 4, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    this->value[2] -= m[2];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    this->value[2] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    this->value[2] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    ++this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q>& mat<3, 4, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    --this->value[2];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> mat<3, 4, T, Q>::operator++(int)
  {
    mat<3, 4, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> mat<3, 4, T, Q>::operator--(int)
  {
    mat<3, 4, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m)
  {
    return mat<3, 4, T, Q>(
      -m[0],
      -m[1],
      -m[2]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m, T scalar)
  {
    return mat<3, 4, T, Q>(
      m[0] + scalar,
      m[1] + scalar,
      m[2] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator+(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    return mat<3, 4, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1],
      m1[2] + m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m, T scalar)
  {
    return mat<3, 4, T, Q>(
      m[0] - scalar,
      m[1] - scalar,
      m[2] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator-(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    return mat<3, 4, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1],
      m1[2] - m2[2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator*(mat<3, 4, T, Q> const& m, T scalar)
  {
    return mat<3, 4, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator*(T scalar, mat<3, 4, T, Q> const& m)
  {
    return mat<3, 4, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 4, T, Q>::col_type operator*
    (
      mat<3, 4, T, Q> const& m,
      typename mat<3, 4, T, Q>::row_type const& v
      )
  {
    return typename mat<3, 4, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z,
      m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z,
      m[0][2] * v.x + m[1][2] * v.y + m[2][2] * v.z,
      m[0][3] * v.x + m[1][3] * v.y + m[2][3] * v.z);
  }

  template<typename T, qualifier Q>
  inline typename mat<3, 4, T, Q>::row_type operator*
    (
      typename mat<3, 4, T, Q>::col_type const& v,
      mat<3, 4, T, Q> const& m
      )
  {
    return typename mat<3, 4, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1] + v.z * m[0][2] + v.w * m[0][3],
      v.x * m[1][0] + v.y * m[1][1] + v.z * m[1][2] + v.w * m[1][3],
      v.x * m[2][0] + v.y * m[2][1] + v.z * m[2][2] + v.w * m[2][3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    const T SrcA00 = m1[0][0];
    const T SrcA01 = m1[0][1];
    const T SrcA02 = m1[0][2];
    const T SrcA03 = m1[0][3];
    const T SrcA10 = m1[1][0];
    const T SrcA11 = m1[1][1];
    const T SrcA12 = m1[1][2];
    const T SrcA13 = m1[1][3];
    const T SrcA20 = m1[2][0];
    const T SrcA21 = m1[2][1];
    const T SrcA22 = m1[2][2];
    const T SrcA23 = m1[2][3];

    const T SrcB00 = m2[0][0];
    const T SrcB01 = m2[0][1];
    const T SrcB02 = m2[0][2];
    const T SrcB10 = m2[1][0];
    const T SrcB11 = m2[1][1];
    const T SrcB12 = m2[1][2];
    const T SrcB20 = m2[2][0];
    const T SrcB21 = m2[2][1];
    const T SrcB22 = m2[2][2];
    const T SrcB30 = m2[3][0];
    const T SrcB31 = m2[3][1];
    const T SrcB32 = m2[3][2];

    mat<4, 4, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01 + SrcA20 * SrcB02;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01 + SrcA21 * SrcB02;
    Result[0][2] = SrcA02 * SrcB00 + SrcA12 * SrcB01 + SrcA22 * SrcB02;
    Result[0][3] = SrcA03 * SrcB00 + SrcA13 * SrcB01 + SrcA23 * SrcB02;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11 + SrcA20 * SrcB12;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11 + SrcA21 * SrcB12;
    Result[1][2] = SrcA02 * SrcB10 + SrcA12 * SrcB11 + SrcA22 * SrcB12;
    Result[1][3] = SrcA03 * SrcB10 + SrcA13 * SrcB11 + SrcA23 * SrcB12;
    Result[2][0] = SrcA00 * SrcB20 + SrcA10 * SrcB21 + SrcA20 * SrcB22;
    Result[2][1] = SrcA01 * SrcB20 + SrcA11 * SrcB21 + SrcA21 * SrcB22;
    Result[2][2] = SrcA02 * SrcB20 + SrcA12 * SrcB21 + SrcA22 * SrcB22;
    Result[2][3] = SrcA03 * SrcB20 + SrcA13 * SrcB21 + SrcA23 * SrcB22;
    Result[3][0] = SrcA00 * SrcB30 + SrcA10 * SrcB31 + SrcA20 * SrcB32;
    Result[3][1] = SrcA01 * SrcB30 + SrcA11 * SrcB31 + SrcA21 * SrcB32;
    Result[3][2] = SrcA02 * SrcB30 + SrcA12 * SrcB31 + SrcA22 * SrcB32;
    Result[3][3] = SrcA03 * SrcB30 + SrcA13 * SrcB31 + SrcA23 * SrcB32;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<2, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<2, 3, T, Q> const& m2)
  {
    return mat<2, 4, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2],
      m1[0][3] * m2[0][0] + m1[1][3] * m2[0][1] + m1[2][3] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2],
      m1[0][3] * m2[1][0] + m1[1][3] * m2[1][1] + m1[2][3] * m2[1][2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator*(mat<3, 4, T, Q> const& m1, mat<3, 3, T, Q> const& m2)
  {
    return mat<3, 4, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2],
      m1[0][3] * m2[0][0] + m1[1][3] * m2[0][1] + m1[2][3] * m2[0][2],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2],
      m1[0][3] * m2[1][0] + m1[1][3] * m2[1][1] + m1[2][3] * m2[1][2],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2],
      m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1] + m1[2][2] * m2[2][2],
      m1[0][3] * m2[2][0] + m1[1][3] * m2[2][1] + m1[2][3] * m2[2][2]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator/(mat<3, 4, T, Q> const& m, T scalar)
  {
    return mat<3, 4, T, Q>(
      m[0] / scalar,
      m[1] / scalar,
      m[2] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<3, 4, T, Q> operator/(T scalar, mat<3, 4, T, Q> const& m)
  {
    return mat<3, 4, T, Q>(
      scalar / m[0],
      scalar / m[1],
      scalar / m[2]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]) && (m1[2] == m2[2]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<3, 4, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]) || (m1[2] != m2[2]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 3 columns of 4 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 4, double, defaultp>		dmat3x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double3x4_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 4 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, double, lowp>		lowp_dmat3x4;

  /// 3 columns of 4 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, double, mediump>	mediump_dmat3x4;

  /// 3 columns of 4 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, double, highp>	highp_dmat3x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x4.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 3 columns of 4 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<3, 4, float, defaultp>			mat3x4;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float3x4_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 3 columns of 4 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, float, lowp>		lowp_mat3x4;

  /// 3 columns of 4 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, float, mediump>	mediump_mat3x4;

  /// 3 columns of 4 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<3, 4, float, highp>		highp_mat3x4;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<4, 2, T, Q>
  {
    typedef vec<2, T, Q> col_type;
    typedef vec<4, T, Q> row_type;
    typedef mat<4, 2, T, Q> type;
    typedef mat<2, 4, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[4];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 4; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<4, 2, T, P> const& m);

    explicit constexpr mat(T scalar);
    constexpr mat(
      T x0, T y0,
      T x1, T y1,
      T x2, T y2,
      T x3, T y3);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2,
      col_type const& v3);

    // -- Conversions --

    template<
      typename X0, typename Y0,
      typename X1, typename Y1,
      typename X2, typename Y2,
      typename X3, typename Y3>
    constexpr mat(
      X0 x0, Y0 y0,
      X1 x1, Y1 y1,
      X2 x2, Y2 y2,
      X3 x3, Y3 y3);

    template<typename V1, typename V2, typename V3, typename V4>
    constexpr mat(
      vec<2, V1, Q> const& v1,
      vec<2, V2, Q> const& v2,
      vec<2, V3, Q> const& v3,
      vec<2, V4, Q> const& v4);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<4, 2, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 3, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<4, 2, T, Q>& operator=(mat<4, 2, U, Q> const& m);
    template<typename U>
    mat<4, 2, T, Q>& operator+=(U s);
    template<typename U>
    mat<4, 2, T, Q>& operator+=(mat<4, 2, U, Q> const& m);
    template<typename U>
    mat<4, 2, T, Q>& operator-=(U s);
    template<typename U>
    mat<4, 2, T, Q>& operator-=(mat<4, 2, U, Q> const& m);
    template<typename U>
    mat<4, 2, T, Q>& operator*=(U s);
    template<typename U>
    mat<4, 2, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<4, 2, T, Q>& operator++ ();
    mat<4, 2, T, Q>& operator-- ();
    mat<4, 2, T, Q> operator++(int);
    mat<4, 2, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator*(mat<4, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator*(T scalar, mat<4, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<4, 2, T, Q>::col_type operator*(mat<4, 2, T, Q> const& m, typename mat<4, 2, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<4, 2, T, Q>::row_type operator*(typename mat<4, 2, T, Q>::col_type const& v, mat<4, 2, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator/(mat<4, 2, T, Q> const& m, T scalar);

  template<typename T, qualifier Q>
  mat<4, 2, T, Q> operator/(T scalar, mat<4, 2, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2);
}//namespace glm



namespace glm
{
  // -- Constructors --

















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<4, 2, T, Q>::mat(mat<4, 2, T, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(T s)

    : value{ col_type(s, 0), col_type(0, s), col_type(0, 0), col_type(0, 0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat
  (
    T x0, T y0,
    T x1, T y1,
    T x2, T y2,
    T x3, T y3
  )

    : value{ col_type(x0, y0), col_type(x1, y1), col_type(x2, y2), col_type(x3, y3) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(col_type const& v0, col_type const& v1, col_type const& v2, col_type const& v3)

    : value{ col_type(v0), col_type(v1), col_type(v2), col_type(v3) }

  {






  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X0, typename Y0,
    typename X1, typename Y1,
    typename X2, typename Y2,
    typename X3, typename Y3>
  inline constexpr mat<4, 2, T, Q>::mat
  (
    X0 x0, Y0 y0,
    X1 x1, Y1 y1,
    X2 x2, Y2 y2,
    X3 x3, Y3 y3
  )

    : value{ col_type(x0, y0), col_type(x1, y1), col_type(x2, y2), col_type(x3, y3) }

  {






  }

  template<typename T, qualifier Q>
  template<typename V0, typename V1, typename V2, typename V3>
  inline constexpr mat<4, 2, T, Q>::mat(vec<2, V0, Q> const& v0, vec<2, V1, Q> const& v1, vec<2, V2, Q> const& v2, vec<2, V3, Q> const& v3)

    : value{ col_type(v0), col_type(v1), col_type(v2), col_type(v3) }

  {






  }

  // -- Conversion --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<4, 2, T, Q>::mat(mat<4, 2, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<4, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 2, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(0) }

  {






  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<4, 2, T, Q>::col_type& mat<4, 2, T, Q>::operator[](typename mat<4, 2, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<4, 2, T, Q>::col_type const& mat<4, 2, T, Q>::operator[](typename mat<4, 2, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator=(mat<4, 2, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    this->value[2] = m[2];
    this->value[3] = m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    this->value[2] += s;
    this->value[3] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator+=(mat<4, 2, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    this->value[2] += m[2];
    this->value[3] += m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    this->value[2] -= s;
    this->value[3] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator-=(mat<4, 2, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    this->value[2] -= m[2];
    this->value[3] -= m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    this->value[2] *= s;
    this->value[3] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    this->value[2] /= s;
    this->value[3] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    ++this->value[2];
    ++this->value[3];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q>& mat<4, 2, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    --this->value[2];
    --this->value[3];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> mat<4, 2, T, Q>::operator++(int)
  {
    mat<4, 2, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> mat<4, 2, T, Q>::operator--(int)
  {
    mat<4, 2, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m)
  {
    return mat<4, 2, T, Q>(
      -m[0],
      -m[1],
      -m[2],
      -m[3]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m, T scalar)
  {
    return mat<4, 2, T, Q>(
      m[0] + scalar,
      m[1] + scalar,
      m[2] + scalar,
      m[3] + scalar);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator+(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return mat<4, 2, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1],
      m1[2] + m2[2],
      m1[3] + m2[3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m, T scalar)
  {
    return mat<4, 2, T, Q>(
      m[0] - scalar,
      m[1] - scalar,
      m[2] - scalar,
      m[3] - scalar);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator-(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return mat<4, 2, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1],
      m1[2] - m2[2],
      m1[3] - m2[3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator*(mat<4, 2, T, Q> const& m, T scalar)
  {
    return mat<4, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar,
      m[3] * scalar);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator*(T scalar, mat<4, 2, T, Q> const& m)
  {
    return mat<4, 2, T, Q>(
      m[0] * scalar,
      m[1] * scalar,
      m[2] * scalar,
      m[3] * scalar);
  }

  template<typename T, qualifier Q>
  inline typename mat<4, 2, T, Q>::col_type operator*(mat<4, 2, T, Q> const& m, typename mat<4, 2, T, Q>::row_type const& v)
  {
    return typename mat<4, 2, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z + m[3][0] * v.w,
      m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z + m[3][1] * v.w);
  }

  template<typename T, qualifier Q>
  inline typename mat<4, 2, T, Q>::row_type operator*(typename mat<4, 2, T, Q>::col_type const& v, mat<4, 2, T, Q> const& m)
  {
    return typename mat<4, 2, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1],
      v.x * m[1][0] + v.y * m[1][1],
      v.x * m[2][0] + v.y * m[2][1],
      v.x * m[3][0] + v.y * m[3][1]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    T const SrcA00 = m1[0][0];
    T const SrcA01 = m1[0][1];
    T const SrcA10 = m1[1][0];
    T const SrcA11 = m1[1][1];
    T const SrcA20 = m1[2][0];
    T const SrcA21 = m1[2][1];
    T const SrcA30 = m1[3][0];
    T const SrcA31 = m1[3][1];

    T const SrcB00 = m2[0][0];
    T const SrcB01 = m2[0][1];
    T const SrcB02 = m2[0][2];
    T const SrcB03 = m2[0][3];
    T const SrcB10 = m2[1][0];
    T const SrcB11 = m2[1][1];
    T const SrcB12 = m2[1][2];
    T const SrcB13 = m2[1][3];

    mat<2, 2, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01 + SrcA20 * SrcB02 + SrcA30 * SrcB03;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01 + SrcA21 * SrcB02 + SrcA31 * SrcB03;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11 + SrcA20 * SrcB12 + SrcA30 * SrcB13;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11 + SrcA21 * SrcB12 + SrcA31 * SrcB13;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<3, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    return mat<3, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2] + m1[3][0] * m2[0][3],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2] + m1[3][1] * m2[0][3],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2] + m1[3][0] * m2[1][3],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2] + m1[3][1] * m2[1][3],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2] + m1[3][0] * m2[2][3],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2] + m1[3][1] * m2[2][3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator*(mat<4, 2, T, Q> const& m1, mat<4, 4, T, Q> const& m2)
  {
    return mat<4, 2, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2] + m1[3][0] * m2[0][3],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2] + m1[3][1] * m2[0][3],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2] + m1[3][0] * m2[1][3],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2] + m1[3][1] * m2[1][3],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2] + m1[3][0] * m2[2][3],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2] + m1[3][1] * m2[2][3],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1] + m1[2][0] * m2[3][2] + m1[3][0] * m2[3][3],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1] + m1[2][1] * m2[3][2] + m1[3][1] * m2[3][3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator/(mat<4, 2, T, Q> const& m, T scalar)
  {
    return mat<4, 2, T, Q>(
      m[0] / scalar,
      m[1] / scalar,
      m[2] / scalar,
      m[3] / scalar);
  }

  template<typename T, qualifier Q>
  inline mat<4, 2, T, Q> operator/(T scalar, mat<4, 2, T, Q> const& m)
  {
    return mat<4, 2, T, Q>(
      scalar / m[0],
      scalar / m[1],
      scalar / m[2],
      scalar / m[3]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]) && (m1[2] == m2[2]) && (m1[3] == m2[3]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<4, 2, T, Q> const& m1, mat<4, 2, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]) || (m1[2] != m2[2]) || (m1[3] != m2[3]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 4 columns of 2 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<4, 2, double, defaultp>		dmat4x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double4x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 4 columns of 2 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, double, lowp>		lowp_dmat4x2;

  /// 4 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, double, mediump>	mediump_dmat4x2;

  /// 4 columns of 2 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, double, highp>	highp_dmat4x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float4x2.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 4 columns of 2 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<4, 2, float, defaultp>			mat4x2;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float2x2_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 4 columns of 2 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, float, lowp>		lowp_mat4x2;

  /// 4 columns of 2 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, float, mediump>	mediump_mat4x2;

  /// 4 columns of 2 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 2, float, highp>		highp_mat4x2;

  /// @}
}//namespace glm

namespace glm
{
  template<typename T, qualifier Q>
  struct mat<4, 3, T, Q>
  {
    typedef vec<3, T, Q> col_type;
    typedef vec<4, T, Q> row_type;
    typedef mat<4, 3, T, Q> type;
    typedef mat<3, 4, T, Q> transpose_type;
    typedef T value_type;

  private:
    col_type value[4];

  public:
    // -- Accesses --

    typedef length_t length_type;
    static constexpr length_type length() { return 4; }

    col_type& operator[](length_type i);
    constexpr col_type const& operator[](length_type i) const;

    // -- Constructors --

    constexpr mat() = default;
    template<qualifier P>
    constexpr mat(mat<4, 3, T, P> const& m);

    explicit constexpr mat(T const& x);
    constexpr mat(
      T const& x0, T const& y0, T const& z0,
      T const& x1, T const& y1, T const& z1,
      T const& x2, T const& y2, T const& z2,
      T const& x3, T const& y3, T const& z3);
    constexpr mat(
      col_type const& v0,
      col_type const& v1,
      col_type const& v2,
      col_type const& v3);

    // -- Conversions --

    template<
      typename X1, typename Y1, typename Z1,
      typename X2, typename Y2, typename Z2,
      typename X3, typename Y3, typename Z3,
      typename X4, typename Y4, typename Z4>
    constexpr mat(
      X1 const& x1, Y1 const& y1, Z1 const& z1,
      X2 const& x2, Y2 const& y2, Z2 const& z2,
      X3 const& x3, Y3 const& y3, Z3 const& z3,
      X4 const& x4, Y4 const& y4, Z4 const& z4);

    template<typename V1, typename V2, typename V3, typename V4>
    constexpr mat(
      vec<3, V1, Q> const& v1,
      vec<3, V2, Q> const& v2,
      vec<3, V3, Q> const& v3,
      vec<3, V4, Q> const& v4);

    // -- Matrix conversions --

    template<typename U, qualifier P>
    constexpr mat(mat<4, 3, U, P> const& m);

    constexpr mat(mat<2, 2, T, Q> const& x);
    constexpr mat(mat<3, 3, T, Q> const& x);
    constexpr mat(mat<4, 4, T, Q> const& x);
    constexpr mat(mat<2, 3, T, Q> const& x);
    constexpr mat(mat<3, 2, T, Q> const& x);
    constexpr mat(mat<2, 4, T, Q> const& x);
    constexpr mat(mat<4, 2, T, Q> const& x);
    constexpr mat(mat<3, 4, T, Q> const& x);

    // -- Unary arithmetic operators --

    template<typename U>
    mat<4, 3, T, Q>& operator=(mat<4, 3, U, Q> const& m);
    template<typename U>
    mat<4, 3, T, Q>& operator+=(U s);
    template<typename U>
    mat<4, 3, T, Q>& operator+=(mat<4, 3, U, Q> const& m);
    template<typename U>
    mat<4, 3, T, Q>& operator-=(U s);
    template<typename U>
    mat<4, 3, T, Q>& operator-=(mat<4, 3, U, Q> const& m);
    template<typename U>
    mat<4, 3, T, Q>& operator*=(U s);
    template<typename U>
    mat<4, 3, T, Q>& operator/=(U s);

    // -- Increment and decrement operators --

    mat<4, 3, T, Q>& operator++();
    mat<4, 3, T, Q>& operator--();
    mat<4, 3, T, Q> operator++(int);
    mat<4, 3, T, Q> operator--(int);
  };

  // -- Unary operators --

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m);

  // -- Binary operators --

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator*(mat<4, 3, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator*(T const& s, mat<4, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  typename mat<4, 3, T, Q>::col_type operator*(mat<4, 3, T, Q> const& m, typename mat<4, 3, T, Q>::row_type const& v);

  template<typename T, qualifier Q>
  typename mat<4, 3, T, Q>::row_type operator*(typename mat<4, 3, T, Q>::col_type const& v, mat<4, 3, T, Q> const& m);

  template<typename T, qualifier Q>
  mat<2, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<2, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<3, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<3, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<4, 4, T, Q> const& m2);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator/(mat<4, 3, T, Q> const& m, T const& s);

  template<typename T, qualifier Q>
  mat<4, 3, T, Q> operator/(T const& s, mat<4, 3, T, Q> const& m);

  // -- Boolean operators --

  template<typename T, qualifier Q>
  bool operator==(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2);

  template<typename T, qualifier Q>
  bool operator!=(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2);
}//namespace glm



namespace glm
{
  // -- Constructors --

















  template<typename T, qualifier Q>
  template<qualifier P>
  inline constexpr mat<4, 3, T, Q>::mat(mat<4, 3, T, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(T const& s)

    : value{ col_type(s, 0, 0), col_type(0, s, 0), col_type(0, 0, s), col_type(0, 0, 0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat
  (
    T const& x0, T const& y0, T const& z0,
    T const& x1, T const& y1, T const& z1,
    T const& x2, T const& y2, T const& z2,
    T const& x3, T const& y3, T const& z3
  )

    : value{ col_type(x0, y0, z0), col_type(x1, y1, z1), col_type(x2, y2, z2), col_type(x3, y3, z3) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(col_type const& v0, col_type const& v1, col_type const& v2, col_type const& v3)

    : value{ col_type(v0), col_type(v1), col_type(v2), col_type(v3) }

  {






  }

  // -- Conversion constructors --

  template<typename T, qualifier Q>
  template<
    typename X0, typename Y0, typename Z0,
    typename X1, typename Y1, typename Z1,
    typename X2, typename Y2, typename Z2,
    typename X3, typename Y3, typename Z3>
  inline constexpr mat<4, 3, T, Q>::mat
  (
    X0 const& x0, Y0 const& y0, Z0 const& z0,
    X1 const& x1, Y1 const& y1, Z1 const& z1,
    X2 const& x2, Y2 const& y2, Z2 const& z2,
    X3 const& x3, Y3 const& y3, Z3 const& z3
  )

    : value{ col_type(x0, y0, z0), col_type(x1, y1, z1), col_type(x2, y2, z2), col_type(x3, y3, z3) }

  {






  }

  template<typename T, qualifier Q>
  template<typename V1, typename V2, typename V3, typename V4>
  inline constexpr mat<4, 3, T, Q>::mat(vec<3, V1, Q> const& v1, vec<3, V2, Q> const& v2, vec<3, V3, Q> const& v3, vec<3, V4, Q> const& v4)

    : value{ col_type(v1), col_type(v2), col_type(v3), col_type(v4) }

  {






  }

  // -- Matrix conversions --

  template<typename T, qualifier Q>
  template<typename U, qualifier P>
  inline constexpr mat<4, 3, T, Q>::mat(mat<4, 3, U, P> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<2, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(0, 0, 1), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<3, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<4, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(m[3]) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<2, 3, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0, 0, 1), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<3, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 1), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<2, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(0, 0, 1), col_type(0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<4, 2, T, Q> const& m)

    : value{ col_type(m[0], 0), col_type(m[1], 0), col_type(m[2], 1), col_type(m[3], 0) }

  {






  }

  template<typename T, qualifier Q>
  inline constexpr mat<4, 3, T, Q>::mat(mat<3, 4, T, Q> const& m)

    : value{ col_type(m[0]), col_type(m[1]), col_type(m[2]), col_type(0) }

  {






  }

  // -- Accesses --

  template<typename T, qualifier Q>
  inline typename mat<4, 3, T, Q>::col_type& mat<4, 3, T, Q>::operator[](typename mat<4, 3, T, Q>::length_type i)
  {
    assert(i < this->length());
    return this->value[i];
  }

  template<typename T, qualifier Q>
  inline constexpr typename mat<4, 3, T, Q>::col_type const& mat<4, 3, T, Q>::operator[](typename mat<4, 3, T, Q>::length_type i) const
  {
    assert(i < this->length());
    return this->value[i];
  }

  // -- Unary updatable operators --

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator=(mat<4, 3, U, Q> const& m)
  {
    this->value[0] = m[0];
    this->value[1] = m[1];
    this->value[2] = m[2];
    this->value[3] = m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator+=(U s)
  {
    this->value[0] += s;
    this->value[1] += s;
    this->value[2] += s;
    this->value[3] += s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator+=(mat<4, 3, U, Q> const& m)
  {
    this->value[0] += m[0];
    this->value[1] += m[1];
    this->value[2] += m[2];
    this->value[3] += m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator-=(U s)
  {
    this->value[0] -= s;
    this->value[1] -= s;
    this->value[2] -= s;
    this->value[3] -= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator-=(mat<4, 3, U, Q> const& m)
  {
    this->value[0] -= m[0];
    this->value[1] -= m[1];
    this->value[2] -= m[2];
    this->value[3] -= m[3];
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator*=(U s)
  {
    this->value[0] *= s;
    this->value[1] *= s;
    this->value[2] *= s;
    this->value[3] *= s;
    return *this;
  }

  template<typename T, qualifier Q>
  template<typename U>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator/=(U s)
  {
    this->value[0] /= s;
    this->value[1] /= s;
    this->value[2] /= s;
    this->value[3] /= s;
    return *this;
  }

  // -- Increment and decrement operators --

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator++()
  {
    ++this->value[0];
    ++this->value[1];
    ++this->value[2];
    ++this->value[3];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q>& mat<4, 3, T, Q>::operator--()
  {
    --this->value[0];
    --this->value[1];
    --this->value[2];
    --this->value[3];
    return *this;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> mat<4, 3, T, Q>::operator++(int)
  {
    mat<4, 3, T, Q> Result(*this);
    ++*this;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> mat<4, 3, T, Q>::operator--(int)
  {
    mat<4, 3, T, Q> Result(*this);
    --*this;
    return Result;
  }

  // -- Unary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m)
  {
    return m;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m)
  {
    return mat<4, 3, T, Q>(
      -m[0],
      -m[1],
      -m[2],
      -m[3]);
  }

  // -- Binary arithmetic operators --

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m, T const& s)
  {
    return mat<4, 3, T, Q>(
      m[0] + s,
      m[1] + s,
      m[2] + s,
      m[3] + s);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator+(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return mat<4, 3, T, Q>(
      m1[0] + m2[0],
      m1[1] + m2[1],
      m1[2] + m2[2],
      m1[3] + m2[3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m, T const& s)
  {
    return mat<4, 3, T, Q>(
      m[0] - s,
      m[1] - s,
      m[2] - s,
      m[3] - s);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator-(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return mat<4, 3, T, Q>(
      m1[0] - m2[0],
      m1[1] - m2[1],
      m1[2] - m2[2],
      m1[3] - m2[3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator*(mat<4, 3, T, Q> const& m, T const& s)
  {
    return mat<4, 3, T, Q>(
      m[0] * s,
      m[1] * s,
      m[2] * s,
      m[3] * s);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator*(T const& s, mat<4, 3, T, Q> const& m)
  {
    return mat<4, 3, T, Q>(
      m[0] * s,
      m[1] * s,
      m[2] * s,
      m[3] * s);
  }

  template<typename T, qualifier Q>
  inline typename mat<4, 3, T, Q>::col_type operator*
    (
      mat<4, 3, T, Q> const& m,
      typename mat<4, 3, T, Q>::row_type const& v)
  {
    return typename mat<4, 3, T, Q>::col_type(
      m[0][0] * v.x + m[1][0] * v.y + m[2][0] * v.z + m[3][0] * v.w,
      m[0][1] * v.x + m[1][1] * v.y + m[2][1] * v.z + m[3][1] * v.w,
      m[0][2] * v.x + m[1][2] * v.y + m[2][2] * v.z + m[3][2] * v.w);
  }

  template<typename T, qualifier Q>
  inline typename mat<4, 3, T, Q>::row_type operator*
    (
      typename mat<4, 3, T, Q>::col_type const& v,
      mat<4, 3, T, Q> const& m)
  {
    return typename mat<4, 3, T, Q>::row_type(
      v.x * m[0][0] + v.y * m[0][1] + v.z * m[0][2],
      v.x * m[1][0] + v.y * m[1][1] + v.z * m[1][2],
      v.x * m[2][0] + v.y * m[2][1] + v.z * m[2][2],
      v.x * m[3][0] + v.y * m[3][1] + v.z * m[3][2]);
  }

  template<typename T, qualifier Q>
  inline mat<2, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<2, 4, T, Q> const& m2)
  {
    return mat<2, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2] + m1[3][0] * m2[0][3],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2] + m1[3][1] * m2[0][3],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2] + m1[3][2] * m2[0][3],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2] + m1[3][0] * m2[1][3],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2] + m1[3][1] * m2[1][3],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2] + m1[3][2] * m2[1][3]);
  }

  template<typename T, qualifier Q>
  inline mat<3, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<3, 4, T, Q> const& m2)
  {
    T const SrcA00 = m1[0][0];
    T const SrcA01 = m1[0][1];
    T const SrcA02 = m1[0][2];
    T const SrcA10 = m1[1][0];
    T const SrcA11 = m1[1][1];
    T const SrcA12 = m1[1][2];
    T const SrcA20 = m1[2][0];
    T const SrcA21 = m1[2][1];
    T const SrcA22 = m1[2][2];
    T const SrcA30 = m1[3][0];
    T const SrcA31 = m1[3][1];
    T const SrcA32 = m1[3][2];

    T const SrcB00 = m2[0][0];
    T const SrcB01 = m2[0][1];
    T const SrcB02 = m2[0][2];
    T const SrcB03 = m2[0][3];
    T const SrcB10 = m2[1][0];
    T const SrcB11 = m2[1][1];
    T const SrcB12 = m2[1][2];
    T const SrcB13 = m2[1][3];
    T const SrcB20 = m2[2][0];
    T const SrcB21 = m2[2][1];
    T const SrcB22 = m2[2][2];
    T const SrcB23 = m2[2][3];

    mat<3, 3, T, Q> Result;
    Result[0][0] = SrcA00 * SrcB00 + SrcA10 * SrcB01 + SrcA20 * SrcB02 + SrcA30 * SrcB03;
    Result[0][1] = SrcA01 * SrcB00 + SrcA11 * SrcB01 + SrcA21 * SrcB02 + SrcA31 * SrcB03;
    Result[0][2] = SrcA02 * SrcB00 + SrcA12 * SrcB01 + SrcA22 * SrcB02 + SrcA32 * SrcB03;
    Result[1][0] = SrcA00 * SrcB10 + SrcA10 * SrcB11 + SrcA20 * SrcB12 + SrcA30 * SrcB13;
    Result[1][1] = SrcA01 * SrcB10 + SrcA11 * SrcB11 + SrcA21 * SrcB12 + SrcA31 * SrcB13;
    Result[1][2] = SrcA02 * SrcB10 + SrcA12 * SrcB11 + SrcA22 * SrcB12 + SrcA32 * SrcB13;
    Result[2][0] = SrcA00 * SrcB20 + SrcA10 * SrcB21 + SrcA20 * SrcB22 + SrcA30 * SrcB23;
    Result[2][1] = SrcA01 * SrcB20 + SrcA11 * SrcB21 + SrcA21 * SrcB22 + SrcA31 * SrcB23;
    Result[2][2] = SrcA02 * SrcB20 + SrcA12 * SrcB21 + SrcA22 * SrcB22 + SrcA32 * SrcB23;
    return Result;
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator*(mat<4, 3, T, Q> const& m1, mat<4, 4, T, Q> const& m2)
  {
    return mat<4, 3, T, Q>(
      m1[0][0] * m2[0][0] + m1[1][0] * m2[0][1] + m1[2][0] * m2[0][2] + m1[3][0] * m2[0][3],
      m1[0][1] * m2[0][0] + m1[1][1] * m2[0][1] + m1[2][1] * m2[0][2] + m1[3][1] * m2[0][3],
      m1[0][2] * m2[0][0] + m1[1][2] * m2[0][1] + m1[2][2] * m2[0][2] + m1[3][2] * m2[0][3],
      m1[0][0] * m2[1][0] + m1[1][0] * m2[1][1] + m1[2][0] * m2[1][2] + m1[3][0] * m2[1][3],
      m1[0][1] * m2[1][0] + m1[1][1] * m2[1][1] + m1[2][1] * m2[1][2] + m1[3][1] * m2[1][3],
      m1[0][2] * m2[1][0] + m1[1][2] * m2[1][1] + m1[2][2] * m2[1][2] + m1[3][2] * m2[1][3],
      m1[0][0] * m2[2][0] + m1[1][0] * m2[2][1] + m1[2][0] * m2[2][2] + m1[3][0] * m2[2][3],
      m1[0][1] * m2[2][0] + m1[1][1] * m2[2][1] + m1[2][1] * m2[2][2] + m1[3][1] * m2[2][3],
      m1[0][2] * m2[2][0] + m1[1][2] * m2[2][1] + m1[2][2] * m2[2][2] + m1[3][2] * m2[2][3],
      m1[0][0] * m2[3][0] + m1[1][0] * m2[3][1] + m1[2][0] * m2[3][2] + m1[3][0] * m2[3][3],
      m1[0][1] * m2[3][0] + m1[1][1] * m2[3][1] + m1[2][1] * m2[3][2] + m1[3][1] * m2[3][3],
      m1[0][2] * m2[3][0] + m1[1][2] * m2[3][1] + m1[2][2] * m2[3][2] + m1[3][2] * m2[3][3]);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator/(mat<4, 3, T, Q> const& m, T const& s)
  {
    return mat<4, 3, T, Q>(
      m[0] / s,
      m[1] / s,
      m[2] / s,
      m[3] / s);
  }

  template<typename T, qualifier Q>
  inline mat<4, 3, T, Q> operator/(T const& s, mat<4, 3, T, Q> const& m)
  {
    return mat<4, 3, T, Q>(
      s / m[0],
      s / m[1],
      s / m[2],
      s / m[3]);
  }

  // -- Boolean operators --

  template<typename T, qualifier Q>
  inline bool operator==(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return (m1[0] == m2[0]) && (m1[1] == m2[1]) && (m1[2] == m2[2]) && (m1[3] == m2[3]);
  }

  template<typename T, qualifier Q>
  inline bool operator!=(mat<4, 3, T, Q> const& m1, mat<4, 3, T, Q> const& m2)
  {
    return (m1[0] != m2[0]) || (m1[1] != m2[1]) || (m1[2] != m2[2]) || (m1[3] != m2[3]);
  }
} //namespace glm




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 4 columns of 3 components matrix of double-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<4, 3, double, defaultp>		dmat4x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_double4x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 4 columns of 3 components matrix of double-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, double, lowp>		lowp_dmat4x3;

  /// 4 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, double, mediump>	mediump_dmat4x3;

  /// 4 columns of 3 components matrix of double-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, double, highp>	highp_dmat4x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float4x3.hpp




namespace glm
{
  /// @addtogroup core_matrix
  /// @{

  /// 4 columns of 3 components matrix of single-precision floating-point numbers.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  typedef mat<4, 3, float, defaultp>			mat4x3;

  /// @}
}//namespace glm


/// @ref core
/// @file glm/ext/matrix_float4x3_precision.hpp




namespace glm
{
  /// @addtogroup core_matrix_precision
  /// @{

  /// 4 columns of 3 components matrix of single-precision floating-point numbers using low precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, float, lowp>		lowp_mat4x3;

  /// 4 columns of 3 components matrix of single-precision floating-point numbers using medium precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, float, mediump>	mediump_mat4x3;

  /// 4 columns of 3 components matrix of single-precision floating-point numbers using high precision arithmetic in term of ULPs.
  ///
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.1.6 Matrices</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 4.7.2 Precision Qualifier</a>
  typedef mat<4, 3, float, highp>		highp_mat4x3;

  /// @}
}//namespace glm




namespace glm {
  namespace detail
  {
    template<length_t C, length_t R, typename T, qualifier Q>
    struct outerProduct_trait {};

    template<typename T, qualifier Q>
    struct outerProduct_trait<2, 2, T, Q>
    {
      typedef mat<2, 2, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<2, 3, T, Q>
    {
      typedef mat<3, 2, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<2, 4, T, Q>
    {
      typedef mat<4, 2, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<3, 2, T, Q>
    {
      typedef mat<2, 3, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<3, 3, T, Q>
    {
      typedef mat<3, 3, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<3, 4, T, Q>
    {
      typedef mat<4, 3, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<4, 2, T, Q>
    {
      typedef mat<2, 4, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<4, 3, T, Q>
    {
      typedef mat<3, 4, T, Q> type;
    };

    template<typename T, qualifier Q>
    struct outerProduct_trait<4, 4, T, Q>
    {
      typedef mat<4, 4, T, Q> type;
    };
  }//namespace detail

     /// @addtogroup core_func_matrix
     /// @{

     /// Multiply matrix x by matrix y component-wise, i.e.,
     /// result[i][j] is the scalar product of x[i][j] and y[i][j].
     ///
     /// @tparam C Integer between 1 and 4 included that qualify the number a column
     /// @tparam R Integer between 1 and 4 included that qualify the number a row
     /// @tparam T Floating-point or signed integer scalar types
     /// @tparam Q Value from qualifier enum
     ///
     /// @see <a href="http://www.opengl.org/sdk/docs/manglsl/xhtml/matrixCompMult.xml">GLSL matrixCompMult man page</a>
     /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 8.6 Matrix Functions</a>
  template<length_t C, length_t R, typename T, qualifier Q>
  mat<C, R, T, Q> matrixCompMult(mat<C, R, T, Q> const& x, mat<C, R, T, Q> const& y);

  /// Treats the first parameter c as a column vector
  /// and the second parameter r as a row vector
  /// and does a linear algebraic matrix multiply c * r.
  ///
  /// @tparam C Integer between 1 and 4 included that qualify the number a column
  /// @tparam R Integer between 1 and 4 included that qualify the number a row
  /// @tparam T Floating-point or signed integer scalar types
  /// @tparam Q Value from qualifier enum
  ///
  /// @see <a href="http://www.opengl.org/sdk/docs/manglsl/xhtml/outerProduct.xml">GLSL outerProduct man page</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 8.6 Matrix Functions</a>
  template<length_t C, length_t R, typename T, qualifier Q>
  typename detail::outerProduct_trait<C, R, T, Q>::type outerProduct(vec<C, T, Q> const& c, vec<R, T, Q> const& r);

  /// Returns the transposed matrix of x
  ///
  /// @tparam C Integer between 1 and 4 included that qualify the number a column
  /// @tparam R Integer between 1 and 4 included that qualify the number a row
  /// @tparam T Floating-point or signed integer scalar types
  /// @tparam Q Value from qualifier enum
  ///
  /// @see <a href="http://www.opengl.org/sdk/docs/manglsl/xhtml/transpose.xml">GLSL transpose man page</a>
  /// @see <a href="http://www.opengl.org/registry/doc/GLSLangSpec.4.20.8.pdf">GLSL 4.20.8 specification, section 8.6 Matrix Functions</a>
  template<length_t C, length_t R, typename T, qualifier Q>
  typename mat<C, R, T, Q>::transpose_type transpose(mat<C, R, T, Q> const& x);

  /// Return the determinant of a squared matrix.
  ///
  /// @tparam C Integer between 1 and 4 included that qualify the number a column
  /// @tparam R Integer between 1 and 4 included that qualify the number a row
  /// @tparam T Floating-point or signed integer scalar types
  /// @tparam Q Value from qualifier enum
  ///
  /// @see <a href="http://www.opengl.org/sdk/docs/manglsl/xhtml/determina