/*
  ZynAddSubFX - a software synthesizer

  SVFilter.C - Several state-variable filters
  Copyright (C) 2002-2005 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  Modified for rakarrack by Josep Andreu

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

*/

#include "SVFilter.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

SVFilter::SVFilter(unsigned char Ftype, f32 Ffreq, f32 Fq, unsigned char Fstages)
  : stages(Fstages)
  , type(Ftype)
  , freq(Ffreq)
  , q(Fq)
{
  outgain = 1.0f;
  if (stages >= MAX_FILTER_STAGES)
    stages = MAX_FILTER_STAGES;
  cleanup();
  setfreq_and_q(Ffreq, Fq);
}

void SVFilter::cleanup()
{
  for (i32 i = 0; i < MAX_FILTER_STAGES + 1; i++)
  {
    st[i].low = 0.0;
    st[i].high = 0.0;
    st[i].band = 0.0;
    st[i].notch = 0.0;
  }
  oldabovenq = 0;
  abovenq = 0;
}

void SVFilter::computefiltercoefs()
{
  par.f = freq / fSAMPLE_RATE * 4.0f;
  if (par.f > 0.99999)
    par.f = 0.99999f;
  par.q = 1.0f - atanf(sqrtf(q)) * 2.0f / PI_;
  par.q = powf(par.q, 1.0f / (f32)(stages + 1));
  par.q_sqrt = sqrtf(par.q);
}

void SVFilter::setfreq(f32 frequency)
{
  if (frequency < 0.1)
    frequency = 0.1f;
  f32 rap = freq / frequency;
  if (rap < 1.0)
    rap = 1.0f / rap;

  oldabovenq = abovenq;
  abovenq = frequency > (fSAMPLE_RATE / 2.0f - 500.0f);

  i32 nyquistthresh = (abovenq ^ oldabovenq);

  if ((rap > 3.0) || (nyquistthresh != 0))
  {				//if the frequency is changed fast, it needs interpolation (now, filter and coeficients backup)
    if (firsttime == 0)
      needsinterpolation = 1;
    ipar = par;
  }
  freq = frequency;
  computefiltercoefs();
  firsttime = 0;
}

void SVFilter::setfreq_and_q(f32 frequency, f32 q_)
{
  q = q_;
  setfreq(frequency);
}

void SVFilter::setq(f32 q_)
{
  q = q_;
  computefiltercoefs();
}

void SVFilter::settype(i32 type_)
{
  type = type_;
  computefiltercoefs();
}

void SVFilter::setgain(f32 dBgain)
{
  gain = dB2rap(dBgain);
  computefiltercoefs();
}

void SVFilter::setstages(i32 stages_)
{
  if (stages_ >= MAX_FILTER_STAGES)
    stages_ = MAX_FILTER_STAGES - 1;
  stages = stages_;
  cleanup();
  computefiltercoefs();
}

void SVFilter::singlefilterout(f32* smp, fstage& x, parameters& par_, const i32 blockSize)
{
  f32* out = NULL;
  switch (type)
  {
  case 0:
    out = &x.low;
    break;
  case 1:
    out = &x.high;
    break;
  case 2:
    out = &x.band;
    break;
  case 3:
    out = &x.notch;
    break;
  default:
    unreachable();
  }

  for (i32 i = 0; i < blockSize; i++)
  {
    x.low = x.low + par_.f * x.band;
    x.high = par_.q_sqrt * smp[i] - x.low - par_.q * x.band;
    x.band = par_.f * x.high + x.band;
    x.notch = x.high + x.low;

    smp[i] = *out;
  }
}

void SVFilter::filterout(f32* smp, const i32 blockSize)
{
  f32* ismp = NULL;

  if (needsinterpolation != 0)
  {
    ismp = new f32[blockSize];
    for (i32 i = 0; i < blockSize; i++)
      ismp[i] = smp[i];
    for (i32 i = 0; i < stages + 1; i++)
      singlefilterout(ismp, st[i], ipar, blockSize);
  }

  for (i32 i = 0; i < stages + 1; i++)
    singlefilterout(smp, st[i], par, blockSize);

  if (needsinterpolation != 0)
  {
    for (i32 i = 0; i < blockSize; i++)
    {
      f32 x = (f32)i / fPERIOD;
      smp[i] = ismp[i] * (1.0f - x) + smp[i] * x;
    }
    delete[] ismp;
    needsinterpolation = 0;
  }

  for (i32 i = 0; i < blockSize; i++)
    smp[i] *= outgain;
}

#endif // SHR3D_SFX_CORE_RAKARRACK
