/****************************************************************************
*
* NAME: smbPitchShift.cpp
* VERSION: 1.2
* HOME URL: http://www.dspdimension.com
* KNOWN BUGS: none
*
* SYNOPSIS: Routine for doing pitch shifting while maintaining
* duration using the Short Time Fourier Transform.
*
* DESCRIPTION: The routine takes a pitchShift factor value which is between 0.5
* (one octave down) and 2. (one octave up). A value of exactly 1 does not change
* the pitch. numSampsToProcess tells the routine how many samples in indata[0...
* numSampsToProcess-1] should be pitch shifted and moved to outdata[0 ...
* numSampsToProcess-1]. The two buffers can be identical (ie. it can process the
* data in-place). fftFrameSize defines the FFT frame size used for the
* processing. Typical values are 1024, 2048 and 4096. It may be any value <=
* MAX_FRAME_LENGTH but it MUST be a power of 2. osamp is the STFT
* oversampling factor which also determines the overlap between adjacent STFT
* frames. It should at least be 4 for moderate scaling ratios. A value of 32 is
* recommended for best quality. sampleRate takes the sample rate for the signal
* in unit Hz, ie. 44100 for 44.1 kHz audio. The data passed to the routine in
* indata[] should be in the range [-1.0, 1.0), which is also the output range
* for the data, make sure you scale the data accordingly (for 16bit signed integers
* you would have to divide (and multiply) by 32768).
*
* COPYRIGHT 1999-2006 Stephan M. Bernsee <smb [AT] dspdimension [DOT] com>
*
* 						The Wide Open License (WOL)
*
* Permission to use, copy, modify, distribute and sell this software and its
* documentation for any purpose is hereby granted without fee, provided that
* the above copyright notice and this license appear in all source copies.
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY OF
* ANY KIND. See http://www.dspguru.com/wol.htm for more information.
*
*****************************************************************************/


#ifndef PITCH_H
#define PITCH_H

#include "../../../typedefs.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

#define MAX_FRAME_LENGTH 2048
class PitchShifter
{
public:
  PitchShifter(i32 fftFrameSize, i32 osamp, f32 sampleRate);
  void smbPitchShift(f32 pitchShift, i32 numSampsToProcess,
    i32 fftFrameSize, i32 osamp, f32 sampleRate,
    f32* indata, f32* outdata);
  void smbFft(f32* fftBuffer, i32 fftFrameSize, i32 sign);
  f64 smbAtan2(f64 x, f64 y);
  f32 ratio;
private:
  f32 gInFIFO[MAX_FRAME_LENGTH]{};
  f32 gOutFIFO[MAX_FRAME_LENGTH]{};
  f32 gFFTworksp[2 * MAX_FRAME_LENGTH]{};
  f32 gLastPhase[MAX_FRAME_LENGTH / 2 + 1]{};
  f32 gSumPhase[MAX_FRAME_LENGTH / 2 + 1]{};
  f32 gOutputAccum[2 * MAX_FRAME_LENGTH]{};
  f32 gAnaFreq[MAX_FRAME_LENGTH]{};
  f32 gAnaMagn[MAX_FRAME_LENGTH]{};
  f32 gSynFreq[MAX_FRAME_LENGTH];
  f32 gSynMagn[MAX_FRAME_LENGTH];

  f64 dfftFrameSize, coef_dfftFrameSize, dpi_coef;
  f64 magn, phase, tmp, window, real, imag;
  f64 freqPerBin, expct, coefPB, coef_dpi, coef_mpi;
  i32 qpd, index, inFifoLatency, stepSize, fftFrameSize2, gRover, FS_osamp;
};

#endif // SHR3D_SFX_CORE_RAKARRACK

#endif /*  */
