/*
  ZynAddSubFX - a software synthesizer

  Distorsion.C - Distorsion effect
  Copyright (C) 2002-2005 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  Modified for rakarrack by Josep Andreu & Ryan Billing

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

*/

#include "Derelict.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

#ifdef SHR3D_SFX_CORE_RAKARRACK_LIBSAMPLERATE

#include "../common/AnalogFilter.h"
#include "../common/Filter.h"

#include <string.h>

/*
 * Waveshape (this is called by OscilGen::waveshape and Distorsion::process)
 */

Derelict::Derelict()
  : lpfl(2, 22000, 1, 0)
  , lpfr(2, 22000, 1, 0)
  , hpfl(3, 20, 1, 0)
  , hpfr(3, 20, 1, 0)
  , blockDCl(2, 75.0f, 1, 0)
  , blockDCr(2, 75.0f, 1, 0)
  , DCl(3, 30, 1, 0)
  , DCr(3, 30, 1, 0)

  , filterpars(0, 64, 64)
{
  blockDCl.setfreq(75.0f);
  blockDCr.setfreq(75.0f);

  DCl.setfreq(30.0f);
  DCr.setfreq(30.0f);

  filterpars.Pcategory = 2;
  filterpars.Ptype = 0;
  filterpars.Pfreq = 72;
  filterpars.Pq = 76;
  filterpars.Pstages = 0;
  filterpars.Pgain = 76;

  filterl = new Filter(filterpars);
  filterr = new Filter(filterpars);

  setpreset(0);

  //default values
  Pvolume = 50;
  Plrcross = 40;
  Pdrive = 1;
  Plevel = 32;
  Ptype = 0;
  Pnegate = 0;
  Plpf = 127;
  Phpf = 0;
  Prfreq = 64;
  Pprefiltering = 0;
  Poctave = 0;
  togglel = 1.0;
  octave_memoryl = -1.0;
  toggler = 1.0;
  octave_memoryr = -1.0;
  octmix = 0.0;

  cleanup();
}

void Derelict::cleanup()
{
  lpfl.cleanup();
  hpfl.cleanup();
  lpfr.cleanup();
  hpfr.cleanup();
  blockDCr.cleanup();
  blockDCl.cleanup();
  DCl.cleanup();
  DCr.cleanup();
}

void Derelict::applyfilters(f32** outBlock, const i32 blockSize)
{
  lpfl.filterout(outBlock[0], blockSize);
  hpfl.filterout(outBlock[0], blockSize);
  lpfr.filterout(outBlock[1], blockSize);
  hpfr.filterout(outBlock[1], blockSize);
}

void Derelict::processBlock(f32** inBlock, f32** outBlock, const i32 blockSize)
{
  f32 inputvol = .5f;

  if (Pnegate != 0)
    inputvol *= -1.0f;

  if (Pprefiltering != 0)
    applyfilters(inBlock, blockSize);

  //no optimised, yet (no look table)

  wshapel.waveshapesmps(blockSize, inBlock[0], Ptype, Pdrive, 2);
  wshaper.waveshapesmps(blockSize, inBlock[1], Ptype, Pdrive, 2);

  memcpy(outBlock[0], inBlock[0], blockSize * sizeof(f32));
  memcpy(outBlock[1], inBlock[0], blockSize * sizeof(f32));

  if (octmix > 0.01f)
  {
    for (i32 i = 0; i < blockSize; i++)
    {
      const f32 lout = outBlock[0][i];
      const f32 rout = outBlock[1][i];

      if ((octave_memoryl < 0.0f) && (lout > 0.0f)) togglel *= -1.0f;
      octave_memoryl = lout;

      if ((octave_memoryr < 0.0f) && (rout > 0.0f)) toggler *= -1.0f;
      octave_memoryr = rout;

      octoutl[i] = lout * togglel;
      octoutr[i] = rout * toggler;
    }

    blockDCr.filterout(octoutr, blockSize);
    blockDCl.filterout(octoutl, blockSize);
  }

  filterl->filterout(inBlock[0], blockSize);
  filterr->filterout(inBlock[1], blockSize);

  if (Pprefiltering == 0)
    applyfilters(outBlock, blockSize);

  const f32 level = dB2rap(60.0f * (f32)Plevel / 127.0f - 40.0f);

  for (i32 i = 0; i < blockSize; i++)
  {
    f32 lout = outBlock[0][i];
    f32 rout = outBlock[1][i];

    const f32 l = lout * (1.0f - lrcross) + rout * lrcross;
    const f32 r = rout * (1.0f - lrcross) + lout * lrcross;

    if (octmix > 0.01f)
    {
      lout = l * (1.0f - octmix) + octoutl[i] * octmix;
      rout = r * (1.0f - octmix) + octoutr[i] * octmix;
    }
    else
    {
      lout = l;
      rout = r;
    }

    outBlock[0][i] = lout * level * panning;
    outBlock[1][i] = rout * level * (1.0f - panning);
  }

  DCr.filterout(outBlock[1], blockSize);
  DCl.filterout(outBlock[0], blockSize);
}


/*
 * Parameter control
 */
void Derelict::setvolume(i32 Pvolume_)
{
  Pvolume = Pvolume_;

  outvolume = (f32)Pvolume_ / 127.0f;
  if (Pvolume_ == 0)
    cleanup();
}

void Derelict::setpanning(i32 Ppanning_)
{
  Ppanning = Ppanning_;
  panning = ((f32)Ppanning_ + 0.5f) / 127.0f;
}

void Derelict::setlrcross(i32 Plrcross_)
{
  Plrcross = Plrcross_;
  lrcross = (f32)Plrcross_ / 127.0f * 1.0f;
}

void Derelict::setlpf(i32 value)
{
  Plpf = value;
  f32 fr = (f32)Plpf;
  lpfl.setfreq(fr);
  lpfr.setfreq(fr);
}

void Derelict::sethpf(i32 value)
{
  Phpf = value;
  f32 fr = (f32)Phpf;
  hpfl.setfreq(fr);
  hpfr.setfreq(fr);
}

void Derelict::setoctave(i32 Poctave_)
{
  Poctave = Poctave_;
  octmix = (f32)(Poctave_) / 127.0f;
}

void Derelict::setpreset(i32 npreset)
{
  const i32 PRESET_SIZE = 11;
  i32 presets[][PRESET_SIZE] = {
    //Derelict 1
    {0, 64, 64, 83, 65, 15, 0, 2437, 169, 68, 0},
    //Derelict 2
    {0, 64, 64, 95, 45, 6, 0, 3459, 209, 60, 1},
    //Derelict 3
    {0, 64, 64, 43, 77, 16, 0, 2983, 118, 83, 0}
  };

  for (i32 n = 0; n < PRESET_SIZE; n++)
    changepar(n, presets[npreset][n]);

  cleanup();
}

void Derelict::changepar(i32 npar, i32 value)
{
  switch (npar)
  {
  case 0:
    setvolume(value);
    return;
  case 1:
    setpanning(value);
    return;
  case 2:
    setlrcross(value);
    return;
  case 3:
    Pdrive = value;
    return;
  case 4:
    Plevel = value;
    return;
  case 5:
    Ptype = value;
    return;
  case 6:
    if (value > 1)
      value = 1;
    Pnegate = value;
    return;
  case 7:
    setlpf(value);
    return;
  case 8:
    sethpf(value);
    return;
  case 9:
    Prfreq = value;
    rfreq = expf(powf((f32)value / 127.0f, 0.5f) * logf(25000.0f)) + 40.0f;
    filterl->setfreq(rfreq);
    filterr->setfreq(rfreq);
    return;
  case 10:
    Pprefiltering = value;
    return;
  case 11:
    setoctave(value);
    return;
  }
  assert(false);
};

i32 Derelict::getpar(i32 npar)
{
  switch (npar)
  {
  case 0:
    return Pvolume;
  case 1:
    return Ppanning;
  case 2:
    return Plrcross;
  case 3:
    return Pdrive;
  case 4:
    return Plevel;
  case 5:
    return Ptype;
  case 6:
    return Pnegate;
  case 7:
    return Plpf;
  case 8:
    return Phpf;
  case 9:
    return Prfreq;
  case 10:
    return Pprefiltering;
  case 11:
    return Poctave;
  }
  ASSERT(false);
  return 0;
}

#endif // SHR3D_SFX_CORE_RAKARRACK_LIBSAMPLERATE

#endif // SHR3D_SFX_CORE_RAKARRACK
