/*
  ZynAddSubFX - a software synthesizer

  Distorsion.C - Distorsion effect
  Copyright (C) 2002-2005 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  Modified for rakarrack by Josep Andreu & Ryan Billing

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

*/

#include "Distorsion_.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

#ifdef SHR3D_SFX_CORE_RAKARRACK_LIBSAMPLERATE

#include <string.h>

Distorsion::Distorsion()
  : lpfl(2, 22000, 1, 0)
  , lpfr(2, 22000, 1, 0)
  , hpfl(3, 20, 1, 0)
  , hpfr(3, 20, 1, 0)
  , blockDCl(2, 440.0f, 1, 0)
  , blockDCr(2, 440.0f, 1, 0)
  , DCl(3, 30, 1, 0)
  , DCr(3, 30, 1, 0)
{
  blockDCl.setfreq(75.0f);
  blockDCr.setfreq(75.0f);
  DCl.setfreq(30.0f);
  DCr.setfreq(30.0f);

  setpreset(0);
  cleanup();
}

void Distorsion::cleanup()
{
  lpfl.cleanup();
  hpfl.cleanup();
  lpfr.cleanup();
  hpfr.cleanup();
  blockDCr.cleanup();
  blockDCl.cleanup();
  DCl.cleanup();
  DCr.cleanup();
}

void Distorsion::applyfilters(f32** outBlock, const i32 blockSize)
{
  lpfl.filterout(outBlock[0], blockSize);
  hpfl.filterout(outBlock[0], blockSize);

  if (Pstereo != 0)
  {				//stereo
    lpfr.filterout(outBlock[1], blockSize);
    hpfr.filterout(outBlock[1], blockSize);
  }
}

void Distorsion::processBlock(const f32* const* inBlock, f32** outBlock, const i32 blockSize)
{
  i32 i;
  f32 l, r, lout, rout;

  f32 inputvol = powf(5.0f, ((f32)Pdrive - 32.0f) / 127.0f);
  if (Pnegate != 0)
    inputvol *= -1.0f;

  if (Pstereo != 0)
  {				//Stereo
    for (i = 0; i < blockSize; i++)
    {
      outBlock[0][i] = inBlock[0][i] * inputvol * 2.0f;
      outBlock[1][i] = inBlock[1][i] * inputvol * 2.0f;
    }
  }
  else
  {
    for (i = 0; i < blockSize; i++)
    {
      outBlock[0][i] =
        (inBlock[0][i] + inBlock[1][i]) * inputvol;
    }
  }

  if (Pprefiltering != 0)
    applyfilters(outBlock, blockSize);

  //no optimised, yet (no look table)

  dwshapel.waveshapesmps(blockSize, outBlock[0], Ptype, Pdrive, 1);
  if (Pstereo != 0)
    dwshaper.waveshapesmps(blockSize, outBlock[1], Ptype, Pdrive, 1);

  if (Pprefiltering == 0)
    applyfilters(outBlock, blockSize);

  if (Pstereo == 0)
      memcpy(outBlock[1], outBlock[0], blockSize * sizeof(f32));

  if (octmix > 0.01f)
  {
    for (i = 0; i < blockSize; i++)
    {
      lout = outBlock[0][i];
      rout = outBlock[1][i];


      if ((octave_memoryl < 0.0f) && (lout > 0.0f)) togglel *= -1.0f;

      octave_memoryl = lout;

      if ((octave_memoryr < 0.0f) && (rout > 0.0f)) toggler *= -1.0f;

      octave_memoryr = rout;

      octoutl[i] = lout * togglel;
      octoutr[i] = rout * toggler;
    }

    blockDCr.filterout(octoutr, blockSize);
    blockDCl.filterout(octoutl, blockSize);
  }

  const f32 level = dB2rap(60.0f * (f32)Plevel / 127.0f - 40.0f);

  for (i = 0; i < blockSize; i++)
  {
    lout = outBlock[0][i];
    rout = outBlock[1][i];


    l = lout * (1.0f - lrcross) + rout * lrcross;
    r = rout * (1.0f - lrcross) + lout * lrcross;

    if (octmix > 0.01f)
    {
      lout = l * (1.0f - octmix) + octoutl[i] * octmix;
      rout = r * (1.0f - octmix) + octoutr[i] * octmix;
    }
    else
    {
      lout = l;
      rout = r;
    }

    outBlock[0][i] = lout * 2.0f * level * panning;
    outBlock[1][i] = rout * 2.0f * level * (1.0f - panning);
  }

  DCr.filterout(outBlock[1], blockSize);
  DCl.filterout(outBlock[0], blockSize);
}

void Distorsion::setvolume(i32 Pvolume_)
{
  Pvolume = Pvolume_;

  outvolume = (f32)Pvolume_ / 127.0f;
  if (Pvolume_ == 0)
    cleanup();
}

void Distorsion::setpanning(i32 Ppanning_)
{
  Ppanning = Ppanning_;
  panning = ((f32)Ppanning_ + 0.5f) / 127.0f;
}

void Distorsion::setlrcross(i32 Plrcross_)
{
  Plrcross = Plrcross_;
  lrcross = (f32)Plrcross_ / 127.0f * 1.0f;
}

void Distorsion::setlpf(i32 value)
{
  Plpf = value;
  f32 fr = (f32)Plpf;
  lpfl.setfreq(fr);
  lpfr.setfreq(fr);
}

void Distorsion::sethpf(i32 value)
{
  Phpf = value;
  f32 fr = (f32)Phpf;

  hpfl.setfreq(fr);
  hpfr.setfreq(fr);
  //Prefiltering of 51 is approx 630 Hz. 50 - 60 generally good for OD pedal.
}

void Distorsion::setoctave(i32 Poctave_)
{
  Poctave = Poctave_;
  octmix = (f32)(Poctave_) / 127.0f;
}

void Distorsion::setpreset(i32 npreset)
{
  const i32 PRESET_SIZE = 11;
  i32 presets[][PRESET_SIZE] = {
    //Overdrive 1
    {84, 64, 35, 56, 40, 0, 0, 6703, 21, 0, 0},
    //Overdrive 2
    {85, 64, 35, 29, 45, 1, 0, 25040, 21, 0, 0},
    //Distorsion 1
    {0, 64, 0, 87, 14, 6, 0, 3134, 157, 0, 1},
    //Distorsion 2
    {0, 64, 127, 87, 14, 0, 1, 3134, 102, 0, 0},
    //Distorsion 3
    {0, 64, 127, 127, 12, 13, 0, 5078, 56, 0, 1},
    //Guitar Amp
    {84, 64, 35, 63, 50, 2, 0, 824, 21, 0, 0}
  };

  for (i32 n = 0; n < PRESET_SIZE; n++)
    changepar(n, presets[npreset][n]);

  cleanup();
}

void Distorsion::changepar(i32 npar, i32 value)
{
  switch (npar)
  {
  case 0:
    setvolume(value);
    return;
  case 1:
    setpanning(value);
    return;
  case 2:
    setlrcross(value);
    return;
  case 3:
    Pdrive = value;
    return;
  case 4:
    Plevel = value;
    return;
  case 5:
    Ptype = value;
    return;
  case 6:
    if (value > 1)
      value = 1;
    Pnegate = value;
    return;
  case 7:
    setlpf(value);
    return;
  case 8:
    sethpf(value);
    return;
  case 9:
    if (value > 1)
      value = 1;
    Pstereo = value;
    return;
  case 10:
    Pprefiltering = value;
    return;
  case 11:
    return;
  case 12:
    setoctave(value);
    return;
  }
  ASSERT(false);
}

i32 Distorsion::getpar(i32 npar)
{
  switch (npar)
  {
  case 0:
    return Pvolume;
  case 1:
    return Ppanning;
  case 2:
    return Plrcross;
  case 3:
    return Pdrive;
  case 4:
    return Plevel;
  case 5:
    return Ptype;
  case 6:
    return Pnegate;
  case 7:
    return Plpf;
  case 8:
    return Phpf;
  case 9:
    return Pstereo;
  case 10:
    return Pprefiltering;
  case 11:
    return 0;
  case 12:
    return Poctave;
  }
  assert(false);
  return 0;
}

#endif // SHR3D_SFX_CORE_RAKARRACK_LIBSAMPLERATE

#endif // SHR3D_SFX_CORE_RAKARRACK
