/*
  Shuffle.C - Distorsion effect

  ZynAddSubFX - a software synthesizer
  Copyright (C) 2002-2005 Nasca Octavian Paul
  Author: Nasca Octavian Paul

  Modified for rakarrack by Josep Andreu

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License (version 2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
*/

#include "Shuffle.h"

#ifdef SHR3D_SFX_CORE_RAKARRACK

#include "Distorsion_.h"

/*
 * Waveshape (this is called by OscilGen::waveshape and Distorsion::process)
 */

Shuffle::Shuffle()
  : lr(6, 300.0f, .3f, 0)
  , hr(6, 8000.0f, .3f, 0)
  , mlr(6, 1200.0f, .3f, 0)
  , mhr(6, 2400.0f, .3f, 0)
{
  //default values
  setpreset(0);
  cleanup();
}

void Shuffle::cleanup()
{
  lr.cleanup();
  hr.cleanup();
  mlr.cleanup();
  mhr.cleanup();
}

void Shuffle::processBlock(const f32* const* inBlock, f32** outBlock, const i32 blockSize)
{
  for (i32 i = 0; i < blockSize; i++)
  {
    inputl[i] = inBlock[0][i] + inBlock[1][i];
    inputr[i] = inBlock[0][i] - inBlock[1][i];
  }

  if (E)
  {
    lr.filterout(inputr, blockSize);
    mlr.filterout(inputr, blockSize);
    mhr.filterout(inputr, blockSize);
    hr.filterout(inputr, blockSize);
  }
  else
  {
    lr.filterout(inputl, blockSize);
    mlr.filterout(inputl, blockSize);
    mhr.filterout(inputl, blockSize);
    hr.filterout(inputl, blockSize);
  }

  for (i32 i = 0; i < blockSize; i++)
  {
    outBlock[0][i] = (inputl[i] + inputr[i] - inBlock[0][i]) * 0.333333f;
    outBlock[1][i] = (inputl[i] - inputr[i] - inBlock[1][i]) * 0.333333f;
  }
}

void Shuffle::setvolume(i32 value)
{
  Pvolume = value;
  outvolume = (f32)Pvolume / 128.0f;
}

void Shuffle::setCross1(i32 value)
{
  Cross1 = value;
  lr.setfreq((f32)value);
}

void Shuffle::setCross2(i32 value)
{
  Cross2 = value;
  mlr.setfreq((f32)value);
}

void Shuffle::setCross3(i32 value)
{
  Cross3 = value;
  mhr.setfreq((f32)value);
}

void Shuffle::setCross4(i32 value)
{
  Cross4 = value;
  hr.setfreq((f32)value);
}

void Shuffle::setGainL(i32 value)
{
  PvolL = value + 64;
  volL = 30.0f * ((f32)PvolL - 64.0f) / 64.0f;
  lr.setgain(volL);
}

void Shuffle::setGainML(i32 value)
{
  PvolML = value + 64;
  volML = 30.0f * ((f32)PvolML - 64.0f) / 64.0f;
  mlr.setgain(volML);
}

void Shuffle::setGainMH(i32 value)
{
  PvolMH = value + 64;
  volMH = 30.0f * ((f32)PvolMH - 64.0f) / 64.0f;
  mhr.setgain(volMH);
}

void Shuffle::setGainH(i32 value)
{
  PvolH = value + 64;
  volH = 30.0f * ((f32)PvolH - 64.0f) / 64.0f;
  hr.setgain(volH);
}

void Shuffle::setpreset(i32 npreset)
{
  const i32 PRESET_SIZE = 11;
  i32 presets[][PRESET_SIZE] = {
    //Shuffle 1
    {64, 10, 0, 0, 0,600, 1200,2000, 6000,-14, 1},
    //Shuffle 2
    {64, 0, 0, 0, 0, 120, 1000,2400, 8000,-7, 1},
    //Shuffle 3
    {64, 0, 0, 0, 0, 60, 1800, 3700, 12000, 7, 0},
    //Remover
    {0, 17, 0, 7, 5, 600, 1200, 2000, 13865, -45, 1}
  };

  for (i32 n = 0; n < PRESET_SIZE; n++)
    changepar(n, presets[npreset][n]);

  cleanup();
}

void Shuffle::changepar(i32 npar, i32 value)
{
  switch (npar)
  {
  case 0:
    setvolume(value);
    return;
  case 1:
    setGainL(value);
    return;
  case 2:
    setGainML(value);
    return;
  case 3:
    setGainMH(value);
    return;
  case 4:
    setGainH(value);
    return;
  case 5:
    setCross1(value);
    return;
  case 6:
    setCross2(value);
    return;
  case 7:
    setCross3(value);
    return;
  case 8:
    setCross4(value);
    return;
  case 9:
    PQ = value;
    value += 64;
    tmp = powf(30.0f, ((f32)value - 64.0f) / 64.0f);
    lr.setq(tmp);
    mlr.setq(tmp);
    mhr.setq(tmp);
    hr.setq(tmp);
    return;
  case 10:
    E = value;
    return;
  }
  ASSERT(false);
}

i32 Shuffle::getpar(i32 npar)
{
  switch (npar)
  {
  case 0:
    return Pvolume;
  case 1:
    return PvolL - 64;
  case 2:
    return PvolML - 64;
  case 3:
    return PvolMH - 64;
  case 4:
    return PvolH - 64;
  case 5:
    return Cross1;
  case 6:
    return Cross2;
  case 7:
    return Cross3;
  case 8:
    return Cross4;
  case 9:
    return PQ;
  case 10:
    return E;
  }
  ASSERT(false);
  return 0;			//in case of bogus parameter number
}

#endif // SHR3D_SFX_CORE_RAKARRACK
