// SPDX-License-Identifier: Unlicense

#include "tones.h"

#include "base64.h"
#include "file.h"
#include "ini.h"
#include "global.h"
#include "sfx.h"

#ifdef SHR3D_SFX
static void loadSfxTone(const std::map<std::string, std::string>& sfxTone)
{
  for (const auto& [key, value] : sfxTone)
  {
    ASSERT(key.size() >= 1);

    SfxToneIndex sfxToneIndex;
    i32 slotIndex = -1;
    {
      const char* str = reinterpret_cast<const char*>(key.c_str());
      const SfxBankIndex sfxBankIndex = atoi(str); // atoi stops at ',' We are good.
      i32 toneInBank = -1;
      str += 1;
      while (str[0] != '\0')
      {
        if (str[0] == ',')
        {
          if (toneInBank < 0)
          {
            toneInBank = atoi(str + 1);
          }
          else
          {
            slotIndex = atoi(str + 1);
            break;
          }
        }
        str = str + 1;
      }
      ASSERT(toneInBank / Const::sfxToneTonesPerBank == 0);
      sfxToneIndex = sfxBankIndex * Const::sfxToneTonesPerBank + toneInBank;
    }

    if (slotIndex < 0) // value is bank name
    {
      Global::sfxToneNames[sfxToneIndex] = value;
    }
    else // is sfx
    {
      if (slotIndex > ARRAY_SIZE(Global::effectChain) - 1)
        continue;

      const u64 comma0 = value.find(',');
      const u64 comma1 = value.find(',', comma0 + 1);

      const std::string sfxSystemStr = value.substr(0, comma0);
      const std::string sfxName = value.substr(comma0 + 1, comma1 - comma0 - 1);

      const SfxSystem sfxSystem = Sfx::name2SfxSystem(sfxSystemStr.c_str());
      const std::vector<std::string>& names = Sfx::names[sfxSystem];

      for (SfxIndex k = 0; k < SfxIndex(names.size()); ++k)
      {
        if (sfxName != names[k])
          continue;

        ASSERT(value[comma1 + 1] == '-' || value[comma1 + 1] == 'M');

        Global::sfxTone[sfxToneIndex][slotIndex].id.system = sfxSystem;
        Global::sfxTone[sfxToneIndex][slotIndex].id.sfxIndex = k;
        Global::sfxTone[sfxToneIndex][slotIndex].state = SfxEffectState(value[comma1 + 1]);

        const std::string parameters = value.substr(comma1 + 3);

#ifdef SHR3D_SFX_PLUGIN
        if (Sfx::sfxSystemIsPlugin(sfxSystem))
        {
          const std::vector<u8> decodedData = Base64::decode(parameters.data(), parameters.size());
          const std::string decodedParameters(reinterpret_cast<const char*>(decodedData.data()), decodedData.size());

          Global::sfxParameters[sfxToneIndex][slotIndex] = decodedParameters;
        }
        else
#endif // SHR3D_SFX_PLUGIN
        {
          Global::sfxParameters[sfxToneIndex][slotIndex] = parameters;
        }
        break;
      }
    }
  }
}
#endif // SHR3D_SFX

void Tones::init()
{
  std::map<std::string, std::map<std::string, std::string>> serializedSaves;

  if (File::exists(Settings::pathTonesIni.c_str()))
  {
    const std::vector<u8> tonesIniData = File::read(Settings::pathTonesIni.c_str());
    serializedSaves = Ini::loadIniContent(reinterpret_cast<const char*>(tonesIniData.data()), tonesIniData.size());
  }
#ifdef SHR3D_SFX
  else
  {
#ifdef SHR3D_SFX_CORE_RAKARRACK_PRESETS
    serializedSaves = { { "Tones", {
    { "01,0",   "Rodent Airlines" },
    { "01,0,0", "CoreRakarrack,Expander,-,-55,37,24,46,8050,76,1" },
    { "01,0,1", "CoreRakarrack,StompBox,-,32,-44,0,0,110,2" },
    { "01,0,2", "CoreRakarrack,Echotron,-,63,28,39,8,0,10,0,64,3,32,4,64,1,1,0,1" },
    { "01,0,3", "CoreRakarrack,Reverbtron,-,64,17,1,400,0,0,34,6,1,0,0,64,1,0,2524,0" },
    { "01,1",   "Ambient Flange" },
    { "01,1,0", "CoreRakarrack,Expander,-,-55,37,24,46,2091,20,1" },
    { "01,1,1", "CoreRakarrack,Sustainer,-,66,37" },
    { "01,1,2", "CoreRakarrack,Echotron,-,63,20,20,8,0,61,0,64,3,32,4,64,1,1,1,1" },
    { "01,1,3", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,34,39,1,0,0,64,1,0,2524,76" },
    { "01,2",   "Cathedral Swell" },
    { "01,2,0", "CoreRakarrack,Expander,-,-32,20,1806,118,6673,20,46" },
    { "01,2,1", "CoreRakarrack,Sustainer,-,99,29" },
    { "01,2,2", "CoreRakarrack,Reverbtron,-,103,0,1,400,0,30,0,40,9,0,0,64,0,0,20000,0" },
    { "01,3",   "Go with Him" },
    { "01,3,0", "CoreRakarrack,Ring,-,-64,0,-64,127,44,392,0,74,18,48,17,81,1" },
    { "01,3,1", "CoreRakarrack,Synthfilter,-,54,59,180,0,0,91,67,-40,6,0,0,57,-17,77,176,53" },
    { "01,3,2", "CoreRakarrack,Echoverse,-,119,64,90,52,37,64,64,0,1,114" },
    { "01,3,3", "CoreRakarrack,Pan,-,37,64,180,0,0,127,74,1,1" },
    { "01,3,4", "CoreRakarrack,Reverb,-,69,64,30,24,0,3090,5,83,1,83" },
    { "01,4",   "All is Lost" },
    { "01,4,0", "CoreRakarrack,EQ,-,64,71,75,71,64,62,68,73,64,32,56,64" },
    { "01,4,1", "CoreRakarrack,Echoverse,-,64,64,132,64,64,29,10,10,3,64" },
    { "01,4,2", "CoreRakarrack,Arpie,-,39,64,132,64,100,72,62,127,6,3,3" },
    { "01,4,3", "CoreRakarrack,Chorus,-,64,64,33,0,0,68,40,45,33,119,0,0" },
    { "01,4,4", "CoreRakarrack,Phaser,-,64,64,11,0,0,64,40,64,1,0,0,20" },
    { "01,4,5", "CoreRakarrack,Reverb,-,80,64,63,24,0,85,5,83,1,64" },
    { "01,5",   "Tight Rock" },
    { "01,5,0", "CoreRakarrack,Expander,-,-55,18,50,50,8050,76,0" },
    { "01,5,1", "CoreRakarrack,Distorsion,-,0,64,64,127,76,26,1,2982,645,0,1,0,0" },
    { "01,5,2", "CoreRakarrack,Overdrive,-,0,64,35,9,44,26,0,3327,21,0,0,0,0" },
    { "01,5,3", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,99,25,0,0,0,64,0,0,0,0" },
    { "01,5,4", "CoreRakarrack,Parametric EQ,-,127,76,12,931,53,64,8111,64,64,64" },
    { "01,6",   "Summer at the Pool" },
    { "01,6,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "01,6,1", "CoreRakarrack,Sustainer,-,36,91" },
    { "01,6,2", "CoreRakarrack,Echotron,-,55,72,37,4,0,84,3,47,0,96,-6,64,1,0,0,1" },
    { "01,7",   "Summer at the Pit" },
    { "01,7,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "01,7,1", "CoreRakarrack,StompBox,-,42,-20,0,0,70,2" },
    { "01,7,2", "CoreRakarrack,Echotron,-,51,72,37,4,0,73,3,47,0,96,-6,64,1,0,0,1" },
    { "01,8",   "Summer at the Beach" },
    { "01,8,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "01,8,1", "CoreRakarrack,Sustainer,-,36,91" },
    { "01,8,2", "CoreRakarrack,Analog Phaser,-,64,20,25,50,1,64,93,40,6,48,0,58,1" },
    { "01,8,3", "CoreRakarrack,Echotron,-,64,45,34,4,0,76,3,41,0,96,-13,64,1,1,0,1" },
    { "01,9",   "Suction" },
    { "01,9,0", "CoreRakarrack,Expander,-,-64,22,50,50,1200,70,0" },
    { "01,9,1", "CoreRakarrack,Sustainer,-,75,73" },
    { "01,9,2", "CoreRakarrack,Echotron,-,0,47,28,8,0,92,0,64,3,32,0,64,1,1,0,1" },
    { "01,9,3", "CoreRakarrack,Reverbtron,-,104,0,1,400,0,0,60,15,2,0,0,64,0,0,20000,20" },
    { "02,0",   "Suction Flange +dist" },
    { "02,0,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "02,0,1", "CoreRakarrack,Sustainer,-,75,73" },
    { "02,0,2", "CoreRakarrack,StompBox,-,37,64,0,0,127,4" },
    { "02,0,3", "CoreRakarrack,Echotron,-,64,36,93,8,0,81,0,64,3,32,0,64,1,0,0,1" },
    { "02,0,4", "CoreRakarrack,Reverbtron,-,104,0,1,155,0,0,60,21,2,0,0,64,0,0,20000,37" },
    { "02,1",   "Succulent Clean" },
    { "02,1,0", "CoreRakarrack,Expander,-,-57,19,10,10,8050,76,72" },
    { "02,1,1", "CoreRakarrack,MuTroMojo,-,28,8,35,0,0,64,50,-6,53,58,28,-16,32,2,2600,300,1,0" },
    { "02,1,2", "CoreRakarrack,DualFlange,-,-40,0,0,35,9,67,12,4700,1,1,160,75,0,60" },
    { "02,2",   "EboWeboAmiYeebo" },
    { "02,2,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,483,0" },
    { "02,2,1", "CoreRakarrack,Sustainer,-,58,54" },
    { "02,2,2", "CoreRakarrack,Echoverse,-,64,64,101,48,33,77,0,0,1,67" },
    { "02,2,3", "CoreRakarrack,Sequence,-,30,127,30,49,80,40,110,80,44,101,95,1,0,2,2" },
    { "02,3",   "Rock Amp" },
    { "02,3,0", "CoreRakarrack,Expander,-,-55,37,24,46,8050,76,1" },
    { "02,3,1", "CoreRakarrack,Sustainer,-,36,51" },
    { "02,3,2", "CoreRakarrack,StompBox,-,55,22,15,0,21,0" },
    { "02,3,3", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,72,2,0,-61" },
    { "02,3,4", "CoreRakarrack,Reverbtron,-,108,0,0,181,0,0,0,0,3,0,0,64,1,1,20000,21" },
    { "02,4",   "Overdrive" },
    { "02,4,0", "CoreRakarrack,StompBox,-,127,19,-23,0,5,0" },
    { "02,4,1", "CoreRakarrack,Reverbtron,-,108,0,0,155,0,0,0,0,3,0,0,64,1,1,20000,12" },
    { "02,5",   "Metal Amp" },
    { "02,5,0", "CoreRakarrack,Expander,-,-51,50,59,46,8050,76,1" },
    { "02,5,1", "CoreRakarrack,StompBox,-,48,0,-32,-36,53,6" },
    { "02,5,2", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,88,0,0,0" },
    { "02,5,3", "CoreRakarrack,Reverbtron,-,64,0,0,327,0,0,0,0,7,0,0,64,0,0,1585,12" },
    { "02,6",   "Hard Rock Amp" },
    { "02,6,0", "CoreRakarrack,Expander,-,-55,37,24,46,8050,76,1" },
    { "02,6,1", "CoreRakarrack,Overdrive,-,0,64,35,79,48,26,1,5406,708,0,1,0,0" },
    { "02,6,2", "CoreRakarrack,StompBox,-,120,-3,-21,28,51,0" },
    { "02,6,3", "CoreRakarrack,Convolotron,-,0,64,0,134,0,0,0,51,3,0,0" },
    { "02,6,4", "CoreRakarrack,Reverbtron,-,95,0,0,155,0,0,99,6,0,0,0,64,0,0,20000,21" },
    { "02,7",   "Edgy" },
    { "02,7,0", "CoreRakarrack,Expander,-,-55,18,50,50,8050,20,0" },
    { "02,7,1", "CoreRakarrack,StompBox,-,78,10,-2,5,37,0" },
    { "02,7,2", "CoreRakarrack,Reverbtron,-,64,0,1,110,0,0,48,20,4,0,0,64,0,0,0,0" },
    { "02,7,3", "CoreRakarrack,Looper,-,64,1,0,0,1,0,70,1,0,1,70,1,0,0" },
    { "02,8",   "Dark Alley" },
    { "02,8,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "02,8,1", "CoreRakarrack,Shifter,-,0,64,64,200,200,-9,1,0,0,0" },
    { "02,8,2", "CoreRakarrack,Sustainer,-,79,83" },
    { "02,8,3", "CoreRakarrack,DualFlange,-,-32,0,64,60,10,100,20,16000,0,0,16,90,4,0" },
    { "02,8,4", "CoreRakarrack,Echoverse,-,64,64,90,64,64,64,64,0,1,64" },
    { "02,9",   "D Flange" },
    { "02,9,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "02,9,1", "CoreRakarrack,DualFlange,-,-32,0,0,110,800,10,-27,16000,1,0,40,64,1,10" },
    { "02,9,2", "CoreRakarrack,Sustainer,-,45,107" },
    { "02,9,3", "CoreRakarrack,Echoverse,-,68,64,81,64,33,39,0,0,1,67" },
    { "02,9,4", "CoreRakarrack,Reverb,-,121,64,62,24,0,3134,102,75,1,49" },
    { "03,0",   "Clean Swirl" },
    { "03,0,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "03,0,1", "CoreRakarrack,Sustainer,-,39,91" },
    { "03,0,2", "CoreRakarrack,Analog Phaser,-,64,20,32,50,1,64,93,40,6,48,0,58,1" },
    { "03,0,3", "CoreRakarrack,Echoverse,-,111,64,90,64,64,18,23,0,1,82" },
    { "03,0,4", "CoreRakarrack,Reverb,-,116,64,62,24,0,3134,102,75,1,49" },
    { "03,0,5", "CoreRakarrack,Looper,-,64,0,1,0,1,0,64,1,0,1,64,1,0,0" },
    { "03,1",   "Noise Synth" },
    { "03,1,0", "CoreRakarrack,Expander,-,-49,28,50,50,8050,76,27" },
    { "03,1,1", "CoreRakarrack,StompBox,-,127,-31,0,0,0,7" },
    { "03,1,2", "CoreRakarrack,Synthfilter,-,0,5,75,0,1,64,0,-32,6,0,0,0,11,439,122,0" },
    { "03,1,3", "CoreRakarrack,Echo,-,8,64,575,65,55,105,77,0,0" },
    { "03,1,4", "CoreRakarrack,MuTroMojo,-,28,8,35,0,0,64,50,-6,53,58,28,-16,32,4,2600,300,1,0" },
    { "03,1,5", "CoreRakarrack,DualFlange,-,-15,0,0,42,12,50,-10,1500,0,0,120,0,0,20" },
    { "03,1,6", "CoreRakarrack,Parametric EQ,-,72,64,64,1077,76,64,8111,64,64,104" },
    { "03,2",   "Acoustic Sparkle" },
    { "03,2,0", "CoreRakarrack,Compressor,-,-24,4,0,45,135,1,84,0,1" },
    { "03,2,1", "CoreRakarrack,Distorsion,-,81,64,64,90,81,26,0,9586,15641,1,1,0,0" },
    { "03,2,2", "CoreRakarrack,Reverb,-,88,64,53,20,0,25040,2417,71,1,64" },
    { "03,2,3", "CoreRakarrack,Parametric EQ,-,256,29,64,1932,39,82,11766,66,72,64" },
    { "03,3",   "Extreme" },
    { "03,3,0", "CoreRakarrack,EQ,-,17,31,32,36,42,34,42,56,59,72,58,127" },
    { "03,3,1", "CoreRakarrack,Distorsion,-,0,64,64,19,59,1,0,25040,21,1,0,0,0" },
    { "03,3,2", "CoreRakarrack,Echo,-,49,64,534,62,32,64,0,1,0" },
    { "03,3,3", "CoreRakarrack,Chorus,-,33,64,106,36,0,50,18,34,11,0,0,1" },
    { "03,3,4", "CoreRakarrack,Phaser,-,41,64,4,43,0,24,33,98,8,0,0,110" },
    { "03,3,5", "CoreRakarrack,Flanger,-,46,64,2,34,1,95,39,62,18,0,0,1" },
    { "03,4",   "Heaven Garden" },
    { "03,4,0", "CoreRakarrack,Distorsion,-,80,64,0,24,8,0,0,7678,56,0,0,0,0" },
    { "03,4,1", "CoreRakarrack,WahWah,-,112,64,140,0,0,64,70,64,0,60" },
    { "03,4,2", "CoreRakarrack,Harmonizer,-,20,48,64,24,600,0,9,0,64,64,0" },
    { "03,4,3", "CoreRakarrack,Musical Delay,-,64,0,3,7,125,49,0,127,6,89,90,96,44" },
    { "03,4,4", "CoreRakarrack,Pan,-,0,64,26,0,0,0,32,0,1" },
    { "03,4,5", "CoreRakarrack,Reverb,-,80,64,40,24,0,4002,27,83,1,64" },
    { "03,5",   "Old Pat" },
    { "03,5,0", "CoreRakarrack,Distorsion,-,0,64,64,7,57,6,0,692,21,0,0,1,0" },
    { "03,5,1", "CoreRakarrack,Echo,-,106,64,534,62,32,70,0,1,0" },
    { "03,5,2", "CoreRakarrack,Chorus,-,89,64,7,0,0,42,97,95,90,127,0,0" },
    { "03,5,3", "CoreRakarrack,Cabinet,-,3,64" },
    { "03,6",   "Metal Tone" },
    { "03,6,0", "CoreRakarrack,EQ,-,25,35,58,70,82,104,67,30,14,0,83,32" },
    { "03,6,1", "CoreRakarrack,Overdrive,-,51,64,35,21,48,14,0,824,21,0,0,1,0" },
    { "03,6,2", "CoreRakarrack,Distorsion,-,0,64,0,72,60,15,0,9586,21,0,0,1,0" },
    { "03,6,3", "CoreRakarrack,Parametric EQ,-,159,78,0,569,33,25,4767,57,9,86" },
    { "03,7",   "Dist-Duet-A7 *" },
    { "03,7,0", "CoreRakarrack,Distorsion,-,0,58,0,65,0,0,0,25040,209,0,0,0,0" },
    { "03,7,1", "CoreRakarrack,Harmonizer,-,74,104,64,16,600,1,9,19,64,64,0" },
    { "03,7,2", "CoreRakarrack,WahWah,-,116,64,39,0,0,50,54,54,0,60" },
    { "03,7,3", "CoreRakarrack,Musical Delay,-,113,0,2,7,0,59,0,127,4,59,60,75,75" },
    { "03,7,4", "CoreRakarrack,Cabinet,-,5,34" },
    { "03,8",   "Bass" },
    { "03,8,0", "CoreRakarrack,EQ,-,88,76,70,68,72,64,64,64,64,64,52,49" },
    { "03,8,1", "CoreRakarrack,Harmonizer,-,0,64,90,0,52,0,0,0,70,49,0" },
    { "03,8,2", "CoreRakarrack,Parametric EQ,-,55,70,61,870,64,55,8111,64,64,72" },
    { "03,9",   "Ballada Duet Gm *" },
    { "03,9,0", "CoreRakarrack,Distorsion,-,40,64,127,69,0,0,0,2837,408,0,1,1,0" },
    { "03,9,1", "CoreRakarrack,Harmonizer,-,64,0,64,8,600,1,7,10,64,64,0" },
    { "03,9,2", "CoreRakarrack,Chorus,-,97,64,19,0,0,98,56,90,64,0,0,0" },
    { "03,9,3", "CoreRakarrack,Echo,-,71,64,566,64,23,59,90,1,0" },
    { "03,9,4", "CoreRakarrack,Pan,-,64,64,26,0,0,0,27,0,1" },
    { "03,9,5", "CoreRakarrack,NoiseGate,-,-20,-23,1,132,6703,76,334" },
    { "04,0",   "True Heavy-3" },
    { "04,0,0", "CoreRakarrack,Distorsion,-,0,64,0,109,32,13,0,1157,37,1,1,1,0" },
    { "04,0,1", "CoreRakarrack,Pan,-,53,64,76,0,5,127,86,1,1" },
    { "04,0,2", "CoreRakarrack,Reverb,-,102,64,40,24,0,4002,27,83,1,64" },
    { "04,0,3", "CoreRakarrack,Cabinet,-,0,84" },
    { "04,1",   "Angel's Chorus" },
    { "04,1,0", "CoreRakarrack,WahWah,-,64,64,26,0,0,96,64,0,0,60" },
    { "04,1,1", "CoreRakarrack,Compressor,-,-26,4,-25,20,120,1,10,0,1" },
    { "04,1,2", "CoreRakarrack,Overdrive,-,66,64,35,29,34,1,0,25040,21,0,0,1,0" },
    { "04,1,3", "CoreRakarrack,Echo,-,67,64,347,64,30,59,0,1,64" },
    { "04,1,4", "CoreRakarrack,Chorus,-,64,64,19,0,0,98,56,90,64,19,0,0" },
    { "04,1,5", "CoreRakarrack,Reverb,-,80,64,69,35,0,25040,21,71,0,64" },
    { "04,1,6", "CoreRakarrack,Analog Phaser,-,64,20,14,5,1,64,110,40,6,10,0,70,1" },
    { "04,2",   "Funk Wah" },
    { "04,2,0", "CoreRakarrack,WahWah,-,0,64,1,0,0,64,0,109,0,0" },
    { "04,2,1", "CoreRakarrack,Compressor,-,-30,8,-22,20,63,1,30,1,1" },
    { "04,2,2", "CoreRakarrack,Parametric EQ,-,159,60,37,1077,90,55,4767,62,14,64" },
    { "04,2,3", "CoreRakarrack,Reverb,-,113,64,35,9,0,1785,27,106,1,64" },
    { "04,3",   "Chorus Riff" },
    { "04,3,0", "CoreRakarrack,EQ,-,45,38,28,43,54,71,58,76,26,0,61,68" },
    { "04,3,1", "CoreRakarrack,Overdrive,-,18,64,35,58,48,0,0,1785,21,0,0,1,0" },
    { "04,3,2", "CoreRakarrack,Chorus,-,57,58,9,0,0,69,82,35,56,127,0,0" },
    { "04,3,3", "CoreRakarrack,Parametric EQ,-,29,69,23,1077,42,64,8111,85,64,64" },
    { "04,4",   "Big Stack" },
    { "04,4,0", "CoreRakarrack,Distorsion,-,0,64,61,42,58,14,0,3293,804,0,1,1,16" },
    { "04,4,1", "CoreRakarrack,Overdrive,-,21,64,35,23,51,15,0,1292,22,0,0,1,0" },
    { "04,4,2", "CoreRakarrack,Reverb,-,115,64,63,24,0,4002,27,83,1,64" },
    { "04,4,3", "CoreRakarrack,Cabinet,-,7,64" },
    { "04,5",   "Comphaser" },
    { "04,5,0", "CoreRakarrack,Compressor,-,-55,5,-21,19,23,1,0,0,1" },
    { "04,5,1", "CoreRakarrack,Analog Phaser,-,64,20,32,50,1,64,93,40,6,48,0,58,1" },
    { "04,6",   "Jet Flanger" },
    { "04,6,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,43,26,0,89,68" },
    { "04,6,1", "CoreRakarrack,Distorsion,-,2,64,61,79,64,0,0,334,335,0,1,0,0" },
    { "04,6,2", "CoreRakarrack,Echo,-,0,64,254,64,64,18,0,1,0" },
    { "04,6,3", "CoreRakarrack,Flanger,-,23,64,8,0,0,60,23,3,69,0,0,0" },
    { "04,6,4", "CoreRakarrack,Reverb,-,99,64,35,18,0,4002,27,83,1,64" },
    { "04,6,5", "CoreRakarrack,Parametric EQ,-,128,70,89,825,55,40,7691,57,47,64" },
    { "04,7",   "Nostalgia" },
    { "04,7,0", "CoreRakarrack,EQ,-,37,55,62,56,69,72,65,56,49,44,64,64" },
    { "04,7,1", "CoreRakarrack,Distorsion,-,14,64,64,9,51,17,0,7678,94,0,1,1,0" },
    { "04,7,2", "CoreRakarrack,Pan,-,30,64,46,0,4,64,0,1,1" },
    { "04,7,3", "CoreRakarrack,Reverb,-,117,64,51,24,0,3293,21,83,1,32" },
    { "04,8",   "Panalog Phaser" },
    { "04,8,0", "CoreRakarrack,Compressor,-,-30,2,0,20,120,1,0,0,0" },
    { "04,8,1", "CoreRakarrack,Overdrive,-,60,64,35,8,102,14,0,2316,494,0,1,0,0" },
    { "04,8,2", "CoreRakarrack,Reverb,-,97,64,40,18,0,4002,27,83,1,47" },
    { "04,8,3", "CoreRakarrack,Pan,-,0,53,61,14,0,127,44,1,1" },
    { "04,8,4", "CoreRakarrack,Analog Phaser,-,64,20,61,0,1,74,69,30,6,60,1,64,1" },
    { "04,8,5", "CoreRakarrack,Chorus,-,0,0,1,0,0,0,0,0,0,0,0,0" },
    { "04,8,6", "CoreRakarrack,Parametric EQ,-,72,68,64,1077,64,64,6916,25,12,64" },
    { "04,9",   "Space Invasion" },
    { "04,9,0", "CoreRakarrack,StompBox,-,48,-8,-36,57,58,7" },
    { "04,9,1", "CoreRakarrack,AlienWah,-,7,64,57,0,0,96,102,118,31,0,36" },
    { "04,9,2", "CoreRakarrack,Reverbtron,-,79,0,1,400,0,0,39,44,1,0,-31,64,1,1,20000,12" },
    { "05,0",   "Clean-A-Duet *" },
    { "05,0,0", "CoreRakarrack,EQ,-,64,80,72,64,64,64,64,60,59,47,64,56" },
    { "05,0,1", "CoreRakarrack,Harmonizer,-,46,27,64,9,600,1,9,0,64,64,0" },
    { "05,0,2", "CoreRakarrack,Reverb,-,109,64,45,0,0,16683,435,64,1,85" },
    { "05,0,3", "CoreRakarrack,Cabinet,-,5,71" },
    { "05,1",   "Tricky Dick" },
    { "05,1,0", "CoreRakarrack,EQ,-,38,47,58,65,76,86,81,80,75,68,63,110" },
    { "05,1,1", "CoreRakarrack,Chorus,-,64,64,398,0,0,90,40,85,47,119,0,0" },
    { "05,1,2", "CoreRakarrack,Flanger,-,64,64,39,0,0,60,23,3,62,0,0,0" },
    { "05,1,3", "CoreRakarrack,Reverb,-,111,64,10,26,0,1223,21,77,1,45" },
    { "05,2",   "Dist-WahWah-1" },
    { "05,2,0", "CoreRakarrack,EQ,-,72,63,59,65,62,52,64,64,64,57,64,42" },
    { "05,2,1", "CoreRakarrack,Valve,-,0,64,64,127,48,0,5841,61,1,0,69,1,84" },
    { "05,2,2", "CoreRakarrack,WahWah,-,84,64,7,0,4,28,55,113,1,60" },
    { "05,2,3", "CoreRakarrack,Musical Delay,-,80,0,2,7,0,59,0,127,4,59,60,75,75" },
    { "05,2,4", "CoreRakarrack,Cabinet,-,1,70" },
    { "05,3",   "Rolling Echo" },
    { "05,3,0", "CoreRakarrack,Compressor,-,-10,2,0,20,120,1,0,0,1" },
    { "05,3,1", "CoreRakarrack,Distorsion,-,0,64,64,7,55,1,0,692,21,0,0,1,0" },
    { "05,3,2", "CoreRakarrack,Echo,-,0,64,1220,62,32,103,111,1,0" },
    { "05,3,3", "CoreRakarrack,Chorus,-,89,64,7,0,0,42,97,95,90,127,0,0" },
    { "05,3,4", "CoreRakarrack,Reverb,-,110,64,32,0,0,7340,21,64,1,71" },
    { "05,4",   "Gary Mooore" },
    { "05,4,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,45,26,0,89,68" },
    { "05,4,1", "CoreRakarrack,Overdrive,-,0,64,35,110,23,13,0,2982,21,0,0,0,0" },
    { "05,4,2", "CoreRakarrack,Echo,-,0,64,488,64,64,35,0,1,0" },
    { "05,4,3", "CoreRakarrack,Parametric EQ,-,55,28,72,704,65,104,5896,52,64,92" },
    { "05,5",   "Satriani" },
    { "05,5,0", "CoreRakarrack,EQ,-,18,40,53,58,68,71,64,53,26,0,40,74" },
    { "05,5,1", "CoreRakarrack,Overdrive,-,18,64,35,72,48,0,0,1364,21,0,0,0,0" },
    { "05,5,2", "CoreRakarrack,Echo,-,0,64,254,64,64,18,0,1,0" },
    { "05,5,3", "CoreRakarrack,Phaser,-,39,59,23,0,0,16,91,64,8,0,0,40" },
    { "05,6",   "SubOctave" },
    { "05,6,0", "CoreRakarrack,Distorsion,-,2,64,61,36,51,0,0,1364,76,0,1,1,47" },
    { "05,6,1", "CoreRakarrack,Reverb,-,80,64,27,12,0,4002,27,83,1,64" },
    { "05,6,2", "CoreRakarrack,Cabinet,-,0,76" },
    { "05,7",   "Lil Dirty" },
    { "05,7,0", "CoreRakarrack,Distorsion,-,7,64,0,12,65,14,0,2982,559,0,1,1,0" },
    { "05,7,1", "CoreRakarrack,EQ,-,64,77,74,64,57,59,56,50,47,44,79,61" },
    { "05,8",   "Quasi Acoustic" },
    { "05,8,0", "CoreRakarrack,EQ,-,66,73,63,45,33,45,56,69,80,26,56,64" },
    { "05,8,1", "CoreRakarrack,Chorus,-,85,64,14,0,0,90,66,48,64,119,0,0" },
    { "05,8,2", "CoreRakarrack,Reverb,-,118,64,37,7,13,5078,21,75,1,51" },
    { "05,8,3", "CoreRakarrack,Parametric EQ,-,76,77,69,1404,74,43,6218,73,32,68" },
    { "05,9",   "Satriani 2" },
    { "05,9,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,53,26,0,64,68" },
    { "05,9,1", "CoreRakarrack,Overdrive,-,12,64,35,65,48,0,0,1441,21,0,0,0,0" },
    { "05,9,2", "CoreRakarrack,Echo,-,0,64,254,64,64,18,0,1,0" },
    { "05,9,3", "CoreRakarrack,Phaser,-,39,59,23,0,0,16,91,64,8,0,0,40" },
    { "05,9,4", "CoreRakarrack,Reverb,-,82,64,13,18,0,4002,27,83,0,64" },
    { "05,9,5", "CoreRakarrack,Parametric EQ,-,128,70,89,1932,75,60,11766,72,35,80" },
    { "06,0",   "Classic Chorus" },
    { "06,0,0", "CoreRakarrack,EQ,-,18,40,53,58,51,71,82,69,49,0,88,68" },
    { "06,0,1", "CoreRakarrack,Chorus,-,33,63,43,0,0,73,49,51,64,127,0,0" },
    { "06,0,2", "CoreRakarrack,Reverb,-,99,64,35,18,0,4002,27,83,1,64" },
    { "06,0,3", "CoreRakarrack,Parametric EQ,-,128,70,89,825,55,40,7691,70,47,64" },
    { "06,1",   "Trash Chorus" },
    { "06,1,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,43,26,0,89,68" },
    { "06,1,1", "CoreRakarrack,Distorsion,-,2,64,61,79,36,0,0,334,335,0,1,0,0" },
    { "06,1,2", "CoreRakarrack,Echo,-,0,64,254,64,64,18,0,1,0" },
    { "06,1,3", "CoreRakarrack,Chorus,-,33,63,1,0,0,73,120,86,64,127,0,0" },
    { "06,1,4", "CoreRakarrack,Reverb,-,99,64,35,18,0,4002,27,83,1,64" },
    { "06,1,5", "CoreRakarrack,Parametric EQ,-,128,70,89,825,58,40,7691,57,47,96" },
    { "06,2",   "Ballada Solo" },
    { "06,2,0", "CoreRakarrack,EQ,-,45,38,28,43,54,59,46,21,7,0,56,68" },
    { "06,2,1", "CoreRakarrack,Distorsion,-,40,64,61,127,48,0,0,2837,76,0,1,0,0" },
    { "06,2,2", "CoreRakarrack,Echo,-,48,64,566,64,23,59,90,1,0" },
    { "06,3",   "Spandex" },
    { "06,3,0", "CoreRakarrack,Distorsion,-,18,64,0,85,23,14,0,1785,494,0,1,1,0" },
    { "06,3,1", "CoreRakarrack,Overdrive,-,25,64,35,53,58,17,0,1292,35,0,0,1,0" },
    { "06,3,2", "CoreRakarrack,Echo,-,88,64,566,64,30,59,0,1,0" },
    { "06,3,3", "CoreRakarrack,Reverb,-,47,64,21,10,0,652,21,93,1,24" },
    { "06,4",   "CrunchPhaser" },
    { "06,4,0", "CoreRakarrack,Distorsion,-,0,64,0,87,37,14,0,2564,157,0,1,1,0" },
    { "06,4,1", "CoreRakarrack,Analog Phaser,-,64,20,14,0,1,64,110,40,4,10,0,64,1" },
    { "06,5",   "Joe" },
    { "06,5,0", "CoreRakarrack,EQ,-,23,35,61,73,84,82,69,54,41,18,56,64" },
    { "06,5,1", "CoreRakarrack,Reverb,-,118,64,37,7,13,5078,21,75,1,51" },
    { "06,5,2", "CoreRakarrack,Parametric EQ,-,84,77,77,704,64,80,7293,58,77,72" },
    { "06,6",   "Falling in Fantasy" },
    { "06,6,0", "CoreRakarrack,Distorsion,-,69,64,64,58,67,26,0,9586,10441,1,1,0,0" },
    { "06,6,1", "CoreRakarrack,Analog Phaser,-,64,20,5,0,3,127,127,28,4,10,0,72,1" },
    { "06,6,2", "CoreRakarrack,Echo,-,55,64,1143,65,55,35,21,1,1" },
    { "06,6,3", "CoreRakarrack,Reverb,-,103,64,62,24,0,3134,102,75,1,49" },
    { "06,6,4", "CoreRakarrack,Parametric EQ,-,256,29,64,1932,39,82,11766,66,72,72" },
    { "06,7",   "Underwater I" },
    { "06,7,0", "CoreRakarrack,Distorsion,-,14,64,64,23,31,2,0,25040,21,0,0,1,0" },
    { "06,7,1", "CoreRakarrack,Phaser,-,9,64,31,63,1,117,71,8,2,0,0,40" },
    { "06,7,2", "CoreRakarrack,Echo,-,64,64,472,65,55,68,0,1,0" },
    { "06,7,3", "CoreRakarrack,Reverb,-,99,64,62,24,0,1441,21,83,1,49" },
    { "06,7,4", "CoreRakarrack,Parametric EQ,-,72,73,45,1077,64,64,8111,69,38,64" },
    { "06,8",   "Slap Echo" },
    { "06,8,0", "CoreRakarrack,EQ,-,18,40,53,58,51,71,82,69,49,0,89,68" },
    { "06,8,1", "CoreRakarrack,Echo,-,64,64,254,64,64,18,0,1,0" },
    { "06,8,2", "CoreRakarrack,Parametric EQ,-,128,75,89,968,23,53,9513,82,69,72" },
    { "06,9",   "Aural Phaser" },
    { "06,9,0", "CoreRakarrack,EQ,-,18,40,53,58,51,71,82,69,49,0,80,68" },
    { "06,9,1", "CoreRakarrack,Phaser,-,39,59,23,0,0,16,91,64,8,0,0,40" },
    { "06,9,2", "CoreRakarrack,Reverb,-,99,64,35,18,0,4002,27,83,1,64" },
    { "06,9,3", "CoreRakarrack,Parametric EQ,-,128,70,89,825,55,40,7691,70,47,64" },
    { "07,0",   "Nasty" },
    { "07,0,0", "CoreRakarrack,Distorsion,-,39,64,0,65,35,14,0,428,31,0,1,1,65" },
    { "07,0,1", "CoreRakarrack,Overdrive,-,35,64,35,65,35,11,0,1785,1203,0,1,1,0" },
    { "07,0,2", "CoreRakarrack,Echo,-,88,64,566,64,30,59,0,1,0" },
    { "07,0,3", "CoreRakarrack,Analog Phaser,-,64,20,14,10,0,64,110,80,7,10,1,45,1" },
    { "07,0,4", "CoreRakarrack,Reverb,-,47,64,21,10,0,652,21,93,1,24" },
    { "07,1",   "Aymimadre" },
    { "07,1,0", "CoreRakarrack,Parametric EQ,-,256,59,64,1932,76,82,11766,66,72,88" },
    { "07,1,1", "CoreRakarrack,Overdrive,-,0,64,35,127,23,6,0,314,494,0,1,1,0" },
    { "07,1,2", "CoreRakarrack,Chorus,-,62,64,1,0,1,18,120,85,64,119,0,0" },
    { "07,1,3", "CoreRakarrack,Echo,-,56,64,51,65,55,74,21,1,0" },
    { "07,2",   "Clean-2" },
    { "07,2,0", "CoreRakarrack,Parametric EQ,-,256,59,64,1932,76,82,11766,66,72,60" },
    { "07,2,1", "CoreRakarrack,Flanger,-,112,64,7,85,0,41,77,0,11,17,0,1" },
    { "07,2,2", "CoreRakarrack,Echo,-,52,64,51,65,55,74,21,1,0" },
    { "07,2,3", "CoreRakarrack,Reverb,-,119,64,20,24,0,3134,102,75,0,49" },
    { "07,3",   "Dist-4" },
    { "07,3,0", "CoreRakarrack,Overdrive,-,0,64,35,127,61,0,0,9174,21,0,1,1,0" },
    { "07,3,1", "CoreRakarrack,Parametric EQ,-,44,58,39,1404,66,46,17068,24,46,59" },
    { "07,3,2", "CoreRakarrack,Echo,-,119,64,410,65,55,109,21,1,0" },
    { "07,3,3", "CoreRakarrack,Compressor,-,-27,8,-13,32,144,1,1,0,1" },
    { "07,4",   "Lil Blue" },
    { "07,4,0", "CoreRakarrack,Distorsion,-,0,64,0,8,40,14,1,1785,672,0,1,0,0" },
    { "07,4,1", "CoreRakarrack,Overdrive,-,0,64,35,16,65,14,0,1692,21,0,0,0,0" },
    { "07,4,2", "CoreRakarrack,EQ,-,64,77,74,64,57,59,56,50,47,44,79,61" },
    { "07,4,3", "CoreRakarrack,Reverb,-,116,64,17,6,0,1223,21,97,0,77" },
    { "07,5",   "Funk-1" },
    { "07,5,0", "CoreRakarrack,Parametric EQ,-,89,64,39,1932,76,82,7691,76,37,64" },
    { "07,5,1", "CoreRakarrack,Chorus,-,62,64,1,0,1,18,0,85,64,119,0,0" },
    { "07,5,2", "CoreRakarrack,Phaser,-,122,64,16,47,0,34,89,9,3,127,1,10" },
    { "07,5,3", "CoreRakarrack,Echo,-,89,64,410,65,55,20,21,1,0" },
    { "07,6",   "Goog Bye" },
    { "07,6,0", "CoreRakarrack,Parametric EQ,-,93,69,39,825,78,32,11766,37,46,48" },
    { "07,6,1", "CoreRakarrack,Chorus,-,69,64,156,89,0,125,12,2,38,119,0,0" },
    { "07,6,2", "CoreRakarrack,Echo,-,90,64,690,65,55,35,21,1,0" },
    { "07,6,3", "CoreRakarrack,Reverb,-,113,64,22,24,0,4002,27,83,1,64" },
    { "07,7",   "Naranja One" },
    { "07,7,0", "CoreRakarrack,Parametric EQ,-,72,64,64,600,76,41,8111,64,64,64" },
    { "07,7,1", "CoreRakarrack,Overdrive,-,0,64,35,108,39,0,0,6703,21,0,0,1,0" },
    { "07,7,2", "CoreRakarrack,Echo,-,99,64,566,64,30,59,0,1,0" },
    { "07,7,3", "CoreRakarrack,Chorus,-,72,64,26,0,0,99,43,102,84,119,0,0" },
    { "07,8",   "Like Pat" },
    { "07,8,0", "CoreRakarrack,Parametric EQ,-,104,78,64,600,80,41,13086,42,36,56" },
    { "07,8,1", "CoreRakarrack,Echo,-,117,64,566,64,30,53,0,1,0" },
    { "07,8,2", "CoreRakarrack,Chorus,-,47,64,39,64,0,23,57,43,75,19,0,0" },
    { "07,8,3", "CoreRakarrack,Reverb,-,110,64,42,10,0,25040,81,78,1,64" },
    { "07,9",   "The Trooper" },
    { "07,9,0", "CoreRakarrack,Distorsion,-,0,64,0,31,20,0,0,3134,157,0,1,1,0" },
    { "07,9,1", "CoreRakarrack,Reverb,-,91,64,28,24,0,4002,27,83,1,64" },
    { "07,9,2", "CoreRakarrack,Parametric EQ,-,128,51,99,1481,76,64,10032,46,94,87" },
    { "08,0",   "Hendrix" },
    { "08,0,0", "CoreRakarrack,Distorsion,-,84,64,35,27,34,1,0,824,21,0,0,1,0" },
    { "08,0,1", "CoreRakarrack,Overdrive,-,84,64,35,56,23,0,0,4200,21,0,0,1,0" },
    { "08,0,2", "CoreRakarrack,Reverb,-,111,64,10,26,0,1223,21,77,1,45" },
    { "08,1",   "Talk-Clean" },
    { "08,1,0", "CoreRakarrack,EQ,-,42,62,42,28,21,35,42,65,53,0,83,94" },
    { "08,1,1", "CoreRakarrack,Echo,-,67,64,347,64,30,59,0,1,0" },
    { "08,1,2", "CoreRakarrack,Chorus,-,64,64,7,0,1,42,97,95,90,127,0,0" },
    { "08,1,3", "CoreRakarrack,WahWah,-,64,64,26,0,0,96,64,0,0,60" },
    { "08,1,4", "CoreRakarrack,Pan,-,64,64,19,0,0,0,10,0,1" },
    { "08,1,5", "CoreRakarrack,Reverb,-,110,64,14,0,0,25040,27,71,0,25" },
    { "08,1,6", "CoreRakarrack,Cabinet,-,1,47" },
    { "08,2",   "Acoustic" },
    { "08,2,0", "CoreRakarrack,Exciter,-,16,0,0,0,-32,0,-32,-64,0,48,0,26000,20" },
    { "08,2,1", "CoreRakarrack,Sequence,-,20,30,32,20,14,25,30,7,96,120,61,0,0,6,3" },
    { "08,2,2", "CoreRakarrack,Pan,-,0,64,26,0,0,127,32,0,1" },
    { "08,2,3", "CoreRakarrack,Echotron,-,88,64,16,3,0,120,48,64,1,127,-4,64,1,0,0,0" },
    { "08,3",   "Acoustic Chorus" },
    { "08,3,0", "CoreRakarrack,EQ,-,64,64,64,64,64,64,64,64,64,64,56,64" },
    { "08,3,1", "CoreRakarrack,Exciter,-,64,0,0,0,-32,0,-32,-64,0,48,0,10648,4743" },
    { "08,3,2", "CoreRakarrack,Sequence,-,20,30,32,20,14,25,30,7,72,120,61,0,0,6,3" },
    { "08,3,3", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,30,50,6,0,0,64,0,0,20000,0" },
    { "08,4",   "Chinese Lead" },
    { "08,4,0", "CoreRakarrack,EQ,-,0,0,53,62,67,69,79,81,0,0,65,64" },
    { "08,4,1", "CoreRakarrack,Distorsion,-,0,64,64,127,30,0,0,2316,65,1,1,1,0" },
    { "08,4,2", "CoreRakarrack,Parametric EQ,-,256,46,25,2802,90,53,8111,79,9,16" },
    { "08,4,3", "CoreRakarrack,Flanger,-,64,64,1,0,0,60,102,3,127,64,0,0" },
    { "08,5",   "Soft Stereo" },
    { "08,5,0", "CoreRakarrack,Parametric EQ,-,72,67,64,1077,36,50,4521,72,50,72" },
    { "08,5,1", "CoreRakarrack,Overdrive,-,84,64,35,22,66,13,1,2982,294,1,1,1,0" },
    { "08,5,2", "CoreRakarrack,Echo,-,34,64,488,105,2,20,0,1,0" },
    { "08,5,3", "CoreRakarrack,Chorus,-,14,64,4,0,0,0,107,40,64,72,0,1" },
    { "08,5,4", "CoreRakarrack,Pan,-,36,64,26,0,0,0,34,0,1" },
    { "08,5,5", "CoreRakarrack,Reverb,-,111,64,30,24,0,25040,118,65,1,47" },
    { "08,6",   "Acoustic Bright" },
    { "08,6,0", "CoreRakarrack,EQ,-,0,0,109,30,97,18,83,83,51,0,97,25" },
    { "08,6,1", "CoreRakarrack,Distorsion,-,0,64,0,32,68,0,1,3134,157,0,1,1,0" },
    { "08,6,2", "CoreRakarrack,Reverb,-,90,64,51,10,0,25040,81,78,1,64" },
    { "08,7",   "Rain Echo" },
    { "08,7,0", "CoreRakarrack,EQ,-,0,0,42,95,35,90,88,74,0,0,39,95" },
    { "08,7,1", "CoreRakarrack,Musical Delay,-,48,0,5,1,64,59,23,127,2,69,120,127,127" },
    { "08,7,2", "CoreRakarrack,Flanger,-,64,64,39,0,0,60,23,3,62,0,0,0" },
    { "08,7,3", "CoreRakarrack,Parametric EQ,-,143,79,64,1647,79,64,8111,64,64,44" },
    { "08,7,4", "CoreRakarrack,Cabinet,-,6,64" },
    { "08,8",   "Swirly" },
    { "08,8,0", "CoreRakarrack,Analog Phaser,-,104,80,5,0,4,64,110,88,5,10,1,97,1" },
    { "08,8,1", "CoreRakarrack,Echo,-,48,64,924,64,23,59,90,1,92" },
    { "08,8,2", "CoreRakarrack,Chorus,-,57,58,9,0,0,69,82,35,56,127,0,0" },
    { "08,8,3", "CoreRakarrack,Parametric EQ,-,29,69,23,1077,42,64,8111,74,64,64" },
    { "08,9",   "CleanChord" },
    { "08,9,0", "CoreRakarrack,Parametric EQ,-,72,64,64,1077,64,64,8111,64,64,54" },
    { "08,9,1", "CoreRakarrack,Musical Delay,-,43,0,2,7,0,39,9,127,4,32,36,127,127" },
    { "08,9,2", "CoreRakarrack,Chorus,-,64,64,19,0,0,98,56,90,64,19,0,0" },
    { "08,9,3", "CoreRakarrack,Reverb,-,104,64,53,20,0,25040,2417,71,1,64" },
    { "08,9,4", "CoreRakarrack,Cabinet,-,8,64" },
    { "09,0",   "Shall" },
    { "09,0,0", "CoreRakarrack,Convolotron,-,16,64,1,9,0,0,30,32,0,0,0" },
    { "09,0,1", "CoreRakarrack,DistBand,-,64,64,64,63,24,20,19,26,32,32,24,0,638,1708,0" },
    { "09,0,2", "CoreRakarrack,Parametric EQ,-,412,80,40,3222,88,40,6000,48,40,64" },
    { "09,0,3", "CoreRakarrack,Shelf Boost,-,64,8,6962,1,24" },
    { "09,0,4", "CoreRakarrack,Reverbtron,-,64,0,1,150,0,0,19,48,6,0,0,64,1,0,20000,20" },
    { "09,0,5", "CoreRakarrack,Cabinet,-,4,48" },
    { "09,1",   "Shall2" },
    { "09,1,0", "CoreRakarrack,Convolotron,-,0,64,1,9,0,0,30,32,0,0,0" },
    { "09,1,1", "CoreRakarrack,StompBox,-,48,3,2,33,16,6" },
    { "09,1,2", "CoreRakarrack,Shelf Boost,-,48,8,13274,0,48" },
    { "09,1,3", "CoreRakarrack,Parametric EQ,-,412,79,25,1651,72,25,8545,69,42,56" },
    { "09,1,4", "CoreRakarrack,Reverbtron,-,64,0,1,100,0,18,0,18,4,0,-2,64,1,0,20000,9" },
    { "09,1,5", "CoreRakarrack,Cabinet,-,2,55" },
    { "09,2",   "Amp" },
    { "09,2,0", "CoreRakarrack,Sustainer,-,18,127" },
    { "09,2,1", "CoreRakarrack,Shelf Boost,-,55,23,9493,0,52" },
    { "09,2,2", "CoreRakarrack,StompBox,-,49,32,0,32,16,0" },
    { "09,2,3", "CoreRakarrack,Reverbtron,-,88,0,1,141,0,0,67,24,5,0,0,64,1,0,20000,8" },
    { "09,2,4", "CoreRakarrack,Cabinet,-,1,56" },
    { "09,3",   "Steve Distortion" },
    { "09,3,0", "CoreRakarrack,Compressor,-,-30,2,-6,20,120,1,0,0,1" },
    { "09,3,1", "CoreRakarrack,Valve,-,0,64,64,127,64,0,5078,61,1,0,112,0,30" },
    { "09,3,2", "CoreRakarrack,NoiseGate,-,0,-20,1,2,6703,76,2" },
    { "09,3,3", "CoreRakarrack,StompBox,-,42,0,-17,0,116,6" },
    { "09,3,4", "CoreRakarrack,Reverb,-,100,64,33,0,0,25040,21,106,1,30" },
    { "09,3,5", "CoreRakarrack,Convolotron,-,67,60,1,7,0,0,30,20,4,0,0" },
    { "09,4",   "Clean Smack Back" },
    { "09,4,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "09,4,1", "CoreRakarrack,Compressor,-,-34,3,-8,54,130,1,72,0,1" },
    { "09,4,2", "CoreRakarrack,DualFlange,-,-32,0,64,60,10,100,20,16000,0,0,16,90,4,0" },
    { "09,4,3", "CoreRakarrack,Echoverse,-,77,64,105,53,64,76,0,0,0,64" },
    { "09,4,4", "CoreRakarrack,Looper,-,64,1,0,0,1,0,64,0,1,1,64,1,0,0" },
    { "09,5",   "Soft Dist" },
    { "09,5,0", "CoreRakarrack,DistBand,-,16,64,0,24,48,21,19,21,20,31,25,0,338,800,0" },
    { "09,5,1", "CoreRakarrack,Shelf Boost,-,96,20,14135,0,48" },
    { "09,5,2", "CoreRakarrack,Reverb,-,96,64,24,10,0,6633,5,73,1,74" },
    { "09,5,3", "CoreRakarrack,Parametric EQ,-,207,66,20,2240,68,20,9455,66,1,64" },
    { "09,6",   "VocoHarm" },
    { "09,6,0", "CoreRakarrack,Vocoder,-,0,64,7,48,92,100,10" },
    { "09,6,1", "CoreRakarrack,Harmonizer,-,83,64,100,8,1201,0,0,0,64,17,0" },
    { "09,6,2", "CoreRakarrack,Reverb,-,80,64,23,24,0,4507,5,83,1,85" },
    { "09,7",   "Ambience" },
    { "09,7,0", "CoreRakarrack,EQ,-,72,68,64,64,64,64,64,64,68,72,56,40" },
    { "09,7,1", "CoreRakarrack,Echotron,-,112,64,64,4,0,180,64,64,0,127,-24,64,1,0,0,0" },
    { "09,7,2", "CoreRakarrack,Reverbtron,-,88,0,1,220,0,0,38,16,5,0,0,64,1,0,20000,8" },
    { "09,7,3", "CoreRakarrack,Pan,-,32,64,50,0,0,0,32,0,1" },
    { "09,8",   "Mono Arpeg" },
    { "09,8,0", "CoreRakarrack,Distorsion,-,0,64,0,86,49,27,0,4701,157,0,1,0,0" },
    { "09,8,1", "CoreRakarrack,Ring,-,-42,0,-64,94,74,196,0,55,0,83,26,77,1" },
    { "09,8,2", "CoreRakarrack,Synthfilter,-,76,20,93,0,1,64,110,-40,6,0,0,59,0,500,100,62" },
    { "09,8,3", "CoreRakarrack,Shelf Boost,-,48,29,10979,1,33" },
    { "09,8,4", "CoreRakarrack,Sequence,-,0,40,50,60,70,60,40,0,0,268,64,0,0,5,3" },
    { "09,8,5", "CoreRakarrack,Reverb,-,64,64,33,15,0,5216,5,83,1,74" },
    { "09,9",   "Padpeg" },
    { "09,9,0", "CoreRakarrack,Synthfilter,-,112,20,90,0,1,64,110,-40,6,0,0,59,0,500,100,62" },
    { "09,9,1", "CoreRakarrack,Shelf Boost,-,48,29,10979,1,25" },
    { "09,9,2", "CoreRakarrack,Sequence,-,0,21,33,59,62,89,106,127,104,540,64,0,0,5,3" },
    { "09,9,3", "CoreRakarrack,Pan,-,64,64,90,0,6,127,32,0,1" },
    { "09,9,4", "CoreRakarrack,Reverb,-,64,64,38,9,0,7342,5,74,0,87" },
    { "10,0",   "Rock Band" },
    { "10,0,0", "CoreRakarrack,Parametric EQ,-,220,80,16,3484,72,22,7636,72,12,16" },
    { "10,0,1", "CoreRakarrack,Sustainer,-,48,72" },
    { "10,0,2", "CoreRakarrack,Exciter,-,72,0,0,0,0,20,0,-20,0,20,0,4744,120" },
    { "10,0,3", "CoreRakarrack,DistBand,-,0,64,0,32,64,22,19,26,64,32,32,0,400,1200,0" },
    { "10,0,4", "CoreRakarrack,Shelf Boost,-,32,29,10692,0,48" },
    { "10,0,5", "CoreRakarrack,Pan,-,64,64,115,0,5,0,32,0,1" },
    { "10,0,6", "CoreRakarrack,Reverbtron,-,64,11,1,100,0,0,68,22,5,-7,0,64,1,0,20000,8" },
    { "10,1",   "Rock Band 2" },
    { "10,1,0", "CoreRakarrack,Parametric EQ,-,220,80,16,3484,72,22,7636,72,12,16" },
    { "10,1,1", "CoreRakarrack,Sustainer,-,48,72" },
    { "10,1,2", "CoreRakarrack,Exciter,-,70,0,0,0,0,31,0,-31,0,30,0,4744,120" },
    { "10,1,3", "CoreRakarrack,DistBand,-,0,64,0,24,64,22,19,26,64,32,32,0,400,1200,0" },
    { "10,1,4", "CoreRakarrack,Shelf Boost,-,48,64,16000,0,48" },
    { "10,1,5", "CoreRakarrack,Reverbtron,-,64,11,1,100,0,0,68,22,5,-7,0,64,1,0,20000,8" },
    { "10,2",   "12 Strings" },
    { "10,2,0", "CoreRakarrack,Shifter,-,94,64,64,165,383,-54,12,0,1,126" },
    { "10,2,1", "CoreRakarrack,Exciter,-,64,0,0,-31,0,29,0,-28,0,22,0,12538,5106" },
    { "10,3",   "12 Strings Down" },
    { "10,3,0", "CoreRakarrack,Shifter,-,94,64,64,165,383,-54,12,1,1,126" },
    { "10,3,1", "CoreRakarrack,Exciter,-,64,0,0,-31,0,29,0,-28,0,22,0,12538,5106" },
    { "10,3,2", "CoreRakarrack,Echotron,-,107,64,10,3,0,104,32,64,1,96,-8,64,1,0,0,0" },
    { "10,4",   "12 Strings Crazy" },
    { "10,4,0", "CoreRakarrack,Shifter,-,96,64,64,165,383,-54,12,1,1,126" },
    { "10,4,1", "CoreRakarrack,Exciter,-,64,0,0,-28,0,44,0,-42,0,29,0,6633,1473" },
    { "10,4,2", "CoreRakarrack,Sequence,-,127,0,127,0,127,0,127,0,96,164,64,1,0,5,8" },
    { "10,4,3", "CoreRakarrack,Echotron,-,112,45,34,4,0,164,3,96,0,96,-8,64,1,1,1,1" },
    { "10,5",   "Stereo Tremolo" },
    { "10,5,0", "CoreRakarrack,Distorsion,-,0,64,0,32,49,27,0,4467,110,0,1,0,0" },
    { "10,5,1", "CoreRakarrack,Chorus,-,64,64,50,0,0,64,50,30,32,64,0,1" },
    { "10,5,2", "CoreRakarrack,Sequence,-,15,48,85,106,125,84,61,33,96,200,95,1,3,4,8" },
    { "10,5,3", "CoreRakarrack,Pan,-,64,64,100,0,5,127,64,1,1" },
    { "10,5,4", "CoreRakarrack,Reverbtron,-,64,0,1,113,0,0,30,20,7,0,0,64,1,0,20000,16" },
    { "10,5,5", "CoreRakarrack,Parametric EQ,-,225,72,37,1258,67,32,6000,64,64,40" },
    { "10,6",   "Classic WF" },
    { "10,6,0", "CoreRakarrack,StompBox,-,42,64,0,0,110,4" },
    { "10,6,1", "CoreRakarrack,Chorus,-,64,64,60,0,6,127,24,30,32,64,0,1" },
    { "10,6,2", "CoreRakarrack,Sequence,-,4,8,12,8,4,8,12,8,32,120,64,0,0,0,6" },
    { "10,6,3", "CoreRakarrack,Pan,-,32,64,60,127,1,127,96,1,1" },
    { "10,7",   "Steve Dist 2" },
    { "10,7,0", "CoreRakarrack,NoiseGate,-,-26,-20,1,2,6703,76,2" },
    { "10,7,1", "CoreRakarrack,Coil Crafter,-,45,1,8,3380,10,4466,32,20,1" },
    { "10,7,2", "CoreRakarrack,Compressor,-,-30,2,-6,20,120,1,0,0,1" },
    { "10,7,3", "CoreRakarrack,Valve,-,0,64,64,127,64,0,5078,61,1,0,112,0,30" },
    { "10,7,4", "CoreRakarrack,StompBox,-,45,0,-17,0,116,6" },
    { "10,7,5", "CoreRakarrack,Reverb,-,100,64,33,0,0,25040,21,106,1,30" },
    { "10,8",   "Auto Wah" },
    { "10,8,0", "CoreRakarrack,Compressor,-,-32,16,-16,10,153,1,0,0,0" },
    { "10,8,1", "CoreRakarrack,Shifter,-,64,0,64,0,0,-20,1,1,1,20" },
    { "10,8,2", "CoreRakarrack,Valve,-,0,64,0,24,48,0,3200,110,0,1,0,0,80" },
    { "10,8,3", "CoreRakarrack,Sequence,-,11,55,15,77,13,64,11,36,96,60,110,0,4,1,2" },
    { "10,8,4", "CoreRakarrack,Pan,-,0,64,26,0,0,0,96,0,1" },
    { "10,8,5", "CoreRakarrack,VaryBand,-,32,120,0,64,240,6,127,400,1400,2000,1" },
    { "10,9",   "Auto Wah 2" },
    { "10,9,0", "CoreRakarrack,Compressor,-,-33,4,-16,10,153,1,0,0,0" },
    { "10,9,1", "CoreRakarrack,Shifter,-,64,0,64,0,0,-20,1,1,1,20" },
    { "10,9,2", "CoreRakarrack,Sequence,-,0,0,96,96,0,0,96,96,16,80,104,0,4,0,2" },
    { "10,9,3", "CoreRakarrack,Pan,-,0,64,240,0,0,0,64,0,1" },
    { "11,0",   "Stereo Vibrato" },
    { "11,0,0", "CoreRakarrack,Convolotron,-,16,64,0,20,0,0,30,24,0,0,0" },
    { "11,0,1", "CoreRakarrack,Sequence,-,64,30,45,20,60,25,42,15,64,120,64,0,1,6,3" },
    { "11,0,2", "CoreRakarrack,VaryBand,-,0,200,5,127,200,6,0,400,1000,2000,1" },
    { "11,0,3", "CoreRakarrack,Exciter,-,64,0,0,0,0,3,0,-64,0,64,0,20000,20" },
    { "11,0,4", "CoreRakarrack,Pan,-,32,64,200,0,6,126,64,1,1" },
    { "11,1",   "Dist Dead" },
    { "11,1,0", "CoreRakarrack,Compressor,-,-36,10,-8,10,213,1,26,0,1" },
    { "11,1,1", "CoreRakarrack,StompBox,-,48,0,6,6,24,5" },
    { "11,1,2", "CoreRakarrack,Shuffle,-,32,-5,3,-64,-57,252,1251,3000,6000,-22,1" },
    { "11,1,3", "CoreRakarrack,VaryBand,-,32,234,0,127,234,6,0,400,2600,4000,1" },
    { "11,1,4", "CoreRakarrack,Pan,-,0,64,234,0,0,127,96,0,1" },
    { "11,1,5", "CoreRakarrack,Echotron,-,96,35,34,4,0,58,3,128,0,127,-4,64,1,1,0,1" },
    { "11,2",   "Rhythmic Vibes" },
    { "11,2,0", "CoreRakarrack,Expander,-,-55,37,24,46,8050,80,1" },
    { "11,2,1", "CoreRakarrack,StompBox,-,44,64,-1,23,9,4" },
    { "11,2,2", "CoreRakarrack,VaryBand,-,0,140,0,32,80,4,0,465,1764,3309,1" },
    { "11,3",   "Clean Tremolo" },
    { "11,3,0", "CoreRakarrack,Convolotron,-,32,64,0,50,0,0,30,31,0,0,0" },
    { "11,3,1", "CoreRakarrack,Pan,-,0,64,340,0,1,64,0,1,0" },
    { "11,4",   "Double Tremolo" },
    { "11,4,0", "CoreRakarrack,Convolotron,-,32,64,0,50,0,0,18,16,0,0,0" },
    { "11,4,1", "CoreRakarrack,VaryBand,-,16,70,1,64,140,1,64,514,2324,3946,8" },
    { "11,4,2", "CoreRakarrack,Pan,-,32,64,350,0,5,64,96,1,1" },
    { "11,5",   "Stereo" },
    { "11,5,0", "CoreRakarrack,EQ,-,64,64,64,64,64,64,64,64,64,64,56,64" },
    { "11,5,1", "CoreRakarrack,Shifter,-,64,64,64,0,0,-20,1,0,1,22" },
    { "11,5,2", "CoreRakarrack,Exciter,-,64,0,0,0,0,0,0,-20,0,20,0,15608,100" },
    { "11,5,3", "CoreRakarrack,VaryBand,-,16,160,1,127,80,5,0,500,2500,5000,1" },
    { "11,5,4", "CoreRakarrack,Pan,-,32,64,26,0,0,0,96,0,1" },
    { "11,5,5", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,30,70,22,5,0,0,64,1,0,20000,10" },
    { "11,6",   "Circle Tremolo" },
    { "11,6,0", "CoreRakarrack,Parametric EQ,-,24,64,64,75,64,64,113,64,64,32" },
    { "11,6,1", "CoreRakarrack,Sequence,-,8,40,66,92,105,82,54,15,32,140,64,1,4,4,8" },
    { "11,6,2", "CoreRakarrack,VaryBand,-,32,70,5,0,280,1,127,400,1000,4162,0" },
    { "11,6,3", "CoreRakarrack,Pan,-,96,64,35,0,1,127,96,1,1" },
    { "11,7",   "Rhythm" },
    { "11,7,0", "CoreRakarrack,Shifter,-,64,64,64,165,165,-20,0,1,0,20" },
    { "11,7,1", "CoreRakarrack,Compressor,-,-30,2,-12,20,120,1,0,0,0" },
    { "11,7,2", "CoreRakarrack,Sequence,-,28,59,94,127,120,80,50,24,64,60,59,0,4,0,6" },
    { "11,7,3", "CoreRakarrack,MuTroMojo,-,64,1,60,0,6,64,32,34,96,96,16,-16,16,3,4000,400,1,0" },
    { "11,7,4", "CoreRakarrack,Pan,-,48,64,60,0,4,96,96,1,1" },
    { "11,8",   "Bright Size" },
    { "11,8,0", "CoreRakarrack,Compressor,-,-20,8,-16,20,120,1,0,0,0" },
    { "11,8,1", "CoreRakarrack,Sequence,-,0,21,33,59,62,89,106,127,106,540,64,0,0,5,6" },
    { "11,8,2", "CoreRakarrack,Pan,-,32,64,90,0,6,127,32,0,1" },
    { "11,8,3", "CoreRakarrack,Reverb,-,64,64,38,15,0,7342,747,90,0,80" },
    { "11,9",   "Metal" },
    { "11,9,0", "CoreRakarrack,Sustainer,-,48,100" },
    { "11,9,1", "CoreRakarrack,StompBox,-,48,2,2,2,8,6" },
    { "11,9,2", "CoreRakarrack,Sequence,-,9,35,60,67,67,62,46,16,80,100,88,0,4,0,8" },
    { "11,9,3", "CoreRakarrack,Shelf Boost,-,48,0,9600,0,48" },
    { "11,9,4", "CoreRakarrack,Reverb,-,64,64,18,8,0,4800,200,64,0,72" },
    { "12,0",   "Floyd Trip" },
    { "12,0,0", "CoreRakarrack,Compressor,-,-13,2,-10,20,120,1,0,0,0" },
    { "12,0,1", "CoreRakarrack,Sequence,-,0,21,33,59,62,89,106,127,67,480,64,0,3,5,6" },
    { "12,0,2", "CoreRakarrack,Synthfilter,-,0,5,60,0,0,0,53,-12,4,0,1,21,17,313,158,127" },
    { "12,0,3", "CoreRakarrack,Reverb,-,90,64,35,5,7,7342,170,64,0,83" },
    { "12,1",   "Dreamland" },
    { "12,1,0", "CoreRakarrack,EQ,-,64,64,65,59,60,65,67,70,67,64,64,37" },
    { "12,1,1", "CoreRakarrack,Sustainer,-,35,67" },
    { "12,1,2", "CoreRakarrack,Overdrive,-,0,64,64,2,74,28,0,9467,120,0,1,0,0" },
    { "12,1,3", "CoreRakarrack,Analog Phaser,-,67,0,120,50,1,32,23,34,10,27,0,99,1" },
    { "12,1,4", "CoreRakarrack,Echotron,-,93,64,127,4,0,99,3,41,0,96,-13,64,1,0,0,0" },
    { "12,2",   "Old Tavern" },
    { "12,2,0", "CoreRakarrack,Sustainer,-,8,91" },
    { "12,2,1", "CoreRakarrack,Analog Phaser,-,64,60,180,50,0,48,79,100,1,0,1,53,1" },
    { "12,2,2", "CoreRakarrack,Reverbtron,-,64,25,1,128,0,0,0,72,0,2,-29,68,1,0,20000,88" },
    { "12,3",   "Mild Clean" },
    { "12,3,0", "CoreRakarrack,Coil Crafter,-,32,5,3,2700,21,2900,18,179,1" },
    { "12,3,1", "CoreRakarrack,Sustainer,-,74,54" },
    { "12,3,2", "CoreRakarrack,Echotron,-,40,102,13,8,0,160,0,64,3,32,0,64,1,0,0,0" },
    { "12,3,3", "CoreRakarrack,Analog Phaser,-,83,15,10,50,0,0,79,33,2,0,1,53,1" },
    { "12,3,4", "CoreRakarrack,Reverbtron,-,81,65,0,316,0,0,0,32,3,0,-10,68,1,0,965,46" },
    { "12,4",   "Arppeg Sprinkler" },
    { "12,4,0", "CoreRakarrack,Coil Crafter,-,32,5,3,2700,21,2900,18,179,1" },
    { "12,4,1", "CoreRakarrack,Sustainer,-,74,54" },
    { "12,4,2", "CoreRakarrack,Arpie,-,67,64,35,64,30,59,0,127,0,0,0" },
    { "12,4,3", "CoreRakarrack,Analog Phaser,-,83,15,10,50,0,0,79,33,2,0,1,53,1" },
    { "12,4,4", "CoreRakarrack,Echotron,-,40,102,13,8,0,160,0,64,3,32,0,64,1,0,0,0" },
    { "12,4,5", "CoreRakarrack,Reverbtron,-,81,65,0,316,0,0,0,51,3,0,-10,68,1,0,965,46" },
    { "12,5",   "Backward Tumble" },
    { "12,5,0", "CoreRakarrack,Coil Crafter,-,32,5,3,2700,21,2900,18,179,0" },
    { "12,5,1", "CoreRakarrack,Sustainer,-,95,54" },
    { "12,5,2", "CoreRakarrack,DualFlange,-,-15,0,0,42,12,50,-10,1500,0,0,120,0,0,20" },
    { "12,5,3", "CoreRakarrack,Echoverse,-,60,64,70,64,64,16,39,127,0,0" },
    { "12,5,4", "CoreRakarrack,VaryBand,-,25,50,0,32,30,4,0,305,1764,3309,2" },
    { "12,5,5", "CoreRakarrack,Echotron,-,81,53,34,4,0,70,3,41,0,96,-13,64,1,0,1,1" },
    { "12,5,6", "CoreRakarrack,Reverbtron,-,81,65,0,316,0,0,0,51,3,0,-10,68,1,0,965,46" },
    { "12,6",   "Dig For Shift" },
    { "12,6,0", "CoreRakarrack,Sustainer,-,95,26" },
    { "12,6,1", "CoreRakarrack,Shifter,-,64,64,64,20,20,-5,5,0,0,0" },
    { "12,6,2", "CoreRakarrack,Echotron,-,60,49,46,8,0,59,0,64,3,0,0,64,1,0,0,0" },
    { "12,6,3", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,99,21,0,0,0,64,0,0,20000,0" },
    { "12,7",   "Diode Overdose" },
    { "12,7,0", "CoreRakarrack,Distorsion,-,16,64,64,23,43,29,0,4744,720,0,1,0,0" },
    { "12,7,1", "CoreRakarrack,Overdrive,-,25,64,64,0,58,29,0,722,20,0,0,0,0" },
    { "12,7,2", "CoreRakarrack,Echoverse,-,109,64,107,18,30,67,25,0,0,81" },
    { "12,8",   "Warm Chorus" },
    { "12,8,0", "CoreRakarrack,Coil Crafter,-,44,5,2,3153,34,3464,51,179,0" },
    { "12,8,1", "CoreRakarrack,Sustainer,-,104,38" },
    { "12,8,2", "CoreRakarrack,Analog Phaser,-,83,15,10,50,0,0,79,33,2,0,1,53,1" },
    { "12,8,3", "CoreRakarrack,DualFlange,-,-15,0,0,46,14,50,-10,1500,0,0,120,0,0,20" },
    { "12,8,4", "CoreRakarrack,Shuffle,-,30,36,-13,-13,36,60,858,2436,6000,-24,1" },
    { "12,8,5", "CoreRakarrack,Reverbtron,-,86,37,0,881,0,0,0,17,3,0,-25,68,1,0,965,127" },
    { "12,9",   "SeqDist" },
    { "12,9,0", "CoreRakarrack,Sustainer,-,30,38" },
    { "12,9,1", "CoreRakarrack,StompBox,-,32,64,0,0,110,4" },
    { "12,9,2", "CoreRakarrack,Sequence,-,0,0,62,0,51,21,5,64,0,219,95,1,3,4,2" },
    { "12,9,3", "CoreRakarrack,Vocoder,-,0,64,31,73,97,95,16" },
    { "12,9,4", "CoreRakarrack,Reverbtron,-,79,0,1,128,0,0,99,30,0,0,0,64,0,0,20000,58" },
    { "13,0",   "Music" },
    { "13,0,0", "CoreRakarrack,CompBand,-,64,8,4,4,4,-21,16,16,-21,100,1063,7189,64" },
    { "13,0,1", "CoreRakarrack,Compressor,-,-60,2,0,10,500,1,0,1,1" },
    { "13,1",   "C Melody" },
    { "13,1,0", "CoreRakarrack,Ring,-,50,0,-64,127,38,698,1,95,0,73,0,85,1" },
    { "13,1,1", "CoreRakarrack,Harmonizer,-,64,64,64,17,64,1,0,0,64,64,0" },
    { "13,1,2", "CoreRakarrack,CompBand,-,64,8,2,2,4,6,10,9,-12,691,1063,7189,64" },
    { "13,1,3", "CoreRakarrack,Reverb,-,96,64,18,16,0,6218,220,78,1,83" },
    { "13,1,4", "CoreRakarrack,Pan,-,32,64,50,0,0,0,96,0,1" },
    { "13,2",   "Surf's Up" },
    { "13,2,0", "CoreRakarrack,CompBand,-,26,2,4,3,4,-32,24,-28,-14,383,1000,5055,55" },
    { "13,2,1", "CoreRakarrack,OpticalTrem,-,60,217,10,0,64,64" },
    { "13,2,2", "CoreRakarrack,Overdrive,-,0,64,64,2,75,28,0,4255,21,1,0,0,0" },
    { "13,2,3", "CoreRakarrack,Chorus,-,32,64,120,0,8,90,31,85,64,0,0,1" },
    { "13,2,4", "CoreRakarrack,Convolotron,-,0,64,0,90,0,0,0,85,5,0,0" },
    { "13,2,5", "CoreRakarrack,Reverbtron,-,100,0,1,155,0,30,0,0,9,0,0,64,1,0,2854,60" },
    { "13,3",   "Live Punk Show" },
    { "13,3,0", "CoreRakarrack,Expander,-,-70,33,18,46,8050,80,1" },
    { "13,3,1", "CoreRakarrack,Distorsion,-,49,64,0,30,44,29,0,3134,680,0,1,0,0" },
    { "13,3,2", "CoreRakarrack,Overdrive,-,0,64,64,59,37,28,0,4255,150,1,1,0,0" },
    { "13,3,3", "CoreRakarrack,Convolotron,-,0,75,0,93,0,0,55,77,2,0,0" },
    { "13,3,4", "CoreRakarrack,Reverbtron,-,63,0,1,155,0,30,0,0,9,0,0,64,1,0,2854,60" },
    { "13,4",   "Charred Violets" },
    { "13,4,0", "CoreRakarrack,Expander,-,-71,31,50,50,8050,76,1" },
    { "13,4,1", "CoreRakarrack,Vibe,-,35,240,10,0,0,64,64,64,3,64" },
    { "13,4,2", "CoreRakarrack,StompBox,-,17,-47,5,-26,127,2" },
    { "13,4,3", "CoreRakarrack,Echotron,-,62,78,37,5,0,60,52,64,4,96,-2,64,1,0,0,1" },
    { "13,5",   "Stereo Ecstasy" },
    { "13,5,0", "CoreRakarrack,CompBand,-,26,2,4,6,4,-34,24,-28,-11,431,1000,5055,55" },
    { "13,5,1", "CoreRakarrack,Overdrive,-,0,64,64,2,75,28,0,4255,21,1,0,0,0" },
    { "13,5,2", "CoreRakarrack,Echotron,-,64,36,93,6,0,45,0,64,8,32,0,64,0,0,1,1" },
    { "13,5,3", "CoreRakarrack,DualFlange,-,-15,0,0,42,12,50,-10,1500,0,0,105,0,8,20" },
    { "13,5,4", "CoreRakarrack,VaryBand,-,0,80,0,64,40,0,0,120,1000,2300,1" },
    { "13,5,5", "CoreRakarrack,Reverbtron,-,100,0,1,155,0,30,0,0,9,0,0,64,1,0,2854,60" },
    { "13,6",   "Insanity Mojo" },
    { "13,6,0", "CoreRakarrack,Expander,-,-70,33,18,46,8050,80,1" },
    { "13,6,1", "CoreRakarrack,MuTroMojo,-,0,22,138,0,0,64,0,36,0,20,32,0,5,1,3735,60,0,1" },
    { "13,6,2", "CoreRakarrack,Overdrive,-,0,64,64,59,37,28,0,4255,150,1,1,0,0" },
    { "13,7",   "Random Eyes" },
    { "13,7,0", "CoreRakarrack,Expander,-,-70,33,18,46,8050,80,1" },
    { "13,7,1", "CoreRakarrack,Distorsion,-,49,64,0,30,44,29,0,3134,680,0,1,0,0" },
    { "13,7,2", "CoreRakarrack,Overdrive,-,0,64,64,67,37,28,0,2943,20,1,0,0,0" },
    { "13,7,3", "CoreRakarrack,MuTroMojo,-,0,12,145,0,9,64,59,5,15,40,32,0,-2,3,3125,60,0,1" },
    { "13,7,4", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,53,25,0,0,0,64,0,0,20000,0" },
    { "13,7,5", "CoreRakarrack,Parametric EQ,-,153,75,47,1689,57,25,8927,53,40,64" },
    { "13,8",   "Goin through a phase" },
    { "13,8,0", "CoreRakarrack,Expander,-,-70,33,18,46,8050,80,1" },
    { "13,8,1", "CoreRakarrack,Distorsion,-,49,64,0,30,44,29,0,3134,680,0,1,0,0" },
    { "13,8,2", "CoreRakarrack,Overdrive,-,0,64,64,67,37,28,0,2943,20,1,0,0,0" },
    { "13,8,3", "CoreRakarrack,Analog Phaser,-,64,20,145,50,9,64,92,40,6,0,1,58,1" },
    { "13,8,4", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,53,25,0,0,0,64,0,0,20000,0" },
    { "13,8,5", "CoreRakarrack,EQ,-,64,64,65,59,60,65,67,70,67,64,64,37" },
    { "13,8,6", "CoreRakarrack,Parametric EQ,-,153,75,47,1689,61,25,12829,58,40,64" },
    { "13,9",   "Suction II" },
    { "13,9,0", "CoreRakarrack,Expander,-,-64,22,50,50,1200,70,0" },
    { "13,9,1", "CoreRakarrack,Sustainer,-,75,73" },
    { "13,9,2", "CoreRakarrack,Overdrive,-,0,64,35,4,51,28,0,2382,21,1,0,0,0" },
    { "13,9,3", "CoreRakarrack,Echotron,-,32,46,37,8,0,92,0,64,3,75,0,64,1,1,0,1" },
    { "13,9,4", "CoreRakarrack,Reverbtron,-,104,0,1,400,0,0,60,15,2,0,0,64,0,0,20000,20" },
    { "13,9,5", "CoreRakarrack,Reverb,-,48,64,13,18,0,4002,27,83,0,64" },
    { "14,0",   "Multichorus" },
    { "14,0,0", "CoreRakarrack,Sustainer,-,35,113" },
    { "14,0,1", "CoreRakarrack,Echotron,-,79,64,42,7,0,60,0,61,7,32,0,64,1,1,0,1" },
    { "14,1",   "Clean Twirl" },
    { "14,1,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "14,1,1", "CoreRakarrack,Sustainer,-,39,91" },
    { "14,1,2", "CoreRakarrack,Analog Phaser,-,64,20,32,50,1,64,93,40,6,48,0,58,1" },
    { "14,1,3", "CoreRakarrack,Echoverse,-,111,64,90,64,64,18,23,0,1,82" },
    { "14,1,4", "CoreRakarrack,Reverb,-,116,64,62,24,0,3134,102,75,1,49" },
    { "14,2",   "TriadEchoes" },
    { "14,2,0", "CoreRakarrack,Expander,-,-64,22,50,50,8050,76,0" },
    { "14,2,1", "CoreRakarrack,StompBox,-,42,-20,0,0,70,2" },
    { "14,2,2", "CoreRakarrack,Echotron,-,98,72,37,3,0,73,3,47,0,96,-6,0,1,0,0,1" },
    { "14,2,3", "CoreRakarrack,Echo,-,51,127,500,64,64,35,10,0,0" },
    { "14,3",   "GibSoonMur" },
    { "14,3,0", "CoreRakarrack,Expander,-,-55,30,50,50,1441,157,50" },
    { "14,3,1", "CoreRakarrack,Compressor,-,-26,4,-8,20,120,1,10,1,1" },
    { "14,3,2", "CoreRakarrack,Coil Crafter,-,32,1,6,4400,42,3300,17,20,0" },
    { "14,3,3", "CoreRakarrack,Echotron,-,20,65,34,2,0,96,3,41,0,96,-13,64,1,1,1,0" },
    { "14,3,4", "CoreRakarrack,Parametric EQ,-,72,71,38,1077,64,64,10580,69,38,64" },
    { "14,4",   "GibSoonMurDouble" },
    { "14,4,0", "CoreRakarrack,Expander,-,-55,30,50,50,1441,157,50" },
    { "14,4,1", "CoreRakarrack,Compressor,-,-26,4,-8,20,120,1,10,0,0" },
    { "14,4,2", "CoreRakarrack,Coil Crafter,-,10,1,6,4400,42,3300,17,20,1" },
    { "14,4,3", "CoreRakarrack,Echotron,-,20,65,34,2,0,63,3,41,0,96,-13,64,1,1,1,0" },
    { "14,4,4", "CoreRakarrack,Parametric EQ,-,72,71,38,1077,64,64,10580,69,38,64" },
    { "14,5",   "GibSoonMurder" },
    { "14,5,0", "CoreRakarrack,Expander,-,-55,30,50,50,1441,157,50" },
    { "14,5,1", "CoreRakarrack,Compressor,-,-26,4,-8,20,120,1,10,1,1" },
    { "14,5,2", "CoreRakarrack,Coil Crafter,-,10,1,6,4400,42,3300,17,20,0" },
    { "14,5,3", "CoreRakarrack,Echotron,-,20,65,34,4,0,96,3,41,0,96,-13,64,1,1,4,0" },
    { "14,5,4", "CoreRakarrack,Parametric EQ,-,72,71,38,1077,64,64,10580,69,38,64" },
    { "14,6",   "Igniter" },
    { "14,6,0", "CoreRakarrack,Expander,-,-51,50,59,46,8050,76,1" },
    { "14,6,1", "CoreRakarrack,StompBox,-,38,6,6,6,105,1" },
    { "14,6,2", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,62,0,0,0" },
    { "14,6,3", "CoreRakarrack,Reverbtron,-,64,0,0,327,0,0,0,0,7,0,0,64,0,0,1585,12" },
    { "14,7",   "Ignited" },
    { "14,7,0", "CoreRakarrack,Expander,-,-65,15,10,300,8000,20,30" },
    { "14,7,1", "CoreRakarrack,StompBox,-,60,0,-12,-9,85,6" },
    { "14,7,2", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,74,3,0,0" },
    { "14,7,3", "CoreRakarrack,Overdrive,-,0,64,64,0,79,28,1,10884,170,1,1,0,0" },
    { "14,7,4", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,0,64,0,0,20000,0" },
    { "14,8",   "BlowMoeJo" },
    { "14,8,0", "CoreRakarrack,EQ,-,71,68,64,64,64,64,64,64,66,69,64,40" },
    { "14,8,1", "CoreRakarrack,Compressor,-,-30,2,-13,20,120,1,0,0,0" },
    { "14,8,2", "CoreRakarrack,OpticalTrem,-,127,480,0,3,32,64" },
    { "14,8,3", "CoreRakarrack,Overdrive,-,84,64,35,56,40,0,0,7600,21,0,1,0,0" },
    { "14,8,4", "CoreRakarrack,MuTroMojo,-,16,10,60,0,1,64,0,29,0,0,-16,40,-3,1,2000,450,1,1" },
    { "14,9",   "JoeMoeJo" },
    { "14,9,0", "CoreRakarrack,EQ,-,71,68,64,64,64,64,64,64,66,69,64,40" },
    { "14,9,1", "CoreRakarrack,Compressor,-,-30,2,-12,20,120,1,0,0,0" },
    { "14,9,2", "CoreRakarrack,MuTroMojo,-,0,15,138,0,0,64,0,39,0,30,32,0,5,1,2000,60,0,1" },
    { "15,0",   "Gary Moded" },
    { "15,0,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,45,26,0,89,68" },
    { "15,0,1", "CoreRakarrack,Parametric EQ,-,55,28,72,704,65,104,5896,52,64,92" },
    { "15,0,2", "CoreRakarrack,Sustainer,-,21,127" },
    { "15,0,3", "CoreRakarrack,Overdrive,-,84,64,35,56,40,0,0,7600,21,0,1,0,0" },
    { "15,0,4", "CoreRakarrack,Echo,-,0,64,272,64,64,35,0,1,0" },
    { "15,1",   "Gary Loaded" },
    { "15,1,0", "CoreRakarrack,Expander,-,-45,50,59,250,8050,76,1" },
    { "15,1,1", "CoreRakarrack,Sustainer,-,16,127" },
    { "15,1,2", "CoreRakarrack,Parametric EQ,-,55,28,72,704,65,104,5896,52,64,90" },
    { "15,1,3", "CoreRakarrack,Overdrive,-,84,64,35,56,40,0,0,7600,21,0,1,0,0" },
    { "15,1,4", "CoreRakarrack,Echo,-,37,64,565,75,30,59,1,0,0" },
    { "15,2",   "Oxygen" },
    { "15,2,0", "CoreRakarrack,Vibe,-,75,330,10,0,50,64,0,64,17,64" },
    { "15,2,1", "CoreRakarrack,Reverb,-,48,64,13,18,0,4002,27,83,0,64" },
    { "15,3",   "Oxygenated" },
    { "15,3,0", "CoreRakarrack,EQ,-,18,40,53,58,68,66,61,45,26,0,89,68" },
    { "15,3,1", "CoreRakarrack,Overdrive,-,0,64,35,110,23,13,0,2982,21,0,0,0,0" },
    { "15,3,2", "CoreRakarrack,Sustainer,-,16,127" },
    { "15,3,3", "CoreRakarrack,Parametric EQ,-,55,28,72,704,65,104,5896,52,64,92" },
    { "15,3,4", "CoreRakarrack,Echo,-,0,64,488,64,64,35,0,1,0" },
    { "15,4",   "StartButton" },
    { "15,4,0", "CoreRakarrack,Expander,-,-51,50,59,46,8050,76,1" },
    { "15,4,1", "CoreRakarrack,StompBox,-,72,6,6,6,105,6" },
    { "15,4,2", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,53,0,0,0" },
    { "15,4,3", "CoreRakarrack,Overdrive,-,0,64,64,0,79,28,1,10884,170,1,1,0,0" },
    { "15,5",   "MetalAmploaded" },
    { "15,5,0", "CoreRakarrack,Expander,-,-51,50,59,46,8050,76,1" },
    { "15,5,1", "CoreRakarrack,StompBox,-,48,0,-32,-36,53,6" },
    { "15,5,2", "CoreRakarrack,Convolotron,-,0,64,0,76,0,0,0,70,0,0,0" },
    { "15,5,3", "CoreRakarrack,Reverbtron,-,64,0,0,327,0,0,0,0,7,0,0,64,0,0,1585,12" },
    { "15,6",   "MetalAmpReloaded" },
    { "15,6,0", "CoreRakarrack,Expander,-,-55,30,50,50,1441,157,50" },
    { "15,6,1", "CoreRakarrack,StompBox,-,74,0,-12,0,127,6" },
    { "15,6,2", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,30,34,6,0,0,64,0,0,20000,0" },
    { "15,6,3", "CoreRakarrack,Convolotron,-,67,64,0,100,0,0,30,20,8,0,0" },
    { "15,7",   "MetalAmpOverloaded" },
    { "15,7,0", "CoreRakarrack,Expander,-,-65,31,10,50,1441,157,50" },
    { "15,7,1", "CoreRakarrack,Overdrive,-,18,64,35,25,48,29,1,2854,336,0,1,0,0" },
    { "15,7,2", "CoreRakarrack,StompBox,-,90,-8,-17,0,127,6" },
    { "15,7,3", "CoreRakarrack,Convolotron,-,0,64,0,100,0,0,25,60,8,0,0" },
    { "15,7,4", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,30,34,6,0,0,64,0,0,20000,0" },
    { "15,8",   "Melon Collie" },
    { "15,8,0", "CoreRakarrack,Sustainer,-,88,67" },
    { "15,8,1", "CoreRakarrack,Vibe,-,67,260,10,0,48,70,20,35,0,48" },
    { "15,8,2", "CoreRakarrack,OpticalTrem,-,51,130,10,0,80,64" },
    { "15,8,3", "CoreRakarrack,Reverbtron,-,64,25,1,128,0,0,0,72,0,2,-29,68,1,0,20000,88" },
    { "15,9",   "Melon Collie II" },
    { "15,9,0", "CoreRakarrack,Sustainer,-,88,67" },
    { "15,9,1", "CoreRakarrack,Vibe,-,67,260,10,0,48,70,20,35,0,48" },
    { "15,9,2", "CoreRakarrack,OpticalTrem,-,35,265,10,0,80,64" },
    { "15,9,3", "CoreRakarrack,Reverbtron,-,64,25,1,24,0,0,0,72,0,2,-29,68,1,0,20000,88" },
    { "16,0",   "Sounds on Sound" },
    { "16,0,0", "CoreRakarrack,Vibe,-,51,12,10,9,63,64,0,14,0,30" },
    { "16,0,1", "CoreRakarrack,Reverbtron,-,63,0,1,20,0,0,0,53,8,27,0,68,1,0,492,6" },
    { "16,1",   "Voy Pa Alla" },
    { "16,1,0", "CoreRakarrack,Sequence,-,11,55,15,70,12,60,11,36,79,70,92,0,4,1,8" },
    { "16,1,1", "CoreRakarrack,OpticalTrem,-,80,70,0,6,0,64" },
    { "16,1,2", "CoreRakarrack,WahWah,-,64,64,70,0,0,127,64,0,0,60" },
    { "16,1,3", "CoreRakarrack,Pan,-,32,64,50,0,0,0,97,0,1" },
    { "16,1,4", "CoreRakarrack,Reverb,-,80,64,63,24,0,85,5,83,1,64" },
    { "16,2",   "JimiVibe" },
    { "16,2,0", "CoreRakarrack,Vibe,-,46,120,10,0,48,64,0,64,3,64" },
    { "16,2,1", "CoreRakarrack,Valve,-,9,64,64,120,37,0,5688,483,1,1,106,0,100" },
    { "16,2,2", "CoreRakarrack,StompBox,-,48,1,3,17,0,7" },
    { "16,2,3", "CoreRakarrack,Reverbtron,-,64,16,1,208,0,0,0,42,3,-17,0,64,0,0,20000,0" },
    { "16,3",   "JimiVibe2" },
    { "16,3,0", "CoreRakarrack,Vibe,-,60,197,10,0,48,64,64,64,0,64" },
    { "16,3,1", "CoreRakarrack,Valve,-,9,64,64,120,37,0,5688,273,1,1,72,0,78" },
    { "16,3,2", "CoreRakarrack,StompBox,-,48,1,3,-34,0,7" },
    { "16,3,3", "CoreRakarrack,Reverbtron,-,64,16,1,208,0,0,0,42,3,-17,0,64,0,0,20000,0" },
    { "16,4",   "Sounds and-a-log" },
    { "16,4,0", "CoreRakarrack,Sustainer,-,60,22" },
    { "16,4,1", "CoreRakarrack,Vibe,-,110,75,10,0,32,64,64,14,0,30" },
    { "16,4,2", "CoreRakarrack,Valve,-,60,64,64,111,62,0,5841,747,1,1,28,0,91" },
    { "16,4,3", "CoreRakarrack,Echotron,-,64,65,34,3,0,76,40,65,0,96,-38,64,1,1,8,0" },
    { "16,4,4", "CoreRakarrack,Reverbtron,-,64,25,1,128,0,0,0,72,0,2,-29,68,1,0,20000,88" },
    { "16,5",   "Envelope YaYa" },
    { "16,5,0", "CoreRakarrack,EQ,-,64,62,64,64,68,70,72,74,66,69,64,40" },
    { "16,5,1", "CoreRakarrack,MuTroMojo,-,0,31,138,0,0,64,0,24,0,28,29,0,0,2,5455,60,0,1" },
    { "16,5,2", "CoreRakarrack,Ring,-,-64,0,0,65,100,1820,1,7,36,0,35,81,0" },
    { "16,5,3", "CoreRakarrack,Parametric EQ,-,145,74,30,1077,64,64,12182,51,25,64" },
    { "16,6",   "AutoYa" },
    { "16,6,0", "CoreRakarrack,EQ,-,64,62,64,64,68,70,65,69,66,69,64,64" },
    { "16,6,1", "CoreRakarrack,Compressor,-,-30,2,-12,20,120,1,0,0,0" },
    { "16,6,2", "CoreRakarrack,MuTroMojo,-,0,31,120,0,0,64,28,-5,46,42,29,0,0,2,5455,60,0,1" },
    { "16,6,3", "CoreRakarrack,Ring,-,-50,0,0,76,100,2320,1,7,36,0,35,81,0" },
    { "16,6,4", "CoreRakarrack,Reverb,-,97,64,63,24,0,85,5,83,1,64" },
    { "16,6,5", "CoreRakarrack,Parametric EQ,-,145,70,30,1077,64,64,12182,58,25,64" },
    { "16,7",   "AutoYiYa" },
    { "16,7,0", "CoreRakarrack,EQ,-,64,62,64,64,68,70,72,74,66,69,64,40" },
    { "16,7,1", "CoreRakarrack,MuTroMojo,-,0,31,45,0,4,64,23,-4,62,28,29,0,0,2,5455,60,0,1" },
    { "16,7,2", "CoreRakarrack,Ring,-,-64,0,0,65,100,1820,1,7,36,0,35,81,0" },
    { "16,7,3", "CoreRakarrack,Parametric EQ,-,145,74,30,1077,64,64,12182,51,25,64" },
    { "16,8",   "RandYa" },
    { "16,8,0", "CoreRakarrack,EQ,-,64,62,64,64,68,70,65,69,66,69,64,64" },
    { "16,8,1", "CoreRakarrack,Compressor,-,-30,2,-12,20,120,1,0,0,0" },
    { "16,8,2", "CoreRakarrack,MuTroMojo,-,0,31,120,0,9,64,31,-5,46,42,29,0,0,2,5455,60,0,1" },
    { "16,8,3", "CoreRakarrack,Ring,-,-52,0,0,76,100,2320,1,7,36,0,35,81,0" },
    { "16,8,4", "CoreRakarrack,Reverb,-,97,64,63,24,0,85,5,83,1,64" },
    { "16,8,5", "CoreRakarrack,Parametric EQ,-,145,70,30,1077,64,64,12182,58,25,64" },
    { "16,9",   "Talk to me" },
    { "16,9,0", "CoreRakarrack,EQ,-,64,62,64,64,68,70,65,69,66,69,64,64" },
    { "16,9,1", "CoreRakarrack,Compressor,-,-31,6,-10,20,120,1,22,0,1" },
    { "16,9,2", "CoreRakarrack,MuTroMojo,-,0,31,208,0,8,64,37,-6,28,42,29,0,0,2,5455,60,0,1" },
    { "16,9,3", "CoreRakarrack,Ring,-,-45,0,0,67,100,2550,1,7,36,0,35,81,0" },
    { "16,9,4", "CoreRakarrack,Reverb,-,97,64,63,24,0,85,5,83,1,64" },
    { "16,9,5", "CoreRakarrack,Parametric EQ,-,145,70,30,1077,64,64,12182,58,25,64" },
    { "17,0",   "Helicopter" },
    { "17,0,0", "CoreRakarrack,Expander,-,-80,37,24,46,8050,76,1" },
    { "17,0,1", "CoreRakarrack,StompBox,-,55,27,10,1,18,0" },
    { "17,0,2", "CoreRakarrack,OpticalTrem,-,127,556,0,2,64,64" },
    { "17,0,3", "CoreRakarrack,Valve,-,0,64,64,30,67,0,3327,20,1,1,46,1,67" },
    { "17,0,4", "CoreRakarrack,VaryBand,-,32,3,0,53,6,0,72,269,1509,3309,2" },
    { "17,0,5", "CoreRakarrack,DualFlange,-,0,0,0,107,2291,98,3,12371,0,0,3,64,1,10" },
    { "17,1",   "Octaflange" },
    { "17,1,0", "CoreRakarrack,EQ,-,78,71,64,64,64,64,64,64,71,80,64,42" },
    { "17,1,1", "CoreRakarrack,Flanger,-,64,64,14,0,1,62,12,19,97,0,0,0" },
    { "17,1,2", "CoreRakarrack,Harmonizer,-,64,64,64,0,6000,0,0,0,64,64,0" },
    { "17,1,3", "CoreRakarrack,Compressor,-,-30,4,-6,20,120,1,100,1,1" },
    { "17,2",   "Octaflange - Dirty" },
    { "17,2,0", "CoreRakarrack,Expander,-,-65,35,10,117,6633,20,1" },
    { "17,2,1", "CoreRakarrack,Flanger,-,64,64,14,0,1,62,12,19,97,0,0,0" },
    { "17,2,2", "CoreRakarrack,Harmonizer,-,64,64,64,0,6000,0,0,0,64,64,0" },
    { "17,2,3", "CoreRakarrack,Valve,-,9,64,64,80,64,1,5688,220,1,1,58,1,30" },
    { "17,2,4", "CoreRakarrack,Compressor,-,-30,4,-6,20,120,1,100,1,1" },
    { "17,2,5", "CoreRakarrack,EQ,-,69,71,71,64,65,67,72,65,58,40,64,55" },
    { "17,3",   "OctaVibe - Dirty Echo" },
    { "17,3,0", "CoreRakarrack,Expander,-,-65,35,10,117,6633,20,1" },
    { "17,3,1", "CoreRakarrack,Exciter,-,64,64,-62,0,0,64,0,-45,0,64,0,20000,5106" },
    { "17,3,2", "CoreRakarrack,Vibe,-,46,132,10,0,48,64,23,98,3,86" },
    { "17,3,3", "CoreRakarrack,Harmonizer,-,64,64,64,0,6000,0,0,0,64,85,0" },
    { "17,3,4", "CoreRakarrack,Valve,-,9,64,64,80,64,1,5688,220,1,1,58,1,30" },
    { "17,3,5", "CoreRakarrack,EQ,-,69,71,71,64,65,67,72,65,58,40,64,55" },
    { "17,3,6", "CoreRakarrack,Echoverse,-,64,64,90,64,64,64,64,0,2,96" },
    { "17,4",   "Dirty Junk" },
    { "17,4,0", "CoreRakarrack,EQ,-,64,64,64,64,65,72,64,64,64,64,64,30" },
    { "17,4,1", "CoreRakarrack,Compressor,-,-15,3,0,106,120,1,71,0,1" },
    { "17,4,2", "CoreRakarrack,DistBand,-,14,64,0,65,53,26,28,26,38,45,35,0,341,1004,0" },
    { "17,4,3", "CoreRakarrack,Distorsion,-,0,64,35,7,60,28,0,3327,143,0,1,0,0" },
    { "17,5",   "Jazz Hall" },
    { "17,5,0", "CoreRakarrack,EQ,-,64,64,66,64,61,68,67,64,64,64,64,30" },
    { "17,5,1", "CoreRakarrack,Compressor,-,-11,6,0,10,10,0,71,0,0" },
    { "17,5,2", "CoreRakarrack,Distorsion,-,55,64,64,12,55,28,1,3327,706,0,1,0,0" },
    { "17,5,3", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "17,5,4", "CoreRakarrack,Cabinet,-,8,58" },
    { "17,6",   "Jazz Bright Chorus" },
    { "17,6,0", "CoreRakarrack,EQ,-,64,67,66,64,61,68,67,64,64,64,64,12" },
    { "17,6,1", "CoreRakarrack,DualFlange,-,-13,0,0,37,6,0,17,4743,0,0,88,20,0,9" },
    { "17,6,2", "CoreRakarrack,Compressor,-,-11,6,0,10,10,0,71,0,0" },
    { "17,6,3", "CoreRakarrack,Distorsion,-,55,64,64,12,55,28,1,3327,706,0,1,0,0" },
    { "17,6,4", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "17,6,5", "CoreRakarrack,Cabinet,-,4,58" },
    { "17,7",   "Trippy Harmo" },
    { "17,7,0", "CoreRakarrack,Shifter,-,0,64,64,900,200,-11,3,1,0,0" },
    { "17,7,1", "CoreRakarrack,Synthfilter,-,20,69,88,0,6,0,76,-50,6,2,1,0,19,114,221,127" },
    { "17,8",   "Mutation" },
    { "17,8,0", "CoreRakarrack,Sustainer,-,92,58" },
    { "17,8,1", "CoreRakarrack,OpticalTrem,-,127,120,10,5,0,64" },
    { "17,8,2", "CoreRakarrack,Synthfilter,-,0,20,60,0,0,95,122,-40,4,4,0,32,-32,500,100,0" },
    { "17,9",   "Jazz Vibrato" },
    { "17,9,0", "CoreRakarrack,EQ,-,64,67,66,64,61,68,67,64,64,64,64,12" },
    { "17,9,1", "CoreRakarrack,OpticalTrem,-,0,251,10,1,64,64" },
    { "17,9,2", "CoreRakarrack,Compressor,-,-11,6,0,10,10,0,71,0,0" },
    { "17,9,3", "CoreRakarrack,Distorsion,-,55,64,64,12,55,28,1,3327,706,0,1,0,0" },
    { "17,9,4", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "17,9,5", "CoreRakarrack,Cabinet,-,4,58" },
    { "18,0",   "Reverse delay ambience" },
    { "18,0,0", "CoreRakarrack,EQ,-,64,67,66,64,61,68,67,64,64,64,64,12" },
    { "18,0,1", "CoreRakarrack,Compressor,-,-29,10,-15,45,55,1,100,0,0" },
    { "18,0,2", "CoreRakarrack,DualFlange,-,-13,0,0,37,6,85,17,4743,0,0,88,20,0,9" },
    { "18,0,3", "CoreRakarrack,Echoverse,-,69,64,12,64,98,76,23,127,0,96" },
    { "18,0,4", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "18,1",   "PongChor" },
    { "18,1,0", "CoreRakarrack,EQ,-,64,67,66,64,61,68,67,64,64,64,64,12" },
    { "18,1,1", "CoreRakarrack,Compressor,-,-29,17,-15,45,55,1,100,0,0" },
    { "18,1,2", "CoreRakarrack,DualFlange,-,-13,0,0,37,6,85,17,4743,0,0,88,20,8,9" },
    { "18,1,3", "CoreRakarrack,Echoverse,-,69,64,12,42,98,76,23,0,3,96" },
    { "18,1,4", "CoreRakarrack,Echotron,-,64,61,44,7,0,81,0,64,7,32,0,64,1,1,8,1" },
    { "18,1,5", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "18,2",   "Pitch Vibrato" },
    { "18,2,0", "CoreRakarrack,EQ,-,64,67,66,64,61,68,67,64,64,64,64,12" },
    { "18,2,1", "CoreRakarrack,Vibe,-,51,260,10,0,83,70,0,88,32,63" },
    { "18,2,2", "CoreRakarrack,Compressor,-,-11,6,0,10,10,0,71,0,0" },
    { "18,2,3", "CoreRakarrack,Distorsion,-,55,64,64,12,55,28,1,3327,706,0,1,0,0" },
    { "18,2,4", "CoreRakarrack,Reverbtron,-,88,0,1,400,0,0,88,14,5,0,64,64,1,0,5216,0" },
    { "18,3",   "Violin" },
    { "18,3,0", "CoreRakarrack,Parametric EQ,-,180,79,35,1716,75,58,8182,73,48,64" },
    { "18,3,1", "CoreRakarrack,EQ,-,23,14,48,58,69,60,53,51,42,53,76,60" },
    { "18,3,2", "CoreRakarrack,Flanger,-,64,64,96,18,0,0,9,0,16,64,0,0" },
    { "18,3,3", "CoreRakarrack,Chorus,-,64,64,34,0,0,90,40,85,64,119,0,0" },
    { "18,3,4", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,48,20,4,0,0,64,0,0,20000,0" },
    { "18,4",   "Violin Distortion" },
    { "18,4,0", "CoreRakarrack,Distorsion,-,0,64,127,42,24,0,1,3134,104,0,0,0,25" },
    { "18,4,1", "CoreRakarrack,Parametric EQ,-,180,72,35,1716,55,53,8182,37,60,64" },
    { "18,4,2", "CoreRakarrack,Flanger,-,64,64,99,18,0,0,9,0,16,64,0,0" },
    { "18,4,3", "CoreRakarrack,Chorus,-,85,63,1,0,0,73,120,86,64,127,0,0" },
    { "18,4,4", "CoreRakarrack,Reverbtron,-,64,0,1,400,0,0,48,20,4,0,0,64,0,0,20000,0" },
    { "18,5",   "Delayed Chorus" },
    { "18,5,0", "CoreRakarrack,CompBand,-,48,16,42,42,16,-16,-10,-11,-21,120,1000,5762,64" },
    { "18,5,1", "CoreRakarrack,Musical Delay,-,32,0,1,7,64,68,0,127,2,68,120,103,107" },
    { "18,5,2", "CoreRakarrack,StereoHarm,-,48,64,12,280,64,12,-280,0,0,0,0,64" },
    { "18,5,3", "CoreRakarrack,Compressor,-,-60,24,0,10,500,1,48,1,1" },
    { "18,6",   "Phasetron" },
    { "18,6,0", "CoreRakarrack,Phaser,-,0,64,6,0,4,0,100,16,1,64,1,21" },
    { "18,6,1", "CoreRakarrack,StereoHarm,-,64,64,12,280,64,12,-280,0,0,0,0,64" },
    { "18,6,2", "CoreRakarrack,Echotron,-,64,47,28,8,0,12,0,64,3,0,0,64,1,1,0,1" },
    { "18,6,3", "CoreRakarrack,Pan,-,64,64,48,0,0,0,64,1,1" },
    { "18,7",   "JZ Power" },
    { "18,7,0", "CoreRakarrack,Sequence,-,30,127,30,50,80,40,110,80,0,240,95,1,1,2,2" },
    { "18,7,1", "CoreRakarrack,MuTroMojo,-,16,10,60,0,0,64,0,40,10,32,-16,40,-3,1,2000,400,1,1" },
    { "18,7,2", "CoreRakarrack,Musical Delay,-,64,0,2,7,0,59,0,127,4,59,120,75,75" },
    { "18,7,3", "CoreRakarrack,Compressor,-,-4,2,0,20,411,1,0,1,1" },
    { "18,8",   "Trigger Chorus" },
    { "18,8,0", "CoreRakarrack,Shuffle,-,64,13,19,-21,22,350,1200,2000,6000,-21,1" },
    { "18,8,1", "CoreRakarrack,Shifter,-,64,64,64,250,100,-10,0,0,0,25" },
    { "18,8,2", "CoreRakarrack,Echo,-,67,75,955,64,30,59,10,0,0" },
    { "18,9",   "EchoPhase" },
    { "18,9,0", "CoreRakarrack,Echo,-,81,60,472,100,127,67,36,0,0" },
    { "18,9,1", "CoreRakarrack,Phaser,-,64,64,10,0,0,88,40,64,3,0,0,20" },
    { "18,9,2", "CoreRakarrack,StereoHarm,-,64,64,12,280,64,12,-280,0,0,0,0,64" },
  }} };
#endif // SHR3D_SFX_CORE_RAKARRACK_PRESETS
  }
#endif // SHR3D_SFX

  {
    for (const std::pair<std::string, std::map<std::string, std::string>> pair : serializedSaves)
    {
      ASSERT(pair.first == "Tones");
#ifdef SHR3D_SFX
      loadSfxTone(pair.second);
#endif // SHR3D_SFX
    }
  }

#ifdef SHR3D_SFX
  { // Load Tone Assignment 0
    for (i32 i = 0; i < ARRAY_SIZE(Global::effectChain); ++i)
    {
      Global::effectChain[i].id = Global::sfxTone[0][i].id;
      Global::effectChain[i].state = Global::sfxTone[0][i].state;
      if (Global::effectChain[i].id.system != SfxSystem::empty)
      {
        i32 instance = 0;
        for (i32 j = 0; j < i; ++j)
          if (Global::effectChain[i].id.system == Global::effectChain[j].id.system && Global::effectChain[i].id.sfxIndex == Global::effectChain[j].id.sfxIndex)
            ++instance;

        Sfx::loadParameters(Global::effectChain[i].id, instance, Global::sfxParameters[0][i]);
      }
    }
  }
#endif // SHR3D_SFX
}

void Tones::fini()
{
#ifdef SHR3D_SFX
  std::map<std::string, std::map<std::string, std::string>> serializedSaves;
  std::map<std::string, std::string> serializedSfxTones;

  for (auto& [sfxToneIndex, effectChain] : Global::sfxTone)
  {
    if (sfxToneIndex < 0)
      continue;

    char keyStr[16];

    if (!Global::sfxToneNames[sfxToneIndex].empty())
    {
      sprintf(reinterpret_cast<char*>(keyStr), "%04d,%02d", sfxToneIndex / Const::sfxToneTonesPerBank, sfxToneIndex % Const::sfxToneTonesPerBank);
      serializedSfxTones.insert({ keyStr, Global::sfxToneNames[sfxToneIndex] });
    }

    for (i32 j = 0; j < ARRAY_SIZE(Global::effectChain); ++j)
    {
      if (effectChain[j].id.system == SfxSystem::empty)
        continue;

      sprintf(reinterpret_cast<char*>(keyStr), "%04d,%02d,%02d", sfxToneIndex / Const::sfxToneTonesPerBank, sfxToneIndex % Const::sfxToneTonesPerBank, j);

      //ASSERT(Global::sfxParameters.contains(toneIndex));

#ifdef SHR3D_SFX_PLUGIN
      if (Sfx::sfxSystemIsPlugin(effectChain[j].id.system))
      {
        const std::string& parametersBase64 = Base64::encode(reinterpret_cast<const u8*>(Global::sfxParameters[sfxToneIndex][j].data()), Global::sfxParameters[sfxToneIndex][j].size());
        serializedSfxTones.insert({ keyStr, std::string(Sfx::sfxSystem2Name(effectChain[j].id.system)) + ',' + Sfx::names[effectChain[j].id.system][effectChain[j].id.sfxIndex] + ',' + (effectChain[j].state == SfxEffectState::muted ? 'M' : '-') + ',' + parametersBase64 });
      }
      else
#endif // SHR3D_SFX_PLUGIN
      {
        const std::string& parameters = Global::sfxParameters[sfxToneIndex][j];
        serializedSfxTones.insert({ keyStr, std::string(Sfx::sfxSystem2Name(effectChain[j].id.system)) + ',' + Sfx::names[effectChain[j].id.system][effectChain[j].id.sfxIndex] + ',' + (effectChain[j].state == SfxEffectState::muted ? 'M' : '-') + ',' + parameters });
      }
    }
  }
  serializedSaves.insert({ "Tones", serializedSfxTones });

  Ini::saveIniFile(Settings::pathTonesIni.c_str(), serializedSaves);
#endif // SHR3D_SFX
}

void Tones::loadSongToneFile(const std::vector<Arrangement::Info>& arrangementInfos, const std::map<std::string, std::vector<Ini::KeyValue>>& tonesIni)
{
  DEBUG_PRINT("loadSongToneFile()\n");

  ASSERT(tonesIni.size() == 1);
  ASSERT(tonesIni.contains("Tones"));

  const std::vector<Ini::KeyValue>& tones = tonesIni.at("Tones");

  for (const auto& [key, value] : tones)
  {
    ASSERT(key.size() >= 1);
    const u64 arrangementPos = key.find_first_of(',');
    const std::string arrangementName = key.substr(0, arrangementPos);

    SfxBankIndex sfxBankIndex = -1;
    for (ArrangementIndex arrangementIndex = 0; arrangementIndex < ArrangementIndex(arrangementInfos.size()); ++arrangementIndex)
    {
      if (arrangementName == arrangementInfos[arrangementIndex].arrangementName)
      {
        sfxBankIndex = Global::songStats.at(arrangementInfos[arrangementIndex].persistentId).sfxBankIndex;
        break;
      }
    }
    ASSERT(sfxBankIndex != -1);

    const std::string toneSlot = key.substr(arrangementPos + 1);
    const u64 toneSlotPos = toneSlot.find_first_of(',');
    const std::string tone = toneSlot.substr(0, toneSlotPos);
    const i32 toneIndex = atoi(reinterpret_cast<const char*>(tone.c_str())) /*+ toneIndexOffset*/;
    const SfxToneIndex sfxToneIndex = sfxBankIndex * Const::sfxToneTonesPerBank - toneIndex;
    if (toneSlotPos == std::string::npos)
    {
      Global::sfxToneNames[sfxToneIndex] = value;
    }
    else
    {
      const std::string slot = toneSlot.substr(toneSlotPos + 1);
      const i32 slotIndex = atoi(reinterpret_cast<const char*>(slot.c_str()));

      ASSERT(toneIndex < Const::sfxToneTonesPerBank);

      if (slotIndex > ARRAY_SIZE(Global::effectChain) - 1)
        continue;

      const u64 comma0 = value.find(',');
      const u64 comma1 = value.find(',', comma0 + 1);

      const std::string sfxSystemStr = value.substr(0, comma0);
      const SfxSystem sfxSystem = Sfx::name2SfxSystem(sfxSystemStr.c_str());

      const std::string sfxName = value.substr(comma0 + 1, comma1 - comma0 - 1);

      const u64 sfxNameSeperator = value.find_first_of(',');
      ASSERT(sfxNameSeperator != std::string::npos);

      for (SfxIndex k = 0; k < SfxIndex(Sfx::names[sfxSystem].size()); ++k)
      {
        if (sfxName != Sfx::names[sfxSystem][k])
          continue;

        Global::sfxTone[sfxToneIndex][slotIndex].state = SfxEffectState(value[comma1 + 1]);
        Global::sfxTone[sfxToneIndex][slotIndex].id.system = sfxSystem;
        Global::sfxTone[sfxToneIndex][slotIndex].id.sfxIndex = k;
        const std::string parameters = value.substr(comma1 + 3);

#ifdef SHR3D_SFX_PLUGIN
        if (Sfx::sfxSystemIsPlugin(sfxSystem))
        {
          const std::vector<u8> decodedData = Base64::decode(parameters.data(), parameters.size());
          const std::string decodedParameters(reinterpret_cast<const char*>(decodedData.data()), decodedData.size());

          Global::sfxParameters[sfxToneIndex][slotIndex] = decodedParameters;
        }
        else
#endif // SHR3D_SFX_PLUGIN
        {
          Global::sfxParameters[sfxToneIndex][slotIndex] = parameters;
        }
        break;
      }
    }
  }
}

#ifndef PLATFORM_EMSCRIPTEN
void Tones::saveSongToneFile(const SongIndex songIndex)
{
  std::map<std::string, std::map<std::string, std::string>> serializedSaves;
  std::map<std::string, std::string> serializedSfxTones;

  const Song::Info& songInfo = Global::songInfos.at(songIndex);
  const std::vector<Arrangement::Info>& arrangementInfos = songInfo.arrangementInfos;
  for (ArrangementIndex arrangementIndex = 0; arrangementIndex < ArrangementIndex(arrangementInfos.size()); ++arrangementIndex)
  {
    const Arrangement::Info& arrangementInfo = songInfo.arrangementInfos[arrangementIndex];

    const SfxBankIndex sfxBankIndex = Global::songStats[arrangementInfo.persistentId].sfxBankIndex;
    if (sfxBankIndex >= -1)
      continue;

    const char* arrangementName = arrangementInfo.arrangementName.c_str();

    for (i8 i = 0; i < Const::sfxToneTonesPerBank; ++i)
    {
      const SfxToneIndex sfxToneIndex = sfxBankIndex * Const::sfxToneTonesPerBank - i;
      if (!Global::sfxToneNames.contains(sfxToneIndex))
        continue;

      char keyStr[64];

      if (!Global::sfxToneNames[sfxToneIndex].empty())
      {
        sprintf(reinterpret_cast<char*>(keyStr), "%s,%02d", reinterpret_cast<const char*>(arrangementName), (-1 * sfxToneIndex) % Const::sfxToneTonesPerBank);
        serializedSfxTones.insert({ keyStr, Global::sfxToneNames[sfxToneIndex] });
      }

      for (i8 j = 0; j < ARRAY_SIZE(Global::effectChain); ++j)
      {
        const SfxChainEffect& sfxChainEffect = Global::sfxTone[sfxToneIndex][j];
        {
          if (sfxChainEffect.id.system == SfxSystem::empty)
            continue;

          sprintf(reinterpret_cast<char*>(keyStr), "%s,%02d,%02d", reinterpret_cast<const char*>(arrangementName), (-1 * sfxToneIndex) % Const::sfxToneTonesPerBank, j);

          //ASSERT(Global::sfxParameters.contains(toneIndex));

#ifdef SHR3D_SFX_PLUGIN
          if (Sfx::sfxSystemIsPlugin(sfxChainEffect.id.system))
          {
            const std::string& parametersBase64 = Base64::encode(reinterpret_cast<const u8*>(Global::sfxParameters[sfxToneIndex][j].data()), Global::sfxParameters[sfxToneIndex][j].size());
            serializedSfxTones.insert({ keyStr, std::string(Sfx::sfxSystem2Name(sfxChainEffect.id.system)) + ',' + Sfx::names[sfxChainEffect.id.system][sfxChainEffect.id.sfxIndex] + ',' + (sfxChainEffect.state == SfxEffectState::muted ? 'M' : '-') + ',' + parametersBase64 });
          }
          else
#endif // SHR3D_SFX_PLUGIN
          {
            const std::string& parameters = Global::sfxParameters[sfxToneIndex][j];
            serializedSfxTones.insert({ keyStr, std::string(Sfx::sfxSystem2Name(sfxChainEffect.id.system)) + ',' + Sfx::names[sfxChainEffect.id.system][sfxChainEffect.id.sfxIndex] + ',' + (sfxChainEffect.state == SfxEffectState::muted ? 'M' : '-') + ',' + parameters });
          }
        }
      }
    }
  }

  if (serializedSfxTones.size() >= 0)
  {
    serializedSaves.insert({ "Tones", serializedSfxTones });

    const std::string& songFilePath = songInfo.filepath;
    const std::string tonesIniFilePath = File::replaceExtension(songFilePath.c_str(), ".tones.ini");

    Ini::saveIniFile(tonesIniFilePath.c_str(), serializedSaves);
  }
}
#endif // PLATFORM_EMSCRIPTEN
